﻿local L = AceLibrary("AceLocale-2.2"):new("sRaidFrames")
local Aura = AceLibrary("SpecialEvents-Aura-2.0")
local roster = AceLibrary("Roster-2.1")
local HealComm = LibStub:GetLibrary("LibHealComm-3.0", true)
local ResComm = LibStub("LibResComm-1.0", true)
local Media = LibStub("LibSharedMedia-3.0")
local Banzai = LibStub("LibBanzai-2.0", true)
local LibTalentQuery = LibStub("LibTalentQuery-1.0", true)

Media:Register("statusbar", "Otravi", "Interface\\AddOns\\sRaidFrames\\textures\\otravi")
Media:Register("statusbar", "Smooth", "Interface\\AddOns\\sRaidFrames\\textures\\smooth")
Media:Register("statusbar", "Striped", "Interface\\AddOns\\sRaidFrames\\textures\\striped")
Media:Register("statusbar", "BantoBar", "Interface\\AddOns\\sRaidFrames\\textures\\bantobar")

sRaidFrames = AceLibrary("AceAddon-2.0"):new(
	"AceDB-2.0",
	"AceEvent-2.0",
	"AceConsole-2.0",
	"FuBarPlugin-2.0"
)

local sRaidFrames = sRaidFrames

local WoTLK = select(4, GetBuildInfo()) >= 30000 

sRaidFrames.hasIcon = "Interface\\Icons\\INV_Helmet_06"
sRaidFrames.defaultMinimapPosition = 180
sRaidFrames.cannotDetachTooltip = true
sRaidFrames.hasNoColor = true
sRaidFrames.clickableTooltip = true
sRaidFrames.hideWithoutStandby = true
sRaidFrames.independentProfile = true

local SpellCache = setmetatable({}, {
	__index = function(table, id)
		table[id] = GetSpellInfo(id)
		return table[id]
	end
})

function sRaidFrames:OnInitialize()

	self:RegisterDB("sRaidFramesDB")

	self:RegisterDefaults("profile", {
		Locked				= false,
		HealthFormat	= 'percent',
		Invert 				= false,
		Scale					= 1,
		Border				= true,
		Texture				= "Otravi",
		BuffType			= "debuffs",
		ShowOnlyDispellable	= 1,
		BackgroundColor		= {r = 0.32, g = 0.50, b = 0.70, a = 0.90},
		BorderColor		= {r = 1, g = 1, b = 1, a = 1},
		HealthTextColor	= {r = 1, g = 1, b = 1, a = 1},
		Growth				= "down",
		Spacing				= 0,
		ShowGroupTitles		= true,
		UnitTooltipMethod		= "notincombat",
		BuffTooltipMethod = "always",
		DebuffTooltipMethod = "always",
		UnitTooltipType	= "ctra",
		BuffFilter			= {},
		BuffBlacklist		= {},
		DebuffFilter		= {},
		DebuffWhitelist = {},
		PowerFilter			= {[SPELL_POWER_MANA] = true,[SPELL_POWER_RAGE] = true, [SPELL_POWER_ENERGY] = true},
		RangeCheck 			= true,
		RangeLimit			= 38,
		RangeFrequency	= 0.2,
		RangeAlpha 			= 0.5,
		ReadyCheck			= true,
		AggroCheck			= false,
		HighlightTarget	= false,
		HighlightHeals	= true,
		HighlightHealsSelf = true,
		HighlightDebuffs = "onlyself",
		Layout					= "CTRA_WithBorders",
		GroupSetup			= L["By class"],
		GroupSetups			= {},
		Positions				= { ['*'] = {} },
		StatusMaps			= {},
		HideInArena			= true,
		BuffDisplay			= "own",
	})

	self:RegisterChatCommand("/srf", "/sraidframes", self.options)

	self.opt = self.db.profile

	self.OnMenuRequest = self.options

	-- Init variables
	self.enabled = false
	self.frames, self.groupframes = {}, {}, {}
	self.res, self.RangeChecks = {}, {}
	self.FramesByUnit = {}
	self.statusstate = {}
	self.statusElements = {}
	self.validateStatusElements = {}
	self.MainTanks = {}
	self.DebuffCache = {}
	self.SpecCache = {}

	self.cooldownSpells = {}
	self.cooldownSpells["WARLOCK"] = SpellCache[27239] -- Soulstone Resurrection
	self.cooldownSpells["DRUID"] = SpellCache[26994] -- Rebirth
	self.cooldownSpells["SHAMAN"] = SpellCache[20608] -- Reincarnation
	self.cooldownSpells["PALADIN"] = SpellCache[19753] -- Divine Intervention

	self.cleanseTypes= {
		["PRIEST"] = {
			["Magic"] = true,
			["Disease"] = true,
		},
		["SHAMAN"] = {
			["Poison"] = true,
			["Disease"] = true,
		},
		["PALADIN"] = {
			["Magic"] = true,
			["Poison"] = true,
			["Disease"] = true,
		},
		["MAGE"] = {
			["Curse"] = true,
		},
		["DRUID"] = {
			["Curse"] = true,
			["Poison"] = true,
		},
	}

	self:AddStatusMap("ReadyCheck_Pending", 80, {"background"}, L["Ready?"], {r = 0.1, g = 0.1, b = 0.1})

	self:AddStatusMap("Death", 70, {"background"}, L["Dead"], {r = 0.1, g = 0.1, b = 0.1, a = 1})

	-- Spirit of Redemption
	self:AddStatusMap("Buff_"..SpellCache[20711], 65, {"statusbar"}, L["Dead"], {r=1,g=0,b=0,a=1})
	-- Divine Intervention
	self:AddStatusMap("Buff_"..SpellCache[19753], 65, {"statusbar"}, L["Intervened"], {r=1,g=0,b=0,a=1})

	self:AddStatusMap("Aggro", 50, {"border"}, "Aggro", {r = 1, g = 0, b = 0})
	self:AddStatusMap("Target", 55, {"border"}, "Target", {r = 1, g = 0.75, b = 0})

	self:AddStatusMap("Debuff_Curse", 55, {"background"}, "Cursed", {r=1, g=0, b=0.75, a=0.5})
	self:AddStatusMap("Debuff_Magic", 54, {"background"}, "Magic", {r=1, g=0, b=0, a=0.5})
	self:AddStatusMap("Debuff_Disease", 53, {"background"}, "Diseased", {r=1, g=1, b=0, a=0.5})
	self:AddStatusMap("Debuff_Poison", 52, {"background"}, "Poisoned", {r=0, g=0.5, b=0, a=0.5})

	-- Shield Wall
	self:AddStatusMap("Buff_"..SpellCache[871], 53, {"statusbar"}, SpellCache[871], {r=1,g=1,b=1,a=1})
	-- Last Stand
	self:AddStatusMap("Buff_"..SpellCache[12975], 52, {"statusbar"}, SpellCache[12975], {r=1,g=1,b=1,a=1})
	-- Vanish
	self:AddStatusMap("Buff_"..SpellCache[26889], 51, {"statusbar"}, L["Vanished"], {r=0,g=1,b=0,a=1})
	-- Invisibility
	self:AddStatusMap("Buff_"..SpellCache[66], 51, {"statusbar"}, SpellCache[66], {r=0,g=1,b=0,a=1})
	-- Gift of Life
	self:AddStatusMap("Buff_"..SpellCache[23725], 51, {"statusbar"}, SpellCache[23725], {r=1,g=1,b=1,a=1})
	-- Evasion
	self:AddStatusMap("Buff_"..SpellCache[38541], 50, {"statusbar"}, SpellCache[38541], {r=1,g=1,b=0,a=1})
	-- Blessing of Protection
	self:AddStatusMap("Buff_"..SpellCache[41450], 50, {"statusbar"}, L["Protection"], {r=1,g=1,b=1,a=1})
	-- Stealth
	self:AddStatusMap("Buff_"..SpellCache[1787], 50, {"statusbar"}, L["Stealthed"], {r=1,g=1,b=1,a=1})
	-- Innervate
	self:AddStatusMap("Buff_"..SpellCache[29166], 51, {"statusbar"}, L["Innervating"], {r=0,g=1,b=0,a=1})
	-- Ice Block
	self:AddStatusMap("Buff_"..SpellCache[45438], 50, {"statusbar"}, SpellCache[45438], {r=1,g=1,b=1,a=1})
	
	-- Feign Death
	self:AddStatusMap("Buff_"..SpellCache[5384], 50, {"statusbar"}, SpellCache[5384], {r=0,g=1,b=0,a=1})
	-- Cloak of Shadows
	self:AddStatusMap("Buff_"..SpellCache[39666], 50, {"statusbar"}, SpellCache[39666], {r=1,g=1,b=1,a=1})
	-- Divine Shield
	self:AddStatusMap("Buff_"..SpellCache[40733], 50, {"statusbar"}, SpellCache[40733], {r=1,g=1,b=1,a=1})
	-- Power Infusion
	self:AddStatusMap("Buff_"..SpellCache[10060], 50, {"statusbar"}, L["Infused"], {r=1,g=1,b=1,a=1})

	-- Misdirection
	self:AddStatusMap("Buff_"..SpellCache[35079], 45, {"statusbar"}, SpellCache[35079], {r=0,g=1,b=0,a=1})
	-- Intervene
	self:AddStatusMap("Buff_"..SpellCache[3411], 45, {"statusbar"}, SpellCache[3411], {r=0,g=1,b=0,a=1})
	-- Fear Ward
	self:AddStatusMap("Buff_"..SpellCache[6346], 40, {"statusbar"}, SpellCache[6346], {r=1,g=1,b=0,a=1})
	-- Heroism
	self:AddStatusMap("Buff_"..SpellCache[32182], 39, {"statusbar"}, SpellCache[32182], {r=1,g=1,b=1,a=1})
	-- Bloodlust
	self:AddStatusMap("Buff_"..SpellCache[2825], 39, {"statusbar"}, SpellCache[2825], {r=1,g=1,b=1,a=1})

	-- Drums of Battle
	self:AddStatusMap("Buff_"..SpellCache[35476], 38, {"statusbar"}, SpellCache[35476], {r=1,g=1,b=1,a=1})

	self:AddStatusMap("Heal", 36, {"statusbar"}, "Inc. heal", {r = 0, g = 1, b = 0})
	self:AddStatusMap("Buff_"..SpellCache[15473], 35, {"statusbar"}, SpellCache[15473], {r=1,g=0,b=0.75,a=1})

	self:RegisterStatusElement("border", "Border", 
		function(self, frame, status)
			if status == nil then
				frame:SetBackdropBorderColor(self.opt.BorderColor.r, self.opt.BorderColor.g, self.opt.BorderColor.b, self.opt.BorderColor.a or 1)
			else
				frame:SetBackdropBorderColor(status.color.r, status.color.g, status.color.b, status.color.a or 1)
			end
		end
	)
	
	self:RegisterStatusElement("background", "Background",
		function(self, frame, status)
			if status == nil then
				frame:SetBackdropColor(self.opt.BackgroundColor.r, self.opt.BackgroundColor.g, self.opt.BackgroundColor.b, self.opt.BackgroundColor.a or 1)
			else
				frame:SetBackdropColor(status.color.r, status.color.g, status.color.b, status.color.a or 1)
			end
		end
	)
	
	self:RegisterStatusElement("statusbar", "StatusBar",
		function(self, frame, status)
			if status == nil then
				frame.statustext:SetText(nil)
			else
				frame.statustext:SetText(status.text)
				frame.statustext:SetTextColor(status.color.r, status.color.g, status.color.b, status.color.a or 1)
			end
		end
	)

	self:DefaultGroupSetups()

	self:chatUpdateFilterMenu()
	self:chatUpdateBuffMenu()
	self:chatUpdateDebuffMenu()
	self:chatUpdateStatusElements()

	self.master = CreateFrame("Frame", "sRaidFrame", UIParent)
	self.master:SetMovable(true)
	self.master:SetScale(self.opt.Scale)

	self.master:SetHeight(200);
	self.master:SetWidth(200);
end

function sRaidFrames:OnProfileEnable()
	self.opt = self.db.profile
	self:SetPosition()
end

function sRaidFrames:OnEnable()
	self.PlayerClass = select(2, UnitClass("player"))

	self:AddRangeFunction(10, function (unit) return CheckInteractDistance(unit, 3) == 1 end)
	self:AddRangeFunction(28, function (unit) return CheckInteractDistance(unit, 4) == 1 end)
	self:AddRangeFunction(38, function (unit) return UnitInRange(unit) end)
	self:AddRangeFunction(100, function (unit) return UnitIsVisible(unit) == 1 end)

	self:RegisterBucketEvent("RAID_ROSTER_UPDATE", 0.2, "UpdateRoster")
	self:RegisterEvent("RosterLib_UnitChanged")
	self:RegisterEvent("SharedMedia_SetGlobal")

	self:ScanSpellbookForRange()

	self:UpdateRoster()
end

function sRaidFrames:OnDisable()
	self.enabled = false
	self.master:Hide()
end

local EventsHealth = {"UNIT_HEALTH", "UNIT_MAXHEALTH"}
local EventsPower = {"UNIT_MANA", "UNIT_RAGE", "UNIT_ENERGY", "UNIT_FOCUS", "UNIT_RUNIC_POWER", "UNIT_MAXMANA", "UNIT_MAXRAGE", "UNIT_MAXFOCUS", "UNIT_MAXENERGY", "UNIT_MAXRUNIC_POWER", "UNIT_DISPLAYPOWER"}

function sRaidFrames:EnableFrames()
	self.enabled = true
	self.statusstate = {}
	self.SpecCache = {}

	self:CreateFrames()

	self:RegisterBucketEvent(EventsHealth, 0.05, "UNIT_HEALTH")
	self:RegisterBucketEvent(EventsPower, 1, "UNIT_POWER")
	self:RegisterBucketEvent("UNIT_AURA", 0.2)

	self:RegisterEvent("PLAYER_TARGET_CHANGED", "UpdateTarget")

	if HealComm then
		HealComm.RegisterCallback(self, "HealComm_DirectHealStart")
		HealComm.RegisterCallback(self, "HealComm_DirectHealStop")
		HealComm.RegisterCallback(self, "HealComm_DirectHealDelayed")
		HealComm.RegisterCallback(self, "HealComm_HealModifierUpdate")
	end
	
	if ResComm then
		ResComm.RegisterCallback(self, "ResComm_ResStart")
		ResComm.RegisterCallback(self, "ResComm_ResEnd")
		ResComm.RegisterCallback(self, "ResComm_Ressed")
		ResComm.RegisterCallback(self, "ResComm_CanRes")
		ResComm.RegisterCallback(self, "ResComm_ResExpired")
	end

	if Banzai then
		Banzai:RegisterCallback(sRaidFrames.Banzai_Callback)
	end
	
	if LibTalentQuery then
		LibTalentQuery.RegisterCallback(self, "TalentQuery_Ready")
	end

	self:RegisterEvent("oRA_MainTankUpdate")

	self:RegisterEvent("SpecialEvents_UnitDebuffGained")
	self:RegisterEvent("SpecialEvents_UnitDebuffLost")
	self:RegisterEvent("SpecialEvents_UnitBuffGained")
	self:RegisterEvent("SpecialEvents_UnitBuffLost")

	self:RegisterEvent("READY_CHECK")
	self:RegisterEvent("READY_CHECK_CONFIRM")
	self:RegisterEvent("READY_CHECK_FINISHED")

	self:ScheduleRepeatingEvent("sRaidFramesRangeCheck", self.RangeCheck, self.opt.RangeFrequency, self)

	self:UpdateRoster()
	self:oRA_MainTankUpdate()

	self.master:Show()
end

function sRaidFrames:DisableFrames()
	if not self.enabled then return end
	self.statusstate = {}
	self.FramesByUnit = {}

	self:UnregisterBucketEvent(EventsHealth)
	self:UnregisterBucketEvent(EventsPower)
	self:UnregisterBucketEvent("UNIT_AURA")

	self:UnregisterEvent("PLAYER_TARGET_CHANGED")

	if Banzai then
		Banzai:UnregisterCallback(sRaidFrames.Banzai_Callback)
	end

	self:UnregisterEvent("oRA_MainTankUpdate")

	self:UnregisterEvent("SpecialEvents_UnitDebuffGained")
	self:UnregisterEvent("SpecialEvents_UnitDebuffLost")
	self:UnregisterEvent("SpecialEvents_UnitBuffGained")
	self:UnregisterEvent("SpecialEvents_UnitBuffLost")

	self:CancelScheduledEvent("sRaidFramesRangeCheck")

	if HealComm then
		HealComm.UnregisterCallback(self, "HealComm_DirectHealStart")
		HealComm.UnregisterCallback(self, "HealComm_DirectHealStop")
		HealComm.UnregisterCallback(self, "HealComm_DirectHealDelayed")
		HealComm.UnregisterCallback(self, "HealComm_HealModifierUpdate")
	end
	
	if ResComm then
		ResComm.UnregisterCallback(self, "ResComm_ResStart")
		ResComm.UnregisterCallback(self, "ResComm_ResEnd")
		ResComm.UnregisterCallback(self, "ResComm_Ressed")
		ResComm.UnregisterCallback(self, "ResComm_CanRes")
		ResComm.UnregisterCallback(self, "ResComm_ResExpired")
	end
	
	if LibTalentQuery then
		LibTalentQuery.UnregisterCallback(self, "TalentQuery_Ready")
	end

	self.enabled = false
	self.master:Hide()
end

function sRaidFrames:TalentQuery_Ready(e, name)
	local unit = roster:GetUnitIDFromName(name)
	if not unit then return end
	
	local frominspect = not UnitIsUnit(unit, "player")
	for tab = 1, GetNumTalentTabs(frominspect) do
		local treename, _, pointsspent = GetTalentTabInfo(tab, frominspect)
		if pointsspent >= 31 then
			self.SpecCache[unit] = treename
		end
	end
end

function sRaidFrames:SharedMedia_SetGlobal(type, handle)
	if type == "statusbar" then
		local texture = Media:Fetch("statusbar", handle)
		for _, frame in pairs(self.frames) do
			frame.hpbar:SetStatusBarTexture(texture)
			frame.mpbar:SetStatusBarTexture(texture)
		end
	end
end

function sRaidFrames:ScanSpellbookForRange()
	local i = 1
	while true do
		local SpellId = i
		local name, _, _, _, _, _, _, _, maxRange = GetSpellInfo(SpellId, BOOKTYPE_SPELL)
		if not name then return end

		if maxRange and IsSpellInRange(SpellId, "spell", "player") ~= nil then
			self:AddRangeFunction(tonumber(maxRange), function (unit) return IsSpellInRange(SpellId, "spell", unit) == 1 end)
		end
		i = i + 1
	end
end

function sRaidFrames:UpdateRoster()
	local inRaid = GetNumRaidMembers() > 0
	local inBG = select(2, IsInInstance()) == "pvp"
	local inArena = select(2, IsInInstance()) == "arena"

	if not self.enabled then
		if inRaid and not (self.opt.HideInArena and inArena) then
			self:EnableFrames()
		end
	end
	
	if self.enabled then
		if not inRaid then
			self:DisableFrames()
		end
		if self.opt.HideInArena and inArena then
			self:DisableFrames()
		end
	end	
end

function sRaidFrames:UpdateAllUnits()
	for unit in pairs(self:GetAllUnits()) do
		self:UpdateAll(unit)
	end
end

function sRaidFrames:UpdateAll(unit)
	self:UpdateUnitDetails(unit)
	self:UpdateUnitHealth(unit)
	self:UpdateUnitPower(unit)
	self:UpdateStatuses(unit)
	self:UpdateAuras(unit)
end

function sRaidFrames:UNIT_POWER(units)
	for unit in pairs(units) do
		self:UpdateUnitPower(unit)
	end
end

function sRaidFrames:UNIT_HEALTH(units)
	for unit in pairs(units) do
		self:UpdateUnitHealth(unit)
	end
end

function sRaidFrames:UNIT_AURA(units)
	for unit in pairs(units) do
		self:UpdateAuras(unit)
	end
end

function sRaidFrames:READY_CHECK(author)
	if not self.opt.ReadyCheck or not (IsRaidLeader() or IsRaidOfficer()) then return end

	local authorid = roster:GetUnitIDFromName(author)

	for unitid in pairs(self:GetAllUnits()) do
		if unitid ~= authorid then
			self:SetStatus(unitid, "ReadyCheck_Pending")
		end
	end
end

function sRaidFrames:READY_CHECK_CONFIRM(id, confirm)
	if not self.opt.ReadyCheck then return end

	local unitid = "raid"..id

	self:UnsetStatus(unitid, "ReadyCheck_Pending")
	if confirm then
		self:SetStatus(unitid, "ReadyCheck_Ready")
	else
		self:SetStatus(unitid, "ReadyCheck_NotReady")
	end
end

function sRaidFrames:READY_CHECK_FINISHED()
	if not self.opt.ReadyCheck then return end

	for unitid in pairs(self:GetAllUnits()) do
		self:UnsetStatus(unitid, "ReadyCheck_Pending")
		self:UnsetStatus(unitid, "ReadyCheck_Ready")
		self:UnsetStatus(unitid, "ReadyCheck_NotReady")
	end
end

function sRaidFrames:RosterLib_UnitChanged(unitid, name, class, subgroup, rank, oldname, oldunitid, oldclass, oldsubgroup, oldrank)
	if not (unitid or oldunitid):match("^raid(%d+)$") then return end

	ShouldUpdateFrameCache = true

	if not name then
		-- Unit left
		self.statusstate[oldunitid] = nil
		self.SpecCache[oldunitid] = nil
	elseif not oldname then
		-- Unit joined
		self.statusstate[unitid] = {}
		self.SpecCache[unitid] = nil
		self:UpdateAll(unitid)
	else
		-- Unit changed
		self.statusstate[unitid] = {}
		self.SpecCache[unitid] = nil
		self:UpdateAll(unitid)
	end
end

local LastTarget
function sRaidFrames:UpdateTarget()
	if not self.opt.HighlightTarget then return end

	if LastTarget then
		self:UnsetStatus(LastTarget, "Target")
	end

	for unit in pairs(self:GetAllUnits()) do
		if UnitIsUnit(unit, "target") then
			self:SetStatus(unit, "Target")
			LastTarget = unit
			break
		end
	end
end

function sRaidFrames.Banzai_Callback(aggro, name, ...)
	local self = sRaidFrames
	if not self.opt.AggroCheck then return end

	local unit = roster:GetUnitIDFromName(name)
	if not unit or not self:IsTracking(unit) then return end

	local aggro = Banzai:GetUnitAggroByUnitName(name)
	if aggro then
		self:SetStatus(unit, "Aggro")
	else
		self:UnsetStatus(unit, "Aggro")
	end
end

function sRaidFrames:SpecialEvents_UnitDebuffGained(unit, debuff, apps, type, tex, rank, index)
	tinsert(self.DebuffCache, debuff)
	if not self:IsTracking(unit) then return end
	if not type or type == "" then return end
	if self.opt.DebuffFilter[debuff] then return end

	local HighLightRule = self.opt.HighlightDebuffs

	if HighLightRule == "never" then
		return -- Don't highlight it
	elseif HighLightRule == "onlyself" and not self:CanDispell(type) then
		return -- Don't highlight it
	end

	self:SetStatus(unit, "Debuff_".. type, debuff)
end

function sRaidFrames:SpecialEvents_UnitDebuffLost(unit, debuff, apps, type, tex, rank)
	if not self:IsTracking(unit) then return end
	if not type or type == "" then return end

	if not Aura:UnitHasDebuffType(unit, type) then
		self:UnsetStatus(unit, "Debuff_".. type)
	end
end

function sRaidFrames:SpecialEvents_UnitBuffGained(unit, buff, index, apps, tex, rank)
	if not self:IsTracking(unit) then return end

	-- Divine Intervention
	if buff == SpellCache[19753] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Misdirection
	elseif buff == SpellCache[35079] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Feign Death
	elseif buff == SpellCache[5384] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Intervene
	elseif buff == SpellCache[3411] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Innervate
	elseif buff == SpellCache[29166] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Spirit of Redemption
	elseif buff == SpellCache[20711] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Shield Wall
	elseif buff == SpellCache[871] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Last Stand
	elseif buff == SpellCache[12975] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Gift of Life
	elseif buff == SpellCache[23725] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Ice Block
	elseif buff == SpellCache[45438] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Blessing of Protection
	elseif buff == SpellCache[41450] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Divine Shield
	elseif buff == SpellCache[40733] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Vanish
	elseif buff == SpellCache[26889] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Cloak of Shadows
	elseif buff == SpellCache[39666] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Invisibility
	elseif buff == SpellCache[66] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Stealth
	elseif buff == SpellCache[1787] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Evasion
	elseif buff == SpellCache[38541] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Power Infusion
	elseif buff == SpellCache[10060] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Heroism
	elseif buff == SpellCache[32182] and tex == "Interface\\Icons\\Ability_Shaman_Heroism" then
		self:SetStatus(unit, "Buff_"..buff)

	-- Bloodlust
	elseif buff == SpellCache[2825] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Drums of Battle
	elseif buff == SpellCache[35476] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Fear Ward
	elseif buff == SpellCache[6346] then
		self:SetStatus(unit, "Buff_"..buff)

	-- Shadowform
	elseif buff == SpellCache[15473] then
		self:SetStatus(unit, "Buff_"..buff)
	end
end

function sRaidFrames:SpecialEvents_UnitBuffLost(unit, buff, apps, type, tex, rank)
	if not self:IsTracking(unit) then return end
	self:UnsetStatus(unit, "Buff_"..buff)
end

function sRaidFrames:ResComm_ResStart(event, _, _, target)
	local unit = roster:GetUnitIDFromName(target)
	if unit then
		self.res[unit] = 3
		self:UpdateUnitHealth(unit)
	end
end

function sRaidFrames:ResComm_ResEnd(event, _, target)
	if ResComm:IsUnitBeingRessed(target) then return end
	
	local unit = roster:GetUnitIDFromName(target)
	if unit then
		self.res[unit] = nil
		self:UpdateUnitHealth(unit)
	end
end

function sRaidFrames:ResComm_CanRes(event, target)
	local unit = roster:GetUnitIDFromName(target)
	if unit then
		self.res[unit] = 1
		self:UpdateUnitHealth(unit)
	end
end

function sRaidFrames:ResComm_Ressed(event, target)
	local unit = roster:GetUnitIDFromName(target)
	if unit then
		self.res[unit] = 2
		self:UpdateUnitHealth(unit)
	end
end

function sRaidFrames:ResComm_ResExpired(event, target)
	local unit = roster:GetUnitIDFromName(target)
	if unit then
		self.res[unit] = nil
		self:UpdateUnitHealth(unit)
	end
end

function sRaidFrames:oRA_MainTankUpdate()
	if not oRA or not oRA.maintanktable then return end
	for i = 0, 10 do
		self.MainTanks[i] = oRA.maintanktable[i] or ""
	end

	self:ScheduleLeaveCombatAction(self.SetGroupFilters, self)
end

function sRaidFrames:IsUnitInRange(unit, range)
	if not self.RangeChecks[range] then
		return false
	end
	return self.RangeChecks[range](unit)
end

function sRaidFrames:CanDispell(type)
	return (self.cleanseTypes[self.PlayerClass] and self.cleanseTypes[self.PlayerClass][type])
end

function sRaidFrames:RangeCheck()
	if not self.opt.RangeCheck then return end
	local RangeLimit = self.opt.RangeLimit
	local RangeAlpha = self.opt.RangeAlpha

	for unit in pairs(self:GetAllUnits()) do
		if self:IsUnitInRange(unit, RangeLimit) then
			for _, f in pairs(self:FindUnitFrames(unit)) do
				f:SetAlpha(1)
			end
		else
			for _, f in pairs(self:FindUnitFrames(unit)) do
				f:SetAlpha(RangeAlpha)
			end
		end
	end
end

function sRaidFrames:AddRangeFunction(range, check)
	if self.RangeChecks[range] then return end
	self.RangeChecks[range] = check
	self:UpdateRangeLimitOptions()
end

function sRaidFrames:UpdateRangeFrequency()
		self:CancelScheduledEvent("sRaidFramesRangeCheck")
		self:ScheduleRepeatingEvent("sRaidFramesRangeCheck", self.RangeCheck, self.opt.RangeFrequency, self)
end

function sRaidFrames:UpdateStatuses(unit)
	for element in pairs(self.statusElements) do
		self:UpdateUnitStatusElement(unit, element)
	end
end

function sRaidFrames:UpdateUnitDetails(unit)
	if not self:IsTracking(unit) then return end
	for _, f in pairs(self:FindUnitFrames(unit)) do
		local class = select(2, UnitClass(unit))
		if class then
			f.title:SetText(UnitName(unit))

			local color = RAID_CLASS_COLORS[class]
			f.title:SetTextColor(color.r, color.g, color.b, 1)
		else
			f.title:SetText(UnitName(unit) or L["Unknown"])
		end
	end
end

function sRaidFrames:UpdateUnitHealth(unit)
	if not self:IsTracking(unit) then return end
	for _, f in pairs(self:FindUnitFrames(unit)) do
		local status, dead, ghost = nil, UnitIsDead(unit), UnitIsGhost(unit)

		if not UnitIsConnected(unit) then status = "|cffff0000"..L["Offline"].."|r"
		elseif dead and self.res[unit] == 1 then status = "|cff00ff00"..L["Can Recover"].."|r"
		elseif (dead or ghost) and self.res[unit] == 2 then status = "|cff00ff00"..L["Resurrected"].."|r"
		elseif (dead or ghost) and self.res[unit] == 3 then status = "|cffff8c00"..L["Resurrecting"].."|r"
		elseif ghost then status = "|cffff0000"..L["Released"].."|r"
		elseif dead then status = "|cffff0000"..L["Dead"].."|r"
		end

		if status then
			f.hpbar.text:SetText(status)
			f.hpbar:SetValue(0)
			f.mpbar:SetValue(0)
			self:SetStatus(unit, "Death")
		else
			self:UnsetStatus(unit, "Death")
			self.res[unit] = nil
			local hp = UnitHealth(unit) or 0
			local hpmax = UnitHealthMax(unit)
			local hpp = (hpmax ~= 0) and ceil((hp / hpmax) * 100) or 0
			local hptext, hpvalue = nil, nil

			local format = self.opt.HealthFormat
			if format == "percent" then
				hptext = hpp .."%"
			elseif format == "deficit" then
				hptext = (hp-hpmax) ~=  0 and (hp-hpmax) or nil
			elseif format == "current" then
				hptext = hp
			elseif format == "curmax" then
				hptext = hp .."/".. hpmax
			elseif format == "curdeficit" then
				hptext = (hp-hpmax) ~=  0 and  hp .." |cffff0000".. (hp-hpmax) or hpmax
			end

			if self.opt.Invert then
				hpvalue = 100 - hpp
			else
				hpvalue = hpp
			end

			f.hpbar.text:SetText(hptext)
			f.hpbar:SetValue(hpvalue)
			f.hpbar:SetStatusBarColor(self:GetHPSeverity(hpp/100))
		end
	end
end

function sRaidFrames:UpdateUnitPower(unit)
	if not self:IsTracking(unit) then return end
	for _, f in pairs(self:FindUnitFrames(unit)) do
		local powerType = UnitPowerType(unit)
		if self.opt.PowerFilter[powerType] == false then
			f.mpbar:SetValue(0)
		else
			local color = ManaBarColor and ManaBarColor[powerType] or PowerBarColor[powerType]
			local mp = UnitMana(unit) or 0
			local mpmax = UnitManaMax(unit)
			local mpp = (mpmax ~= 0) and ceil((mp / mpmax) * 100) or 0
			f.mpbar:SetStatusBarColor(color.r, color.g, color.b)
			f.mpbar:SetValue(mpp)
		end
	end
end

function sRaidFrames:UpdateAuras(unit)
	if not self:IsTracking(unit) then return end
	local layout = self:GetLayout()
	for _, f in pairs(self:FindUnitFrames(unit)) do
		for i = 1, layout.debuffCount do
			f["aura".. i]:Hide()
		end

		for i = 1, layout.buffCount do
			f["buff".. i]:Hide()
		end

		local BuffType = self.opt.BuffType
		local DebuffSlots = 0
		if BuffType == "debuffs" or BuffType == "buffsifnotdebuffed" then
			local DebuffFilter = self.opt.DebuffFilter
			local DebuffWhitelist = self.opt.DebuffWhitelist
			local ShowOnlyDispellable = self.opt.ShowOnlyDispellable
			for i=1,40 do
				local debuffName, _, debuffTexture, debuffApplications, debuffType = UnitDebuff(unit, i)
				if not debuffTexture then break end
				if not DebuffFilter[debuffName] and ((ShowOnlyDispellable and (self:CanDispell(debuffType) or DebuffWhitelist[debuffName])) or not ShowOnlyDispellable) then
					DebuffSlots = DebuffSlots + 1
					local debuffFrame = f["aura".. DebuffSlots]
					if not debuffFrame then break end
					debuffFrame.unitid = unit
					debuffFrame.debuffid = i
					debuffFrame.count:SetText(debuffApplications > 1 and debuffApplications or nil);
					debuffFrame.texture:SetTexture(debuffTexture)
					debuffFrame:Show()
				end

				if DebuffSlots == layout.debuffCount then break end
			end
		end

		if BuffType == "buffs" or (BuffType == "buffsifnotdebuffed" and DebuffSlots == 0) then
			local BuffSlots = 0
			local BuffFilter = self.opt.BuffFilter
			local HasBuffFilter = next(BuffFilter) and true or false
			local BuffBacklist = self.opt.BuffBlacklist
			local BuffDisplay = self.opt.BuffDisplay
			local showOnlyCastable = BuffDisplay == "own" or BuffDisplay == "class" or false

			for i=1,32 do
				local buffName, _, buffTexture, buffApplications, buffDuration = UnitBuff(unit, i, showOnlyCastable)
				if not buffTexture then break end

				if ((showOnlyCastable and buffDuration and BuffDisplay == "own") or (showOnlyCastable and BuffDisplay == "class") or BuffDisplay == "all") and not BuffBacklist[buffName] and (not HasBuffFilter or (HasBuffFilter and BuffFilter[buffName])) then
						BuffSlots = BuffSlots + 1
						local buffFrame = f["buff".. BuffSlots]
						if not buffFrame then break end
						buffFrame.buffid = i
						buffFrame.unitid = unit
						buffFrame.showCastable = showOnlyCastable
						buffFrame.count:SetText(buffApplications > 1 and buffApplications or nil)
						buffFrame.texture:SetTexture(buffTexture)
						buffFrame:Show()
				end

				if BuffSlots == layout.buffCount then break end
			end
		end
	end
end

function sRaidFrames:AddStatusMap(statuskey, priority, elements, text, color)
	if self.opt.StatusMaps[statuskey] then return end

	self.opt.StatusMaps[statuskey] = {["priority"] = priority, ["elements"] = {}, ["text"] = text, ["color"] = color, ["enabled"] = true}
	for _, element in pairs(elements) do
		self.opt.StatusMaps[statuskey].elements[element] = true
	end

	self:chatUpdateStatusElements()
end

function sRaidFrames:GetStatus(unit, statuskey)
	return self.statusstate[unit] and self.statusstate[unit][statuskey] or nil
end

function sRaidFrames:SetStatus(unit, statuskey, text, color, update)
	if not self:IsTracking(unit) then return end

	local map = self.opt.StatusMaps[statuskey]

	if not map then return end
	if map.enabled == false then return end

	if not self.statusstate[unit] then
		self.statusstate[unit] = {}
	end

	local hasStatus = self:GetStatus(unit, statuskey)
	if hasStatus and not update then return end

	if text then
		map.text = text
	end

	if color then
		map.color = color
	end

	self.statusstate[unit][statuskey] = map

	self:UpdateStatusElements(unit, statuskey)
end

function sRaidFrames:UnsetStatus(unit, statuskey)
	if not self:IsTracking(unit) then return end
	if self.opt.StatusMaps[statuskey] and self:GetStatus(unit, statuskey) then
		self.statusstate[unit][statuskey] = nil
		self:UpdateStatusElements(unit, statuskey)
	end
end

function sRaidFrames:UpdateStatusElements(unit, statuskey)
	for element in pairs(self.opt.StatusMaps[statuskey].elements) do
		self:UpdateUnitStatusElement(unit, element)
	end
end

function sRaidFrames:RegisterStatusElement(element, name, func)
	self.statusElements[element] = { ["func"] = func, ["name"] = name }
	self.validateStatusElements[element] = name
end

function sRaidFrames:UpdateUnitStatusElement(unit, element)
	local status = self:GetTopStatus(element, unit)

	for _, frame in pairs(self:FindUnitFrames(unit)) do
		if self.statusElements[element] then
			self.statusElements[element].func(self, frame, status)
		end
	end
end

function sRaidFrames:GetTopStatus(element, unit)
	if not self.statusstate[unit] then
		return nil
	end

	local TopStatus, TopPriority = nil, 0
	for name, data in pairs(self.statusstate[unit]) do
		if self.opt.StatusMaps[name].elements[element] and self.opt.StatusMaps[name].priority > TopPriority then
			TopPriority = self.opt.StatusMaps[name].priority
			TopStatus = name
		end
	end

	return self.statusstate[unit][TopStatus] or nil
end

function sRaidFrames:GetHPSeverity(percent)
	if (percent >= 0.5) then
		return (1.0-percent)*2, 1.0, 0.0
	else
		return 1.0, percent*2, 0.0
	end
end

function sRaidFrames:QueryTooltipDisplay(value)
	if value == "never" then
		return false
	elseif value == "notincombat" and UnitAffectingCombat("player") then
		return false
	else
		return true
	end
end

function sRaidFrames:UnitTooltip(frame)
	local name, rank, subgroup, level, class, eclass, zone, _, _, role  = GetRaidRosterInfo(frame.id)
	local unit = frame:GetAttribute("unit")
	if not unit or not name then return end
	
	GameTooltip:SetOwner(frame)
	
	if self.opt.UnitTooltipType == "blizz" then
			GameTooltip:SetUnit(unit)
			GameTooltip:Show()
			return
	end

	GameTooltip:AddDoubleLine(name, level > 0 and level or nil, RAID_CLASS_COLORS[eclass].r, RAID_CLASS_COLORS[eclass].g, RAID_CLASS_COLORS[eclass].b, 1, 1, 1)
	if UnitIsAFK(unit) then
		GameTooltip:AddLine(L["AFK: Away From Keyboard"], 1, 1, 0)
	end

	if LibTalentQuery and not self.SpecCache[unit] then
		if UnitIsUnit(unit, "player") then
			self:TalentQuery_Ready(_, name)
		else
			LibTalentQuery:Query(unit)
		end
	end
	
	if LibTalentQuery and self.SpecCache[unit] then
		GameTooltip:AddDoubleLine(UnitRace(unit) .. " " .. class, self.SpecCache[unit] or UNKNOWN, 1, 1, 1, RAID_CLASS_COLORS[eclass].r, RAID_CLASS_COLORS[eclass].g, RAID_CLASS_COLORS[eclass].b);
	else
		GameTooltip:AddLine(UnitRace(unit) .. " " .. class, 1, 1, 1);
	end
	
	GameTooltip:AddDoubleLine(zone or UNKNOWN, L["Group %d"]:format(subgroup), 1, 1, 1, 1, 1, 1);

	local cooldownSpell = self.cooldownSpells[eclass]
	if oRA and oRA:HasModule("OptionalCooldown") and cooldownSpell then
		if oRA:GetModule("OptionalCooldown").db.realm.cooldowns and oRA:GetModule("OptionalCooldown").db.realm.cooldowns[name] then
			local expire = oRA:GetModule("OptionalCooldown").db.realm.cooldowns[name]-time()
			if expire > 0 then
				GameTooltip:AddDoubleLine(cooldownSpell, SecondsToTime(expire), nil, nil, nil, 1, 0, 0)
			else
				GameTooltip:AddDoubleLine(cooldownSpell, L["Ready!"], nil, nil, nil, 0, 1, 0)
			end
		else
			GameTooltip:AddDoubleLine(cooldownSpell, UNKNOWN, nil, nil, nil, 1, 1, 0)
		end
	end

	GameTooltip:Show()
end

function sRaidFrames_initialHeaderConfigFunction(frame)
	local layout = sRaidFrames:GetLayout()
	frame:SetAttribute("initial-width", layout.unitframeWidth)
	frame:SetAttribute("initial-height", layout.unitframeHeight)
	frame:SetAttribute("type1", "target")
	frame:SetAttribute("*type1", "target")
end

function sRaidFrames_InitUnitFrame(frame)
	sRaidFrames:CreateUnitFrame(frame)
end

local ShouldUpdateFrameCache = false
function sRaidFrames_OnAttributeChanged(frame, name, value)
	if name == "unit" then
		ShouldUpdateFrameCache = true

		if value then
			frame.id = select(3, value:find("(%d+)"))
		end
	end
end

function sRaidFrames:UpdateFrameCache()
	self.FramesByUnit = {}
	for k, frame in pairs(self.frames) do
		local unit = frame:GetAttribute("unit")
		if unit then
			if not self.FramesByUnit[unit] then
				self.FramesByUnit[unit] = {}
			end
			tinsert(self.FramesByUnit[unit], frame)
		end
	end
	ShouldUpdateFrameCache = false
	self:UpdateAllUnits()
end

local function BuffFrame_OnEnter(this)
	if sRaidFrames:QueryTooltipDisplay(sRaidFrames.opt.BuffTooltipMethod) then
		GameTooltip:SetOwner(this)
		GameTooltip:SetUnitBuff(this.unitid, this.buffid, this.showCastable)
	end
end

local function BuffFrame_OnLeave(this)
	GameTooltip:Hide()
end

local function DebuffFrame_OnEnter(this)
	if sRaidFrames:QueryTooltipDisplay(sRaidFrames.opt.DebuffTooltipMethod) then
		GameTooltip:SetOwner(this)
		GameTooltip:SetUnitDebuff(this.unitid, this.debuffid)
	end
end

local function DebuffFrame_OnLeave(this)
	GameTooltip:Hide()
end

local DebuffFrame_OnLeave = BuffFrame_OnLeave

local function UnitFrame_OnEnter(this)
	if sRaidFrames:QueryTooltipDisplay(sRaidFrames.opt.UnitTooltipMethod) then
		sRaidFrames:UnitTooltip(this)
	end
end

local UnitFrame_OnLeave = BuffFrame_OnLeave


-- Adapts frames created by Secure Headers
function sRaidFrames:CreateUnitFrame(f)
	local layout = self:GetLayout()
	f:EnableMouse(true)
	f:RegisterForClicks("AnyUp")

	f:SetScript("OnEnter", UnitFrame_OnEnter)
	f:SetScript("OnLeave", UnitFrame_OnLeave)

	f.title = f:CreateFontString(nil, "ARTWORK")
	f.title:SetFontObject(GameFontNormalSmall)
	f.title:SetJustifyH("LEFT")


	for i = 1, layout.debuffCount do
		local debuffFrame = CreateFrame("Button", nil, f)
		debuffFrame:SetScript("OnEnter", DebuffFrame_OnEnter);
		debuffFrame:SetScript("OnLeave", DebuffFrame_OnLeave)
		debuffFrame.texture = debuffFrame:CreateTexture(nil, "ARTWORK")
		debuffFrame.texture:SetAllPoints(debuffFrame);
		debuffFrame.count = debuffFrame:CreateFontString(nil, "OVERLAY")
		debuffFrame.count:SetFontObject(GameFontHighlightSmallOutline)
		debuffFrame.count:SetJustifyH("CENTER")
		debuffFrame.count:SetPoint("CENTER", debuffFrame, "CENTER", 0, 0);
		debuffFrame:Hide()
		f["aura"..i] = debuffFrame
	end

	for i = 1, layout.buffCount do
		local buffFrame = CreateFrame("Button", nil, f)
		buffFrame:SetScript("OnEnter", BuffFrame_OnEnter)
		buffFrame:SetScript("OnLeave", BuffFrame_OnLeave)
		buffFrame.texture = buffFrame:CreateTexture(nil, "ARTWORK")
		buffFrame.texture:SetAllPoints(buffFrame)
		buffFrame.count = buffFrame:CreateFontString(nil, "OVERLAY")
		buffFrame.count:SetFontObject(GameFontHighlightSmallOutline)
		buffFrame.count:SetJustifyH("CENTER")
		buffFrame.count:SetPoint("CENTER", buffFrame, "CENTER", 0, 0);
		buffFrame:Hide()
		f["buff"..i] = buffFrame
	end

	local texture = Media:Fetch("statusbar", self.opt.Texture)

	f.hpbar = CreateFrame("StatusBar", nil, f)
	f.hpbar:SetStatusBarTexture(texture)
	f.hpbar:SetMinMaxValues(0,100)
	f.hpbar:SetValue(0)

	f.hpbar.text = f.hpbar:CreateFontString(nil, "ARTWORK")
	f.hpbar.text:SetFontObject(GameFontHighlightSmall)
	f.hpbar.text:SetJustifyH("CENTER")

	local color = self.opt.HealthTextColor
	f.hpbar.text:SetTextColor(color.r, color.g, color.b, color.a)

	f.mpbar = CreateFrame("StatusBar", nil, f)
	f.mpbar:SetStatusBarTexture(texture)
	f.mpbar:SetMinMaxValues(0,100)
	f.mpbar:SetValue(0)

	f.statustext = f.mpbar:CreateFontString(nil, "ARTWORK")
	f.statustext:SetFontObject(GameFontHighlightSmall)
	f.statustext:SetJustifyH("CENTER")

	layout.StyleUnitFrame(f)

	f:Hide();

	tinsert(self.frames, f)
	ShouldUpdateFrameCache = true

	ClickCastFrames = ClickCastFrames or {}
	ClickCastFrames[f] = true
end

function sRaidFrames:CreateFrames()
	if not self.enabled then return end

	local neededGroups = #self:GetCurrentGroupSetup()
	local createdGroups = #self.groupframes
	for i = createdGroups+1, neededGroups do
		self:CreateGroupFrame(i)
	end
	self:SetPosition()
	self:SetGroupFilters()
	self:SetGrowth()
end

function sRaidFrames:SetGroupFilters()
	if InCombatLockdown() then return end

	for _, f in pairs(self.groupframes) do
		local id = f:GetID()
		local frame = self:GetCurrentGroupSetup()[id]
		if frame and not frame.hidden then
			for attribute, value in pairs(frame.attributes) do
				f.header:SetAttribute(attribute, value or nil)
			end
			if frame.MagicMaintankList then
				f.header:SetAttribute("nameList", table.concat(self.MainTanks, ","))
			end
			f.title:SetText(frame.caption)
			f.header:Show()
		else
			f.header:Hide()
			f.header:SetAttribute("groupFilter", false)
			f.header:SetAttribute("nameList", false)
		end
		self:UpdateTitleVisibility(f.header)
	end
end

function sRaidFrames:GroupFrameGetNumChildren(frame)
	local i = 1
	local child = frame:GetAttribute("child"..i)
	while child do
		if not child:IsVisible() or not UnitExists(child:GetAttribute("unit")) then
			break
		end
		i = i + 1
		child = frame:GetAttribute("child"..i)
	end
	return (i-1)
end

function sRaidFrames:UpdateTitleVisibility(frame)
	if self.opt.ShowGroupTitles and self:GroupFrameGetNumChildren(frame) > 0 then
		frame:GetParent().anchor:Show()
	else
		frame:GetParent().anchor:Hide()
	end
end

function sRaidFrames:SetGrowth()
	for _, f in pairs(self.groupframes) do
		f.header:ClearAllPoints();
		if self.opt.Growth == "up" then
			f.header:SetAttribute("point", "TOP")
			f.header:SetPoint("BOTTOM", f.anchor, "TOP")
		elseif self.opt.Growth == "right" then
			f.header:SetAttribute("point", "LEFT")
			f.header:SetPoint("LEFT", f.anchor, "RIGHT")
		elseif self.opt.Growth == "left" then
			f.header:SetAttribute("point", "RIGHT")
			f.header:SetPoint("RIGHT", f.anchor, "LEFT")
		elseif self.opt.Growth == "down" then
			f.header:SetAttribute("point", "BOTTOM")
			f.header:SetPoint("TOP", f.anchor, "BOTTOM")
		end
	end
	self:SetSpacing()
end

function sRaidFrames:SetSpacing()
	local s = sRaidFrames.opt.Spacing
	for _, f in pairs(self.groupframes) do
		if self.opt.Growth == "down" then
			f.header:SetAttribute("xOffset", 0)
			f.header:SetAttribute("yOffset", s)
		elseif self.opt.Growth == "up" then
			f.header:SetAttribute("xOffset", 0)
			f.header:SetAttribute("yOffset", -s)
		elseif self.opt.Growth == "left" then
			f.header:SetAttribute("xOffset", s)
			f.header:SetAttribute("yOffset", 0)
		elseif self.opt.Growth == "right" then
			f.header:SetAttribute("xOffset", -s)
			f.header:SetAttribute("yOffset", 0)
		end
	end
end

function sRaidFrames:StartMovingAll(f)
	this.multidrag = 1
	local id = f:GetID()
	local fg = self.groupframes[id]
	local x, y = fg:GetLeft(), fg:GetTop()
	if ( not x or not y ) then
		return
	end
	for k, f in pairs(self.groupframes) do
		if k ~= id then
			local oX, oY = f:GetLeft(), f:GetTop()
			if ( oX and oY ) then
				f:ClearAllPoints()
				f:SetPoint("TOPLEFT", fg, "TOPLEFT", oX-x, oY-y)
			end
		end
	end
end

function sRaidFrames:StopMovingOrSizingAll(f)
	this.multidrag = nil
	local id = this:GetID()
	local fg = self.groupframes[id]
	for k, f in pairs(self.groupframes) do
		if k ~= id then
			local oX, oY = f:GetLeft(), f:GetTop()
			if ( oX and oY ) then
				f:ClearAllPoints()
				f:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", oX, oY)
			end
		end
	end
end

local function sRaidFrames_OnGroupFrameEvent(frame, event)
	if event == "PARTY_MEMBERS_CHANGED" and frame:IsVisible() then
  	sRaidFrames:ScheduleLeaveCombatAction(sRaidFrames.UpdateTitleVisibility, sRaidFrames, frame)
	end
end

function sRaidFrames:CreateGroupFrame(id)
	local layout = self:GetLayout()
	local f = CreateFrame("Frame", "sRaidFramesGroupBase".. id, self.master)
	f:SetHeight(layout.headerHeight)
	f:SetWidth(layout.headerWidth)
	f:SetMovable(true)
	f:SetID(id)

	f.header = CreateFrame("Frame", "sRaidFramesGroupHeader" .. id, f, "SecureRaidGroupHeaderTemplate")
	f.header:SetAttribute("template", "sRaidFrameUnitSecure")
	f.header.initialConfigFunction = sRaidFrames_initialHeaderConfigFunction
	f.header:HookScript("OnEvent", sRaidFrames_OnGroupFrameEvent)

	-- hack to reduce login lockup times
	f.header:UnregisterEvent("UNIT_NAME_UPDATE")

	f.anchor = CreateFrame("Button", "sRaidFramesAnchor"..id, self.master)
	f.anchor:SetHeight(layout.headerHeight)
	f.anchor:SetWidth(layout.headerWidth)
	f.anchor:SetScript("OnDragStart", function() if self.opt.Locked then return end if IsAltKeyDown() then self:StartMovingAll(f) end f:StartMoving() end)
	f.anchor:SetScript("OnDragStop", function() if this.multidrag == 1 then self:StopMovingOrSizingAll() end f:StopMovingOrSizing(f) self:SavePosition() end)
	f.anchor:EnableMouse(true)
	f.anchor:RegisterForDrag("LeftButton")
	f.anchor:RegisterForClicks("AnyUp")
	f.anchor:SetPoint("CENTER", f)

	f.title = f.anchor:CreateFontString(nil, "OVERLAY", "GameFontNormal")
	f.title:SetJustifyH("CENTER")
	f.title:ClearAllPoints()
	f.title:SetPoint("CENTER", f.anchor)

	self.groupframes[id] = f
end

function sRaidFrames:SetWHP(frame, width, height, p1, relative, p2, x, y)
	frame:SetWidth(width)
	frame:SetHeight(height)

	if (p1) then
		frame:ClearAllPoints()
		frame:SetPoint(p1, relative, p2, x, y)
	end
end

function sRaidFrames:GetAllUnits()
	if ShouldUpdateFrameCache then self:UpdateFrameCache() end
	return self.FramesByUnit
end

function sRaidFrames:FindUnitFrames(unit)
	if ShouldUpdateFrameCache then self:UpdateFrameCache() end
	return self.FramesByUnit[unit] or {}
end

function sRaidFrames:IsTracking(unit)
	if ShouldUpdateFrameCache then self:UpdateFrameCache() end
	return (self.FramesByUnit[unit] ~= nil)
end

function sRaidFrames:SetPosition()
	if #self.opt.Positions[self.opt.GroupSetup] <= 0 then
		self:ResetPosition()
	else
		self:RestorePosition()
	end
end

function sRaidFrames:SavePosition()
	if not self.opt.Positions[self.opt.GroupSetup] then
		self.opt.Positions[self.opt.GroupSetup] = {}
	end

	for k,frame in pairs(self.groupframes) do
		local dbentry = {}
		local x, y, s = frame:GetLeft(), frame:GetTop(), frame:GetEffectiveScale()
		x, y = x * s, y  * s
		
		dbentry.x = x
		dbentry.y = y
		dbentry.s = s

		self.opt.Positions[self.opt.GroupSetup][k] = dbentry
	end
end

function sRaidFrames:RestorePosition()
	local aryPos = self.opt.Positions[self.opt.GroupSetup]
	local scale = self.master:GetEffectiveScale()

	for k, data in pairs(aryPos) do
		local groupframe = self.groupframes[k]
		if groupframe then
			local x, y, s = data.x, data.y, groupframe:GetEffectiveScale()
			x, y = x and x / s, y and y / s
			groupframe:ClearAllPoints()
			groupframe:SetPoint(x and "topleft" or "center", UIParent, y and "bottomleft" or "center", x or 0, y or 0)
		end
	end
end

function sRaidFrames:ResetPosition()
	self:PositionLayout("ctra", 200, -200)
end

function sRaidFrames:PositionLayout(layout, xBuffer, yBuffer)
	local xMod, yMod, i = 0, 0, -1
	local frameHeight = self:GetLayout().unitframeHeight+3+self.opt.Spacing
	local framePadding = MEMBERS_PER_RAID_GROUP

	for k,frame in pairs(self.groupframes) do
		i = i + 1
		if layout == "horizontal" then
			yMod = i * frame:GetWidth()
			xMod = 0
		elseif layout == "vertical" then
			if i ~= 0 and math.fmod(i, 2) == 0 then
				xMod = xMod + (-1*framePadding*frameHeight)
				yMod = 0
				i = 0
			else
				yMod = i * frame:GetWidth()
			end
		elseif layout == "ctra" then
			if i ~= 0 and math.fmod(i, 2) == 0 then
				yMod = yMod + frame:GetWidth()
				xMod = 0
				i = 0
			else
				xMod = i * (-1*framePadding*frameHeight)
			end
		end

		frame:ClearAllPoints()
		frame:SetPoint("TOPLEFT", UIParent, "TOPLEFT", xBuffer+yMod, yBuffer+xMod)
	end

	self:SavePosition()
end

local Tablet = AceLibrary("Tablet-2.0")
function sRaidFrames:OnTooltipUpdate()
	local cat = Tablet:AddCategory("columns", 2,
																 "text", L["Filter set"],
																 "justify", "LEFT",
																 "text2", L["State"],
																 "justify2", "LEFT",
																 "child_justify", "LEFT",
																 "child_justify2", "LEFT")
	local func, clicktext
	for name, _ in pairs(self:GetAllGroupFilters()) do
		if name == self.opt.GroupSetup then
			clicktext = L["|cff00ff00<Active>|r"]
			func = nil
		elseif InCombatLockdown() then
			clicktext = "|cffff0000<In Combat>|r"
			func = nil
		else
			clicktext = L["|cffffffff<Click to activate>|r"]
			func = self.SetCurrentGroupSetup
		end

		cat:AddLine("text", name,
							  "text2", clicktext,
							  "func", func,
							  "arg1", self,
							  "arg2", name)
	end
	--Tablet:SetHint(L["Right-click for options."])
end