--[[-------------------------------------------------------------------------
	Copyright (c) 2006, Threesf, Rupture
	All rights reserved.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
	A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
	OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
	SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
	LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
	DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
	THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
	OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
---------------------------------------------------------------------------]]

-- Locals
local _G = getfenv(0)
local pairs = pairs
local ipairs = ipairs

local formatgs = function(gstring, anchor)
	gstring = string.gsub(gstring,"([%^%(%)%.%[%]%*%+%-%?])","%%%1")
	gstring = string.gsub(gstring,"%%s","(.+)")
	gstring = string.gsub(gstring,"%%d","(%-?%%d+)")
	if anchor then gstring = "^"..gstring end
	return gstring
end

local classhex = function(class)
	return string.format("|cff%02x%02x%02x", RAID_CLASS_COLORS[class].r * 255, RAID_CLASS_COLORS[class].g * 255, RAID_CLASS_COLORS[class].b * 255)
end

local capitalize = function(text)
	local up = function(first, rest)
		return string.upper(first)..string.lower(rest)
	end
	text = string.gsub(string.lower(text), "(%l)([%w_']*)", up)
	return text
end

---------------------------------------------------------
-- Minimap button
local dragmini = function(self)
	if self.isMoving then
		local xpos, ypos = GetCursorPosition()
		local xmin, ymin = Minimap:GetLeft(), Minimap:GetBottom()
		xpos = xmin - xpos/Minimap:GetEffectiveScale() + 70
		ypos = ypos / Minimap:GetEffectiveScale() - ymin - 70
		local angle = math.deg(math.atan2(ypos, xpos))
		xpos = 80 * cos(angle)
		ypos = 80 * sin(angle)
		self:SetPoint("TOPLEFT", "Minimap", "TOPLEFT", 52-xpos, ypos-52)
	end
end

local button = CreateFrame("Button", "Rupture_Minimap", Minimap)
button:SetWidth(33)
button:SetHeight(33)
button:SetFrameStrata("LOW")
button:SetPoint("CENTER", -12, -80)
button:RegisterForClicks("AnyUp")
button:RegisterForDrag("LeftButton")
button:SetMovable(true)
button:SetScript("OnClick", function(self, arg1)
		if arg1 == "RightButton" then
			Rupture_Options:Hide()
			if Rupture_Bindings:IsShown() then
				Rupture_Bindings:Hide()
			else
				Rupture_Bindings:Show()
			end
		else
			Rupture_Bindings:Hide()
			if Rupture_Options:IsShown() then
				Rupture_Options:Hide()
			else
				Rupture_Options:Show()
			end
		end
	end)
button:SetScript("OnEnter", function(self)
		local version = GetAddOnMetadata("Rupture", "Version")
		GameTooltip:SetOwner(self, "ANCHOR_LEFT")
		GameTooltip:AddLine("|cffff0000Rupture vers.|r "..version, 1, 1, 1)
		GameTooltip:AddLine("|cff00ffffLeft Click:|r Options Menu", 1, 1, 1)
		GameTooltip:AddLine("|cff00ffffRight Click:|r Bindings Menu", 1, 1, 1)
		GameTooltip:AddLine("|cff00ffffShift Drag:|r Moves Frames", 1, 1, 1)
		GameTooltip:Show()
	end)	
button:SetScript("OnLeave", function()
		GameTooltip:Hide()
	end)
button:SetScript("OnDragStart", function(self)
		self.isMoving = true
		self:SetScript("OnUpdate", function(self) dragmini(self) end)
	end)
button:SetScript("OnDragStop", function(self)
		self.isMoving = nil
		self:SetScript("OnUpdate", nil)
		self:SetUserPlaced(true)
	end)

local texture = button:CreateTexture(nil, "ARTWORK")
texture:SetWidth(52)
texture:SetHeight(52)
texture:SetTexture("Interface\\Minimap\\MiniMap-TrackingBorder")
texture:SetPoint("TOPLEFT")

texture = button:CreateTexture(nil, "ARTWORK")
texture:SetWidth(33)
texture:SetHeight(33)
texture:SetTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight")
texture:SetBlendMode("ADD")
texture:SetPoint("TOPLEFT")
button:SetHighlightTexture(texture)

texture = button:CreateTexture(nil, "BACKGROUND")
texture:SetWidth(23)
texture:SetHeight(23)
texture:SetTexture("Interface\\Addons\\Rupture\\logo")
texture:SetPoint("CENTER", -1, 1)

---------------------------------------------------------
-- Cooldowns
local cooldowntext = function(self)
	if self.cool then
		local remain = floor((self.start + self.duration) - GetTime())
		if remain >= 0 then
			if self.aura and not RUPTURE_TOURNEY_SAVED.aura then
				self.text:SetText(nil)
				self.cool = nil
				return
			end
			local scale = self:GetScale()
			local height = floor(22 / scale)
			local fheight = select(2, self.text:GetFont())
			local r, g, b = 1, 0, 0
			if remain >= 3600 then
				remain = math.floor(remain / 3600).."h"
				height = floor(14 / scale)
				r, g, b = 0.6, 0.6, 0.6
			elseif remain >= 60 then
				local min = math.floor(remain / 60)
				if self.aura then
					remain = min
				elseif min < 10 then
					local sec = math.floor(math.fmod(remain, 60))
					remain = string.format("%2d:%02s", min, sec)
				else
					remain = min.."m"
				end
				height = floor(14 / scale)
				r, g, b = 1, 1, 0
			end
			self.text:SetText(remain)
			self.text:SetTextColor(r, g, b)
			if not self.aura and (height ~= fheight) then
				self.text:SetFont("Fonts\\FRIZQT__.TTF", height, "OUTLINE")
			end
		else
			self.text:SetText(nil)
			self.cool = nil
		end
	end
end

local cooldowns = function(this, start, duration, enable)
	if not this.text then
		this.text = this:CreateFontString(nil, "ARTWORK")
		this.text:SetPoint("CENTER")
		this.text:SetFont("Fonts\\FRIZQT__.TTF", (ceil(this:GetWidth() * 0.6) or 14), "OUTLINE")
		this:SetScript("OnUpdate", function(self) cooldowntext(self) end)
		this.aura = this:GetParent().isaura
	end

	if not this.text:GetFont() then
		return
	end

	if start > 2 and duration > 2 then
		this.cool = true
		this.start = start
		this.duration = duration
	else
		this.cool = nil
		this.text:SetText(nil)
	end
end

hooksecurefunc("CooldownFrame_SetTimer", cooldowns)

---------------------------------------------------------
-- UnitFrames
local getinfo = function(unit, type)
	local curr, max, perc, r, g, b
	curr = _G["Unit"..type](unit)
	max = _G["Unit"..type.."Max"](unit)
	perc = curr / max

	if type == "Health" then
		if perc > 0.5 then
			r = (1.0 - perc) * 2
			g = 1.0
		else
			r = 1.0
			g = perc * 2
		end
		b = 0.0
	end
	perc = format("%.0f", floor(perc * 100))
	return curr, max, perc, r, g, b
end

local castshow = function(bar)
	bar:ClearAllPoints()
	if UnitBuff(bar.unit, 1) then
		bar:SetPoint("TOP", bar:GetParent(), "BOTTOM", 10, -18)
	else
		bar:SetPoint("TOP", bar:GetParent(), "BOTTOM", 10, -1)
	end
end

local castevent = function()
	local newevent = event
	local newarg1 = arg1
	if event == "PLAYER_TARGET_CHANGED" then
		local nameChannel  = UnitChannelInfo(this.unit)
		local nameSpell  = UnitCastingInfo(this.unit)
		if nameChannel then
			newevent = "UNIT_SPELLCAST_CHANNEL_START"
			newarg1 = "target"
		elseif nameSpell then
			newevent = "UNIT_SPELLCAST_START"
			newarg1 = "target"
		else
			this.casting = nil
			this.channeling = nil
			this:SetMinMaxValues(0, 0)
			this:SetValue(0)
			this:Hide()
			return
		end
	end
	CastingBarFrame_OnEvent(newevent, newarg1)
end

local updateauras = function(self)
	local button, count, name, rank, texture, app, cure, border, color, duration, left
	for i = 1, 10 do
		button = _G[self:GetName().."buff"..i]
		if not button then
			break
		end
		name, rank, texture, app, duration, left = UnitBuff(self.unit, i)
		if name then
			_G[button:GetName().."Icon"]:SetTexture(texture)
			count = _G[button:GetName().."Count"]
			if app > 1 then
				count:SetText(app)
				count:Show()
			else
				count:Hide()
			end
			if duration and duration > 0 then
				CooldownFrame_SetTimer(_G[button:GetName().."Cooldown"], GetTime()-(duration-left), duration, 1)
			else
				_G[button:GetName().."Cooldown"]:Hide()
			end
			button:Show()
		else
			button:Hide()
		end
	end

	for i = 1, 6 do
		button = _G[self:GetName().."debuff"..i]
		if not button then
			break
		end
		name, rank, texture, app, cure, duration, left = UnitDebuff(self.unit, i)
		if name then
			_G[button:GetName().."Icon"]:SetTexture(texture)
			count = _G[button:GetName().."Count"]
			border = _G[button:GetName().."Border"]
			if app > 1 then
				count:SetText(app)
				count:Show()
			else
				count:Hide()
			end

			if cure then
				color = DebuffTypeColor[cure]
			else
				color = DebuffTypeColor["none"]
			end
			if duration and duration > 0 then
				CooldownFrame_SetTimer(_G[button:GetName().."Cooldown"], GetTime()-(duration-left), duration, 1)
			else
				_G[button:GetName().."Cooldown"]:Hide()
			end
			border:SetVertexColor(color.r, color.g, color.b)
			button:Show()
		else
			button:Hide()
		end
	end

	local cast = _G[self:GetName().."SpellBar"]
	if cast then
		castshow(cast)
	end
end

local updatename = function(self)
	local _, eclass = UnitClass(self.unit)
	local color = RAID_CLASS_COLORS[eclass] or { r = 1, g = 1, b = 1 }
	local id, found = gsub(self.unit, "party([1-4])", "%1")
	local name = UnitName(self.unit)
	if found == 1 and string.len(self.unit) == 6 then
		local binding = GetBindingText(GetBindingKey("TARGETPARTYMEMBER"..id), "KEY_")
		name = "|cffffffff["..binding.."] |r"..name
	end
	self.name:SetText(name)
	self.name:SetTextColor(color.r, color.g, color.b)
end

local updateframe = function(self, notext)
	local curr, max, perc, r, g, b = getinfo(self.unit, "Health")
	self.hp:SetStatusBarColor(r, g, b)
	self.hp:SetMinMaxValues(0, max)
	self.hp:SetValue(curr)
	self.hpperc:SetText(perc.."%")
	self.hpperc:SetTextColor(r, g, b)
	if self.hptext then
		self.hptext:SetText(curr.."/"..max)
		local miss = curr - max
		if miss < 0 then
			self.hpmiss:SetText(miss)
		else
			self.hpmiss:SetText(nil)
		end
	end

	if self.mp then
		curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
		local color = ManaBarColor[UnitPowerType(self.unit)]
		self.mp:SetStatusBarColor(color.r, color.g, color.b)
		self.mpbg:SetVertexColor(color.r, color.g, color.b, 0.25)
	end

	if not notext then
		updatename(self)

		if UnitIsEnemy(self.unit, "player")  then
			self:SetBackdropBorderColor(1, 0, 0)
		elseif UnitIsUnit(self.unit, "player") or UnitIsUnit(self.unit, "pet") then
			self:SetBackdropBorderColor(1, 1, 0)
		elseif UnitInParty(self.unit) then
			self:SetBackdropBorderColor(0, 1, 0)
		else
			self:SetBackdropBorderColor(1, 1, 1)
		end
	end
	updateauras(self)
end

local totupdate = function(self, e)
	local unit, notext
	unit = SecureButton_GetUnit(self)
	if UnitExists(unit) then
		notext = true
		if not self.lastname or self.lastname ~= UnitName(unit) then
			self.lastname = UnitName(unit)
			notext = nil
		end
		updateframe(self, notext)
	else
		self.lastname = nil
	end
end

local events = {
	["PLAYER_ENTERING_WORLD"] = function(self, ...)
		updateframe(self)
		local unit = SecureButton_GetUnit(self)
		local id, found = gsub(unit, "party([1-4])", "%1")
		if unit == "player" then
			if IsPartyLeader() then
				self.leader:Show()
			else
				self.leader:Hide()
			end
		elseif found == 1 then
			if GetPartyLeaderIndex() == tonumber(id) then
				self.leader:Show()
			else
				self.leader:Hide()
			end
		end
	end,
	
	["PLAYER_TARGET_CHANGED"] = function(self, ...)
		if self.unit == "target" then
			updateframe(self)
		else
			if UnitName("target") == UnitName(self.unit) then
				self:LockHighlight()
			else
				self:UnlockHighlight()
			end
		end
	end,

	["PLAYER_FOCUS_CHANGED"] = function(self, ...)
		updateframe(self)
	end,

	["PARTY_LEADER_CHANGED"] = function(self)
		local unit = SecureButton_GetUnit(self)
		local id, found = gsub(unit, "party([1-4])", "%1")
		if unit == "player" then
			if IsPartyLeader() then
				self.leader:Show()
			else
				self.leader:Hide()
			end
		elseif found == 1 then
			if GetPartyLeaderIndex() == tonumber(id) then
				self.leader:Show()
			else
				self.leader:Hide()
			end
		end
	end,

	["PARTY_MEMBERS_CHANGED"] = function(self, ...)
		updateframe(self)
	end,
	
	["UNIT_HEALTH"] = function(self, ...)
		local curr, max, perc, r, g, b = getinfo(self.unit, "Health")
		self.hp:SetStatusBarColor(r, g, b)
		self.hp:SetValue(curr)
		self.hptext:SetText(curr.."/"..max)
		local miss = curr - max
		if miss < 0 then
			self.hpmiss:SetText(miss)
		else
			self.hpmiss:SetText(nil)
		end
		self.hpperc:SetText(perc.."%")
		self.hpperc:SetTextColor(r, g, b)
	end,
	
	["UNIT_MAXHEALTH"] = function(self, ...)
		local curr, max, perc, r, g, b = getinfo(self.unit, "Health")
		self.hp:SetStatusBarColor(r, g, b)
		self.hp:SetMinMaxValues(0, max)
		self.hp:SetValue(curr)
		self.hptext:SetText(curr.."/"..max)
		self.hpperc:SetText(perc.."%")
		self.hpperc:SetTextColor(r, g, b)
	end,
	
	["UNIT_MANA"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_MAXMANA"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_ENERGY"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_MAXENERGY"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_RAGE"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_MAXRAGE"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,

	["UNIT_FOCUS"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_MAXFOCUS"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
	end,
	
	["UNIT_DISPLAYPOWER"] = function(self, ...)
		local curr, max, perc = getinfo(self.unit, "Mana")
		self.mp:SetMinMaxValues(0, max)
		self.mp:SetValue(curr)
		self.mptext:SetText(curr.."/"..max)
		local color = ManaBarColor[UnitPowerType(self.unit)]
		self.mp:SetStatusBarColor(color.r, color.g, color.b)
		self.mpbg:SetVertexColor(color.r, color.g, color.b, 0.25)
	end,
		
	["UNIT_NAME_UPDATE"] = function(self, ...)
		updatename(self)
	end,

	["UPDATE_BINDINGS"] = function(self)
		updatename(self)
	end,

	["UNIT_AURA"] = function(self, ...)
		updateauras(self)
	end,
}

local onevent = function(self, event, ...)
	if UnitExists(self.unit) then
		if string.find(event, "^UNIT_") and arg1 ~= self.unit then
			return
		end
		events[event](self, ...)
	end
end

local units = {
	["player"] = {
		"PLAYER_ENTERING_WORLD",
		"PARTY_LEADER_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
	},
	["target"] = {
		"PLAYER_ENTERING_WORLD",
		"PLAYER_TARGET_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
	},
	["pet"] = {
		"PLAYER_ENTERING_WORLD",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
	},
	["focus"] = {
		"PLAYER_ENTERING_WORLD",
		"PLAYER_FOCUS_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
	},
	["party1"] = {
		"PLAYER_ENTERING_WORLD",
		"PLAYER_TARGET_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"PARTY_LEADER_CHANGED",
		"PARTY_MEMBERS_CHANGED",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
		"UPDATE_BINDINGS",
	},
	["party2"] = {
		"PLAYER_ENTERING_WORLD",
		"PLAYER_TARGET_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"PARTY_LEADER_CHANGED",
		"PARTY_MEMBERS_CHANGED",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
		"UPDATE_BINDINGS",
	},
	["party3"] = {
		"PLAYER_ENTERING_WORLD",
		"PLAYER_TARGET_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"PARTY_LEADER_CHANGED",
		"PARTY_MEMBERS_CHANGED",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
		"UPDATE_BINDINGS",
	},
	["party4"] = {
		"PLAYER_ENTERING_WORLD",
		"PLAYER_TARGET_CHANGED",
		"UNIT_HEALTH",
		"UNIT_MAXHEALTH",
		"UNIT_MANA",
		"UNIT_MAXMANA",
		"UNIT_ENERGY",
		"UNIT_MAXENERGY",
		"UNIT_RAGE",
		"UNIT_MAXRAGE",
		"UNIT_FOCUS",
		"UNIT_MAXFOCUS",
		"UNIT_DISPLAYPOWER",
		"PARTY_LEADER_CHANGED",
		"PARTY_MEMBERS_CHANGED",
		"UNIT_NAME_UPDATE",
		"UNIT_AURA",
		"UPDATE_BINDINGS",
	},
}

-- Create frames
for unit, events in pairs(units) do
	local frame = CreateFrame("Button", "Rupture_"..unit, UIParent, "SecureUnitButtonTemplate")
	frame.unit = unit
	frame:SetWidth(180)
	frame:SetHeight(50)
	frame:RegisterForClicks("AnyUp")
	frame:RegisterForDrag("LeftButton")
	frame:SetMovable(true)
	frame:EnableMouse(true)
	frame:EnableMouseWheel(true)
	frame:SetFrameStrata("LOW")
	frame:SetClampedToScreen(true)
	frame:SetBackdrop({
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
		edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
		tile = true,
		tileSize = 16,
		edgeSize = 16,
		insets = { left = 5, right = 5, top = 5, bottom = 5 },
	})
	frame:SetBackdropColor(0, 0, 0, 0.75)

	local dropdown, menufunc
	local id, found = string.gsub(unit, "party([1-4])", "%1")
	if found == 1 then
		dropdown = _G["PartyMemberFrame"..id.."DropDown"]
		local highlight = frame:CreateTexture(nil, "ARTWORK")
		highlight:SetWidth(130)
		highlight:SetHeight(14)
		highlight:SetPoint("TOPRIGHT", -3, -3)
		highlight:SetTexture("Interface\\QuestFrame\\UI-QuestTitleHighlight")
		highlight:SetBlendMode("ADD")
		highlight:Hide()
		frame:SetHighlightTexture(highlight)
	else
		dropdown = _G[string.gsub(unit, "^%l", string.upper).."FrameDropDown"]
	end

	if dropdown then
		menufunc = function() ToggleDropDownMenu(1, nil, dropdown, "cursor") end
	end
	SecureUnitButton_OnLoad(frame, frame.unit, menufunc)
	RegisterUnitWatch(frame)
	
	local child = CreateFrame("PlayerModel", "Rupture_"..unit.."Model", frame)
	child:SetWidth(40)
	child:SetHeight(40)
	child:SetModelScale(1.9)
	child:SetFrameStrata("LOW")
	child:SetFrameLevel(1)
	child:SetPoint("LEFT", 5, 0)
	child:RegisterEvent("PLAYER_ENTERING_WORLD")
	child:RegisterEvent("DISPLAY_SIZE_CHANGED")
	child:RegisterEvent("UNIT_MODEL_CHANGED")
	if unit == "target" then
		child:RegisterEvent("PLAYER_TARGET_CHANGED")
	elseif unit == "pet" then
		child:RegisterEvent("UNI_PET")
	elseif unit == "focus" then
		child:RegisterEvent("PLAYER_FOCUS_CHANGED")
	elseif found == 1 then
		child:RegisterEvent("PARTY_MEMBERS_CHANGED")
	end
	child.unit = unit
	child:SetScript("OnEvent", function(self, event)
			if event == "DISPLAY_SIZE_CHANGED" then
				self:RefreshUnit()
			else
				self:SetUnit(self.unit)
			end
		end)
	child:SetScript("OnUpdate", function(self) self:SetCamera(0) end)
	
	frame.hp = CreateFrame("StatusBar", "Rupture_"..unit.."HP", frame)
	frame.hp:SetWidth(130)
	frame.hp:SetHeight(15)
	frame.hp:SetPoint("TOPRIGHT", -5, -17)
	frame.hp:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")

	local bg = frame.hp:CreateTexture(nil, "BACKGROUND")
	bg:SetAllPoints(frame.hp)
	bg:SetTexture("Interface\\TargetingFrame\\UI-StatusBar")
	bg:SetVertexColor(1, 0, 0, 0.25)

	frame.hptext = frame.hp:CreateFontString(nil, "ARTWORK")
	frame.hptext:SetAllPoints(frame.hp)
	frame.hptext:SetJustifyH("LEFT")
	frame.hptext:SetFontObject("GameFontNormalSmall")
	frame.hptext:SetTextColor(1, 1, 1)

	frame.hpmiss = frame.hp:CreateFontString(nil, "ARTWORK")
	frame.hpmiss:SetAllPoints(frame.hp)
	frame.hpmiss:SetJustifyH("RIGHT")
	frame.hpmiss:SetFontObject("GameFontNormalSmall")
	frame.hpmiss:SetTextColor(1, 0.25, 0)

	frame.hpperc = frame:CreateFontString(nil, "ARTWORK")
	frame.hpperc:SetWidth(125)
	frame.hpperc:SetJustifyH("RIGHT")
	frame.hpperc:SetFontObject("GameFontNormal")
	frame.hpperc:SetPoint("TOPRIGHT", -5, -5)

	frame.mp = CreateFrame("StatusBar", "Rupture_"..unit.."MP", frame)
	frame.mp:SetWidth(130)
	frame.mp:SetHeight(13)
	frame.mp:SetPoint("TOPRIGHT", _G["Rupture_"..unit.."HP"], "BOTTOMRIGHT", 0, -1)
	frame.mp:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
	frame.mptext = frame.mp:CreateFontString(nil, "ARTWORK")
	frame.mptext:SetAllPoints(frame.mp)
	frame.mptext:SetJustifyH("LEFT")
	frame.mptext:SetFontObject("GameFontNormalSmall")
	frame.mptext:SetTextColor(1, 1, 1)

	frame.mpbg = frame.mp:CreateTexture(nil, "BACKGROUND")
	frame.mpbg:SetAllPoints(frame.mp)
	frame.mpbg:SetTexture("Interface\\TargetingFrame\\UI-StatusBar")
	frame.mpbg:SetVertexColor(1, 0, 0, 0.25)
	
	frame.name = frame:CreateFontString(nil, "ARTWORK")
	frame.name:SetWidth(100)
	frame.name:SetHeight(12)
	frame.name:SetJustifyH("LEFT")
	frame.name:SetFontObject("GameFontNormal")
	frame.name:SetPoint("TOPLEFT", 46, -4)

	frame.leader = frame:CreateTexture(nil, "OVERLAY")
	frame.leader:SetWidth(13)
	frame.leader:SetHeight(13)
	frame.leader:SetTexture("Interface\\GroupFrame\\UI-Group-LeaderIcon")
	frame.leader:SetPoint("TOPLEFT", 3, -2)
	frame.leader:Hide()

	if not frame:IsUserPlaced() then
		frame:SetPoint("CENTER")
	end
	frame:SetScript("OnEnter", UnitFrame_OnEnter)
	frame:SetScript("OnLeave", UnitFrame_OnLeave)
	frame:SetScript("OnEvent", function(self, event, ...) onevent(self, event, ...) end)
	frame:SetScript("OnDragStart", function(self) if IsShiftKeyDown() then self:StartMoving() end end)
	frame:SetScript("OnDragStop", function(self) self:StopMovingOrSizing() end)
	
	for _, event in ipairs(events) do
		frame:RegisterEvent(event)
	end

	local aura
	for i = 1, 10 do
		aura = CreateFrame("Button", "Rupture_"..unit.."buff"..i, frame, "TargetBuffButtonTemplate")
		aura.id = i
		aura.unit = unit
		aura.isaura = true
		aura:SetWidth(17)
		aura:SetHeight(17)
		aura:Hide()
		if i == 1 then
			aura:SetPoint("TOPLEFT", frame, "BOTTOMLEFT", 5, 2)
		else
			aura:SetPoint("LEFT", _G["Rupture_"..unit.."buff"..(i - 1)], "RIGHT", 0, 0)
		end

		aura:SetScript("OnEnter", function(self)
					GameTooltip:SetOwner(self, "ANCHOR_BOTTOMRIGHT", 0, 0);
					GameTooltip:SetUnitBuff(self.unit, self.id);
				end)
	end

	for i = 1, 6 do
		aura = CreateFrame("Button", "Rupture_"..unit.."debuff"..i, frame, "TargetDebuffButtonTemplate")
		aura.id = i
		aura.unit = unit
		aura.isaura = true
		aura:SetWidth(18)
		aura:SetHeight(18)
		aura:Hide()
		_G["Rupture_"..unit.."debuff"..i.."Border"]:SetAllPoints(aura)
		if i == 1 then
			aura:SetPoint("TOPLEFT", frame, "TOPRIGHT", -2, -6)
		else
			if math.fmod(2, i) == 0 then
				aura:SetPoint("TOP", _G["Rupture_"..unit.."debuff"..(i - 1)], "BOTTOM", 0, -1)
			else
				aura:SetPoint("LEFT", _G["Rupture_"..unit.."debuff"..(i - 2)], "RIGHT", 0, 0)
			end
		end

		aura:SetScript("OnEnter", function(self)
					GameTooltip:SetOwner(self, "ANCHOR_BOTTOMRIGHT", 0, 0);
					GameTooltip:SetUnitDebuff(self.unit, self.id);
				end)
	end

	-- Add tot frames
	if unit ~= "player" then
		local target = CreateFrame("Button", "Rupture_"..unit.."target", frame, "SecureUnitButtonTemplate")
		target.unit = unit.."target"
		target:SetWidth(129)
		target:SetHeight(19)
		target:RegisterForClicks("AnyUp")
		target:EnableMouse(true)
		target:SetClampedToScreen(true)
		target:SetPoint("BOTTOMLEFT", frame, "TOPLEFT", 3, -4)
		target:SetBackdrop({
			bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
			edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
			tile = true,
			tileSize = 14,
			edgeSize = 14,
			insets = { left = 5, right = 5, top = 5, bottom = 5 },
		})
		target:SetBackdropColor(0, 0, 0, 0.75)
		SecureUnitButton_OnLoad(target, target.unit, nil)
		RegisterUnitWatch(target)

		target.hp = CreateFrame("StatusBar", "Rupture_"..unit.."targetHP", target)
		target.hp:SetWidth(121)
		target.hp:SetHeight(12)
		target.hp:SetPoint("RIGHT", -4, 0)
		target.hp:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
		target.hpperc = target.hp:CreateFontString(nil, "ARTWORK")
		target.hpperc:SetAllPoints(target.hp)
		target.hpperc:SetJustifyH("RIGHT")
		target.hpperc:SetFontObject("GameFontNormalSmall")
		target.hpperc:SetTextColor(1, 1, 1)

		target.name = target.hp:CreateFontString(nil, "ARTWORK")
		target.name:SetWidth(100)
		target.name:SetHeight(12)
		target.name:SetJustifyH("LEFT")
		target.name:SetFontObject("GameFontNormalSmall")
		target.name:SetPoint("LEFT", 0, 0)

		target.update = 0
		target:SetScript("OnUpdate", function(self, arg1) totupdate(self, arg1) end)

		for i = 1, 3 do
			aura = CreateFrame("Button", "Rupture_"..unit.."targetdebuff"..i, target, "TargetDebuffButtonTemplate")
			aura.id = i
			aura.isaura = true
			aura.unit = unit.."target"
			aura:SetWidth(15)
			aura:SetHeight(15)
			aura:Hide()
			_G["Rupture_"..unit.."targetdebuff"..i.."Border"]:SetAllPoints(aura)
			if i == 1 then
				aura:SetPoint("LEFT", target, "RIGHT", -1, 0)
			else
				aura:SetPoint("LEFT", _G["Rupture_"..unit.."targetdebuff"..(i - 1)], "RIGHT", 0, 0)
			end

			aura:SetScript("OnEnter", function(self)
						GameTooltip:SetOwner(self, "ANCHOR_BOTTOMRIGHT", 0, 0);
						GameTooltip:SetUnitDebuff(self.unit, self.id);
					end)
		end
	end

	-- Add casting bars
	if unit == "target" or unit == "focus" then
		local casting = CreateFrame("StatusBar", "Rupture_"..unit.."SpellBar", frame, "CastingBarFrameTemplate")
		casting:Hide()
		casting:SetWidth(150)
		casting:SetHeight(14)
		if unit == "target" then
			casting:RegisterEvent("PLAYER_TARGET_CHANGED")
		else
			casting:RegisterEvent("PLAYER_FOCUS_CHANGED")
		end

		this = casting
		CastingBarFrame_OnLoad(unit, false)
		_G["Rupture_"..unit.."SpellBarIcon"]:Show()

		local frameText = _G[this:GetName().."Text"]
		if frameText then
			frameText:SetTextHeight(10)
			frameText:ClearAllPoints()
			frameText:SetPoint("CENTER")
		end

		local frameBorder = _G[this:GetName().."Border"]
		if frameBorder then
			frameBorder:SetTexture("Interface\\CastingBar\\UI-CastingBar-Border-Small")
			frameBorder:SetWidth(197)
			frameBorder:SetHeight(60)
			frameBorder:ClearAllPoints()
			frameBorder:SetPoint("CENTER")
		end

		local frameFlash = _G[this:GetName().."Flash"]
		if frameFlash then
			frameFlash:SetTexture("Interface\\CastingBar\\UI-CastingBar-Flash-Small")
			frameFlash:SetWidth(197)
			frameFlash:SetHeight(49)
			frameFlash:ClearAllPoints()
			frameFlash:SetPoint("TOP", casting, "TOP", 0, 20)
		end

		casting:SetScript("OnShow", castshow)
		casting:SetScript("OnEvent", castevent)
	end
end

-- Create combo bar
local combo = CreateFrame("Frame", "Rupture_Combo")
combo:SetWidth(256)
combo:SetHeight(32)
combo:RegisterForDrag("LeftButton")
combo:SetMovable(true)
combo:EnableMouse(true)
combo:EnableMouseWheel(true)
combo:SetFrameStrata("LOW")
combo:SetClampedToScreen(true)
combo:RegisterEvent("PLAYER_LOGIN")
combo:RegisterEvent("PLAYER_COMBO_POINTS")
combo:RegisterEvent("PLAYER_TARGET_CHANGED")

if not combo:IsUserPlaced() then
	combo:SetPoint("CENTER")
end

combo.bg = combo:CreateTexture(nil, "BACKGROUND")
combo.bg:SetTexture("Interface\\Addons\\Rupture_Tourney\\combobg")
combo.bg:SetAllPoints(combo)
for i = 1, 4 do
	combo["combo"..i] = combo:CreateTexture(nil, "ARTWORK")
	combo["combo"..i]:SetWidth(13)
	combo["combo"..i]:SetHeight(13)
	combo["combo"..i]:SetTexture("Interface\\Addons\\Rupture_Tourney\\combo")
	combo["combo"..i]:Hide()
	if i == 1 then
		combo["combo"..i]:SetPoint("LEFT", 73, 2)
	elseif i == 4 then
		combo["combo"..i]:SetPoint("LEFT", combo["combo"..(i-1)], "RIGHT", 10, 0)
	else
		combo["combo"..i]:SetPoint("LEFT", combo["combo"..(i-1)], "RIGHT", 11, 0)
	end
end
combo.combo5 = combo:CreateTexture(nil, "ARTWORK")
combo.combo5:SetWidth(17)
combo.combo5:SetHeight(17)
combo.combo5:SetTexture("Interface\\Addons\\Rupture_Tourney\\combo")
combo.combo5:SetPoint("LEFT", combo.combo4, "RIGHT", 12, 0)
combo.combo5:Hide()
combo.text = combo:CreateFontString(nil, "ARTWORK")
combo.text:SetPoint("TOP")
combo.text:SetFont("Fonts\\FRIZQT__.TTF", 24, "OUTLINE")
combo:SetScript("OnDragStart", function(self) if IsShiftKeyDown() then self:StartMoving() end end)
combo:SetScript("OnDragStop", function(self) self:StopMovingOrSizing() end)
combo:SetScript("OnEvent", function(self, event)
		if event == "PLAYER_LOGIN" then
			local class = select(2, UnitClass("player"))
			if class ~= "ROGUE" and class ~= "DRUID" then
				self:Hide()
			end
		else
			local comboPoints = GetComboPoints()
			combo.text:SetText(comboPoints)
			if comboPoints == 5 then
				combo.text:SetTextColor(1, 0, 0)
			elseif comboPoints > 2 then
				combo.text:SetTextColor(1, 1, 0)
			else
				combo.text:SetTextColor(0, 1, 0)
			end
			
			if RUPTURE_TOURNEY_SAVED.combonum then
				return
			end
			
			for i = 1, 5 do
				local point = combo["combo"..i]
				if comboPoints >= i then
					point:Show()
				else
					point:Hide()
				end
			end
		end
	end)

-- Clear temp tables
units = nil

-- Hide default frames
PlayerFrame:UnregisterAllEvents()
PlayerFrame:Hide()
TargetFrame:UnregisterAllEvents()
TargetFrame:Hide()
ComboFrame:UnregisterAllEvents()
ComboFrame:Hide()
PetFrame:UnregisterAllEvents()
PetFrame:Hide()
for i = 1, 4 do
	local party = _G["PartyMemberFrame"..i]
	party:UnregisterAllEvents()
	party:Hide()
end
ShowPartyFrame = function() end

---------------------------------------------------------
-- Spell Alert
local casting = formatgs(SPELLCASTOTHERSTART)
local totem = formatgs(SPELLCASTGOOTHER)
local buff = formatgs(AURAADDEDOTHERHELPFUL)

local istotem = "Totem"

if GetLocale()=="koKR" then
	istotem = "토템"
elseif GetLocale()=="zhCN" then
	istotem = "图腾"
elseif GetLocale()=="zhTW" then
	istotem = "圖騰"
elseif GetLocale()=="esES" then
	istotem = "Tótem"
end

local inparty = function(unit)
	for i = 1, GetNumPartyMembers() do
		local name = UnitName("party"..i)
		if name == unit then
			return true
		end
	end
end

local spellevent = function()
	local unit, spell
	if event == "CHAT_MSG_SPELL_HOSTILEPLAYER_DAMAGE" then
		for unit, spell in string.gmatch(arg1, casting) do
			if not inparty(unit) then
				Rupture_SpellAlert:AddMessage(format(SPELLCASTOTHERSTART, unit, "|cffff0000"..spell.."|r"))
			end
		end
	elseif event == "CHAT_MSG_SPELL_HOSTILEPLAYER_BUFF" then
		for unit, spell in string.gmatch(arg1, casting) do
			if not inparty(unit) then
				Rupture_SpellAlert:AddMessage(format(SPELLCASTOTHERSTART, unit, "|cffffff00"..spell.."|r"))
			end
			return
		end

		for unit, spell in string.gmatch(arg1, totem) do
			if not inparty(unit) and string.find(spell, istotem) then
				Rupture_SpellAlert:AddMessage(format(SPELLCASTGOOTHER, unit, "|cff999999"..spell.."|r"))
			end
			return
		end

		for unit, spell in string.gmatch(arg1, buff) do
			if string.find(spell, "^%a") then
				Rupture_BuffAlert:AddMessage(format(AURAADDEDOTHERHELPFUL, unit, "|cff00ff00"..spell.."|r"))
			end
			return
		end
	elseif event == "CHAT_MSG_SPELL_PERIODIC_HOSTILEPLAYER_BUFFS" then
		for unit, spell in string.gmatch(arg1, buff) do
			if string.find(spell, "^%a") then
				Rupture_BuffAlert:AddMessage(format(AURAADDEDOTHERHELPFUL, unit, "|cff00ff00"..spell.."|r"))
			end
		end
	end
end

CreateFrame("MessageFrame", "Rupture_SpellAlert", UIParent)
Rupture_SpellAlert:SetWidth(UIParent:GetWidth())
Rupture_SpellAlert:SetHeight(25)
Rupture_SpellAlert:SetInsertMode("TOP")
Rupture_SpellAlert:SetFrameStrata("HIGH")
Rupture_SpellAlert:SetTimeVisible(1)
Rupture_SpellAlert:SetFadeDuration(0.5)
Rupture_SpellAlert:SetFont("Fonts\\FRIZQT__.TTF", 23, "OUTLINE")
Rupture_SpellAlert:SetPoint("CENTER", 0, 95)
Rupture_SpellAlert:RegisterEvent("CHAT_MSG_SPELL_HOSTILEPLAYER_DAMAGE")
Rupture_SpellAlert:RegisterEvent("CHAT_MSG_SPELL_HOSTILEPLAYER_BUFF")
Rupture_SpellAlert:RegisterEvent("CHAT_MSG_SPELL_PERIODIC_HOSTILEPLAYER_BUFFS")
Rupture_SpellAlert:SetScript("OnEvent", spellevent)

CreateFrame("MessageFrame", "Rupture_BuffAlert", UIParent)
Rupture_BuffAlert:SetWidth(UIParent:GetWidth())
Rupture_BuffAlert:SetHeight(25)
Rupture_BuffAlert:SetInsertMode("TOP")
Rupture_BuffAlert:SetFrameStrata("HIGH")
Rupture_BuffAlert:SetTimeVisible(1)
Rupture_BuffAlert:SetFadeDuration(0.5)
Rupture_BuffAlert:SetFont("Fonts\\FRIZQT__.TTF", 23, "OUTLINE")
Rupture_BuffAlert:SetPoint("BOTTOM", Rupture_SpellAlert, "TOP", 0, 2)

---------------------------------------------------------
-- Move Tooltip
local mousetot = function(self, e)
	self.update = self.update - e
	if self.update <= 0 then
		if self:IsUnit("mouseover") and UnitIsPlayer("mouseover") then
			local name = UnitName("mouseover")
			local class = select(2, UnitClass("mouseover"))
			local _, _, perc, r, g, b = getinfo("mouseover", "Health")
			GameTooltipTextLeft1:SetTextColor(r, g, b)
			GameTooltipTextLeft1:SetText(classhex(class)..name.. "|r ("..perc.."%)")
			if UnitExists("mouseovertarget") then
				name = UnitName("mouseovertarget")
				if UnitIsUnit("mouseovertarget", "player") then
					name = "|cffffffff"..YOU.."|r"
				else
					class = select(2, UnitClass("mouseovertarget"))
					name = classhex(class)..name.."|r"
				end
				local _, _, perc, r, g, b = getinfo("mouseovertarget", "Health")
				if not self.totline then
					self:AddLine("["..name.."]")
					self:Show()
					self.totline = self:NumLines()
				else
					local line = _G["GameTooltipTextLeft"..self.totline]
					line:SetTextColor(r, g, b)
					line:SetText("["..name.." ("..perc.."%)]")
					self:Show()
				end
			end
		end
		self.update = 0.15
	end
end

local settool = function(tooltip, parent)
	if tooltip == GameTooltip and parent == UIParent and UnitExists("mouseover") then
		if RUPTURE_TOURNEY_SAVED.cursor then
			tooltip:SetOwner(parent, "ANCHOR_CURSOR")
		end
		tooltip.totline = nil
	end
end

hooksecurefunc("GameTooltip_SetDefaultAnchor", settool)
GameTooltip.update = 0
GameTooltip:SetScript("OnUpdate", function (self, arg1) mousetot(self, arg1) end)

---------------------------------------------------------
-- Slash commands
SLASH_RUPTURETOURNEY1 = "/rtourney"
SlashCmdList["RUPTURETOURNEY"] = function(msg)
	msg = string.lower(msg)
	if msg == "cursor" then
		if RUPTURE_TOURNEY_SAVED.cursor then
			RUPTURE_TOURNEY_SAVED.cursor = nil
			DEFAULT_CHAT_FRAME:AddMessage("|cffff0000Rupture Tourney: .:|r Tooltips BottomRight ")
		else
			RUPTURE_TOURNEY_SAVED.cursor = true
			DEFAULT_CHAT_FRAME:AddMessage("|cffff0000Rupture Tourney: .:|r Tooltips Cursor ")
		end
	elseif msg == "aura" then
		if RUPTURE_TOURNEY_SAVED.aura then
			RUPTURE_TOURNEY_SAVED.aura = nil
			DEFAULT_CHAT_FRAME:AddMessage("|cffff0000Rupture Tourney: .:|r Aura Cooldowns Off ")
		else
			RUPTURE_TOURNEY_SAVED.aura = true
			DEFAULT_CHAT_FRAME:AddMessage("|cffff0000Rupture Tourney: .:|r Aura Cooldowns On ")
		end
	elseif msg == "team" then
		teamframe:Show()
	else
		DEFAULT_CHAT_FRAME:AddMessage("|cffff0000Rupture Tourney: .:|r cursor | aura | team |cffff0000:.|r")
	end
end