----------------------------
--      Declaration       --
----------------------------

qComments = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0", "AceHook-2.1", "AceDebug-2.0")

local tablet = AceLibrary("Tablet-2.0")
local quixote = AceLibrary("Quixote-1.0")
local dewdrop = AceLibrary("Dewdrop-2.0")

----------------------------
--      Main Functions    --
----------------------------

local defaults = {
	debug = false,
	cartographer = true,
	deletenotes = true,
	setnotes = false,
	width = 900,
	tablet_data = {detached = true}
}

local isVisible = false ;

function qComments:DisplayMessage(msg)
	DEFAULT_CHAT_FRAME:AddMessage(msg);
end
						
function qComments:GetComments(object, iQuestId)
		
		local title, iLevel, strTag, iGroup, _, iObjectives, zone
		local faction = UnitFactionGroup("player")
		
		if (IsAddOnLoaded("qComments_" .. faction .. "DB") == nil) then
				local loaded, reason = LoadAddOn("qComments_" .. faction .. "DB")
				if not loaded then
						self:DisplayMessage("The " .. faction .. " quest commentaries database could not be dynamically loaded (" .. reason .. "). You might try to uninstall and reinstall qComments to fix this problem.")
						return
				end
		end	
		
        self:Debug(string.format("GetComments(%s): %s", type(qQuestId), tostring(iQuestId)) )

        if ( type(iQuestId) == "number" ) then
                title, iLevel, strTag, iGroup, _, iObjectives, zone = quixote:GetQuestById(iQuestId)
        else
                title, iLevel, strTag, iGroup, _, iObjectives, zone = quixote:GetQuestByName(iQuestId)
        end

        if (title == nil or faction == nil or zone == nil) then
                return
        end

        self:Debug("GetComments: " .. faction .. " : " .. title)		
		
		local qData = self:GetFactionData(title)
		
        if (qData == nil) then
				if (IsAddOnLoaded("qComments_CommonDB") == nil) then
						local loaded, reason = LoadAddOn("qComments_CommonDB")
						if not loaded then
								self:DisplayMessage("The quest commentaries database for both factions could not be dynamically loaded (" .. reason .. "). You might try to uninstall and reinstall qComments to fix this problem.")
								return
						end
				end
				qData = self:GetCommonData(title)
		end		
		return title, qData, zone
end

function qComments:OnInitialize()
	self.options = {
		type="group",
		args = {
			debug = {
				name =	"Debug", type = "toggle",
				desc =	"Debug on/off",
				get =	function()
							return self.db.profile.debug
						end,
						set = function(v)
							self.db.profile.debug = v
						end,
			},
			setnotes = {
				name =	"SetNotes", type = "toggle",
				desc =	"Set coordinate notes on the map (required Cartographer) on quest for new quests",
				get =	function()
							return self.db.profile.setnotes
						end,
						set = function(v)
							self.db.profile.setnotes = v
						end,
			},
			deletenotes = {
				name =	"DeleteNotes", type = "toggle",
				desc =	"Delete notes from the map on quest completion",
				get =	function()
							return self.db.profile.deletenotes
						end,
						set = function(v)
							self.db.profile.deletenotes = v
						end,
			},
			width = {
				name =	"Frame width", type = "text",
				desc =	"Set frame width",
				usage = "width",
				get =	function()
							return self.db.profile.width
						end,
						set = function(v)
							self.db.profile.width = v
						end,
			},
		},
	}
	
	-- Register everything
	self:RegisterDB("qCommentsDB")
	self:RegisterDefaults('profile', defaults)
	self:RegisterChatCommand({ "/qcomments", "/qcm" }, self.options )
end

function qComments:OnEnable()
	if ( not nQuestLog ) then
		self:Print("qComments: No nQuestLog addon detected... aborted...") 
		return false
	end

	self:Hook(nQuestLog, "ShowQuestToolTip") 
	self:Hook(GameTooltip, "Hide", true)

	self:RegisterEvent("Quixote_Leaderboard_Update", "OnQuest_Leaderboard_Update")
	self:RegisterEvent("Quixote_Quest_Gained", "OnQuest_Gained")
end

function qComments:OnDisable()
	self:UnhookAll() 
	self:UnregisterAllEvents()
end

function qComments:OnQuest_Gained(strTitle)
	if (self.db.profile.setnotes) then
		self:DoNote(strTitle, 0)
	end
end

function qComments:OnQuest_Leaderboard_Update(title, _, _, desc)

	if ( self.db.profile.deletenotes) then
		local _, _, _, _, complete = quixote:GetQuestByName(title)
		self:Debug(string.format("OnQuest_Leaderboard_Update : %s - %d", title, complete or -1) )
		if (  complete == 1 ) then
			self:DoNote(title, complete) 
		end
	end
end

local function sortedpairs(t,comparator)
	local sortedKeys = {};
	table.foreach(t, function(k,v) table.insert(sortedKeys,k) end);
	table.sort(sortedKeys,comparator);
	local i = 0;
	local function _f(_s,_v)
		i = i + 1;
		local k = sortedKeys[i];
		if (k) then
			return k,t[k];
		end
	end
	return _f,nil,nil;
end

function qComments:DoNote(strTitle, delete)

	self:Debug(string.format("DoNote: %s - %d", strTitle, delete) )

	local title, qData, zone = self:GetComments(nil, strTitle)
	
	if ( title == nil or qData == nil or zone == nil ) then
		self:Debug("DoNote: GetComments returns nil")
		return
	end

	self:Debug(string.format("DoNote: %s - %d", title, delete) )

	if (Cartographer_Notes and qData["comments"]) then
		local BZ = Rock("LibBabble-Zone-3.0")
		local BZL = BZ:GetLookupTable()
		local BZH = BZ:GetUnstrictLookupTable()
		local BZR = BZ:GetReverseLookupTable()
		if not BZH[zone] then
			if BZR[zone] then
				zone = BZR[zone]
			else
				self:Debug(string.format("DoNote: Zone \"%s\" not found, returning nil", zone))
				return
			end
		end
		for k, v in sortedpairs(qData["comments"]) do 
			for x,y in string.gmatch(v, "%[(%d+) (%d+)%]") do
				nx = tonumber(x) / 100 ;
				ny = tonumber(y) / 100 ;
				if ( nx and ny and nx < 1 and nx >0 and ny < 1 and ny > 0) then
					self:Debug(string.format("DoNote: %s - %f,%f", title, nx, ny) )
					if ( delete == 0) then
						Cartographer_Notes:SetNote(zone, nx, ny, "Cross", "qComments (weekly)", 'title', title)
					else
						if ( Cartographer_Notes:GetNote(zone, nx, ny) ) then
							Cartographer_Notes:DeleteNote(zone, nx, ny)
						end
					end
				else
					self:Debug(string.format("DoNote: couldn't convert to number or bad coords '%s' or '%s'", x, y) ) 
				end
			end
		end
	end
end

function qComments:Hide(...)
	if ( isVisible and IsAltKeyDown()) then tablet:Close("qComments") end
	return self.hooks[GameTooltip]["Hide"](...)
end

function qComments:Debug( msg )
	if self.db.profile.debug then
		if DEFAULT_CHAT_FRAME then
			if type( msg ) == "table" then
				for k, v in pairs( msg ) do
					if not v then
						msg[k] = "nil"
					end
				end
				qComments:Print( table.concat( msg ) )
			else
				qComments:Print( msg )
			end
		end
	end
end

local gtitle, glvl, aComments = {}

function qComments:ShowQuestToolTip(...)
	
	self.hooks[nQuestLog]["ShowQuestToolTip"](...)
	
	if (IsAltKeyDown()) then
		local qData, zone, msg = ""
		gtitle, qData, zone = self:GetComments(...) 
		
		if (qData and qData["comments"]) then
			if ( IsControlKeyDown() ) then
				self:DoNote(gtitle, 0)
			elseif ( IsShiftKeyDown() ) then
				self:DoNote(gtitle, 1)
			end
			aComments = qData["comments"]
			glvl = qData["reqlevel"]
		else
			aComments = {}
		end
			
		for k, v in sortedpairs(aComments) do 
			self:Debug(string.format("ShowQuestToolTip [%s] = %s", k or "null", v or "null") )
		end
		if not tablet:IsRegistered('qComments') then
			tablet:Register(
				'qComments',
				'parent', nQuestLogFrame,
				'point', function() return "BOTTOMRIGHT", "TOPLEFT" end,
				'detachedData', self.db.profile.tablet_data,
				'dontHook', true, 
				'showTitleWhenDetached', true,
				'maxWidth', self.db.profile.width,
				'wrap', true,
				'cantAttach', true,
				'menu',
					function()
						dewdrop:AddLine(
							'text', "Close all",
							'closeWhenClicked', true,
							'func',
								function()
									tablet:Close("qComments")							
								end
						)
					end,
				'children', 
					function()
						tablet:SetTitle(string.format("               %64s%" .. 64-string.len(gtitle) .. "s               ", gtitle, " "))
						local cat = tablet:AddCategory(gtitle)
						if (#aComments == 0) then
							cat:AddLine('text', "\n\n  No commentaries found.\n\n  Why not adding your own on www.wowhead.com? :)\n\n")
						else
							cat:AddLine('text', "Required Level: " .. glvl)
							for k, v in sortedpairs(aComments) do 
								local c = string.gsub(string.gsub(v, "%[(%d+) (%d+)%]", "|cffffd700 %1,%2 |r"),
												"(<http:.->)", "|cff0101ee %1 |r")														
								cat:AddLine('text', "|cffaa4444" .. k .. ".|r  " .. c, 
											'textR', 0.9,
											'textG', 0.9,
											'textB', 0.9,
											'wrap', true
								)
							end
						end
					end
			)
		end
		tablet:Open("qComments")
		tablet:Refresh("qComments")
		isVisible = true
	end
end
