﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------

Unit Skinning Options: Status

class options =
{
	string orient				-- location of the first role icon on the bar: LEFT (grows right), RIGHT (grows left), TOP (grows down), BOTTOM (grows up)
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	float height				-- height of the bar
	float width					-- width of the bar
	int level					-- the frame level to use (1-9) or nil to use the parent frame's level+1
	
	class label					-- location of the status text within the frame or nil to use default of centered text
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 75%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local CreateFrame         = CreateFrame;
local GetTime             = GetTime;
local IsResting           = IsResting;
local UnitAffectingCombat = UnitAffectingCombat;
local UnitInRange         = UnitInRange;
local UnitIsAFK           = UnitIsAFK;
local UnitIsConnected     = UnitIsConnected;
local UnitIsDND           = UnitIsDND;
local UnitIsDead          = UnitIsDead;
local UnitIsFeignDeath    = UnitIsFeignDeath;
local UnitIsGhost         = UnitIsGhost;
local UnitIsPartyLeader   = UnitIsPartyLeader;
local UnitOnTaxi          = UnitOnTaxi;

-------------------------------------------------------------------------------
-- unit status event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Status", nUI_Unit.Drivers )

local StatusCallbacks    = {};
local StatusUnits        = {};
local timer              = 0;

nUI_Unit.Drivers.Status  = frame;

-------------------------------------------------------------------------------

local function onStatusUpdate( who, elapsed )
	
	timer = timer + elapsed;
	
	if timer > 0.2 then -- check at 5fps -- plenty fast enough to know this info
	
		timer = 0;
			
		-- we're only going to bother looking at the lists that have at least one listener
			
		for list_unit in pairs( StatusCallbacks ) do
			if #StatusCallbacks[list_unit] > 0 then				
				frame.newUnitInfo( list_unit, nUI_Unit:getUnitInfo( list_unit ) );
			end
		end				
	end
end

frame:SetScript( "OnUpdate", onStatusUpdate );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for unit status changes GUID

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateStatusInfo( list_unit, unit_info );
	local callbacks = StatusCallbacks;
	local unitlist  = StatusUnits;
	
	nUI_Unit:notifyCallbacks( nUI_L["unit status"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of unit status listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the status info of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerStatusCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = StatusCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not StatusCallbacks[unit_id] then
			StatusCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		
		if #list == 1 then
			nUI_Unit:registerRangeCallback( unit_id, nUI_Unit.Drivers.Label );
		end
		
		-- collect status information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateStatusInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterStatusCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = StatusCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterRangeCallback( unit_id, nUI_Unit.Drivers.Label );
		end
	end
end

-------------------------------------------------------------------------------
-- update the status information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateStatusInfo( unit_id, unit_info )

	local modified  = false;
	
	if unit_info then
		
		local raid_info   = unit_info.in_raid and unit_info.raid_info or nil;
		local range_info  = unit_info.range_info;
		local raid_leader = raid_info and raid_info.rank == 2 or false;
		local is_player   = unit_info.is_player;
		local status_info = unit_info.status_info or {};
		local is_dead     = UnitIsDead( unit_id ) and true or false;
		local is_ghost    = UnitIsGhost( unit_id ) and true or false;
		local is_resting  = unit_info.is_self and IsResting() and true or false;
		local is_afk      = is_player and UnitIsAFK( unit_id ) and true or false;
		local is_dnd      = is_player and UnitIsDND( unit_id ) and true or false;
		local is_oor      = range_info and range_info.min and range_info.min >= 40 or false;
		local is_fd       = is_player and UnitIsFeignDeath( unit_id )and true or false;
		local on_taxi     = is_player and UnitOnTaxi( unit_id ) and true or false;
		local is_leader   = is_player and (raid_leader or UnitIsPartyLeader( unit_id )) and true or false;
		local is_online   = not is_player or (UnitIsConnected( unit_id ) and true or false);
		local in_range    = not is_player or (UnitInRange( unit_id ) and true or false);
		local in_combat   = UnitAffectingCombat( unit_id ) and true or false;
		
		-- we get a few status values for pets from the status values of their owner
		
		if unit_info.pet_owner then
			
			local pet_owner = nUI_Unit:getGUIDInfo( unit_info.pet_owner );
			
			if pet_owner and pet_owner.status_info then
				is_afk     = pet_owner.status_info.is_afk;
				is_dnd     = pet_owner.status_info.is_dnd;
				is_onlne   = pet_owner.status_info.is_online;
				is_resting = pet_owner.status_info.is_resting;
			end
		end
		
		if status_info.is_dead    ~= is_dead
		or status_info.is_ghost   ~= is_ghost
		or status_info.is_resting ~= is_resting
		or status_info.is_afk     ~= is_afk
		or status_info.is_dnd     ~= is_dnd
		or status_info.is_oor     ~= is_oor
		or status_info.is_online  ~= is_online
		or status_info.is_fd      ~= is_fd
		or status_info.is_leader  ~= is_leader
		or status_info.on_taxi    ~= on_taxi
		or status_info.in_range   ~= in_range
		or status_info.in_combat  ~= in_combat
		then
			
			status_info.is_dead    = is_dead;
			status_info.is_ghost   = is_ghost;
			status_info.is_resting = is_resting
			status_info.is_afk     = is_afk;
			status_info.is_dnd     = is_dnd;
			status_info.is_oor     = is_oor;
			status_info.is_online  = is_online;
			status_info.is_fd      = is_fd;
			status_info.is_leader  = is_leader;
			status_info.on_taxi    = on_taxi;
			status_info.in_range   = in_range;
			status_info.in_combat  = in_combat;
			
			modified              = true;
			unit_info.modified    = true;
			unit_info.last_change = GetTime();
			unit_info.status_info = status_info;
			
		end
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered unit status listeners, even if there's no 
-- change in data... typically used when the status color options change
-- or entering the world

function nUI_Unit:refreshStatusCallbacks()

	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit status"], StatusCallbacks, StatusUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateStatusInfo( list_unit, unit_info ); 
		end 
	);

end

-------------------------------------------------------------------------------
-- create a new unit status frame

function nUI_Unit:createStatusFrame( parent, unit_id, id, options )

	local frame      = nUI_Unit:createFrame( "$parent_Status"..(id or ""), parent, unit_id, false );
	frame.text       = frame:CreateFontString( "$parentText" );
	frame.text.value = "";
	frame.Super      = {};
	
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated
	
	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateStatusFrame( frame );
		end
		
	end
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerStatusCallback( frame.unit, frame );
				nUI_Unit:updateStatusFrame( frame );
			else
				nUI_Unit:unregisterStatusCallback( frame.unit, frame );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame.

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );

		frame.configText( frame.text, frame.options.label );
		
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.

	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		frame.Super.applyOptions( options );
		
		frame.parent:SetAlpha( frame.active and frame.options.fade_unit or 1 );
		
		nUI_Unit:updateStatusFrame( frame );
		
	end

	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerStatusCallback( frame.unit, frame );	
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a unit status frame

function nUI_Unit:deleteStatusFrame( frame )

	nUI_Unit:unregisterStatusCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
	frame.parent.applyFrameFader( frame, false );	
	
end

-------------------------------------------------------------------------------
-- display the appropriate icon for the unit's status
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateStatusFrame( frame )
	
	local unit_info = frame.unit_info;
	
	-- if there is no unit or we don't know it's status, then hide the icon
	
	if not unit_info or not unit_info.status_info then
		
		if frame.active then
			frame.active = false;
			frame:SetAlpha( 0 );

			if frame.options.fade_unit then
				frame.parent:SetAlpha( 1 );
			end
		end
	
	-- otherwise, show the icon and clip the appropriate region
	
	else

		-- if the status has changed from what we last knew, then update 

		local msg;
		local status_info = unit_info.status_info;
		
		if status_info.is_ghost then msg = nUI_L["GHOST"];
		elseif status_info.is_dead then msg = nUI_L["DEAD"];
		elseif status_info.is_fd then msg = nUI_L["FD"];
		elseif status_info.on_taxi then msg = nUI_L["TAXI"];
		elseif status_info.is_afk then msg = nUI_L["AFK"];
		elseif not status_info.is_online then msg = nUI_L["OFFLINE"];
		elseif status_info.is_oor then msg = nUI_L["OOR"];
		end
		
		if frame.text.value ~= msg then
			
			frame.text.value = msg;

			if msg then
				
				-- if the status is hidden, show it
				
				if not frame.active then
					frame.active = true;
					frame:SetAlpha( 1 );
				end
		
				frame.text:SetText( msg );
			
			elseif frame.active then
				
				frame.active = false;
				frame:SetAlpha( 0 );

			end
		end
		
		frame.parent.applyFrameFader( frame, msg ~= nil and frame.options.fade_unit );
		
	end
end
