﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--]]---------------------------------------------------------------------------
	
if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local CheckInteractDistance = CheckInteractDistance;
local CreateFrame           = CreateFrame;
local GetNetStats           = GetNetStats;
local GetTime               = GetTime;
local GetTalentTabInfo      = GetTalentTabInfo;
local NotifyInspect         = NotifyInspect;

-------------------------------------------------------------------------------
-- unit spec event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Spec", nUI_Unit.Drivers )

local first_queue      = true;
local updating_queue   = 0;
local queue_check      = 1 / nUI_DEFAULT_FRAME_RATE;
local num_queued       = 0;
local TalentBuildQueue = {};
local QueueCount       = {};
local SpecCallbacks    = {};
local SpecUnits        = {};

nUI_Unit.Drivers.Spec  = frame;

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for spec changes GUID or when the unit reaction information changes

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateSpecInfo( list_unit, unit_info );
	local callbacks = SpecCallbacks;
	local unitlist  = SpecUnits;

	nUI_Unit:notifyCallbacks( nUI_L["unit spec"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- add a request to the talent build inspection queue (API entry point)

local function GetTalentBuild( unit_id, unit_info )

	updating_queue      = updating_queue+1;
	QueueCount[unit_id] = (QueueCount[unit_id] or 0)+1;
	num_queued          = num_queued+1;

--	nUI:debug( "nUI_UnitSpec: adding "..(unit_info.name or unit_id).." to the inspection queue", 1 );
	
	if first_queue then		
		first_queue = false;
	end
	
	TalentBuildQueue[num_queued] = 
	{
		ready    = false,
		pending  = false,
		unit     = unit_info,
		guid     = unit_info.guid,
		id       = unit_id,
		expires  = 0;
	};
		
	updating_queue = updating_queue-1;
	
end

-------------------------------------------------------------------------------
-- remove the first item from the talent build inspection queue

local function Dequeue( queue )
	
	if num_queued > 0 then

		unit = queue.unit;

--		nUI:debug( "nUI_UnitSpec: dequeuing "..(unit.name or queue.id), 1 );

		if QueueCount[queue.id] > 0 then
			QueueCount[queue.id] = QueueCount[queue.id]-1;
		end
		
		num_queued = num_queued-1;

		for i=1,num_queued do
			TalentBuildQueue[i] = TalentBuildQueue[i+1];
		end				
	end	
end

-------------------------------------------------------------------------------
-- move a pending talent build inspection to the end of the queue

local function Requeue( queue )
	
	if num_queued > 0 then
		
		local unit_info = queue.unit;
		local unit_id   = queue.id;
		
		Dequeue( queue );
		GetTalentBuild( unit_id, unit_info );
		
	end
end

-------------------------------------------------------------------------------
-- read the talent table

local function ReadTalents( queue, unit )

	local id        = queue.id;
	local guid      = queue.guid;
	local is_player = unit and unit.is_self or false;
	local match     = unit and unit.guid == guid or false;
	local build     = {};
	
	if match then

--		nUI:debug( "nUI_UnitSpec: reading talent spec for "..(unit.name or id), 1 );

		local remoteInspect = not is_player;
		
		build.name1, build.iconTexture1, build.num1, build.background1  = GetTalentTabInfo( 1, remoteInspect );
		build.name2, build.iconTexture2, build.num2, build.background2  = GetTalentTabInfo( 2, remoteInspect );
		build.name3, build.iconTexture3, build.num3, build.background3  = GetTalentTabInfo( 3, remoteInspect );

		-- make sure that the read we are doing is for the unit we intend it to be for
	
		if build.name1 == nil 
		or build.name2 == nil 
		or build.name3 == nil
		or build.num1 == nil 
		or build.num2 == nil 
		or build.num3 == nil 
		or (build.num1 == 0 and build.num2 == 0 and build.num3 == 0)
		then
			build = nil;
		end
	end
	
	if QueueCount[id] == 0 then
		
--		nUI:debug( "nUI_UnitSpec: passing new unit info for "..(unit.name or id), 1 );

		frame.build = match and build or nil;
		frame.newUnitInfo( id, queue.unit  );
		
	end	
end

-------------------------------------------------------------------------------
-- flag the unit spec ready for reading on a unit we're waiting for

local function onTalentBuildEvent()
	
	if num_queued > 0 then
	
		local queue = TalentBuildQueue[1];
		
		queue.ready = true;

--		nUI:debug( "nUI_UnitSpec: flagged "..(queue.unit.name or queue.id).." as ready", 1 );
		
	end	
end

frame:SetScript( "OnEvent", onTalentBuildEvent );
frame:RegisterEvent( "INSPECT_TALENT_READY" );

-------------------------------------------------------------------------------
-- process the talent build inspection queue

local function onTalentBuildUpdate( who, elapsed )
	
	if num_queued > 0 then
			
		queue_check = queue_check - elapsed;
		
		if updating_queue == 0
		and queue_check <= 0
		then
		
			queue_check = nUI_Unit.frame_rate;

			for i=1,num_queued do			
				
				-- get this first item on the queue
				
				local queue       = TalentBuildQueue[1];			
				local id          = queue.id;
				local unit        = queue.unit;
				local unit_info   = nUI_Unit:getUnitInfo( id );
				local is_player   = unit_info and unit_info.is_self or false;
	
				-- if the unit we're looking for does not match the one that's
				-- in the queue, then the unit has changed GUID and we can dequeue
				
				if not unit_info
				or unit.guid ~= unit_info.guid
				then
					
					Dequeue( queue );
					
				-- if the pending talent build ready is ready, get it now
				
				elseif queue.ready then
		
--					nUI:debug( "nUI_UnitSpec: queue is ready for "..(unit.name or id), 1 );
					
					Dequeue( queue );
					ReadTalents( queue, unit_info );
					
				-- if it is not already pending processing, start it now
	
				elseif not queue.pending then
	
					local _,_,latency = GetNetStats();
	
					queue.pending = true;
					queue.expires = GetTime() + latency + 1;
					
					-- if this is the player, run it locally now
					
					if is_player 
					then 
					
--						nUI:debug( "nUI_UnitSpec: fetching player talents for "..id, 1 );
					
						Dequeue( queue );
						ReadTalents( queue, unit_info );
					
					-- if this unit already has a talent build, then just use what's cached
					
					elseif unit_info.build
					then
					
						Dequeue( queue );
						
						frame.build = unit.build;
						frame.newUnitInfo( id, unit );
						
					-- if the unit no longer exists or it is no longer player controlled
					
					elseif not unit_info
					or not unit_info.is_player
					then

--						nUI:debug( "nUI_UnitSpec: "..(unit.name or id).." no longer exists or is not a player", 1 );					

						Dequeue( queue );
	
					-- if the unit is out of range, requeue it to try again later
					
					elseif not unit_info.is_visible
					or not CheckInteractDistance( id, 1 ) then

--						nUI:debug( "nUI_UnitSpec: "..(unit.name or id).." is out of inspection range", 1 );
					
						Requeue( queue );
	
						if num_queued == 1 then break; end
	
					-- otherwise, intiate a new unit inspection and we're done
					
					else

--						nUI:debug( "nUI_UnitSpec: requesting inspection for "..(unit.name or id), 1 );
					
						NotifyInspect( id );
						break;
						
					end
	
				-- if the unit is pending and no longer exists, we can drop the request
				
				elseif not unit_info
				or not unit_info.is_player
				then

--					nUI:debug( "nUI_UnitSpec: "..(unit.name or id).." no longer exists or is not a player", 1 );
					
					Dequeue( queue );
	
				-- if the unit has moved out of range, we can requeue and try again later

				elseif not unit_info.is_visible
				or not CheckInteractDistance( id, 1 ) then

--					nUI:debug( "nUI_UnitSpec: "..(unit.name or id).." moved out of inspection range", 1 );
					
					Requeue( queue );
					
					if num_queued == 1 then break; end
			
				-- or if it's expired, give it up
				
				elseif GetTime() > queue.expires then

--					nUI:debug( "nUI_UnitSpec: timed out waiting for inspection of "..(unit.name or id), 1 );
					
					Requeue( queue );
					
					if num_queued == 1 then break; end
				
				end
			end
		end
	end
end

frame:SetScript( "OnUpdate", onTalentBuildUpdate );

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of unit spec listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the unit spec of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerSpecCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = SpecCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not SpecCallbacks[unit_id] then
			SpecCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		-- or the unit reaction status changes (enters/leaves a party/raid, etc.)
		
		if #list == 1 then
			nUI_Unit:registerUnitChangeCallback( unit_id, nUI_Unit.Drivers.Spec );
		end
		
		-- collect unit spec information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateSpecInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterSpecCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = SpecCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterUnitChangeCallback( unit_id, nUI_Unit.Drivers.Spec );
		end
	end
end

-------------------------------------------------------------------------------
-- update the unit spec information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateSpecInfo( unit_id, unit_info )

	local modified  = false;
	
	if unit_info then

		-- if this is not a player character, then our only concern is 
		-- whether or not it is an elite, rare, world boss, etc.
		
		local family         = UnitCreatureFamily( unit_id );
		local type           = UnitCreatureType( unit_id );
		local race           = UnitRace( unit_id );
		local sex            = UnitSex( unit_id );
		local classification = UnitClassification( unit_id );
		local build          = frame.build or unit_info.build;
		local spec;
		
		-- if this unit is not a player, then there is no such thing
		-- as a talen build for it
		
		if not unit_info.is_player 
		then build = nil;
		
		-- otherwise, if this is a player and we don't already have
		-- a talent build for it, then queue a request for one
		
		elseif not build and not unit_info.build
		then GetTalentBuild( unit_id, unit_info )			
		
		-- otherwise, if we do have a build, see if anything has changed
		
		elseif build then
			
			if not unit_info.build
			or unit_info.build.num1  ~= build.num1
			or unit_info.build.num2  ~= build.num2
			or unit_info.build.num3  ~= build.num3
			or unit_info.build.name1 ~= build.name1
			or unit_info.build.name2 ~= build.name2
			or unit_info.build.name3 ~= build.name3
			or unit_info.build.icon1 ~= build.name1
			or unit_info.build.icon2 ~= build.icon2
			or unit_info.build.icon3 ~= build.icon3
			or unit_info.build.bg1   ~= build.bg1
			or unit_info.build.bg2   ~= build.bg2
			or unit_info.build.bg3   ~= build.bg3
			then

				build.last_update = GetTime();
				
				-- construct the strings because something changed
				
				build.points = build.num1.." / "..build.num2.." / "..build.num3;
				
				if build.num1 >= build.num2 and build.num1 >= build.num3 then
					build.name = build.name1;
				elseif build.num2 >= build.num1 and build.num2 >= build.num3 then
					build.name = build.name2;
				else
					build.name = build.name3;
				end

			-- if nothing has changed, so make sure the two builds
			-- will compare as being equal
			
			else					
				build = unit_info.build;
			end
		end		
		
		-- has anything about the unit changed?
		
		if unit_info.family         ~= family
		or unit_info.type           ~= type
		or unit_info.race           ~= race
		or unit_info.sex            ~= sex
		or unit_info.classification ~= classification
		or unit_info.build          ~= build
		then
			
			modified                 = true;
			unit_info.modified       = true;
			unit_info.last_change    = GetTime();
			unit_info.family         = family;
			unit_info.type           = type;
			unit_info.race           = race;
			unit_info.sex            = sex;
			unit_info.classification = classification;
			unit_info.build         = build;
			
		end
	end
	
	frame.build = nil;
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered unit spec listeners, even if there's no 
-- change in data.

function nUI_Unit:refreshSpecCallbacks()
	
	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit spec"], SpecCallbacks, SpecUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateSpecInfo( list_unit, unit_info ); 
		end 
	);

end

-------------------------------------------------------------------------------
-- create a new unit spec frame

function nUI_Unit:createSpecFrame( parent, unit_id, id, options, clickable )

	local frame = nUI_Unit:createFrame( "$parent_Spec"..(id or ""), parent, unit_id, clickable );
	frame.text  = frame:CreateFontString( "$parentLabel", "OVERLAY" );
	frame.icon  = frame:CreateTexture( "$parentIcon", "ARTWORK" );
	frame.Super = {};
	
	frame.icon:SetAlpha( 0 );
	
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated

	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )

		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateSpecFrame( frame );
		end
		
	end;
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerSpecCallback( frame.unit, frame );
				nUI_Unit:updateSpecFrame( frame );
			else
				nUI_Unit:unregisterSpecCallback( frame.unit, frame );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame. Is also recreates
	-- the font to present clear, sharp, readable text versus the blurred text you get
	-- as a result of frame:SetScale() or text:SetTextHeight()
	
	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );

		if frame.options and frame.options.icon then
			
			local width  = frame.options.icon.width * frame.scale * nUI.scale;
			local height = frame.options.icon.height * frame.scale * nUI.scale;
			
			if frame.icon.width  ~= width
			or frame.icon.height ~= height
			then
		
				frame.icon.width  = width;
				frame.icon.height = height;
				
				frame.icon:SetWidth( width );
				frame.icon:SetHeight( height );
				
			end
		end
		
		frame.configText( frame.text, frame.options.label );
		
	end
	
	-- anchor the frame and its icon if it has one
	
	frame.Super.applyAnchor = frame.applyAnchor;
	frame.applyAnchor       = function( anchor )
		
		frame.Super.applyAnchor( anchor );
		
		if frame.options and frame.options.icon then
			
			local options = frame.options.icon;
			local anchor_pt   = options.anchor_pt or "CENTER";
			local relative_to = options.relative_to or frame:GetName();
			local relative_pt = options.relative_pt or anchor_pt;
			local xOfs        = (options.xOfs or 0) * frame.scale * nUI.scale;
			local yOfs        = (options.yOfs or 0) * frame.scale * nUI.scale;
			
			if frame.icon.anchor_pt   ~= anchor_pt
			or frame.icon.relative_to ~= relative_to
			or frame.icon.relative_pt ~= relative_pt
			or frame.icon.xOfs        ~= xOfs
			or frame.icon.yOfs        ~= yOfs
			then
				
				frame.icon.anchor_pt = anchor_pt;
				frame.icon.relative_to = relative_to;
				frame.icon.relative_pt = relative_pt;
				frame.icon.xOfs        = xOfs;
				frame.icon.yOfs        = yOfs;
				
				frame.icon:ClearAllPoints();
				frame.icon:SetPoint( anchor_pt, relative_to:gsub( "$parent", frame.parent:GetName() ), relative_pt, xOfs, yOfs );
				
			end
		end
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.

	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		frame.Super.applyOptions( options );
		
		if not frame.options.icon 
		or not frame.options.icon.enabled
		then
			
			frame.icon.enabled = false;
			frame.icon:Hide();
			
		else
			
			frame.icon.enabled = true;
			frame.icon:Show();
			
		end
		
		nUI_Unit:updateSpecFrame( frame );
		
	end
	
	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerSpecCallback( frame.unit, frame );
	frame.applyOptions( options );
	
	return frame;
		
end

-------------------------------------------------------------------------------
-- remove a unit spec frame

function nUI_Unit:deleteSpecFrame( frame )

	nUI_Unit:unregisterSpecCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- display the appropriate text for the unit's spec
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateSpecFrame( frame )
	
	local unit_info = frame.unit_info;

	-- if there is no unit or we don't know it's unit spec, then hide the panel
	
	if not unit_info 
	then
		
		if frame.active then

			frame.active = false;
			frame:SetAlpha( 0 );
		end
	
	-- otherwise, show the unit spec
	
	else

		local spec;
		local icon;
		local coord;
		
		-- if the unit spec is hidden, show it
		
		if not frame.active then

			frame.active = true;
			frame:SetAlpha( 1 );
		end

		-- if this unit has a talent build, then we build the label from that
		
		if unit_info.build then
		
			if frame.options.show_name and frame.options.show_points then
				spec = unit_info.build.name.." ("..unit_info.build.points..")";
			elseif frame.options.show_name then
				spec = unit_info.build.name;
			elseif frame.options.show_points then
				spec = unit_info.build.points;
			end
		
		-- otherwise is there some generic mob information we need to display about this unit?
		
		elseif unit_info.classification == "worldboss" then			
			icon  = frame.options.icon  and frame.options.icon.world_boss_icon or nil;
			coord = frame.options.icon and frame.options.icon.world_boss_coord or nil;
			spec  = nUI_L["World Boss"];
			
		elseif unit_info.classification == "elite" then			
			icon  = frame.options.icon and frame.options.icon.elite_icon or nil;
			coord = frame.options.icon and frame.options.icon.elite_coord or nil;
			spec  = nUI_L["Elite"];
			
		elseif unit_info.classification == "rare" then			
			icon  = frame.options.icon and frame.options.icon.rare_icon or nil;
			coord = frame.options.icon and frame.options.icon.rare_coord or nil;
			spec  = nUI_L["Rare"];
			
		elseif unit_info.classification == "rareelite" then
			icon  = frame.options.icon and frame.options.icon.rare_elite_icon or nil;
			coord = frame.options.icon and frame.options.icon.rare_elite_coord or nil;
			spec  = nUI_L["Rare Elite"];
			
		elseif frame.options.show_family then
			spec = unit_info.family;
			
		elseif frame.options.show_type then
			spec = unit_info.type;
			
		end
		
		-- if the unit spec has changed from what we last knew, then update 
		
		if frame.spec ~= spec then

			frame.spec = spec;			
			frame.text:SetText( spec );
			
		end		
		
		-- set the frame icon if required
		
		if frame.icon.enabled
		and (frame.icon.texture ~= icon or frame.icon.coord ~= coord)
		then
			
			frame.icon.texture = icon;
			frame.icon.coord   = coord;
			frame.icon:SetTexture( icon );
			frame.icon:SetTexCoord( unpack( coord or { 0, 0, 0, 1, 1, 0, 1, 1 } ) );

			if not frame.icon.active then
				
				frame.icon.active = true;
				frame.icon:SetAlpha( 1 );
				
			end
			
		elseif frame.icon.active then
			
			frame.icon.active = false;
			frame.icon:SetAlpha( 0 );
			
		end
	end
end
