﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------

Unit Skinning Options: Label

class options =
{
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	string text					-- sets a fixed, unchanging, label regardless of the underlying mob's name
	float height				-- height of the bar
	float width					-- width of the bar
	int level					-- the frame level to use (1-9) or nil to use the parent frame's level+1
	boolean show_reaction		-- sets the background color of the frame green, yellow or red based on the unit's hostility to the player 
	boolean class_colors		-- colors the unit name text based on the class of the unit (warrior, druid, etc.)
	
	class label					-- location of the unit name within the frame or nil to use default of centered text
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 50%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local GetTime   = GetTime;
local UnitName  = UnitName;

-------------------------------------------------------------------------------
-- unit label event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Label", nUI_Unit.Drivers )

local LabelCallbacks    = {};
local LabelUnits        = {};

nUI_Unit.Drivers.Label  = frame;

-------------------------------------------------------------------------------

local function onLabelEvent()

	-- for these events, we don't know which units are affected, so
	-- we span the list of all known interested units to see who is watching
		
	if event == "PLAYER_ENTERING_WORLD" then
		
		nUI_Unit:refreshLabelCallbacks();
		
	-- we're only going to look at the event if there's someone who cares about it
		
	elseif LabelCallbacks[arg1] and #LabelCallbacks[arg1] > 0 then
			
		frame.newUnitInfo( arg1, nUI_Unit:getUnitInfo( arg1 ) );
		
	end		
end

frame:SetScript( "OnEvent", onLabelEvent );
frame:RegisterEvent( "PLAYER_ENTERING_WORLD" );
frame:RegisterEvent( "UNIT_NAME_UPDATE" );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for unit label changes GUID or when the unit reaction information changes

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateLabelInfo( list_unit, unit_info );
	local callbacks = LabelCallbacks;
	local unitlist  = LabelUnits;
	
	nUI_Unit:notifyCallbacks( nUI_L["unit label"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of unit label listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the label info of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerLabelCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = LabelCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not LabelCallbacks[unit_id] then
			LabelCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		
		if #list == 1 then
			nUI_Unit:registerStatusCallback( unit_id, nUI_Unit.Drivers.Label );
		end
		
		-- collect label information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateLabelInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterLabelCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit id and remove this callback
		
		local list = LabelCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterStatusCallback( unit_id, nUI_Unit.Drivers.Label );
		end
	end
end

-------------------------------------------------------------------------------
-- update the label information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateLabelInfo( unit_id, unit_info )

	local modified  = false;
	
	if unit_info then
		
		-- check to see if the label has changed
		
		local name, realm = UnitName( unit_id );

		if unit_info.status_info and unit_info.status_info.is_dnd then
			name = ("%s |cFF00FFFF(%s)|r"):format( name, nUI_L["DND"] );
		end
		
		if unit_info.name ~= name
		or unit_info.realm ~= realm
		then
			
			modified              = true;
			unit_info.last_change = GetTime();
			unit_info.name        = name;
			unit_info.realm       = realm;
			
		end
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered unit label listeners, even if there's no 
-- change in data.

function nUI_Unit:refreshLabelCallbacks()
	
	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit label"], LabelCallbacks, LabelUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateLabelInfo( list_unit, unit_info ); 
		end 
	);

end

-------------------------------------------------------------------------------
-- create a new unit label frame

function nUI_Unit:createLabelFrame( parent, unit_id, id, options, clickable )

	local frame          = nUI_Unit:createFrame( "$parent_Label"..(id or ""), parent, unit_id, clickable );	
	frame.text           = frame:CreateFontString( "$parentText", "OVERLAY" );
	frame.texture        = frame:CreateTexture( "$parentReaction", "ARTWORK" );
	frame.texture.active = true;
	frame.Super   = {};
		
	frame.texture:SetPoint( "CENTER", frame, "CENTER", 0, 0 );
	frame.texture:SetTexture( 1, 1, 1, 1 );
	frame.texture:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
	
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated
	
	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateLabelFrame( frame );
		end
		
	end;
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				if frame.show_reaction then 
					nUI_Unit:registerReactionCallback( frame.unit, frame );
				end
				if frame.class_colors then
					nUI_Unit:registerClassCallback( frame.unit, frame );
				end
				frame.unit_info = nUI_Unit:registerLabelCallback( frame.unit, frame );
				nUI_Unit:updateLabelFrame( frame );
			else
				if frame.show_reaction then
					nUI_Unit:unregisterReactionCallback( frame.unit, frame );
				end
				if frame.class_colors then
					nUI_Unit:unregisterClassCallback( frame.unit, frame );
				end
				nUI_Unit:unregisterLabelCallback( frame.unit, frame );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame. Is also recreates
	-- the font to present clear, sharp, readable text versus the blurred text you get
	-- as a result of frame:SetScale() or text:SetTextHeight()

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );
		
		frame.configText( frame.text, frame.options.label );
		
		if frame.texture.size   ~= frame.size 
		or frame.texture.width  ~= frame.width
		or frame.texture.height ~= frame.height 
		or frame.texture.inset  ~= frame.inset
		then
			
			frame.texture.size   = frame.size;
			frame.texture.width  = frame.width;
			frame.texture.height = frame.height;
			frame.texture.inset  = frame.inset;
			
			frame.texture:SetWidth( (frame.size or frame.width) - frame.inset );
			frame.texture:SetHeight( (frame.size or frame.height) - frame.inset );

		end		
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.

	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		frame.Super.applyOptions( options );
		
		-- if we want to show unit reaction in the frame background then
		-- register for callbacks on reaction updates
		
		if options.show_reaction and not frame.show_reaction then
			frame.show_reaction = true;
			frame.unit_info = nUI_Unit:registerReactionCallback( frame.unit, frame );
			
		-- otherwise make sure we're not getting callbacks we don't cre about
		
		elseif frame.show_reaction then
			frame.show_reaction = false;
			nUI_Unit:unregisterReactionCallback( frame.unit, frame );
		end
		
		-- if we want to use class colors on the label text, then register
		-- for callbacks on class updates
		
		if options.class_colors and not frame.class_colors then
			frame.class_colors = true;
			frame.unit_info = nUI_Unit:registerClassCallback( frame.unit, frame );
			
		-- otthewise, again, makes sure we're not wasting CPU cycles
		
		elseif frame.class_colors then
			frame.class_colors = false;
			nUI_Unit:unregisterClassCallback( frame.unit, frame );
		end
		
		-- refresh the frame
		
		nUI_Unit:updateLabelFrame( frame );
		
	end
	
	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerLabelCallback( frame.unit, frame );
	
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a unit label frame

function nUI_Unit:deleteLabelFrame( frame )

	nUI_Unit:unregisterLabelCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- display the appropriate icon for the unit's label
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateLabelFrame( frame )
	
	local unit_info = frame.unit_info;
	
	-- if there is no unit or we don't know it's label, then hide the panel
	
	if not unit_info or not unit_info.name then
		
		if frame.active then
			frame.active = false;
			frame:SetAlpha( 0 );
		end
	
	-- otherwise, show the label
	
	else

		-- if the label is hidden, show it
		
		if not frame.active then
			frame.active = true;
			frame:SetAlpha( 1 );
		end

		-- if the label has changed from what we last knew, then update 
		
		local name = frame.options.text or unit_info.name;

		if frame.name ~= name then

			frame.name = name;
			
			frame.text:SetText( name );
		
		end
		
		-- check and see if the label color has changed (showing class colors)

		local class_colors = frame.options.class_colors and unit_info.class_info and unit_info.class_info.color ~= nil;
		local label_color  = class_colors and unit_info.class_info.color;
		
		if label_color
		and frame.class_color ~= label_color
		then
							
			frame.class_color = label_color;
			
			frame.text:SetTextColor( label_color.r, label_color.g, label_color.b, 1 );
			
		end

		-- check and see if the unit reaction color has changed
		
		local show_reaction  = frame.show_reaction and unit_info.reaction_color ~= nil;
		local reaction_color = show_reaction and unit_info.reaction_color or (frame.options.border and frame.options.border.color.backdrop or {});
		
		local r = reaction_color.r or 0;
		local g = reaction_color.g or 0;
		local b = reaction_color.b or 0;
		local a = reaction_color.a or 0;
		
		if frame.reaction_r ~= r
		or frame.reaction_g ~= g
		or frame.reaction_b ~= b
		or frame.reaction_a ~= a
		then
							
			frame.reaction_r = r;
			frame.reaction_g = g;
			frame.reaction_b = b;
			frame.reaction_a = a;
			
			frame.texture:SetVertexColor( r, g, b, 1 );
			frame.texture:SetAlpha( a );
			
		end
		
	end
end
