﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------
-- to do: add cast icon options to the bar
--        add GCD bar

Unit Skinning Options: Casting

class options =
{
	string orient				-- location of the zero value on the bar: LEFT (grows right), RIGHT (grows left), TOP (grows down), BOTTOM (grows up)
	string overlay				-- path to the overlay texture (tranparency) to use on the bar or nil for a flat bar
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	float height				-- height of the bar
	float width					-- width of the bar
	int level					-- the frame level to use (1-9) or nil to use the parent frame's level+1
	boolean show_bar			-- displays the graphic casting bar when true -- setting false allows for text only casting displays
	boolean show_gcd			-- displays the global cooldown bar as an overlay when true
	boolean show_latency		-- displays a latency bar overlapping the cast bar when true
	
	class cur_time 				-- enables current casting time or channel time remaining text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class max_time 				-- enables total casting time or channel time text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class pct_time 				-- enables casting time or channel time remaining as a percent of total time text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class spell_name			-- enables spell name text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class msg_label				-- enables "Interrupted", "Missed" message text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the bar: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class spell_icon			-- enables display of the spell icon with the bar when not nil
	{
		string	anchor_pt		-- point on the icon to use as the origin: LEFT, TOPLEFT, BOTTOM, etc.
		string  relative_pt		-- the point on the bar to use as the origin: LEFT, TOPLEFT, BOTTOM, etc.
		float	xOfs        	-- the horizontal offset of anchor_pt on the icon from relative_pt on the bar
		float	yOfs			-- the vertical offset of the anchor
	}
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local CreateFrame     = CreateFrame;
local GetNetStats     = GetNetStats;
local GetTime         = GetTime;
local UnitCastingInfo = UnitCastingInfo;
local UnitChannelInfo = UnitChannelInfo;

-------------------------------------------------------------------------------
-- default options for the casting bar colors

nUI_DefaultConfig.CastBarColors =
{
	["Casting"]    = { r = 0.9, g = 0.75, b = 0, a = 1, },
	["Channeling"] = { r = 0, g = 0.75, b = 0.25, a = 1, },
	["Latency"]    = { r = 0.1, g = 0.25, b = 1, a = 1, },
};

-------------------------------------------------------------------------------
-- unit casting bar event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Casting", nUI_Unit.Drivers )

local CastingCallbacks    = {};
local CastingUnits        = {};
local CastingStatusList   = {};
local ActiveCastingUnits  = {};
local CastMessages        = {};
local update_timer        = 1 / nUI_DEFAULT_FRAME_RATE;

nUI_Unit.Drivers.Casting  = frame;

-------------------------------------------------------------------------------
-- helper unitility to copy what we know about a cast from one table to another

local function CopyCast( target, source )
	
	target.spell       = source.spell;
	target.rank        = source.rank;
	target.spell_name  = source.spell_name;
	target.icon        = source.icon;
	target.start_time  = source.start_time;
	target.end_time    = source.end_time;
	target.pct_time    = source.pct_time;
	target.cur_time    = source.cur_time;
	target.max_time    = source.max_time;
	target.pct_latency = source.pct_latency;
	target.cur_latency = source.cur_latency;
	target.max_latency = source.max_latency;
	target.active      = source.active;
	target.channeling  = source.channeling;
	target.complete    = source.complete;
	target.stopped     = source.stopped;
	target.interrupted = source.interrupted;
	target.failed      = source.failed;
	target.delayed     = source.delayed;
	target.succeeded   = source.succeeded;
	target.missed      = source.missed;
	target.tradeskill  = source.tradeskill;				
	target.bar_color   = source.bar_color;
end

-------------------------------------------------------------------------------

local function onCastingEvent()
	
	if event == "VARIABLES_LOADED" then
		
		nUI:patchConfig();
		nUI_Unit:configCasting();

	-- we don't care about the event unless there are observers for the unit id
	
	elseif arg1 and CastingCallbacks[arg1] and #CastingCallbacks[arg1] > 0 then

		local unit_info = nUI_Unit:getUnitInfo( arg1 );
		
		if unit_info then
				
			local cast_info = {};
			
			-- starting a new cast spell
			
			if event == "UNIT_SPELLCAST_START" then

				cast_info.spell, cast_info.rank, cast_info.spell_name, 
				cast_info.icon, cast_info.start_time, cast_info.end_time, 				
				cast_info.tradeskill = UnitCastingInfo( arg1 );
				
				if cast_info.spell then
					cast_info.bar_color  = nUI_UnitOptions.CastBarColors["Casting"];		
					cast_info.start_time = cast_info.start_time / 1000;
					cast_info.end_time   = cast_info.end_time / 1000;
				end
				
			-- starting a new channeled spell
			
			elseif event == "UNIT_SPELLCAST_CHANNEL_START" then			
				
				cast_info.spell, cast_info.rank, frame.channel_name, 
				cast_info.icon, cast_info.start_time, cast_info.end_time, 				
				cast_info.tradeskill = UnitChannelInfo( arg1 );				
				
				if cast_info.spell then
					
					cast_info.channeling = true;
					cast_info.bar_color  = nUI_UnitOptions.CastBarColors["Channeling"];		
					cast_info.start_time = cast_info.start_time / 1000;
					cast_info.end_time   = cast_info.end_time / 1000;
					
				end

			-- otherwise, we're updating a spellcast that is already underway
			
			else
				
				-- copy what we already know about the spellcast
				
				local cached = unit_info.cast_info or {};
				
				CopyCast( cast_info, cached );
				
				-- updating time information about the spell
				
				if event == "UNIT_SPELLCAST_DELAYED" then
				
					cast_info.spell, cast_info.rank, cast_info.spell_name, 
					cast_info.icon, cast_info.start_time, cast_info.end_time, 					
					cast_info.tradeskill = UnitCastingInfo( arg1 );			
					
					if cast_info.spell then
						cast_info.start_time = cast_info.start_time / 1000;
						cast_info.end_time   = cast_info.end_time / 1000;
					end
				
				elseif event == "UNIT_SPELLCAST_CHANNEL_UPDATE" then

					cast_info.spell, cast_info.rank, frame.channel_name, 
					cast_info.icon, cast_info.start_time, cast_info.end_time, 
					cast_info.tradeskill = UnitChannelInfo( arg1 );				
					
					if cast_info.spell then
						cast_info.start_time = cast_info.start_time / 1000;
						cast_info.end_time   = cast_info.end_time / 1000;
					end
					
				-- the spellcast has been stopped
				
				elseif event == "UNIT_SPELLCAST_STOP" then
				
					if not cast_info.channeling then
						cast_info.stopped = true;
					end
					
				elseif event == "UNIT_SPELLCAST_CHANNEL_STOP" then
					
					if cast_info.channeling then
						cast_info.stopped = true;
					end
					
				-- spell succeeded
				
				elseif event == "UNIT_SPELLCAST_SUCCEEDED" then
					
					cast_info.succeeded = true;
					
				-- spell failed
				
				elseif event == "UNIT_SPELLCAST_FAILED" then
					
					cast_info.failed = true;
					
				-- spell missed
				
				elseif event == "UNIT_SPELLCAST_SPELLMISS" then
					
					cast_info.missed = true;
					
				-- spell interrupted
				
				elseif event == "UNIT_SPELLCAST_INTERRUPTED" then
					
					cast_info.interrupted = true;
					
				end
			end

			-- process the spellcast update
			
			frame.cast_info = cast_info;
			
			frame.newUnitInfo( arg1, unit_info );
			
			frame.cast_info = nil;
			
		end
	end
end

frame:SetScript( "OnEvent", onCastingEvent );
frame:RegisterEvent( "VARIABLES_LOADED" );
frame:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_START" );
frame:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_STOP" );
frame:RegisterEvent( "UNIT_SPELLCAST_CHANNEL_UPDATE" );
frame:RegisterEvent( "UNIT_SPELLCAST_START" );
frame:RegisterEvent( "UNIT_SPELLCAST_STOP" );
frame:RegisterEvent( "UNIT_SPELLCAST_SUCCEEDED" );
frame:RegisterEvent( "UNIT_SPELLCAST_SPELLMISS" );
frame:RegisterEvent( "UNIT_SPELLCAST_DELAYED" );
frame:RegisterEvent( "UNIT_SPELLCAST_FAILED" );
frame:RegisterEvent( "UNIT_SPELLCAST_INTERRUPTED" );

-------------------------------------------------------------------------------
-- casting bar update event handler

local function onCastingUpdate( who, elapsed )

	if #ActiveCastingUnits > 0 then
		
		update_timer = update_timer - elapsed;
		
		if update_timer <= 0 then -- update casting bars at the user selected frame rate
		
			update_timer = nUI_Unit.frame_rate;

			local update_list = {};
			
			for i=#ActiveCastingUnits, 1, -1 do
					
				local unit_id   = ActiveCastingUnits[i];
				local unit_info = nUI_Unit:getUnitInfo( unit_id );
				local new_data;
				
				if unit_info then
						
					new_data = update_list[unit_info] or nUI_Unit:updateCastingInfo( unit_id, unit_info ) ~= nil;				
					update_list[unit_info] = new_data;
					
				end

				if new_data or not unit_info then 
				
					nUI_Unit:notifyCallbacks( nUI_L["casting bar"], CastingCallbacks, CastingUnits, unit_info, unit_id, new_data );
					
				end
			end
		end
	end
end

frame:SetScript( "OnUpdate", onCastingUpdate );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for casting bar changes GUID

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateCastingInfo( list_unit, unit_info );
	local callbacks = CastingCallbacks;
	local unitlist  = CastingUnits;
	
	nUI_Unit:notifyCallbacks( nUI_L["casting bar"], callbacks, unitlist, unit_info, list_unit, new_data );

end

-------------------------------------------------------------------------------
-- initialize configuration for the casting bar colors
-- 
-- this method is called when the mod's saved variables have been loaded by Bliz and
-- may be called again whenever the casting bar color configuration has been changed
-- by the player or programmatically. Passing true or a non-nil value for "use_default"
-- will cause the player's current cast bar color configuration to be replaced with
-- the default settings defined at the top of this file (which cannot be undone!)

function nUI_Unit:configCasting( use_default )
	
	if not nUI_UnitOptions then nUI_UnitOptions = {}; end
	if not nUI_UnitOptions.CastBarColors then nUI_UnitOptions.CastBarColors = {}; end

	nUI_Unit:configCastBar( "Casting", use_default );
	nUI_Unit:configCastBar( "Channeling", use_default );
	nUI_Unit:configCastBar( "Latency", use_default );

end

-------------------------------------------------------------------------------
-- configure casting bar colors for the named bar

function nUI_Unit:configCastBar( name, use_default )
	
	local config  = nUI_UnitOptions.CastBarColors[name] or {};
	local default = nUI_DefaultConfig.CastBarColors[name];
	
	if use_default then
			
		config.r = default.r;
		config.g = default.g;
		config.b = default.b;
		config.a = default.a;

	else
			
		config.r = tonumber( config.r or default.r );
		config.g = tonumber( config.g or default.g );
		config.b = tonumber( config.b or default.b );
		config.a = tonumber( config.a or default.a );

	end
		
	nUI_UnitOptions.CastBarColors[name] = config;
	
	nUI_Unit:refreshCastingCallbacks();
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of casting bar listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the casting info of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerCastingCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = CastingCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not CastingCallbacks[unit_id] then
			CastingCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		
		if #list == 1 then
			nUI_Unit:registerUnitChangeCallback( unit_id, nUI_Unit.Drivers.Casting );
		end
		
		-- collect casting information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateCastingInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterCastingCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = CastingCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterUnitChangeCallback( unit_id, nUI_Unit.Drivers.Casting );
		end
	end
end

-------------------------------------------------------------------------------
-- update the casting information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateCastingInfo( unit_id, unit_info )

	local modified  = false;
	
	if unit_info then

		local this_cast = frame.cast_info or unit_info.cast_info or {};
		
		-- no longer have a valid spell?
		
		if not this_cast.spell
		or not this_cast.start_time
		or not this_cast.end_time
		or this_cast.complete
		then
			
			if unit_info.is_casting then
					
				modified              = true;
				unit_info.modified    = true;
				unit_info.last_change = GetTime();
				unit_info.is_casting  = false;
				unit_info.cast_info   = nil;			

			end
			
		-- we have a valid spell, do something with it
		
		else
	
			local _,_,latency = GetNetStats();
			local cast_info = {};
			local proc_time = GetTime();
			
			CopyCast( cast_info, this_cast );				
			
			cast_info.max_time    = cast_info.end_time - cast_info.start_time;
			cast_info.cur_time    = proc_time - cast_info.start_time;
			cast_info.pct_time    = max( 0, min( 1, cast_info.cur_time / cast_info.max_time ));
			cast_info.cur_latency = cast_info.cur_time + latency / 1000;
			cast_info.pct_latency = max( 0, min( 1, cast_info.cur_latency / cast_info.max_time ));
			cast_info.max_latency = cast_info.max_time;

			-- if this is a channeled spell, the time is time remaining so we invert the percentage value
			
			if cast_info.channeling then
				
				cast_info.pct_time    = 1 - cast_info.pct_time;
				cast_info.pct_latency = 1 - cast_info.pct_latency;
				
				-- by default, when channeling a spell, the localized spell name you get
				-- from UnitChannelInfo() is "channeling" instead of the spell name... so
				-- here we do some extra work to get the actual name of the spell 

				if not cast_info.spell_name
				or not unit_info.cast_info
				or unit_info.cast_info.spell ~= cast_info.spell
				then
					
					cast_info.spell_name = nUI_L[cast_info.spell_name] or GetSpellInfo( cast_info.spell ) or frame.channel_name;
					
				end
				
			end
			
			-- if the spell has stopped, then mark it complete so we can dequeue it
			-- on the next update... this makes sure that all the listeners get a
			-- notice on the completion before we dequeue it.
			
			cast_info.complete = cast_info.stopped;
			
			-- has the spell been completed?

			cast_info.is_casting =	not cast_info.stopped 
									and not cast_info.interrupted 
									and not cast_info.failed
									and not cast_info.missed;
				
			-- is the spell actively being cast at this time?
			
			cast_info.active =	cast_info.is_casting
								and cast_info.start_time <= proc_time
								and cast_info.end_time >= proc_time;

			-- update the cache if the status of this spell changed
			
			local cached = unit_info.cast_info or {};
			
			if not cached
			or not unit_info.is_casting
			or cached.pct_time    ~= cast_info.pct_time
			or cached.cur_time    ~= cast_info.cur_time
			or cached.max_time    ~= cast_info.max_time
			or cached.pct_latency ~= cast_info.pct_latency
			or cached.cur_latency ~= cast_info.cur_latency
			or cached.max_latency ~= cast_info.max_latency
			or cached.spell       ~= cast_info.spell
			or cached.rank        ~= cast_info.rank
			or cached.failed      ~= cast_info.failed
			or cached.interrupted ~= cast_info.interrupted
			or cached.succeeded   ~= cast_info.succeeded  
			or cached.missed      ~= cast_info.missed     
			or cached.stopped     ~= cast_info.stopped    
			or cached.active      ~= cast_info.active
			or cached.complete    ~= cast_info.complete
			or cached.delayed     ~= cast_info.delayed
			then
				
				modified              = true;
				unit_info.modified    = true;
				unit_info.is_casting  = true;
				unit_info.last_change = proc_time;
				unit_info.cast_info   = cast_info;
				
				-- if this is a new cast, add it to the list of active units
				-- for our casting bar updates
				
				if not CastingStatusList[unit_id] then
	
					nUI:TableInsertByValue( ActiveCastingUnits, unit_id );				
					CastingStatusList[unit_id] = true;
					
				end			
			end
		end
		
		-- if this unit was an active caster then remove it from active cast bar list

		if not unit_info.cast_info and CastingStatusList[unit_id] then
			
			nUI:TableRemoveByValue( ActiveCastingUnits, unit_id );				
			CastingStatusList[unit_id] = false;
		
		end
		
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered casting bar listeners, even if there's no 
-- change in data... typically used when the cast bar color options change
-- or entering the world

function nUI_Unit:refreshCastingCallbacks()

	nUI_Unit:refreshCallbacks( 
	
		nUI_L["casting bar"], CastingCallbacks, CastingUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateCastingInfo( list_unit, unit_info ); 
		end 
	);
	
end

-------------------------------------------------------------------------------
-- create a new unit casting bar frame

function nUI_Unit:createCastingFrame( parent, unit_id, id, options, clickable )

	local frame      = nUI_Unit:createFrame( "$parent_Casting"..(id or ""), parent, unit_id, clickable );	
	frame.bar        = nUI_Bars:createStatusBar( "$parentBar", frame );
	frame.latency    = nUI_Bars:createStatusBar( "$parentLatency", frame );
	frame.bar_anchor = frame:CreateTexture( "$parentAnchor", "ARTWORK" );
	frame.cur        = frame:CreateFontString( "$parentCurrent", "OVERLAY" );			-- shows current casting time as text
	frame.max        = frame:CreateFontString( "$parentMaximum", "OVERLAY" );			-- shows maximum casting time as text
	frame.pct        = frame:CreateFontString( "$parentPercent", "OVERLAY" );			-- shows percent percent of casting time as text
	frame.lbl        = frame:CreateFontString( "$parentSpellName", "OVERLAY" );			-- shows the name of the spell
	frame.msg        = frame.parent:CreateFontString( "$parentMessage", "OVERLAY" );	-- shows alert messages (failed, interrupted, etc)
	frame.Super      = {};

	frame.bar_anchor:SetPoint( "CENTER", frame, "CENTER", 0, 0 );
	
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated

	frame.Super.newUnitInfo = frame.newUnitInfo;	
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateCastingFrame( frame );
		end
		
	end;
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerCastingCallback( frame.unit, frame );
				frame.bar.setEnabled( frame.options.show_bar );
				frame.latency.setEnabled( frame.options.show_latency );
				nUI_Unit:updateCastingFrame( frame );
			else
				nUI_Unit:unregisterCastingCallback( frame.unit, frame );
				frame.bar.setEnabled( false );
				frame.latency.setEnabled( false );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame.

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );

		if frame.options then
				
			frame.font_size = (frame.size or (frame.bar.horizontal and frame.height or frame.width)) * 0.7;		
			
			frame.configText( frame.cur, frame.options.cur_time );
			frame.configText( frame.max, frame.options.max_time );
			frame.configText( frame.pct, frame.options.pct_time );
			frame.configText( frame.lbl, frame.options.spell_name );
			frame.configText( frame.msg, frame.options.msg_label );
	
			if frame.bar_anchor.size   ~= frame.size 
			or frame.bar_anchor.width  ~= frame.width
			or frame.bar_anchor.height ~= frame.height 
			or frame.bar_anchor.inset  ~= frame.inset
			then
	
				local width  = (frame.size or frame.width) - frame.inset;
				local height = (frame.size or frame.height) - frame.inset;
				
				frame.bar_anchor.size   = frame.size;
				frame.bar_anchor.width  = frame.width;
				frame.bar_anchor.height = frame.height;
				frame.bar_anchor.inset  = frame.inset;
							
				frame.bar_anchor:SetWidth( width );
				frame.bar_anchor:SetHeight( height );
				
				if frame.bar.horizontal then
					
					frame.bar:SetWidth( width );
					frame.bar:SetHeight( height * (frame.latency.enabled and 0.7 or 1) );
					
					frame.latency:SetWidth( width );
					frame.latency:SetHeight( height * 0.3 );
					
				else
					
					frame.bar:SetWidth( width * (frame.latency.enabled and 0.7 or 1) );
					frame.bar:SetHeight( height );
					
					frame.latency:SetWidth( width * 0.3 );
					frame.latency:SetHeight( height );
									
				end
			end
		end		
	end
	
	-- overload the frame anchor method to also anchor the bars correctly
	
	frame.Super.applyAnchor = frame.applyAnchor;
	frame.applyAnchor       = function( anchor )
		
		frame.Super.applyAnchor( anchor );
		
		if frame.bar.horizontal then
			
			frame.bar:ClearAllPoints();
			frame.bar:SetPoint( "TOP", frame.bar_anchor, "TOP", 0, 0 );
			
			frame.latency:ClearAllPoints();
			frame.latency:SetPoint( "TOP", frame.bar, "BOTTOM", 0, 0 );
			
		else
			
			frame.bar:ClearAllPoints();
			frame.bar:SetPoint( "LEFT", frame.bar_anchor, "LEFT", 0, 0 );
			
			frame.latency:ClearAllPoints();
			frame.latency:SetPoint( "LEFT", frame.bar, "RIGHT", 0, 0 );
			
		end				
	end

	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.
	
	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		if options then
			frame.bar.setEnabled( options.show_bar );
			frame.latency.setEnabled( options.show_latency );
		end

		frame.Super.applyOptions( options );

		-- extra frame level work
		
		frame.bar:SetFrameStrata( frame:GetFrameStrata() );
		frame.bar:SetFrameLevel( frame:GetFrameLevel() );
		
		frame.latency:SetFrameStrata( frame.bar:GetFrameStrata() );
		frame.latency:SetFrameLevel( frame.bar:GetFrameLevel() );

		-- set the bar texture
		
		if options.overlay then
			frame.bar_anchor:SetAlpha( 1 );
			frame.bar_anchor:SetTexture( options.overlay );
			frame.bar_anchor:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
		else
			frame.bar_anchor:SetAlpha( 0 );
		end
		
		-- enable or disable the display of a graphic casting bar
		
		if not frame.bar.enabled then
			
			frame.bar:SetAlpha( 0 );
			
		else
			
			frame.bar:SetAlpha( 1 );
			frame.bar.setOrientation( options.orient or "LEFT" );
			frame.bar.setBar();
			
		end
		
		-- enable or disable the display of a graphic latency bar
		
		if not frame.latency.enabled then
			
			frame.latency:SetAlpha( 0 );
			
		else
			
			frame.latency:SetAlpha( 1 );
			frame.latency.setOrientation( options.orient or "LEFT" );
			frame.latency.setBar();
			
		end
		
		-- and refresh the frame
		
		nUI_Unit:updateCastingFrame( frame );
		
	end
	
	-- initiate the frame
			
	frame.unit_info = nUI_Unit:registerCastingCallback( frame.unit, frame );
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a unit healt frame

function nUI_Unit:deleteCastingFrame( frame )

	frame.bar.deleteBar();
	
	nUI_Unit:unregisterCastingCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- update the casting bar text and casting bar as required
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateCastingFrame( frame )
	
	local unit_info = frame.unit_info;
	local cast_info = unit_info and unit_info.cast_info or nil;
	local proc_time = GetTime();
	local message   = nil;
	
	-- if we don't have a spell anymore, then shut down the casting bar
	
	if not cast_info 
	or cast_info.end_time < proc_time then
		
		if frame.active then
			
			local persist = frame.options.persist;
			frame.active  = false;
			
			frame:SetAlpha( persist and 1 or 0 );
			frame.bar.updateBar();
			frame.latency.updateBar();
			
		end
	
		if frame.pct.active then
			frame.pct.active = false;
			frame.pct.value  = nil;
			frame.pct:SetText( "" );
		end
		
		if frame.max.active then
			frame.max.active = false;
			frame.max.value  = nil;
			frame.max:SetText( "" );
		end
		
		if frame.cur.active then
			frame.cur.active = false;
			frame.cur.value  = nil;
			frame.cur:SetText( "" );
		end
		
		if frame.lbl.active then
			frame.lbl.active = false;
			frame.lbl.value  = nil;
			frame.lbl:SetText( "" );
		end			
	
	-- do we have a new message to show the user?
	
	elseif cast_info.failed      then message = nUI_L["~FAILED~"];
	elseif cast_info.missed      then message = nUI_L["~MISSED~"];
	elseif cast_info.interrupted then message = nUI_L["~INTERRUPTED~"];
	end

	if message 
	and frame.msg.enabled 
	and frame.msg.value ~= message 
	then
			
		frame.msg.active     = true;
		frame.msg.value      = value;
		frame.msg.start_time = GetTime()+1;
		
--		frame.msg:SetText( message );		
--		nUI:TableInsertByValue( CastMessages, frame.msg );
		
	end
		
	-- if we have an active casting bar, then update it
	
	if cast_info and cast_info.active then

		local pct = ("%0.0f%%"):format( cast_info.pct_time * 100 );
		local max = ("%0.0f"):format( cast_info.max_time );
		local cur = ("%0.0f"):format( cast_info.cur_time );
		local lbl = cast_info.spell_name;

		-- make sure our bar is visible
		
		if not frame.active then
			
			frame.active = true;			
			frame:SetAlpha( 1 );
			
		end
	
		-- update the labels as required

		if frame.pct.enabled and frame.pct.value ~= pct then
			
			frame.pct.active = true;
			frame.pct.value  = pct;
			frame.pct:SetText( pct );
			
		end
		
		if frame.max.enabled and frame.max.value ~= max then
			
			frame.max.active = true;
			frame.max.value  = max;
			frame.max:SetText( max );
			
		end
		
		if frame.cur.enabled and frame.cur.value ~= cur then
			
			frame.cur.active = true;
			frame.cur.value  = cur;			
			frame.cur:SetText( cur );
			
		end
		
		if frame.lbl.enabled and frame.lbl.value ~= lbl then

			frame.lbl.active = true;
			frame.lbl.value  = lbl;
			frame.lbl:SetText( lbl );
			
		end

		-- update the casting bars

		frame.bar.updateBar( 
		
			frame.bar.enabled 
			and cast_info.pct_time ~= 0
			and cast_info.pct_time ~= 1
			and cast_info.pct_time or nil, 
			
			frame.bar.enabled 
			and cast_info.bar_color or nil );
		
		frame.latency.updateBar( 
		
			frame.latency.enabled 
			and cast_info.pct_latency ~= 0 
			and cast_info.pct_latency ~= 1 
			and cast_info.pct_latency or nil, 
			
			frame.latency.enabled 
			and nUI_UnitOptions.CastBarColors["Latency"] or nil 
		);
		
	end
end
