﻿--[[

Copyright (C) 2004 by neuron
aagaande AT gmail DOT com

This program is free software; you can redistribute it and--or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

]]

local nQuestLog = nQuestLog

local REVISION = tonumber(("$Revision: 177 $"):match("%d+"))
if ( nQuestLog.revision < REVISION ) then
	nQuestLog.version = (nQuestLog.versionstring):format(REVISION)
	nQuestLog.revision = REVISION
	nQuestLog.date = ("$Date: 2008-03-28 23:06:38 +0000 (Fri, 28 Mar 2008) $"):match("%d%d%d%d%-%d%d%-%d%d")
end

--Load libraries
local Dewdrop = AceLibrary("Dewdrop-2.0")
local Quixote = AceLibrary("Quixote-1.0")

--Load localization
local L = AceLibrary("AceLocale-2.2"):new("nQuestLog")

--Declare string constants
local DATACOLOR = "|cFF66FF66"
local ENDCOLOR = "|r"
local NEWLINE = "\n"
local WHITE = "|cFFFFFFFF"

function nQuestLog:AddWayPoint(x, y, qTitle, qLevel, npc, c, z, zName)
	if ( TomTom ) or ( Cartographer_Waypoints ) then
		local note = ""
		if ( npc ) then
			note = note..WHITE..npc..ENDCOLOR
		end
		if ( qTitle ) then
			if ( npc ) then
				note = note..NEWLINE
			end
			local qColor = WHITE
			if ( qLevel ) then
				local colorTable = GetDifficultyColor(qLevel)
				if ( colorTable ) then
					qColor = self:RGBPercToHex(colorTable.r or 1, colorTable.g or 1, colorTable.b or 1)
				end
			end
			note = note..self:FormatQuestInfo(qLevel, nil, qTitle, qColor)
		end
		if ( c ) and ( z ) then
			if ( TomTom ) and ( type(TomTom.AddZWaypoint) == "function" ) then
				TomTom:AddZWaypoint(c, z, x, y, note)
			end
		else
			if ( TomTom ) and ( type(TomTom.AddWaypoint) == "function" ) then
				TomTom:AddWaypoint(x, y, note)
			end
		end
		if ( qTitle ) then
			note = note..NEWLINE
		end
		if ( not zName ) then
			zName = GetZoneText()
		end
		note = note..self:FormatZoneByFaction(zName, self:Round(x, 1), self:Round(y, 1))
		if ( Cartographer_Waypoints ) and ( type(Cartographer_Waypoints.AddLHWaypoint) == "function" ) then
			Cartographer_Waypoints:AddLHWaypoint(c, z, x, y, note)
		end
	elseif ( MobMapDotParentFrame ) then
		if ( npc ) and ( type(MobMap_ParseQuestObjective) == "function" ) then
			MobMap_ParseQuestObjective(npc)
		end
	end
end

function nQuestLog:ItemClick(link)
	if ( IsControlKeyDown() ) then
		DressUpItemLink(link)
	elseif ( IsShiftKeyDown() ) then
		ChatFrameEditBox:Show()
		ChatFrameEditBox:Insert(link)
	else
		ShowUIPanel(ItemRefTooltip)
		if ( not ItemRefTooltip:IsVisible() ) then
			ItemRefTooltip:SetOwner(UIParent, "ANCHOR_PRESERVE")
		end
		ItemRefTooltip:SetHyperlink(link)
	end
end

function nQuestLog:ItemHover(link, unsafe, id)
	if ( not link ) then
		return
	end
	if ( unsafe ) then
		--Add GameToolTip warning about unsafe item link
		--Copied from AtlasLoot
		GameTooltip:SetOwner(_G.this, "ANCHOR_PRESERVE")
		GameTooltip:AddLine("|cFFFF0000"..L["Unsafe Item"], nil, nil, nil, 1)
		if ( id ) then
			GameTooltip:AddLine("|cFF0070DD"..L["ItemID:"].." "..id, nil, nil, nil, 1)
		end
		GameTooltip:AddLine(L["unsafeItemWarning"], nil, nil, nil, 1)
		GameTooltip:AddLine(" ")
		GameTooltip:AddLine(L["queryItemWarning"], nil, nil, nil, 1)
	else
		GameTooltip:SetOwner(_G.this, "ANCHOR_PRESERVE")
		GameTooltip:SetHyperlink(link)
		if ( IsShiftKeyDown() ) then
			GameTooltip_ShowCompareItem()
		end
	end
	GameTooltip:Show()
end

function nQuestLog:SendToChat(text, editBox)
	if ( text ) then
		if ( editBox ) or ( ChatFrameEditBox:IsVisible() ) then
			ChatFrameEditBox:Show()
			ChatFrameEditBox:Insert(text)
			ChatFrameEditBox:SetFocus()
			ChatFrameEditBox:HighlightText()
		else
			local chatType = DEFAULT_CHAT_FRAME.editBox:GetAttribute("chatType")
			SendChatMessage(text, chatType)
		end
	end
end

------------------------------------------------------
--		Mouse click events							--
------------------------------------------------------

function nQuestLog:OnQuestLogClick(button, questid, zone)
	--if questid is nil then it's a zone
	if ( IsShiftKeyDown() ) then
		if ( questid ) then
			nQuestLog:OnQuestLogShiftClick(button, questid)
		end
		return
	end
	if ( button == "RightButton" ) and ( IsAltKeyDown() ) then
		self:OnQuestLogAltRightClick()
	elseif ( questid == nil ) or ( button == "RightButton") then
		--We do if button == "RightButton" to open the zone menu on rightclick
		self:OnQuestLogClickZone(button, this.strZone)
	else
		self:OnQuestLogClickQuest(button, questid, zone)
	end
end

function nQuestLog:OnQuestLogAltRightClick()
	local popupMenu = { type = "group", args = {
		header = { order = 1, type = "header", name = L["Zone visibility:"] } } }
	for i, zone in Quixote:IterZones() do
		prettyzone = self:Colorize(zone, "zoneHeader")
		popupMenu.args[zone] = { order = i + 5, type = "toggle",
			name = prettyzone,
			desc = L["Toggle showing"].." "..zone,
			get = function() return not self.db.char.hiddenZones[zone] end,
			set = function(v)
				self.db.char.hiddenZones[zone] = nil
				if ( not v ) then
					self.db.char.hiddenZones[zone] = true
				end
				self:Refresh()
			end }
	end
	Dewdrop:Open( nQuestLogFrame, "cursorX", true, "cursorY", true, "children",
		function()
			Dewdrop:FeedAceOptionsTable(popupMenu)
		end
	)
end

function nQuestLog:OnQuestLogShiftClick(button, questid)
	local title, level, tag, _, _, objectives = Quixote:GetQuestById(questid)
	if ( title == nil ) then
		return
	end
	--Link quest to chat
	local text = "["..level..tag.."] "
	local link = GetQuestLink(questid)
	if ( link ) then
		text = text..link.." "
	else
		text = text..title.." "
	end
	local totalGot, totalNeeded = 0, 0
	local extraText = ""
	if ( objectives > 0 ) then
		for i=1, objectives do
			local desc, _, got, needed = Quixote:GetQuestStatusById(questid, i)
			totalGot = totalGot + got
			totalNeeded = totalNeeded + needed
			if ( button == "RightButton" ) then
				--Link quest objectives
				if ( i > 1 ) then
					extraText = extraText..", "
				end
				extraText = extraText..desc.." "..got.."/"..needed
			end
		end
		if ( extraText ~= "" ) then
			text = text.."("..extraText..") "
		end
	end
	if ( IsControlKeyDown() ) then
		--Link total objectives
		text = text..totalGot.."/"..totalNeeded.." "
	end
	nQuestLog:SendToChat(text)
end

function nQuestLog:OnQuestLogClickZone(button, zone)
	if ( button == "LeftButton" ) then
		--Fold/Unfold zone
		if ( self.db.char.hiddenZones[zone] ) then
			self.db.char.hiddenZones[zone] = nil
		else
			self.db.char.hiddenZones[zone] = true
		end
		self:Refresh()
	elseif ( button == "RightButton" ) and ( not self.db.profile.minionLocked ) then
		--Tracker unlocked on right mouse button, lets show a lock menu
		local popupMenu = { type = "group", args = {
			showHide = { order = 1, type = "execute",
				name = L["Lock minion"],
				desc = L["Minion must be locked for proper functionality!"],
				func = function()
					self.db.profile.minionLocked = true
					Dewdrop:Close()
				end } } }
		Dewdrop:Open( nQuestLogFrame, "cursorX", true, "cursorY", true, "children",
			function() Dewdrop:FeedAceOptionsTable(popupMenu) end )
	elseif ( button == "RightButton" ) then
		--Show popup menu for this zone
		local zoneVisibility = L["Hide zone"]
		if ( self.db.char.hiddenZones[zone] ) then
			zoneVisibility = L["Show zone"]
		end
		zoneVisibility = self:Colorize(zoneVisibility, "zoneHeader")
		local popupMenu = { type = "group", args = {
			showHide = { order = 1, type  = "execute",
				name  = zoneVisibility,
				desc  = L["Toggle zone visibility"],
				func  = function()
					if ( self.db.char.hiddenZones[zone] ) then
						self.db.char.hiddenZones[zone] = nil
					else
						self.db.char.hiddenZones[zone] = true
					end
					self:Refresh()
					Dewdrop:Close()
				end } } }
		for _, id in Quixote:QuestsByZone(zone) do
			local title, level, tag, iGroup, complete, iObjectives = Quixote:GetQuestById(id)
			local objective = Quixote:GetQuestText(id)
			local _, got, needed = self:Format_QuestObjectives(title, iObjectives, objective, id)
			local formattedTitle = self:Format_QuestTitle(title, level, tag, got, needed, complete)
			popupMenu.args[title] = {
				order = id + 5,
				name  = formattedTitle,
				type  = "toggle",
				desc  = L["Toggle showing"].." "..title,
				get = function() return self.db.char.watchedQuests[title] end,
				set = function(v)
					self.db.char.watchedQuests[title] = nil
					if ( v ) then
						self.db.char.watchedQuests[title] = true
					end
					self:Refresh()
				end }
		end
		Dewdrop:Open( nQuestLogFrame, "cursorX", true, "cursorY", true, "children",
			function() Dewdrop:FeedAceOptionsTable(popupMenu) end )
	end
end

function nQuestLog:OnQuestLogClickQuest(button, questid, zone)
	local title, level, _, _, _, objectives = Quixote:GetQuestById(questid)
	if ( title == nil ) then
		return
	end
	if ( button == "LeftButton" ) then
		if ( IsAltKeyDown() ) then
			--Toggle objective visibility
			if ( self:ShouldShowObjective(title) ) then
				self.db.char.watchedObjectives[title] = false
				if ( self:IsModuleEnabled("SmartVisibility") ) then
					self:GetModule("SmartVisibility"):OnQuest_Lost(title)
				end
			else
				self.db.char.watchedObjectives[title] = true
			end
			self:Refresh()
		else
			self:ShowLog(questid)
		end
	end
end

------------------------------------------------------
--		Frame visibility							--
------------------------------------------------------

function nQuestLog:ExpandCollapse()
	local function toggleZones(var)
		for _, zone in Quixote:IterZones() do
			self.db.char.hiddenZones[zone] = var
		end
	end
	local Z, subZ = GetRealZoneText(), GetSubZoneText()
	if ( self.db.char.hiddenZones[subZ] ) then
		self.db.char.hiddenZones[subZ] = nil
	elseif ( self.db.char.hiddenZones[Z] ) then
		self.db.char.hiddenZones[Z] = nil
	else
		for _, zone in Quixote:IterZones() do
			if ( self.db.char.hiddenZones[zone] ) then
				toggleZones()
				self:Refresh()
				return
			end
		end
		toggleZones(true)
	end
	self:Refresh()
end

function nQuestLog:Fade(fadeIn)
	local db = self.db.profile
	if (fadeIn) then
		self:CancelScheduledEvent("nQuestLog_FadeOut")
		if ( nQuestLogFrame:GetAlpha() < db.minionFadeIn ) and ( self.isFading ~= "in" ) then
			local alphaStart, alphaCurrent, alphaEnd = db.minionFadeOut, nQuestLogFrame:GetAlpha(), db.minionFadeIn
			local alphaRange, fadeRemaining = alphaEnd - alphaStart, 0
			if ( alphaRange > 0 ) then
				fadeRemaining = ( alphaEnd - alphaCurrent ) / alphaRange
			end
			local fadeTimer = db.minionFadeInTime * fadeRemaining
			self.isFading = "in"
			UIFrameFadeIn(nQuestLogFrame, fadeTimer, alphaCurrent, alphaEnd)
			nQuestLogFrame.fadeInfo.finishedFunc = function() nQuestLog.isFading = "" end
		end
	else
		if ( nQuestLogFrame:GetAlpha() > db.minionFadeOut ) and ( self.isFading ~= "out" ) then
			local alphaStart, alphaCurrent, alphaEnd = db.minionFadeIn, nQuestLogFrame:GetAlpha(), db.minionFadeOut
			local alphaRange, fadeRemaining = alphaStart - alphaEnd, 0
			if ( alphaRange > 0 ) then
				fadeRemaining = ( alphaCurrent - alphaEnd ) / alphaRange
			end
			local fadeTimer = db.minionFadeOutTime * fadeRemaining
			self:ScheduleEvent("nQuestLog_FadeOut", function ()
				self.isFading = "out"
				UIFrameFadeOut(nQuestLogFrame, fadeTimer, alphaCurrent, alphaEnd)
				nQuestLogFrame.fadeInfo.finishedFunc = function() nQuestLog.isFading = "" end
			end, db.minionFadeOutDelay)
		end
	end
end

function nQuestLog:ShowLog(questid)
	--Show details frame
	if ( self:IsModuleEnabled("DetailsFrame") ) then
		self:GetModule("DetailsFrame"):ShowLog(questid)
	else
		local wasHidden = not QuestLogFrame:IsVisible()
		HideUIPanel(QuestLogFrame)
		if ( GetQuestLogSelection() ~= questid ) or ( wasHidden ) then
			SelectQuestLogEntry(questid)
			QuestLog_SetSelection(questid)
			ShowUIPanel(QuestLogFrame)
			if ( LightHeaded ) then
				local link = GetQuestLink(questid)
				if ( link ) then
					local qid = tonumber(link:match(":(%d+):"))
					if ( LightHeaded.db.profile.singlepage ) then
						LightHeaded:UpdateFrame(qid, nil)
					else
						local lhframe = LightHeadedFrameSub
						lhframe.current_page = 1
						LightHeaded:UpdateFrame(qid, lhframe.current_page)
					end
				end
			end
		end
	end
end

------------------------------------------------------
--		Quest events								--
------------------------------------------------------

function nQuestLog:OnQuestLog_Ready()
	local newDb = true
	--Lets clean the databases as well:
	local function CleanUp(database)
		for name in pairs(database) do
			if ( not Quixote:GetQuestByName(name) ) then
				database[name] = nil
			end
			newDb = false
		end
	end
	local function CleanUpZones()
		for name in pairs(self.db.char.hiddenZones) do
			local zones = select(2, Quixote:QuestsByZone(name))
			if ( not zones ) then
				self.db.char.hiddenZones[name] = nil
			end
			newDb = false
		end
	end
	CleanUp(self.db.char.watchedQuests)
	CleanUp(self.db.char.watchedObjectives)
	CleanUpZones()
	if ( newDb ) then
		--This has got to be a new user, lets add all his quests
		for _, id in Quixote:QuestsByLevel() do
			local title = Quixote:GetQuestById(id)
			self.db.char.watchedQuests[title] = true
		end
		if ( self:IsModuleEnabled("SmartVisibility") ) then
			self:GetModule("SmartVisibility"):OnZoneChange()
		end
		if ( self:IsModuleEnabled("Highlight") ) then
			self:GetModule("Highlight"):OnZoneChange()
		end
	end
	--Refresh and show db.
	self.refreshHold = nil
	if ( self.db.char.visible ) then
		self:ToggleVisible(true)
	end
	if ( nQuestLogFuBar ) then
		nQuestLogFuBar:UpdateText()
	end
end

function nQuestLog:OnQuest_Gained(title)
	if ( self.db.profile.trackNewQuests ) then
		self.db.char.watchedQuests[title] = true
	end
	if ( nQuestLogFuBar ) then
		nQuestLogFuBar:UpdateText()
	end
end

function nQuestLog:OnQuest_Lost(title)
	self.db.char.watchedQuests[title] = nil
	self.db.char.watchedObjectives[title] = nil
	if ( nQuestLogFuBar ) then
		nQuestLogFuBar:UpdateText()
	end
end

function nQuestLog:OnQuest_Leaderboard_Update(title, _, _, desc)
	if ( self.db.profile.trackWhenLooting ) and ( not self.db.char.watchedQuests[title] ) then
		self.db.char.watchedQuests[title] = true
		self:Print("Found "..desc.." for "..title.." adding to quest tracker.")
	end
end

------------------------------------------------------
--		Refresh checker								--
------------------------------------------------------

function nQuestLog:OnRefreshCheck()
	--Modules that needs refresh and that triggers any zone/quest event should not do a manual update, but set this flag instead.
	if ( self.refresh ) then
		self:Refresh()
	end
end
