﻿--[[

Copyright (C) 2004 by neuron
aagaande AT gmail DOT com

This program is free software; you can redistribute it and--or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

]]

local nQuestLog = nQuestLog

local REVISION = tonumber(("$Revision: 165 $"):match("%d+"))
if ( nQuestLog.revision < REVISION ) then
	nQuestLog.version = (nQuestLog.versionstring):format(REVISION)
	nQuestLog.revision = REVISION
	nQuestLog.date = ("$Date: 2007-12-17 02:55:24 +0000 (Mon, 17 Dec 2007) $"):match("%d%d%d%d%-%d%d%-%d%d")
end

local moduleName = "Tooltips"
local Tooltips = nQuestLog:NewModule(moduleName, "AceConsole-2.0", "AceEvent-2.0", "AceHook-2.1")

--Load libraries
local Quixote = AceLibrary("Quixote-1.0")

--Load localization
local L = AceLibrary("AceLocale-2.2"):new("Tooltips")

local MobMap_Installed = false
--local MobInfo2_Installed = false --Not currently in use

function Tooltips:OnModuleLoad()
	self.menuName = L["Tooltips"]
	self.description = L["Description"]
	if ( not self.defaultsRegistered ) then
		self.core:RegisterDefaults(moduleName, "profile", { enabled = true,
			tooltipAnchor = "DEFAULT", itemToolTip = true, trimToolTip = 0,
			mobToolTip = true, mobToolTipTrigger = "both", disableInRaidInstance = true,
			useMobMap = true, MobMapDropRateFilter = 0.01, addMobMapDropRates = true, addKillsNeeded = true })
		self.defaultsRegistered = true
	end
	if ( MobMapDotParentFrame ) then
		MobMap_Installed = true
		if ( self.db.profile.useMobMap ) then
			self:MobMap_Enable()
		end
		--Used to cache mob names and their associated loot in memory, thus reducing lag
		self.mobNameCache, self.mobLootCache = {}, {}
	end
end

function Tooltips:ToggleEnabled(v)
	self.db.profile.enabled = v
	self:ToggleItemToolTip()
	self:ToggleMobToolTip()
end

function Tooltips:GetMenu(oldMenu)
	local menu = { order = oldMenu.order, type = "group",
		name = self.menuName, desc = self.description,
		cmdHidden = true,
		args = {
			toggleEnabled = oldMenu,
			spacer = { order = oldMenu.order + 1, type = "header" },
			tooltipAnchor = { order = oldMenu.order + 2, type = "text",
				name = L["tooltipAnchor"], desc = L["tooltipAnchorDesc"],
				validate = { DEFAULT = L["DEFAULT"], ANCHOR_TOPRIGHT = L["TOPRIGHT"], ANCHOR_RIGHT = L["RIGHT"],
					ANCHOR_BOTTOMRIGHT = L["BOTTOMRIGHT"], ANCHOR_TOPLEFT = L["TOPLEFT"],
					ANCHOR_LEFT = L["LEFT"], ANCHOR_BOTTOMLEFT = L["BOTTOMLEFT"] },
				get = function() return self.db.profile.tooltipAnchor end,
				set = function(v) self.db.profile.tooltipAnchor = v end },
			itemToolTip = { order = oldMenu.order + 3, type = "toggle",
				name = L["itemToolTip"], desc = L["itemToolTipDesc"],
				get = function() return self.db.profile.itemToolTip end,
				set = function(v)
					self.db.profile.itemToolTip = v
					self:ToggleItemToolTip()
				end },
			mobToolTipHeader = { order = oldMenu.order + 4, type = "group",
				name = L["mobToolTip"], desc = L["mobToolTipDesc"],
				args = {
					mobToolTip = { order = 1, type = "toggle",
						name = L["Enabled"], desc = L["mobToolTipDesc"],
						get = function() return self.db.profile.mobToolTip end,
						set = function(v)
							self.db.profile.mobToolTip = v
							self:ToggleMobToolTip()
						end },
					mobToolTipTrigger = { order = 2, type = "text",
						name = L["mobToolTipTrigger"], desc = L["mobToolTipTriggerDesc"],
						validate = { both = L["mobToolTipTriggerBoth"], mobOnly = L["mobToolTipTriggerMob"] },
						get = function() return self.db.profile.mobToolTipTrigger end,
						set = function(v)
							self.db.profile.mobToolTip = false
							self:ToggleMobToolTip()
							self.db.profile.mobToolTip = true
							self.db.profile.mobToolTipTrigger = v
							self:ToggleMobToolTip()
						end },
					disableInRaidInstance = { order = 3, type = "toggle",
						name = L["disableInRaidInstance"], desc = L["disableInRaidInstanceDesc"],
						get = function() return self.db.profile.disableInRaidInstance end,
						set = function(v) self.db.profile.disableInRaidInstance = v end },
					useMobMap = { order = 4, type = "toggle",
						name = L["useMobMap"], desc = L["useMobMapDesc"],
						disabled = ( not MobMap_Installed ),
						get = function() return self.db.profile.useMobMap end,
						set = function(v)
							self.db.profile.useMobMap = v
							self:MobMap_Enable(v)
						end },
					MobMapDropRateFilter = { order = 5, type = "range",
						name = "- "..L["MobMapDropRateFilter"], desc = L["MobMapDropRateFilterDesc"],
						min = 0, max = 1, step = 0.01, isPercent = true,
						disabled = function() return not self.db.profile.useMobMap end,
						hidden = ( not MobMap_Installed ),
						get = function() return self.db.profile.MobMapDropRateFilter end,
						set = function(v) self.db.profile.MobMapDropRateFilter = v end },
					addMobMapDropRates = { order = 6, type = "toggle",
						name = "- "..L["addMobMapDropRates"], desc = L["addMobMapDropRatesDesc"],
						disabled = function() return not self.db.profile.useMobMap end,
						hidden = ( not MobMap_Installed ),
						get = function() return self.db.profile.addMobMapDropRates end,
						set = function(v) self.db.profile.addMobMapDropRates = v end },
					addKillsNeeded = { order = 7, type = "toggle",
						disabled = function() return not self.db.profile.useMobMap end,
						name = "- "..L["addKillsNeeded"], desc = L["addKillsNeededDesc"],
						hidden = ( not MobMap_Installed ),
						get = function() return self.db.profile.addKillsNeeded end,
						set = function(v) self.db.profile.addKillsNeeded = v end } } },
			trimToolTip = { order = oldMenu.order + 5, type = "range",
				name = L["trimToolTip"], desc = L["trimToolTipDesc"],
				min = 0, max = 100, step = 1,
				get = function() return self.db.profile.trimToolTip end,
				set = function(v) self.db.profile.trimToolTip = v end } } }
	
	return menu
end

function Tooltips:ToggleItemToolTip()
	if ( self.db.profile.itemToolTip ) and ( not self.itemHooked ) then
		self.itemHooked = true
		self:HookScript(GameTooltip, "OnTooltipSetItem")
	elseif ( not self.db.profile.itemToolTip ) and ( self.itemHooked ) then
		self.itemHooked = nil
		self:Unhook(GameTooltip, "OnTooltipSetItem")
	end
end

function Tooltips:ToggleMobToolTip()
	if ( self.db.profile.mobToolTip ) and ( not self.mobHooked ) then
		self.mobHooked = true
		if ( self.db.profile.mobToolTipTrigger == "both" ) then
			self:HookScript(GameTooltip, "OnTooltipSetUnit")
		else
			self:RegisterEvent("UPDATE_MOUSEOVER_UNIT", "OnTooltipSetUnit")
		end
	elseif ( not self.db.profile.mobToolTip ) and ( self.mobHooked ) then
		self.mobHooked = nil
		if ( self.db.profile.mobToolTipTrigger == "both" ) then
			self:Unhook(GameTooltip, "OnTooltipSetUnit")
		else
			self:UnregisterEvent("UPDATE_MOUSEOVER_UNIT", "OnTooltipSetUnit")
		end
	end
end

function Tooltips:MobMap_Enable()
	if ( MobMap_LoadDatabase ) then
		if ( not IsAddOnLoaded("MobMapDatabaseStub"..MOBMAP_DROP_DATABASE) ) then
			MobMap_LoadDatabase(MOBMAP_DROP_DATABASE)
		end
		if ( not IsAddOnLoaded("MobMapDatabaseStub"..MOBMAP_MOBNAME_DATABASE) ) then
			MobMap_LoadDatabase(MOBMAP_MOBNAME_DATABASE)
		end
	end
	if ( not MobMap_GetIDForMobName ) or ( not MobMap_GetMobLootTable ) then
		MobMap_Installed = false
	end
end

function Tooltips:MobMap_AddInfo(mobName)
	local mobIsCached, mobLootCacheStart, mobLootCacheEnd = false, 0, 0
	--Check to see if the mob has been cached in memory.  If found, set mobIsCached to true and
	--set the starting and ending indexes for the mobLootCache table stored in memory for that mob
	if ( #(self.mobNameCache) > 0 ) then
		for mobIndex=1, #(self.mobNameCache), 1 do
			if ( self.mobNameCache[mobIndex].mobName == mobName ) then
				mobIsCached = true
				mobLootCacheStart = self.mobNameCache[mobIndex].lootIndexStart
				mobLootCacheEnd = self.mobNameCache[mobIndex].lootIndexEnd
			end
		end
	end
	local lootTable, lootTableRow = {}, 1
	--If mob is cached in memory, get the cached loot table
	--Otherwise, get it from MobMap and add it to the cache
	if ( mobIsCached ) then
		for lootIndex=mobLootCacheStart, mobLootCacheEnd, 1 do
			--Filter loot table and remove items whose drop rates are lower than the threshold chosen (default 1%)
			if ( (self.mobLootCache[lootIndex].chance / 100) >= self.db.profile.MobMapDropRateFilter ) then
				lootTable[lootTableRow] = self.mobLootCache[lootIndex]
				lootTableRow = lootTableRow + 1
			end
		end
	else
		local mobID = MobMap_GetIDForMobName(mobName)
		if ( mobID ) then
			local mobLoot = MobMap_GetMobLootTable(mobID)
			if ( mobLoot ) then
				local lootIndexStart, lootIndexEnd = 1, 1
				if ( #(self.mobLootCache) > 0 ) then
					lootIndexStart = #(self.mobLootCache) + 1
				end
				lootIndexEnd = lootIndexStart + #(mobLoot) - 1
				--Filter loot table and remove items whose drop rates are lower than the threshold chosen (default 1%)
				for lootIndex=1, #(mobLoot), 1 do
					if ( (mobLoot[lootIndex].chance / 100) >= self.db.profile.MobMapDropRateFilter ) then
						lootTable[lootTableRow] = mobLoot[lootIndex]
						lootTableRow = lootTableRow + 1
					end
					--Add mob's loot table to the cache stored in memory
					tinsert(self.mobLootCache, mobLoot[lootIndex])
				end
				--Add mob's name and loot table start/end rows to the cache stored in memory
				tinsert(self.mobNameCache, {mobName = mobName, lootIndexStart = lootIndexStart, lootIndexEnd = lootIndexEnd})
			end
		end
	end
	--If the mob's loot table contains any items after filtering
	if ( #(lootTable) > 0 ) then
		--Iterate through each quest, and then through each objective
		for qID, _, _, _, _, _, qObjs in Quixote:IterateQuests() do
			for oIndex=1, qObjs do
				local oDesc, oType, oGot, oNeeded = Quixote:GetQuestStatusById(qID, oIndex)
				--Only process objectives of type "item" or "object"
				if ( oType == "item" ) or ( oType == "object" ) then
					local MobMapMatch
					--Iterate through mob's loot item table
					for lootIndex=1, #(lootTable), 1 do
						--If loot item matches quest objective
						if ( lootTable[lootIndex].itemname == oDesc ) then
							local dropRate, dropChance = "", lootTable[lootIndex].chance
							--Add MobMap item drop rate, if enabled
							if ( self.db.profile.addMobMapDropRates ) and ( not self.db.profile.addKillsNeeded ) and ( dropChance ) then
								dropRate = " ("..dropChance.."%)"
							end
							if ( self.db.profile.trimToolTip > 0 ) and ( strlen(oDesc) > self.db.profile.trimToolTip ) then
								oDesc = strsub(oDesc, 1, self.db.profile.trimToolTip)
								oDesc = oDesc.."..."
							end
							local lineLeft = self.core:Colorize(oDesc..dropRate, "objectiveProgress", oGot/oNeeded)
							local lineRight = self.core:Colorize(oGot.."/"..oNeeded, "objectiveProgress", oGot/oNeeded)
							--Add objective, drop rate (if enabled), and completion status, colored by % complete, to tooltip
							GameTooltip:AddDoubleLine(lineLeft, lineRight)
							if ( self.db.profile.addKillsNeeded ) and ( dropChance ) then
								local killsNeeded = ceil((oNeeded - oGot) / (dropChance / 100))
								--Add MobMap item drop rate, if enabled
								if ( self.db.profile.addMobMapDropRates ) then
									dropRate = " @ "..dropChance.."%"
								end
								lineLeft = self.core:Colorize("- "..L["killsNeeded"]..dropRate, "objectiveProgress", oGot/oNeeded)
								lineRight = self.core:Colorize(killsNeeded, "objectiveProgress", oGot/oNeeded)
								GameTooltip:AddDoubleLine(lineLeft, lineRight)
							end
							MobMapMatch = true
							break
						end
					end
					--If objective isn't in mob's loot table in MobMap
					if ( not MobMapMatch ) then
						self:MatchMobFromObjective(mobName, oDesc, oGot, oNeeded)
					end
				end
			end
		end
	end
end

--[[	--Most of the code for this function came from QuestIon
		--Not currently in use, due to MobMap
local function GetMI2DropChance(name, level, oDesc)
	local mobData = MI2_GetMobData(name, level)
	local combined = {}
	local dropChance = ""
	MI2_AddTwoMobs(combined, mobData)
	for levelToCombine = level - 4, level + 4, 1 do
		if ( levelToCombine ~= level ) then
			if ( MobInfoDB[name..":"..level] ) then
				local dataToCombine = MI2_GetMobData(name, levelToCombine)
				MI2_AddTwoMobs(combined, dataToCombine)
			end
		end
	end
	mobData = combined
	if ( not mobData.itemList ) then
		mobData.itemList = {}
	end
	for k, v in pairs(mobData.itemList) do
		local itemName = GetItemInfo(k)
		if ( itemName ) then
			if ( strfind(oDesc, itemName) ) then
				dropChance = v / mobData.loots * 100
				if ( dropChance < 1 ) then
					dropChance = strsub(tostring(dropChance), 1, 4).."%"
				elseif ( dropChance < 10 ) then
					dropChance = strsub(tostring(dropChance), 1, 3).."%"
				elseif ( dropChance == 100 ) then
					dropChance = dropChance.."%"
				else
					dropChance = strsub(tostring(dropChance), 1, 2).."%"
				end
				dropChance = "["..dropChance.."] "
			end
		end
	end
	return dropChance
end ]]

--Mouseover tooltips

function Tooltips:OnTooltipSetUnit(tooltip, ...)
	local instanceType = select(2, IsInInstance())
	--Disable tooltip parsing while in a raid instance, unless the option is unchecked in the menu
	if ( instanceType ~= "raid" ) or ( not self.db.profile.disableInRaidInstance ) then
		local mobName, unit
		if ( tooltip ) then
			mobName, unit = tooltip:GetUnit()
		end
		if ( not unit ) then
			unit = "mouseover"
			mobName = UnitName(unit)
		end
		--Make sure that the unit isn't a player and is attackable before processing quest objectives
		if ( mobName ) and ( not UnitPlayerControlled(unit) ) and ( UnitCanAttack("player", unit) ) then
			if ( Quixote:IsQuestMob(mobName) ) then
				for _, qName, qGot, qNeeded in Quixote:IterateQuestsForMob(mobName) do
					local qLevel, _, qGroup = select(2, Quixote:GetQuestByName(qName))
					local qLevelG = qLevel
					if ( qGroup > 0 ) then
						qLevelG = qLevel.."G"..qGroup
					end
					if ( self.db.profile.trimToolTip > 0 ) and ( strlen(qName) > self.db.profile.trimToolTip ) then
						qName = strsub(qName, 1, self.db.profile.trimToolTip)
						qName = qName.."..."
					end
					lineLeft = self.core:Colorize("["..qLevelG.."] "..qName, "questHeader", nil, nil, qLevel)
					lineRight = self.core:Colorize(qGot.."/"..qNeeded, "objectiveProgress", qGot/qNeeded)
					GameTooltip:AddDoubleLine(lineLeft, lineRight)
				end
			end
			if ( MobMap_Installed ) and ( self.db.profile.useMobMap ) then
				self:MobMap_AddInfo(mobName)
			else
				--Iterate through each quest, and then through each objective
				for qID, _, _, _, _, _, qObjs in Quixote:IterateQuests() do
					for oIndex=1, qObjs do
						local oDesc, oType, oGot, oNeeded = Quixote:GetQuestStatusById(qID, oIndex)
						--Only process objectives of type "item" or "object"
						if ( oType == "item" ) or ( oType == "object" ) then
							self:MatchMobFromObjective(mobName, oDesc, oGot, oNeeded)
						end
					end
				end
			end
		end
	end
	if ( tooltip ) then
		return self.hooks[tooltip].OnTooltipSetUnit(tooltip, ...)
	else
		GameTooltip:Show()
	end
end

function Tooltips:MatchMobFromObjective(mobName, oDesc, oGot, oNeeded)
	--Look for mob name in quest objective, return start and end positions if found
	local startPos, endPos = strfind(oDesc, mobName)
	--If quest objective starts with mob name
	if ( startPos == 1 ) then
		--Ensures proper matching by checking for a space with or without possessive characters after mob name
		if ( strfind(oDesc, mobName.." ") ) or ( strfind(oDesc, mobName.."' ") ) or ( strfind(oDesc, mobName.."'s ") ) then
			--Check if any spaces remain after the space after the mob name
			local spaceRemains = strfind(oDesc, "%s", endPos + 2)
			--If no spaces remain (meaning only one word is after the mob name)
			--We can safely assume that this objective is related to the current mob
			if ( not spaceRemains ) then
				if ( self.db.profile.trimToolTip > 0 ) and ( strlen(oDesc) > self.db.profile.trimToolTip ) then
					oDesc = strsub(oDesc, 1, self.db.profile.trimToolTip)
					oDesc = oDesc.."..."
				end
				--Add objective and completion status, colored by % complete, to tooltip
				lineLeft = self.core:Colorize(oDesc, "objectiveProgress", oGot/oNeeded)
				lineRight = self.core:Colorize(oGot.."/"..oNeeded, "objectiveProgress", oGot/oNeeded)
				GameTooltip:AddDoubleLine(lineLeft, lineRight)
			end
		end
	end
end

function Tooltips:OnTooltipSetItem(tooltip, ...)
	local item = tooltip:GetItem()
	if ( not item ) then return end
	if ( Quixote:IsQuestItem(item) ) then
		for qID, qTitle, qLevel, _, qGroup, _, qObjs in Quixote:IterateQuests() do
			for oIndex=1, qObjs do 
				local oDesc, _, oGot, oNeeded = Quixote:GetQuestStatusById(qID, oIndex)
				if ( oDesc == item ) then
					local qLevelG = qLevel
					if ( qGroup > 0 ) then
						qLevelG = qLevel.."G"..qGroup
					end
					if ( self.db.profile.trimToolTip > 0 ) and ( strlen(qTitle) > self.db.profile.trimToolTip ) then
						qTitle = strsub(qTitle, 1, self.db.profile.trimToolTip)
						qTitle = qTitle.."..."
					end
					lineLeft = self.core:Colorize("["..qLevelG.."] "..qTitle, "questHeader", nil, nil, qLevel)
					lineRight = self.core:Colorize(oGot.."/"..oNeeded, "objectiveProgress", oGot/oNeeded)
					tooltip:AddDoubleLine(lineLeft, lineRight)
					tooltip:Show()
					break
				end
			end
		end
	end
	return self.hooks[tooltip].OnTooltipSetItem(tooltip, ...)
end

-- Minion tooltips

function Tooltips:ShowUnlockedToolTip()
	if ( self.db.profile["tooltipAnchor"] == "DEFAULT" ) then
		GameTooltip_SetDefaultAnchor(GameTooltip, nQuestLogFrame)
	else
		GameTooltip:SetOwner(nQuestLogFrame, self.db.profile.tooltipAnchor)
	end
	GameTooltip:SetText(L["MinionUnlocked"])
	GameTooltip:AddLine(L["PlaceMinion"], 1, 1, 1, 1)
	GameTooltip:Show()
end

function Tooltips:ShowZoneToolTip(zone)
	if ( self.db.profile["tooltipAnchor"] == "DEFAULT" ) then
		GameTooltip_SetDefaultAnchor(GameTooltip, nQuestLogFrame)
	else
		GameTooltip:SetOwner(nQuestLogFrame, self.db.profile.tooltipAnchor)
	end
	--Set Title text
	GameTooltip:SetText(zone)
	
	local function showZoneQuests(showHidden)
		local quests, nQuests = {}, 0
		for _, iQuestId in Quixote:QuestsByZone(zone) do
			local title, iLevel, strTag, iGroup, iComplete, iObjectives = Quixote:GetQuestById(iQuestId)
			local strObjective = Quixote:GetQuestText(iQuestId)
			if (( showHidden ) and ( not self.core.db.char.watchedQuests[title] ))
			or (( not showHidden ) and ( self.core.db.char.watchedQuests[title] )) then
				local objectives, iTotalGot, iTotalNeeded = self.core:Format_QuestObjectives(title, iObjectives, strObjective, iQuestId, true)
				title = self.core:Format_QuestTitle(title, iLevel, strTag, iTotalGot, iTotalNeeded, iComplete)
				nQuests = nQuests + 1
				quests[title] = true
			end
		end
		return quests, nQuests
	end
	local function printQuests(quests)
		for name in pairs(quests) do
			GameTooltip:AddLine(name, 1, 0.5, 0.5, 0)
		end
	end
	
	if ( self.core.db.char.hiddenZones[zone] ) then
		GameTooltip:AddLine("  -  "..L["ZoneHidden"], 1, 1, 1, 0)
	end
	local quests, nQuests = showZoneQuests(false)
	if ( nQuests > 0 ) then
		GameTooltip:AddLine(L["TrackedQuests"].." ("..nQuests..")", 1, 0.5, 0.5, 0)
		printQuests(quests)
	end
	quests, nQuests = showZoneQuests(true)
	if ( nQuests > 0 ) then
		GameTooltip:AddLine(L["HiddenQuests"].." ("..nQuests..")", 1, 0.5, 0.5, 0)
		printQuests(quests)
	end
	GameTooltip:AddLine("\n"..L["ZoneTips"], 1, 1, 1, 0)
	GameTooltip:Show()
end

function Tooltips:ShowQuestToolTip(iQuestId)
	--TODO: layout options, ShowObjectives,ShowAll,ShowShort?
	--TODO: maybe for example show all quest data on modifier key?
	local title, iLevel, strTag, iGroup, _, iObjectives, zone = Quixote:GetQuestById(iQuestId)
	local strObjective = Quixote:GetQuestText(iQuestId)
	
	if ( title == nil ) then return end
	if ( self.db.profile["tooltipAnchor"] == "DEFAULT" ) then
		GameTooltip_SetDefaultAnchor(GameTooltip, nQuestLogFrame)
	else
		GameTooltip:SetOwner(nQuestLogFrame, self.db.profile["tooltipAnchor"])
	end
	--Set Title text
	GameTooltip:SetText(title)
	--Add objective description
	GameTooltip:AddLine(self.core:Colorize(strObjective, "objectiveText"), 0.5, 0.5, 0.5, 1)
	--Add progress
	local objectives, iTotalGot, iTotalNeeded = self.core:Format_QuestObjectives(title, iObjectives, strObjective, iQuestId, true)
	if ( iTotalNeeded > 0 ) then
		objectives = "\n"..L["Total"].." : "..self.core:Colorize(iTotalGot.."/"..iTotalNeeded, "questProgress", iTotalGot/iTotalNeeded)..objectives
	end
	GameTooltip:AddLine(objectives, 1, 1, 1, 0)
	--Add who in party has the quest
	local ipartyMembers = GetNumPartyMembers()
	if ( ipartyMembers > 0 ) then
		GameTooltip:AddLine("\n", 1, 1, 1, 0)
		for i=1, ipartyMembers do
			if ( IsUnitOnQuest(iQuestId, "party"..i) ) then
				--Party member is on the quest
				local strName = UnitName("party"..i)
				
				--Lets see if we can get quest progress too
				local iPartyTotal = 0
				if ( iObjectives > 0 ) then
					for i=1, iObjectives do
						local strDesc, strType, iGot, iNeeded = Quixote:GetQuestStatusById(iQuestId, i)
						local iPartyGot = Quixote:GetPartyQuestStatus(strName, iQuestId, strDesc)
						if ( iPartyGot ) then
							if ( strType == "reputation" ) then 
								if ( iNeeded == iPartyGot ) then
									iPartyTotal = iPartyTotal + 1
								end
							else
								iPartyTotal = iPartyTotal + iPartyGot
							end
						end
					end
					strName = strName.." "..self.core:Colorize(iPartyTotal.."/"..iTotalNeeded, "objectiveProgress", iPartyTotal/iTotalNeeded)
				end
				GameTooltip:AddLine(strName, 0, 1, 0, 0)
			else
				--Party member is not on the quest
				GameTooltip:AddLine(UnitName("party"..i), 1, 0, 0, 0)
			end
		end
	end
	GameTooltip:AddLine("\n"..L["QuestTips"], 1, 1, 1, 0)
	GameTooltip:Show()
end
