﻿--[[

Copyright (C) 2004 by neuron
aagaande AT gmail DOT com

This program is free software; you can redistribute it and--or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

]]

local nQuestLog = nQuestLog

local REVISION = tonumber(("$Revision: 183 $"):match("%d+"))
if ( nQuestLog.revision < REVISION ) then
	nQuestLog.version = (nQuestLog.versionstring):format(REVISION)
	nQuestLog.revision = REVISION
	nQuestLog.date = ("$Date: 2008-04-27 18:10:26 +0000 (Sun, 27 Apr 2008) $"):match("%d%d%d%d%-%d%d%-%d%d")
end

local moduleName = "Comments"
local Comments = nQuestLog:NewModule(moduleName, "AceEvent-2.0", "AceHook-2.1")

--Load libraries
local Abacus = LibStub("LibAbacus-3.0")
local Dewdrop = AceLibrary("Dewdrop-2.0")
local Quixote = AceLibrary("Quixote-1.0")
local Tablet = AceLibrary("Tablet-2.0")

--Load localization
local L = AceLibrary("AceLocale-2.2"):new(moduleName)

function Comments:OnModuleLoad()
	self.menuName = L[moduleName]
	self.description = L["Description"]
	if ( not self.defaultsRegistered ) then
		self.core:RegisterDefaults(moduleName, 'profile', { enabled = true,
			data = { fontSizePercent = 0.85, detached = true },
			minWidth = 350, maxHeight = 350 })
		self.defaultsRegistered = true
	end
end

function Comments:ToggleEnabled(v)
	self.db.profile.enabled = v
	if ( v ) then
		self:SecureHook(nQuestLog, "ShowLog")
		if ( nQuestLog:IsModuleActive("DetailsFrame") ) then
			self:SecureHook(nQuestLog:GetModule("DetailsFrame"), "UpdateQuestDetail")
		end
	else
		self:Unhook(nQuestLog, "ShowLog")
		if ( nQuestLog:IsModuleActive("DetailsFrame") ) then
			self:Unhook(nQuestLog:GetModule("DetailsFrame"), "UpdateQuestDetail")
		end
	end
end

function Comments:GetMenu(oldMenu)
	local menu = { order = oldMenu.order, type = 'group',
		name = self.menuName,
		desc = self.description,
		cmdHidden = true,
		args = {
			toggleEnabled = oldMenu,
			minWidth = { order = oldMenu.order + 1, type = 'range',
				name = L["Minimum width"],
				desc = L["Set the minimum width of the Comments frame"],
				min = 250, max = 500, step = 1,
				get = function() return self.db.profile.minWidth end,
				set = function(v)
					self.db.profile.minWidth = v
					if ( Tablet:IsRegistered("nQuestLog_Comments") ) then
						Tablet:Close("nQuestLog_Comments")
						Tablet:Unregister("nQuestLog_Comments")
					end
					self:Update()
				end },
			maxHeight = { order = oldMenu.order + 2, type = 'range',
				name = L["Maximum height"],
				desc = L["Set the maximum height of the Comments frame"],
				min = 200, max = 800, step = 1,
				get = function() return self.db.profile.maxHeight end,
				set = function(v)
					self.db.profile.maxHeight = v
					if ( Tablet:IsRegistered("nQuestLog_Comments") ) then
						Tablet:Close("nQuestLog_Comments")
						Tablet:Unregister("nQuestLog_Comments")
					end
					self:Update()
				end
			} } }
	return menu
end

function Comments:ShowLog(object, questid)
	if ( LightHeaded ) and ( nQuestLog:IsModuleEnabled("DetailsFrame") ) and ( questid ) then
		self:ScheduleEvent(self.Update, 0, self) --Schedule an event because tablet bugs out otherwise.
		self.altTitle, self.altLevel = "", nil
	end
end

function Comments:UpdateQuestDetail(object)
	if ( LightHeaded ) then
		self:Update()
	end
end

function Comments:GetQuestId()
	local questid = 1
	if ( nQuestLog:IsModuleEnabled("DetailsFrame") ) then
		questid = nQuestLog:GetModule("DetailsFrame").questid
	end
	return questid
end

function Comments:Update()
	if ( Tablet:IsRegistered("nQuestLog_Comments") ) then
		--Disable the lock every time the tablet is shown
		--I'd rather remove the "Lock" option from the Tablet Dewdrop menu, but that
		--  doesn't appear to be possible at this time without modification of Tablet-2.0
		if ( Tablet:IsLocked("nQuestLog_Comments") ) then
			Tablet:ToggleLocked("nQuestLog_Comments")
		end
		--Refresh the Comments pane
		Tablet:Refresh("nQuestLog_Comments")
		return
	end
	
	self.altID, self.altTitle, self.altLevel = nil, "", nil
	local badQIDs = {}
	
	--Declare constants
	local DATA_COLOR = "|cFF66FF66"
	local END_COLOR = "|r"
	local INDENT_SIZE = 14
	local LABEL_COLOR = nQuestLog:RGBPercToHex(NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
	local NEW_LINE = "\n"
	local WHITE = "|cFFFFFFFF"
	--Blizzard-localized strings
	local LEVEL_PHRASE = ITEM_LEVEL --"Level %d"
	local MONEY = MONEY --"Money"
	local XP = XP --"XP"
	local ZONE = ZONE --"Zone"
	
	--Add function pointers to core functions used in this module
	local function addWayPoint(...) nQuestLog:AddWayPoint(...) end
	local function formatQuestInfo(...) return nQuestLog:FormatQuestInfo(...) end
	local function formatZoneByFaction(...) return nQuestLog:FormatZoneByFaction(...) end
	local function itemClick(...) nQuestLog:ItemClick(...) end
	local function itemHover(...) nQuestLog:ItemHover(...) end
	local function RGBPercToHex(...) return nQuestLog:RGBPercToHex(...) end
	local function round(...) return nQuestLog:Round(...) end
	local function sendToChat(...) nQuestLog:SendToChat(...) end
	
	local function buildItemString(itemid)
		return "item:"..itemid..":0:0:0:0:0:0:0"
	end
	
	local function toggleDisplay(id, idtype)
		if ( idtype == "npc" ) then
			if ( self.npcid == id ) then
				self.npcid = nil
			else
				self.npcid = id
			end
		elseif ( idtype == "quest" ) then
			if ( self.qid == id ) then
				self.qid = nil
			else
				self.qid = id
			end
		elseif ( idtype == "object" ) then
			if ( self.objid == id ) then
				self.objid = nil
			else
				self.objid = id
			end
		end
		self:Update()
	end
	
	local function showAltQuest(id)
		local lvl = select(3, LightHeaded:GetQuestInfo(id))
		local name = LightHeaded:GetQuestName(id)
		if ( name ) and ( lvl ) then
			self.altID = id
			self.altTitle = name
			self.altLevel = tonumber(lvl)
			self:Update()
		end
	end
	
	local function addURLs(cat, linkid, linktype)
		local aURLtitle, aURL = "-> "..DATA_COLOR.."Allakhazam URL"..END_COLOR.." <-", "http://wow.allakhazam.com/"
		local tURLtitle, tURL = "-> "..DATA_COLOR.."Thottbot URL"..END_COLOR.." <-", "http://www.thottbot.com/"
		local wURLtitle, wURL = "-> "..DATA_COLOR.."Wowhead URL"..END_COLOR.." <-", "http://www.wowhead.com/"
		cat:AddLine('text', wURLtitle, 'indentation', INDENT_SIZE * 2,
			'func', sendToChat, 'arg1', wURL.."?"..linktype.."="..linkid, 'arg2', true)
		if ( linktype == "npc" ) then
			cat:AddLine('text', tURLtitle, 'indentation', INDENT_SIZE * 2,
				'func', sendToChat, 'arg1', tURL.."c"..linkid, 'arg2', true)
			cat:AddLine('text', aURLtitle, 'indentation', INDENT_SIZE * 2,
				'func', sendToChat, 'arg1', aURL.."db/mob.html?wmob="..linkid, 'arg2', true)
		elseif ( linktype == "object" ) then
			cat:AddLine('text', tURLtitle, 'indentation', INDENT_SIZE * 2,
				'func', sendToChat, 'arg1', tURL.."o"..linkid, 'arg2', true)
		elseif ( linktype == "quest" ) then
			cat:AddLine('text', tURLtitle, 'indentation', INDENT_SIZE * 2,
				'func', sendToChat, 'arg1', tURL.."q"..linkid, 'arg2', true)
			cat:AddLine('text', aURLtitle, 'indentation', INDENT_SIZE * 2,
				'func', sendToChat, 'arg1', aURL.."db/quest.html?wquest="..linkid, 'arg2', true)
		end
	end
	
	local function addNpcLocs(cat, id, name, qTitle, qLevel)
		id = tonumber(id)
		if ( LightHeaded:LoadNPCData(id) ) then
			for c, z, x, y in LightHeaded:IterateNPCLocs(id) do
				local zoneNames = { GetMapZones(c) }
				local zone = zoneNames[z]
				cat:AddLine('text', formatZoneByFaction(zone, x, y), 'indentation', INDENT_SIZE * 2,
					'tooltipTitle', "", 'func', addWayPoint, 'arg1', tonumber(x),
					'arg2', tonumber(y), 'arg3', qTitle, 'arg4', qLevel, 'arg5', name,
					'arg6', tonumber(c), 'arg7', tonumber(z), 'arg8', zone)
			end
			return true
		end
		return false
	end
	
	local function popupLinks(links, qTitle, qLevel)
		Dewdrop:Open(moduleName, 'cursorX', true, 'cursorY', true, 'children',
			function()
				for i=1, #(links), 1 do
					local x, y, url, name = links[i].x, links[i].y, links[i].url, links[i].name
					if ( x ) and ( y ) then
						Dewdrop:AddLine('text', LABEL_COLOR..L["Coordinates"]..":  "..DATA_COLOR..
							"["..x..", "..y.."]", 'notCheckable', true, 'tooltipTitle', "",
							'func', addWayPoint, 'arg1', tonumber(x), 'arg2', tonumber(y),
							'arg3', qTitle, 'arg4', qLevel)
					elseif ( url ) then
						Dewdrop:AddLine('text', LABEL_COLOR.."URL:  "..DATA_COLOR..url,
							'notCheckable', true, 'tooltipTitle', "",
							'func', sendToChat, 'arg1', url, 'arg2', true)
						local urlType, urlID = select(3, url:find("http://www.wowhead.com/%?(%a+)=(%d+)"))
						if ( urlID ) then
							urlID = tonumber(urlID)
							if ( urlType == "quest" ) then
								local questValid = true
								for _, v in ipairs(badQIDs) do
									if ( v == urlID ) then
										questValid = false
										break
									end
								end
								if ( questValid ) then
									local link
									local lvl = select(3, LightHeaded:GetQuestInfo(urlID))
									local name = LightHeaded:GetQuestName(urlID)
									if ( name ) and ( lvl ) then
										local DC = GetDifficultyColor(lvl)
										link = RGBPercToHex(DC.r, DC.g, DC.b)..
											"["..lvl.."] "..name
									end
									if ( not link ) then
										link = LABEL_COLOR..L["Quest"].." "..urlID
									end
									Dewdrop:AddLine('text', link, 'tooltipTitle', "",
										'func', showAltQuest, 'arg1', urlID )
								end
							end
							if ( urlType == "item" ) then
								local itemString = buildItemString(urlID)
								local name, link, rarity = GetItemInfo(itemString)
								local unsafe = false
								if ( not name ) then
									--Enable unsafe item link
									link = itemString
									name = L["<not yet cached>"]
									unsafe = true
								end
								local item = "["..name.."]"
								if ( rarity ) then
									local itemColor = select(4, GetItemQualityColor(rarity))
									item = itemColor..item
								end
								Dewdrop:AddLine('text', LABEL_COLOR..item,
									'func', itemClick, 'arg1', link,
									'tooltipFunc', itemHover, 'tooltipArg1', link,
									'tooltipArg2', unsafe, 'tooltipArg3', urlID)
							end
							if ( urlType == "npc" ) then
								if ( name ) then
									Dewdrop:AddLine('text', WHITE..name,
										'isTitle', true, 'tooltipTitle', "")
								else
									name = "NPC".." "..urlID
								end
								addNpcLocs(Dewdrop, urlID, name, qTitle, qLevel)
							end
						end
					end
				end;
			end
		)
	end
	
	local function getMobMapQuestIdByName(questname, mmQID)
		if ( mobmap_questnames ) then
			for k, v in pairs(mobmap_questnames) do
				if ( v == questname ) and ( k > tonumber(mmQID) ) then
					return k
				end
			end
		end
		return nil
	end
	
	Tablet:Register("nQuestLog_Comments", 'detachedData', self.db.profile.data, 'cantAttach', true,
		'dontHook', true, 'hideWhenEmpty', true, 'showTitleWhenDetached', true, 'strata', "HIGH",
		'minWidth', self.db.profile.minWidth, 'maxHeight', self.db.profile.maxHeight, 'children',
		function()
			local questid = self:GetQuestId()
			if ( questid ) and ( LightHeaded.GetQuestInfo )
			and ( LightHeaded.IterateQuestComments ) and ( LightHeaded.IterateQuestInfo ) then
				local title, level, tag = Quixote:GetQuestById(questid)
				local qid, sharable, _, reqlev, stype, sname, sid, etype, ename, eid, xp, rep, series
				if ( self.altID ~= nil ) and ( self.altTitle ~= "" ) then
					for iqid, isharable, _, ireqlev, istype, isname, isid,
					    ietype, iename, ieid, ixp, irep, iseries in
						LightHeaded:IterateQuestInfo(self.altID) do
						if ( tonumber(iqid) == tonumber(self.altID) ) then
							qid, sharable, reqlev, stype, sname, sid,
							etype, ename, eid, xp, rep, series =
								iqid, isharable, ireqlev, istype, isname, isid,
								ietype, iename, ieid, ixp, irep, iseries
							break
						end
					end
					if ( qid ) then
						title = self.altTitle
						level = self.altLevel
					else
						tinsert(badQIDs, self.altID)
					end
				end
				if ( not qid ) then
					local link = GetQuestLink(questid)
					if ( not link ) then return end
					qid = tonumber(link:match(":(%d+):"))
					qid, sharable, _, reqlev, stype, sname, sid, etype, ename, eid, xp, rep, series =
						LightHeaded:GetQuestInfo(qid)
					if ( not qid ) then return end
				end
				level, qid = tonumber(level), tonumber(qid)
				local qColorHex
				if ( level ) then
					local qColor = GetDifficultyColor(level)
					qColorHex = RGBPercToHex(qColor.r, qColor.g, qColor.b)
				else
					qColorHex = LABEL_COLOR
				end
				
				--TITLE  |  Comments  |
				Tablet:SetTitle(L["Comments"])
				
				--CATEGORY  |  LightHeaded Info  |  Level, title, color, id, reqlvl, sharable, start, end
				local cat = Tablet:AddCategory('text', NEW_LINE..L["%s Info"]:format("Lightheaded"),
					'columns', 1, 'wrap', true)
				
				--LINE  |  [<quest level>] <quest title>  |  Colored by difficulty
				cat:AddLine('text', formatQuestInfo(level, tag, title, qColorHex), 'indentation', INDENT_SIZE)
				
				--LINE  |  Quest ID [<quest id>] - Req Lvl <required level> - Sharable  |  Clickable
				local qInfo = L["%s ID"]:format(L["Quest"])..DATA_COLOR.." ["..qid.."]"..END_COLOR
				if ( reqlev ~= "" ) then
					qInfo = qInfo.." - "..L["Req Lvl"].." "..DATA_COLOR..reqlev..END_COLOR
				end
				if ( sharable ~= "" ) then
					qInfo = qInfo.." - "..DATA_COLOR..L["Sharable"]
				end
				cat:AddLine('text', qInfo, 'indentation', INDENT_SIZE,
					'func', toggleDisplay, 'arg1', qid, 'arg2', "quest")
				if ( qid == self.qid ) then
					--LINES  |  -> <site> URL <-  |  Clickable
					--Add URLs for the quest when the "Quest ID" line is clicked
					addURLs(cat, qid, "quest")
				end
				
				if ( sname ~= "" ) then
					--LINE  |  Starts: [<starting npc/object>]  |  Clickable
					cat:AddLine('text', L["Starts"]..": "..DATA_COLOR.."["..sname.."]",
						'indentation', INDENT_SIZE, 'func', toggleDisplay, 'arg1', sid, 'arg2', stype)
					if ( sid == self.npcid ) then
						--LINES  |  <zone> [<xcoord>, <ycoord>]  |  Clickable
						--Add zone coordinates for the NPC when the "Starts:" line is clicked
						local npcValid = addNpcLocs(cat, sid, sname, title, level)
						if ( not npcValid ) then
							addURLs(cat, sid, "npc")
						end
					end
					if ( sid == self.objid ) then
						--LINES  |  -> <site> URL <-  |  Clickable
						--Add URLs for the object when the "Starts:" line is clicked
						addURLs(cat, sid, stype)
					end
				end
				
				if ( ename ~= "" ) then
					--LINE  |  Ends: [<ending npc/object>]  |  Clickable
					cat:AddLine('text', L["Ends"]..": "..DATA_COLOR.."["..ename.."]", 'indentation', INDENT_SIZE,
						'func', toggleDisplay, 'arg1', eid, 'arg2', etype)
					if ( eid == self.npcid ) then
						--LINES  |  <zone> [<xcoord>, <ycoord>]  |  Clickable
						--Add zone coordinates for the NPC when the "Ends:" line is clicked
						local npcValid = addNpcLocs(cat, eid, ename, title, level)
						if ( not npcValid ) then
							addURLs(cat, eid, "npc")
						end
					end
					if ( eid == self.objid ) then
						--LINES  |  -> <site> URL <-  |  Clickable
						--Add URLs for the object when the "Ends:" line is clicked
						addURLs(cat, eid, etype)
					end
				end
				
				if ( xp ~= "" ) or ( rep ~= "" ) then
					--CATEGORY  |  LightHeaded Rewards  |  XP and rep rewards
					cat = Tablet:AddCategory('text', L["%s Rewards"]:format("Lightheaded"), 'columns', 1)
					
					if ( xp ~= "" ) then
						local maxXP, playerLevel = UnitXPMax("player"), UnitLevel("player")
						local xpText = XP..": "..DATA_COLOR..xp..END_COLOR.." ("..DATA_COLOR..
							round((xp / maxXP) * 100, 1).."%"..END_COLOR..")"
						cat:AddLine('text', xpText, 'indentation', INDENT_SIZE)
						if ( level ) and ( playerLevel >= (level + 6) ) then
							--Quest XP formula from http://www.wowwiki.com/Formulas:Quest_XP
							--Could not determine the exact formula, but this one works well enough
							--Estimates are accurate within 3% based on limited testing
							local effXP, effXPText
							if ( playerLevel == (level + 6) ) then
								effXP = round(xp*0.8 / 5, 0) * 5
							elseif ( playerLevel == (level + 7) ) then
								effXP = round(xp*0.6 / 5, 0) * 5
							elseif ( playerLevel == (level + 8) ) then
								effXP = round(xp*0.4 / 5, 0) * 5
							elseif ( playerLevel == (level + 9) ) then
								effXP = round(xp*0.2 / 5, 0) * 5
							elseif ( playerLevel >= (level + 10) ) then
								effXP = round(xp*0.1 / 5, 0) * 5
							end
							effXPText = XP.." @ "..LEVEL_PHRASE:format(playerLevel)..": "..
								DATA_COLOR..effXP..END_COLOR.." ("..DATA_COLOR..
								round((effXP / maxXP) * 100, 1).."%"..END_COLOR..")"
							cat:AddLine('text', effXPText, 'indentation', INDENT_SIZE)
						end
					end
					
					if ( rep ~= "" ) then
						local factionTable = {}
						for factionIndex=1, GetNumFactions() do
							local name, _, standingId = GetFactionInfo(factionIndex)
							tinsert(factionTable, {name, standingId})
						end
						for name, value in rep:gmatch("([^\029]+)\029([^\029]+)") do
							for _, v in ipairs(factionTable) do
								if ( name == v[1] ) then
									local r, g, b = FACTION_BAR_COLORS[v[2]].r,
										FACTION_BAR_COLORS[v[2]].g, FACTION_BAR_COLORS[v[2]].b
									local hexColor = "|cFF"..
										("%02x%02x%02x"):format(r * 255, g * 255, b * 255)
									cat:AddLine('text', L["Rep"]..": "..hexColor..name..
										DATA_COLOR.." +"..value, 'indentation', INDENT_SIZE)
								end
							end
						end
					end
				end --if ( xp ~= "" ) or ( rep ~= "" )
				
				if ( MobMapDotParentFrame ) then
					if ( MobMap_LoadDatabase ) then
						if ( not IsAddOnLoaded("MobMapDatabaseStub"..MOBMAP_QUEST_DATABASE) ) then
							MobMap_LoadDatabase(MOBMAP_QUEST_DATABASE)
						end
					end
					local mmQID = getMobMapQuestIdByName(title, 0)
					local mmLvl, mmZone, mmHorde, mmAlli, mmMoney, mmNPC, mmSrc, mmType, mmAlways, mmChoice
					if ( mmQID ) and ( MobMap_GetDetailsForQuest ) then
						if ( sname ~= "" ) and ( MobMap_GetMobName ) and ( MobMap_GetQuestSourceName ) then
							local l, z, h, a, m, n, s, t, aw, c, lastQID
							while not (( tonumber(mmLvl) == level )
							and (( sname == MobMap_GetMobName(mmNPC) )
							or ( sname == MobMap_GetQuestSourceName(mmSrc) ))) do
								l, z, _, _, h, a, m, n, s, t, aw, c = MobMap_GetDetailsForQuest(mmQID)
								if ( tonumber(l) == level ) then
									mmLvl, mmZone, mmHorde, mmAlli, mmMoney, mmNPC, mmSrc, mmType,
										mmAlways, mmChoice = l, z, h, a, m, n, s, t, aw, c
									lastQID = mmQID
								end
								mmQID = getMobMapQuestIdByName(title, mmQID)
								if ( mmQID == nil ) then
									mmQID = lastQID
									break
								end
							end
						else
							while ( tonumber(mmLvl) ~= level ) do
								mmQID = getMobMapQuestIdByName(title, mmQID)
								if ( mmQID == nil ) then
									break
								end
								mmLvl, mmZone, _, _, mmHorde, mmAlli, mmMoney, mmNPC, mmSrc, mmType,
									mmAlways, mmChoice = MobMap_GetDetailsForQuest(mmQID)
							end
						end
					end
					if ( mmQID ) then
						cat = Tablet:AddCategory('text', L["%s Info"]:format("MobMap"), 'columns', 1)
						local faction = ""
						if ( mmHorde ) and ( mmAlli ) then
							faction = " |cFFFFFF00Horde/Alliance"
						elseif ( mmHorde ) then
							faction = " |cFF00FF00"..L["%s-only"]:format("Horde")
						elseif ( mmAlli ) then
							faction = " |cFF00FF00"..L["%s-only"]:format("Alliance")
						end
						cat:AddLine('text', L["%s Type"]:format(L["Quest"])..": "..DATA_COLOR..
							MOBMAP_QUEST_TYPES[mmType]..faction, 'indentation', INDENT_SIZE)
						if ( MobMap_GetZoneName ) then
							cat:AddLine('text', ZONE..": "..
								formatZoneByFaction(MobMap_GetZoneName(mmZone)),
								'indentation', INDENT_SIZE)
						end
						if ( sname == "" ) then
							local source = ""
							if ( mmSrc ~= 0 ) and ( MobMap_GetQuestSourceName ) then
								source = MobMap_GetQuestSourceName(mmSrc)
							elseif ( mmNPC ~= 0 ) and ( MobMap_GetMobName ) then
								source = MobMap_GetMobName(mmNPC)
							end
							if ( source ~= "" ) and ( MobMap_ParseQuestObjective ) then
								cat:AddLine('text', L["Source"]..": "..DATA_COLOR.."["..source.."]",
									'func', MobMap_ParseQuestObjective, 'arg1', source,
									'indentation', INDENT_SIZE)
							end
						end
						if ( xp ~= "" ) or ( #(mmChoice) > 0 ) or ( #(mmAlways) > 0 ) or ( mmMoney > 0 ) then
							local function addRewardItems(rewardTable)
								for _, id in pairs(rewardTable) do
									local itemString = buildItemString(id)
									local name, link, rarity, _, _, _, _, _, _, texture =
										GetItemInfo(itemString)
									local unsafe = false
									if ( not name ) then
										--Enable unsafe item link
										link = itemString
										name = L["<not yet cached>"]
										unsafe = true
									end
									local item = "["..name.."]"
									if ( rarity ) then
										local itemColor = select(4, GetItemQualityColor(rarity))
										item = itemColor..item
									end
									cat:AddLine('text', item, 'indentation', INDENT_SIZE + 2,
										'checked', true, 'hasCheck', true,
										'checkIcon', texture, 'func', itemClick, 'arg1', link,
										'onEnterFunc', itemHover, 'onEnterArg1', link,
										'onEnterArg2', unsafe, 'onEnterArg3', id,
										'onLeaveFunc', GameTooltip.Hide,
										'onLeaveArg1', GameTooltip)
								end
							end
							
							cat = Tablet:AddCategory('text', L["%s Rewards"]:format("MobMap"),
								'columns', 1)
							if ( self.altID ) then
								if ( #(mmChoice) > 0 ) then
									cat:AddLine('text', L["Choose from"]..":",
										'indentation', INDENT_SIZE)
									addRewardItems(mmChoice)
								end
								if ( #(mmAlways) > 0 ) then
									cat:AddLine('text', L["Always receive"]..":",
										'indentation', INDENT_SIZE)
									addRewardItems(mmAlways)
								end
							end
							local xpToCopper = 0
							if ( xp ~= "" ) then
								if ( tonumber(xp) >= 1000 ) then
									xpToCopper = round((xp+1)*0.01, 0)*60
								else
									xpToCopper = round((xp+1)*0.02, 0)*30
								end
							end
							if ( mmMoney > 0 ) then
								mmMoney = mmMoney - xpToCopper
							else
								mmMoney = 0
							end
							local money = Abacus:FormatMoneyFull(mmMoney, true)
							if ( xpToCopper > 0 ) then
								money = money.." (+"..Abacus:FormatMoneyFull(xpToCopper,
									true).." @ 70)"
							end
							cat:AddLine('text', MONEY..": "..WHITE..money,
								'indentation', INDENT_SIZE)
						end --if ( #(mmChoice) > 0 ) or ( #(mmAlways) > 0 ) or ( mmMoney > 0 )
					end --if ( mmQID )
				end --if ( MobMapDotParentFrame )
				
				if ( series ~= "" ) then
					cat = Tablet:AddCategory('text', L["%s %s Series"]:format("Lightheaded", L["Quest"]), 'columns', 1)
					
					for step, id in series:gmatch("([^\029]+)\029([^\029]+)") do
						local questValid = true
						id = tonumber(id)
						for _, v in ipairs(badQIDs) do
							if ( v == id ) then
								questValid = false
								break
							end
						end
						if ( questValid ) then
							local link
							local name = LightHeaded:GetQuestName(id)
							if ( name ) then
								link = name
								local lvl, reqlvl = select(3, LightHeaded:GetQuestInfo(id))
								if ( tonumber(lvl) ) then
									local color = GetDifficultyColor(lvl)
									link = RGBPercToHex(color.r, color.g, color.b)..
										"["..lvl.."] "..link
								elseif ( tonumber(reqlvl) == 70 ) then
									local color = GetDifficultyColor(reqlvl)
									link = RGBPercToHex(color.r, color.g, color.b)..
										"["..reqlvl.."] "..link
								else
									link = "[??] "..link
								end
							else
								link = step..". "..L["Quest"].." "..id
							end
							local isChecked = false
							if ( tonumber(qid) == id ) then
								isChecked = true
							end
							cat:AddLine('text', link, 'hasCheck', true, 'checked', isChecked,
								'func', showAltQuest, 'arg1', id)
						end
					end
				end --if ( series ~= "" )
				
				cat = Tablet:AddCategory('text', L["%s Comments"]:format("Lightheaded"), 'columns', 2)
				if ( LightHeaded:GetNumQuestComments(qid) > 0 ) then
					qid = tonumber(qid)
					for idx, cqid, cid, rating, indent, parent, date, poster, comment in
					LightHeaded:IterateQuestComments(qid) do
						if ( qid == tonumber(cqid) ) then
							cat:AddLine('text', (DATA_COLOR.."[%s - %s - %s]"):format(poster, date,
								rating), 'wrap', true, 'indentation', indent * INDENT_SIZE)
							local links = {}
							--Process wiki-formatted URLs
							for url, name in
							comment:gmatch("%[url=(http://[^%s%[%]%(%)%,]+)%]([^%[]+)%[/url%]") do
								--Add URL and name to links table
								tinsert(links, { url = url, name = name })
								--Escape magic characters in the URL string
								url = url:gsub("([%^%$%(%)%%%.%[%]%*%+%-%?])", "%%%1")
								--For any matches, remove the URL string from the comment
								--This avoids duplicates when normal URLs are processed
								comment = comment:gsub(url, "")
							end
							--Process normal URLs
							for url in comment:gmatch("http://[^%s%[%]%(%)%,]+") do
								--Add URL to links table
								tinsert(links, { url = url })
							end
							--Process coordinates
							for xCoord, yCoord in
							comment:gmatch("(%d%d?%.?%d?%d?)%s*[, ]%s*(%d%d?%.?%d?%d?)") do
								--Add X and Y coords to links table
								tinsert(links, { x = xCoord, y = yCoord })
							end
							--If there are any links in the comment, make the comment clickable
							--Clicking the comment will open a popup menu with all the links
							if ( #(links) > 0 ) then
								cat:AddLine('text', comment, 'indentation', indent * INDENT_SIZE,
									'wrap', true, 'func', popupLinks, 'arg1', links,
									'arg2', title, 'arg3', level)
							--No links in this comment, so no reason to make it clickable
							else
								cat:AddLine('text', comment, 'indentation', indent * INDENT_SIZE,
									'wrap', true)
							end
							cat:AddLine('text', " ") --spacer
						end
					end
				else
					cat:AddLine('text', L["No comments found"].."...", 'indentation', INDENT_SIZE)
					cat:AddLine('text', " ") --spacer
				end --if ( LightHeaded:GetNumQuestComments(title, level) > 0 )
				
				cat:AddLine('text', L["%s data is from %s"]:format("Lightheaded", "Wowhead"),
					'hasCheck', true, 'checked', true,
					'checkIcon', "Interface\\Addons\\LightHeaded\\images\\wh_icon",
					'func', sendToChat, 'arg1', "http://www.wowhead.com", 'arg2', true)
			end --if ( questid ) and ( LightHeaded.GetQuestInfo )
			    --and ( LightHeaded.IterateQuestComments ) and ( LightHeaded.IterateQuestInfo )
		end) --Tablet:Register()
	Tablet:Open("nQuestLog_Comments")
end
