﻿--[[

Copyright (C) 2004 by neuron
aagaande AT gmail DOT com

This program is free software; you can redistribute it and--or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

]]

local nQuestLog = nQuestLog

local REVISION = tonumber(("$Revision: 180 $"):match("%d+"))
if ( nQuestLog.revision < REVISION ) then
	nQuestLog.version = (nQuestLog.versionstring):format(REVISION)
	nQuestLog.revision = REVISION
	nQuestLog.date = ("$Date: 2008-04-23 05:00:35 +0000 (Wed, 23 Apr 2008) $"):match("%d%d%d%d%-%d%d%-%d%d")
end

local moduleName = "Coloring"
local Coloring = nQuestLog:NewModule(moduleName,"AceEvent-2.0","AceHook-2.1", "AceConsole-2.0")

--Load libraries
local Crayon = LibStub("LibCrayon-3.0")
local Tourist = LibStub("LibTourist-3.0")

--Load localization
local L = AceLibrary("AceLocale-2.2"):new("Coloring")

function Coloring:OnModuleLoad()
	self.menuName = L["Coloring"]
	self.description = L["Description"]
	if ( not self.defaultsRegistered ) then
		self.core:RegisterDefaults(moduleName, "profile", { enabled = true,
			levelTrivial = { 0.5, 0.5, 0.5 },
			levelStandard = { 0.25, 0.75, 0.25 },
			levelDifficult = { 1, 1, 0 },
			levelVeryDifficult = { 1, 0.5, 0.25 },
			levelImpossible = { 1, 0.1, 0.1 },
			showProgress50 = true,
			progressStart = { 1, 0, 0 },
			progress25 = { 1, 0.5, 0 },
			progress50 = { 1, 1, 0 },
			progress75 = { 0.5, 1, 0 },
			progressEnd = { 0, 1, 0 },
			trackerBorderColor = { r = 255, g = 255, b = 255, a = 0 },
			trackerBackgroundColor = { r = 31, g = 32, b = 94, a = 0.6 },
			zoneHeader = { r = 119, g = 187, b = 255, threshold = false },
			zoneLevel = { r = 119, g = 187, b = 255, threshold = false },
			questHeader = { r = 255, g = 255, b = 255, threshold = true },
			questHighlight = { r = 91, g = 135, b = 229, a = 0.8 },
			questProgress = { r = 255, g = 255, b = 255, threshold = true },
			questFailed = { r = 255, g = 255, b = 255 },
			questGoto = { r = 255, g = 255, b = 255 },
			questDone = { r = 255, g = 255, b = 255 },
			objectiveHeader = { r = 255, g = 255, b = 255, threshold = false },
			objectiveProgress = { r = 255, g = 255, b = 255, threshold = true },
			objectiveText = { r = 237, g = 165, b = 95 },
			notificationColor = { r = 255, g = 255, b = 0 } })
		self.defaultsRegistered = true
	end
end

function Coloring:GetMenu(oldMenu)
	local colorList = {
		{ name = "trackerBorderColor" },
		{ name = "trackerBackgroundColor" },
		{ name = "zoneHeader", extra = "level" },
		{ name = "zoneLevel", extra = "level" },
		{ name = "questHeader", extra = "level" },
		{ name = "questHighlight" },
		{ name = "questProgress", extra = "progress" },
		{ name = "questFailed" },
		{ name = "questGoto" },
		{ name = "questDone" },
		{ name = "objectiveHeader", extra = "progress" },
		{ name = "objectiveProgress", extra = "progress" },
		{ name = "objectiveText" },
		{ name = "notificationColor" } }
	local function getColor(var)
		return self.db.profile[var].r / 255, self.db.profile[var].g / 255,
			self.db.profile[var].b / 255, self.db.profile[var].a
	end
	local function setColor(var, r, g, b, a)
		self.db.profile[var] = { r = r * 255, g = g * 255, b = b * 255, a = a}
		self.core:RefreshTracker()
	end
	local function getToggle(var)
		return self.db.profile[var].threshold
	end
	local function setToggle(var, v)
		self.db.profile[var].threshold = v
		self.core:RefreshTracker()
	end
	local menu = { order = oldMenu.order, type = "group",
		name = self.menuName, desc = self.description,
		cmdHidden = true,
		args = { toggleEnabled = oldMenu,
			level = { order = oldMenu.order + 1, type = "group",
				name = L["colorBy"]:format(L["level"]), desc = L["colorByDesc"]:format(L["level"]),
				args = {
					custom = { order = 1, type = "toggle",
						name = L["customColor"]:format(L["level"]),
						desc = L["customColorDesc"]:format(L["level"]),
						get = function() return self.db.profile.levelCustom end,
						set = function(v)
							self.db.profile.levelCustom = v
							self.core:RefreshTracker()
						end },
					spacer = { order = 2, type = "header" },
					impossible = { order = 3, type = "color",
						name = L["Impossible"], desc = L["colorFor"]:format(L["Impossible"])
							.."  "..L["colorDefault"]:format(L["red"]),
						disabled = function() return not self.db.profile.levelCustom end,
						get = function() return unpack(self.db.profile.levelImpossible) end,
						set = function(r, g, b, a)
							self.db.profile.levelImpossible = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					veryDifficult = { order = 4, type = "color",
						name = L["Very difficult"], desc = L["colorFor"]:format(L["Very difficult"])
							.."  "..L["colorDefault"]:format(L["orange"]),
						disabled = function() return not self.db.profile.levelCustom end,
						get = function() return unpack(self.db.profile.levelVeryDifficult) end,
						set = function(r, g, b, a)
							self.db.profile.levelVeryDifficult = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					difficult = { order = 5, type = "color",
						name = L["Difficult"], desc = L["colorFor"]:format(L["Difficult"])
							.."  "..L["colorDefault"]:format(L["yellow"]),
						disabled = function() return not self.db.profile.levelCustom end,
						get = function() return unpack(self.db.profile.levelDifficult) end,
						set = function(r, g, b, a)
							self.db.profile.levelDifficult = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					standard = { order = 6, type = "color",
						name = L["Standard"], desc = L["colorFor"]:format(L["Standard"])
							.."  "..L["colorDefault"]:format(L["green"]),
						disabled = function() return not self.db.profile.levelCustom end,
						get = function() return unpack(self.db.profile.levelStandard) end,
						set = function(r, g, b, a)
							self.db.profile.levelStandard = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					trivial = { order = 7, type = "color",
						name = L["Trivial"], desc = L["colorFor"]:format(L["Trivial"])
							.."  "..L["colorDefault"]:format(L["gray"]),
						disabled = function() return not self.db.profile.levelCustom end,
						get = function() return unpack(self.db.profile.levelTrivial) end,
						set = function(r, g, b, a)
							self.db.profile.levelTrivial = {r, g, b, a}
							self.core:RefreshTracker()
						end } } },
			progress = { order = oldMenu.order + 2, type = "group",
				name = L["colorBy"]:format(L["progress"]), desc = L["colorByDesc"]:format(L["progress"]),
				args = {
					custom = { order = 1, type = "toggle",
						name = L["customColor"]:format(L["progress"]),
						desc = L["customColorDesc"]:format(L["progress"]),
						get = function() return self.db.profile.progressCustom end,
						set = function(v)
							self.db.profile.progressCustom = v
							self.core:RefreshTracker()
						end },
					custom25 = { order = 2, type = "toggle",
						name = L["showProgress"]:format("25%"),
						desc = L["showProgressDesc"]:format("25%"),
						disabled = function() return not self.db.profile.progressCustom end,
						get = function() return self.db.profile.showProgress25 end,
						set = function(v)
							self.db.profile.showProgress25 = v
							self.core:RefreshTracker()
						end },
					custom50 = { order = 3, type = "toggle",
						name = L["showProgress"]:format("50%"),
						desc = L["showProgressDesc"]:format("50%"),
						disabled = function() return not self.db.profile.progressCustom end,
						get = function() return self.db.profile.showProgress50 end,
						set = function(v)
							self.db.profile.showProgress50 = v
							self.core:RefreshTracker()
						end },
					custom75 = { order = 4, type = "toggle",
						name = L["showProgress"]:format("75%"),
						desc = L["showProgressDesc"]:format("75%"),
						disabled = function() return not self.db.profile.progressCustom end,
						get = function() return self.db.profile.showProgress75 end,
						set = function(v)
							self.db.profile.showProgress75 = v
							self.core:RefreshTracker()
						end },
					colorSpacer = { order = 5, type = "header" },
					color00 = { order = 6, type = "color",
						name = L["Start"].." (0%)", desc = L["progressDesc"]:format("0%"),
						disabled = function() return not self.db.profile.progressCustom end,
						get = function() return unpack(self.db.profile.progressStart) end,
						set = function(r, g, b, a)
							self.db.profile.progressStart = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					color25 = { order = 7, type = "color",
						name = "1/4 (25%)", desc = L["progressDesc"]:format("25%"),
						disabled = function() return not self.db.profile.progressCustom end,
						hidden = function() return not self.db.profile.showProgress25 end,
						get = function() return unpack(self.db.profile.progress25) end,
						set = function(r, g, b, a)
							self.db.profile.progress25 = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					color50 = { order = 8, type = "color",
						name = L["Halfway"].." (50%)", desc = L["progressDesc"]:format("50%"),
						disabled = function() return not self.db.profile.progressCustom end,
						hidden = function() return not self.db.profile.showProgress50 end,
						get = function() return unpack(self.db.profile.progress50) end,
						set = function(r, g, b, a)
							self.db.profile.progress50 = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					color75 = { order = 9, type = "color",
						name = "3/4 (75%)", desc = L["progressDesc"]:format("75%"),
						disabled = function() return not self.db.profile.progressCustom end,
						hidden = function() return not self.db.profile.showProgress75 end,
						get = function() return unpack(self.db.profile.progress75) end,
						set = function(r, g, b, a)
							self.db.profile.progress75 = {r, g, b, a}
							self.core:RefreshTracker()
						end },
					color100 = { order = 10, type = "color",
						name = L["End"].." (100%)", desc = L["progressDesc"]:format("100%"),
						disabled = function() return not self.db.profile.progressCustom end,
						get = function() return unpack(self.db.profile.progressEnd) end,
						set = function(r, g, b, a)
							self.db.profile.progressEnd = {r, g, b, a}
							self.core:RefreshTracker()
						end } } },
			header = { order = oldMenu.order + 3, type = "header" } } }
	for i, color in ipairs(colorList) do
		local name = colorList[i].name
		local extra = colorList[i].extra
		local hasAlpha = false
		if self.db.profile[name].a then
			hasAlpha = true
		end
		local colorMenu = { order = 1200 + i, type = "color",			
			name = L[name],
			desc = L["Change color settings for"].." "..L[name],
			get = function() return getColor(name) end,
			set = function(r, g, b, a) setColor(name, r, g, b, a) end,
			hasAlpha = hasAlpha }
		if ( extra ) then
			menu.args[name] = { order = 1200 + i, type = "group",
				name = L[name],
				desc = L["Change color settings for"].." "..L[name],
				args = {
					name = colorMenu,
					extra = {
						type = "toggle",
						name = "- "..L["colorBy"]:format(L[extra]),
						desc = L["colorBy"]:format(L[extra]),
						order = 1200 + i + 1,
						get = function() return getToggle(name) end,
						set = function(v) setToggle(name, v) end } } }
		else
			menu.args[name] = colorMenu
		end
	end
	return menu
end

function Coloring:Colorize(text, variable, range, zone, level)
	--Colorize text, using variable for settings
	--Range is for threshold colorizing, so is zone
	--If level and threshold then colorize by level

	--Range and zone is optional, will be used if threshold=true
	if ( range ) and ( self.db.profile[variable].threshold ) then
		--Color by range, for example got/needed
		local color
		if ( self.db.profile.progressCustom ) then
			color = self:GetProgressHexColor(range)
		else
			color = Crayon:GetThresholdHexColor(range)
		end
		return Crayon:Colorize(color, text)
	elseif ( zone ) and ( self.db.profile[variable].threshold ) then
		--Color by zone
		local color
		if ( self.db.profile.levelCustom ) then
			color = self:GetZoneDifficultyHexColor(zone)
		else
			local r, g, b = Tourist:GetLevelColor(zone)
			color = ("%02x%02x%02x"):format(r*255, g*255, b*255)
		end
		return Crayon:Colorize(color, text)
	elseif ( level ) and ( self.db.profile[variable].threshold ) then
		local color
		if ( self.db.profile.levelCustom ) then
			color = self:GetLevelHexColor(level)
		else
			color = GetDifficultyColor(level)
			color = ("%02x%02x%02x"):format(color.r * 255, color.g * 255, color.b * 255)
		end
		return Crayon:Colorize(color, text)
	end
	--Take predefined color
	local color = ("%02x%02x%02x"):format(self.db.profile[variable].r, self.db.profile[variable].g, self.db.profile[variable].b)
	return Crayon:Colorize(color, text)
end

function Coloring:GetColor(variable)
	return self.db.profile[variable]
end

function Coloring:GetLevelHexColor(level)
	local r, g, b
	--GetDifficultyColor() is a Blizzard function
	--Returns r,g,b (each ranging from 0 to 1) in table form
	local color = GetDifficultyColor(level)
	--QuestDifficultyColor is a table defined in Blizzard's QuestLogFrame.lua
	if ( color == QuestDifficultyColor.trivial ) then
		r, g, b = unpack(self.db.profile.levelTrivial)
	elseif ( color == QuestDifficultyColor.standard ) then
		r, g, b = unpack(self.db.profile.levelStandard)
	elseif color == QuestDifficultyColor.difficult then
		r, g, b = unpack(self.db.profile.levelDifficult)
	elseif color == QuestDifficultyColor.verydifficult then
		r, g, b = unpack(self.db.profile.levelVeryDifficult)
	elseif color == QuestDifficultyColor.impossible then
		r, g, b = unpack(self.db.profile.levelImpossible)
	end
	return ("%02x%02x%02x"):format(r*255, g*255, b*255)
end

function Coloring:GetProgressHexColor(progress)
	local r, g, b
	local db = self.db.profile
	if ( progress <= 0 ) then
		r, g, b = unpack(db.progressStart)
	elseif ( progress >= 1 ) then
		r, g, b = unpack(db.progressEnd)
	else
		local r1, g1, b1, r2, g2, b2
		if ( db.showProgress25 ) then
			if ( db.showProgress50 ) then
				--25%, 50%, and 75% progress colors are being used
				if ( db.showProgress75 ) then
					if ( progress <= 0.25 ) then
						progress = progress / 0.25
						r1, g1, b1 = unpack(db.progressStart)
						r2, g2, b2 = unpack(db.progress25)
					elseif ( progress <= 0.5 ) then
						progress = (progress - 0.25) / 0.25
						r1, g1, b1 = unpack(db.progress25)
						r2, g2, b2 = unpack(db.progress50)
					elseif ( progress <= 0.75 ) then
						progress = (progress - 0.50) / 0.25
						r1, g1, b1 = unpack(db.progress50)
						r2, g2, b2 = unpack(db.progress75)
					else
						progress = (progress - 0.75) / 0.25
						r1, g1, b1 = unpack(db.progress75)
						r2, g2, b2 = unpack(db.progressEnd)
					end
				--Only the 25% and 50% progress colors are being used
				else
					if ( progress <= 0.25 ) then
						progress = progress / 0.25
						r1, g1, b1 = unpack(db.progressStart)
						r2, g2, b2 = unpack(db.progress25)
					elseif ( progress <= 0.5 ) then
						progress = (progress - 0.25) / 0.25
						r1, g1, b1 = unpack(db.progress25)
						r2, g2, b2 = unpack(db.progress50)
					else
						progress = (progress - 0.5) / 0.5
						r1, g1, b1 = unpack(db.progress50)
						r2, g2, b2 = unpack(db.progressEnd)
					end
				end
			--Only the 25% and 75% progress colors are being used
			elseif ( db.showProgress75 ) then
				if ( progress <= 0.25 ) then
					progress = progress / 0.25
					r1, g1, b1 = unpack(db.progressStart)
					r2, g2, b2 = unpack(db.progress25)
				elseif ( progress <= 0.75 ) then
					progress = (progress - 0.25) / 0.5
					r1, g1, b1 = unpack(db.progress25)
					r2, g2, b2 = unpack(db.progress75)
				else
					progress = (progress - 0.75) / 0.25
					r1, g1, b1 = unpack(db.progress75)
					r2, g2, b2 = unpack(db.progressEnd)
				end
			--Only the 25% progress color is being used
			else
				if ( progress <= 0.25 ) then
					progress = progress / 0.25
					r1, g1, b1 = unpack(db.progressStart)
					r2, g2, b2 = unpack(db.progress25)
				else
					progress = (progress - 0.25) / 0.75
					r1, g1, b1 = unpack(db.progress25)
					r2, g2, b2 = unpack(db.progressEnd)
				end
			end
		elseif ( db.showProgress50 ) then
			--Only the 50% and 75% progress colors are being used
			if ( db.showProgress75 ) then
				if ( progress <= 0.5 ) then
					progress = progress / 0.5
					r1, g1, b1 = unpack(db.progressStart)
					r2, g2, b2 = unpack(db.progress50)
				elseif ( progress <= 0.75 ) then
					progress = (progress - 0.5) / 0.25
					r1, g1, b1 = unpack(db.progress50)
					r2, g2, b2 = unpack(db.progress75)
				else
					progress = (progress - 0.75) / 0.25
					r1, g1, b1 = unpack(db.progress75)
					r2, g2, b2 = unpack(db.progressEnd)
				end
			--Only the 50% progress color is being used
			else
				if ( progress <= 0.5 ) then
					progress = progress / 0.5
					r1, g1, b1 = unpack(db.progressStart)
					r2, g2, b2 = unpack(db.progress50)
				else
					progress = (progress - 0.5) / 0.5
					r1, g1, b1 = unpack(db.progress50)
					r2, g2, b2 = unpack(db.progressEnd)
				end
			end
		--Only the 75% progress color is being used
		elseif ( db.showProgress75 ) then
			if ( progress <= 0.75 ) then
				progress = progress / 0.75
				r1, g1, b1 = unpack(db.progressStart)
				r2, g2, b2 = unpack(db.progress75)
			else
				progress = (progress - 0.75) / 0.25
				r1, g1, b1 = unpack(db.progress75)
				r2, g2, b2 = unpack(db.progressEnd)
			end
		--No additional progress colors are being used, aside from start and end
		else
			r1, g1, b1 = unpack(db.progressStart)
			r2, g2, b2 = unpack(db.progressEnd)
		end
		r, g, b = r1+(r2-r1)*progress, g1+(g2-g1)*progress, b1+(b2-b1)*progress
	end
	return ("%02x%02x%02x"):format(r*255, g*255, b*255)
end

function Coloring:GetZoneDifficultyHexColor(zone)
	local r, g, b = 1, 1, 1
	local zLevelLow, zLevelHigh = Tourist:GetLevel(zone)
	if ( zLevelLow > 0 ) and ( zLevelHigh > 0 ) then
		local playerLevel = UnitLevel("player")
		if ( playerLevel <= zLevelLow - 3 ) then
			r, g, b = unpack(self.db.profile.levelImpossible)
		elseif ( playerLevel <= (zLevelLow + zLevelHigh)/2 ) then
			local progress = (playerLevel - zLevelLow)/(((zLevelLow + zLevelHigh)/2) - zLevelLow)
			r1, g1, b1 = unpack(self.db.profile.levelVeryDifficult)
			r2, g2, b2 = unpack(self.db.profile.levelDifficult)
			r, g, b = r1+(r2-r1)*progress, g1+(g2-g1)*progress, b1+(b2-b1)*progress
		elseif ( playerLevel < zLevelHigh + 3 ) then
			local progress = (playerLevel - (zLevelLow + zLevelHigh)/2)/(zLevelHigh - (zLevelLow + zLevelHigh)/2)
			r1, g1, b1 = unpack(self.db.profile.levelDifficult)
			r2, g2, b2 = unpack(self.db.profile.levelStandard)
			r, g, b = r1+(r2-r1)*progress, g1+(g2-g1)*progress, b1+(b2-b1)*progress
		else
			r, g, b = unpack(self.db.profile.levelTrivial)
		end
	end
	return ("%02x%02x%02x"):format(r*255, g*255, b*255)
end
