local _G = _G

local tonumber = _G.tonumber
local select = _G.select
local next = _G.next
local ipairs = _G.ipairs
local pairs = _G.pairs
local ceil = _G.ceil
local floor = _G.floor
local abs = _G.abs
local getglobal = _G.getglobal

local CreateFrame = _G.CreateFrame
local GetCVar = _G.GetCVar
local SetCVar = _G.SetCVar
local UnitPowerType = _G.UnitPowerType
local UnitHealth = _G.UnitHealth
local UnitMana = _G.UnitMana
local GetActionInfo = _G.GetActionInfo
local GetActionCount = _G.GetActionCount
local IsConsumableAction = _G.IsConsumableAction
local GetItemCount = _G.GetItemCount
local UnitIsDeadOrGhost = _G.UnitIsDeadOrGhost
local SPELL_REAGENTS = _G.SPELL_REAGENTS


local Gratuity = LibStub("LibGratuity-3.0")
local Deformat
do
-- This is very much a rippof of Deformat, simplified in that it does not
-- handle merged patterns
	local select = select
	local tonumber = tonumber
	local string_match = string.match
	local function donothing() end

	local sequences = {
		["%d*d"] = "%%-?%%d+",
		["s"] = ".+",
		["[fg]"] = "%%-?%%d+%%.?%%d*",
		["%%%.%d[fg]"] = "%%-?%%d+%%.?%%d*",
		["c"] = ".",
	}

	local function get_first_pattern(s)
		local first_pos, first_pattern
		for pattern in pairs(sequences) do
			local pos = s:find("%%%%"..pattern)
			if pos and (not first_pos or pos < first_pos) then
				first_pos, first_pattern = pos, pattern
			end
		end
		return first_pattern
	end

	local function get_indexed_pattern(s, i)
		for pattern in pairs(sequences) do
			if s:find("%%%%" .. i .. "%%%$" .. pattern) then
				return pattern
			end
		end
	end

	local function bubble(f, i, a1, ...)
		if f[i] then a1 = tonumber(a1) end
		if not ... then return a1 end
		return a1, bubble(f, i + 1, ...)
	end

	local function bubble_num(f, o, i, ...)
		if not o[i] then return end
		local a1 = select(o[i], ...)
		if f[i] then a1 = tonumber(a1) end
		return a1, bubble_num(f, o, i + 1, ...)
	end

	local function unpattern_unordered(unpattern, f)
		local i = 1
		while true do
			local pattern = get_first_pattern(unpattern)
			if not pattern then return unpattern, i > 1 end

			unpattern = unpattern:gsub("%%%%" .. pattern, "(" .. sequences[pattern] .. ")", 1)
			f[i] = (pattern ~= "c" and pattern ~= "s")
			i = i + 1
		end
	end

	local function unpattern_ordered(unpattern, f)
		local i = 1
		while true do
			local pattern = get_indexed_pattern(unpattern, i)
			if not pattern then return unpattern, i > 1 end

			unpattern = unpattern:gsub("%%%%" .. i .. "%%%$" .. pattern, "(" .. sequences[pattern] .. ")", 1)
			f[i] = (pattern ~= "c" and pattern ~= "s")
			i = i + 1
		end
	end

	local function curry(pattern)
		local unpattern, f, matched = '^' .. pattern:gsub("([%(%)%.%*%+%-%[%]%?%^%$%%])", "%%%1") .. '$', {}
		if not pattern:find("%1$", nil, true) then
			unpattern, matched = unpattern_unordered(unpattern, f)
			if not matched then
				return donothing
			else
				return function(text)
					return bubble(f, 1, string_match(text, unpattern))
				end
			end
		else
			unpattern, matched = unpattern_ordered(unpattern, f)
			if not matched then
				return donothing
			else
				local i, o = 1, {}
				pattern:gsub("%%(%d)%$", function(w) o[i] = tonumber(w); i = i + 1; end)
				return function(text)
					return bubble_num(f, o, 1, string_match(text, unpattern))
				end
			end
		end
	end

	local patterns = setmetatable({}, {
--		__mode = "k", -- you can't make 'weak' strings this way.
		__index = function (self, pattern)
			local c = curry(pattern)
			self[pattern] = c
			return c
		end,
	})

	Deformat = function (string, pattern)
		return patterns[pattern](string)
	end
end

local L = LibStub("AceLocale-3.0"):GetLocale("lern2count")

local ltc = LibStub("AceAddon-3.0"):NewAddon("lern2count", "AceConsole-3.0", "AceTimer-3.0")

local Plugins = {}
local CostList = _G.setmetatable({}, { __mode = "kv" })
local ReagentList = {}
local CountList = {}

local MAX_COUNT_UPDATED_PER_UPDATE = 24 -- two full bars

--[[
	count types:
		0 : mana
		1 : rage
		P : focus
		3 : energy
		H : health
		R : reagents
		D : stack
]]

local CostTable = {
	[0] = { base = _G.MANA_COST, timed = _G.MANA_COST_PER_TIME },
	[1] = { base = _G.RAGE_COST, timed = _G.RAGE_COST_PER_TIME },
	P   = { base = _G.FOCUS_COST, timed = _G.FOCUS_COST_PER_TIME },
	[3] = { base = _G.ENERGY_COST, timed = _G.ENERGY_COST_PER_TIME },
	H   = { base = _G.HEALTH_COST, timed = _G.HEALTH_COST_PER_TIME },
}

local ltc = ltc

local defaults = {
	profile = {
		colors = {
			[0] = "88aaff",
			[1] = "ffff33",
			P   = "ff8033",
			[3] = "ffff33",
			H   = "ff0033",
			R   = "00ff33",
			D   = "ffffff",
		},
		sizes = {
			[0] = 14,
			[1] = 14,
			P   = 14,
			[3] = 14,
			H   = 14,
			R   = 14,
			D   = 14,
		},
		hidden = {
			[0] = false,
			[1] = false,
			P   = false,
			[3] = false,
			H   = false,
			R   = false,
			D   = false,
		},
		hidezero = {
			[0] = false,
			[1] = true,
			P   = false,
			[3] = true,
			H   = false,
			R   = false,
			D   = false,
		}
	}
}

local function getRGB(hex)
	return tonumber(hex:sub(1, 2), 16) / 255,
		tonumber(hex:sub(3, 4), 16) / 255,
		tonumber(hex:sub(5, 6), 16) / 255
end

local function setRGB(r, g, b)
	return("%02x%02x%02x"):format(floor(r * 255), floor(g * 255), floor(b * 255))
end

do
	local nameMap = {
		[0] = L["mana"],
		[1] = L["rage"],
		P   = L["focus"],
		[3] = L["energy"],
		H   = L["health"],
		R   = L["reagent"],
		D   = L["count"],
	}
	local options = {
		desc = L["An addon to show reagent count on action buttons"],
		type = "group",
		childGroups = "tab",
		args = {
			options = {
				order = 1,
				name = L["Options"],
				type = "group",
				args = {
					colors = {
						name = L["colors"],
						desc = L["Color of the text for the different elements"],
						type = "group",
						args = {},
					},
					sizes = {
						name = L["sizes"],
						desc = L["Size of the text for the different elements"],
						type = "group",
						args = {},
					},
					hidden = {
						name = L["hide"],
						desc = L["Hide the text for the different elements"],
						type = "group",
						args = {},
					},
					hidezero = {
						name = L["hide zero"],
						desc = L["Hide the count if it is zero"],
						type = "group",
						args = {},
					},
				},
			},
		},
	}
	function ltc:OnInitialize()
		self.db = LibStub("AceDB-3.0"):New("lern2countDB", defaults)
		local p = self.db.profile
		local args = options.args.options.args
		for k, v in pairs(nameMap) do
			local index = k
			args.colors.args[v] = {
				name = v,
				desc = L["Color for %s"]:format(v),
				type = "color",
				get = function () return getRGB(self.db.profile.colors[index]) end,
				set = function (_, r, g, b) self.db.profile.colors[index] = setRGB(r, g, b) ; self:UpdateAllCounts(true) end,
			}
			args.sizes.args[v] = {
				name = v,
				desc = L["Fontsize for %s"]:format(v),
				type = "range",
				min = 6,
				max = 20,
				step = 1,
				get = function () return self.db.profile.sizes[index] end,
				set = function (_, v) self.db.profile.sizes[index] = v ; self:UpdateAllCounts(true) end,
			}
			args.hidden.args[v] = {
				name = v,
				desc = L["Hide %s"]:format(v),
				type = "toggle",
				get = function () return self.db.profile.hidden[index] end,
				set = function (_, v) self.db.profile.hidden[index] = v ; self:CheckTooltipOption() ; self:UpdateAllCounts(true) end,
			}
			args.hidezero.args[v] = {
				name = v,
				desc = L["Hide %s"]:format(v),
				type = "toggle",
				get = function () return self.db.profile.hidezero[index] end,
				set = function (_, v) self.db.profile.hidezero[index] = v ; self:UpdateAllCounts(true) end,
			}
		end

		local AceDBOptions = LibStub("AceDBOptions-3.0", true)
		if AceDBOptions then
			options.args.profiles = AceDBOptions:GetOptionsTable(self.db)
			options.args.profiles.order = 200
		end

		LibStub("AceConfig-3.0"):RegisterOptionsTable("lern2count", options)
		self.optionsFrame = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("lern2count", "lern2count")
		self:RegisterChatCommand("lern2count", "OnChatCommand")
		self.OnInitialize = nil
	end
end

function ltc:OnChatCommand(input)
	if not input or input:trim() == "" then
		InterfaceOptionsFrame_OpenToFrame(self.optionsFrame)
	else
		LibStub("AceConfigCmd-3.0").HandleCommand(self, "lern2count", "lern2count", input)
	end
end

do
	local checktypes = { 0, 1, "P", 3, "H", "R" }
	function ltc:CheckTooltipOption()
		if GetCVar("UberTooltips") ~= "0" then return end
		for _,v in ipairs(checktypes) do
			if not self.db.profile.hidden[v] then
				self:Print(L["UberTooltips are required for lern2count correctly displaying information, activating it"])
				SetCVar("UberTooltips", "1")
				break
			end
		end
	end
end

local EventHandlers = {}

EventHandlers.UNIT_HEALTH = function (_, unit)
	if unit == "player" or unit == "pet" then
		ltc:UpdateAllCounts()
	end
end

local LibDruidMana = select(2, UnitClass("player")) == "DRUID" and LibStub("LibDruidMana-1.0", true)

if LibDruidMana then
	LibDruidMana:AddListener(function() EventHandlers.UNIT_HEALTH(nil, "player") end)
else
	EventHandlers.UNIT_MANA = EventHandlers.UNIT_HEALTH
end

EventHandlers.UNIT_RAGE = EventHandlers.UNIT_HEALTH
EventHandlers.UNIT_ENERGY = EventHandlers.UNIT_HEALTH
EventHandlers.UNIT_FOCUS = EventHandlers.UNIT_HEALTH

EventHandlers.UNIT_AURA = function (_, unit)
	if unit == "player" then
		ltc:FullUpdate()
	end
end

EventHandlers.BAG_UPDATE = function ()
	ltc:UpdateAllCounts()
end

EventHandlers.ACTIONBAR_SLOT_CHANGED = function ()
	ltc:FullUpdate()
end

EventHandlers.ACTIONBAR_HIDEGRID = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.ACTIONBAR_PAGE_CHANGED = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.ACTIONBAR_UPDATE_USABLE = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.UPDATE_BONUS_ACTIONBAR = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.UPDATE_SHAPESHIFT_FORMS = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.PET_BAR_HIDEGRID = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.PET_BAR_UPDATE = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.PET_BAR_UPDATE_COOLDOWN = EventHandlers.ACTIONBAR_SLOT_CHANGED
EventHandlers.MODIFIER_STATE_CHANGED = EventHandlers.ACTIONBAR_SLOT_CHANGED

local combat
EventHandlers.PLAYER_REGEN_DISABLED = function ()
	combat = true
	ltc:UpdateAllCounts()
end

EventHandlers.PLAYER_REGEN_ENABLED = function ()
	combat = false
	ltc:UpdateAllCounts()
end

local ltcFrame = CreateFrame("Frame", "ltcFrame", nil)
ltcFrame:Hide()

for event, handler in pairs(EventHandlers) do
	ltcFrame:RegisterEvent(event)
	ltcFrame[event] = handler
end
ltcFrame:SetScript("OnEvent", function (self, event, ...)
	local handler = self[event]
	if handler then handler(self, ...) end
end)
ltcFrame:SetScript("OnUpdate", function (self, elapsed)
	ltc:UpdateStep()
end)

function ltc:OnDisable()
	self.update = nil
	ltcFrame:Hide()
end

function ltc:OnEnable()
	self:ScheduleTimer(self.FullUpdate, 1, self) --Delay initializing the addon a second, thats enough for everything to finish what its doing (I hope)
end

local GratuityHandler = {
	action = function (id) Gratuity:SetAction (id) end,
	pet = function (id) Gratuity:SetPetAction (id) end,
	shapeshift = function (id) Gratuity:SetShapeshift (id) end,
}

local GetCost
if LibDruidMana then
	GetCost = function(ctype, type, cost)
		if type == "pet" then
			return floor(UnitMana("pet") / cost)
		elseif tonumber(ctype) then -- mana, rage, energy
			local ptype = UnitPowerType("player")
			if ptype == ctype then
				if ctype == 3 then
					if not UnitIsDeadOrGhost("player") then
						local v = cost - UnitMana("player")
						if v > 0 then
							return ceil(v / 20)
						end
					end
				else
					return floor(UnitMana("player") / cost)
				end
			elseif ctype == 0 then
				return floor(LibDruidMana:GetCurrentMana() / cost)
			else
				return 0
			end
		elseif ctype == "H" then
			return floor(UnitHealth("player") / cost)
		end
	end
else
	GetCost = function(ctype, type, cost)
		if type == "pet" then
			return floor(UnitMana("pet") / cost)
		elseif tonumber(ctype) then -- mana, rage, energy
			if ctype == UnitPowerType("player") then
				if ctype == 3 then -- energy
					if not UnitIsDeadOrGhost("player") then
						local v = cost - UnitMana("player")
						if v > 0 then
							return ceil(v / 20)
						end
					end
				else
					return floor(UnitMana("player") / cost)
				end
			else
				return 0
			end
		elseif ctype == "H" then
			return floor(UnitHealth("player") / cost)
		end
	end
end


function ltc:UpdateCount(frame, info)
	if not info or not info.shown then return end
	local id = info.id
	local type = info.type

	local ctype, cost
	local cl = CostList[frame]
	if cl then
		ctype = cl.type
		cost = cl.cost
	end

	local reagent = ReagentList[frame]
	if reagent and IsConsumableAction (id) then
		reagent = GetActionCount(id)
	else
		reagent = nil
	end

	local count = cost and GetCost(ctype, type, cost)

	if combat and reagent and (not count or reagent < count) or not combat and reagent then
		count =  reagent
		ctype = "R"
	end

	if not count and type == "action" and IsConsumableAction (id) then
		count = GetActionCount (id)
		ctype = "D"
	end

	local p = self.db.profile
	if p.hidezero[ctype] and count == 0 then
		count = nil
	end
	if count and ctype and not p.hidden[ctype] then
		frame:SetFormattedText("|cff%s%d|r", p.colors[ctype], count)
		local new_height = p.sizes[ctype]

		local font, height, flag = frame:GetFont()
		if abs(height - new_height) > .5 then
			frame:SetFont(font, new_height, flag)
		end
	else
		(frame.ltc_prev_SetText or frame.SetText)(frame, "")
	end
end

function ltc:GetCountList()
	return CountList
end

function ltc:UpdateAllCounts()
	if not self.update then
		self.update, self.step = self.UpdateCountStep
		ltcFrame:Show()
	end
end

function ltc:FullUpdate()
	self:CheckTooltipOption()
	local UpdateCountListStep = self.UpdateCountListStep
	if self.update ~= UpdateCountListStep then
		self.update, self.step = UpdateCountListStep
		ltcFrame:Show()
	end
end

function ltc:RegisterPlugin(func)
	Plugins[#Plugins + 1] = func
end

local recursive
local function Frame_SetText(self, ...)
	if recursive then
		error("Recursive call detected !")
	end
	recursive = true
	local info = CountList[frame]
	if info then
		ltc:UpdateCount(self, info)
	else
		self:ltc_prev_SetText(...)
	end
	recursive = nil
end

local function AddFrameCount(frame, type, id)
	if type ~= "action" or GetActionInfo(id) then
		local t = CountList[frame]
		if not t then
			t = {}
			CountList[frame] = t
		end
		t.type = type
		t.id = id
		t.shown = true
		if frame.SetText ~= Frame_SetText then
			frame.ltc_prev_SetText = frame.SetText
			frame.SetText = Frame_SetText
		end
	end
end

function ltc:UpdateCountListStep(step)
	if not step then
		for _, v in pairs(CountList) do
			if v.shown then
				v.shown = false
			end
		end
		return next(Plugins) or "end"
	elseif step == "end" then
		for f, v in pairs(CountList) do
			if v.shown == false then
				v.shown = nil
				f:SetText("")
			end
		end
		return nil, self.UpdateCostListStep
	else
		local func = Plugins[step]
		func (AddFrameCount)
		return next(Plugins, step) or "end"
	end
end

function ltc:UpdateCostListStep(step)
	local max = MAX_COUNT_UPDATED_PER_UPDATE
	while max > 0 do
		local frame, info = next(CountList, step)
		if not frame then
			return nil, self.UpdateCountStep
		end
		self:UpdateCostList(frame, info)
		max = max - 1
		step = frame
	end
	return step
end

function ltc:UpdateCountStep(step)
	local max = MAX_COUNT_UPDATED_PER_UPDATE
	while max > 0 do
		local frame, info = next(CountList, step)
		if not frame then
			return
		end
		self:UpdateCount(frame, info)
		max = max - 1
		step = frame
	end
	return step
end

function ltc:UpdateStep()
	local update = self.update
	if update then
		local new_step, new_update = update(self, self.step)
		if not new_step then
			self.update, self.step = new_update, nil
		else
			self.step = new_step
		end
	else
		ltcFrame:Hide()
	end
end

function ltc:UpdateCostList(frame, info)
	if info and info.shown then
		GratuityHandler[info.type](info.id)
		if Gratuity:NumLines() >= 2 then

			local found
			for type, c in pairs(CostTable) do
				local text = Gratuity:GetLine(2)
				local cost = text and (Deformat(text, c.base) or Deformat(text, c.timed))

				if cost then
					local t = CostList[frame]
					if not t then
						t = {}
						CostList[frame] = t
					end
					t.type = type
					t.cost = tonumber(cost)
					found = true
					break
				end
			end
			if not found and CostList[frame] then
				CostList[frame].cost = nil
			end

			ReagentList[frame] = Gratuity:Find(SPELL_REAGENTS, 2, 99, false, true, false) and true
		end
	end
end

_G.ltc = ltc
