--------------------------
--		Initialization     --
--------------------------

local MAX_ROWS = 5
local MAX_BUTTONS = 5

local buttons = {} -- holds the rows which hold the actual buttons
local buttonFactory -- declare outside do statement so it can be used throughout file

local selector 
local selectedRow
local selectedButton

-- keybinds 	- these are used in OnFrameShow to set the temporary keybinds
local keybindFireAction, keybindMoveSelectorUp, keybindMoveSelectorDown, keybindMoveSelectorRight, keybindMoveSelectorLeft, keybindCloseWindow1, keybindCloseWindow2

local kGo = LibStub("AceAddon-3.0"):GetAddon("kGo")
local Grid = kGo:NewModule("Grid", "AceEvent-3.0")
local db

local frame

local defaults = {
	char = {
		rows = 3,
		buttons = 3, 
		
		initialSelectedRow = 2,
		initialSelectedButton = 2,
		
		keybinds = {
			fireAction = "F",
			moveSelectorUp = "W",
			moveSelectorDown = "S",
			moveSelectorRight = "D",
			moveSelectorLeft = "A",
			closeWindow1 = "TAB",
			closeWindow2 = "ESCAPE",
		},
		
		actions = {
			[1] = {
				[1] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[2] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[3] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[4] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[5] = {
					type = "empty",
					value = "",
					texture = "",
				},
			},
			[2] = {
				[1] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[2] = {
					type = "item",
					value = "Hearthstone",
					texture = "Interface\\Icons\\INV_Misc_Rune_01",
				},
				[3] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[4] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[5] = {
					type = "empty",
					value = "",
					texture = "",
				},
			},
			[3] = {
				[1] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[2] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[3] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[4] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[5] = {
					type = "empty",
					value = "",
					texture = "",
				},
			},
			[4] = {
				[1] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[2] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[3] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[4] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[5] = {
					type = "empty",
					value = "",
					texture = "",
				},
			},
			[5] = {
				[1] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[2] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[3] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[4] = {
					type = "empty",
					value = "",
					texture = "",
				},
				[5] = {
					type = "empty",
					value = "",
					texture = "",
				},
			},
		},
	},
}

local options
Grid.options = {
--	type = "group",
--	name = "Grid Display",
--	args = {
	container = {
		type = "group",
		name = "Grid Display",
		args = {
			intro = {
				order = 1,
				type = "description",
				name = "Grid Display will display buttons in a grid when you press the keybind. You can then use WASD to select a button with the highlighter and press the keybind again to execute that button. You can also click on the buttons. Currently the buttons are only configureable from this menu and can only take spells, macros and items.",
			},
			rows = {
				order = 2, 
				type = "range",
				name = "Number of rows",
				get = function() return db.char.rows end,
				set = function(info, value) 
							db.char.rows = value 
							Grid:CreateRows()
							Grid:CreateButtons()
							Grid:CreateDummyActionTables()
							Grid:Refresh()
						end,
				min = 1, max = 5, step = 1,
			},
			buttons = {
				order = 3,
				type = "range",
				name = "Number of buttons per row",
				get = function() return db.char.buttons end,
				set = function(info, value) 
							db.char.buttons = value 	
						    Grid:CreateButtons()
							Grid:CreateDummyActionTables()
							Grid:Refresh()
						end,
				min = 1, max = 5, step = 1,
			},
				initialRow = {
				order = 4,
				type = "range",
				name = "Initially highlighted row",
				get = function() return db.char.initialSelectedRow end,
				set = function(info, value) 
							-- quick fix to avoid errors. the real fix should be able to dynamically set 
							-- the max value to db.char.rows
							if value > db.char.rows then
								value = math.floor(db.char.rows/2)
							end
							db.char.initialSelectedRow = value 
						end,
				min = 1, max = 5, step = 1,				
			},
				initialButton = {
				order = 5,
				type = "range",
				name = "Initially highlighted button",
				get = function() return db.char.initialSelectedButton end,
				set = function(info, value) 
							-- same as above for initialRow
							if value > db.char.buttons then
								value = math.floor(db.char.buttons/2)
							end
							db.char.initialSelectedButton = value 
						end,
				min = 1, max = 5, step = 1,				
			},
			actions = {
				order = 6,
				type = "group",
				name = "Actions",
				desc = "Configure the button contents here.",
				args = {
					
				}
			},
			keybindings = {
			order = 7,
			type = "group",
			name = "Keybindings",
			desc = "Configure the keybindings which control the behaviour of the window when it is open.",
			args = {
				fireAction = {
					order = 1,
					name = "Fire Action",
					type = "keybinding",
					desc = "Fire the highlighted action. eg cast spell which highlighter is over.",
					get = function() return db.char.keybinds.fireAction end,
					set = function(info, value) 
								db.char.keybinds.fireAction = value 
								keybindFireAction = value
							end,
				},
				moveSelectorUp = {
					order = 2,
					name = "Move Selector Up",
					type = "keybinding",
					desc = "Move selector up.",
					get = function() return db.char.keybinds.moveSelectorUp end,
					set = function(info, value) 
								db.char.keybinds.moveSelectorUp = value 
								keybindMoveSelectorUp = value
							end,
				},
				moveSelectorDown = {
					order = 3,
					name = "Move Selector Down",
					type = "keybinding",
					desc = "Move selector down.",
					get = function() return db.char.keybinds.moveSelectorDown end,
					set = function(info, value) 
								db.char.keybinds.moveSelectorDown = value 
								keybindMoveSelectorDown = value
							end,
				},			
				moveSelectorRight = {
					order = 4,
					name = "Move Selector Right",
					type = "keybinding",
					desc = "Move selector right.",
					get = function() return db.char.keybinds.moveSelectorRight end,
					set = function(info, value) 
								db.char.keybinds.moveSelectorRight = value 
								keybindMoveSelectorRight = value
							end,
				},
				moveSelectorLeft = {
					order = 5,
					name = "Move Selector Left",
					type = "keybinding",
					desc = "Move selector left.",
					get = function() return db.char.keybinds.moveSelectorLeft end,
					set = function(info, value) 
								db.char.keybinds.moveSelectorLeft = value 
								keybindMoveSelectorLeft = value
							end,
				},
				closeWindow1 = {
					order = 6,
					name = "Close Window 1",
					type = "keybinding",
					desc = "Close window.",
					get = function() return db.char.keybinds.closeWindow1 end,
					set = function(info, value) 
								db.char.keybinds.closeWindow1 = value 
								keybindCloseWindow1 = value
							end,
				},
				closeWindow2 = {
					order = 7,
					name = "Close Window 2",
					type = "keybinding",
					desc = "Close window.",
					get = function() return db.char.keybinds.closeWindow2 end,
					set = function(info, value) 
								db.char.keybinds.closeWindow2 = value 
								keybindCloseWindow2 = value
							end,
				},
			}
		},
		}
	}
--	}
}

--[[
creates the options table for the actions 
only adds a category dropdown option and a value name (spell/item/etc name)
note on category values table: keys have same name as values due to the saving system
if you use numbered keys eg {"item", "spell"} it will really be {[1] = "item", [2] = "spell"}
the key is what is passed as the value in the set function. Hence it will save a number.
I prefer to save the values as strings to avoid confusion and having a reference table which
tells you what each number stands for. Unnecessary.
]]
local function createActionsOptions()
	local actionsOpts = Grid.options.container.args.actions.args 
	local rowKey, buttonKey
	for i=1, MAX_ROWS do
		rowKey = "Row"..i
		actionsOpts[rowKey] = {}
		actionsOpts[rowKey].type = "group"
		actionsOpts[rowKey].name = "Row "..i
		actionsOpts[rowKey].order = i
		actionsOpts[rowKey].args = {}
		for j=1, MAX_BUTTONS do
			buttonKey = "Button"..j
			actionsOpts[rowKey].args[buttonKey] = {}
			actionsOpts[rowKey].args[buttonKey].type = "group"
			actionsOpts[rowKey].args[buttonKey].name = "Button "..j
			actionsOpts[rowKey].args[buttonKey].order = j
			actionsOpts[rowKey].args[buttonKey].args = {
				category = {
					order = 1,
					type = "select",
					name = "Category",
					get = function() return db.char.actions[i][j].type end,
					set = function(info, value) db.char.actions[i][j].type = value    
						
						end,
					values = {["item"] = "item", ["spell"] = "spell",["empty"] = "empty"},
					style = "radio",
				},
				name = {
					order = 2,
					type = "input",
					name = "Item/Spell Name",
					usage = "<spell or item name>",
					get = function() return db.char.actions[i][j].value end,
					set = function(info, value) 
						db.char.actions[i][j].value = value 
						local texture = ""
						if db.char.actions[i][j].type == "item" then
							texture = select(10, GetItemInfo(value))
						elseif db.char.actions[i][j].type == "spell" then
							texture = select(3, GetSpellInfo(value))
						end
						db.char.actions[i][j].texture = texture
						if buttons[i] then
							if buttons[i][j] then 
								buttons[i][j].texture:SetTexture(texture)
							end
						end
						
					end,
				},
			}
		end
	end
end

function Grid:OnInitialize()
	self.db = kGo.db:RegisterNamespace("Grid", defaults)
	db = self.db
	
	local enabled
	if kGo.db.char.display == "Grid" then
		enabled = true
	else
		enable = false
	end
	self:SetEnabledState(enabled)
	
	self.buttons = buttons
	
	createActionsOptions()
	kGo.options.plugins["Grid Display"] = self.options
end

function Grid:OnEnable()
	-- reassign local variable
	-- this has to be assigned in onEnable, because the frame isn't created until after onInit
	frame = kGo.frame
	-- create selector frame 
	if not selector then 
		self:CreateSelector()
	end
	-- set main background window to approriate size in case compact mode was used previously
	if kGo.db.char.display == "Grid" then
		local height = (db.char.rows*100)
		local width = (db.char.buttons*100)
		frame:SetHeight(height)
		frame:SetWidth(width)
		
		frame:SetScript("OnShow", function() self:OnFrameShow() end)
	end
	
	self:RegisterEvent("PLAYER_REGEN_DISABLED")
	
	-- set local keybind variables 
	-- these are used in OnFrameShow to set the temporary keybinds
	keybindFireAction = db.char.keybinds.fireAction
	keybindMoveSelectorUp = db.char.keybinds.moveSelectorUp
	keybindMoveSelectorDown = db.char.keybinds.moveSelectorDown
	keybindMoveSelectorRight = db.char.keybinds.moveSelectorRight
	keybindMoveSelectorLeft = db.char.keybinds.moveSelectorLeft
	keybindCloseWindow1 = db.char.keybinds.closeWindow1
	keybindCloseWindow2 = db.char.keybinds.closeWindow2
end

function Grid:OnDisable()
	-- clear main frame script handlers 
	ClearOverrideBindings(frame) 
	frame:SetScript("OnShow", nil)
	frame:Hide()
end

--------------------------
--		Button Creation   --
--------------------------

-- major props to nymbia/ominous for this button factory code
do
	buttonFactory = {
		__index = function(t, k)
			local name = "kGoButton"..tostring(math.random(1, 1000))
			local button = CreateFrame("Button", name , kGoFrame, "SecureActionButtonTemplate")
			t[k] = button 
			
			button.name = name
			
			button:SetHeight(80)
			button:SetWidth(80)
			
			button:SetBackdrop({
				bgFile = "Interface/Tooltips/UI-Tooltip-Background",
				edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
				tile = true, tileSize = 0, edgeSize = 8,
				insets = {left = 2, right = 2, top = 2, bottom = 2}
			})
		
			button:SetBackdropColor(0,0,0,0.5)
			
			button.texture = button:CreateTexture(nil, "OVERLAY")
			button.texture:SetPoint("CENTER", button, "CENTER")
			button.texture:SetTexture("")
			button.texture:SetHeight(70)
			button.texture:SetWidth(70)
			
			button.highlight = button:CreateTexture(nil, "HIGHLIGHT")
			button.highlight:SetPoint("CENTER", button, "CENTER")
			button.highlight:SetTexture(1, 1, 1, 0.5)
			button.highlight:SetHeight(70)
			button.highlight:SetWidth(70)
			
			button:SetScript("PostClick", function() ClearOverrideBindings(frame) frame:Hide()  end)
			
			button:EnableMouse(true)
		end
	}
	
end

-- create extra rows if needed
function Grid:CreateRows()
	if #buttons >= db.char.rows then
		return
	end
	
	for i=1, db.char.rows do
		if not buttons[i] then
			buttons[i] = setmetatable({}, buttonFactory)
		end
	end
end
-- Create empty buttons. Mainly used when increasing number of rows/buttons in the options.
-- else buttons are created when frame is shown through indexing the buttons table and 
-- firing off the __index function which fires off the buttonfactory. 
function Grid:CreateButtons()
	local scale = kGo.db.profile.scale
	for i=1, db.char.rows do
		for j=1, db.char.buttons do
			-- figure out where to place our buttons
			-- each button is 100px wide, plus 10 for left side padding, minus 90 because 1st button should be 10px away, not 100
			local x = j*100 - 90  
			local y = -i*100 + 90
			local b = buttons[i][j]
			buttons[i][j]:SetPoint("TOPLEFT", frame, "TOPLEFT", x, y)
			buttons[i][j].texture:SetTexture(db.char.actions[i][j].texture)
		end
	end
end
-- fill in blank action tables so that buttons show up empty
function Grid:CreateDummyActionTables()
	for i=1, db.char.rows do
		for j=1, db.char.buttons do
			if not db.char.actions[i] then
				db.char.actions[i] = {}
			end
			if not db.char.actions[i][j] then
				db.char.actions[i][j] = {}
				db.char.actions[i][j].type = "none"
				db.char.actions[i][j].value = "none"
				db.char.actions[i][j].texture = ""
			end
		end
	end
end

function Grid:Refresh()
	local rows = db.char.rows
	local numButtons = db.char.buttons
	local scale = kGo.db.profile.scale
	
	-- make sure our rows exist
	if #buttons < db.char.rows then
		self:CreateRows()
	end
	
	for i=1, #buttons do
		for j=1, #buttons[i] do
			-- figure out where to place our buttons
			-- each button is 100px wide, plus 10 for left side padding, minus 90 because 1st button should be 10px away, not 100
			local x = j*100 - 90  
			local y = -i*100 + 90
			local b = buttons[i][j]
			buttons[i][j]:SetPoint("TOPLEFT", frame, "TOPLEFT", x, y)
			buttons[i][j].texture:SetTexture(db.char.actions[i][j].texture)
			
			-- hide buttons if button number is larger than db.char.buttons (number of buttons per row)
			-- hide buttons if row number is larger than db.char.rows (number of rows)
			-- show buttons if button number is lower than db.char.buttons
			-- show buttons if row number is lower than db.char.rows
			if (j <= numButtons and not buttons[i][j]:IsVisible()) and i <= rows then
				buttons[i][j]:Show()
			elseif j > numButtons or i > rows then
				buttons[i][j]:Hide()
			end
		end
	end	
	
	frame:SetWidth(numButtons*100)
	frame:SetHeight(rows*100)
	frame:SetScale(scale)
	
end
---------------------------
--		Selector Functions  --
---------------------------

-- selector is the highlighted frame behind the buttons which moves with keypresses
function Grid:CreateSelector()
	kGo.frame.selector = CreateFrame("Frame", nil, kGoFrame)
	selector = kGo.frame.selector
	
	selector:SetHeight(95)
	selector:SetWidth(95)
	
	selector:SetBackdrop({
		bgFile = "Interface/Tooltips/UI-Tooltip-Background",
		edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
		tile = true, tileSize = 0, edgeSize = 8,
		insets = {left = 2, right = 2, top = 2, bottom = 2}
	})
	
	selector:SetBackdropColor(0.8, 0.2, 0, 1)	
end
-- Below functions move the selector
function Grid:MoveSelectorUp()
	if selectedRow == 1 then 
		selectedRow = db.char.rows
	else
		selectedRow = selectedRow - 1
	end
	selector:SetPoint("CENTER", buttons[selectedRow][selectedButton], "CENTER")
	SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
end
function Grid:MoveSelectorLeft()
	if selectedButton == 1 then 
		selectedButton = db.char.buttons
	else
		selectedButton = selectedButton - 1
	end
	selector:SetPoint("CENTER", buttons[selectedRow][selectedButton], "CENTER")
	SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
end
function Grid:MoveSelectorDown()
	if selectedRow == db.char.rows then 
		selectedRow = 1
	else
		selectedRow = selectedRow + 1
	end
	selector:SetPoint("CENTER", buttons[selectedRow][selectedButton], "CENTER")
	SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
end
function Grid:MoveSelectorRight()
	if selectedButton == db.char.buttons then 
		selectedButton = 1
	else
		selectedButton = selectedButton + 1
	end
	selector:SetPoint("CENTER", buttons[selectedRow][selectedButton], "CENTER")
	SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
end

--------------------------
--		Event Handlers    --
--------------------------

function Grid:OnFrameShow()
	-- make sure our rows exist
	if #buttons < db.char.rows then
		self:CreateRows()
	end

	-- position our buttons, buttons are automatically created if need be
	for i=1, db.char.rows do
		for j=1, db.char.buttons do
			-- figure out where to place our buttons
			-- each button is 100px wide, plus 10 for left side padding, minus 90 because 1st button should be 10px away, not 100
			local x = j*100 - 90  
			local y = -i*100 + 90
			local b = buttons[i][j]
			buttons[i][j]:SetPoint("TOPLEFT", frame, "TOPLEFT", x, y)
			buttons[i][j].texture:SetTexture(db.char.actions[i][j].texture)
			local attribute 
			if db.char.actions[i][j].type == "macro" then
				attribute = "macrotext1"
			else
				attribute = db.char.actions[i][j].type.."1"
			end
			if db.char.actions[i][j].type ~= "none" then
				buttons[i][j]:SetAttribute("type1", db.char.actions[i][j].type)
				buttons[i][j]:SetAttribute(attribute, db.char.actions[i][j].value)
			end

		end
	end
	
	-- set selector position
	selector:SetPoint("CENTER", buttons[db.char.initialSelectedRow][db.char.initialSelectedButton], "CENTER")
	selector:Show()
	
	selectedRow = db.char.initialSelectedRow
	selectedButton = db.char.initialSelectedButton
	
	-- override keybinds
	SetOverrideBinding(frame, true, keybindMoveSelectorUp, "Move kGo Selector Up")
	SetOverrideBinding(frame, true, keybindMoveSelectorLeft, "Move kGo Selector Left")
	SetOverrideBinding(frame, true, keybindMoveSelectorDown, "Move kGo Selector Down")
	SetOverrideBinding(frame, true, keybindMoveSelectorRight, "Move kGo Selector Right")
	
	SetOverrideBinding(frame, true, keybindCloseWindow1, "Close kGo Window")
	SetOverrideBinding(frame, true, keybindCloseWindow2, "Close kGo Window")
	
	SetOverrideBindingClick(frame, true, keybindFireAction,buttons[selectedRow][selectedButton].name)
end

-- player enters combat
-- hide the frame, clear overriden binds
function Grid:PLAYER_REGEN_DISABLED()
	ClearOverrideBindings(frame) 
	frame:Hide()
end
--------------------------
--		Button Actions    --
--------------------------

-- refresh button textures to texture of appropriate action
function Grid:RefreshButtonTextures()
	for i=1, db.char.rows do
		for j=1, db.char.buttons do 
			buttons[i][j].texture:SetTexture(db.char.actions[i][j].texture)
		end
	end
end


--[[

									DEPRECATED CODE 
									
keeping it for now incase I need it later

-- execute the action assigned to the selected button which has been clicked/'hit' by keyboard
function Grid:FireAction(action)

	
	local type = action.type
	local value = action.value
	
	if type == "item" then
	
	elseif type == "equip" then
		EquipItemByName(value)
	elseif type =="food" then
		
	elseif type == "spell" then
	
	end
end

			local macro
			local type = db.char.actions[i][j].type
			local value = db.char.actions[i][j].value
			kGo:Print(type, value)
			if type == "spell" then
				macro = "/cast "..value.."\n/script ClearOverrideBindings(kGo.frame) kGo.frame:Hide()"
			elseif type == "food" or type == "item" then
				macro = "/use "..value.."\n/script ClearOverrideBindings(kGo.frame) kGo.frame:Hide()"
			elseif type == "equip" then
				macro = "/equip "..value.."\n/script ClearOverrideBindings(kGo.frame) kGo.frame:Hide()"
			end
			buttons[i][j]:SetAttribute("type1", "macro")
			buttons[i][j]:SetAttribute("macrotext1", macro)
--------------------------
--		Keyboard Handlers  --
--------------------------

function Grid:OnKeyDown(arg1)
	kGo:Print(arg1)
	
	-- close window
	if arg1 == "ESCAPE" or arg1 == "TAB" then
		ClearOverrideBindings(frame)
		frame:Hide()
	end
	
	-- move the selector
	if arg1 == "W" then
		self:MoveSelectorUp()
		SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
	end
	if arg1 == "A" then
		self:MoveSelectorLeft()
		SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
	end
	if arg1 == "S" then
		self:MoveSelectorDown()
		SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
	end
	if arg1 == "D" then
		self:MoveSelectorRight()
		SetOverrideBindingClick(frame, true, "F",buttons[selectedRow][selectedButton].name)
		kGo:Print("Moved right")
	end
	
	-- fire action associated with button 
	if arg1 == "F" then
		--self:FireAction(db.char.actions[selectedRow][selectedButton])
		kGo:Print(buttons[selectedRow][selectedButton].name)
	end
end
]]

