------------------------------------------------------
-- FeedOMatic.lua
------------------------------------------------------

-- Food quality by itemLevel
--
-- levelDelta = petLevel - foodItemLevel
-- levelDelta > 30 = won't eat
-- 30 >= levelDelta > 20 = 8 happiness per tick
-- 20 >= levelDelta > 10 = 17 happiness per tick
-- 10 >= levelDelta = 35 happiness per tick

-- constants
FOM_WARNING_INTERVAL = 10; -- don't warn more than once per this many seconds
MAX_KEEPOPEN_SLOTS = 150;

FOM_BonusFoodsCache = {};

-- Variables
FOM_ShouldFeed = false;
FOM_LastWarning = 0;

FOM_LastPetName = nil;

-- Anti-freeze code borrowed from ReagentInfo (in turn, from Quest-I-On):
-- keeps WoW from locking up if we try to scan the tradeskill window too fast.
FOM_TradeSkillLock = { };
FOM_TradeSkillLock.Locked = false;
FOM_TradeSkillLock.EventTimer = 0;
FOM_TradeSkillLock.EventCooldown = 0;
FOM_TradeSkillLock.EventCooldownTime = 1;

FOM_DifficultyLabels = {
	GFWUtils.Hilite("all"),
	GFWUtils.ColorText("easy", QuestDifficultyColor["standard"]),
	GFWUtils.ColorText("medium", QuestDifficultyColor["difficult"]),
	GFWUtils.ColorText("difficult", QuestDifficultyColor["verydifficult"]),
	GFWUtils.ColorText("unknown", QuestDifficultyColor["impossible"]),
};

-- State variable used to track required quantities of quest food when it's in more than one stack
FOM_Quantity = { };

-- Remember how item IDs map to food names at runtime, but don't bloat long-term memory with it...
FOM_FoodIDsToNames = {};

-- workaround for sound bug in WoW 2.2
local soundQueue = {};

function FOM_FeedButton_PostClick(self, button, down)
	if (not down) then
		if (button == "RightButton") then
			FOM_ShowOptions();
		elseif (FOM_NextFoodLink and not FOM_NoFoodError and not InCombatLockdown()) then
		elseif (FOM_NoFoodError and not IsAltKeyDown()) then
			if (FOM_NextFoodLink) then
				GFWUtils.Note(FOM_NoFoodError.."\n"..string.format(FOM_FALLBACK_MESSAGE, FOM_NextFoodLink));
			else
				GFWUtils.Note(FOM_NoFoodError);
			end
		end
	end
end

function FOM_FeedButton_OnEnter()
	if ( PetFrameHappiness.tooltip ) then
		GameTooltip:SetOwner(PetFrameHappiness, "ANCHOR_RIGHT");
		GameTooltip:SetText(PetFrameHappiness.tooltip);
		if ( PetFrameHappiness.tooltipDamage ) then
			GameTooltip:AddLine(PetFrameHappiness.tooltipDamage, "", 1, 1, 1);
		end
		if ( PetFrameHappiness.tooltipLoyalty ) then
			GameTooltip:AddLine(PetFrameHappiness.tooltipLoyalty, "", 1, 1, 1);
		end
		if (FOM_NoFoodError) then
			GameTooltip:AddLine(FOM_NoFoodError, RED_FONT_COLOR.r, RED_FONT_COLOR.g, RED_FONT_COLOR.b, 1);
			if (FOM_NextFoodLink) then
				GameTooltip:AddLine(string.format(FOM_FEEDBUTTON_TOOLTIP1_FALLBACK, FOM_NextFoodLink), GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b);
			end
		else
			GameTooltip:AddLine(string.format(FOM_FEEDBUTTON_TOOLTIP1, FOM_NextFoodLink), GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b);
		end
		GameTooltip:AddLine(FOM_FEEDBUTTON_TOOLTIP2, GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b);
		
		if (FOM_Debug) then
			GameTooltip:AddLine(" ");
			GameTooltip:AddLine("Next Foods:");
			for _, foodInfo in pairs(SortedFoodList) do
				local line = string.format("%dx%s (bag %d, slot %d)", foodInfo.count, foodInfo.link, foodInfo.bag, foodInfo.slot);
				if (foodInfo.useful) then
					line = line .. " (useful)";
				end
				if (foodInfo.temp) then
					line = line .. " (temp)";
				end
				local color;
				if (foodInfo.delta > 30) then
					color = QuestDifficultyColor["trivial"];
				elseif (foodInfo.delta > 20 and levelDelta <= 30) then
					color = QuestDifficultyColor["standard"];
				elseif (foodInfo.delta > 10 and levelDelta <= 20) then
					color = QuestDifficultyColor["difficult"];
				elseif (foodInfo.delta <= 10) then
					color = QuestDifficultyColor["verydifficult"];
				end
				GameTooltip:AddLine(line, color.r, color.g, color.b);
			end
		end
		
		GameTooltip:Show();
	end
end

function FOM_FeedButton_OnLeave()
	GameTooltip:Hide();
end

function FOM_OnLoad()

	-- Register for Events
	this:RegisterEvent("VARIABLES_LOADED");
	this:RegisterEvent("SPELLS_CHANGED");

	-- Register Slash Commands
	SLASH_FEEDOMATIC1 = "/feedomatic";
	SLASH_FEEDOMATIC2 = "/fom";
	SLASH_FEEDOMATIC3 = "/feed";
	SLASH_FEEDOMATIC4 = "/petfeed"; -- Rauen's PetFeed compatibility
	SLASH_FEEDOMATIC5 = "/pf";
	SlashCmdList["FEEDOMATIC"] = function(msg)
		FOM_ChatCommandHandler(msg);
	end
		
	BINDING_HEADER_GFW_FEEDOMATIC = GetAddOnMetadata("GFW_FeedOMatic", "Title"); -- gets us the localized title if needed
	--GFWUtils.Debug = true;

end

function FOM_HookTooltip(frame)
	if (frame:GetScript("OnTooltipSetItem")) then
		frame:HookScript("OnTooltipSetItem", FOM_OnTooltipSetItem);
	else
		frame:SetScript("OnTooltipSetItem", FOM_OnTooltipSetItem);
	end
end

function FOM_OnTooltipSetItem()
	local name, link = this:GetItem();

	if (FOM_Config.Tooltip and link and UnitExists("pet")) then
				
		local itemID = FOM_IDFromLink(link);
		if (not FOM_IsInDiet(itemID)) then
			return false;
		end
		
		local color;
		local _, _, _, itemLevel = GetItemInfo(itemID);
		if (itemLevel) then
			local levelDelta = UnitLevel("pet") - itemLevel;
			local petName = UnitName("pet");
			if (levelDelta >= 30) then
				color = QuestDifficultyColor["trivial"];
				this:AddLine(string.format(FOM_QUALITY_UNDER, petName), color.r, color.g, color.b);
				return true;
			elseif (levelDelta >= 20 and levelDelta < 30) then
				color = QuestDifficultyColor["standard"];
				this:AddLine(string.format(FOM_QUALITY_WILL, petName), color.r, color.g, color.b);
				return true;
			elseif (levelDelta >= 10 and levelDelta < 20) then
				color = QuestDifficultyColor["difficult"];
				this:AddLine(string.format(FOM_QUALITY_LIKE, petName), color.r, color.g, color.b);
				return true;
			elseif (levelDelta < 10) then
				color = QuestDifficultyColor["verydifficult"];
				this:AddLine(string.format(FOM_QUALITY_LOVE, petName), color.r, color.g, color.b);
				return true;
			end
		end
	end
end

function FOM_GetFeedPetSpellName()
	for tabIndex = 1, GetNumSpellTabs() do
		local tabName, tabIcon, offset, numSpells = GetSpellTabInfo(tabIndex);
		if (tabIcon == "Interface\\Icons\\Ability_Hunter_BeastTaming") then
			for spellID = offset + 1, offset + numSpells do
				local spellIcon = GetSpellTexture(spellID, BOOKTYPE_SPELL);
				if (spellIcon == "Interface\\Icons\\Ability_Hunter_BeastTraining") then
					FOM_FeedPetSpellName = GetSpellName(spellID, BOOKTYPE_SPELL);
					return FOM_FeedPetSpellName;
				end
			end
		end
	end
end

function FOM_Initialize()
	
	local _, realClass = UnitClass("player");
	if (realClass ~= "HUNTER") then
	 	this:UnregisterAllEvents();
		return;
	end
	
	if (UnitLevel("player") < 10) then return; end
		
	-- track whether foods are useful for Cooking 
	this:RegisterEvent("TRADE_SKILL_SHOW");
	this:RegisterEvent("TRADE_SKILL_UPDATE");

	-- Catch when feeding happened so we can notify/emote
	this:RegisterEvent("CHAT_MSG_PET_INFO");
	this:RegisterEvent("CHAT_MSG_SPELL_TRADESKILLS");
	
	-- Only subscribe to inventory updates once we're in the world
	this:RegisterEvent("PLAYER_ENTERING_WORLD");
	this:RegisterEvent("PLAYER_LEAVING_WORLD");

	-- Events for trying to catch when the pet needs feeding
	this:RegisterEvent("PET_BAR_SHOWGRID");
	this:RegisterEvent("UNIT_NAME_UPDATE");
	this:RegisterEvent("PET_BAR_UPDATE");
	this:RegisterEvent("PET_UI_UPDATE");
	this:RegisterEvent("UNIT_HAPPINESS");
	this:RegisterEvent("PLAYER_REGEN_ENABLED");
	
	FOM_FeedButton = CreateFrame("Button", "FOM_FeedButton", PetFrameHappiness, "SecureActionButtonTemplate");
	FOM_FeedButton:SetAllPoints(PetFrameHappiness);
	FOM_FeedButton:RegisterForClicks("LeftButtonUp", "RightButtonUp");
	FOM_FeedButton:SetScript("PostClick", FOM_FeedButton_PostClick);
	FOM_FeedButton:SetScript("OnEnter", FOM_FeedButton_OnEnter);
	FOM_FeedButton:SetScript("OnLeave", FOM_FeedButton_OnLeave);

	-- set key binding to click FOM_FeedButton
	FOM_UpdateBindings();
	this:RegisterEvent("UPDATE_BINDINGS");
	
	FOM_HookTooltip(GameTooltip);
	FOM_HookTooltip(ItemRefTooltip);

	this:UnregisterEvent("VARIABLES_LOADED");
	this:UnregisterEvent("SPELLS_CHANGED");

	FOM_Initialized = true;
		
end

function FOM_OnEvent(event, arg1)
	--DevTools_Dump(event)

	if ( event == "VARIABLES_LOADED" or event == "SPELLS_CHANGED") then
				
		if (not FOM_Initialized) then FOM_Initialize(); end
		return;

	elseif ( event == "UPDATE_BINDINGS" ) then

		FOM_UpdateBindings();
		
	elseif ( event == "PLAYER_ENTERING_WORLD" ) then

		this:RegisterEvent("BAG_UPDATE");
		if (InCombatLockdown()) then
			FOM_PickFoodQueued = true;
		else
			FOM_PickFoodForButton();
		end
		return;

	elseif ( event == "PLAYER_LEAVING_WORLD" ) then

		this:UnregisterEvent("BAG_UPDATE");
		
	elseif (event == "BAG_UPDATE" ) then
		
		if (FOM_BagIsQuiver(arg1)) then return; end
		
		if (InCombatLockdown()) then
			FOM_PickFoodQueued = true;
		else
			FOM_PickFoodForButton();
		end
		return;
	
	elseif ((event == "UNIT_NAME_UPDATE" and arg1 == "pet") or event == "PET_BAR_UPDATE") then
	
		if (InCombatLockdown()) then
			FOM_PickFoodQueued = true;
		else
			FOM_PickFoodForButton();
		end
		return;
	
	elseif (event == "TRADE_SKILL_SHOW" or event == "TRADE_SKILL_UPDATE") then
		FOM_ScanTradeSkill();
		return;

	elseif (event == "PLAYER_REGEN_ENABLED" and FOM_PickFoodQueued) then
		FOM_PickFoodForButton();
		return;
	elseif (event == "CHAT_MSG_PET_INFO" or event == "CHAT_MSG_SPELL_TRADESKILLS") then
		if (not FOM_FEEDPET_LOG_FIRSTPERSON) then
			FOM_FEEDPET_LOG_FIRSTPERSON = GFWUtils.FormatToPattern(FEEDPET_LOG_FIRSTPERSON);
		end
		local _, _, foodEaten = string.find(arg1, FOM_FEEDPET_LOG_FIRSTPERSON);
		if (foodEaten) then
			local foodName = foodEaten;
			if (FOM_NextFoodLink and FOM_NameFromLink(FOM_NextFoodLink) == foodEaten) then
				foodName = FOM_NextFoodLink;
			end
			local pet = UnitName("pet");
			if (pet) then
				if ( FOM_Config.NotifyType == 2) then
					GFWUtils.Print(string.format(FOM_FEEDING_EAT, pet, foodName));
				elseif ( FOM_Config.NotifyType == 1) then
					SendChatMessage(string.format(FOM_FEEDING_FEED, pet, foodName).. FOM_RandomEmote(foodName), "EMOTE");
				end
			end
		end
	end
 	
	if (FOM_Config.WarningLevel ~= 3) then
		FOM_CheckHappiness();
	end
	
	if (FOM_PickFoodQueued and not InCombatLockdown()) then
		FOM_PickFoodForButton();
	end
	
end

function FOM_ScanTradeSkill()
	if (GetTradeSkillLine() and GetTradeSkillLine() == FOM_CookingSpellName()) then
		if (FOM_Config.SaveForCooking) then
			-- Update Cooking reagents list so we can avoid consuming food we could skillup from.
			if (FOM_Cooking == nil) then
				FOM_Cooking = { };
			end
			if (FOM_Cooking and TradeSkillFrame and TradeSkillFrame:IsVisible() and not FOM_TradeSkillLock.Locked) then
				-- This prevents further update events from being handled if we're already processing one.
				-- This is done to prevent the game from freezing under certain conditions.
				FOM_TradeSkillLock.Locked = true;

				GFWUtils.DebugLog("scanning Cooking list");
				for i=1, GetNumTradeSkills() do
					local itemName, type, _, _ = GetTradeSkillInfo(i);
					if (type ~= "header") then
						for j=1, GetTradeSkillNumReagents(i) do
							local reagentLink = GetTradeSkillReagentItemLink(i, j);
							local itemID = FOM_IDFromLink(reagentLink);
							
							if (itemID and FOM_IsKnownFood(itemID)) then
								if (FOM_Cooking[itemID] == nil) then
									FOM_Cooking[itemID] = FOM_DifficultyToNum(type);
								else           
									FOM_Cooking[itemID] = max(FOM_Cooking[itemID], FOM_DifficultyToNum(type));
								end
							end
						end
					end
				end
			end
		end
	end
end

function FOM_UpdateBindings()
	ClearOverrideBindings(FOM_FeedButton);
	local key = GetBindingKey("FOM_FEED");
	if (key) then
		SetOverrideBindingClick(FOM_FeedButton, nil, key, "FOM_FeedButton");
	end
end

-- Update our list of quest objectives so we can avoid consuming food we want to accumulate for a quest.
function FOM_ScanQuests()
	FOM_QuestFood = nil;
	for questNum=1, GetNumQuestLogEntries() do
		local QText, level, questTag, isHeader, isCollapsed, isComplete  = GetQuestLogTitle(questNum);
		if (not isHeader) then
			for objectiveNum=1, GetNumQuestLeaderBoards(questNum) do
				local text, type, finished = GetQuestLogLeaderBoard(objectiveNum, questNum);
				if (text and strlen(text) > 0) then
					local _, _, objectiveName, numCurrent, numRequired = string.find(text, "(.*): (%d+)/(%d+)");
					if (FOM_IsKnownFood(objectiveName)) then
						if (FOM_QuestFood == nil) then
							FOM_QuestFood = { };
						end
						if (FOM_QuestFood[objectiveName] == nil) then
							FOM_QuestFood[objectiveName] = tonumber(numRequired);
						else             
							FOM_QuestFood[objectiveName] = max(FOM_QuestFood[objectiveName], tonumber(numRequired));
						end
					end
				end
			end
		end
	end
end

function FOM_DifficultyToNum(level)
	if (level == "optimal" or level == "orange") then
		return 4;
	elseif (level == "medium" or level == "yellow") then
		return 3;
	elseif (level == "easy" or level == "green") then
		return 2;
	elseif (level == "trivial" or level == "gray" or level == "grey") then
		return 1;
	else -- bad input
		return nil;
	end
end

function FOM_OnUpdate(elapsed)
	
	_, realClass = UnitClass("player");
	if (realClass ~= "HUNTER") then return; end
	
	-- workaround for sound bug in WoW 2.2
	for file in pairs(soundQueue) do
          PlaySoundFile(file);
          soundQueue[file] = nil;
    end

	-- If it's been more than a second since our last tradeskill update,
	-- we can allow the event to process again.
	FOM_TradeSkillLock.EventTimer = FOM_TradeSkillLock.EventTimer + elapsed;
	if (FOM_TradeSkillLock.Locked) then
		FOM_TradeSkillLock.EventCooldown = FOM_TradeSkillLock.EventCooldown + elapsed;
		if (FOM_TradeSkillLock.EventCooldown > FOM_TradeSkillLock.EventCooldownTime) then

			FOM_TradeSkillLock.EventCooldown = 0;
			FOM_TradeSkillLock.Locked = false;
		end
	end
		
	--GFWUtils.Debug = true;

	if (FOM_ShouldFeed and FOM_Config.IconWarning and PetFrameHappiness and not InCombatLockdown()) then
		if (PetFrameHappiness:IsVisible() and PetFrameHappiness:GetAlpha() == 1 and not FOM_HasFeedEffect()) then
			FOM_FadeOut();
		end
	end
end

function FOM_FadeOut()
    local fadeInfo = {};
    fadeInfo.mode = "OUT";
    fadeInfo.timeToFade = 0.5;
    fadeInfo.finishedFunc = FOM_FadeIn;
    UIFrameFade(PetFrameHappiness, fadeInfo);
end

-- hack since a frame can't have a reference to itself in it
function FOM_FadeIn()
    UIFrameFadeIn(PetFrameHappiness, 0.5);
end

-- record observed creature family and diet names since they're localized;
-- we can have users send in their SavedVariables files to learn the new names.
function FOM_CheckLocaleInfo()
	if (FOM_ObservedDiets == nil) then
		FOM_ObservedDiets = {};
	end
	FOM_ObservedDiets[UnitCreatureFamily("pet")] = {GetPetFoodTypes()};
	if (FOM_ObservedFamilies == nil) then
		FOM_ObservedFamilies = {};
	end
	table.insert(FOM_ObservedFamilies, UnitCreatureFamily("pet"));
end

function FOM_ChatCommandHandler(msg)

	if ( msg == "" ) then
		FOM_ShowOptions();
		return;
	end
	
	-- Check for Pet (we don't really need one for most of our chat commands, but we conveniently use its name.)
	if ( UnitExists("pet") ) then
		petName = UnitName("pet");
		FOM_CheckLocaleInfo();
	else
		petName = "Your pet";
	end
	
	-- Print Help
	if ( msg == "help" ) then
		local version = GetAddOnMetadata("GFW_FeedOMatic", "Version");
		GFWUtils.Print("Fizzwidget Feed-O-Matic "..version..":");
		GFWUtils.Print("/feedomatic /fom <command>");
		GFWUtils.Print("- "..GFWUtils.Hilite("help").." - Print this helplist.");
		GFWUtils.Print("- "..GFWUtils.Hilite("reset").." - Reset to default settings.");
		GFWUtils.Print("- "..GFWUtils.Hilite("add <diet> <name>").." - Add food to list.");
		GFWUtils.Print("- "..GFWUtils.Hilite("remove <diet> <name>").." - Remove food from list.");
		GFWUtils.Print("- "..GFWUtils.Hilite("show <diet>").." - Show food list.");
		return;
	end

	if ( msg == "version" ) then
		local version = GetAddOnMetadata("GFW_FeedOMatic", "Version");
		GFWUtils.Print("Fizzwidget Feed-O-Matic "..version..":");
		return;
	end
		

	if ( msg == "debug" ) then
		FOM_Debug = not FOM_Debug;
		GFWUtils.Print((not FOM_Debug and "Not " or "").."Showing food list in happiness icon tooltip.");
	end
	
	-- Reset Variables
	if ( msg == "reset" ) then
		GFW_FeedOMatic:ResetProfile();
		FOM_Cooking = nil;
		FOM_AddedFoods = nil;
		FOM_RemovedFoods = nil;
		FOM_QuestFood = nil;
		GFWUtils.Print("Feed-O-Matic configuration reset.");
		FOM_ChatCommandHandler("status");
		return;
	end

	local _, _, cmd, diet, foodString = string.find(msg, "(%w+) (%w*) *(.*)");
	if ( cmd == "add" or cmd == "remove" or cmd == "show" or cmd == "list" ) then
	
		diet = string.lower(diet); -- let's be case insensitive
		if ( diet ~= "" and FOM_Foods[diet] == nil and diet ~= FOM_DIET_ALL) then
			local usageString = "Usage: "..GFWUtils.Hilite("/feedomatic "..cmd..FOM_DIET_MEAT).." | "..GFWUtils.Hilite(FOM_DIET_FISH).." | "..GFWUtils.Hilite(FOM_DIET_BREAD).." | "..GFWUtils.Hilite(FOM_DIET_CHEESE).." | "..GFWUtils.Hilite(FOM_DIET_FRUIT).." | "..GFWUtils.Hilite(FOM_DIET_FUNGUS)
			if (cmd ~= "show" and cmd ~= "list") then
				usageString = usageString.." <item link>.";
			end
			GFWUtils.Print(usageString);
			return;
		end

		if (cmd == "show" or cmd == "list") then
			local diets;
			if ( diet == FOM_DIET_ALL ) then
				diets = {FOM_DIET_MEAT, FOM_DIET_FISH, FOM_DIET_BREAD, FOM_DIET_CHEESE, FOM_DIET_FRUIT, FOM_DIET_FUNGUS};
			else
				diets = {diet};
			end
			
			for _, aDiet in pairs(diets) do
				local capDiet = string.upper(string.sub(aDiet, 1, 1)) .. string.sub(aDiet, 2); -- print a nicely capitalized version
				GFWUtils.Print("Feed-O-Matic "..GFWUtils.Hilite(capDiet).." List:");
				local dietFoods = FOM_Foods[aDiet];
				if (FOM_AddedFoods and FOM_AddedFoods[aDiet]) then
					dietFoods = GFWTable.Merge(dietFoods, FOM_AddedFoods[aDiet]);
				end
				if (FOM_RemovedFoods and FOM_RemovedFoods[aDiet]) then
					dietFoods = GFWTable.Subtract(dietFoods, FOM_RemovedFoods[aDiet]);
				end
				table.sort(dietFoods);
				for _, food in pairs(dietFoods) do
					local foodName = GetItemInfo(food);
					if (foodName) then
						if (FOM_FoodIDsToNames == nil) then
							FOM_FoodIDsToNames = {};
						end
						FOM_FoodIDsToNames[food] = foodName;
						GFWUtils.Print(GFWUtils.Hilite(" - ")..foodName);
					else
						GFWUtils.Print(GFWUtils.Hilite(" - ").."item id "..food.." (name not available)");
					end
				end
			end
			return;
		else
		
			local inputFoods = { };
			for itemLink in string.gmatch(foodString, "|c%x+|Hitem:[-%d:]+|h%[.-%]|h|r") do
				table.insert(inputFoods, itemLink);
				local foodID = FOM_IDFromLink(itemLink);
				if (foodID) then
					local foodName = FOM_NameFromLink(itemLink);
					if (FOM_FoodIDsToNames == nil) then
						FOM_FoodIDsToNames = {};
					end
					FOM_FoodIDsToNames[foodID] = foodName;
				end
			end
			if (table.getn(inputFoods) == 0) then
				GFWUtils.Print("The "..GFWUtils.Hilite("/fom "..cmd).." command requires an item link; shift-click an item to insert a link.");
				return;
			end

			local capDiet = string.upper(string.sub(diet, 1, 1)) .. string.sub(diet, 2); -- print a nicely capitalized version
			if ( cmd == "add" ) then
				for _, food in pairs(inputFoods) do
					local foodID = FOM_IDFromLink(food);
					if ( FOM_AddFood(diet, tonumber(foodID)) ) then
						GFWUtils.Print("Added "..food.." to "..GFWUtils.Hilite(capDiet).." list.");
					else
						GFWUtils.Print(food.." already in "..GFWUtils.Hilite(capDiet).." list.");
					end
				end
				if (FOM_Config.AvoidQuestFood) then
					FOM_ScanQuests(); -- in case any of the newly added foods are quest objectives
				end
				return;
			elseif (cmd == "remove" ) then
				for _, food in pairs(inputFoods) do
					local foodID = FOM_IDFromLink(food);

					local diets;
					if ( diet == "" ) then
						diets = {FOM_DIET_MEAT, FOM_DIET_FISH, FOM_DIET_BREAD, FOM_DIET_CHEESE, FOM_DIET_FRUIT, FOM_DIET_FUNGUS};
					else
						diets = {diet};
					end
					
					for _, aDiet in pairs(diets) do
						local capDiet = string.upper(string.sub(aDiet, 1, 1)) .. string.sub(aDiet, 2); -- print a nicely capitalized version
						if ( FOM_RemoveFood(aDiet, tonumber(foodID)) ) then
							GFWUtils.Print("Removed "..food.." from "..GFWUtils.Hilite(capDiet).." list.");
						elseif (#diets == 1) then
							GFWUtils.Print("Could not find "..food.." in "..GFWUtils.Hilite(capDiet).." list.");
						end
					end
				end
				return;
			end
		end
	end
	
	-- if we got down to here, we got bad input
	FOM_ChatCommandHandler("help");
end

-- Add a food to a list
function FOM_AddFood(diet, food)

	if (FOM_Foods[diet] == nil) then
		GFWUtils.DebugLog("FOM_Foods[diet] == nil");
	end
	if (FOM_AddedFoods == nil or FOM_AddedFoods[diet] == nil) then
		GFWUtils.DebugLog("FOM_AddedFoods == nil or FOM_AddedFoods[diet] == nil");
	end
	if (FOM_RemovedFoods == nil or FOM_RemovedFoods[diet] == nil) then
		GFWUtils.DebugLog("FOM_RemovedFoods == nil or FOM_RemovedFoods[diet] == nil");
	end
	if ( GFWTable.IndexOf(FOM_Foods[diet], food) == 0 ) then
		if (FOM_AddedFoods == nil) then
			FOM_AddedFoods = {};
		end
		if (FOM_AddedFoods[diet] == nil) then
			FOM_AddedFoods[diet] = {};
		end
		if ( GFWTable.IndexOf(FOM_AddedFoods[diet], food) == 0 ) then
			table.insert( FOM_AddedFoods[diet], food );
			table.sort( FOM_AddedFoods[diet] );
			if (FOM_RemovedFoods and FOM_RemovedFoods[diet] and GFWTable.IndexOf(FOM_RemovedFoods[diet], food) ~= 0) then
				table.remove( FOM_RemovedFoods[diet], GFWTable.IndexOf(FOM_RemovedFoods[diet], food) );
				table.sort( FOM_RemovedFoods[diet] );
			end
			return true;
		else
			return false;
		end
	else
		return false;
	end

end

-- Remove a food from a list
function FOM_RemoveFood(diet, food)
	
	if (FOM_Foods[diet] == nil) then
		GFWUtils.DebugLog("FOM_Foods[diet] == nil");
	end
	if (FOM_AddedFoods == nil or FOM_AddedFoods[diet] == nil) then
		GFWUtils.DebugLog("FOM_AddedFoods == nil or FOM_AddedFoods[diet] == nil");
	end
	if (FOM_RemovedFoods == nil or FOM_RemovedFoods[diet] == nil) then
		GFWUtils.DebugLog("FOM_RemovedFoods == nil or FOM_RemovedFoods[diet] == nil");
	end
	if ( GFWTable.IndexOf(FOM_Foods[diet], food) ~= 0 ) then
		if (FOM_RemovedFoods == nil) then
			FOM_RemovedFoods = {};
		end
		if (FOM_RemovedFoods[diet] == nil) then
			FOM_RemovedFoods[diet] = {};
		end
		if ( GFWTable.IndexOf(FOM_RemovedFoods[diet], food) == 0 ) then
			table.insert( FOM_RemovedFoods[diet], food );
			table.sort( FOM_RemovedFoods[diet] );
			if (FOM_AddedFoods and FOM_AddedFoods[diet] and GFWTable.IndexOf(FOM_AddedFoods[diet], food) ~= 0) then
				table.remove( FOM_AddedFoods[diet], GFWTable.IndexOf(FOM_AddedFoods[diet], food) );
				table.sort( FOM_AddedFoods[diet] );
			end
			return true;
		else
			return false;
		end
	else
		if (FOM_AddedFoods and FOM_AddedFoods[diet] and GFWTable.IndexOf(FOM_AddedFoods[diet], food) ~= 0) then
			table.remove( FOM_AddedFoods[diet], GFWTable.IndexOf(FOM_AddedFoods[diet], food) );
			table.sort( FOM_AddedFoods[diet] );
			return true;
		end
		return false;
	end

end

-- Check Happiness
function FOM_CheckHappiness()

	-- Check for pet
	if not ( UnitExists("pet") ) then 
		FOM_ShouldFeed = nil;		
		return;
	end
		
	-- Get Pet Info
	local pet = UnitName("pet");
	local happiness, damage, loyalty = GetPetHappiness();
	
	-- Check No Happiness
	if ( happiness == 0 ) or ( happiness == nil ) then return; end
	
	local level = 0;
	if (FOM_Config.WarningLevel == 1 or FOM_Config.WarningLevel == 2) then
		level = FOM_Config.WarningLevel;
	end
	
	-- Check if Need Feeding
	if ( happiness < level + 1 ) then
	
		if (UnitIsDead("pet")) then return; end
		if (UnitAffectingCombat("pet")) then return; end
		if (UnitAffectingCombat("player")) then return; end
		
		FOM_ShouldFeed = true;
		if (not FOM_HasFeedEffect() and GetTime() - FOM_LastWarning > FOM_WARNING_INTERVAL) then
			if (FOM_Config.TextWarning) then
				local msg;
				if (level - happiness == 0) then
					msg = FOM_PET_HUNGRY;
				else
					msg = FOM_PET_VERY_HUNGRY;
				end
				if (FOM_NoFoodError) then
					msg = msg .. "\n" .. FOM_NoFoodError;
				end
				GFWUtils.Print(string.format(msg, pet));
				GFWUtils.Note(string.format(msg, pet));
			end
			FOM_PlayHungrySound();
			FOM_LastWarning = GetTime();
		end
	else
		FOM_ShouldFeed = nil;
	end
	
end

FOM_HungrySounds = {
  	[FOM_BAT]		    = "Sound\\Creature\\FelBat\\FelBatDeath.wav",
  	[FOM_BEAR]		    = "Sound\\Creature\\Bear\\mBearDeathA.wav",
  	[FOM_BOAR]		    = "Sound\\Creature\\Boar\\mWildBoarAggro2.wav",
  	[FOM_CAT]		    = "Sound\\Creature\\Tiger\\mTigerStand2A.wav",
  	[FOM_CARRION_BIRD]	= "Sound\\Creature\\Carrion\\mCarrionWoundCriticalA.wav",
  	[FOM_CRAB]		    = "Sound\\Creature\\Crab\\CrabDeathA.wav",
  	[FOM_CROCOLISK]	    = "Sound\\Creature\\Basilisk\\mBasiliskSpellCastA.wav",
  	[FOM_GORILLA]	    = "Sound\\Creature\\Gorilla\\GorillaDeathA.wav",
  	[FOM_HYENA]		    = "Sound\\Creature\\Hyena\\HyenaPreAggroA.wav",
  	[FOM_OWL]		    = "Sound\\Creature\\OWl\\OwlPreAggro.wav",
  	[FOM_RAPTOR]	    = "Sound\\Creature\\Raptor\\mRaptorWoundCriticalA.wav",
  	[FOM_SCORPID]	    = "Sound\\Creature\\SilithidWasp\\mSilithidWaspStand2A.wav",
  	[FOM_SPIDER]	    = "Sound\\Creature\\Tarantula\\mTarantulaFidget2a.wav",
  	[FOM_TALLSTRIDER]   = "Sound\\Creature\\TallStrider\\tallStriderPreAggroA.wav",
  	[FOM_TURTLE]	    = "Sound\\Creature\\SeaTurtle\\SeaTurtleWoundCritA.wav",
  	[FOM_WIND_SERPENT]	= "Sound\\Creature\\WindSerpant\\mWindSerpantDeathA.wav",
  	[FOM_WOLF]		    = "Sound\\Creature\\Wolf\\mWolfFidget2c.wav",
	[FOM_DRAGONHAWK]	= "Sound\\Creature\\DragonHawk\\DragonHawkWoundCrit.wav",
	[FOM_NETHER_RAY]	= "Sound\\Creature\\SporeBat\\SporebatWoundCrit.wav",
	[FOM_RAVAGER]		= "Sound\\Creature\\Crawler\\CrawlerWoundCrit.wav",
	[FOM_SERPENT]		= "Sound\\Creature\\Serpent\\SerpentPreAggro.wav",
	[FOM_SPOREBAT]		= "Sound\\Creature\\SporeBat\\SporebatWoundCrit.wav",
	[FOM_WARP_STALKER]	= "Sound\\Creature\\WarpStalker\\WarpStalkerWoundCrit.wav",
};
function FOM_PlayHungrySound()
	if (FOM_Config.AudioWarning) then
		local type = UnitCreatureFamily("pet");
		local sound = FOM_HungrySounds[type];
		if (sound == nil or FOM_Config.AudioWarningBell) then
			FOM_PlaySoundFile("Sound\\Doodad\\BellTollNightElf.wav");
		else
			FOM_PlaySoundFile(sound);
		end
	end
end

-- workaround for sound bug in WoW 2.2
function FOM_PlaySoundFile(file)
	soundQueue[file] = true;
end

-- Check Feed Effect
function FOM_HasFeedEffect()

	local i = 1;
	local _, _, buff = UnitBuff("pet", i);
	while buff do
		if ( string.find(buff, "Ability_Hunter_BeastTraining") ) then
			return true;
		end
		i = i + 1;
		_, _, buff = UnitBuff("pet", i);
	end
	return false;

end

function FOM_PickFoodForButton()

	local pet = UnitName("pet");
	if (not pet) then 
		FOM_PickFoodQueued = true;
		return;
	end
	local dietList = {GetPetFoodTypes()};
	if ( dietList == nil or #dietList == 0) then
		FOM_PickFoodQueued = true;
		return ;
	end
	
	local foodBag, foodSlot;
	foodBag, foodSlot, FOM_NextFoodLink = FOM_NewFindFood();
	FOM_SetupButton(foodBag, foodSlot);
	
	if ( foodBag == nil) then
		local fallbackBag, fallbackSlot;
		fallbackBag, fallbackSlot, FOM_NextFoodLink = FOM_NewFindFood(1);
		if (fallbackBag) then
			FOM_NoFoodError = string.format(FOM_ERROR_NO_FOOD_NO_FALLBACK, pet);
			FOM_SetupButton(fallbackBag, fallbackSlot, "alt");
		else
			-- No Food Could be Found
			FOM_NoFoodError = string.format(FOM_ERROR_NO_FOOD, pet);
			FOM_NextFoodLink = nil;
			--GFWUtils.Print("Can't feed? #SortedFoodList:"..#SortedFoodList);
			--DevTools_Dump(GetPetFoodTypes());
		end
		PetFrameHappinessTexture:SetVertexColor(0.5, 0.5, 1);
	else
		FOM_NoFoodError = nil;
		PetFrameHappinessTexture:SetVertexColor(1, 1, 1);
	end
	
	-- debug
	if (false and FOM_NextFoodLink) then
		if (FOM_NoFoodError) then
			GFWUtils.PrintOnce("Next food (fallback):"..FOM_NextFoodLink, 1);
		else
			GFWUtils.PrintOnce("Next food:"..FOM_NextFoodLink, 1);
		end
	end
end

function FOM_SetupButton(bag, slot, modifier)
	if (not FOM_GetFeedPetSpellName()) then
		local version = GetAddOnMetadata("GFW_FeedOMatic", "Version");
		GFWUtils.PrintOnce(GFWUtils.Red("Feed-O-Matic v."..version.." error:").."Can't find Feed Pet spell. (Have you finished your level 10 Hunter quests?)");
		return;
	end
	if (modifier) then
		modifier = modifier.."-";
	else
		modifier = "";
	end
	if (bag and slot) then
		FOM_FeedButton:SetAttribute(modifier.."type1", "spell");
		FOM_FeedButton:SetAttribute(modifier.."spell1", FOM_FeedPetSpellName);
		FOM_FeedButton:SetAttribute("target-bag", bag);
		FOM_FeedButton:SetAttribute("target-slot", slot);
	else
		FOM_FeedButton:SetAttribute(modifier.."type", ATTRIBUTE_NOOP);
		FOM_FeedButton:SetAttribute(modifier.."spell", ATTRIBUTE_NOOP);
		FOM_FeedButton:SetAttribute(modifier.."type1", ATTRIBUTE_NOOP);
		FOM_FeedButton:SetAttribute(modifier.."spell1", ATTRIBUTE_NOOP);
		FOM_FeedButton:SetAttribute("target-bag", nil);
		FOM_FeedButton:SetAttribute("target-slot", nil);
	end
	FOM_PickFoodQueued = nil;
end

function FOM_RandomEmote(foodLink)
	
	local randomEmotes = {};
	if (UnitSex("pet") == 2) then
		randomEmotes = GFWTable.Merge(randomEmotes, FOM_Emotes["male"]);
	elseif (UnitSex("pet") == 3) then
		randomEmotes = GFWTable.Merge(randomEmotes, FOM_Emotes["female"]);
	end
	
	randomEmotes = GFWTable.Merge(randomEmotes, FOM_Emotes[UnitCreatureFamily("pet")]);
	randomEmotes = GFWTable.Merge(randomEmotes, FOM_Emotes[FOM_NameFromLink(foodLink)]);
	randomEmotes = GFWTable.Merge(randomEmotes, FOM_Emotes["any"]);
	
	return randomEmotes[math.random(table.getn(randomEmotes))];

end

function FOM_IsTemporaryFood(itemLink)
	
	local _, _, link = string.find(itemLink, "(item:%d+)");
	if (link == nil or link == "") then 
		return false; 
	end
	FOMTooltip:ClearLines();
	FOMTooltip:SetHyperlink(link);
	if (FOMTooltipTextLeft2:GetText() == ITEM_CONJURED) then
		return true;
	else
		return false;
	end	

end

function FOM_FlatFoodList()
	local foodList = {};
	FOM_Quantity = { };
	local petLevel = UnitLevel("pet");
	for bagNum = 0, 4 do
		if (not FOM_BagIsQuiver(bagNum) ) then
		-- skip bags that can't contain food
			for itemNum = 1, GetContainerNumSlots(bagNum) do
				local itemLink = GetContainerItemLink(bagNum, itemNum);
				-- debug
				--if (bagNum == 0 and itemNum == 1) then _, itemLink = GetItemInfo(21023); end
				if (itemLink) then
					local itemID = FOM_IDFromLink(itemLink);
					local _, itemCount = GetContainerItemInfo(bagNum, itemNum);
					-- debug
					--if (bagNum == 0 and itemNum == 1) then itemCount = 10; end
					if ( FOM_IsInDiet(itemID) ) then
						if (FOM_FoodIDsToNames == nil) then
							FOM_FoodIDsToNames = {};
						end
						local name = FOM_NameFromLink(itemLink);
						FOM_FoodIDsToNames[itemID] = name;
						local isUseful = FOM_IsUsefulFood(itemID, itemCount);
						local _, _, _, level = GetItemInfo(itemID);
						if (petLevel - level < 30) then
							table.insert(foodList, {bag=bagNum, slot=itemNum, link=itemLink, count=itemCount, delta=(petLevel - level), useful=isUseful, temp=FOM_IsTemporaryFood(itemLink)});
						end
					end
				end
			end
		end
	end
	return foodList;
end

function FOM_NewFindFood(fallback)
	SortedFoodList = FOM_FlatFoodList(fallback);

	-- if there are any conjured foods, drop everything else from the list
	local tempFoodsOnly = {};
	for _, foodInfo in pairs(SortedFoodList) do
		if (foodInfo.temp) then
			table.insert(tempFoodsOnly, foodInfo);
		end
	end
	if (table.getn(tempFoodsOnly) > 0) then
		SortedFoodList = tempFoodsOnly;
	end
	
	
	table.sort(SortedFoodList, FOM_SortCount); -- small stacks first
	if (FOM_NumOpenBagSlots() > FOM_Config.KeepOpenSlots) then
		if (not FOM_Config.UseLowLevelFirst) then
			table.sort(SortedFoodList, FOM_SortQualityDescending); -- higher quality first
		else
			table.sort(SortedFoodList, FOM_SortQualityAscending); -- lower quality first
		end
	end
	if ((FOM_Config.AvoidQuestFood or FOM_Config.AvoidBonusFood or FOM_Config.SaveForCooking) and not fallback) then
		local nonUsefulFoodsOnly = {};
		for _, foodInfo in pairs(SortedFoodList) do
			if (not foodInfo.useful) then
				table.insert(nonUsefulFoodsOnly, foodInfo);
			end
		end
		SortedFoodList = nonUsefulFoodsOnly;
	else
		table.sort(SortedFoodList, FOM_SortUseful); -- non-useful first
	end
	
	if (GFWUtils.Debug) then
		if (fallback) then
			GFWUtils.DebugLog("Food list (with fallback):")
		else
			GFWUtils.DebugLog("Food list:")
		end
		for num, foodInfo in pairs(SortedFoodList) do
			GFWUtils.DebugLog(string.format("%d: %dx%s, delta %d", num, foodInfo.count, foodInfo.link, foodInfo.delta));
		end
	end
	for _, foodInfo in pairs(SortedFoodList) do
		return foodInfo.bag, foodInfo.slot, foodInfo.link;
	end
	
	return nil;
end

function FOM_SortCount(a, b)
	return a.count < b.count;
end

function FOM_SortQualityDescending(a, b)
	return a.delta < b.delta;
end

function FOM_SortQualityAscending(a, b)
	return a.delta > b.delta;
end

function FOM_SortUseful(a, b)
	if (a.useful) then
		aUseful = 1;
	else
		aUseful = 0;
	end
	if (b.useful) then
		bUseful = 1;
	else
		bUseful = 0;
	end
	return aUseful < bUseful;
end

function FOM_IsUsefulFood(itemID, quantity)
	if (itemID == 28112) then
		return false;	-- treat Underspore Pod as not useful because you can always summon more
	end
	local foodName = GetItemInfo(itemID);
	if (foodName == nil) then
		GFWUtils.DebugLog("Can't get info for item ID "..itemID..", assuming it's OK to eat.");
		return false;
	end
	if (FOM_Config.SaveForCooking) then
		if (FOM_Cooking and FOM_Cooking[itemID]) then
			if (FOM_Cooking[itemID] >= FOM_Config.CookingLevel) then
				GFWUtils.DebugLog("Skipping "..quantity.."x "..foodName.."; is good for cooking.");
				return true;
			end
		elseif( FOM_Config.CookingUnknown and FOM_CookingFoods[itemID]) then
			GFWUtils.DebugLog("Skipping "..quantity.."x "..foodName.."; is used in cooking recipes we don't know.");
			return true;
		end
	end
	if (FOM_Config.AvoidQuestFood) then
		FOM_ScanQuests();
		if (FOM_QuestFood and FOM_QuestFood[foodName]) then
			if (FOM_Quantity[foodName] == nil) then
				FOM_Quantity[foodName] = quantity;
			else
				FOM_Quantity[foodName] = FOM_Quantity[foodName] + quantity;
			end
			if (FOM_Quantity[foodName] > FOM_QuestFood[foodName]) then
				GFWUtils.DebugLog("Not skipping "..quantity.."x "..foodName.."; is needed for quest, but we have more than enough.");
				return false;
			else
				GFWUtils.DebugLog("Skipping "..quantity.."x "..foodName.."; is needed for quest.");
				return true;
			end
		end
	end
	if (FOM_IsBonusFood(itemID)) then
		GFWUtils.DebugLog("Skipping "..quantity.."x "..foodName.."; has bonus effect when eaten by player.");
		return true;
	end
	--GFWUtils.DebugLog("Not skipping "..quantity.."x "..foodName.."; doesn't have other uses.");
	return false;
end

local FOM_BonusTokens = {
	SPELL_STAT1_NAME,
	SPELL_STAT2_NAME,
	SPELL_STAT3_NAME,
	SPELL_STAT4_NAME,
	SPELL_STAT5_NAME,
	MANA,
	ATTACK_POWER_TOOLTIP,
	COMBAT_RATING_NAME1,
	COMBAT_RATING_NAME2,
	COMBAT_RATING_NAME3,
	COMBAT_RATING_NAME4,
	COMBAT_RATING_NAME5,
	COMBAT_RATING_NAME6,
	COMBAT_RATING_NAME7,
	COMBAT_RATING_NAME8,
	COMBAT_RATING_NAME9,
	COMBAT_RATING_NAME10,
	COMBAT_RATING_NAME11,
	COMBAT_RATING_NAME15,
	SCORE_HEALING_DONE,
	ATTACK_SPEED,
	DAMAGE,
	"%%",
};
function FOM_IsBonusFood(itemID)
	if (FOM_Config.AvoidBonusFood) then
		if (FOM_BonusFoodsCache[itemID]) then return true; end
		FOMTooltip:ClearLines();
		FOMTooltip:SetHyperlink("item:"..itemID);

		for lineNum = 1, FOMTooltip:NumLines() do
			local leftText = getglobal("FOMTooltipTextLeft"..lineNum):GetText();
			if (leftText and string.sub(leftText, 1, string.len(ITEM_SPELL_TRIGGER_ONUSE)) == ITEM_SPELL_TRIGGER_ONUSE) then
				--GFWUtils.Print("Found "..leftText)
				for _, statName in pairs(FOM_BonusTokens) do
					if (string.find(string.lower(leftText), string.lower(statName))) then
						FOM_BonusFoodsCache[itemID] = 1;
						return true;
					end
				end
			end
		end
	end
	return false;
end

function FOM_NumOpenBagSlots()
	local openSlots = 0;
	for bagNum = 0, 4 do
		if (not FOM_BagIsQuiver(bagNum) ) then
		-- skip bags that can't contain food

			local bagSize = GetContainerNumSlots(bagNum);
			for itemNum = 1, bagSize do
				if (GetContainerItemInfo(bagNum, itemNum) == nil) then
					openSlots = openSlots + 1;
				end
			end
		end
	end
	return openSlots;
end

function FOM_IsInDiet(food, dietList)

	if ( dietList == nil ) then
		dietList = {GetPetFoodTypes()};
	end
	if ( dietList == nil or #dietList == 0) then
		FOM_PickFoodQueued = true;
		return false;
	end
	if (type(dietList) ~= "table") then
		dietList = {dietList};
	end
	for _, diet in pairs(dietList) do 
		diet = string.lower(diet); -- let's be case insensitive
		if (FOM_Foods[diet] == nil) then
			GFWUtils.DebugLog("FOM_Foods[diet] == nil");
		end
		if (FOM_RemovedFoods and FOM_RemovedFoods[diet] and GFWTable.IndexOf(FOM_RemovedFoods[diet], food) ~= 0) then
			return false;
		end
		if (FOM_AddedFoods and FOM_AddedFoods[diet] and GFWTable.IndexOf(FOM_AddedFoods[diet], food) ~= 0) then
			return true;
		end
		if (GFWTable.IndexOf(FOM_Foods[diet], food) ~= 0) then
			return true;
		end
	end
	
	return false;

end

function FOM_IsKnownFood(food)
	return FOM_IsInDiet(food, {FOM_DIET_MEAT, FOM_DIET_FISH, FOM_DIET_BREAD, FOM_DIET_CHEESE, FOM_DIET_FUNGUS, FOM_DIET_FRUIT});
end

function FOM_GetItemName(bag, slot)

	local itemLink = GetContainerItemLink(bag, slot);
	if (itemLink) then
		return FOM_NameFromLink(itemLink);
	else
		return "";
	end
end

-- The icon for the cooking spell is unique and the same in all languages; use that to determine the localized name.
function FOM_CookingSpellName()
	FOM_COOKING_ICON = "Interface\\Icons\\INV_Misc_Food_15"; 
	if (FOM_COOKING_NAME == nil) then
		local spellName;
		local i = 0;
		repeat
			i = i + 1;
			spellName = GetSpellName(i, BOOKTYPE_SPELL);
			if (spellName and GetSpellTexture(i, BOOKTYPE_SPELL) == FOM_COOKING_ICON) then
				FOM_COOKING_NAME = spellName;
				return FOM_COOKING_NAME;
			end
		until (spellName == nil);
	end
	return FOM_COOKING_NAME;	
end

function FOM_BagIsQuiver(bagNum)
	if (bagNum == 0 or bagNum == -2) then return false; end
	local invSlotID = ContainerIDToInventoryID(bagNum);
	local bagLink = GetInventoryItemLink("player", invSlotID);
	if (bagLink == nil) then
		return false;	
	end
	local _, _, itemID  = string.find(bagLink, "item:(%d+)");
	if (tonumber(itemID)) then
		itemID = tonumber(itemID);
		local name, link, rarity, level, minLevel, type, subType, stackCount, equipLoc, texture = GetItemInfo(itemID);
		if (type == "Ammo Pouch" or type == "Quiver" or subType == "Ammo Pouch" or subType == "Quiver") then
			return true;
		end
		if (type == FOM_AMMO_POUCH or type == FOM_QUIVER or subType == FOM_AMMO_POUCH or subType == FOM_QUIVER) then
			return true;
		end
	end	
	return false;
end

function FOM_IDFromLink(itemLink)
	if (itemLink == nil) then return nil; end
	local _, _, itemID  = string.find(itemLink, "item:(%d+)");
	if (tonumber(itemID)) then
		return tonumber(itemID);
	else
		return nil;
	end
end

function FOM_NameFromLink(itemLink)
	if (itemLink == nil) then return nil; end
	local _, _, name = string.find(itemLink, "%[(.-)%]"); 
	if (name) then
		return name;
	end
	return itemLink;
end

-- options window

FOM_Bindings = {
	Feed = "CLICK FOM_FeedButton:LeftButton",
	Options = "CLICK FOM_FeedButton:RightButton",
};

------------------------------------------------------
-- Dongle & GFWOptions stuff
------------------------------------------------------

GFW_FeedOMatic = {};
local GFWOptions = DongleStub("GFWOptions-1.0");

local MAX_KEEP_OPEN_SLOTS = 100;

local function buildOptionsUI(panel)

	GFW_FeedOMatic.optionsText = {
		Tooltip				= FOM_OPTIONS_TOOLTIP,
		UseLowLevelFirst	= FOM_OPTIONS_LOW_LVL_1ST,
		KeepOpenSlots		= FOM_OPTIONS_OPEN_SLOTS,
		KeepOpenSlots_MinLabel		= 0,
		KeepOpenSlots_MaxLabel		= MAX_KEEP_OPEN_SLOTS,
		AvoidQuestFood		= FOM_OPTIONS_AVOID_QUEST,
		AvoidBonusFood		= FOM_OPTIONS_AVOID_BONUS,
		SaveForCooking		= FOM_OPTIONS_AVOID_COOK,
		CookingUnknown		= FOM_OPTIONS_COOK_UNKNOWN,
		IconWarning			= FOM_OPTIONS_WARN_ICON,
		TextWarning			= FOM_OPTIONS_WARN_TEXT,
		AudioWarning		= FOM_OPTIONS_WARN_SOUND,
		AudioWarningBell	= FOM_OPTIONS_SOUND_BELL,
		WarningLevel		= FOM_OPTIONS_WARN_LEVEL,
		NotifyType			= FOM_OPTIONS_FEED_NOTIFY,
	};
	
	local s, widget, lastWidget, lastWidgetLeft, options;

	widget = panel:CreateCheckButton("Tooltip", false);
	widget:SetPoint("TOPLEFT", panel.contentAnchor, "BOTTOMLEFT", -2, -2);
	lastWidget = widget;
	
	widget = panel:CreateCheckButton("UseLowLevelFirst", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateSlider("KeepOpenSlots", 0, MAX_KEEP_OPEN_SLOTS, 1);
	widget:SetPoint("TOP", lastWidget, "CENTER", 0, -5);
	widget:SetPoint("RIGHT", -20, 0);
	
	s = panel:CreateFontString("FOM_OptionsPanel_AvoidHeader", "ARTWORK", "GameFontNormal");
	s:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 2, -16);
	s:SetText(FOM_OPTIONS_HEADER_AVOID);
	lastWidget = s;

	widget = panel:CreateCheckButton("AvoidQuestFood", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", -2, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("AvoidBonusFood", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("SaveForCooking", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;
	lastWidgetLeft = widget;
	lastWidgetLeft.dependentControls = {};
	
	options = {
		format(FOM_DIFFICULTY_RECIPES, GFWUtils.Hilite(FOM_DIFFICULTY_ALL)),
		format(FOM_DIFFICULTY_RECIPES, GFWUtils.ColorText(FOM_DIFFICULTY_GREEN, QuestDifficultyColor["standard"])),
		format(FOM_DIFFICULTY_RECIPES, GFWUtils.ColorText(FOM_DIFFICULTY_YELLOW, QuestDifficultyColor["difficult"])),
		format(FOM_DIFFICULTY_RECIPES, GFWUtils.ColorText(FOM_DIFFICULTY_ORANGE, QuestDifficultyColor["verydifficult"])),		
	};
	widget = panel:CreateDropDown("CookingLevel", options, 120);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 10, 0);
	table.insert(lastWidgetLeft.dependentControls, widget);
	lastWidget = widget;

	widget = panel:CreateCheckButton("CookingUnknown", false, true);
	widget:SetPoint("LEFT", lastWidget, "RIGHT", 10, 2);
	table.insert(lastWidgetLeft.dependentControls, widget);
	
	s = panel:CreateFontString("FOM_OptionsPanel_WarnHeader", "ARTWORK", "GameFontNormal");
	s:SetPoint("TOPLEFT", lastWidgetLeft, "BOTTOMLEFT", 2, -40);
	s:SetText(FOM_OPTIONS_HEADER_WARNING);
	lastWidget = s;
	lastWidgetLeft = s;

	widget = panel:CreateCheckButton("IconWarning", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", -2, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("TextWarning", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("AudioWarning", false);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("AudioWarningBell", false, true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 16, 0);
	lastWidget.dependentControls = { widget };
	lastWidget = widget;

	options = {
		{ text=PET_HAPPINESS1, icon="Interface\\PetPaperDollFrame\\UI-PetHappiness", 
			tCoordLeft=0.375, tCoordRight=0.5625, tCoordTop=0, tCoordBottom=0.359375 },
		{ text=PET_HAPPINESS2, icon="Interface\\PetPaperDollFrame\\UI-PetHappiness", 
			tCoordLeft=0.1875, tCoordRight=0.375, tCoordTop=0, tCoordBottom=0.359375 },
		FOM_OPTIONS_LEVEL_NONE,
	};
	widget = panel:CreateDropDown("WarningLevel", options, 130);
	widget:SetPoint("TOPLEFT", lastWidgetLeft, "BOTTOMLEFT", 175, -20);
	lastWidget = widget;

	options = {
		FOM_OPTIONS_NOTIFY_EMOTE,
		FOM_OPTIONS_NOTIFY_TEXT,
		FOM_OPTIONS_NOTIFY_NONE,
	};
	widget = panel:CreateDropDown("NotifyType", options, 130);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -20);
	
end

function FOM_ShowOptions()
	InterfaceOptionsFrame_OpenToFrame(FOM_OptionsPanel);
end

function GFW_FeedOMatic:Initialize()
	self.defaults = { 
		profile = {
			Tooltip				= true,
			UseLowLevelFirst	= true,
			KeepOpenSlots		= 8,
			AvoidQuestFood		= true,
			AvoidBonusFood		= true,
			SaveForCooking		= true,
			CookingLevel		= 1,
			CookingUnknown		= false,
			IconWarning			= false,
			TextWarning			= false,
			AudioWarning		= false,
			AudioWarningBell	= false,
			WarningLevel		= 1,
			NotifyType			= 1,
		}
	};
	self.db = self:InitializeDB("GFW_FeedOMaticDB", self.defaults);
	FOM_Config = self.db.profile;
end

function GFW_FeedOMatic:Enable()
	-- conditionalize 2.4 stuff for now so we can run on 2.3
	if (InterfaceOptions_AddCategory) then
		GFWOptions:CreateMainPanel("GFW_FeedOMatic", "FOM_OptionsPanel", FOM_OPTIONS_SUBTEXT);
		FOM_OptionsPanel.BuildUI = buildOptionsUI;
	end
end

GFW_FeedOMatic = DongleStub("Dongle-1.2"):New("GFW_FeedOMatic", GFW_FeedOMatic);

