------------------------------------------------------
-- FactionFriend.lua
------------------------------------------------------
local ADDON_NAME = "GFW_FactionFriend";

FACTION_BAR_COLORS = {
	[1] = {r = 1, g = 0.2, b = 0.15},	-- Hated
	[2] = {r = 0.8, g = 0.3, b = 0.22},	-- Hostile		}
	[3] = {r = 0.75, g = 0.27, b = 0},	-- Unfriendly	} same as default
	[4] = {r = 0.9, g = 0.7, b = 0},	-- Neutral		}
	[5] = {r = 0, g = 0.6, b = 0.1},	-- Friendly		}
	[6] = {r = 0, g = 0.6, b = 0.33},	-- Honored
	[7] = {r = 0, g = 0.7, b = 0.5},	-- Revered
	[8] = {r = 0, g = 0.7, b = 0.7},	-- Exalted
};

FFF_LastBarSwitchTime = 0;

function FFF_HookTooltip(frame)
	if (frame:GetScript("OnTooltipSetItem")) then
		frame:HookScript("OnTooltipSetItem", FFF_OnTooltipSetItem);
	else
		frame:SetScript("OnTooltipSetItem", FFF_OnTooltipSetItem);
	end
end

function FFF_OnTooltipSetItem()
	
	local name, link = this:GetItem();
	if (not link) then return; end
	if (not FFF_Config.NoTooltip and link) then
		local _, _, itemID  = string.find(link, "item:(%d+)");
		itemID = tonumber(itemID);
		
		if (FFF_ReverseCache == nil) then
			FFF_SetupReverseCache();
		end
		
		local itemInfo = FFF_ReverseCache[itemID];
		if (itemInfo) then
			if (type(itemInfo) == "table") then
				FFF_AddTooltipLine(this, FFF_FACTION_TURNIN..":", itemInfo[1]);
				for i = 2, #itemInfo do
					FFF_AddTooltipLine(this, " ", itemInfo[i]);
				end
			else
				FFF_AddTooltipLine(this, FFF_FACTION_TURNIN, itemInfo, 1);
			end
			return true;
		end

	end
	
end

function FFF_AddTooltipLine(tooltip, leftText, faction, isOnlyLine)
	local factionText = faction;
	if (faction == FFF_RACIAL_FACTIONS) then
		faction = UnitFactionGroup("player");
		factionText = faction..FFF_RACIAL_FACTIONS;
	end
	if (FFF_Notes[itemID] and FFF_Notes[itemID][faction]) then
		factionText = factionText.." ("..FFF_Notes[itemID][faction]..")";
	end
	local _, _, standingID = FFF_GetFactionInfoByName(faction);
	if (standingID == nil) then
		standingID = 4;
	end
	local baseColor = FACTION_BAR_COLORS[standingID];
	local color = {};
	for key, value in pairs(baseColor) do 
		color[key] = math.min(1, value * 1.25);
	end
	if (isOnlyLine) then
		this:AddLine(leftText.." "..factionText, color.r, color.g, color.b);
	else
		local c = HIGHLIGHT_FONT_COLOR;
		this:AddDoubleLine(leftText, factionText, c.r, c.g, c.b, color.r, color.g, color.b);
	end
end

function FFF_OnLoad()

	hooksecurefunc("ReputationWatchBar_Update", FFF_ReputationWatchBar_Update);
	ReputationWatchBar:HookScript("OnEnter", FFF_ReputationWatchBar_OnEnter);
	ReputationWatchBar:HookScript("OnLeave", FFF_ReputationWatchBar_OnLeave);

	FFF_HookTooltip(GameTooltip);
	FFF_HookTooltip(ItemRefTooltip);
		
	this:RegisterEvent("UPDATE_FACTION");
	this:RegisterEvent("PLAYER_ENTERING_WORLD");
	this:RegisterEvent("ADDON_LOADED");
	this:RegisterEvent("PLAYER_LEAVING_WORLD");
	this:RegisterEvent("PLAYER_CONTROL_GAINED");
	this:RegisterEvent("CHAT_MSG_COMBAT_FACTION_CHANGE");
	this:RegisterEvent("ZONE_CHANGED");
	this:RegisterEvent("ZONE_CHANGED_NEW_AREA");
	
	-- Register Slash Commands
	SLASH_FFF1 = "/factionfriend";
	SLASH_FFF2 = "/ff";
	SlashCmdList["FFF"] = function(msg)
		FFF_ShowOptions();
	end
	
end

function FFF_OnEvent(event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
	if ( event == "PLAYER_ENTERING_WORLD" or (event == "ADDON_LOADED" and arg1 == ADDON_NAME)) then
		this:RegisterEvent("BAG_UPDATE");
		this:RegisterEvent("UNIT_INVENTORY_CHANGED");
	elseif( event == "PLAYER_LEAVING_WORLD" ) then
		this:UnregisterEvent("UNIT_INVENTORY_CHANGED");
		this:UnregisterEvent("BAG_UPDATE");
	elseif( event == "PLAYER_CONTROL_GAINED" ) then
		if (FFF_OnTaxi) then
			FFF_OnTaxi = nil;
			FFF_CheckZone();
		end
	elseif ( event == "ZONE_CHANGED_NEW_AREA" or event == "ZONE_CHANGED" ) then
		if (UnitOnTaxi("player")) then
			FFF_OnTaxi = 1;
			return;
		end
		FFF_CheckZone();
	elseif ( event == "CHAT_MSG_COMBAT_FACTION_CHANGE" ) then
		if (FFF_Config.NoRepGained) then return; end
		
		local currentZone = GetRealZoneText();
		local zoneFaction = FFF_ZoneFactions[UnitFactionGroup("player")][currentZone] or FFF_ZoneFactions.Neutral[currentZone];
		if (FFF_FACTION_STANDING_INCREASED == nil) then
			FFF_FACTION_STANDING_INCREASED = GFWUtils.FormatToPattern(FACTION_STANDING_INCREASED);
		end
		local _, _, factionName = string.find(arg1, FFF_FACTION_STANDING_INCREASED);
		if (FFF_Config.NoZones) then
			local watchedFaction = GetWatchedFactionInfo();
			if (watchedFaction ~= factionName and GetTime() - FFF_LastBarSwitchTime > 5) then
				FFF_SetWatchedFaction(factionName);
			end
		else
			if (not zoneFaction or zoneFaction == factionName) then
				local watchedFaction = GetWatchedFactionInfo();
				if (watchedFaction ~= factionName and GetTime() - FFF_LastBarSwitchTime > 5) then
					FFF_SetWatchedFaction(factionName);
				end
			end
		end
	elseif ( event == "BAG_UPDATE") then
		FFF_ReputationWatchBar_Update();
	elseif ( event == "UNIT_INVENTORY_CHANGED") then
		if (arg1 == "player") then
			FFF_ReputationWatchBar_Update();
		end
	end
end

local racialFactions = {
	["Alliance"] = { FACTION_DWARF, FACTION_NELF, FACTION_GNOME, FACTION_HUMAN, FACTION_DRAENEI },
	["Horde"] = { FACTION_ORC, FACTION_TAUREN, FACTION_TROLL, FACTION_FORSAKEN, FACTION_BELF },
};
function FFF_SetupReverseCache()
	FFF_ReverseCache = {};
	FFF_Notes = {};
	for faction, quests in pairs(FFF_ItemInfo) do
		local myExcludedFactions = FFF_ExcludedFactions[UnitFactionGroup("player")];
		if (GFWTable.KeyOf(myExcludedFactions, faction) == nil) then
			for quest, questInfo in pairs(quests) do
				if (not questInfo.otherFactionRequired) then
					for itemID in pairs(questInfo.items) do
						if (FFF_ReverseCache[itemID]) then
							if (type(FFF_ReverseCache[itemID]) == "table") then
								if (GFWTable.KeyOf(FFF_ReverseCache[itemID], faction) == nil) then
									table.insert(FFF_ReverseCache[itemID], faction);
								end
							elseif (FFF_ReverseCache[itemID] ~= faction) then
								local oldValue = FFF_ReverseCache[itemID];
								FFF_ReverseCache[itemID] = {};
								table.insert(FFF_ReverseCache[itemID], oldValue);
								table.insert(FFF_ReverseCache[itemID], faction);
							end
						else
							FFF_ReverseCache[itemID] = faction;
						end
						if (questInfo.note) then
							if (FFF_Notes[itemID] == nil) then
								FFF_Notes[itemID] = {};
							end
							FFF_Notes[itemID][faction] = questInfo.note;
						end
					end
				end
			end
		end
	end
	local myRacialFactions = racialFactions[UnitFactionGroup("player")];
	for itemID, factionInfo in pairs(FFF_ReverseCache) do
		if (type(factionInfo) == "table") then
			local racialSubtracted = GFWTable.Subtract(factionInfo, myRacialFactions);
			if (#factionInfo - #racialSubtracted == #myRacialFactions) then
				if (#racialSubtracted == 0) then
					FFF_ReverseCache[itemID] = FFF_RACIAL_FACTIONS;
				else
					FFF_ReverseCache[itemID] = racialSubtracted;
					table.insert(FFF_ReverseCache[itemID], FFF_RACIAL_FACTIONS);
				end
			end
		end
	end
end

function FFF_GetFactionInfoByName(factionName)
	for factionIndex = 1, GetNumFactions() do
		local name, description, standingID, bottomValue, topValue, earnedValue, atWarWith, canToggleAtWar, isHeader, isCollapsed, isWatched = GetFactionInfo(factionIndex)
		if (name == factionName) then
			return name, description, standingID, bottomValue, topValue, earnedValue, atWarWith, canToggleAtWar, isHeader, isCollapsed, isWatched;
		end
	end
end

function FFF_CheckZone()
	if (FFF_Config.NoZones) then return; end
	
	local currentZone = GetRealZoneText();
	local zoneFaction = FFF_ZoneFactions[UnitFactionGroup("player")][currentZone] or FFF_ZoneFactions.Neutral[currentZone];
	if (zoneFaction) then
		local watchedFaction = GetWatchedFactionInfo();
		if (watchedFaction ~= zoneFaction and GetTime() - FFF_LastBarSwitchTime > 5) then
			FFF_SetWatchedFaction(zoneFaction);
		end
	end
end

function FFF_SetWatchedFaction(factionName)
	for index = 1, GetNumFactions() do
		local name = GetFactionInfo(index);
		if (name == factionName and not IsFactionInactive(index)) then
			SetWatchedFactionIndex(index);
			FFF_LastBarSwitchTime = GetTime();
			break;
		end
	end
end

function FFF_IsFactionItem(itemID)
	for factionName, factionInfo in pairs(FFF_ItemInfo) do
		for quest, qInfo in pairs(factionInfo) do
			for id, qtyPerTurnin in pairs(qInfo.items) do
				if (id == itemID) then 
					return true; 
				end
			end
		end
	end
end

function FFF_GetItemCount(itemID, includeBank)
	if (FFF_FakeItemCount and FFF_FakeItemCount[itemID]) then
		-- useful for debugging
		return FFF_FakeItemCount[itemID];
	end
	return GetItemCount(itemID, includeBank);
end

FFF_SpecialItems = {
	[12844] = { n=FFF_AD_TOKEN, q=2 },
	[19858] = { n=FFF_ZG_TOKEN, q=2 },
}
function FFF_ItemLink(itemID)
	local _, link = GetItemInfo(itemID);
	if (link) then return link; end
	
	local info = FFF_SpecialItems[itemID];
	if (info) then
		local _, _, _, colorCode = GetItemQualityColor(info.q);
		return colorCode .. info.n .. FONT_COLOR_CODE_CLOSE;
	end
end

FFF_PointsPerStanding = {
	[1] = 36000,	-- Hated
	[2] = 3000,		-- Hostile
	[3] = 3000,		-- Unfriendly
	[4] = 3000,		-- Neutral
	[5] = 6000,		-- Friendly
	[6] = 12000,	-- Honored
	[7] = 21000,	-- Revered
	[8] = 1000,
};
FFF_AbsMinForStanding = {
	[1] = 0 - FFF_PointsPerStanding[3] - FFF_PointsPerStanding[2] - FFF_PointsPerStanding[1],
	[2] = 0 - FFF_PointsPerStanding[3] - FFF_PointsPerStanding[2],
	[3] = 0 - FFF_PointsPerStanding[3],
	[4] = 0,
	[5] = FFF_PointsPerStanding[4],
	[6] = FFF_PointsPerStanding[4] + FFF_PointsPerStanding[5],
	[7] = FFF_PointsPerStanding[4] + FFF_PointsPerStanding[5] + FFF_PointsPerStanding[6],
	[8] = FFF_PointsPerStanding[4] + FFF_PointsPerStanding[5] + FFF_PointsPerStanding[6] + FFF_PointsPerStanding[7],
	[9] = FFF_PointsPerStanding[4] + FFF_PointsPerStanding[5] + FFF_PointsPerStanding[6] + FFF_PointsPerStanding[7] + FFF_PointsPerStanding[8],
};

function FFF_StandingForValue(value)
	value = math.min(value, FFF_AbsMinForStanding[9] - 1);	-- cap at max Exalted
	value = math.max(value, FFF_AbsMinForStanding[1]);	-- and min Hated
	for standingID = 1, 8 do
		if (value >= FFF_AbsMinForStanding[standingID] and value < FFF_AbsMinForStanding[standingID+1]) then
			local pointsInto = value - FFF_AbsMinForStanding[standingID];
			return standingID, pointsInto, FFF_PointsPerStanding[standingID];
		end
	end
end

function FFF_GetWatchedFactionPotential(withReport)
	return FFF_GetFactionPotential(nil, withReport)
end

function FFF_GetFactionPotential(faction, withReport)
	local factionName, standing, min, max, value;
	if (faction == nil) then
		factionName, standing, min, max, value = GetWatchedFactionInfo();
	elseif (type(faction) == "string") then
		factionName = faction;
		_, _, standing, min, max, value = FFF_GetFactionInfoByName(factionName);
		if (standing == nil) then return 0; end
	elseif (type(faction) == "number") then
		_, _, standing, min, max, value = GetFactionInfo(faction);
		if (standing == nil) then return 0; end
	else
		error("bad argument #1 to 'FFF_GetFactionPotential' (expected number, string, or nil, got "..type(faction)..")", 2);
	end	
		
	local factionInfo = FFF_ItemInfo[factionName];
	if (factionInfo == nil) then return 0; end
	
	local totalPotential = 0;
	local reportLines;
	if (withReport) then
		reportLines = {};
	end
	local multiplier = 1;
	local _, race = UnitRace("player");
	if (race == "Human") then
		multiplier = 1.1;
	end
	local itemsAccountedFor = {};
	local itemsCreated = {};
	for quest, qInfo in GFWTable.PairsByKeys(factionInfo, function(a,b) return a > b end) do
		local meetsRequirements;
		
		meetsRequirements = (standing >= (qInfo.minStanding or 1));
		meetsRequirements = meetsRequirements and (standing <= (qInfo.maxStanding or 8));

		if (meetsRequirements and qInfo.otherFactionRequired) then
			local _, _, standingID = FFF_GetFactionInfoByName(qInfo.otherFactionRequired.faction);
			if (standingID) then
				meetsRequirements = meetsRequirements and (standingID >= (qInfo.otherFactionRequired.minStanding or 1));
				meetsRequirements = meetsRequirements and (standingID <= (qInfo.otherFactionRequired.maxStanding or 8));
			end
		end
		
		if (meetsRequirements) then
			local potentialValue = 0;
			
			-- first, figure out how many turnins' worth we have of each item the quest requres
			local turninCounts = {};
			local reportItemLines = {};
			for itemID, qtyPerTurnin in pairs(qInfo.items) do
				local countIncludingBank = FFF_GetItemCount(itemID, true);
				local created = itemsCreated[itemID] or 0;
				local alreadyCounted = itemsAccountedFor[itemID] or 0;
				local turnins = math.floor((countIncludingBank + created - alreadyCounted) / qtyPerTurnin);
				table.insert(turninCounts, turnins);
			end
			
			-- if a quest requires multiple item, the number of turnins we can actually make
			-- is decided by whichever item we have the fewest turnins' worth of
			local numTurnins;
			if (table.getn(turninCounts) == 1) then
				numTurnins = turninCounts[1];
			elseif (table.getn(turninCounts) > 1) then
				numTurnins = math.min(unpack(turninCounts));
			else
				numTurnins = 0;
			end
			
			-- some turnins create items which can be used for later "turnins"
			if (qInfo.creates) then
				for createdID, qtyCreated in pairs(qInfo.creates) do
					itemsCreated[createdID] = (itemsCreated[createdID] or 0) + (qtyCreated * numTurnins);
				end
			end

			-- now that we know how many we're actually using of each item for this quest,
			-- account for it so they don't get used again for another quest,
			-- and include it in the report/tooltip
			for itemID, qtyPerTurnin in pairs(qInfo.items) do
				itemsAccountedFor[itemID] = numTurnins * qtyPerTurnin;
				
				if (withReport and numTurnins > 0) then
					local itemsForTurnin = itemsAccountedFor[itemID];
					local itemLink = FFF_ItemLink(itemID) or string.format(FFF_UNKNOWN_ITEM, itemID);
					local lineItem = string.format(FFF_REPORT_LINE_ITEM, itemsForTurnin, itemLink);
					local inBags = FFF_GetItemCount(itemID, false);
					local inBank = FFF_GetItemCount(itemID, true) - inBags;

					local lineItemAdditions = {};
					if (inBank > 0 and inBags < itemsForTurnin) then
						table.insert(lineItemAdditions, string.format(FFF_COUNT_IN_BANK, math.min(inBank, itemsForTurnin)));
					end
					local created = itemsCreated[itemID];
					if (created and inBank == 0 and created == itemsForTurnin) then
						table.insert(lineItemAdditions, FFF_ALL_CREATED);
					elseif (created and created > 0 and inBags + inBank < itemsForTurnin) then
						table.insert(lineItemAdditions, string.format(FFF_COUNT_CREATED, math.min(created, itemsForTurnin)));
					end
					if (#lineItemAdditions > 0) then
						lineItem = lineItem ..GRAY_FONT_COLOR_CODE.. " (".. table.concat(lineItemAdditions, ", ") .. ")" .. FONT_COLOR_CODE_CLOSE;
					end
					table.insert(reportItemLines, lineItem);
				end
			end
			
			-- Now we can figure how much rep we stand to gain from this quest...
			potentialValue = numTurnins * qInfo.value * multiplier;
			-- ... but we may need to adjust it in case it'd push us up to a new standing
			local potentialStanding = standing;
			local pointsIntoStanding;
			if (potentialValue > max) then
				local potentialTotal = value + potentialValue;
				potentialStanding, pointsIntoStanding = FFF_StandingForValue(potentialTotal);
			end
			if (qInfo.maxStanding and potentialStanding >= qInfo.maxStanding) then
				local maxValue = qInfo.maxValue or FFF_PointsPerStanding[qInfo.maxStanding];
				local absMaxValue = FFF_AbsMinForStanding[qInfo.maxStanding] + maxValue;
				if (value >= absMaxValue) then
					potentialValue = 0;
				end
				local maxPotential = absMaxValue - value;
				potentialValue = math.min(potentialValue, maxPotential);
			end
			
			-- Finally, throw what we've figured out into the report/tooltip
			if (potentialValue > 0 and withReport) then
				local reportLineHeader = string.format(FFF_REPORT_NUM_POINTS, potentialValue);
				if (numTurnins > 1 and not qInfo.useItem) then
					reportLineHeader = reportLineHeader  ..GRAY_FONT_COLOR_CODE..string.format(FFF_REPORT_NUM_TURNINS, numTurnins)..FONT_COLOR_CODE_CLOSE;
				end
				reportLineHeader = reportLineHeader ..":";
				table.insert(reportLines, {[reportLineHeader] = reportItemLines});
			end
			
			totalPotential = totalPotential + potentialValue;
		end
	end
	return totalPotential, reportLines;

end

function FFF_ReputationWatchBar_Update(newLevel)
	
	local name, standing, min, max, value = GetWatchedFactionInfo();
	if (not name) then return; end
	
	local potential = FFF_GetWatchedFactionPotential();
	local totalValue = value + potential;
	local tickSet = ((totalValue - min) / (max - min)) * ReputationWatchStatusBar:GetWidth();
	local tickSet = math.max(tickSet, 0);
	local tickSet = math.min(tickSet, ReputationWatchStatusBar:GetWidth());
	FFF_ReputationTick:ClearAllPoints();
	if (potential == 0 or FFF_Config.NoShowPotential) then
	    FFF_ReputationTick:Hide();
	    FFF_ReputationExtraFillBarTexture:Hide();
	else
		if (totalValue > FFF_AbsMinForStanding[standing+1]) then 
			FFF_ReputationTick:Hide();
		else
		    FFF_ReputationTick:Show();
		end
	    FFF_ReputationTick:SetPoint("CENTER", "ReputationWatchStatusBar", "LEFT", tickSet, 0);
	    FFF_ReputationExtraFillBarTexture:Show();
	    FFF_ReputationExtraFillBarTexture:SetPoint("TOPRIGHT", "ReputationWatchStatusBar", "TOPLEFT", tickSet, 0);
	    local color = FACTION_BAR_COLORS[standing];
		FFF_ReputationTickHighlight:SetVertexColor(color.r, color.g, color.b);
		if (totalValue > FFF_AbsMinForStanding[standing+1]) then 
			local potentialStanding = FFF_StandingForValue(totalValue);
			color = FACTION_BAR_COLORS[potentialStanding];
		    FFF_ReputationExtraFillBarTexture:SetVertexColor(color.r, color.g, color.b, 0.25);
		else
		    FFF_ReputationExtraFillBarTexture:SetVertexColor(color.r, color.g, color.b, 0.15);
		end
	end

end

function FFF_ReputationTick_Tooltip()
	-- TODO: if the tick is past the end of the bar, the tooltip should say how far into the next level you have turnins for
	local x,y;
	x,y = FFF_ReputationTick:GetCenter();
	if ( FFF_ReputationTick:IsVisible() ) then
		FFF_ReputationTick:LockHighlight();
		if ( x >= ( GetScreenWidth() / 2 ) ) then
			GameTooltip:SetOwner(FFF_ReputationTick, "ANCHOR_LEFT");
		else
			GameTooltip:SetOwner(FFF_ReputationTick, "ANCHOR_RIGHT");
		end
	else
		GameTooltip_SetDefaultAnchor(GameTooltip, UIParent);
	end
	
	local potential, reportLines = FFF_GetWatchedFactionPotential(1);
	local tooltipText = string.format(FFF_REPUTATION_TICK_TOOLTIP, potential);
	local name, standing, min, max, value = GetWatchedFactionInfo();
	local color = FACTION_BAR_COLORS[standing];

	-- First line: name of faction and label for current standing
	-- e.g. "The Aldor: Friendly"
	local standingName = getglobal("FACTION_STANDING_LABEL"..standing);
	if (UnitSex("player") == 3) then
		standingName = getglobal("FACTION_STANDING_LABEL"..standing.."_FEMALE");
	end
	GameTooltip:SetText(name..": "..standingName, color.r, color.g, color.b);

	if (potential == 0) then return; end

	-- Report lines summarizing faction available from turnins
	-- e.g. "500 points (2 turnins): 20x[Mark of Sargeras] (13 in bank)"
	GameTooltip:AddLine(tooltipText);
	for _, reportLine in pairs(reportLines) do
		for reportHeader, itemLines in pairs(reportLine) do
			local r, g, b = HIGHLIGHT_FONT_COLOR.r, HIGHLIGHT_FONT_COLOR.g, HIGHLIGHT_FONT_COLOR.b;
			GameTooltip:AddDoubleLine(reportHeader, itemLines[1], r, g, b, r, g, b);
			if (table.getn(itemLines) > 1) then
				for i = 2, table.getn(itemLines) do
					GameTooltip:AddDoubleLine(" ", itemLines[i], r, g, b, r, g, b);
				end
			end
		end
	end
	
	-- Last line (preceded by blank): standing after turnins (if different from current standing)
	-- e.g. "After turnins: Honored (100/12000)"
	local potentialTotal = potential + value;
	local potentialStanding, pointsInto, localMax = FFF_StandingForValue(potentialTotal);
	if (potentialStanding ~= standing) then
		GameTooltip:AddLine(" ");
		local potentialStandingName = getglobal("FACTION_STANDING_LABEL"..potentialStanding);
		if (UnitSex("player") == 3) then
			potentialStandingName = getglobal("FACTION_STANDING_LABEL"..potentialStanding.."_FEMALE");
		end
		local summary = string.format(FFF_AFTER_TURNINS_INFO, potentialStandingName, pointsInto, localMax);
		local c1 = HIGHLIGHT_FONT_COLOR;
		local c2 = FACTION_BAR_COLORS[potentialStanding];
		GameTooltip:AddDoubleLine(FFF_AFTER_TURNINS_LABEL, summary, c1.r, c1.g, c1.b, c2.r, c2.g, c2.b);
	end
	
	GameTooltip:Show();

end

function FFF_ReputationWatchBar_OnEnter()
	if (FFF_Config.NoShowPotential) then return; end
	FFF_ReputationTick_Tooltip();
	FFF_ShowingTooltip = true;
end

function FFF_ReputationWatchBar_OnLeave()
	FFF_ReputationTick:UnlockHighlight();
	if (FFF_ShowingTooltip) then
		GameTooltip:Hide();
		FFF_ShowingTooltip = false;
	end
end

------------------------------------------------------
-- ReputationWatchBar menu shamelessly, er, borrowed from AnduinLothar. Thanks!
------------------------------------------------------

function FFF_ToggleMenu()
	FFF_MenuDropDown.displayMode = "MENU";
	ToggleDropDownMenu(1, nil, FFF_MenuDropDown, "cursor", 0, 0, "TOP");
	PlaySound("igMainMenuOptionCheckBoxOn");
end

function FFF_ReputationWatchBar_OnClick(button)
	if (button == "RightButton") then
		FFF_ToggleMenu();
	end
end

function FFF_LoadMenu()
		
	local gender = UnitSex("player");
	local info;
	for i = 1, GetNumFactions() do
		if (not IsFactionInactive(i)) then
			local name, description, standingID, barMin, barMax, barValue, atWarWith, canToggleAtWar, isHeader, isCollapsed, isWatched = GetFactionInfo(i);
			
			if (not (isHeader and name == FACTION_INACTIVE)) then
				info = {};
				info.value = i;
				if (isHeader) then
					info.text = name;
					info.notClickable = 1;
					info.notCheckable = 1;
					info.isTitle = isHeader;
				else
					local baseColor = FACTION_BAR_COLORS[standingID];
					local color = {};
					for key, value in pairs(baseColor) do 
						color[key] = math.min(1, value * 1.25);
					end
					local standingLabel = GetText("FACTION_STANDING_LABEL"..standingID, gender);
					info.text = name..GFWUtils.ColorText(" ("..standingLabel..")", color);
					if (isWatched) then
						info.func = SetWatchedFactionIndex;
						info.arg1 = 0;
					else
						info.func = SetWatchedFactionIndex;
						info.arg1 = info.value;
					end
				end
				info.checked = isWatched;
				UIDropDownMenu_AddButton(info, 1);
			end
		end
	end
	
	-- spacer
	info = {};
	info.disabled = 1;
	info.text = "";
	UIDropDownMenu_AddButton(info);
	
	info = {};
	info.text = FFF_SHOW_REPUTATION_PANE;
	info.func = function() 
		ToggleCharacter("ReputationFrame"); 
	end;
	info.notCheckable = 1;
	UIDropDownMenu_AddButton(info);

	info = {};
	info.text = FFF_SHOW_OPTIONS;
	info.func = function()
		FFF_ShowOptions();
	end;
	info.notCheckable = 1;
	UIDropDownMenu_AddButton(info);
	
	
	-- nil the displayMode so that the rest of ToggleDropDownMenu doesn't error on the fake frame
	FFF_MenuDropDown.displayMode = nil;
end

-- Fake menu frame.... who says you need a frame, bah!
FFF_MenuDropDown = {
	initialize = FFF_LoadMenu;
	GetName = function() return "FFF_MenuDropDown" end;
	SetHeight = function() end;
}

local oldFunc = ReputationWatchBar:GetScript("OnMouseDown");
if (oldFunc) then
	ReputationWatchBar:SetScript("OnMouseDown", function() oldFunc(); FFF_ReputationWatchBar_OnClick(arg1); end);
else
	ReputationWatchBar:SetScript("OnMouseDown", function() FFF_ReputationWatchBar_OnClick(arg1); end);
end

------------------------------------------------------
-- Dongle & GFWOptions stuff
------------------------------------------------------

GFW_FactionFriend = {};
local GFWOptions = DongleStub("GFWOptions-1.0");

local function buildOptionsUI(panel)

	GFW_FactionFriend.optionsText = {
		Zones = FFF_OPTION_ZONES,
		RepGained = FFF_OPTION_REP_GAINED,
		ShowPotential = FFF_OPTION_SHOW_POTENTIAL,
		Tooltip = FFF_OPTION_TOOLTIP,
	};
	
	local s, widget, lastWidget;

	widget = panel:CreateCheckButton("ShowPotential", true);
	widget:SetPoint("TOPLEFT", panel.contentAnchor, "BOTTOMLEFT", -2, -16);
	lastWidget = widget;
	
	widget = panel:CreateCheckButton("Tooltip", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;
	
	s = panel:CreateFontString("FFF_OptionsPanel_SwitchHeader", "ARTWORK", "GameFontNormal");
	s:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 2, -16);
	s:SetText(FFF_OPTIONS_SWITCHBAR);
	lastWidget = s;

	widget = panel:CreateCheckButton("Zones", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", -2, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("RepGained", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

end

function FFF_ShowOptions()
	InterfaceOptionsFrame_OpenToFrame(FFF_OptionsPanel);
end

function GFW_FactionFriend:Initialize()
	self.defaults = { 
		profile = {
			NoZones = false,
			NoRepGained = false,
			NoShowPotential = false,
			NoTooltip = false,
		}
	};
	self.db = self:InitializeDB("GFW_FactionFriendDB", self.defaults);
	FFF_Config = self.db.profile;
end

function GFW_FactionFriend:Enable()
	-- conditionalize 2.4 stuff for now so we can run on 2.3
	if (InterfaceOptions_AddCategory) then
		GFWOptions:CreateMainPanel("GFW_FactionFriend", "FFF_OptionsPanel");
		FFF_OptionsPanel.BuildUI = buildOptionsUI;
	end
end

function GFW_FactionFriend:OptionsChanged()
	FFF_ReputationWatchBar_Update();
end

GFW_FactionFriend = DongleStub("Dongle-1.2"):New("GFW_FactionFriend", GFW_FactionFriend);

