------------------------------------------------------
-- DisenchantPredictor.lua
------------------------------------------------------

-- constants
local FDP_DUST = 1;
local FDP_ESSENCE = 2;
local FDP_SHARD = 3;
local FDP_CRYSTAL = 4;

-- Configuration
FDP_Config = { };
FDP_Config.Tooltip = true;
FDP_Config.Reagents = true;
FDP_Config.Items = true;
FDP_Config.Verbose = true;
FDP_Config.AutoLoot = true;
FDP_Exceptions = {};

FDP_ItemIDs = {
	["DUST_STRANGE"]			= 10940;
	["DUST_SOUL"]				= 11083;
	["DUST_VISION"]				= 11137;
	["DUST_DREAM"]				= 11176;
	["DUST_ILLUSION"]			= 16204;

	["DUST_ARCANE"]				= 22445;	-- Expansion
	
	["ESSENCE_MAGIC_LESSER"]	= 10938;
	["ESSENCE_MAGIC_GREATER"]	= 10939;
	["ESSENCE_ASTRAL_LESSER"]	= 10998;
	["ESSENCE_ASTRAL_GREATER"]	= 11082;
	["ESSENCE_MYSTIC_LESSER"]	= 11134;
	["ESSENCE_MYSTIC_GREATER"]	= 11135;
	["ESSENCE_NETHER_LESSER"]	= 11174;
	["ESSENCE_NETHER_GREATER"]	= 11175;
	["ESSENCE_ETERNAL_LESSER"]	= 16202;
	["ESSENCE_ETERNAL_GREATER"]	= 16203;

	["ESSENCE_PLANAR_LESSER"]	= 22447;	-- Expansion
	["ESSENCE_PLANAR_GREATER"]	= 22446;	-- Expansion

	["SHARD_GLIMMER_SMALL"]		= 10978;
	["SHARD_GLIMMER_LARGE"]		= 11084;
	["SHARD_GLOWING_SMALL"]		= 11138;
	["SHARD_GLOWING_LARGE"]		= 11139;
	["SHARD_RADIANT_SMALL"]		= 11177;
	["SHARD_RADIANT_LARGE"]		= 11178;
	["SHARD_BRILLIANT_SMALL"]	= 14343;
	["SHARD_BRILLIANT_LARGE"]	= 14344;

	["SHARD_PRISMATIC_SMALL"]	= 22448;	-- Expansion
	["SHARD_PRISMATIC_LARGE"]	= 22449;	-- Expansion

	["NEXUS_CRYSTAL"]			= 20725;
	["VOID_CRYSTAL"]			= 22450;	-- Expansion
}

-- req lvl	ilvl
-- 57		81
-- 58		84
-- 59		87
-- 60		90
-- 61		93		lesser planar
-- 62		96		lesser
-- 63		99		lesser
-- 64		102		greater planar
-- 65		105		greater planar
-- 66		108
-- 67		111
-- 68		114
-- 69		117?
-- 70		120?

-- peacekeeper boots, ilvl 68: small prismatic
-- zulian defender, ilvl 68: small prismatic

-- formula for dust by level: max(1, ceil((item level - 15) / 10))
FDP_DustNames = {
	"DUST_STRANGE",		-- item level 1-25      req level 1-20
	"DUST_SOUL",		-- item level 26-35     req level 21-30
	"DUST_VISION",		-- item level 36-45     req level 31-40
	"DUST_DREAM",		-- item level 46-65     req level 41-50
	"DUST_ILLUSION",	-- item level 56-65     req level 51-60
	"DUST_ARCANE",		-- item level 66+
}

-- formula for essence by level: max(1, ceil((item level - 10) / 5))
FDP_EssenceNames = {
	"ESSENCE_MAGIC_LESSER",		-- item level 6-15 
	"ESSENCE_MAGIC_GREATER",	-- item level 16-20
	"ESSENCE_ASTRAL_LESSER",	-- item level 21-25
	"ESSENCE_ASTRAL_GREATER",	-- item level 26-30
	"ESSENCE_MYSTIC_LESSER",	-- item level 31-35
	"ESSENCE_MYSTIC_GREATER",	-- item level 36-40
	"ESSENCE_NETHER_LESSER",	-- item level 41-45
	"ESSENCE_NETHER_GREATER",	-- item level 46-50
	"ESSENCE_ETERNAL_LESSER",	-- item level 51-55
	"ESSENCE_ETERNAL_GREATER",	-- item level 56-60
	"ESSENCE_ETERNAL_GREATER",	-- item level 61-65
	"ESSENCE_PLANAR_LESSER",	-- item level 66-70
	"ESSENCE_PLANAR_LESSER",	-- item level 71-75
	"ESSENCE_PLANAR_LESSER",	-- item level 76-80
	"ESSENCE_PLANAR_LESSER",	-- item level 81-85
	"ESSENCE_PLANAR_LESSER",	-- item level 86-90
	"ESSENCE_PLANAR_LESSER",	-- item level 91-95
	"ESSENCE_PLANAR_LESSER",	-- item level 96-100
	"ESSENCE_PLANAR_GREATER",	-- item level 101-105
}
	
-- formula for shard by level: max(1, ceil((item level - 20) / 5))
FDP_ShardNames = {
	"SHARD_GLIMMER_SMALL",		-- item level 1-25
	"SHARD_GLIMMER_LARGE",		-- item level 26-30
	"SHARD_GLOWING_SMALL",		-- item level 31-35
	"SHARD_GLOWING_LARGE",		-- item level 36-40
	"SHARD_RADIANT_SMALL",		-- item level 41-45
	"SHARD_RADIANT_LARGE",		-- item level 46-50
	"SHARD_BRILLIANT_SMALL",	-- item level 51-55
	"SHARD_BRILLIANT_LARGE",	-- item level 56-70 pre-BC items
	"SHARD_PRISMATIC_SMALL",	-- item level 66-99 BC items 
	"SHARD_PRISMATIC_LARGE",	-- item level 100+
}

FDP_CrystalNames = {
	"NEXUS_CRYSTAL",	-- item level 56-94 epics
	"VOID_CRYSTAL",		-- item level 95+ epics
}

local X1 = "57-63 "..FDP_BC;
local X2 = "64+ "..FDP_BC;
local Xall = "57+ "..FDP_BC;

FDP_DustLevels = {
	"1-20", "21-30", "31-40", "41-50", "51-60", Xall,
}

FDP_EssenceLevels = {
	"1-10", "11-15", "16-20", "21-25", "26-30", "31-35", "36-40", "41-45", "46-50", "51-60", "51-60", X1, X1, X1, X1, X1, X1, X1, X2,
}
	
FDP_ShardLevels = {
	"1-20", "21-25", "26-30", "31-35", "36-40", "41-45", "46-50", "51-60", X1, X2
}

FDP_CrystalLevels = {
	"51-60", Xall,
}

FDP_WeaponTypes = { "INVTYPE_2HWEAPON", "INVTYPE_WEAPON", "INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPONOFFHAND", "INVTYPE_RANGED", "INVTYPE_RANGEDRIGHT", "INVTYPE_THROWN" };
FDP_ArmorTypes = { "INVTYPE_BODY", "INVTYPE_CHEST", "INVTYPE_CLOAK", "INVTYPE_FEET", "INVTYPE_FINGER", "INVTYPE_HAND", "INVTYPE_HEAD", "INVTYPE_HOLDABLE", "INVTYPE_LEGS", "INVTYPE_NECK", "INVTYPE_RANGED", "INVTYPE_ROBE", "INVTYPE_SHIELD", "INVTYPE_SHOULDER", "INVTYPE_TRINKET", "INVTYPE_WAIST", "INVTYPE_WRIST", "INVTYPE_RELIC", };

function FDP_HookTooltip(frame)
	if (frame:GetScript("OnTooltipSetItem")) then
		frame:HookScript("OnTooltipSetItem", FDP_OnTooltipSetItem);
	else
		frame:SetScript("OnTooltipSetItem", FDP_OnTooltipSetItem);
	end
end

function FDP_OnTooltipSetItem()
	local name, link = this:GetItem();
	if ( FDP_Config.Tooltip and link) then
		
		local _, _, itemID = string.find(link, "item:(%d+)");
		if (itemID == nil or tonumber(itemID) == nil) then return false; end
		itemID = tonumber(itemID);
		if (FDP_StaticExceptions[itemID] or FDP_Exceptions[itemID]) then
			return false;
		end

		if (FDP_Config.Reagents) then
			if (FDP_TooltipForReagent(this, itemID)) then
				return true;
			end
		end
		
		if (FDP_Config.Items) then
			local quality, level, kind = FDP_ItemInfoForLink(link);
			if (level and level > 0 and kind) then
				for lineNum = 1, this:NumLines() do
					local leftText = getglobal(this:GetName().."TextLeft"..lineNum):GetText();
					if (leftText == ITEM_DISENCHANT_NOT_DISENCHANTABLE) then
						FDP_Exceptions[itemID] = 1;
						return false;
					end
				end
				
				local dustIndex, essenceIndex, shardIndex, crystalIndex = FDP_DisenchantIndex(level, quality, itemID);
				local dustName, essenceName, shardName, crystalName;
				if (quality == 4 and crystalIndex ) then
					crystalName = FDP_DisplayName(FDP_CrystalNames[crystalIndex]);
					this:AddLine(FDP_CAN_DIS_TO.." "..crystalName, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
					return true;
				elseif (quality == 3) then
					shardName = FDP_DisplayName(FDP_ShardNames[shardIndex]);
					if (not shardName) then return nil; end
					if (crystalIndex) then
						crystalName = FDP_DisplayName(FDP_CrystalNames[crystalIndex]);
						if (FDP_Config.Verbose) then
							this:AddLine(FDP_CAN_DIS_TO, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
							this:AddLine("  "..shardName.." "..FDP_MOST_LIKELY, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
							this:AddLine("  "..crystalName.." "..FDP_RARELY, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
						else
							this:AddLine(FDP_CAN_DIS_TO_SHORT, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
							this:AddDoubleLine(" ", shardName..crystalName);
						end
					else
						this:AddLine(FDP_CAN_DIS_TO.." "..shardName, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
					end
					return true;
				elseif (quality == 2) then
					dustName = FDP_DisplayName(FDP_DustNames[dustIndex]);
					essenceName = FDP_DisplayName(FDP_EssenceNames[essenceIndex]);
					shardName = FDP_DisplayName(FDP_ShardNames[shardIndex]);
					if (not dustName or not essenceName or not shardName) then return nil; end
					if (FDP_Config.Verbose) then
						this:AddLine(FDP_CAN_DIS_TO, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
						this:AddLine("  "..dustName.." "..FDP_MOST_LIKELY, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
						if (kind == "WEAPON") then
							this:AddLine("  "..essenceName.." "..FDP_JUST_LIKELY, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
						else
							this:AddLine("  "..essenceName.." "..FDP_OCCASIONALLY, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);		
						end
						this:AddLine("  "..shardName.." "..FDP_RARELY, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
					else
						this:AddDoubleLine(FDP_CAN_DIS_TO_SHORT, shardName, GFW_FONT_COLOR.r, GFW_FONT_COLOR.g, GFW_FONT_COLOR.b);
						this:AddDoubleLine(" ", essenceName..dustName);
					end
					return true;
				end
			end
		end
	end
	return false;
end

function FDP_ContainerFrameItemButton_OnClick(button)
	if ( button == "LeftButton" and not FDP_IsDisenchanting ) then
		local bagID = this:GetParent():GetID();
		local slotID = this:GetID();
	
		FDP_ClickedItem = GetContainerItemLink(bagID, slotID);
	end
end

function FDP_PaperDollItemSlotButton_OnClick(button)	
	if ( button == "LeftButton" and not FDP_IsDisenchanting ) then
		FDP_ClickedItem = GetInventoryItemLink("player", this:GetID());
	end
end

function FDP_TooltipForReagent(frame, itemID)
	
	local identifier = GFWTable.KeyOf(FDP_ItemIDs, itemID);
	if (identifier == nil) then 
		return false; -- not an itemID we care about
	end
	
	local kind, levelRange;
	local namesTables = {FDP_DustNames, FDP_EssenceNames, FDP_ShardNames, FDP_CrystalNames};
	local levelTables = {FDP_DustLevels, FDP_EssenceLevels, FDP_ShardLevels, FDP_CrystalLevels};
	for tableID, aTable in pairs(namesTables) do
		local index = GFWTable.KeyOf(aTable, identifier);
		if (index) then
			levelRange = levelTables[tableID][index];
			kind = tableID;
			break;
		end
	end

	if (kind) then
		frame:AddLine(string.format(FDP_CAN_DIS_FROM_FORMAT, levelRange), 0.7,0.7,0.7);
		if (FDP_Config.Verbose) then
			if (kind == FDP_CRYSTAL) then
				frame:AddLine(FDP_CRYSTAL_VERBOSE, 0.7,0.7,0.7);
			elseif (kind == FDP_SHARD) then
				frame:AddLine(FDP_SHARD_VERBOSE, 0.7,0.7,0.7);
			elseif (kind == FDP_ESSENCE) then
				frame:AddLine(FDP_ESSENCE_VERBOSE, 0.7,0.7,0.7);
			elseif (kind == FDP_DUST) then
				frame:AddLine(FDP_DUST_VERBOSE, 0.7,0.7,0.7);
			end
		end
		return true;
	end
end

function FDP_OnLoad()

	-- Register Slash Commands
	SLASH_FDP1 = "/enchant";
	SLASH_FDP2 = "/disenchant";
	SLASH_FDP3 = "/ench";
	SLASH_FDP4 = "/dis";
	SLASH_FDP5 = "/de";
	SLASH_FDP6 = "/dp";
	SlashCmdList["FDP"] = function(msg)
		FDP_ChatCommandHandler(msg);
	end
			
	DisenchantPredictorFrame:RegisterEvent("LOOT_OPENED");	
	DisenchantPredictorFrame:RegisterEvent("UNIT_SPELLCAST_START");	
	DisenchantPredictorFrame:RegisterEvent("UNIT_SPELLCAST_INTERRUPTED");	
	
	FDP_HookTooltip(GameTooltip);
	FDP_HookTooltip(ItemRefTooltip);

	hooksecurefunc("ContainerFrameItemButton_OnClick", FDP_ContainerFrameItemButton_OnClick);
	hooksecurefunc("PaperDollItemSlotButton_OnClick", FDP_PaperDollItemSlotButton_OnClick);

end

local LootOpenedTime = 0;
local MIN_TRY_AGAIN_TIME = 1;

function FDP_OnEvent(event, arg1)
	if (event == "LOOT_OPENED") then
		if (GetTime() - LootOpenedTime < MIN_TRY_AGAIN_TIME) then
			return; -- UIParent likes to get lost in infinite recursion...
		end
		LootOpenedTime = GetTime();

		if (FDP_IsDisenchanting and LootFrame:IsVisible()) then
			for slot = 1, GetNumLootItems() do
				if (LootSlotIsCoin(slot)) then
					return; -- coins don't come from disenchanting.
				else
					local link = GetLootSlotLink(slot);
					if (link == nil) then 
						return; 
					end
					local _, _, itemID = string.find(link, "item:(%d+)");
					itemID = tonumber(itemID);
					if (itemID == nil) then return; end
					local identifier = GFWTable.KeyOf(FDP_ItemIDs, itemID);
					if (identifier == nil) then 
						-- if it's not one of our known disenchanted reagents, we probably didn't get here by disenchanting.
						return; 
					end
					if (FDP_IsDisenchanting and FDP_ClickedItem) then
						local quality, level, kind = FDP_ItemInfoForLink(FDP_ClickedItem);
						local dustIndex, essenceIndex, shardIndex, crystalIndex = FDP_DisenchantIndex(level, quality, itemID);
						local indices = {dustIndex, essenceIndex, shardIndex, crystalIndex};
						
						for i, table in pairs({FDP_DustNames, FDP_EssenceNames, FDP_ShardNames, FDP_CrystalNames}) do
							if (GFWTable.KeyOf(table, identifier)) then
								local expectedName = table[indices[i]];
								if (expectedName ~= identifier) then
									local version = GetAddOnMetadata("GFW_DisenchantPredictor", "Version");
									local _, _, _, colorCode = GetItemQualityColor(quality);
									local qualityName = colorCode..getglobal("ITEM_QUALITY"..quality.."_DESC")..FONT_COLOR_CODE_CLOSE;
									GFWUtils.Print(string.format(GFWUtils.Red("Disenchant Predictor %s:").." Expected %s from a level %s %s item, but got %s instead. Please check www.fizzwidget.com for updates.", version, FDP_DisplayName(expectedName), GFWUtils.Hilite(level), qualityName, FDP_DisplayName(identifier)));
								end
							end
						end
					end
				end
			end
			if (FDP_Config.AutoLoot) then
				CloseLoot(); -- closing the loot window, if it's from disenchanting, automatically loots the items.
			end
		end
		FDP_ClickedItem = nil;	
		FDP_IsDisenchanting = nil;
	elseif (event == "UNIT_SPELLCAST_START" and arg1 == "player") then
		local spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo("player");
		if (FDP_ClickedItem and icon == "Interface\\Icons\\INV_Enchant_Disenchant") then
			local _, _, _, numSpells = GetSpellTabInfo(1);
			-- only check the first tab of spellbook; same icon for DE is on other spells, but not on any other General spells.
			for spellIndex = 1, numSpells do
				local texture = GetSpellTexture(spellIndex, BOOKTYPE_SPELL);
				local spellName, spellRank = GetSpellName(spellIndex, BOOKTYPE_SPELL);
				if (texture == icon and spellName == spell and spellRank == rank) then
					if (FDP_Config.AutoLoot) then
						GFWUtils.Print(string.format(FDP_DISENCHANTING_STATUS, FDP_ClickedItem));
					end
					FDP_IsDisenchanting = true;
				end
			end
		end
	elseif (event == "UNIT_SPELLCAST_INTERRUPTED" and arg1 == "player") then
		FDP_ClickedItem = nil;	
		FDP_IsDisenchanting = nil;
	end
end

function FDP_ChatCommandHandler(msg)

	-- Print Help
	if ( msg == "help" ) or ( msg == "" ) then
		local version = GetAddOnMetadata("GFW_DisenchantPredictor", "Version");
		GFWUtils.Print("Fizzwidget Disenchant Predictor "..version..":");
		GFWUtils.Print("/enchant (or /ench or /disenchant or /dis or /de)");
		GFWUtils.Print("- "..GFWUtils.Hilite("help").." - "..FDP_HELP_HELP);
		GFWUtils.Print("- "..GFWUtils.Hilite("status").." - "..FDP_HELP_STATUS);
		GFWUtils.Print("- "..GFWUtils.Hilite("reagents on").." | "..GFWUtils.Hilite("off").." - "..FDP_HELP_REAGENTS);
		GFWUtils.Print("- "..GFWUtils.Hilite("items on").." | "..GFWUtils.Hilite("off").." - "..FDP_HELP_ITEMS);
		GFWUtils.Print("- "..GFWUtils.Hilite("tooltip on").." | "..GFWUtils.Hilite("off").." - "..FDP_HELP_TOOLTIP);
		GFWUtils.Print("- "..GFWUtils.Hilite("verbose on").." | "..GFWUtils.Hilite("off").." - "..FDP_HELP_VERBOSE);
		GFWUtils.Print("- "..GFWUtils.Hilite("autoloot on").." | "..GFWUtils.Hilite("off").." - "..FDP_HELP_AUTOLOOT);
		GFWUtils.Print("- "..GFWUtils.Hilite(FDP_CMD_LINK).." - "..FDP_HELP_LINK);
		return;
	end
	
	if (msg == "version") then
		local version = GetAddOnMetadata("GFW_DisenchantPredictor", "Version");
		GFWUtils.Print("Fizzwidget Disenchant Predictor "..version);
		return;
	end
	
	if (msg == "reagents on") then
		FDP_Config.Reagents = true;
		FDP_Config.Tooltip = true;
		GFWUtils.Print(FDP_STATUS_REAGENTS_ON);
		return;
	end
	if (msg == "reagents off") then
		FDP_Config.Reagents = false;
		GFWUtils.Print(FDP_STATUS_REAGENTS_OFF);
		return;
	end
	if (msg == "items on") then
		FDP_Config.Items = true;
		FDP_Config.Tooltip = true;
		GFWUtils.Print(FDP_STATUS_ITEMS_ON);
		return;
	end
	if (msg == "items off") then
		FDP_Config.Items = false;
		GFWUtils.Print(FDP_STATUS_ITEMS_OFF);
		return;
	end
	if (msg == "tooltip on") then
		FDP_Config.Tooltip = true;
		GFWUtils.Print(FDP_STATUS_TOOLTIP_ON);
		return;
	end
	if (msg == "tooltip off") then
		FDP_Config.Tooltip = false;
		GFWUtils.Print(FDP_STATUS_TOOLTIP_OFF);
		return;
	end
	if (msg == "verbose on") then
		FDP_Config.Verbose = true;
		GFWUtils.Print(FDP_STATUS_VERBOSE_ON);
		return;
	end
	if (msg == "verbose off") then
		FDP_Config.Verbose = false;
		GFWUtils.Print(FDP_STATUS_VERBOSE_OFF);
		return;
	end
	if (msg == "autoloot on") then
		FDP_Config.AutoLoot = true;
		GFWUtils.Print(FDP_STATUS_AUTOLOOT_ON);
		return;
	end
	if (msg == "autoloot off") then
		FDP_Config.AutoLoot = false;
		GFWUtils.Print(FDP_STATUS_AUTOLOOT_OFF);
		return;
	end
	
	if ( msg == "status" ) then
		if ( FDP_Config.Tooltip and (FDP_Config.Items or FDP_Config.Reagents)) then
			if (FDP_Config.Items) then
				GFWUtils.Print(FDP_STATUS_ITEMS_ON);
			else
				GFWUtils.Print(FDP_STATUS_ITEMS_OFF);				
			end
			if (FDP_Config.Reagents) then
				GFWUtils.Print(FDP_STATUS_REAGENTS_ON);
				if ( FDP_Config.Verbose ) then
					GFWUtils.Print(FDP_STATUS_VERBOSE_ON);
				else
					GFWUtils.Print(FDP_STATUS_VERBOSE_OFF);
				end
			else
				GFWUtils.Print(FDP_STATUS_REAGENTS_OFF);				
			end
		else
			GFWUtils.Print(FDP_STATUS_TOOLTIP_OFF);
		end
		if ( FDP_Config.AutoLoot ) then
			GFWUtils.Print(FDP_STATUS_AUTOLOOT_ON);
		else
			GFWUtils.Print(FDP_STATUS_AUTOLOOT_OFF);
		end
		return;
	end

	
	local _, _, link  = string.find(msg, "(|c%x+|Hitem:[-%d:]+|h%[.-%]|h|r)");
	if (link and link ~= "") then

		local _, _, itemID = string.find(link, "item:([-%d]+)");
		itemID = tonumber(itemID);
		if (itemID == nil) then 
			GFWUtils.Print(string.format(FDP_ERROR_ITEMID_FORMAT, link));
			return; 
		end
		
		local identifier = GFWTable.KeyOf(FDP_ItemIDs, itemID);
		if (identifier) then 
			local kind, levelRange;
			local namesTables = {FDP_DustNames, FDP_EssenceNames, FDP_ShardNames, FDP_CrystalNames};
			local levelTables = {FDP_DustLevels, FDP_EssenceLevels, FDP_ShardLevels, FDP_CrystalLevels};
			for tableID, aTable in pairs(namesTables) do
				local index = GFWTable.KeyOf(aTable, identifier);
				if (index) then
					levelRange = levelTables[tableID][index];
					kind = tableID;
					break;
				end
			end
			if (kind) then
				GFWUtils.Print(link..": "..string.format(FDP_CAN_DIS_FROM_FORMAT, levelRange));
				if (FDP_Config.Verbose) then
					if (kind == FDP_CRYSTAL) then
						GFWUtils.Print(FDP_CRYSTAL_VERBOSE);
					elseif (kind == FDP_SHARD) then
						GFWUtils.Print(FDP_SHARD_VERBOSE);
					elseif (kind == FDP_ESSENCE) then
						GFWUtils.Print(FDP_ESSENCE_VERBOSE);
					elseif (kind == FDP_DUST) then
						GFWUtils.Print(FDP_DUST_VERBOSE);
					end
				end
				return;
			end
		else
			GFWUtils.Print(string.format(FDP_ERROR_ITEMID_FORMAT, link));
			return; 
		end
		
		if (FDP_StaticExceptions[itemID] or FDP_Exceptions[itemID]) then
			GFWUtils.Print(string.format(FDP_CANT_DIS_EXCEPTION_FORMAT, link));
			return;
		end
		local quality, level, kind = FDP_ItemInfoForLink(link);
		if (quality < 2 or quality > 0) then
			GFWUtils.Print(string.format(FDP_CANT_DIS_QUALITY_FORMAT, link));
			return;
		end
		
		if (level and kind and level > 0) then
			local dustIndex, essenceIndex, shardIndex, crystalIndex = FDP_DisenchantIndex(level, quality, itemID);
			local dustName, essenceName, shardName, crystalName;
			if (quality == 4 and kind and crystalIndex) then
				crystalName = FDP_DisplayName(FDP_CrystalNames[crystalIndex]);
				GFWUtils.Print(link ..": "..FDP_CAN_DIS_TO.." ".. crystalName);
				return;
			elseif (quality == 3 and kind) then
				shardName = FDP_DisplayName(FDP_ShardNames[shardIndex]);
				if (crystalIndex) then
					crystalName = FDP_DisplayName(FDP_CrystalNames[crystalIndex]);
					GFWUtils.Print(link ..": "..FDP_CAN_DIS_TO);
					GFWUtils.Print(" - "..shardName.." "..FDP_MOST_LIKELY);
					GFWUtils.Print(" - "..crystalName.." "..FDP_RARELY);
				else
					GFWUtils.Print(link ..": "..FDP_CAN_DIS_TO.." "..shardName);
				end
				return;
			elseif (quality == 2 and kind) then
				dustName = FDP_DisplayName(FDP_DustNames[dustIndex]);
				essenceName = FDP_DisplayName(FDP_EssenceNames[essenceIndex]);
				shardName = FDP_DisplayName(FDP_ShardNames[shardIndex]);
				GFWUtils.Print(link ..": "..FDP_CAN_DIS_TO);
				GFWUtils.Print(" - "..dustName.." "..FDP_MOST_LIKELY);
				if (kind == "WEAPON") then
					GFWUtils.Print(" - "..essenceName.." "..FDP_JUST_LIKELY);
				else
					GFWUtils.Print(" - "..essenceName.." "..FDP_OCCASIONALLY);		
				end
				GFWUtils.Print(" - "..shardName.." "..FDP_RARELY);
				return;
			end
		else
			GFWUtils.Print(string.format(FDP_CANT_DIS_TYPE_FORMAT, link));
			return;
		end
		GFWUtils.Print(string.format(FDP_BAIL_FORMAT, link));
		return;
	end
	
	-- If we're this far, we probably have bad input.
	FDP_ChatCommandHandler("help");
end

local BCBlueExceptions = {
	[23835]	= 1,	-- Gnomish Poultryizer
	[23836]	= 1,	-- Goblin Rocket Launcher
	[25653]	= 1,	-- Riding Crop
	[32863]	= 1,	-- Skybreaker Whip
};
function FDP_DisenchantIndex(level, quality, itemID)
    local dustIndex = math.min(6, math.max(1, math.ceil((level - 15) / 10)));
    local essenceIndex = math.min(19, math.max(1, math.ceil((level - 10) / 5)));
    local shardIndex = math.min(10, math.max(1, math.ceil((level - 20) / 5)));
	local crystalIndex;

	-- The pattern gets disrupted for shards and crystals as we get into the BC level range
	if (BCBlueExceptions[itemID]) then
		shardIndex = 9;		-- Small Prismatic
		crystalIndex = 1;	-- Nexus
	elseif (quality == 2 or quality == 3) then
	 	if (level >= 56 and level < 65) then
			shardIndex = 8;		-- Large Brilliant
			crystalIndex = 1;	-- Nexus
		elseif (level >= 66 and level < 100) then
			shardIndex = 9;		-- Small Prismatic
			crystalIndex = 1;	-- Nexus
		elseif (level >= 100) then
			shardIndex = 10;	-- Large Prismatic
			crystalIndex = 2;	-- Void
		end
	elseif (quality == 4 and level >= 56 and level < 95) then
		crystalIndex = 1;	-- Nexus
	elseif (quality == 4 and level >= 95) then
		crystalIndex = 2;	-- Void
	end
    return dustIndex, essenceIndex, shardIndex, crystalIndex;
end

-- returns a link if possible, colored localized name otherwise.
function FDP_DisplayName(identifier)
	local itemID = FDP_ItemIDs[identifier];
	if (itemID == nil) then
		return getglobal(identifier); -- shouldn't happen anyways, right?
	end

	local namesTables = {FDP_DustNames, FDP_EssenceNames, FDP_ShardNames, FDP_CrystalNames};
	local kind;
	for tableID, aTable in pairs(namesTables) do
		local index = GFWTable.KeyOf(aTable, identifier);
		if (index) then
			kind = tableID;
			break;
		end
	end
	local _, _, _, colorCode = GetItemQualityColor(kind);
	
	local localizedName, link = GetItemInfo(itemID);
	if (localizedName) then
		return link;
	else
		localizedName = getglobal(identifier);
		return colorCode..localizedName..FONT_COLOR_CODE_CLOSE;
	end
end

function FDP_ItemInfoForLink(itemLink)
	
	_, _, itemID = string.find(itemLink, "item:(%d+)");
	itemID = tonumber(itemID);
	if (itemID == nil) then return nil; end
	
	local name, link, quality, itemLevel, minLevel, type, subType, stackCount, equipLoc, texture = GetItemInfo(itemID);	

	-- we use equipLoc instead of type because it's unlocalized
	local kind;		
	if (GFWTable.KeyOf(FDP_WeaponTypes, equipLoc)) then
		kind = "WEAPON";
	end
	if (GFWTable.KeyOf(FDP_ArmorTypes, equipLoc)) then
		kind = "ARMOR";
	end
	
	if (kind) then
		return quality, itemLevel, kind;
	end
end

function FDP_TooltipInfoForLink(link)
	FDPHiddenTooltip:ClearLines();
	FDPHiddenTooltip:SetHyperlink(link);
	local level, skill, kind;
	if (FDP_ITEM_DURATION_DAYS == nil) then
		FDP_ITEM_DURATION_DAYS = GFWUtils.FormatToPattern(ITEM_DURATION_DAYS);
	end
	if (FDP_ITEM_DURATION_HOURS == nil) then
		FDP_ITEM_DURATION_HOURS = GFWUtils.FormatToPattern(ITEM_DURATION_HOURS);
	end
	if (FDP_ITEM_DURATION_MIN == nil) then
		FDP_ITEM_DURATION_MIN = GFWUtils.FormatToPattern(ITEM_DURATION_MIN);
	end
	if (FDP_ITEM_DURATION_SEC == nil) then
		FDP_ITEM_DURATION_SEC = GFWUtils.FormatToPattern(ITEM_DURATION_SEC);
	end
	if (FDP_ITEM_MIN_LEVEL == nil) then
		FDP_ITEM_MIN_LEVEL = GFWUtils.FormatToPattern(ITEM_MIN_LEVEL);
	end
	if (FDP_ITEM_MIN_SKILL == nil) then
		FDP_ITEM_MIN_SKILL = GFWUtils.FormatToPattern(ITEM_MIN_SKILL);
	end
	for lineNum = 1, FDPHiddenTooltip:NumLines() do
		local leftText = getglobal("FDPHiddenTooltipTextLeft"..lineNum):GetText();
		if (string.find(leftText, FDP_ITEM_DURATION_DAYS) or string.find(leftText, FDP_ITEM_DURATION_HOURS) or string.find(leftText, FDP_ITEM_DURATION_MIN) or string.find(leftText, FDP_ITEM_DURATION_SEC)) then
			return nil; -- items with a duration can't be disenchanted even if we might otherwise think they can.
		end
		local _, _, skillName, skillString = string.find(leftText, FDP_ITEM_MIN_SKILL);
		if (levelString == nil and skillString and tonumber(skillString)) then
			if (lineNum < 4) then
				kind = "RECIPE"; 
				-- if it's got a skill level on one of the first couple of lines and no armor slot or speed, it's very likely a recipe
			end
			if (skill) then
				kind = "RECIPE"; -- if it lists a required skill level more than once, it's almost certainly a recipe
			end
			skill = tonumber(skillString);
		end
	end
	for lineNum = 1, FDPHiddenTooltip:NumLines() do
		-- for some reason ClearLines alone isn't clearing the right-side text
		getglobal("FDPHiddenTooltipTextLeft"..lineNum):SetText(nil);
		getglobal("FDPHiddenTooltipTextRight"..lineNum):SetText(nil);
	end
	--DevTools_Dump({skill=skill, level=level});
	if (kind == "RECIPE") then
		return nil; -- if there's both a skill requirement and a level requirement, it's probably a recipe (which can't be DE'ed)
	end
	return skill;
end

------------------------------------------------------
-- Runtime loading
------------------------------------------------------

FDP_OnLoad();
