------------------------------------------------------
-- AdSpace.lua
------------------------------------------------------

FAS_SPECIAL_VENDOR_COLOR = {r=1.0, g=0.5, b=0.25};
FAS_INDENT = "  ";

FAS_AllClasses = { "PALADIN", "SHAMAN", "MAGE", "PRIEST", "WARLOCK", "WARRIOR", "HUNTER", "ROGUE", "DRUID", "ANY" };

-- Configuration
FAS_NewItemInfo = { };
FAS_NewVendorInfo = { };
FAS_NewVendorLocations = { };

function FAS_HookTooltip(frame)
	if (frame:GetScript("OnTooltipSetItem")) then
		frame:HookScript("OnTooltipSetItem", FAS_OnTooltipSetItem);
	else
		frame:SetScript("OnTooltipSetItem", FAS_OnTooltipSetItem);
	end
end

function FAS_OnTooltipSetItem()
	if (MerchantFrame:IsVisible()) then
		for buttonIndex = 1, MERCHANT_ITEMS_PER_PAGE do
			local button = getglobal("MerchantItem"..buttonIndex.."ItemButton");
			if (this:IsOwned(button)) then return; end
		end
	end

	local name, link = this:GetItem();
	if (not link) then return; end
	-- workaround for OnTooltipSetItem being called twice for recipes
	local _, _, _, _, _, itemType, _, _, _, _ = GetItemInfo(link);
	if (itemType == FAS_RECIPE and this.lastLink ~= link) then
		for i = 1, this:NumLines() do
			local line = getglobal(this:GetName().."TextLeft"..i);
			local text = line:GetText();
			if (text and string.sub(text, 1, 1) == "\n") then 
				this.lastLink = link;
				return; 
			end
		end
	end
	this.lastLink = nil;
	
	if (link) then
		local _, _, itemID  = string.find(link, "item:(%d+)");
		itemID = tonumber(itemID);
		local itemInfo = FAS_ItemInfo[itemID];
		if (not itemInfo) then
			itemInfo = FAS_NewItemInfo[itemID];
		end
		
		if (not FAS_Config.NoRecipes and itemInfo) then
			local myFaction = UnitFactionGroup("player");
			local vendors = GFWTable.Merge(FAS_VendorInfo[myFaction][itemID], FAS_VendorInfo["Neutral"][itemID]);
			if (FAS_NewVendorInfo[myFaction] and FAS_NewVendorInfo[myFaction][itemID]) then
				vendors = GFWTable.Merge(vendors, FAS_NewVendorInfo[myFaction][itemID]);
			end
			if (FAS_NewVendorInfo["Neutral"] and FAS_NewVendorInfo["Neutral"][itemID]) then
				vendors = GFWTable.Merge(vendors, FAS_NewVendorInfo["Neutral"][itemID]);
			end
			local note = itemInfo.note;
			
			local color;
			local intro;
			if (note) then
				color = FAS_SPECIAL_VENDOR_COLOR;
			else
				color = GFW_FONT_COLOR;
			end
			if (not FAS_Config.NoShowCost) then
				local priceSummmary;
				if (itemInfo.b and itemInfo.b > 0) then
				 	priceSummmary = GFWUtils.TextGSC(itemInfo.b);
				else
					priceSummmary = "";
				end
				if (itemInfo.h) then
					priceSummmary = priceSummmary .. " + ".. itemInfo.h.. " honor";
				end
				if (itemInfo.a) then
					priceSummmary = priceSummmary .. " + ".. itemInfo.a.. " arena";
				end
				if (itemInfo.i) then
					priceSummmary = priceSummmary .. " + ".. itemInfo.i;
				end
				priceSummmary = string.gsub(priceSummmary, "^ %+ ", "");
				priceSummmary = string.gsub(priceSummmary, " %+ $", "");
				
				intro = string.format(SOLD_FOR_PRICE_BY, GFWUtils.Hilite(priceSummmary));
			else
				intro = SOLD_BY;
			end
					
			if (table.getn(vendors) > 1) then
				this:AddLine(intro..":", color.r, color.g, color.b);
				for i, aVendor in pairs(vendors) do
					local vendorLoc = FAS_VendorLocations[aVendor];
					if (not vendorLoc) then
						vendorLoc = FAS_NewVendorLocations[aVendor];
					end
					if (vendorLoc) then
						local vendorName = FAS_Localized[aVendor] or aVendor;
						local vendorLocation = FAS_Localized[vendorLoc] or vendorLoc;
						this:AddLine(FAS_INDENT..string.format(VENDOR_LOCATION_FORMAT, vendorName, vendorLocation), color.r, color.g, color.b);
					else
						local version = GetAddOnMetadata("GFW_AdSpace", "Version");
						GFWUtils.PrintOnce(GFWUtils.Red("AdSpace "..version.." error: ").."Can't find location for "..aVendor..". Please check www.fizzwidget.com for updates.", 60);
						return false;
					end
				end
			elseif (table.getn(vendors) == 1) then
				local vendorLoc = FAS_VendorLocations[vendors[1]];
				if (not vendorLoc) then
					vendorLoc = FAS_NewVendorLocations[vendors[1]];
				end
				if (vendorLoc) then
					local vendorName = FAS_Localized[vendors[1]] or vendors[1];
					local vendorLocation = FAS_Localized[vendorLoc] or vendorLoc;
					this:AddLine(intro.." "..string.format(VENDOR_LOCATION_FORMAT, vendorName, vendorLocation), color.r, color.g, color.b);
				else
					local version = GetAddOnMetadata("GFW_AdSpace", "Version");
					GFWUtils.PrintOnce(GFWUtils.Red("AdSpace "..version.." error: ").."Can't find location for "..vendorName..". Please check www.fizzwidget.com for updates.", 60);
					return false;
				end
			else
				local found = false;
				for _, faction in pairs({"Alliance", "Horde", "Neutral"}) do
					if (FAS_VendorInfo[faction][itemID]) then
						found = true;
					end
				end
				if not (found) then
					local version = GetAddOnMetadata("GFW_AdSpace", "Version");
					GFWUtils.PrintOnce(GFWUtils.Red("AdSpace "..version.." error: ")..link.."("..itemID..") is listed but has no vendors. Please check www.fizzwidget.com for updates.", 60);
					return false;
				end
			end
			if (note ~= "") then
				this:AddLine(note, color.r, color.g, color.b);
			end
			return true;
		end
		
		local libramInfo = FAS_LibramInfo[itemID];
		if (not FAS_Config.NoLibrams and libramInfo) then
			local color = FAS_SPECIAL_VENDOR_COLOR;
			local returnToName = FAS_Localized[libramInfo.name] or libramInfo.name;
			local returnToLocation = FAS_Localized[FAS_VendorLocations[libramInfo.name]] or FAS_VendorLocations[libramInfo.name];
			local bonus = FAS_Localized[libramInfo.bonus] or libramInfo.bonus;
			this:AddLine(RETURN_TO.." "..string.format(VENDOR_LOCATION_FORMAT, returnToName, returnToLocation), color.r, color.g, color.b);
			this:AddLine(string.format(ARCANUM_FORMAT, bonus), color.r, color.g, color.b);
			return true;
		end

		local tokenInfo = FAS_TokenInfo[itemID];
		if ((not FAS_Config.NoZG or not FAS_Config.NoAQ20 or not FAS_Config.NoAQ40) and  tokenInfo) then
			local color = FAS_SPECIAL_VENDOR_COLOR;
			local addedLines;
			for _, faction in pairs(FAS_TokenFactions) do
				local _, _, factionAbbrev = string.find(faction, "(.-)_FACTION");
				if (FAS_Config[factionAbbrev]) then
					local reportLines = {};
					for _, class in pairs(FAS_AllClasses) do
						if (not (class == "SHAMAN" and UnitFactionGroup("player") == "Alliance") and 
						    not (class == "PALADIN" and UnitFactionGroup("player") == "Horde")) then
							for _, rewardID in pairs(tokenInfo) do
								local reward = FAS_TokenRewards[rewardID];
								if (reward and reward.class == class and reward.faction == faction) then
									local repNeeded;
									if (reward.rep) then
										repNeeded = getglobal("FACTION_STANDING_LABEL"..reward.rep);
									end
									local reportLine = "";
									reportLine = reportLine .. reward.type;
									if (repNeeded) then
										reportLine = reportLine .. " ("..repNeeded..")";
									end
									reportLine = reportLine .. ", ";
									if (reportLines[class]) then
										reportLines[class] = reportLines[class] .. reportLine;
									else
										reportLines[class] = reportLine;
									end
								end
							end
							if (reportLines[class]) then
								reportLines[class] = string.gsub(reportLines[class], ", $", "");
							end
						end
					end
					if (GFWTable.Count(reportLines) > 0) then
						this:AddLine(string.format(FAS_FACTION_REWARDS, getglobal(faction)), color.r, color.g, color.b);
						addedLines = true;
						for class, reportLine in pairs(reportLines) do
							this:AddLine("  "..getglobal(class)..": "..reportLine, color.r, color.g, color.b);
						end
					end
				end
			end
			if (addedLines) then
				return true;
			end
		end

		-- FactionFriend also shows tooltips for this stuff; if it's present, let it take over
		if (FFF_Config and not FFF_Config.NoTooltip) then return; end

		local darkmoonInfo = FAS_DarkmoonInfo[itemID];
		if (not FAS_Config.NoDarkmoon and darkmoonInfo) then
			local color = FAS_SPECIAL_VENDOR_COLOR;
			this:AddLine(darkmoonInfo, color.r, color.g, color.b);
			return true;
		end

		for key, tokenSetList in pairs(FAS_FactionTokenSets) do
			if (not FAS_Config["No"..key]) then
				
				for _, coinSet in pairs(tokenSetList) do
					local found;
					local otherCoins = {};
					for _, coinID in pairs(coinSet) do
						if (itemID == coinID) then
							found = true;
						else
							local itemText = FAS_TokenNames[coinID];
							itemText = FAS_Localized[itemText] or itemText;
							table.insert(otherCoins, itemText);
						end
					end
					if (found) then
						local color = FAS_SPECIAL_VENDOR_COLOR;
						this:AddLine(FAS_TURNIN.." "..getglobal(key.."_FACTION"), color.r, color.g, color.b);
						if (table.getn(otherCoins) > 0) then
							this:AddDoubleLine(" ", FAS_WITH.." "..table.concat(otherCoins, ", "), color.r, color.g, color.b, color.r, color.g, color.b);
						end
						return true;
					end
				end
			end
		end
	end
	
end

function FAS_OnLoad()

	-- Register Slash Commands
	SLASH_FAS1 = "/adspace";
	SLASH_FAS2 = "/ads";
	SlashCmdList["FAS"] = function(msg)
		FAS_ChatCommandHandler(msg);
	end
	
	-- Register for Events
	this:RegisterEvent("MERCHANT_SHOW");
	this:RegisterEvent("MERCHANT_UPDATE");

	FAS_HookTooltip(GameTooltip);
	FAS_HookTooltip(ItemRefTooltip);
	
	table.insert(UISpecialFrames,"FAS_OptionsFrame");	
end

function FAS_OnEvent(event, arg1)

	if( event == "MERCHANT_SHOW" or event == "MERCHANT_UPDATE" ) then
		FAS_ScanMerchant();
	end
end

function FAS_ChatCommandHandler(msg)

	if ( msg == "" ) then
		FAS_ShowOptions();
		return;
	end

	-- Print Help
	if ( msg == "help" ) or ( msg == "" ) then
		local version = GetAddOnMetadata("GFW_AdSpace", "Version");
		GFWUtils.Print("Fizzwidget AdSpace "..version..":");
		GFWUtils.Print("/adspace (or /ads)");
		GFWUtils.Print("- "..GFWUtils.Hilite("help").." - Print this helplist.");
		GFWUtils.Print("- "..GFWUtils.Hilite("[item link]").." - Show info for an item in the chat window.");
		GFWUtils.Print("- "..GFWUtils.Hilite("status").." - Check current settings.");
		GFWUtils.Print("- "..GFWUtils.Hilite("recipes on").." | "..GFWUtils.Hilite("off").." - Show info for vendor-supplied recipes in tooltips.");
		GFWUtils.Print("- "..GFWUtils.Hilite("showcost on").." | "..GFWUtils.Hilite("off").." - Also show vendor prices for recipes.");
		GFWUtils.Print("- "..GFWUtils.Hilite("librams on").." | "..GFWUtils.Hilite("off").." - Show info for librams in tooltips.");
		GFWUtils.Print("- "..GFWUtils.Hilite("darkmoon on").." | "..GFWUtils.Hilite("off").." - Show info for Darkmoon Faire grey item turn-ins in tooltips.");
		GFWUtils.Print("- "..GFWUtils.Hilite("zg on").." | "..GFWUtils.Hilite("off").." - Show info for for special raid loot from Zul'Gurub (Zandalar Tribe rewards) in tooltips.");
		GFWUtils.Print("- "..GFWUtils.Hilite("aq20 on").." | "..GFWUtils.Hilite("off").." - Show info for for special raid loot from Ruins of Ahn'Qiraj (Cenarion Circle rewards) in tooltips.");
		GFWUtils.Print("- "..GFWUtils.Hilite("aq40 on").." | "..GFWUtils.Hilite("off").." - Show info for for special raid loot from Ahn'Qiraj (Brood of Nozdormu rewards) in tooltips.");
		GFWUtils.Print("- "..GFWUtils.Hilite("post on").." | "..GFWUtils.Hilite("off").." - Post to raid/party chat when getting raid loot info via "..GFWUtils.Hilite("/ads [item link]")..".");
		return;
	end
	
	if (msg == "version") then
		local version = GetAddOnMetadata("GFW_AdSpace", "Version");
		GFWUtils.Print("Fizzwidget AdSpace "..version);
		return;
	end
	
	if ( msg == "status" ) then
		local gotSomething;
		if (not FAS_Config.NoRecipes) then
			GFWUtils.Print("Showing info for "..GFWUtils.Hilite("vendor-supplied recipes").." in tooltips.");
			gotSomething = 1;
			if (not FAS_Config.NoShowCost) then
				GFWUtils.Print("Also showing vendor price for recipes.");
			end
		end
		if (not FAS_Config.NoLibrams) then
			GFWUtils.Print("Showing info for "..GFWUtils.Hilite("Librams").." in tooltips.");
			gotSomething = 1;
		end
		if (not FAS_Config.NoDarkmoon) then
			GFWUtils.Print("Showing info for "..GFWUtils.Hilite("Darkmoon Faire grey item turn-ins").." in tooltips.");
			gotSomething = 1;
		end
		if (not FAS_Config.NoZG) then
			GFWUtils.Print("Showing info for special raid loot from "..GFWUtils.Hilite("Zul'Gurub (Zandalar Tribe rewards)").." in tooltips.");
			gotSomething = 1;
		end
		if (not FAS_Config.NoAQ20) then
			GFWUtils.Print("Showing info for special raid loot from "..GFWUtils.Hilite("Ruins of Ahn'Qiraj (AQ20 Cenarion Circle rewards)").." in tooltips.");
			gotSomething = 1;
		end
		if (not FAS_Config.NoAQ40) then
			GFWUtils.Print("Showing info for special raid loot from "..GFWUtils.Hilite("Ahn'Qiraj (AQ40 Brood of Nozdormu rewards)").." in tooltips.");
			gotSomething = 1;
		end
		if ((not FAS_Config.NoZG or not FAS_Config.NoAQ20 or not FAS_Config.NoAQ40) and not FAS_Config.NoPostToRaid) then
			GFWUtils.Print("Will post to raid/party chat when showing info for raid loot via "..GFWUtils.Hilite("/ads [link]")..".");
		end
		if (not gotSomething) then
			GFWUtils.Print("Not adding any info to tooltips.");
		end
		return;
	end
	
	if (msg == "test") then
		local itemInfoCount = 0;
		for itemID in pairs(FAS_ItemInfo) do
			local found = false;
			for _, faction in pairs({"Alliance", "Horde", "Neutral"}) do
				if (FAS_VendorInfo[faction][itemID]) then
					found = true;
				end
			end
			if not (found) then
				GFWUtils.Print("Item ID "..itemID.." not found in FAS_VendorInfo.");
			end
			itemInfoCount = itemInfoCount + 1;
		end
		GFWUtils.Print(itemInfoCount.." entries in FAS_ItemInfo.");
		for _, faction in pairs({"Alliance", "Horde", "Neutral"}) do
			local vendorInfoCount = 0;
			for itemID in pairs(FAS_VendorInfo[faction]) do
				if (FAS_ItemInfo[itemID] == nil) then
					GFWUtils.Print("Item ID "..itemID.." not found in FAS_ItemInfo.");
				end
				vendorInfoCount = vendorInfoCount + 1;
			end
			GFWUtils.Print(vendorInfoCount.." entries in FAS_VendorInfo["..faction.."].");
		end
		return;
	end

	local _, _, cmd, args = string.find(msg, "^([%l%d']+) *(.*)");
	if (cmd) then cmd = string.lower(cmd); end
		
	if (args == nil or args == "") then
		args = msg;
	end
	local postedText;
	for itemLink in string.gmatch(args, "|c%x+|Hitem:[-%d:]+|h%[.-%]|h|r") do
		postedText = nil;
		local _, _, itemID  = string.find(itemLink, "item:(%d+)");
		if (itemID == nil or itemID == "") then
			GFWUtils.Print("Usage: "..GFWUtils.Hilite("/ads info <item link>"));
			return;
		end
		itemID = tonumber(itemID);
		
		local itemInfo = FAS_ItemInfo[itemID];
		if (itemInfo) then
			local myFaction = UnitFactionGroup("player");
			local vendors = GFWTable.Merge(FAS_VendorInfo[myFaction][itemID], FAS_VendorInfo["Neutral"][itemID]);
			if (FAS_NewVendorInfo[myFaction] and FAS_NewVendorInfo[myFaction][itemID]) then
				vendors = GFWTable.Merge(vendors, FAS_NewVendorInfo[myFaction][itemID]);
			end
			if (FAS_NewVendorInfo["Neutral"] and FAS_NewVendorInfo["Neutral"][itemID]) then
				vendors = GFWTable.Merge(vendors, FAS_NewVendorInfo["Neutral"][itemID]);
			end
			local note = itemInfo.note;
			local intro = itemLink..": "..string.format(SOLD_FOR_PRICE_BY, GFWUtils.TextGSC(itemInfo.b));

			if (vendors == nil or vendors == {}) then
				local version = GetAddOnMetadata("GFW_AdSpace", "Version");
				GFWUtils.Print(GFWUtils.Red("AdSpace "..version.." error: ")..itemLink.."("..itemID..") is listed but has no vendors. Please check www.fizzwidget.com for updates.");
				return;
			end
			GFWUtils.Print(intro);
			for i, aVendor in pairs(vendors) do
				local vendorName = FAS_Localized[aVendor] or aVendor;
				local vendorLocation = FAS_Localized[FAS_VendorLocations[aVendor]] or FAS_VendorLocations[aVendor];
				if (not vendorLocation) then
					vendorLocation = FAS_Localized[FAS_NewVendorLocations[aVendor]] or FAS_NewVendorLocations[aVendor];
				end
				GFWUtils.Print(string.format(VENDOR_LOCATION_FORMAT, vendorName, vendorLocation));
			end
			if (note and note ~= "") then
				GFWUtils.Print(GFWUtils.Hilite(note));
			end
			postedText = 1;
		end
		
		local libramInfo = FAS_LibramInfo[itemID];
		if (libramInfo) then
			local returnToName = FAS_Localized[libramInfo.name] or libramInfo.name;
			local returnToLocation = FAS_Localized[FAS_VendorLocations[libramInfo.name]] or FAS_VendorLocations[libramInfo.name];
			local bonus = FAS_Localized[libramInfo.bonus] or libramInfo.bonus;
			GFWUtils.Print(itemLink..": "..RETURN_TO.." "..string.format(VENDOR_LOCATION_FORMAT, returnToName, returnToLocation));
			GFWUtils.Print(bonus);
			postedText = 1;
		end
		
		local darkmoonInfo = FAS_DarkmoonInfo[itemID];
		if (darkmoonInfo) then
			GFWUtils.Print(itemLink..": "..darkmoonInfo);
			postedText = 1;
		end

		local tokenInfo = FAS_TokenInfo[itemID];
		if (tokenInfo) then
			for _, faction in pairs(FAS_TokenFactions) do
				local reportLines = {};
				for _, class in pairs(FAS_AllClasses) do
					for _, rewardID in pairs(tokenInfo) do
						local reward = FAS_TokenRewards[rewardID];
						if (reward and reward.class == class and reward.faction == faction) then
							local link = GFWUtils.ItemLink(rewardID);
							local repNeeded;
							if (reward.rep) then
								repNeeded = getglobal("FACTION_STANDING_LABEL"..reward.rep);
							end
							local reportLine = "";
							if (link) then
								reportLine = reportLine .. link .. " - ";
							end
							reportLine = reportLine .. reward.type;
							if (repNeeded) then
								reportLine = reportLine .. " ("..repNeeded..")";
							end
							reportLine = reportLine .. ", ";
							if (reportLines[class]) then
								reportLines[class] = reportLines[class] .. reportLine;
							else
								reportLines[class] = reportLine;
							end
						end
					end
					if (reportLines[class]) then
						reportLines[class] = string.gsub(reportLines[class], ", $", "");
					end
				end
				if (GFWTable.Count(reportLines) > 0) then
					local _, _, factionAbbrev = string.find(faction, "(.-)_FACTION");
					postedText = 1;
					if (FAS_Config[factionAbbrev]) then
						FAS_Post(itemLink..": "..string.format(FAS_FACTION_REWARDS, getglobal(faction)));
						for class, reportLine in pairs(reportLines) do
							FAS_Post("   "..getglobal(class)..": "..reportLine);
						end
					else
						FAS_Post(itemLink..": "..string.format(FAS_FACTION_REWARDS_COUNT, GFWTable.Count(reportLines), getglobal(faction)));
					end
				end
			end
		end

		local rewardInfo = FAS_TokenRewards[itemID];
		if (rewardInfo) then
			local link = GFWUtils.ItemLink(itemID);
			FAS_Post(link..": "..string.format(ITEM_REQ_REPUTATION, getglobal(rewardInfo.faction), getglobal("FACTION_STANDING_LABEL"..rewardInfo.rep)));
			local reportLines = {};
			for tokenID, rewards in pairs(FAS_TokenInfo) do
				if (GFWTable.KeyOf(rewards, itemID)) then
					local itemText = GFWUtils.ItemLink(tokenID);
					local itemQuality = FAS_TokenQuality[tokenID];
					if (itemText == nil) then
						itemText = FAS_TokenNames[tokenID];
						itemText = FAS_Localized[itemText] or itemText;
						local _, _, _, color = GetItemQualityColor(math.floor(itemQuality));
						itemText = color..itemText..FONT_COLOR_CODE_CLOSE;
					end
					if (rewardInfo == ENSCRIBE) then
						-- ZG enchants take 1 each of any reagent
						itemText = "1 x "..itemText;
					else
						-- other token quests take 1 epic only, or 5 of one green + 5 another green + 2 blue + 1 "special"
						if (itemQuality == 2) then
							itemText = "5 x "..itemText;
						elseif (itemQuality == 3) then
							itemText = "2 x "..itemText;					
						else
							itemText = "1 x "..itemText;
						end
					end
					table.insert(reportLines, itemText);
				end
			end
			table.sort(reportLines);
			for _, line in pairs(reportLines) do
				FAS_Post(line);
			end	
			postedText = 1;
		end
		
		if (not postedText) then
			GFWUtils.Print("Nothing known about "..itemLink..".");
		end
	end
	if (postedText) then
		return;
	end
	-- if we made it down here, there were args we didn't understand... time to remind the user what to do.
	FAS_ChatCommandHandler("help");

end

function FAS_StripColor(text)
	if (string.find(text, "|c%x+|Hitem:[-%d:]+|h%[.-%]|h|r")) then
		return text;
	else
		return string.gsub(text, "|c"..string.rep("%x", 8).."(.-)|r", "%1");
	end
end

function FAS_Post(msg)
	if (not FAS_Config.NoPostToRaid and GetNumRaidMembers() > 0) then
		msg = FAS_StripColor(msg);
		SendChatMessage(msg, "RAID");	
	elseif (not FAS_Config.NoPostToRaid and GetNumPartyMembers() > 0) then
		msg = FAS_StripColor(msg);
		SendChatMessage(msg, "PARTY");	
	else
		GFWUtils.Print(msg);
	end
end

function FAS_CheckMerchant(itemID)
	for merchantIndex = 1, GetMerchantNumItems() do
		local link = GetMerchantItemLink(merchantIndex);
		local _, _, merchantItemID  = string.find(link, "item:(%d+)");
		if (tonumber(merchantItemID) == itemID) then
			return true;
		end
	end
	return false;
end

function FAS_ScanMerchant()
	for index = 1, GetMerchantNumItems() do
		local link = GetMerchantItemLink(index);
		if ( link ) then
			local _, _, _, _, _, itemType, subType, _, _, _ = GetItemInfo(link);
			if (itemType == FAS_RECIPE and subType ~= FAS_BOOK) then
				local itemID = GFWUtils.DecomposeItemLink(link);
				local added, price, honorPoints, arenaPoints;
				_, _, price, _, _, _, extendedCost = GetMerchantItemInfo(index);
				local priceSummmary;
				if (price and price > 0) then
				 	priceSummmary = GFWUtils.TextGSC(price);
				else
					priceSummmary = "";
				end
				local itemSummary = "";
				if (price == 0 and extendedCost) then
					honorPoints, arenaPoints, itemCount = GetMerchantItemCostInfo(index);
					if (honorPoints > 0) then
						priceSummmary = priceSummmary .. " + ".. honorPoints.. " honor";
					else
						honorPoints = nil;
					end
					if (arenaPoints > 0) then
						priceSummmary = priceSummmary .. " + ".. arenaPoints.. " arena";
					else
						arenaPoints = nil;
					end
					if (itemCount > 0) then
						for costIndex = 1, itemCount, 1 do
							local itemTexture, itemValue = GetMerchantItemCostItem(index, costIndex);
							FASHiddenTooltip:ClearLines();
							FASHiddenTooltip:SetOwner(UIParent, "ANCHOR_NONE");
							FASHiddenTooltip:Show();
							FASHiddenTooltip:SetMerchantCostItem(index, costIndex);
							local itemName = FASHiddenTooltipTextLeft1:GetText();
							if (not itemName) then
								itemName = "Unknown Item";
							end
							itemSummary = itemSummary .. string.format(" + %d %s", itemValue, itemName);
						end
					end
					priceSummmary = priceSummmary .. itemSummary;
					itemSummary = string.gsub(itemSummary, "^ %+ ", "");
					priceSummmary = string.gsub(priceSummmary, "^ %+ ", "");
					priceSummmary = string.gsub(priceSummmary, " %+ $", "");
					if (itemSummary == "") then
						itemSummary = nil;
					end
				end
				if (not FAS_ItemInfo[itemID]) then
					local requiresFaction;
					if (FAS_ItemRequiresFaction(link)) then
						requiresFaction = REQ_FACTION;
					end
					FAS_NewItemInfo[itemID] = { b=price, h=honorPoints, a=arenaPoints, i=itemSummary, note=requiresFaction };
					added = true;
				end
				local vendorName = UnitName("npc");
				local vendorFaction = UnitFactionGroup("npc");
				if (vendorFaction ~= UnitFactionGroup("player")) then
					vendorFaction = "Neutral";
				end
				if (FAS_VendorInfo[vendorFaction][itemID] == nil or GFWTable.KeyOf(FAS_VendorInfo[vendorFaction][itemID], vendorName) == nil) then
					if (FAS_NewVendorInfo[vendorFaction] == nil) then
						FAS_NewVendorInfo[vendorFaction] = {};
					end
					if (FAS_NewVendorInfo[vendorFaction][itemID] == nil) then
						FAS_NewVendorInfo[vendorFaction][itemID] = {};
					end
					if (GFWTable.KeyOf(FAS_NewVendorInfo[vendorFaction][itemID], vendorName) == nil) then
						table.insert(FAS_NewVendorInfo[vendorFaction][itemID], vendorName);
						added = true;
					end
				end
				if (not FAS_VendorLocations[vendorName] and not FAS_NewVendorLocations[vendorName]) then
					FAS_NewVendorLocations[vendorName] = GetRealZoneText();
					added = true;
				end
				if (added) then
					local version = GetAddOnMetadata("GFW_AdSpace", "Version");
					GFWUtils.PrintOnce(string.format("Fizzwidget AdSpace v.%s found new vendor recipe: %s sold for %s by %s (%s) in %s. Please check www.fizzwidget.com for updates.", version, link, GFWUtils.Hilite(priceSummmary), GFWUtils.Hilite(vendorName), vendorFaction, GFWUtils.Hilite(GetRealZoneText())));
				end
			end
		end
	end
end

function FAS_ItemRequiresFaction(link)
	FASHiddenTooltip:ClearLines();
	FASHiddenTooltip:SetHyperlink(link);
	for lineNum = 1, FASHiddenTooltip:NumLines() do
		local leftText = getglobal("FASHiddenTooltipTextLeft"..lineNum):GetText();
		if (FAS_ITEM_REQ_REPUTATION == nil) then
			FAS_ITEM_REQ_REPUTATION = GFWUtils.FormatToPattern(ITEM_REQ_REPUTATION);
		end
		local _, _, faction, rep = string.find(leftText, FAS_ITEM_REQ_REPUTATION);
		if (faction and rep) then
			return true;
		end
	end
end

-- private, for building localization tables
function FAS_Translate(langCode)

	tempTranslations = {};
	
	local localizedVendorInfo = getglobal("FAS_VendorInfo_"..langCode);
	local localizedVendorLocations = getglobal("FAS_VendorLocations_"..langCode);
	for faction, factionVendorList in pairs(FAS_VendorInfo) do
		for itemID, vendorList in pairs(factionVendorList) do
			local localizedVendors = localizedVendorInfo[faction][itemID];
			if (localizedVendors and type(localizedVendors) == "table") then
				for index, name in pairs(vendorList) do
					local localizedName = localizedVendors[index];
					
					if (localizedName == nil) then break; end
					
					if (localizedName ~= name) then
						if (tempTranslations[name] == nil) then
							tempTranslations[name] = {};
						end
						table.insert(tempTranslations[name], localizedName);
					end
					
					local location = FAS_VendorLocations[name];
					local localizedLocation = localizedVendorLocations[localizedName];
					if (localizedLocation and localizedLocation ~= location) then
						if (tempTranslations[location] == nil) then
							tempTranslations[location] = {};
						end
						table.insert(tempTranslations[location], localizedLocation);
					end
				end
			end
		end
	end
	
	local localizedLibramInfo = getglobal("FAS_LibramInfo_"..langCode);
	for itemID, libramInfo in pairs(FAS_LibramInfo) do
		local localizedInfo = localizedLibramInfo[itemID];
		if (localizedInfo and type(localizedInfo) == "table") then
					
			if (localizedInfo.name ~= libramInfo.name) then
				if (tempTranslations[libramInfo.name] == nil) then
					tempTranslations[libramInfo.name] = {};
				end
				table.insert(tempTranslations[libramInfo.name], localizedInfo.name);
			end
			if (localizedInfo.bonus ~= libramInfo.bonus) then
				if (tempTranslations[libramInfo.bonus] == nil) then
					tempTranslations[libramInfo.bonus] = {};
				end
				table.insert(tempTranslations[libramInfo.bonus], localizedInfo.bonus);
			end
			
			local location = FAS_VendorLocations[libramInfo.name];
			local localizedLocation = localizedVendorLocations[localizedInfo.name];
			if (localizedLocation and localizedLocation ~= location) then
				if (tempTranslations[location] == nil) then
					tempTranslations[location] = {};
				end
				table.insert(tempTranslations[location], localizedLocation);
			end
		end
	end
	
	FAS_Config[langCode] = {};
	for baseString, translations in pairs(tempTranslations) do
		if (table.getn(translations) == 1) then
			FAS_Config[langCode][baseString] = translations[1];
		else
			local mergedTranslations = {}
			for _, translation in pairs(translations) do
				if (GFWTable.KeyOf(mergedTranslations, translation) == nil) then
					table.insert(mergedTranslations, translation);
				end
			end
			if (table.getn(mergedTranslations) == 1) then
				FAS_Config[langCode][baseString] = mergedTranslations[1];
			else
				FAS_Config[langCode][baseString] = mergedTranslations;
			end
		end
	end

end

-- private, for generating base tables from auto-gathered data
function FAS_GetNames()
	for itemID, itemInfo in pairs(FAS_NewItemInfo) do
		local name = GetItemInfo(itemID);
		itemInfo.n = name;
	end
	for faction, vendorList in pairs(FAS_NewVendorInfo) do
		for itemID, vendors in pairs(vendorList) do
			if (FAS_VendorInfo[faction][itemID]) then
				for _, vendor in pairs(FAS_VendorInfo[faction][itemID]) do
					if (not GFWTable.KeyOf(vendors, vendor)) then
						table.insert(vendors, vendor);
					end
				end
			end
			local name = GetItemInfo(itemID);
			vendors.n = name;
		end
	end
end

------------------------------------------------------
-- Dongle & GFWOptions stuff
------------------------------------------------------

GFW_AdSpace = {};
local GFWOptions = DongleStub("GFWOptions-1.0");

local function buildOptionsUI(panel)

	GFW_AdSpace.optionsText = {
		Recipes = FAS_OPTION_RECIPES,
		ShowCost = FAS_OPTION_RECIPE_COST,
		Librams = FAS_OPTION_LIBRAM,
		Darkmoon = FAS_OPTION_DARKMOON,
		AD = FAS_OPTION_AD,
		ZG = FAS_OPTION_ZG.." "..GFWUtils.Gray(FAS_OPTION_ZG_FACTION),
		AQ20 = FAS_OPTION_AQ20.." "..GFWUtils.Gray(FAS_OPTION_AQ20_FACTION),
		AQ40 = FAS_OPTION_AQ40.." "..GFWUtils.Gray(FAS_OPTION_AQ40_FACTION),
		PostToRaid = FAS_OPTION_POST_RAID,
	};
	
	local s, widget, lastWidget;
	s = panel:CreateFontString("FAS_OptionsPanel_GeneralHeader", "ARTWORK", "GameFontNormal");
	s:SetPoint("TOPLEFT", panel.contentAnchor, "BOTTOMLEFT", 0, -16);
	s:SetText(FAS_OPTIONS_GENERAL);
	lastWidget = s;

	widget = panel:CreateCheckButton("Recipes", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", -2, -2);
	lastWidget = widget;
	
	widget = panel:CreateCheckButton("ShowCost", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 16, -2);
	lastWidget.dependentControls = { widget };
	lastWidget = widget;

	widget = panel:CreateCheckButton("Librams", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", -16, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("Darkmoon", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("AD", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;
	
	s = panel:CreateFontString("FAS_OptionsPanel_RaidHeader", "ARTWORK", "GameFontNormal");
	s:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 2, -16);
	s:SetText(FAS_OPTIONS_RAID);
	lastWidget = s;

	widget = panel:CreateCheckButton("ZG", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", -2, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("AQ20", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("AQ40", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);
	lastWidget = widget;

	widget = panel:CreateCheckButton("PostToRaid", true);
	widget:SetPoint("TOPLEFT", lastWidget, "BOTTOMLEFT", 0, -2);

end

function FAS_ShowOptions()
	InterfaceOptionsFrame_OpenToFrame(FAS_OptionsPanel);
end

function GFW_AdSpace:Initialize()
	self.defaults = { 
		profile = {
			NoRecipes = false,
			NoShowCost = false,
			NoLibrams = false,
			NoDarkmoon = false,
			NoAD = false,
			NoZG = false,
			NoAQ20 = false,
			NoAQ40 = false,
			NoPostToRaid = false,
		}
	};
	self.db = self:InitializeDB("GFW_AdSpaceDB", self.defaults);
	FAS_Config = self.db.profile;
end

function GFW_AdSpace:Enable()
	-- conditionalize 2.4 stuff for now so we can run on 2.3
	if (InterfaceOptions_AddCategory) then
		GFWOptions:CreateMainPanel("GFW_AdSpace", "FAS_OptionsPanel");
		FAS_OptionsPanel.BuildUI = buildOptionsUI;
	end
end

GFW_AdSpace = DongleStub("Dongle-1.2"):New("GFW_AdSpace", GFW_AdSpace);

