--------------------------------------------------------------------------------
--
-- dbCraft by Mike Summers (Rhaithe - Order of Exiles - Proudmoore - Alliance)
--
-- In-Game Database of Guild Crafting Skills
--
-- Based on iGuildTradeSkill v1.1 by HPi (Woopadigo / Defias Brotherhood)
-- Original version: GuildTradeSkill by Orlan (Darkmoon Faire))
--
-- Dependancies: Chronos
--
--------------------------------------------------------------------------------
--
-- Variables
--
--------------------------------------------------------------------------------

DBCRAFT_VERSION = GetAddOnMetadata("dbCraft", "Version");
local VerHigh, VerLow, VerBeta = strsplit(".", DBCRAFT_VERSION);

local dbCraftDefaultOptions = {
  ["VersionString"] = DBCRAFT_VERSION;
  ["Version"] =            { High = tonumber(VerHigh), Low = tonumber(VerLow), Beta = tonumber(VerBeta) };
  ["HighestVersionSeen"] = { High = tonumber(VerHigh), Low = tonumber(VerLow), Beta = tonumber(VerBeta) };
  ["RealmName"] = GetRealmName();
  ["ButtonShown"] = true;
  ["ButtonPosition"] = 15;
  ["ButtonRadius"] = 78;
  ["RemoteSay"] = true;
  ["RemoteGuild"] = true;
  ["RemoteEmote"] = true;
  ["RemoteDoEmote"] = true;
  ["HideMessages"] = true;
  ["Reminder"] = 7;
  ["Locked"] = false;
  ["Clamped"] = true;
  ["Alpha"] = 1.0;
  ["Scale"] = 1.0;
};

textColor = {
  ["AddonName"]       = "|cff8080ff";
  ["Version"]         = "|cffffffff";
  ["NumberOfRecords"] = "|cffffffff";
  ["NumberOfSkills"]  = "|cffffffff";
  ["CharacterName"]   = "|cffffffff";
  ["SenderName"]      = "|cffffffff";
  ["ProfessionName"]  = "|cffffffff";
  ["GeneralText"]     = "|cff808080";
  ["Highlight"]       = "|cffffffff";
};

dbCraft = {
  BufferRecords = nil;
  BufferPosition = nil;
  SearchString = "";
  FooTradeSkill = "";
  SearchTimer = 0;
  TLink = nil;
--  ScrollPosition = 0;
--  ScrollDistance = 100;
--  ScrollTop = 0;
--  ScrollBottom = 100;
};

dbCraftOptions = dbCraftOptions or {};
dbCraftDatabase = dbCraftDatabase or {};
dbCraftCharacterData = dbCraftCharacterData or {};

--------------------------------------------------------------------------------
--
-- Event Processing
--
--------------------------------------------------------------------------------

function dbCraft_OnLoad()
  this:RegisterEvent("ADDON_LOADED");
  this:RegisterEvent("VARIABLES_LOADED");
--  this:RegisterEvent("SPELLS_CHANGED");
--  this:RegisterEvent("PLAYER_LOGIN");
--  this:RegisterEvent("PLAYER_ENTERING_WORLD");
--  this:RegisterEvent("PLAYER_ALIVE");
  this:RegisterEvent("CHAT_MSG_ADDON");


  --Allows dbCraft to be closed with the Escape key
  tinsert(UISpecialFrames, "dbCraftFrame");

  dbCraftFrame:RegisterForDrag("LeftButton");
end

function dbCraft_OnShow()
  dbCraft_ShowHeader();
  dbCraft_ViewTradeSkills(nil , true);
end

function dbCraft_ShowHeader()
    local RealmName = GetRealmName();
    local GuildName = GetGuildInfo("player");

    if GuildName == nil then
      GuildName = "Unguilded Alts";
    end

    local text = "<html><body>";
    text = text .. '<p align="center">Crafting Database for |cffffff00' .. GuildName .. '|r (|cff00ff00' .. RealmName .. '|r)</p>';

    if  (dbCraftOptions.HighestVersionSeen.High >  dbCraftOptions.Version.High)
    or ((dbCraftOptions.HighestVersionSeen.High == dbCraftOptions.Version.High)
    and (dbCraftOptions.HighestVersionSeen.Low  >  dbCraftOptions.Version.Low)) then
      text = text .. '<p align="center">|cffff0000New Version Detected (v';
      text = text .. dbCraftOptions.HighestVersionSeen.High .. ".";
      text = text .. dbCraftOptions.HighestVersionSeen.Low .. ".";
      text = text .. dbCraftOptions.HighestVersionSeen.Beta .. ")|r</p>";
      text = text .. '<p align="center">You can download the new version at |cff8080ffhttp://wow.curse.com/downloads/details/10093/|r</p>';
    end
    text = text .. "</body></html>";
    dbCraftInfo:SetText(text);
end

function dbCraft_OnEvent()
  if (event == "ADDON_LOADED" and arg1 == "dbCraft") then
    dbCraft_Init();
    dbCraft_HandleLoaded();
  elseif (event == "VARIABLES_LOADED") then
    dbCraft_UpdateCrafterList();

    -- Clear Character Data from Guild Database that was put there in the v2.0.0 database change
    dbCraftDatabase["Alchemy"] = nil;
    dbCraftDatabase["Blacksmithing"] = nil;
    dbCraftDatabase["Cooking"] = nil;
    dbCraftDatabase["Enchanting"] = nil;
    dbCraftDatabase["Engineering"] = nil;
    dbCraftDatabase["First Aid"] = nil;
    dbCraftDatabase["Jewelcrafting"] = nil;
    dbCraftDatabase["Jewelrycrafting"] = nil;
    dbCraftDatabase["Leatherworking"] = nil;
    dbCraftDatabase["Smelting"] = nil;
    dbCraftDatabase["Tailoring"] = nil;

    -- Clear old data from before dbCraftDatabase[RealmName][GuildName][TradeSkillKey] structure was adopted in v2.0.0
    dbCraftDatabase["GuildAlchemy"] = nil;
    dbCraftDatabase["GuildBlacksmithing"] = nil;
    dbCraftDatabase["GuildCooking"] = nil;
    dbCraftDatabase["GuildEnchanting"] = nil;
    dbCraftDatabase["GuildEngineering"] = nil;
    dbCraftDatabase["GuildFirst Aid"] = nil;
    dbCraftDatabase["GuildJewelcrafting"] = nil;
    dbCraftDatabase["GuildJewelrycrafting"] = nil;
    dbCraftDatabase["GuildLeatherworking"] =nil;
    dbCraftDatabase["GuildSmelting"] = nil;
    dbCraftDatabase["GuildTailoring"] = nil;
    dbCraftDatabase["lastscan"] = nil;
    dbCraftDatabase["lastsend"] = nil;

--  elseif (event == "SPELLS_CHANGED") then
--  elseif (event == "PLAYER_LOGIN") then
--  elseif (event == "PLAYER_ENTERING_WORLD") then
--  elseif (event == "PLAYER_ALIVE") then
  elseif (event == "CHAT_MSG_ADDON") and (arg1 == "dbCraft") and (arg3 == "GUILD") then
    dbCraft_HandleRecord(arg2, arg4);
  end
end

--------------------------------------------------------------------------------
--
-- Slash Commands
--
--------------------------------------------------------------------------------

function dbCraft_SlashCommand(msg)
  if msg == "" then
    dbCraft_Help();
    return;
  end

  local command, arg = dbCraft_ParseSlashCommand(msg);
  command = strlower(command); arg = strlower(arg);

  if command == "request" or command == "r" then
    if arg == "" then
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - A data request requires a recipient.|r");
    else
      dbCraft_SendDataRequest(arg);
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - Sending data request to " .. strlower(arg) .. ".|r");
    end
  elseif command == "say" or command == "s" then
    if arg == "" then
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a recipient.|r");
    else
      local recipient, text = dbCraft_ParseSlashCommand(arg);
      recipient = strlower(recipient); text = strlower(text);
      if arg == "" then
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a message.|r");
      else
        dbCraft_SendChatRequest("say", strlower(recipient), text);
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - Sending chat request to " .. strlower(recipient) .. ".|r");
      end
    end
  elseif command == "guild" or command == "g" then
    if arg == "" then
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a recipient.|r");
    else
      local recipient, text = dbCraft_ParseSlashCommand(arg);
      recipient = strlower(recipient); text = strlower(text);
      if arg == "" then
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a message.|r");
      else
        dbCraft_SendChatRequest("guild", strlower(recipient), text);
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - Sending chat request to " .. strlower(recipient) .. ".|r");
      end
    end
  elseif command == "emote" or command == "e" then
    if arg == "" then
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a recipient.|r");
    else
      local recipient, text = dbCraft_ParseSlashCommand(arg);
      recipient = strlower(recipient); text = strlower(text);
      if arg == "" then
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a message.|r");
      else
        dbCraft_SendChatRequest("emote", strlower(recipient), text);
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - Sending chat request to " .. strlower(recipient) .. ".|r");
      end
    end
  elseif command == "doemote" or command == "d" then
    if arg == "" then
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a recipient.|r");
    else
      local recipient, text = dbCraft_ParseSlashCommand(arg);
      recipient = strlower(recipient); text = strlower(text);
      if arg == "" then
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - A chat request requires a message.|r");
      else
        dbCraft_SendChatRequest("doemote", strlower(recipient), text);
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - Sending chat request to " .. strlower(recipient) .. ".|r");
      end
    end
  elseif command == "remotesay" or command == "remoteguild" or command == "remoteemote" or command == "remotedoemote" then
    local cmd;
    if command == "remotesay" then
      cmd = "RemoteSay";
    elseif command == "remoteguild" then
      cmd = "RemoteGuild";
    elseif command == "remoteemote" then
      cmd = "RemoteEmote";
    elseif command == "remotedoemote" then
      cmd = "RemoteDoEmote";
    end
    if ((arg ~= "on") and (arg ~= "off")) then
      dbCraft_Print("|cff8080ffdbCraft|cff808080 - Usage: /dbCraft " .. cmd .. " on | off.|r");
    else
      if arg == "off" then
        dbCraftOptions[cmd] = false;
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - " .. cmd .. " disabled.|r");
      elseif arg == "on" then
        dbCraftOptions[cmd] = true;
        dbCraft_Print("|cff8080ffdbCraft|cff808080 - " .. cmd .. " enabled.|r");
      end
    end
  elseif command == "settings" then
    dbCraft_Print("|cff8080ffdbCraft|cffffffff Settings:|r");
    dbCraft_DisplaySetting("RemoteSay");
    dbCraft_DisplaySetting("RemoteGuild");
    dbCraft_DisplaySetting("RemoteEmote");
    dbCraft_DisplaySetting("RemoteDoEmote");
  else
    dbCraft_Help();
  end
end

function dbCraft_ParseSlashCommand(msg)
  if msg then
    local a, b, c = strfind(msg, "(%S+)");
    if a then
      return c, strsub(msg, b + 2);
    else
      return "";
    end
  end
end

function dbCraft_DisplaySetting(setting)
  local text = "|cff8080ffdbCraft|cff808080 " .. setting;
  if dbCraftOptions[setting] == true then
    text = text .. " : |cffffffffon";
  else
    text = text .. " : |cffffffffoff";
  end
  dbCraft_Print(text .. ".|r");
end

function dbCraft_Help()
  dbCraft_Print(textColor.AddonName .. "dbCraft " .. textColor.Version .. "v" .. DBCRAFT_VERSION .. textColor.GeneralText .. " - In-Game Guild Crafting Database.|r");
  if dbCraftFrame:IsVisible() then
    dbCraftFrame:Hide();
  else
    dbCraftFrame:Show();
  end
end

--[[
function dbCraft_testParser()
  local arg = {};

  arg = dbCraft_Parser(msg);
  for i = 1, 6 do
    if arg[i] ~= nil then
      dbCraft_Print("arg[" .. i .. "] = " .. arg[i]);
    end
  end
end

function dbCrafter_Parser(msg)
  local arg = {};
  local ctr = 0;

  while msg do
    ptr = strfind(msg, " ");
dbCraft_Print("msg = " .. msg .. ", ptr = " .. ptr);
    if ptr == 1 then
      msg = strsub(msg, 2);
    elseif ptr > 1 then
      ctr = ctr + 1;
      arg[ctr] = strsub(msg, 1, ptr - 1);
      msg = strsub(msg, ptr + 1);
    end
  end
  return arg;
end
]]

--------------------------------------------------------------------------------
--
-- Interface Control
--
--------------------------------------------------------------------------------

function dbCraft_Init()
  if ( dbCraftOptions == nil or dbCraftOptions["VersionString"] ~= DBCRAFT_VERSION) then
    dbCraft_FreshOptions();
  end

  dbCraft_UpdateLock();
  dbCraft_UpdateAlpha();
  dbCraftFrame:SetClampedToScreen(dbCraftOptions.Clamped);
  dbCraftOptions_Init();

  SlashCmdList["DBCRAFT"] = dbCraft_SlashCommand;
  SLASH_DBCRAFT1 = "/dbCraft";
  SLASH_DBCRAFT2 = "/dbc";
end

function dbCraft_SearchAndRefresh(text)
    dbCraft.SearchString = text;
    dbCraft_HandleSearchString();
end

function dbCraft_HandleSearchString()
  dbCraft_ViewTradeSkills(dbCraft.FooTradeSkill);
end

function dbCraft_FreshOptions()
  dbCraftOptions = dbCraft_CloneTable(dbCraftDefaultOptions);
end

function dbCraft_ToggleLock()
  if(dbCraftOptions.Locked) then
    dbCraftOptions.Locked = false;
    dbCraft_UpdateLock();
  else
    dbCraftOptions.Locked = true;
    dbCraft_UpdateLock();
  end
end

function dbCraft_UpdateLock()
  if(dbCraftOptions.Locked) then
    dbCraftLockNorm:SetTexture("Interface\\AddOns\\dbCraft\\Images\\LockButton-Locked-Up");
    dbCraftLockPush:SetTexture("Interface\\AddOns\\dbCraft\\Images\\LockButton-Locked-Down");
  else
    dbCraftLockNorm:SetTexture("Interface\\AddOns\\dbCraft\\Images\\LockButton-Unlocked-Up");
    dbCraftLockPush:SetTexture("Interface\\AddOns\\dbCraft\\Images\\LockButton-Unlocked-Down");
  end
end

function dbCraft_StartMoving()
  if(not dbCraftOptions.Locked) then
    dbCraftFrame:StartMoving();
  end
end

function dbCraft_UpdateAlpha()
  dbCraftFrame:SetAlpha(dbCraftOptions.Alpha);
end

function dbCraft_UpdateScale()
  dbCraftFrame:SetScale(dbCraftOptions.Scale);
end

function dbCraft_Toggle()
  if(dbCraftFrame:IsVisible()) then
    HideUIPanel(dbCraftFrame);
  else
    ShowUIPanel(dbCraftFrame);
  end
end

function dbCraft_myScroll()
  -- not used atm
  if arg1 == 1 then
    if dbCraft.ScrollPosition - dbCraft.ScrollDistance < dbCraft.ScrollTop then
      dbCraft.ScrollPosition = dbCraft.ScrollTop;
    else
      dbCraft.ScrollPosition = dbCraft.ScrollPosition - dbCraft.ScrollDistance;
    end
  else
    if dbCraft.ScrollPosition + dbCraft.ScrollDistance > dbCraft.ScrollBottom then
      dbCraft.ScrollPosition = dbCraft.ScrollBottom;
    else
      dbCraft.ScrollPosition = dbCraft.ScrollPosition + dbCraft.ScrollDistance;
    end
  end
  dbCraftViewScrollFrame:SetVerticalScroll(dbCraft.ScrollPosition);
end

--------------------------------------------------------------------------------
--
-- General Functions
--
--------------------------------------------------------------------------------

function dbCraft_DisplayText(text)
  dbCraftViewEditBox:SetText(text);
  dbCraft_UpdateScroll();
end

function dbCraft_UpdateScroll()
--  dbCraft.ScrollPosition = 0;
  dbCraftViewScrollFrame:SetVerticalScroll(0);
  dbCraftViewScrollFrame:UpdateScrollChildRect();
--  dbCraft.ScrollBottom = dbCraftViewScrollFrame:GetVerticalScrollRange();

--  dbCraft_Print("bottom = " .. dbCraft.ScrollBottom);
end

function dbCraft_mySort(table)
  local ctr = 0, link, name, key, k;
  local t = {}; n = {}; temp = {};

  for key in pairs(table) do
    ctr = ctr + 1;
    tinsert(t, ctr, key);
  end

  repeat
    k = 0;
    for i = 1, getn(t) - 1 do
      if t[i] > t[i+1] then
        temp = t[i]; t[i] = t[i+1]; t[i+1] = temp;
        k = 1;
      end
    end
  until (k == 0)
  return t;
end

function dbCraft_myLinkSort(table)
  local ctr = 0, link, name, key, k;
  local t = {}; n = {}; temp = {};

  for link in pairs(table) do
    ctr = ctr + 1;
    name = string.gsub(tostring(link),"^.-%[(.*)%].*", "%1");
    tinsert(t, ctr, link);
    tinsert(n, ctr, name);
  end

  repeat
    k = 0;
    for i = 1, getn(n) - 1 do
      if n[i] > n[i+1] then
        temp = n[i]; n[i] = n[i+1]; n[i+1] = temp;
        temp = t[i]; t[i] = t[i+1]; t[i+1] = temp;
        k = 1;
      end
    end
  until (k == 0)
  return t;
end

function dbCraft_myFullLinkSort(table, category, level)
  local link, name, k;
  local t = {}; n = {}; temp = {};

  for i, link in ipairs(table) do
    name = string.gsub(tostring(link),"^.-%[(.*)%].*", "%1");
    tinsert(t, i, link);
    tinsert(n, i, name);
  end

  repeat
    k = 0;
    for i = 1, getn(n) - 1 do
      if n[i] > n[i+1] then
        temp = n[i]; n[i] = n[i+1]; n[i+1] = temp;
        temp = t[i]; t[i] = t[i+1]; t[i+1] = temp;
        temp = category[i]; category[i] = category[i+1]; category[i+1] = temp;
        temp = level[i]; level[i] = level[i+1]; level[i+1] = temp;
        k = 1;
      end
    end
  until (k == 0)
  return t, category, level;
end

function dbCraft_GetTableSize(aTable)
  local size = 0;

  for key in pairs(aTable) do
    size = size + 1;
  end

  return size;
end

function dbCraft_CloneTable(t)
  local new = {};
  local i, v = next(t, nil);
  while i do
    if type(v) == "table" then
      v = dbCraft_CloneTable(v);
    end
    new[i] = v;
    i, v = next(t, i);
  end
  return new;
end

function dbCraft_Print(...)
  local text = "";
  if DEFAULT_CHAT_FRAME then
    for i = 1, select("#", ...) do
      if i == 1 then
        text = select(i, ...);
      else
        text = text .. " " .. select(i, ...);
      end
    end

    DEFAULT_CHAT_FRAME:AddMessage(text, 1.0, 0.35, 0.15);
  end
end

function dbCraft:ShowTooltip(link) -- (Do not change dbCraft:ShowTooltip, it needs a colon)
  if IsShiftKeyDown() then
    if strsub(link, 1, 6) == "player" then
      SendWho("n-" .. strsub(link, 8));
    else
      local RealmName = GetRealmName();
      local GuildName = GetGuildInfo("player");

      if GuildName == nil then
        GuildName = "-> Unguilded <-";
      end

      local Language;
      local faction, faction2 = UnitFactionGroup("player");
      if faction == "Alliance" then
        Language = "Common";
      else
        Language = "Orcish";
      end

      local tradeSkillKey = dbCraft_GetdbCraftKey(dbCraft.FooTradeSkill);
      dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
      dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
      dbCraftDatabase[RealmName][GuildName][tradeSkillKey] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey] or {};
      dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"] or {};

      local category, level, myItem = dbCraft_SearchDatabaseForLink(link);
      if myItem ~= nil then
        local item = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][category][level][myItem];
        local text = "dbCraft: " .. myItem .. " requires: ";
        local ctr = strlen(text);
        local t = dbCraft_myLinkSort(item.Materials);
        for i = 1, getn(t) do
          if (i > 1)  and (ctr > 1) then
            text = text .. ", ";
          end
          materialLink = t[i];
          if (ctr + 4 + strlen(materialLink)) > 250 then
          SendChatMessage(text, "GUILD", Language); 
            text = "dbCraft: ";
            ctr = strlen(text);
          end
          text = text .. materialLink;
          ctr = ctr + strlen(materialLink);
          if item.Materials[materialLink] > 1 then
            text = text .. " x" .. item.Materials[materialLink];
            ctr = ctr + 2 + strlen(item.Materials[materialLink]);
          end
        end
        SendChatMessage(text, "GUILD", Language); 
      end
    end
  else
    -- display a popup instead
    SetItemRef(link, nil, 'RightButton');
  end
end

function dbCraft_SearchDatabaseForLink(link)
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local tradeSkillKey = dbCraft_GetdbCraftKey(dbCraft.FooTradeSkill);
  local category, level, item;
  for category in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"]) do
    for  level in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][category]) do
      for item in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][category][level]) do
        if strfind(item, link) then
          return category, level, item;
        end
      end
    end
  end
  return nil, nil, nil;
end

function dbCraft_SplitTabs(s)
  local result = {};

  local i = string.find(s, '\t');
  while i ~= nil do
    table.insert(result, string.sub(s, 1, i - 1));
    s = string.sub(s, i + 1);
    i = string.find(s, '\t');
  end

  table.insert(result, s);

  return result;
end

--------------------------------------------------------------------------------
--
-- Crafters
--
--------------------------------------------------------------------------------

function dbCraft_AddSenderToCrafterList(sender, record)
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local senderVersion = dbCraft_SplitTabs(record);
  local senderVersionHigh = tonumber(senderVersion[1]);
  local senderVersionLow  = tonumber(senderVersion[2]);
  local senderVersionBeta = tonumber(senderVersion[3]);

  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"] = dbCraftDatabase[RealmName][GuildName]["Crafters"] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"][sender] = dbCraftDatabase[RealmName][GuildName]["Crafters"][sender] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"][sender]["Version"] = dbCraftDatabase[RealmName][GuildName]["Crafters"][sender]["Version"] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"][sender]["Version"]["High"] = senderVersionHigh;
  dbCraftDatabase[RealmName][GuildName]["Crafters"][sender]["Version"]["Low"]  = senderVersionLow;
  dbCraftDatabase[RealmName][GuildName]["Crafters"][sender]["Version"]["Beta"] = senderVersionBeta;
  dbCraftDatabase[RealmName][GuildName]["Crafters"][sender]["Last Send"] = time();
end

function dbCraft_AddSelfToCrafterList()
  local character = UnitName("player");
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"] = dbCraftDatabase[RealmName][GuildName]["Crafters"] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"][character] = dbCraftDatabase[RealmName][GuildName]["Crafters"][character] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"][character]["Version"] = dbCraftDatabase[RealmName][GuildName]["Crafters"][character]["Version"] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"][character]["Version"]["High"] = dbCraftOptions.Version.High;
  dbCraftDatabase[RealmName][GuildName]["Crafters"][character]["Version"]["Low"]  = dbCraftOptions.Version.Low;
  dbCraftDatabase[RealmName][GuildName]["Crafters"][character]["Version"]["Beta"] = dbCraftOptions.Version.Beta;
  dbCraftDatabase[RealmName][GuildName]["Crafters"][character]["Last Send"] = time();
end

function dbCraft_UpdateCrafterList()
  local i, tradeSkill, item, crafter;
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"] = dbCraftDatabase[RealmName][GuildName]["Crafters"] or {};

  for i, tradeSkill in ipairs(dbCraft.TradeSkills) do
    local tradeSkillKey = dbCraft_GetdbCraftKey(tradeSkill);
    dbCraftDatabase[RealmName][GuildName][tradeSkillKey] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey] or {};
    dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"] or {};

    for item in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"]) do
      for crafter in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"][item]) do
        if dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter] == nil then
          dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter] = {};
          dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Version"] = { High = 0, Low = 0, Beta = 0};
          dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Last Send"] = 0 ;
        end
      end
    end
  end
end

function dbCraft_ViewCrafterList()
  local i, text, crafter;
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName]["Crafters"] = dbCraftDatabase[RealmName][GuildName]["Crafters"] or {};

  text = "<html><body>";
  text = text .. "<h1>|cff8080ffCrafters|r|n|n</h1>";

  local t = dbCraft_mySort(dbCraftDatabase[RealmName][GuildName]["Crafters"]);
  for i = 1, getn(t) do
    crafter = t[i];
    text = text .. "<h1>" .. crafter .. "|n</h1>";
    text = text .. "<p>|cff808080";
    if dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Version"].High == 0 then
      text = text .. "Last send prior to v2.1.0";
    else
      text = text .. "[" .. dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Version"].High;
      text = text .. "." .. dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Version"].Low;
      text = text .. "." .. dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Version"].Beta .. "] ";
      text = text .. "Data is " .. dbCraft_FormatTime(difftime(time(), dbCraftDatabase[RealmName][GuildName]["Crafters"][crafter]["Last Send"])) .. " old.";
    end
    text = text .. "|r|n|n</p>";
  end

  if getn(t) == 0 then
    text = text .. "<br/><br/><br/><br/><p align='center'>|cff808080There are currently no crafters in the database.|r</p>";
  end

  text = text .. "</body></html>";
  dbCraft_DisplayText(text);
end

function dbCraft_FormatTime(t)
  local text = "";

  local time = date("!*t", t);
  time.year = time.year - 1970;
  time.month = time.month - 1;
  time.day = time.day - 1;

  if time.year > 0 then
    text = text  .. "over "  .. time.year  .. " year";
    if time.year > 1 then
      text = text .. "s";
    end
  elseif time.month > 0 then
    text = text  .. "over "  .. time.month  .. " month";
    if time.month > 1 then
      text = text .. "s";
    end
  elseif time.day > 0 then
    text = text   .. time.day  .. " day";
    if time.day > 1 then
      text = text .. "s";
    end
  elseif time.hour > 0 then
    text = text   .. time.hour  .. " hour";
    if time.hour > 1 then
      text = text .. "s";
    end
  elseif time.min > 0 then
    text = text   .. time.min  .. " minute";
    if time.min > 1 then
      text = text .. "s";
    end
  else
    text = text   .. "less than 1 minute";
  end

  return text;
end

--------------------------------------------------------------------------------
--
-- Send Data
--
--------------------------------------------------------------------------------

function dbCraft_ProcessTick()
  dbCraft_SendTradeSkills();
  dbCraft_SendRecordPortion();
end

function dbCraft_SendTradeSkills(b)
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    if b then
      dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Broadcast request aborted. You are not in a guild.|r");
    end
    return;
  end

  if b~=true then
    if dbCraftCharacterData["lastscan"] == nil then return; end
    if (dbCraftCharacterData["lastsend"] ~= nil) and (dbCraftCharacterData["lastsend"] > time() - 3 * 60 * 60) then return; end
  end

  local sessionId = time();

  local records = dbCraft_GetAllItemRecords(sessionId);

  local headerRecord = dbCraft_GetHeaderRecord(sessionId, table.getn(records));

  dbCraft_Print(textColor.AddonName .. "dbCraft " .. textColor.GeneralText .. "starts broadcasting your trade skill data (" .. textColor.NumberOfRecords .. (table.getn(records) + 1) .. " records" .. textColor.GeneralText .. ") to your guild.|r");

  dbCraft_SendRecord("Version\t" .. dbCraftOptions.Version.High .. "\t" .. dbCraftOptions.Version.Low .. "\t" .. dbCraftOptions.Version.Beta);
  dbCraft_SendRecord(headerRecord);

  dbCraft.BufferRecords = records;
  dbCraft.BufferPosition = 1;

  dbCraftCharacterData["lastsend"] = time();
end

function dbCraft_SendDataRequest(recipient)
  dbCraft_SendRecord("Request\t" .. recipient);
end

function dbCraft_SendChatRequest(chatType, recipient, msg)
  dbCraft_SendRecord("Chat\t" .. chatType .. "\t" .. recipient .. "\t" .. msg);
end

function dbCraft_SendRecordPortion()
  if dbCraft.BufferPosition == nil then return; end
  if dbCraft.BufferRecords == nil then return; end
  if dbCraft.BufferPosition > table.getn(dbCraft.BufferRecords) then return; end -- getn returns string

  local nextBufferPosition = dbCraft.BufferPosition + 50;
  while (dbCraft.BufferPosition < nextBufferPosition) and (dbCraft.BufferPosition <= table.getn(dbCraft.BufferRecords)) do
    dbCraft_SendRecord(dbCraft.BufferRecords[dbCraft.BufferPosition]);
    dbCraft.BufferPosition = dbCraft.BufferPosition + 1;
  end

  if dbCraft.BufferPosition > table.getn(dbCraft.BufferRecords) then
    dbCraft_Print(textColor.AddonName .. "dbCraft " .. textColor.GeneralText .. "finished broadcasting your trade skill data (" .. textColor.NumberOfRecords .. (table.getn(dbCraft.BufferRecords) + 1) .. " records" .. textColor.GeneralText .. ").|r");
    dbCraft.BufferRecords = nil;
    dbCraft.BufferPosition = nil;
  end
end

function dbCraft_GetAllItemRecords(sessionId)
  local i, tradeSkill;
  local records = {};

  for i, tradeSkill in ipairs(dbCraft.TradeSkills) do
    dbCraft_GetGivenItemRecords(sessionId, tradeSkill, records);
  end

  return records;
end

function dbCraft_GetHeaderRecord(sessionId, recordCount)
  return "Session\t" .. sessionId .. "\t" .. recordCount;
end

function dbCraft_SendRecord(record)
  SendAddonMessage("dbCraft", record, "GUILD");
end

function dbCraft_GetGivenItemRecords(sessionId, tradeSkill, records)
  local i, item;

  dbCraftCharacterData[tradeSkill] = dbCraftCharacterData[tradeSkill] or {};

  dbCraft_AddRecord(records, sessionId, "TradeSkill\t" .. tradeSkill);
  for i, item in ipairs(dbCraftCharacterData[tradeSkill]) do
    dbCraft_GetItemRecords(sessionId, tradeSkill, item, records);
  end
end

function dbCraft_GetItemRecords(sessionId, tradeSkill, item, records)
  local materialLink;

  dbCraft_AddRecord(records, sessionId, "Item\t" .. item.ItemLink .. "\t" .. item.Category .. "\t" .. item.Count .. "\t" .. item.RequiredUserLevel);

  for materialLink in pairs(item.Materials) do
    dbCraft_AddRecord(records, sessionId, "Material\t" .. materialLink .. "\t" .. item.Materials[materialLink]);
  end
end

function dbCraft_AddRecord(records, sessionId, data)
  local record = "Rec\t" .. (table.getn(records) + 1) .. "\t" .. sessionId .. "\t" .. data;
  table.insert(records, record);
end

--------------------------------------------------------------------------------
--
-- Record Processing
--
--------------------------------------------------------------------------------

function dbCraft_HandleRecord(record, sender)

  if string.sub(record, 1, 8) == 'Session\t' then
    if sender ~= UnitName("player") then
      dbCraft_HandleSessionRecord(string.sub(record, 9), sender);
    end
  elseif string.sub(record, 1, 4) == 'Rec\t' then
    if sender ~= UnitName("player") then
      dbCraft_HandleDataRecord(string.sub(record, 5), sender);
    end
  elseif string.sub(record, 1, 8) == 'Version\t' then
    if sender ~= UnitName("player") then
      dbCraft_HandleVersionRecord(string.sub(record, 9), sender);
    end
  elseif string.sub(record, 1, 8) == 'Request\t' then
    dbCraft_HandleRequestRecord(string.sub(record, 9), sender);
  elseif string.sub(record, 1, 5) == 'Chat\t' then
    dbCraft_HandleChatRecord(string.sub(record, 6), sender);
  end
end

function dbCraft_HandleSessionRecord(record, sender)
  local i = string.find(record, '\t');
  local sessionId = tonumber(string.sub(record, 1, i - 1));
  local recordCount = tonumber(string.sub(record, i + 1));

  dbCraft.Sessions = dbCraft.Sessions or {};
  dbCraft.Sessions[sender] = dbCraft.Sessions[sender] or {};
  dbCraft.Sessions[sender][sessionId] = {};
  dbCraft.Sessions[sender][sessionId]['recordCount'] = recordCount;
  dbCraft.Sessions[sender][sessionId]['recordsReceived'] = 0;

  if dbCraftOptions.HideMessages == false then
    dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Receiving " .. textColor.SenderName .. sender .. "'s" .. textColor.GeneralText .. " trade skill data (" .. textColor.NumberOfRecords .. (recordCount + 1) .. " records" .. textColor.GeneralText .. ").|r");
  end
end

function dbCraft_HandleDataRecord(record, sender)
  local i = string.find(record, '\t');
  local recordNumber = tonumber(string.sub(record, 1, i - 1));
  local restOfRecord = string.sub(record, i + 1);
  local j = string.find(restOfRecord, '\t');
  local sessionId = tonumber(string.sub(restOfRecord, 1, j - 1));
  local data = string.sub(restOfRecord, j + 1);

  if dbCraft.Sessions == nil then return; end
  if dbCraft.Sessions[sender] == nil then return; end
  if dbCraft.Sessions[sender][sessionId] == nil then return; end
  if dbCraft.Sessions[sender][sessionId][recordNumber] ~= nil then return; end

  dbCraft.Sessions[sender][sessionId][recordNumber] = data;

  dbCraft.Sessions[sender][sessionId]['recordsReceived'] = dbCraft.Sessions[sender][sessionId]['recordsReceived'] + 1;
  if dbCraft.Sessions[sender][sessionId]['recordsReceived'] == dbCraft.Sessions[sender][sessionId]['recordCount'] then
    dbCraft_HandleSession(sender, sessionId);
  end
end

function dbCraft_HandleVersionRecord(record, sender)
  local senderVersion = dbCraft_SplitTabs(record);
  local senderVersionHigh = tonumber(senderVersion[1]);
  local senderVersionLow  = tonumber(senderVersion[2]);
  local senderVersionBeta = tonumber(senderVersion[3]);

  dbCraft_AddSenderToCrafterList(sender, record);

  if senderVersionBeta == 0 then -- If sender is using a beta version then ignore it.
    if (senderVersionHigh > dbCraftOptions.Version.High) or ((senderVersionHigh == dbCraftOptions.Version.High) and (senderVersionLow > dbCraftOptions.Version.Low)) then
      if (senderVersionHigh > dbCraftOptions.HighestVersionSeen.High) or ((senderVersionHigh == dbCraftOptions.HighestVersionSeen.High) and (senderVersionLow > dbCraftOptions.HighestVersionSeen.Low)) then
        dbCraftOptions.HighestVersionSeen.High = senderVersionHigh;
        dbCraftOptions.HighestVersionSeen.Low = senderVersionLow;
        dbCraftOptions.HighestVersionSeen.Beta = senderVersionBeta;
        dbCraft_ShowHeader();
      end
      local text = textColor.AddonName .. "dbCraft: " .. textColor.SenderName .. sender .. textColor.GeneralText .. " is running a newer version than you (";
      text = text .. textColor.Version .. "v" .. senderVersionHigh .. "." .. senderVersionLow .. "." .. senderVersionBeta .. textColor.GeneralText .. ").|r";
      dbCraft_Print(text);
    end
  end
end

function dbCraft_HandleRequestRecord(recipient, sender)
  if strlower(recipient) == strlower(UnitName("player")) then
    dbCraftCharacterData["lastsend"] = nil;
    dbCraft_SendTradeSkills(true);
  end
end

function dbCraft_HandleChatRecord(record, sender)
  local chatType, recipient, msg = strsplit("\t", record);

  if ((recipient == strlower(UnitName("player"))) or (recipient == "all")) then
    if chatType == "say" then
      if dbCraftOptions.remotesay == true then
        SendChatMessage(msg, "SAY", "Common"); 
      end
    elseif chatType == "guild" then
      if dbCraftOptions.remoteguild == true then
        SendChatMessage(msg, "GUILD", "Common"); 
      end
    elseif chatType == "emote" then
      if dbCraftOptions.remoteemote == true then
        SendChatMessage(msg, "EMOTE", "Common"); 
      end
    elseif chatType == "doemote" then
      if dbCraftOptions.remotedoemote == true then
        DoEmote(msg);
      end
    end
  end
end

function dbCraft_HandleSession(sender, sessionId)
  local i;
  local values = nil;
  local tradeSkill = nil;
  local item = nil;

  for i = 1, dbCraft.Sessions[sender][sessionId]['recordCount'] do
    values = dbCraft_SplitTabs(dbCraft.Sessions[sender][sessionId][i]);
    if values[1] == 'TradeSkill' then
      if item ~= nil then
        dbCraft_AddKnownItem(tradeSkill, item);
        dbCraft_AddItemCreator(tradeSkill, item.ItemLink, sender);
      end

      tradeSkill = values[2];
      item = nil;

      dbCraft_RemoveCharacterGivenSkills(tradeSkill, sender);
    elseif values[1] == 'Item' then
      if item ~= nil then
        dbCraft_AddKnownItem(tradeSkill, item);
        dbCraft_AddItemCreator(tradeSkill, item.ItemLink, sender);
      end

      item = {};
      item.ItemLink = values[2];
      item.Category = values[3];
      item.Count = tonumber(values[4]);
      item.RequiredUserLevel = tonumber(values[5]);
      item.Materials = {};
    elseif values[1] == 'Material' then
      item.Materials[values[2]] = tonumber(values[3]);
    else
      return;
    end
  end

  if item ~= nil then
    dbCraft_AddKnownItem(tradeSkill, item);
    dbCraft_AddItemCreator(tradeSkill, item.ItemLink, sender);
  end

  if dbCraftOptions.HideMessages == false then
    dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Finished receiving " .. textColor.SenderName .. sender .. "'s" .. textColor.GeneralText .. " trade skill data (" .. textColor.NumberOfRecords .. (dbCraft.Sessions[sender][sessionId]['recordCount'] + 1) .. " records" .. textColor.GeneralText .. ").|r");
  end

  dbCraft.Sessions[sender] = nil;
end

function dbCraft_AddKnownItem(tradeSkill, item)
  if dbCraft_GetTableSize(item.Materials) == 0 then return; end

  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local tradeSkillKey = dbCraft_GetdbCraftKey(tradeSkill);
  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][item.Category] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][item.Category] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][item.Category][item.RequiredUserLevel] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][item.Category][item.RequiredUserLevel] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][item.Category][item.RequiredUserLevel][item.ItemLink] = item;
end

function dbCraft_AddItemCreator(tradeSkill, itemLink, character)
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local tradeSkillKey = dbCraft_GetdbCraftKey(tradeSkill);
  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"][itemLink] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"][itemLink] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"][itemLink][character] = 1;
end

function dbCraft_GetdbCraftKey(tradeSkill)
  return "Guild" .. tradeSkill;
end

--------------------------------------------------------------------------------
--
-- Import Character Data
--
--------------------------------------------------------------------------------

function dbCraft_ImportCurrentCharacter()
  local i, tradeSkill;
  for i, tradeSkill in ipairs(dbCraft.TradeSkills) do
    dbCraft_ImportCurrentCharacterGivenSkills(tradeSkill);
  end
end

function dbCraft_ImportCurrentCharacterGivenSkills(tradeSkill)
  local character = UnitName("player");

  dbCraft_RemoveCharacterGivenSkills(tradeSkill, character);

  dbCraftCharacterData[tradeSkill] = dbCraftCharacterData[tradeSkill] or {};

  local i, item;
  for i, item in ipairs(dbCraftCharacterData[tradeSkill]) do
    dbCraft_AddKnownItem(tradeSkill, item);
    dbCraft_AddItemCreator(tradeSkill, item.ItemLink, character);
  end
end

function dbCraft_RemoveCharacterGivenSkills(tradeSkill, character)
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local tradeSkillKey = dbCraft_GetdbCraftKey(tradeSkill);
  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"] or {};

  local i, item;
  for i, item in ipairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"]) do
    if item[character] == 1 then
      item[character] = 0;
    end
  end
end

--------------------------------------------------------------------------------
--
-- View Trade Skills
--
--------------------------------------------------------------------------------

function dbCraft_ViewTradeSkills(tradeSkill, b)
  local text;
  if (tradeSkill == nil) then
    text = dbCraft_LoadSplashScreen();
  else
    dbCraft.FooTradeSkill = tradeSkill;

    if (b  == true) then
      dbCraft.SearchString = "";
    end

    text = dbCraft_BuildSkillList(tradeSkill);
  end
  dbCraft_DisplayText(text);
end


function dbCraft_LoadSplashScreen()
    local text = "<html><body><br/>";
    text = text .. "<h1 align='center'>|cff0000ffdbCraft v" .. DBCRAFT_VERSION .. "|n|n</h1>";
    text = text .. "<h2 align='center'>|cff808080In-Game Guild Crafting Database|n|n</h2>";
    text = text .. "<h3 align='center'>|cffffff00Written by Mike Summers|n|n</h3>";
    text = text .. "<p align='center'>|cff808080(Rhaithe - Order of Exiles - Proudmoore Alliance)</p>";
--    text = text .. "<p align='center'>|cff808080dbcraft@rhaithe.com -- http://dbcraft.rhaithe.com</p>";
    text = text .. "<p align='center'>|cffff0000|nCredits|n|n</p>";
    text = text .. "<p>|cff808080dbCraft is based on iGuildTradeSkill by HPi which was based on GuildTradeSkill by Orlan. The Interface and Mini-Map are based on what I learned from examining the Atlas Addon. These people did some truly amazing work and deserve most of the credit.|n|n</p>";

    text = text .. "<h1 align='center'>|n|cff00ff00New Features|n|n</h1>";
    text = text .. "<p>|cffffffff[Version 2.1.0]|n|n</p>";
    text = text .. "<p>|cff808080Skills are now listed alphabetically and are no longer grouped by category.|r|n|n</p>";
    text = text .. "<p>|cff808080Displays converted to SimpleHTML format to improve readability.|r|n|n</p>";
    text = text .. "<p>|cff808080Shift-Clicking on a recipe now links that recipe and it's materials list to guild chat channel.|r|n|n</p>";
    text = text .. "<p>|cff808080Removed unused data from database.|r|n|n</p>";
    text = text .. "<p>|cff808080Corrected minor display error on splash screen.|r|n|n</p>";
    text = text .. "<p>|cff808080Lots of general code cleanup and standardization of variable and function names. Localized variables and tightened up code where possible.|r|n|n</p>";
    text = text .. "<p>|cff808080Added table of crafters to database and a button to display them. Crafters are listed alphabetically with version and age of data.|r|n|n</p>";

    text = text .. "<h1 align='center'>|n|cff00ff00Change Log|n|n</h1>";
    text = text .. "<p>|cffffffff[Version 2.0.0]|n|n</p>";
    text = text .. "<p>|cff808080Crafter names are now displayed in a comma-separated list instead of one per line.|r|n|n</p>";
    text = text .. "<p>|cff808080All characters on the same account share the same database. (Options are still on a per character basis.)|r|n|n</p>";
    text = text .. "<p>|cff808080Added option to hide 'data received' messages.|r|n|n</p>";
    text = text .. "<p>|cff808080Added option to specify days between scan reminder.|r|n|n</p>";
    text = text .. "<p>|cff808080Only trade skills of guild members are displayed (since shared DB may include data for alts in other guilds).|r|n|n</p>";
    text = text .. "<p>|cff808080Unguilded Alts will now see data for your other Unguilded Alts.|r|n|n</p>";
    text = text .. "<p>|cff808080Corrected issue with frame elements displaying on inappropriate layers.|r|n|n</p>";
    text = text .. "<p>|cff808080Added Guild and Realm identifier to database and to display.|r|n|n</p>";
    text = text .. "<p>|cff808080Added new version detection and a message informing you when a new version is available and where you can download it.|r|n|n</p>";
    text = text .. "<p>|cff808080Changed description from 'On-Line...' to 'In-Game Guild Crafting Database'.|r|n|n</p>";
    text = text .. "<p>|cff808080Created unique artwork for interface and mini-map button.|r|n|n</p>";
    text = text .. "<p>|cff808080Added secret option for Eli to interface between his coffee pot and iPhone.|r|n|n</p>";
    text = text .. "<p>|cff808080Added Shift-Click detection on recipes to add link to chat channel.|r|n|n</p>";
    text = text .. "<p>|cff808080Added Right Click menu on crafter names (Use Shift Click on crafter names, sometimes this takes a couple shift clicks, to get /who info. This will let you easily see if they are currently on-line.)|r|n|n</p>";
    text = text .. "<p>|cff808080Improved the look of the addon messages.|r|n|n</p>";
    text = text .. "</body></html>";
    return text;
end

function dbCraft_BuildSkillList(tradeSkill)
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local tradeSkillKey = dbCraft_GetdbCraftKey(tradeSkill);
  dbCraftDatabase[RealmName] = dbCraftDatabase[RealmName] or {};
  dbCraftDatabase[RealmName][GuildName] = dbCraftDatabase[RealmName][GuildName] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey] or {};
  dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"] = dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"] or {};

  local t = {}; local c = {}; local l = {};
  local category, level, item;
  for category in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"]) do
    for  level in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][category]) do
      for item in pairs(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][category][level]) do
        tinsert(t, getn(t) + 1, item);
        tinsert(c, getn(c) + 1, category);
        tinsert(l, getn(l) + 1, level);
      end
    end
  end

  t, c, l = dbCraft_myFullLinkSort(t, c, l);

  local i;
  local text = "<html><body>";
  text = text .. "<h1>|cff8080ff" .. tradeSkill .. "|r|n|n</h1>";
  for i = 1, getn(t) do
    item = t[i];
    category = c[i];
    level = tonumber(l[i]);
    text = text .. dbCraft_BuildItemDescription(tradeSkill, dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Items"][category][level][item]);
  end
  if getn(t) == 0 then
    text = text .. "<br/><br/><br/><br/><p align='center'>|cff808080There is currently no data available for this tradeskill.|r</p>";
  end
  text = text .. "</body></html>";
  return text;
end

function dbCraft_BuildItemDescription(tradeSkill, item)
  local RealmName = GetRealmName();
  local GuildName = GetGuildInfo("player");
  local tradeSkillKey = dbCraft_GetdbCraftKey(tradeSkill);

  if GuildName == nil then
    GuildName = "-> Unguilded <-";
  end

  local justName = string.gsub(tostring(item.ItemLink),"^.-%[(.*)%].*", "%1");

  if string.find(strlower(justName), dbCraft.SearchString) == nil then
    return "";
  end

  local text = "<h1>" .. item.ItemLink;
  if item.Count > 1 then
    text = text .. textColor.GeneralText .. " x" .. item.Count .. "|r";
  end
  text = text .. "|n</h1>";

  text = text .. "<p>|cffffff00Materials:|r|n</p>";

  local i, materialLink;
  local t = dbCraft_myLinkSort(item.Materials);
  for i = 1, getn(t) do
    materialLink = t[i];

    text = text .. "<p>|cff0d000d . . . |r" .. materialLink;

    if item.Materials[materialLink] > 1 then
      text = text .. textColor.GeneralText .. " x" .. item.Materials[materialLink] .. "|r";
    end

    text = text .. "|n</p>";
  end

  text = text .. "<p>|cffffff00Crafters:|r ";

  local ctr = 10;
  local firstCrafter = true;

  local t = dbCraft_myLinkSort(dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"][item.ItemLink]);
  for i = 1, getn(t) do
    crafterName = t[i];
    if dbCraftDatabase[RealmName][GuildName][tradeSkillKey]["Creators"][item.ItemLink][crafterName] == 1 then
      if not firstCrafter then
        text = text .. ", ";
        ctr = ctr + 2;
      end
      if ctr + 2 + strlen(crafterName) > 80 then
        text = text .. "|n|cff0d000d . . . |r";
        ctr = 6;
        firstCrater = true;
      end

      text = text .. "|cff00ffff|Hplayer:" .. crafterName .. "|h[" .. crafterName .. "]|h|r";
      ctr = ctr + 2 + strlen(crafterName);

      firstCrafter = false;
    end
  end

  text = text .. "|n|n</p>";
  
  return text;
end

--------------------------------------------------------------------------------
--
-- Now it gets weird
--
--------------------------------------------------------------------------------

function dbCraft_HandleLoaded()
  dbCraft.Scanner = dbCraft.TradeSkillScanner:new();

  Chronos.scheduleRepeating("dbCraft:MessageSend", 3, function() dbCraft_ProcessTick(); end);

  dbCraft_Print(textColor.AddonName .. "dbCraft " .. textColor.Version .. "v" .. DBCRAFT_VERSION .. textColor.GeneralText .. " has been loaded. Type /dbCraft or use Mini-Map Button to open.|r");

  if (dbCraftCharacterData["lastscan"] == nil) or (dbCraftCharacterData["lastscan"] < (time() - 60 * 60 * 24 * tonumber(dbCraftOptions.Reminder))) then
    local text = textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "It's time for you to scan your trade skills.|r";
    dbCraft_Print(text);
    dbCraft.Scanner:AskScan();
  end
end

function dbCraft_ScanTradeSkills(b)
  dbCraft.Scanner:Scan();
  return;
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

dbCraft.TradeSkills = {
  "Alchemy",
  "Blacksmithing",
  "Cooking",
  "Enchanting",
  "Engineering",
  "First Aid",
  "Jewelcrafting",
  "Leatherworking",
  "Smelting",
  "Tailoring"
};

dbCraft.TradeSkillNames = {
  ["Alchemy"] = "Alchemy",
  ["Blacksmithing"] = "Blacksmithing",
  ["Cooking"] = "Cooking",
  ["Enchanting"] = "Enchanting",
  ["Engineering"] = "Engineering",
  ["First Aid"] = "First Aid",
  ["Jewelcrafting"] = "Jewelcrafting",
  ["Leatherworking"] = "Leatherworking",
  ["Smelting"] = "Mining",
  ["Tailoring"] = "Tailoring"
};

dbCraft.TradeSkillScanner = {};
local TradeSkillScanner = dbCraft.TradeSkillScanner;

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------

function TradeSkillScanner:new(object)
  object = object or {};
  setmetatable(object, self);
  self.__index = self;

  object:CreateDialog();

  local frame = CreateFrame("Frame");
  frame:Hide();
  frame:SetScript("OnEvent",
    function()
      object:HandleTradeSkillUpdate();
    end);
  frame:RegisterEvent("TRADE_SKILL_UPDATE");
  frame:RegisterEvent("CRAFT_UPDATE");
  frame:RegisterEvent("SPELLS_CHANGED");

  return object;
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

TradeSkillScanner.scanInProgress = false;
TradeSkillScanner.working = false;

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------

function TradeSkillScanner:Scan()
  if self.scanInProgress then
    dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "scan is already in progress.|r");
    return;
  end

  self.scanInProgress = true;

  dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Scanning your current trade skills...|r");

  self.currentSkillIndex = 0;

  self:ScanNextTradeSkill();
end

function TradeSkillScanner:CancelScan()
  dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Scan cancelled.");
  self.scanInProgress = false;
end

function TradeSkillScanner:CloseDialog()
  self.dialog:Hide();
end

function TradeSkillScanner:CreateDialog()
  local frame = CreateFrame("Frame", nil, UIParent);
  frame:Hide();
  frame:SetPoint("CENTER", "UIParent", "CENTER");
  frame:SetFrameStrata("DIALOG");
  frame:SetHeight(100);
  frame:SetWidth(200);
  frame:SetBackdrop({
    bgFile = "Interface/Tooltips/UI-Tooltip-Background",
    edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
    tile = true, tileSize = 32, edgeSize = 32,
    insets = { left = 9, right = 9, top = 9, bottom = 9 }
  });
  frame:SetBackdropColor(0, 0, 0, 1);

  frame.Do = CreateFrame("Button", "dbCraftViewCloseButton", frame, "OptionsButtonTemplate");
  frame.Do:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", 10, 10);

  frame.Cancel = CreateFrame("Button", "dbCraftViewCloseButton", frame, "OptionsButtonTemplate");
  frame.Cancel:SetText("Cancel");
  frame.Cancel:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -10, 10);
  frame.Cancel:SetScript("OnClick",
    function()
      self:CancelScan();
      self:CloseDialog();
    end);

  frame.Box = CreateFrame("SimpleHTML", "dbCraftViewEditBox", frame);
  frame.Box:SetHeight(70);
  frame.Box:SetWidth(170);
  frame.Box:SetFontObject(GameFontHighlight);
  frame.Box:SetPoint("TOPLEFT", frame, "TOPLEFT", 15, -15);

  self.dialog = frame;
end

function TradeSkillScanner:ShowDialog(text, doCaption, doFunction)
  self.dialog.Do:SetText(doCaption);
  self.dialog.Do:SetScript("OnClick",
    function()
      self.dialog:Hide();
      doFunction();
    end);
  self.dialog.Box:SetText(text);
  self.dialog:Show();
end

function TradeSkillScanner:AskScan()
  self:ShowDialog(
    textColor.AddonName .. "dbCraft|n|n" .. textColor.Highlight .. "It's time for you to scan your trade skills.|r",
    "Scan Now",
    function()
      self:Scan();
    end);
end

function TradeSkillScanner:AskContinue()
  self:ShowDialog(
    textColor.AddonName .. "dbCraft|n|n" .. textColor.Highlight .. "Scanning next profession. Press continue, please.|r",
    "Continue",
    function()
      self:ScanNextTradeSkill();
    end);
end

function TradeSkillScanner:HandleTradeSkillUpdate()
  if self.scanInProgress and not self.working then
    --dbCraft_Print("dbCraft: Missing data received.");
    self:ResumeScan();
  end
end

function TradeSkillScanner:ScanNextTradeSkill()
  self.currentSkillIndex = self.currentSkillIndex + 1;

  local profession = dbCraft.TradeSkills[self.currentSkillIndex];

  if profession == nil then
    self:HandleScanFinished();
    return;
  end

  self.currentProfessionScanner = dbCraft.ProfessionScanners[profession]:new({ Profession = profession });

  if self.currentProfessionScanner:IsKnownProfession() then
    self:ResumeScan();
  else
    self:ScanNextTradeSkill();
  end
end

function TradeSkillScanner:ResumeScan()
  self.working = true;

  local success = self.currentProfessionScanner:Scan();

  self.working = false;

  if not success then
    if self.currentProfessionScanner.Wait then
      dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Waiting 3 seconds to make sure data received from server.|r");
      self.working = true;
      Chronos.schedule(3, self.ResumeScan, self);
    else
      --dbCraft_Print("dbCraft: Requesting missing data from server...");

      Chronos.schedule(2, self.HandleTradeSkillUpdate, self);
    end
    return;
  end

  self.working = true;
  self:AskContinue();
end

function TradeSkillScanner:HandleScanFinished()
  dbCraft_AddSelfToCrafterList();
  dbCraftCharacterData["lastscan"] = time();
  
  dbCraft_ImportCurrentCharacter();

  self.scanInProgress = false;

  dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Trade skills scan complete.|r");
  message(textColor.AddonName .. "dbCraft|n|n" .. textColor.Highlight .. "Trade skills scan complete.|r");
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

TradeSkillScanner.ProfessionScannerBase = {};
local ProfessionScannerBase = TradeSkillScanner.ProfessionScannerBase;

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------

function ProfessionScannerBase:new(object)
  object = object or {};
  setmetatable(object, self);
  self.__index = self;

  object.data = {};

  return object;
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

ProfessionScannerBase.scanStarted = false;
ProfessionScannerBase.scanFinished = false;
ProfessionScannerBase.itemCount = 0;
ProfessionScannerBase.itemIndex = 0;
ProfessionScannerBase.wait = false;
ProfessionScannerBase.currentCategory = "General";

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------

function ProfessionScannerBase:Scan()
  self.Wait = false;

  if not self.scanStarted then
    return not self:StartScan();
  end

  while self:ScanNextItem() do end

  return self.scanFinished;
end

function ProfessionScannerBase:ScanNextItem()
  local openProfession = self:GetOpenProfession();

  if openProfession ~= dbCraft.TradeSkillNames[self.Profession] then
    if (openProfession ~= nil) and (openProfession ~= "UNKNOWN") then
      dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Invalid profession got opened: " .. textColor.ProfessionName .. openProfession .. textColor.GeneralText .. ". Please open " .. textColor.ProfessionName.. self.Profession .. textColor.GeneralText .. ".|r");
    else
      dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.GeneralText .. "Profession open failed. Please open " .. textColor.ProfessionName .. self.Profession .. textColor.GeneralText .. ".|r");
    end
    return false;
  end

  if self.itemIndex == 0 then
    self:ResetFilters();
  end

  self.itemIndex = self.itemIndex + 1;
  if self.itemIndex > self:GetItemCount() then
    self:HandleScanFinished();
    return false;
  end

  self:SelectCurrentItem();

  if self:CurrentItemIsHeader() then
    self.currentCategory = self:GetCurrentCategoryName();
    self:EnsureCurrentItemExpanded();
  else
    local itemDetails = self:GetItemDetails();

    if itemDetails.ItemLink == nil then
      return false;
    end

    if dbCraft_GetTableSize(itemDetails.Materials) == 0 then
      return false;
    end

    if not itemDetails.IsBop then
      self.itemCount = self.itemCount + 1;
      self.data[self.itemCount] = itemDetails;
    end
  end

  return true;
end

function ProfessionScannerBase:StartScan()
  self.scanStarted = true;

  if not self:IsKnownProfession() then
    dbCraft_Print("You don't know " .. self.Profession .. ".");
    self.scanFinished = true;
    return false;
  end

  self:OpenProfession();
  self.lastItemCount = -1;

  return true;
end

function ProfessionScannerBase:HandleScanFinished()
  if self.itemCount ~= self.lastItemCount then
    self.lastItemCount = self.itemCount;
    self.itemIndex = 0;
    self.itemCount = 0;
    self.data = {};
    self.currentCategory = "General";
    self.Wait = true;
  else
    self:CloseProfession();

    self.scanFinished = true;

    dbCraftCharacterData[self.Profession] = self.data;

    dbCraft_Print(textColor.AddonName .. "dbCraft: " .. textColor.ProfessionName .. self.Profession .. textColor.GeneralText .. " scan complete. " .. textColor.NumberOfSkills .. self.itemCount .. textColor.GeneralText .. " skills found.|r");
  end
end

function ProfessionScannerBase:IsKnownProfession()
  return IsUsableSpell(self.Profession);
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

TradeSkillScanner.TradeSkillProfessionScanner = ProfessionScannerBase:new();
local TradeSkillProfessionScanner = TradeSkillScanner.TradeSkillProfessionScanner;

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------

function TradeSkillProfessionScanner:GetOpenProfession()
  return GetTradeSkillLine();
end

function TradeSkillProfessionScanner:CloseProfession()
  CloseTradeSkill();
end

function TradeSkillProfessionScanner:OpenProfession()
  CastSpellByName(self.Profession);
end

function TradeSkillProfessionScanner:GetItemCount()
  return GetNumTradeSkills();
end

function TradeSkillProfessionScanner:SelectCurrentItem()
  SelectTradeSkill(self.itemIndex);
end

function TradeSkillProfessionScanner:GetCurrentCategoryName()
  local skillName = GetTradeSkillInfo(self.itemIndex);
  return skillName;
end

function TradeSkillProfessionScanner:CurrentItemIsHeader()
  local _, skillType = GetTradeSkillInfo(self.itemIndex);
  return skillType == "header";
end

function TradeSkillProfessionScanner:EnsureCurrentItemExpanded()
  local _, _, _, isExpanded = GetTradeSkillInfo(self.itemIndex);
  if not isExpanded then
    ExpandTradeSkillSubClass(self.itemIndex);
  end
end

function TradeSkillProfessionScanner:GetItemDetails()
  local item = {};

  item.ItemLink = GetTradeSkillItemLink(self.itemIndex);
  item.Category = self.currentCategory;
  item.Count = GetTradeSkillNumMade(self.itemIndex);
  if item.ItemLink == nil then
    item.Materials = {};
    item.RequiredUserLevel = 0;
    item.IsBop = false;
  else
    item.Materials = self:GetTradeSkillMaterials();
    _, _, _, _, item.RequiredUserLevel = GetItemInfo(item.ItemLink);
    item.IsBop = self:IsBop(item.ItemLink)
  end

  return item;
end

function TradeSkillProfessionScanner:IsBop(link)
  local _, _, id = string.find(link, "(item[:%d+]+)");
  dbCraftBopTooltip:ClearLines();
  dbCraftBopTooltip:SetHyperlink(id);

  return dbCraftBopTooltipTextLeft2:GetText() == ITEM_BIND_ON_PICKUP;
end

function TradeSkillProfessionScanner:GetTradeSkillMaterials()
  local materialCount = GetTradeSkillNumReagents(self.itemIndex);
  local materials = {};
  local materialIndex = 1;
  while materialIndex <= materialCount do
    local materialName, materialTexture, materialCount = GetTradeSkillReagentInfo(self.itemIndex, materialIndex);
    local materialLink = GetTradeSkillReagentItemLink(self.itemIndex, materialIndex);

    if materialLink ~= nil then
      materials[materialLink] = materialCount;
    end

    materialIndex = materialIndex + 1;
  end

  return materials;
end

function TradeSkillProfessionScanner:ResetFilters()
  SetTradeSkillInvSlotFilter(0, 1);
  SetTradeSkillSubClassFilter(0, 1);
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

TradeSkillScanner.CraftProfessionScanner = ProfessionScannerBase:new();
local CraftProfessionScanner = TradeSkillScanner.CraftProfessionScanner;

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------

function CraftProfessionScanner:GetOpenProfession()
  return GetCraftDisplaySkillLine();
end

function CraftProfessionScanner:CloseProfession()
  CloseCraft();
end

function CraftProfessionScanner:OpenProfession()
  CastSpellByName(self.Profession);
end

function CraftProfessionScanner:GetItemCount()
  return GetNumCrafts();
end

function CraftProfessionScanner:SelectCurrentItem()
  SelectCraft(self.itemIndex);
end

function CraftProfessionScanner:CurrentItemIsHeader()
  return false;
end

function CraftProfessionScanner:GetItemDetails()
  local item = {};

  item.ItemLink = GetCraftItemLink(self.itemIndex);
  item.Category = self.currentCategory;
  item.Count = 1;
  if item.ItemLink == nil then
    item.Materials = {};
  else
    item.Materials = self:GetCraftMaterials();
  end
  item.RequiredUserLevel = 0;
  item.IsBop = false;

  return item;
end

function CraftProfessionScanner:GetCraftMaterials()
  local materialCount = GetCraftNumReagents(self.itemIndex);
  local materials = {};
  local materialIndex = 1;
  while materialIndex <= materialCount do
    local materialName, materialTexture, materialCount = GetCraftReagentInfo(self.itemIndex, materialIndex);
    local materialLink = GetCraftReagentItemLink(self.itemIndex, materialIndex);

    if materialLink ~= nil then
      materials[materialLink] = materialCount;
    end

    materialIndex = materialIndex + 1;
  end

  return materials;
end

function CraftProfessionScanner:ResetFilters()
end

--------------------------------------------------------------------------------
--
-- Rogue code that doesn't want to be moved
--
--------------------------------------------------------------------------------

dbCraft.ProfessionScanners =
{
  ["Alchemy"] = TradeSkillProfessionScanner,
  ["Blacksmithing"] = TradeSkillProfessionScanner,
  ["Cooking"] = TradeSkillProfessionScanner,
  ["Enchanting"] = CraftProfessionScanner,
  ["Engineering"] = TradeSkillProfessionScanner,
  ["First Aid"] = TradeSkillProfessionScanner,
  ["Jewelcrafting"] = TradeSkillProfessionScanner,
  ["Leatherworking"] = TradeSkillProfessionScanner,
  ["Smelting"] = TradeSkillProfessionScanner,
  ["Tailoring"] = TradeSkillProfessionScanner
};

--------------------------------------------------------------------------------
--
-- End of rogue code
--
--------------------------------------------------------------------------------
