-- -*- indent-tabs-mode: t; tab-width: 4; lua-indent-level: 4 -*-
----------------------------
--      Declaration       --
----------------------------

XRS = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0", "AceDebug-2.0")
local XRS = XRS

local L = AceLibrary("AceLocale-2.2"):new("XRS")
local dewdrop = AceLibrary("Dewdrop-2.0")
local crayon = AceLibrary("Crayon-2.0")
local SEA = AceLibrary("SpecialEvents-Aura-2.0")

local setmetatable, type = setmetatable, type
local tinsert, tremove = tinsert, tremove
local pairs, ipairs = pairs, ipairs


-- Initialize textures
local media = LibStub("LibSharedMedia-3.0")
local mType = media.MediaType and media.MediaType.STATUSBAR or "statusbar"

local options = {
	type = "group",
	handler = XRS,
	args = {
		scale = {
			name = L["Scale"], type = "range",
			desc = L["Scale_D"],
			get = function() return XRS.db.profile.Scale end,
			set = function(v)
				XRS.db.profile.Scale = v
				XRS:UpdateScale()
			end,
			min = 0.5,
			max = 1.5,
			step = 0.05,
			order = 2
		},
		width = {
			name = L["Width"], type = "range",
			desc = L["Width_D"],
			get = function() return XRS.db.profile.Width end,
			set = function(v)
				XRS.db.profile.Width = v
				XRS:SetWidth()
			end,
			min = 70,
			max = 200,
			step = 5,
			order = 3
		},
		texture = {
			name = L["Textures"], type = 'text',
			desc = L["Textures_D"],
			get = function()
				return XRS.db.profile.Texture
			end,
			set = "SetBarTexture",
			validate = media:List(mType),
			order = 4,
		},
		color = {
			name = L["Color_Group"], type = "group",
			desc = L["Color_Group_D"],
			args = {
				background = {
						name = "Background Color", type = 'group',
						desc = "Change the Background Color",
						args = {
							usecolorgradient = {
								name = "Use Color Gradient", type = 'toggle',
								desc = "Toggle usage of color gradient for the window",
								get = "IsColorGradient",
								set = "SetColorGradient",
								map = { [false] = "Disabled", [true] = "Enabled" },
								order = 1,
							},
							colorgradientstart = {
								name = "Color Gradient Start", type = 'color',
								desc = "Set the Start Color of the Gradient",
								get = "GetColorGradientStart",
								set = "SetColorGradientStart",
								hasAlpha = true,
								disabled = "IsNotColorGradient",
								order = 2,
							},
							colorgradientstop = {
								name = "Color Gradient Stop", type = 'color',
								desc = "Set the Stop Color of the Gradient",
								get = "GetColorGradientStop",
								set = "SetColorGradientStop",
								hasAlpha = true,
								disabled = "IsNotColorGradient",
								order = 3,
							},
							backgroundcolor = {
								name = L["BColor"], type = 'color',
								desc = L["BColor_D"],
								get = function()
									local bc = XRS.db.profile.backgroundcolor
									return bc.r, bc.g, bc.b, bc.a
								end,
								set = "UpdateBackgroundColor",
								hasAlpha = true,
								order = 4,
							},
						},
						order = 1
					},
				title = {
					name = L["TColor"], type = 'color',
					desc = L["TColor_D"],
					get = function()
						local tc = XRS.db.profile.titlecolor
						return tc.r, tc.g, tc.b, tc.a
					end,
					set = "UpdateTitleColor",
					hasAlpha = true,
					order = 2,
				},
				border = {
					name = L["BOColor"], type = 'color',
					desc = L["BOColor_D"],
					get = function()
						local boc = XRS.db.profile.bordercolor
						return boc.r, boc.g, boc.b, boc.a
					end,
					set = "UpdateBorderColor",
					hasAlpha = true,
					order = 3,
				},
			},
			order = 5
		},
		updaterate = {
			name = L["Update_Rate"], type = "range",
			desc = L["Update_Rate_D"],
			get = function() return XRS.db.profile.UpdateRate end,
			set = "ModifyUpdateRate",
			min = 0.1,
			max = 1.5,
			step = 0.1,
			order = 6
		},
		lock = {
			name = L["Lock"], type = "toggle",
			desc = L["Lock_D"],
			get = function()
				return XRS.db.profile.Locked
			end,
			set = function(v)
				XRS.db.profile.Locked = v
			end,
			order = 7
		},
		hint = {
			name = L["Hint"], type = "toggle",
			desc = L["Hint_D"],
			get = function()
				return XRS.db.profile.ShowHint
			end,
			set = function(v)
				XRS.db.profile.ShowHint = v
			end,
			order = 8
		},
		new = {
			name = L["New_Group"], type = "group",
			desc = L["New_Group_D"],
			args = {
				bar = {
					name = L["Create_Bar"], type = "execute",
					desc = L["Create_Bar_D"],
					func = "CreateNewBar",
					order = 1
				},
				buff = {
					name = L["Create_Button"], type = "execute",
					desc = L["Create_Button_D"],
					func = "CreateNewBuffButton",
					order = 2
				},
			},
			order = 20,
			disabled = function() return GetNumRaidMembers() == 0 end,
		},
	}
}

local new, del
do
	local cache = setmetatable({},{__mode='k'})
	function new(...)
		local t = next(cache)
		if t then
			cache[t] = nil
			for i = 1, select("#", ...) do
				t[i] = select(i, ...)
			end
			return t
		else
			return {...}
		end
	end
	function del(t)
		if type(t) == "table" then
			for k in pairs(t) do t[k] = nil end
			setmetatable(t, nil)
			cache[t] = true
		end
		return nil
	end
end

local buffTable
local bars
local updateBars
local buffs

----------------------------
--      Main Functions    --
----------------------------

function XRS:OnInitialize()
	if not self.version then self.version = GetAddOnMetadata("XRS", "Version") end
	local rev = string.gsub(GetAddOnMetadata("XRS", "X-Build"), "%$Revision: (%d+) %$", "%1")
	self.version = self.version .. " |cffff8888r"..rev.."|r"

	-- Debugging state
	--self:SetDebugging(true)

	-- Register everything
	self:RegisterDB("XRSDB")
	self:RegisterDefaults("profile", XRS_DEFAULTS )	
	self:RegisterChatCommand("/xraidstatus", "/xrs", options, "XRS")

	media:Register(mType, "Diagonal", "Interface\\Addons\\XRS\\images\\Diagonal")
	media:Register(mType, "BantoBar", "Interface\\Addons\\XRS\\images\\BantoBar")
	media:Register(mType, "Skewed", "Interface\\Addons\\XRS\\images\\Skewed")
	media:Register(mType, "Smooth", "Interface\\Addons\\XRS\\images\\Smooth")
	media:Register(mType, "Otravi", "Interface\\Addons\\XRS\\images\\Otravi")
	media:Register(mType, "Charcoal", "Interface\\Addons\\XRS\\images\\Charcoal")
	media:Register(mType, "Rounded", "Interface\\Addons\\XRS\\images\\Rounded")
	media:Register(mType, "Granite", "Interface\\Addons\\XRS\\images\\Granite")
	media:Register(mType, "Waves", "Interface\\Addons\\XRS\\images\\Waves")
end

function XRS:OnEnable()
	-- Buff table
	buffTable = {
		sta = {
			GetSpellInfo(1243),
			(GetSpellInfo(21562)), -- The last value in the table needs the surrounding () to trim down to 1 return value
		},
		motw = {
			GetSpellInfo(1126),
			(GetSpellInfo(21849)),
		},
		ai = {
			GetSpellInfo(1459),
			(GetSpellInfo(23028)),
		},
		spi = {
			GetSpellInfo(14752),
			(GetSpellInfo(27681)),
		},
		sp = {
			GetSpellInfo(976),
			(GetSpellInfo(27683)),
		}
	}

	-- initialize necessary table for storing bars etc.
	bars = new()
	updateBars = new()
	buffs = new()

	-- Register event to update the raid
	self:RegisterEvent("RosterLib_RosterUpdated")

	-- One time event to check for a raid ...
	self:RegisterEvent("MEETINGSTONE_CHANGED", "MEETINGSTONE_CHANGED", true)
	self:MEETINGSTONE_CHANGED()
end

function XRS:OnDisable()
	-- no more events to handle
	self:LeaveRaid()
end

--[[--------------------------------------------------------------------------------
  Frame Creation 
-----------------------------------------------------------------------------------]]

function XRS:SetupFrames()
	-- Create Tooltip
	if not self.tooltip then
		self.tooltip = CreateFrame("GameTooltip", "XRSTooltip", UIParent, "GameTooltipTemplate")
	end

	-- Create XRS Frame
	self.frame = CreateFrame("Frame", "XRSFrame", UIParent)
	self.frame:EnableMouse(true)
	self.frame:SetFrameStrata("MEDIUM")
	self.frame:SetMovable(true)
	self.frame:SetWidth(130)
	self.frame:SetHeight(100)
	-- Create Font String
	self.xrsfs = self.frame:CreateFontString("$parentTitle","ARTWORK","GameFontNormal")
	self.xrsfs:SetText("XRaidStatus")
	self.xrsfs:SetPoint("TOP",0,-5)
	local tc = self.db.profile.titlecolor
	self.xrsfs:SetTextColor(tc.r,tc.g,tc.b,tc.a)
	self.xrsfs:Show()
	-- Backdrop options
	self.frame:SetBackdrop( { 
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", 
		edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", tile = true, tileSize = 16, edgeSize = 16, 
		insets = { left = 5, right = 5, top = 5, bottom = 5 }
	})
	local boc = self.db.profile.bordercolor
	self.frame:SetBackdropBorderColor(boc.r,boc.g,boc.b,boc.a)
	local bc = self.db.profile.backgroundcolor
	self.frame:SetBackdropColor(bc.r,bc.g,bc.b,bc.a)
	
	-- Create color gradient
	gradient = self.frame:CreateTexture(nil, "BORDER")
	gradient:SetTexture("Interface\\ChatFrame\\ChatFrameBackground")
	gradient:SetPoint("TOPLEFT", self.frame, "TOPLEFT", 4, -4)
	gradient:SetPoint("BOTTOMRIGHT", self.frame, "BOTTOMRIGHT", -4, 4)
	gradient:SetBlendMode("ADD")
	
	local cgstart = self.db.profile.colorgradientstart
	local cgstop = self.db.profile.colorgradientstop
	gradient:SetGradientAlpha("VERTICAL", cgstart.r, cgstart.g, cgstart.b, cgstart.a, cgstop.r, cgstop.g, cgstop.b, cgstop.a)
	if not self.db.profile.colorgradient then
		gradient:Hide()
	end
	self.framegradient = gradient
	
	self.frame:ClearAllPoints()
	self.frame:SetPoint("CENTER", UIParent, "CENTER", 0, 0)
	self.frame:SetScript("OnMouseDown",function()
		if ( arg1 == "LeftButton" ) then
			if not self.db.profile.Locked then
				this:StartMoving()
			end
		end
	end)
	self.frame:SetScript("OnMouseUp",function()
		if ( arg1 == "LeftButton" ) then
			this:StopMovingOrSizing()
			self:SavePosition()
		end
	end)
	self.frame:SetScript("OnHide",function() this:StopMovingOrSizing() end)
	self.frame:SetScript("OnShow",function() 
		for _,v in ipairs(bars) do
			if v:GetType() ~= "blank" then
				v:UpdateBar()
			end
		end
	end)

	-- Frame cannot be dragged off the screen
	self.frame:SetClampedToScreen(true)

	-- Loads the position of the frame
	self:LoadPosition()

	-- The scale from the db
	self:UpdateScale()

	self:Debug("XRS Frame created!")

	-- Create a button for raid leader options
	if (IsRaidLeader() or IsRaidOfficer()) then
		self:CreateLeaderMenu()
	end

	-- Create all bars and buffs
	self:SetupBars()
	self:SetupBuffs()
	self:SetWidth()
	
	dewdrop:Register(self.frame,
		'children', function()
			dewdrop:FeedAceOptionsTable(options)
		end)
end

--[[--------------------------------------------------------------------------------
  Switch Database
-----------------------------------------------------------------------------------]]

function XRS:OnProfileEnable()
	-- this is called every time your profile changes (after the change)
	for _,v in ipairs(bars) do
		v:RemoveBar()
	end
	for _,v in ipairs(buffs) do
		v:RemoveBuff()
	end

	bars = del(bars)
	buffs = del(buffs)

	bars = new()
	buffs = new()

	self:SetupBars()
	self:SetupBuffs()
end

--[[--------------------------------------------------------------------------------
  Event Handling
-----------------------------------------------------------------------------------]]

local healthEvents = {"UNIT_HEALTH", "UNIT_MAXHEALTH"}
local manaEvents = {"UNIT_MANA", "UNIT_MAXMANA"}
function XRS:RegisterRaidEvents()
	-- Register the WoW events
	self:RegisterBucketEvent(healthEvents, self.db.profile.UpdateRate, "UNIT_HEALTH")
	self:RegisterBucketEvent(manaEvents, self.db.profile.UpdateRate, "UNIT_MANA")

	self:Debug("Raid events registered")
end

function XRS:UnregisterRaidEvents()
	-- Unregister the WoW events
	self:UnregisterBucketEvent(healthEvents)
	self:UnregisterBucketEvent(manaEvents)
	
	self:Debug("Raid events unregistered")
end

function XRS:MEETINGSTONE_CHANGED()
	if( GetNumRaidMembers() > 0 ) then
		self:RosterLib_RosterUpdated()
	end
end

--[[--------------------------------------------------------------------------------
  Main processing
-----------------------------------------------------------------------------------]]

local grouped = nil
function XRS:RosterLib_RosterUpdated()
	if GetNumRaidMembers() > 0 then
		if not self.frame then self:SetupFrames() end
		if not grouped then
			grouped = true
			self:RegisterRaidEvents()
			self:ScheduleRepeatingEvent("barsID", self.FrequentBarsUpdate, self.db.profile.UpdateRate, self)
		end
		
		if not self.frame:IsVisible() then
			self.frame:Show()
		end
		
		if (IsRaidLeader() or IsRaidOfficer()) then
			self:CreateLeaderMenu()
		else
			self:RemoveLeaderMenu()
		end
		for _,v in ipairs(bars) do
			if v:GetType() ~= "blank" then
				v:UpdateBar()
			end
		end
	elseif grouped and GetNumRaidMembers() == 0 then
		self:LeaveRaid()
		self:UnregisterRaidEvents()
	end
end

function XRS:SetupBars()
	local bt = self.db.profile.barTable
	local texture = media:Fetch(mType, self.db.profile.Texture)

	for k,v in ipairs(bt) do
		-- get the data
		local d = bt[k]
		-- create the bar
		local bar = self.bar:new(k, d, texture)
		-- add the bar to the bartable
		tinsert(bars, bar)
	end

	self:RebuildClassBarTable()
end

function XRS:SetupBuffs()
	local bt = self.db.profile.buffTable

	for k,v in pairs(bt) do
		-- create the buff
		local buff = self.buff:new(k-1, v)
		-- add the buff to the bufftable
		tinsert(buffs, buff)
	end

	self:RebuildClassBarTable()
end

function XRS:RebuildClassBarTable()
	self.life = del(self.life)
	self.mana = del(self.mana)
	self.alive = del(self.alive)
	self.dead = del(self.dead)
	self.range = del(self.range)
	self.offline = del(self.offline)
	self.afk = del(self.afk)
	self.pvp = del(self.pvp)
	self.health_below = del(self.health_below)
	self.mana_below = del(self.mana_below)

	for k,v in ipairs(bars) do
		local classes = v:GetClasses()
		local w = v:GetType()
		
		-- Those types need a constant update, no event and no class specific
		if type(w) == "string" and (w == "range" or w == "afk" or w == "pvp") then
			if not self[w] then self[w] = new() end
			tinsert(self[w], v)
		elseif type(w) == "string" and w ~= "blank" then
			if not self[w] then self[w] = new() end
			-- add the bar to the class bar table (life, mana, ...) for later access
			for _,i in ipairs(classes) do
				i = i:lower()
				if not self[w][i] then self[w][i] = new() end
				tinsert(self[w][i], v)
			end
		end
		
		self:Debug("Bar added: "..w)
	end

	self:SetWidth()
end

function XRS:LeaveRaid()
	-- cancel the event to update the bars
	self:CancelScheduledEvent("barsID")
	-- hide the frame
	if self.frame then
		self.frame:Hide()
	end
	-- remove the leader menu (if there is one)
	self:RemoveLeaderMenu()

	-- disable all buff icons (own events)
	for k,v in ipairs(buffs) do
		v:Disable()
	end
	
	for k,v in ipairs(bars) do
		v:RemoveBar()
	end

	-- nil out everything
	bars = del(bars)
	buffs = del(buffs)

	bars = new()
	buffs = new()

	self.frame = nil
	self.xrsfs = nil
	
	grouped = nil
end

function XRS:COMBAT_START()
	self:SetVisual()
end

function XRS:COMBAT_STOP()
	self:RaidCheck()
	self:SetVisual()
end

function XRS:UNIT_HEALTH(units)
	for unit in pairs(units) do
		local class = select(2, UnitClass(unit))
		if not class then return end
		class = class:lower()
		if type(self.life) == "table" and self.life[class] then
			for _,v in ipairs(self.life[class]) do
				-- Update every bar which is affected by that event and class
				self:AddToQueue(v)
			end
		end
		
		if type(self.alive) == "table" and self.alive[class] then
			for _,v in ipairs(self.alive[class]) do
				self:AddToQueue(v)
			end
		end
		
		if type(self.dead) == "table" and self.dead[class] then
			for _,v in ipairs(self.dead[class]) do
				self:AddToQueue(v)
			end
		end
		
		if type(self.offline) == "table" and self.offline[class] then
			for _,v in ipairs(self.offline[class]) do
				self:AddToQueue(v)
			end
		end

		if type(self.health_below) == "table" and self.health_below[class] then
			for _,v in ipairs(self.health_below[class]) do
				-- Update every bar which is affected by that event and class
				self:AddToQueue(v)
			end
		end
	end
	
	self:UpdateNecessaryBars()
end

function XRS:UNIT_MANA(units)
	for unit in pairs(units) do
		local class = select(2, UnitClass(unit))
		if not class then return end
		class = class:lower()
		if type(self.mana) == "table" and self.mana[class] then
			for _,v in ipairs(self.mana[class]) do
				self:AddToQueue(v)
			end
		end
		if type(self.mana_below) == "table" and self.mana_below[class] then
			for _,v in ipairs(self.mana_below[class]) do
				self:AddToQueue(v)
			end
		end
	end
	
	self:UpdateNecessaryBars()
end

function XRS:SavePosition()
	local scale = self.frame:GetEffectiveScale()
	local worldscale = UIParent:GetEffectiveScale()
	
	local x,y = self.frame:GetLeft()*scale,self.frame:GetTop()*scale - (UIParent:GetTop())*worldscale

	if not self.db.profile.Position then
		self.db.profile.Position = {}
	end
	
	self.db.profile.Position.x = x
	self.db.profile.Position.y = y
end

function XRS:LoadPosition()
	if(self.db.profile.Position) then
		local x = self.db.profile.Position.x
		local y = self.db.profile.Position.y
		local scale = self.frame:GetEffectiveScale()
		
		self.frame:SetPoint("TOPLEFT", UIParent,"TOPLEFT", x/scale, y/scale)
	else
		self.frame:SetPoint("CENTER", UIParent, "CENTER")
	end
end

function XRS:UpdateScale()
	if self.frame then
		self.frame:SetScale(self.db.profile.Scale)
		self:LoadPosition()
	end
end

function XRS:CreateNewBar()
	local classes = {"Druid", "Shaman", "Paladin", "Priest", "Mage", "Warrior", "Rogue", "Warlock", "Hunter"}
	local pos = #self.db.profile.barTable + 1
	local name = "New bar <"..pos..">"
	local w = "life"

	-- Create temp table
	local tempTable = {}
	tempTable.name = name
	tempTable.c = classes
	tempTable.w = w

	-- Create the bar
	local bar = self.bar:new(pos, tempTable, self.db.profile.Texture)

	-- Add the temp table to the table in the db for saving
	tinsert(self.db.profile.barTable, pos, tempTable)
	tinsert(bars, bar)

	-- Update class bar table
	self:RebuildClassBarTable()
end

-- Delete the specified bar
function XRS:DeleteBar(bar)
	for k,v in ipairs(bars) do
		if v == bar then
			tremove(bars, k)
			tremove(self.db.profile.barTable, k)
		end
	end

	-- Update class bar table
	self:RebuildClassBarTable()

	-- Update visual position
	for k,v in ipairs(bars) do
		v:SetPosition(k)
	end
end

-- Move the bar up
function XRS:BarUp(bar)
	local pos = bar:GetPosition()
	local bt = self.db.profile.barTable

	if bar:GetPosition() == 1 then return end

	bars[pos], bars[pos-1] = bars[pos-1], bars[pos]
	bt[pos], bt[pos-1] = bt[pos-1], bt[pos]

	bars[pos]:SetPosition(pos)
	bars[pos-1]:SetPosition(pos-1)
end

-- Move the bar down
function XRS:BarDown(bar)
	local pos = bar:GetPosition()
	local bt = self.db.profile.barTable

	if bar:GetPosition() == #bars then return end

	bars[pos], bars[pos+1] = bars[pos+1], bars[pos]
	bt[pos], bt[pos+1] = bt[pos+1], bt[pos]

	bars[pos]:SetPosition(pos)
	bars[pos+1]:SetPosition(pos+1)
end

-- Create a button for raid leader options
function XRS:CreateLeaderMenu()
	if not self.leaderbutton then
		self.leaderbutton = CreateFrame("Button", nil, self.frame)
		self.leaderbutton:SetWidth(16)
		self.leaderbutton:SetHeight(16)
		self.leaderbutton:SetNormalTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollDown-Up")
		self.leaderbutton:SetPushedTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollDown-Down") 
		self.leaderbutton:SetHighlightTexture("Interface\\Buttons\\UI-Common-MouseHilight")
		self.leaderbutton:SetScript("OnClick",function() dewdrop:Open(self.leaderbutton) end)
		self.leaderbutton:ClearAllPoints()
		self.leaderbutton:SetPoint("TOPLEFT", self.frame, "TOPLEFT", 5, -5)
		
		-- Register dewdrop
		dewdrop:Register(self.leaderbutton, 'dontHook', true, 'children', function(level, value) self:CreateDDLeaderMenu(level, value) end)
		
		self.leaderbutton:Show()
		
		self:Debug("Leader Button created!")
	end
end

-- Removes the button if the player is not a raid leader
function XRS:RemoveLeaderMenu()
	if self.leaderbutton then
		self.leaderbutton:Hide()
		dewdrop:Unregister(self.leaderbutton)
		self.leaderbutton = nil
	end
end

function XRS:CreateDDLeaderMenu(level, value)
	-- Create drewdrop menu
	if level == 1 then
		dewdrop:AddLine( 'text', L["Raid_Leader_Options"], 'isTitle', true )
		dewdrop:AddLine( 'text', L["Buff_Check"],
						 'hasArrow', true,
						 'value', "bc",
						 'tooltipTitle', L["Buff_Check"],
						 'tooltipText', L["Buff_Check_TTText"]
					)
		dewdrop:AddLine( 'text', L["Ready_Check"],
						 'func', function()
							DoReadyCheck()
						 end,
						 'tooltipTitle', L["Ready_Check"],
						 'tooltipText', L["Ready_Check_TTText"]
					)

		dewdrop:AddLine( 'text', L["PromoteAll"],
						 'func', function()
							 if IsRaidLeader() then
								 for i=1, GetNumRaidMembers() do
									 local name = GetRaidRosterInfo(i)
									 PromoteToAssistant(name)
								 end
							 end
						 end,
						 'tooltipTitle', L["PromoteAll"],
						 'tooltipText', L["PromoteAll_TTText"]
					)
	elseif level == 2 then
		if value == "bc" then
			dewdrop:AddLine( 'text', L["Select_Buffs"],
						 'hasArrow', true,
						 'value', "buffs",
						 'tooltipTitle', L["Select_Buffs"],
						 'tooltipText', L["Select_Buffs_TTText"]
					)
			dewdrop:AddLine( 'text', L["Start_Check"],
						 'func', function()
							XRS:BuffCheck()
						 end,
						 'tooltipTitle', L["Start_Check"],
						 'tooltipText', L["Start_Check_TTText"]
					)
			dewdrop:AddLine( 'text', L["Select_Buffs_Twentyfive_Only"],
						 'checked', self.db.profile.buffcheck.twentyfive,
						 'func', function()
						 		self.db.profile.buffcheck.twentyfive = not self.db.profile.buffcheck.twentyfive
						 		self.db.profile.buffcheck.ten = false
						 end,
						 'tooltipTitle', L["Select_Buffs_Twentyfive_Only"],
						 'tooltipText', L["Select_Buffs_Twentyfive_Only_TTText"]
					)

			dewdrop:AddLine( 'text', L["Select_Buffs_Ten_Only"],
						 'checked', self.db.profile.buffcheck.ten,
						 'func', function()
						 		self.db.profile.buffcheck.ten = not self.db.profile.buffcheck.ten
						 		self.db.profile.buffcheck.twentyfive = false
						 end,
						 'tooltipTitle', L["Select_Buffs_Ten_Only"],
						 'tooltipText', L["Select_Buffs_Ten_Only_TTText"]
					)
		end
	elseif level == 3 then
		if value == "buffs" then
			dewdrop:AddLine( 'text', GetSpellInfo(1459),
							 'checked', self.db.profile.buffcheck.ai,
							 'func', function() self.db.profile.buffcheck.ai = not self.db.profile.buffcheck.ai end)
			dewdrop:AddLine( 'text', GetSpellInfo(1126),
							 'checked', self.db.profile.buffcheck.motw,
							 'func', function() self.db.profile.buffcheck.motw = not self.db.profile.buffcheck.motw end)
			dewdrop:AddLine( 'text', GetSpellInfo(1243),
							 'checked', self.db.profile.buffcheck.sta,
							 'func', function() self.db.profile.buffcheck.sta = not self.db.profile.buffcheck.sta end)
			dewdrop:AddLine( 'text', GetSpellInfo(14752),
							 'checked', self.db.profile.buffcheck.spi,
							 'func', function() self.db.profile.buffcheck.spi = not self.db.profile.buffcheck.spi end)
			dewdrop:AddLine( 'text', GetSpellInfo(976),
							 'checked', self.db.profile.buffcheck.sp,
							 'func', function() self.db.profile.buffcheck.sp = not self.db.profile.buffcheck.sp end)
		end
	end
end

-- Starts the buff check
function XRS:BuffCheck()
	if not IsRaidLeader() and not IsRaidOfficer() then return end
	local missingTable = new()

	local bt = buffTable

	for num=1,40 do
		local unitID = "raid"..num
		
		-- Limit buff checks to either groups 1-2 or groups 1-5, if enabled
		local checkThisUnit = true
		if self.db.profile.buffcheck.ten then
			local _, _, subgroup, _, _, _, _, _, _ = GetRaidRosterInfo(num)
			if (subgroup > 2) then
				checkThisUnit = false
			end
		elseif self.db.profile.buffcheck.twentyfive then
			local _, _, subgroup, _, _, _, _, _, _ = GetRaidRosterInfo(num)
			if (subgroup > 5) then
				checkThisUnit = false
			end
		end

		if UnitExists(unitID) and UnitIsConnected(unitID) and checkThisUnit then
			local raidname = UnitName(unitID)
			if self.db.profile.buffcheck.sta then
				if not missingTable.sta then missingTable.sta = new() end
				local b1 = SEA:UnitHasBuff(unitID, bt.sta[1])
				local b2 = SEA:UnitHasBuff(unitID, bt.sta[2])
				if (not (b1 or b2)) then tinsert(missingTable.sta, raidname) end
			end

			if self.db.profile.buffcheck.motw then
				if not missingTable.motw then missingTable.motw = new() end
				local b1 = SEA:UnitHasBuff(unitID, bt.motw[1])
				local b2 = SEA:UnitHasBuff(unitID, bt.motw[2])
				if (not (b1 or b2)) then tinsert(missingTable.motw, raidname) end
			end

			if self.db.profile.buffcheck.sp then
				if not missingTable.sp then missingTable.sp = new() end
				local b1 = SEA:UnitHasBuff(unitID, bt.sp[1])
				local b2 = SEA:UnitHasBuff(unitID, bt.sp[2])
				if (not (b1 or b2)) then tinsert(missingTable.sp, raidname) end
			end

			local _, class = UnitClass(unitID) 
			if (class ~= "WARRIOR" and class ~= "ROGUE") then
				if self.db.profile.buffcheck.spi then
					if not missingTable.spi then missingTable.spi = new() end
					local b1 = SEA:UnitHasBuff(unitID, bt.spi[1])
					local b2 = SEA:UnitHasBuff(unitID, bt.spi[2])
					if (not (b1 or b2)) then tinsert(missingTable.spi, raidname) end
				end

				if self.db.profile.buffcheck.ai then
					if not missingTable.ai then missingTable.ai = new() end
					local b1 = SEA:UnitHasBuff(unitID, bt.ai[1])
					local b2 = SEA:UnitHasBuff(unitID, bt.ai[2])
					if (not (b1 or b2)) then tinsert(missingTable.ai, raidname) end
				end
			end
		end
	end

	local count = 0
	for k,v in pairs(missingTable) do
		count = count + #missingTable[k]
	end

	if (count==0) then self:RaidOutput("XRS :: "..L["No Buffs Needed!"]) return end
	if (count>0) then self:RaidOutput("XRS :: "..count..L[" missing buffs."]) end

	for k,v in pairs(missingTable) do
		if #missingTable[k] > 0 then
			local msg = "<"..buffTable[k][1].."> : "
			msg = msg..table.concat(missingTable[k], ", ")
			self:RaidOutput(msg)
		end
		missingTable[k] = del(missingTable[k])
	end

	missingTable = del(missingTable)
end

-- Message output to the raid frame
function XRS:RaidOutput(msg)
	SendChatMessage(msg, "RAID")
end

-- Sets the bar textures of every bar
function XRS:SetBarTexture(name)
	self.db.profile.Texture = name
	local texture = media:Fetch(mType, name)
	for _,v in ipairs(bars)do
		v:SetBarTexture(texture)
	end
end

-- Updates the Background Color of the main frame
function XRS:UpdateBackgroundColor(r,g,b,a)
	local bc = self.db.profile.backgroundcolor
	bc.r, bc.g, bc.b, bc.a = r, g, b, a
	self.frame:SetBackdropColor(r,g,b,a)
end

-- Updates the Title Color
function XRS:UpdateTitleColor(r,g,b,a)
	local tc = self.db.profile.titlecolor
	tc.r, tc.g, tc.b, tc.a = r, g, b, a
	self.xrsfs:SetTextColor(r,g,b,a)
end

-- Updates the Border Color
function XRS:UpdateBorderColor(r,g,b,a)
	local boc = self.db.profile.bordercolor
	boc.r, boc.g, boc.b, boc.a = r, g, b, a
	self.frame:SetBackdropBorderColor(r,g,b,a)
end

function XRS:ModifyUpdateRate(rate)
	self.db.profile.UpdateRate = rate

	if GetNumRaidMembers() > 0 and self.frame then
		self:RegisterBucketEvent({"UNIT_HEALTH", "UNIT_MAXHEALTH"}, rate, "UNIT_HEALTH")
		self:RegisterBucketEvent({"UNIT_MANA", "UNIT_MAXMANA"}, rate, "UNIT_MANA")
		self:ScheduleRepeatingEvent("barsID", self.FrequentBarsUpdate, rate, self)
	end
end

-- Add the bar to the queue to update it later
function XRS:AddToQueue(bar)
	if not self:AlreadyQueued(bar) then updateBars[bar] = bar end
end

-- Checks if the bar is already queued for an update
function XRS:AlreadyQueued(bar)
	if updateBars[bar] then return true end
	return false
end

-- Update all bars where an event occured
--local inittime, initmem, endtime, endmem
function XRS:UpdateNecessaryBars()
	--inittime, initmem = GetTime(), gcinfo()
	if type(updateBars) == "table" then
		for _,v in pairs(updateBars) do
			v:UpdateBar()
		end
	end
	updateBars = del(updateBars)
	updateBars = new()
	--endtime, endmem = GetTime(), gcinfo()
	--self:Print(format("%s - %s (%s / %s)", endmem-initmem, endtime-inittime, initmem, endmem)) 
end

function XRS:FrequentBarsUpdate()
	if type(self.range) == "table" then
		for _,v in ipairs(self.range) do
			v:UpdateBar()
		end
	end
	if type(self.pvp) == "table" then
		for _,v in ipairs(self.pvp) do
			v:UpdateBar()
		end
	end
	if type(self.afk) == "table" then
		for _,v in ipairs(self.afk) do
			v:UpdateBar()
		end
	end
end

-- Create a new buff button
function XRS:CreateNewBuffButton()
	local classes = {"Druid", "Shaman", "Paladin", "Priest", "Mage", "Warrior", "Rogue", "Warlock", "Hunter"}
	local b = {GetSpellInfo(1459), (GetSpellInfo(23028))}

	-- Create temp table
	local tempTable = {}
	tempTable.c = classes
	tempTable.buffs = b

	local position = #buffs

	tinsert(self.db.profile.buffTable, tempTable)
	tinsert(buffs, self.buff:new(position, tempTable))

	-- Update class bar table
	self:RebuildClassBarTable()
end

-- Delete the specified bar
function XRS:DeleteBuff(buff)
	for k,v in ipairs(buffs) do
		if v == buff then
			tremove(buffs, k)
			tremove(self.db.profile.buffTable, k)
			break
		end
	end

	-- Update class bar table
	self:RebuildClassBarTable()

	-- Update visual position
	for k,v in ipairs(buffs) do
		v:SetPosition(k)
	end
end

function XRS:GetHintOption()
	return self.db.profile.ShowHint
end

function XRS:SetWidth()
	local width = self.db.profile.Width
	
	local barcount = #bars
	local buffscount = math.ceil(#buffs / math.floor(width/22))
	if self.frame then 
		self.frame:SetHeight(30 + (barcount * 16) + ((buffscount)*22))
		self.frame:SetWidth(width + 15)
	end
	
	if type(bars) == "table" then
		for _,v in ipairs(bars) do
			v:SetWidth(width)
		end
	end
	if type(buffs) == "table" then
		for _,v in ipairs(buffs) do
			v:SetWidth(width)
		end
	end
end

function XRS:IsColorGradient()
	return self.db.profile.colorgradient
end

function XRS:IsNotColorGradient()
	return not self.db.profile.colorgradient
end

function XRS:SetColorGradient(value)
	self.db.profile.colorgradient = value
	if value then
		self.framegradient:Show()
	else
		self.framegradient:Hide()
	end
end

function XRS:GetColorGradientStart()
	local cgs = self.db.profile.colorgradientstart
	return cgs.r, cgs.g, cgs.b, cgs.a
end

function XRS:SetColorGradientStart(r, g, b, a)
	local cgs = self.db.profile.colorgradientstart
	cgs.r, cgs.g, cgs.b, cgs.a = r, g, b, a
	self:UpdateColorGradient()
end

function XRS:GetColorGradientStop()
	local cgs = self.db.profile.colorgradientstop
	return cgs.r, cgs.g, cgs.b, cgs.a
end

function XRS:SetColorGradientStop(r, g, b, a)
	local cgs = self.db.profile.colorgradientstop
	cgs.r, cgs.g, cgs.b, cgs.a = r, g, b, a
	self:UpdateColorGradient()
end

function XRS:UpdateColorGradient()
	local cgstart = self.db.profile.colorgradientstart
	local cgstop = self.db.profile.colorgradientstop
	self.framegradient:SetGradientAlpha("VERTICAL", cgstart.r, cgstart.g, cgstart.b, cgstart.a, cgstop.r, cgstop.g, cgstop.b, cgstop.a)
end
