-- X-Perl UnitFrames
-- Author: Zek <Boodhoof-EU>
-- License: GNU GPL v3, 29 June 2007 (see LICENSE.txt)

local conf, hconf
XPerl_RequestConfig(function(new) conf = new hconf = new.hud end)

local L
if (XPerlHUDLoc) then
	local loc = XPerlHUDLoc
	L = function(s) return loc[s] or s end
else
	L = function(s) return s end
end

local playerClass
local hud = CreateFrame("Frame", "XPerlHUD", UIParent)
local ManaColours = {"mana", "rage", "focus", "energy"}
local cw, ch = 125, 25
local largeNumDiv = XPERL_LOC_LARGENUMDIV
local largeNumTag = XPERL_LOC_LARGENUMTAG

-- SetBarValue
local function SetBarValue(self, percent)
	-- We don't have an actual status bar, but rather a texture which resize and adjust the texcoords for
	-- 0 is the top of the bar textures
	-- 0.75390625 is the bottom
	-- Left and right are stored in self as tcl & tcr
	if (percent <= 0) then
		self:Hide()
	else
		self:Show()
		self:SetHeight(200 * percent)			-- 300 max height for 100%
		self:SetTexCoord(self.tcl, self.tcr, 0.75390625 * (1 - percent), 0.75390625)
	end
end

-- SmoothColour
local function SmoothColour(self, percentage)
	local r, g, b
	if (conf.colour.classic) then
		if (percentage < 0.5) then
			r = 1
			g = 2*percentage
			b = 0
		else
			g = 1
			r = 2*(1 - percentage)
			b = 0
		end
	else
		local c = conf.colour.bar
		r = c.healthEmpty.r + ((c.healthFull.r - c.healthEmpty.r) * percentage)
		g = c.healthEmpty.g + ((c.healthFull.g - c.healthEmpty.g) * percentage)
		b = c.healthEmpty.b + ((c.healthFull.b - c.healthEmpty.b) * percentage)
	end

	if (r >= 0 and g >= 0 and b >= 0 and r <= 1 and g <= 1 and b <= 1) then
		if (self.SetStatusBarColor) then
			self:SetStatusBarColor(r, g, b)
		else
			self:SetVertexColor(r, g, b)
		end
		if (self.bg) then
			self.bg:SetVertexColor(r, g, b, 0.25)
		end
	end
end

-- TargetDebuffCount
local function TargetDebuffCount(spellName, withDuration)
	for i = 1,40 do
		local name, rank, tex, count, debuffType, max, dur = UnitDebuff("target", i)
		if (not name) then
			break
		elseif (strfind(name, spellName) and (not withDuration or max)) then
			return count, tex, max, dur, spellName
		end
	end
	return 0
end

-- PlayerBuffCount()
local function PlayerBuffCount(sepllName)
	for i = 1,40 do
		local name, rank, tex, count, max, dur = UnitBuff("player", i)
		if (not name) then
			break
		elseif (name == spellName) then
			return count, tex, max, dur
		end
	end
	return 0
end

-- GetComboColour
local function GetComboColour(num)
	if (num == 5) then		return 1,   0, 0
	elseif (num == 4) then		return 1, 0.5, 0
	elseif (num == 3) then		return 1,   1, 0
	elseif (num == 2) then		return 0.5, 1, 0
	elseif (num == 1) then		return 0,   1, 0
	end
end

-- hud:Create()
function hud:Create()

	local list = {
		UNIT_NAME_UPDATE = function(self)
			local name = UnitName(self.partyid)
			self.text:SetText(name)
			if (self.partyid == "player") then
				XPerl_ColourFriendlyUnit(self.text, self.partyid)
			else
				XPerl_SetUnitNameColor(self.text, self.partyid)
			end
		end,

		UNIT_HEALTH = function(self)
			local hp, hpMax
			if (XPerl_Target_GetHealth) then
				hp, hpMax = XPerl_Target_GetHealth(self)
			else
				hp, hpMax = UnitHealth(self.partyid), UnitHealthMax(self.partyid)
			end

			SetBarValue(self.healthBar, min(1, max(0, hp / hpMax)))
			SmoothColour(self.healthBar, min(1, max(0, hp / hpMax)))

			self.healthPercent:SetFormattedText("%d%%", ((hp / hpMax) + 0.005) * 100)
			if (hpMax > 100000) then
				self.health:SetFormattedText("%.1f%s", hp / largeNumDiv, largeNumTag)
				self.healthMax:SetFormattedText("%.1f%s", hpMax / largeNumDiv, largeNumTag)
			else
				self.health:SetFormattedText("%d", hp)
				self.healthMax:SetFormattedText("%d", hpMax)
			end
		end,

		UNIT_MANA = function(self)
			local mana, manaMax = UnitMana(self.partyid), UnitManaMax(self.partyid)
			if (manaMax == 0) then
				self.manaBar:Hide()
				self.manaBar.bg:Hide()
				self.mana:Hide()
				self.manaMax:Hide()
			else
				self.manaBar:Show()
				self.manaBar.bg:Show()
				SetBarValue(self.manaBar, min(1, max(0, mana / manaMax)))
				if (manaMax > 100000) then
					self.mana:SetFormattedText("%.1f%s", mana / largeNumDiv, largeNumTag)
					self.manaMax:SetFormattedText("%.1f%s", manaMax / largeNumDiv, largeNumTag)
				else
					self.mana:SetText(mana)
					self.manaMax:SetText(manaMax)
				end
				self.mana:Show()
				self.manaMax:Show()
			end

			if (self == hud.player and hud.manaBar) then
				hud.manaBar:SetMinMaxValues(0, manaMax)
				hud.manaBar:SetValue(mana)
				hud.manaBar.text:SetFormattedText("%d/%d", mana, manaMax)
			end
		end,

		UNIT_DISPLAYPOWER = function(self)
			local p = UnitPowerType(self.partyid)
			if (p) then
				local c = conf.colour.bar[ManaColours[p + 1]]
				self.manaBar:SetVertexColor(c.r, c.g, c.b, 1)
				self.manaBarbg:SetVertexColor(c.r, c.g, c.b, 0.25)
				self.manaBar:Show()
				self.manaBarbg:Show()

				if (self == hud.player) then
					if (hud.InitTicker) then
						hud:InitTicker()
					end
					if (hud.manaBar) then
						hud.manaBar:SetStatusBarColor(c.r, c.g, c.b, 1)
						hud.manaBar.bg:SetVertexColor(c.r, c.g, c.b, 0.25)
					end
					hud:TickerShowHide()
				end
			else
				self.manaBar:Hide()
			end
		end,

		UNIT_AURA = function(self)
			hud:HideShow()
			if (self.partyid == "player") then
				hud:HideShow()
				if (hud.UpdatePlayerAuraCounters) then
					hud:UpdatePlayerAuraCounters()
				end
			else
				if (hud.UpdateTargetAuraCounters) then
					hud:UpdateTargetAuraCounters()
				end
			end
		end,

		UNIT_ATTACK_SPEED = function(self)
			hud:DoStats()
		end,

		UNIT_STATS = function(self)
			hud:DoStats()
		end,

		UNIT_MODEL_CHANGED = function(self)
			if (self == hud.target) then
				if (hconf.model and UnitExists("target")) then
					self.model:Show()
					self.model:SetUnit("target")
				else
					self.model:Hide()
				end
			else
				hud:GetWeaponSpeed()
				hud:DoStats()
			end
		end,

		UNIT_LEVEL = function(self)
			local l = UnitLevel(self.partyid)
			if (l == -1) then
				self.level:SetText("")
			else
				if (UnitIsPlusMob(self.partyid)) then
					self.level:SetFormattedText("%d+", l)
				else
					self.level:SetText(l)
				end
				local c = GetDifficultyColor(l)
				self.level:SetTextColor(c.r, c.g, c.b)
			end
		end,

		UNIT_FACTION = function(self)
			if (UnitIsPVP(self.partyid)) then
				self.pvpIcon:SetTexture("Interface\\TargetingFrame\\UI-PVP-"..(UnitFactionGroup(self.partyid) or "FFA"))
				self.pvpIcon:Show()
			else
				self.pvpIcon:Hide()
			end
			self:UNIT_NAME_UPDATE()
		end,

		DrawClass = function(self)
			if (UnitIsPlayer(self.partyid)) then
				local class = select(2, UnitClass(self.partyid))
				local l, r, t, b = XPerl_ClassPos(class)
				self.class:SetTexCoord(l + 0.02, r - 0.02, t + 0.02, b - 0.02)
				self.class:Show()
			else
				self.class:Hide()
			end
		end,

		UNIT_ENERGY = function(self)
			self:UNIT_MANA()
			if (self == hud.player) then
				local e = UnitMana("player")
				if (e == self.EnergyLast + 20 and hud.energyTicker) then
					hud.energyTicker.EnergyTime = GetTime()
				end
				self.EnergyLast = e

				hud:TickerShowHide()
			end
		end,

		UNIT_SPELLCAST_SUCCEEDED = function(self, spell, rank)
			if (self == hud.player) then
				if (spell == L("Summon Water Elemental")) then
					hud:SpecialCountdown(L("Water Elemental"), "Interface\\Icons\\Spell_Frost_SummonWaterElemental_2", 45)
				end
			end
		end
	}

	list.UNIT_MAXHEALTH = list.UNIT_HEALTH
	list.UNIT_MAXMANA = list.UNIT_MANA
	list.UNIT_MAXENERGY = list.UNIT_MANA
	list.UNIT_RAGE = list.UNIT_MANA
	list.UNIT_MAXRAGE = list.UNIT_MANA
	list.UNIT_FOCUS = list.UNIT_MANA
	list.UNIT_MAXFOCUS = list.UNIT_MANA

	list.UpdateDisplay = function(self)
		self:UNIT_FACTION()
		self:UNIT_LEVEL()
		self:UNIT_HEALTH()
		self:UNIT_DISPLAYPOWER()
		self:UNIT_MANA()
		self:UNIT_MODEL_CHANGED()
		self:UNIT_AURA()
		self:DrawClass()
	end

	local function Create1(t)
		local frame = CreateFrame("Frame", self:GetName()..t, self, "XPerl_HUDUnitTemplate")
		XPerl_SetChildMembers(frame)
		frame.healthBar.bg = frame.healthBarbg
		frame.manaBar.bg = frame.manaBarbg

		frame.healthBar:ClearAllPoints()
		frame.healthBar.bg:ClearAllPoints()
		frame.manaBar:ClearAllPoints()
		frame.manaBarbg:ClearAllPoints()
		frame.pvpIcon:ClearAllPoints()
		frame.raidIcon:ClearAllPoints()
		frame.class:ClearAllPoints()
		frame.level:ClearAllPoints()
		frame.text:ClearAllPoints()
		frame.health:ClearAllPoints()
		frame.healthMax:ClearAllPoints()
		frame.mana:ClearAllPoints()
		frame.manaMax:ClearAllPoints()

		frame.text:SetWidth(90)
		frame.text:SetHeight(90)
		frame.text:SetJustifyV("TOP")

		local BOTTOMLEFT, BOTTOMRIGHT, TOPRIGHT, TOPLEFT, LEFT, RIGHT, mod
		if (t == "player") then
			LEFT = "LEFT"
			RIGHT = "RIGHT"
			mod = -1
		else
			LEFT = "RIGHT"
			RIGHT = "LEFT"
			mod = 1
		end
		TOPLEFT = "TOP"..LEFT
		BOTTOMLEFT = "BOTTOM"..LEFT
		TOPRIGHT = "TOP"..RIGHT
		BOTTOMRIGHT = "BOTTOM"..RIGHT

		frame:SetPoint(BOTTOMRIGHT, UIParent, "CENTER", 100 * mod, -130)
		frame.healthBar:SetPoint(BOTTOMRIGHT, 0, 0)
		frame.healthBar.bg:SetPoint(BOTTOMRIGHT, 0, 0)
		frame.manaBar:SetPoint(BOTTOMLEFT, 0, 0)
		frame.manaBar.bg:SetPoint(BOTTOMLEFT, 0, 0)

		frame.level:SetPoint(TOPRIGHT, frame, BOTTOMRIGHT, 15 * mod, 0)
		frame.text:SetPoint(TOPRIGHT, frame.level, TOPLEFT, 2 * mod, 0)
		frame.text:SetJustifyH(RIGHT)

		frame.health:SetPoint(BOTTOMLEFT, frame, BOTTOMRIGHT, 10 * mod, 2)
		frame.healthMax:SetPoint(TOPLEFT, frame, TOPRIGHT, 0, 0)
		frame.healthPercent:SetPoint(BOTTOMRIGHT, frame, BOTTOMRIGHT, 15 * mod, 3)
		frame.mana:SetPoint(BOTTOMRIGHT, frame, BOTTOMLEFT, 0, 2)
		frame.manaMax:SetPoint(TOPRIGHT, frame, TOPLEFT, -30 * mod, 0)

		frame.stats = frame:CreateFontString(nil, "BACKGROUND", "GameFontNormalSmall")
		frame.stats:SetTextColor(0.7, 0.7, 0.7)
		frame.stats:SetPoint(TOPRIGHT, frame.manaMax, BOTTOMRIGHT, 0, -10)
		frame.stats:SetJustifyH(RIGHT)
		frame.stats:SetJustifyV("TOP")

		frame.class:SetPoint(TOPRIGHT, frame.level, BOTTOMRIGHT, 0, -2)
		frame.pvpIcon:SetPoint(TOPRIGHT, frame.manaMax, BOTTOMRIGHT, 0, 0)
		frame.raidIcon:SetPoint(BOTTOMLEFT, frame.health, TOPLEFT, 0, 0)

		if (t == "player") then
			frame.healthBar.tcl = 0.23828125
			frame.healthBar.tcr = 0.5
			frame.manaBar.tcl = 0
			frame.manaBar.tcr = 0.19140625
		else
			frame.healthBar.tcl = 0.5
			frame.healthBar.tcr = 0.7578125
			frame.manaBar.tcl = 0.80859375
			frame.manaBar.tcr = 1
		end

		frame.healthBar.bg.tcl = frame.healthBar.tcl
		frame.healthBar.bg.tcr = frame.healthBar.tcr
		frame.manaBar.bg.tcl = frame.manaBar.tcl
		frame.manaBar.bg.tcr = frame.manaBar.tcr

		SetBarValue(frame.healthBar.bg, 1)
		SetBarValue(frame.manaBar.bg, 1)

		frame:SetAttribute("unit", t)
		frame.partyid = t

		for k,v in pairs(list) do
			frame[k] = v
		end

		return frame
	end

	self.player = Create1("player")
	self.target = Create1("target")
	self.counters = {}

	self.player.EnergyLast = 0

	self.counterAnchor = {"TOP", UIParent, "CENTER", 0, -130}

	if (playerClass == "ROGUE" or playerClass == "DRUID" or playerClass == "WARRIOR" or playerClass == "MAGE" or playerClass == "PRIEST" or playerClass == "WARLOCK") then
		self.combo = CreateFrame("Frame", nil, self)
		self.combo:SetWidth(cw)
		self.combo:SetHeight(ch)
		self.combo:SetPoint(unpack(self.counterAnchor))

		self.combo.cell = {}
		for i = 1,5 do
			local c = self.combo:CreateTexture(nil, "BACKGROUND")
			self.combo.cell[i] = c
			c:SetTexture("Interface\\Addons\\XPerl_HUD\\Images\\HudElements")
			c:SetHeight(ch)
			if (i == 1) then
				c:SetWidth(cw / 4)
				c:SetTexCoord(0, 0.125, 0.875, 1)
				c:SetPoint("LEFT", 0, 0)
			elseif (i == 5) then
				c:SetWidth(cw / 4)
				c:SetTexCoord(0.375, 0.5, 0.875, 1)
				c:SetPoint("RIGHT", 0, 0)
			else
				c:SetWidth(cw / 3.15)
				c:SetTexCoord(0.16796875, 0.328125, 0.875, 1)
				c:SetPoint("LEFT", (cw / 7.4) + ((cw / 4.86) * (i - 2)), 0)
			end
		end

		self.combo.SetValue = function(self, val)
			local r, g, b = GetComboColour(val)
			for i = 1,5 do
				if (i > val) then
					self.cell[i]:Hide()
				else
					self.cell[i]:SetVertexColor(r, g, b)
					self.cell[i]:Show()
				end
			end
		end

		self.combo:SetValue(0)
		self.counterAnchor[3] = "BOTTOM"
		self.counterAnchor[2] = self.combo
		if (playerClass ~= "ROGUE") then
			self.counterAnchor[5] = 0
			self.combo.timer = self:CreateCounter()
			self.combo.timer:Hide()
			self.combo.timer.icon:ClearAllPoints()		-- Make the icon span the bar and the combo area
			self.combo.timer.icon:SetPoint("BOTTOMRIGHT", self.combo.timer, "BOTTOMLEFT", 0, 0)
			self.combo.timer.icon:SetHeight(ch + hconf.counterHeight)
			self.combo.timer.icon:SetWidth(ch + hconf.counterHeight)
		else
			self.counterAnchor[5] = -5
		end
	end

	if (playerClass == "ROGUE" or playerClass == "WARRIOR" or class == "DRUID") then
		self.manaBar = CreateFrame("StatusBar", nil, self)
		self.manaBar:SetMinMaxValues(0, UnitManaMax("player"))
		self.manaBar:SetValue(0)
		self.manaBar:SetHeight(10)
		self.manaBar:SetWidth(cw)
		self.manaBar:SetPoint("BOTTOM", self.combo, "TOP", 0, (playerClass ~= "WARRIOR" and 2) or 0)

		local p = UnitPowerType("player")
		local c = conf.colour.bar[ManaColours[p + 1]]

		self.manaBar.bg = self.manaBar:CreateTexture(nil, "BACKGROUND")
		self.manaBar.bg:SetAllPoints()
		self.manaBar:SetStatusBarTexture(XPerl_GetBarTexture())
		self.manaBar.bg:SetTexture(XPerl_GetBarTexture())

		self.manaBar:SetStatusBarColor(c.r, c.g, c.b, 1)
		self.manaBar.bg:SetVertexColor(c.r, c.g, c.b, 0.25)

		self.manaBar.text = self.manaBar:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
		self.manaBar.text:SetAllPoints()
		self.manaBar.text:SetTextColor(1, 1, 1)
	end

	if (self.doHasteBar) then
		self.hasteBar = CreateFrame("StatusBar", nil, self)
		self.hasteBar:SetMinMaxValues(0, 100)
		self.hasteBar:SetValue(0)
		self.hasteBar:SetHeight(10)
		self.hasteBar:SetWidth(cw)
		if (self.combo) then
			self.hasteBar:SetPoint("TOP", self.combo, "BOTTOM", 0, 0)
		else
			self.hasteBar:SetPoint("TOP", UIParent, "CENTER", 0, -165)
		end
		self.hasteBar.text = self.hasteBar:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
		self.hasteBar.text:SetAllPoints()
		self.hasteBar.text:SetTextColor(0.7, 0.7, 0.7)

		self.hasteBar.bg = self.hasteBar:CreateTexture(nil, "BACKGROUND")
		self.hasteBar.bg:SetAllPoints()
		self.hasteBar:SetStatusBarTexture(XPerl_GetBarTexture())
		self.hasteBar.bg:SetTexture(XPerl_GetBarTexture())

		self.counterAnchor[3] = "BOTTOM"
		self.counterAnchor[2] = self.hasteBar
		self.counterAnchor[5] = -5
	end

	-- self.counterAnchor = {"TOP", UIParent, "CENTER", 0, -140}

	if (playerClass == "ROGUE") then
		self.UpdatePlayerAuraCounters = function(self)
			self:ShowCountdown((GetSpellInfo(6774)))						-- Slice and Dice
			self:ShowCountdown((GetSpellInfo(34775)))						-- Haste - From DST
			self:ShowCountdown((GetSpellInfo(37171)))						-- Coup de Grace
			self:ShowCountdown((GetSpellInfo(31242)))						-- Find Weakness
			self:ShowCountdown((GetSpellInfo(40461)))						-- Exploit Weakness
			self:ShowCountdown((GetSpellInfo(37174)))						-- Perceived Weakness
		end
	elseif (playerClass == "WARRIOR") then
		self.UpdateTargetAuraCounters = function(self)
			self:ShowComboBar(TargetDebuffCount(GetSpellInfo(30901)))	-- Sunder Armor
		end
	elseif (playerClass == "MAGE") then
		self.UpdateTargetAuraCounters = function(self)
			if (not self:ShowComboBar(TargetDebuffCount(GetSpellInfo(22959)))) then			-- Fire Vulnerability
				if (not self:ShowComboBar(TargetDebuffCount(GetSpellInfo(28595)))) then		-- Winter's Chill
					self:ShowComboBar(TargetDebuffCount(GetSpellInfo(38636)))				-- Scorch
				end
			end
		end
	elseif (playerClass == "WARLOCK" or playerClass == "PRIEST") then
		self.UpdateTargetAuraCounters = function(self)
			self:ShowComboBar(TargetDebuffCount(GetSpellInfo(15258)))	-- Shadow Vulnerability
		end
	elseif (playerClass == "PALADIN") then
		self.UpdateTargetAuraCounters = function(self)
			self:ShowCountdown(L("^Judgement of "), true)
		end
	elseif (playerClass == "PRIEST") then
		self.UpdatePlayerAuraCounters = function(self)
			self:ShowCountdown((GetSpellInfo(46193)))						-- Power Word: Shield
		end
		self.UpdateTargetAuraCounters = function(self)
			self:ShowCountdown((GetSpellInfo(15290)))						-- Vampiric Embrace
		end
	end

	if (playerClass ~= "WARRIOR") then
		self:InitTicker()
	else
		self.InitTicker = nil
		self.MakeTicker = nil
		self.TickerShowHide = function() end
	end

	self.target.model = CreateFrame("PlayerModel", nil, self.target)
	self.target.model:SetHeight(120)
	self.target.model:SetWidth(80)
	self.target.model:SetPoint("TOP", self.target, "TOPRIGHT", 0, 0)

	self:SetAlpha(0.7)
	self.Create = nil
end

-- CounterOnUpdate
local function CounterOnUpdate(self, elapsed)
	local Min, Max = self:GetMinMaxValues()
	local v = self:GetValue() - elapsed

	if (v < 0) then
		self:Hide()
	else
		self:SetValue(v)
		SmoothColour(self, min(1, max(0, v / Max)))
		if (v < 1) then
			self.text:SetFormattedText("%.1f", v)
		else
			self.text:SetFormattedText("%d", v)
		end
	end
end

-- hud:CreateCounter(tex)
function hud:CreateCounter(tex)
	local barTex = XPerl_GetBarTexture()
	local c = CreateFrame("StatusBar", nil, self)
	c:SetWidth(cw)
	c:SetHeight(hconf.counterHeight)
	c:SetPoint(unpack(self.counterAnchor))
	self.counterAnchor[3] = "BOTTOM"
	self.counterAnchor[2] = c
	self.counterAnchor[5] = 0
	c:SetStatusBarColor(0.8, 0.8, 0.4)
	c:SetStatusBarTexture(barTex)
	c:SetScript("OnUpdate", CounterOnUpdate)

	c.text = c:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
	c.text:SetPoint("LEFT", 2, 0)

	c.bg = c:CreateTexture(nil, "BACKGROUND")
	c.bg:SetTexture(barTex)
	c.bg:SetAllPoints(c)

	c.icon = c:CreateTexture(nil, "BORDER")
	c.icon:SetWidth(hconf.counterHeight)
	c.icon:SetHeight(hconf.counterHeight)
	c.icon:SetPoint("RIGHT", c, "LEFT", 0, 0)
	if (tex) then
		c.icon:SetTexture(tex)
	end

	return c
end

-- OnEvent
function hud:OnEvent(event, a, b, c, d)
	if (strfind(event, "^UNIT_")) then
		local f = self[a]
		if (f) then
			if (not f[event]) then
				error("No handler for event '"..event.."'")
			else
				f[event](f, b, c, d)
			end
		end
	else
		if (not self[event]) then
			error("No handler for event '"..event.."'")
		else
			self[event](self, a, b, c, d)
		end
	end
end

-- SpecialCountdown
function hud:SpecialCountdown(spellName, icon, duration)
	local counter = self.counters[spellName]
	if (not counter) then
		counter = self:CreateCounter(tex)
		self.counters[spellName] = counter
	end

	counter.icon:SetTexture(icon)
	counter:SetMinMaxValues(0, duration)
	counter:SetValue(duration)
	counter:Show()
end

-- GetCounter
function hud:GetCounter(spellName)
	local counter = self.counters[spellName]
	if (not counter) then
		counter = self:CreateCounter(tex)
		self.counters[spellName] = counter
	end
	return counter
end

-- ShowCountdown
function hud:ShowCountdown(spellName, debuff)
	if (not duration) then
		for i = 1,40 do
			local name, rank, count, Type, dur, max, tex
			if (debuff) then
				name, rank, tex, count, Type, max, dur = UnitDebuff("target", i)
			else
				name, rank, tex, count, max, dur = UnitBuff("player", i)
			end
			if (not name) then
				break
			elseif (max and name == spellName) then
				local counter = self:GetCounter(spellName)
				counter.icon:SetTexture(tex)
				counter:SetMinMaxValues(0, max)
				counter:SetValue(dur)
				counter:Show()
				return
			end
		end
	end

	local counter = self.counters[spellName]
	if (counter) then
		counter:Hide()
	end
end

-- PLAYER_TARGET_CHANGED
function hud:PLAYER_TARGET_CHANGED()
	if (not UnitExists("target")) then
		self.target:Hide()
		if (self.combo) then
			self.combo:Hide()
			if (self.combo.timer) then
				self.combo.timer:Hide()
			end
		end
	else
		self.target:Show()
		self.target:UpdateDisplay()
	end
	hud:RAID_TARGET_UPDATE()
end

-- hud:GetWeaponSpeed()
function hud:GetWeaponSpeed()
	if (not self.tooltip) then
		self.tooltip = CreateFrame("GameTooltip", self:GetName().."Tooltip", nil, "GameTooltipTemplate")
		self.tooltip:SetOwner(WorldFrame, "ANCHOR_NONE")
	end

	if (self.doRanged) then
		self.tooltip:SetInventoryItem("player", 18)
	else
		self.tooltip:SetInventoryItem("player", 16)
	end
	for i = 2,10 do
		local line = getglobal(self.tooltip:GetName().."TextRight"..i)
		if (line) then
			local text = line:GetText() or ""
			if (text) then
				local speed = strmatch(text, "^"..WEAPON_SPEED.." (%d.+)")
				if (speed) then
					self.mhWeaponSpeed = tonumber(speed)
					if (self.mhWeaponSpeed < 0 or self.mhWeaponSpeed > 10) then
						self.mhWeaponSpeed = 1.8
					else
						return
					end
				end
			end
		end
	end

	self.mhWeaponSpeed = 1.8			-- ZOMG!
end

-- hud:GetStats()
function hud:GetHaste()
	local mhSpeed, ohSpeed, minHit, maxHit, minHasteDamage, maxHasteDamage, minHasteOffHandDamage, maxHasteOffHandDamage, physicalBonusPos, physicalBonusNeg, dmgPercent
	if (self.doRanged) then
		mhSpeed, minHit, maxHit, physicalBonusPos, physicalBonusNeg, dmgPercent = UnitRangedDamage("player")
	else
		mhSpeed, ohSpeed = UnitAttackSpeed("player")
		minHasteDamage, maxHasteDamage, minHasteOffHandDamage, maxHasteOffHandDamage, physicalBonusPos, physicalBonusNeg, dmgPercent = UnitDamage("player")
	end

	return ((100 * self.mhWeaponSpeed / mhSpeed) - 100) + 0.005, physicalBonusPos, physicalBonusNeg, dmgPercent
end

-- OnHasteUpdate
function hud:OnHasteUpdate(elapsed)
	local val = self:GetValue()

	val = val + (self.movementDiff * (elapsed * 2))
	if ((self.increasing and val > self.targetValue) or (not self.increasing and val < self.targetValue)) then
		val = self.targetValue
		self:SetScript("OnUpdate", nil)
	end

	self:SetValue(val)

	SmoothColour(self, val / 100)

	if (val == 0) then
		self:Hide()
	end
end

local StatStr = {"STR", "AGI", "STA", "INT", "SPR"}
-- DoStats
function hud:DoStats()
	local stats = self.player.stats
	local s, stat, effectiveStat
	local base, posBuff, negBuff
	local power = UnitPowerType("player")

	local haste, physicalBonusPos, physicalBonusNeg, dmgPercent = self:GetHaste()

	if (self.doRanged) then
		base, posBuff, negBuff = UnitRangedAttackPower("player")
	else
		base, posBuff, negBuff = UnitAttackPower("player")
	end

	local ap = max((base + posBuff + negBuff), 0)

	s = format("AP %s%d|r", (negBuff < 0 and "|c00FF0000") or (posBuff > 0 and "|c0000FF00") or "", ap)

	for i = 1,5 do
		if (power > 0 and i > 3) then
			break
		end

		stat, effectiveStat, posBuff, negBuff = UnitStat("player", i)
		s = format("%s\r%s %s%d|r", s, StatStr[i], (negBuff ~= 0 and "|c00FF0000") or (posBuff ~= 0 and "|c0000FF00") or "", effectiveStat)
	end

	if (physicalBonusPos ~= 0 or physicalBonusNeg ~= 0 or dmgPercent ~= 1) then
		s = format("%s\r%s", s, MODIFIERS_COLON, physicalBonusPos)
		if (physicalBonusPos ~= 0) then
			s = format("%s |c00FFFF80+%d|r", s, physicalBonusPos)
		end
		if (physicalBonusNeg ~= 0) then
			s = format("%s |c00FF8080-%d|r", s, physicalBonusNeg)
		end
		if (dmgPercent ~= 1) then
			s = format("%s %s+%d%%|r", s, (dmgPercent >= 1 and "|c0080FF80") or "|c00FF8080", ((dmgPercent + 0.005) * 100) - 100)
		end
	end

	stats:SetText(s)

	-- Haste Bar
	if (self.hasteBar) then
		self.hasteBar.curValue = self.hasteBar:GetValue()
		self.hasteBar.targetValue = min(100, haste)
		self.hasteBar.movementDiff = (self.hasteBar.targetValue - self.hasteBar.curValue)
		self.hasteBar.increasing = self.hasteBar.targetValue > self.hasteBar.curValue

		if (self.hasteBar.curValue ~= self.hasteBar.targetValue) then
			self.hasteBar:Show()
			self.hasteBar:SetScript("OnUpdate", hud.OnHasteUpdate)
		else
			SmoothColour(self.hasteBar, haste / 100)
			if (haste == 0) then
				self.hasteBar:Hide()
			else
				self.hasteBar:Show()
			end
		end

		self.hasteBar.text:SetFormattedText("%.2f%%", haste)
	end
end

-- ShowComboBar
function hud:ShowComboBar(cp, tex, max, dur, spellName)
	if (cp > 0) then
		self.combo:SetValue(cp)
		self.combo:Show()

		if (self.combo.timer) then
			if (max and max > 0) then
				self.combo.timer.icon:SetTexture(tex)
				self.combo.timer:SetMinMaxValues(0, max)
				self.combo.timer:SetValue(dur)
				self.combo.timer:Show()
			else
				self.combo.timer:Hide()
			end
		end
	else
		self.combo:Hide()
		if (self.combo.timer) then
			self.combo.timer:Hide()
		end
	end
end

-- RAID_TARGET_UPDATE
function hud:RAID_TARGET_UPDATE()
	XPerl_Update_RaidIcon(self.player.raidIcon, "player")
	XPerl_Update_RaidIcon(self.target.raidIcon, "target")
end

-- PLAYER_COMBO_POINTS
function hud:PLAYER_COMBO_POINTS()
	self:ShowComboBar(GetComboPoints())
end

-- PLAYER_REGEN_ENABLED
function hud:PLAYER_REGEN_ENABLED()
	self:HideShow()
end

-- PLAYER_REGEN_DISABLED
hud.PLAYER_REGEN_DISABLED = hud.PLAYER_REGEN_ENABLED
hud.PLAYER_UPDATE_RESTING = hud.PLAYER_REGEN_ENABLED
hud.PLAYER_CONTROL_LOST = hud.PLAYER_REGEN_ENABLED
hud.PLAYER_CONTROL_GAINED = hud.PLAYER_REGEN_ENABLED
hud.PLAYER_AURAS_CHANGED = hud.PLAYER_REGEN_ENABLED

-- UPDATE_STEALTH
function hud:UPDATE_STEALTH()
	self:TickerShowHide()
end

-- hud:PLAYER_ENTERING_WORLD()
function hud:PLAYER_ENTERING_WORLD()
	self.player:UpdateDisplay()
	self:PLAYER_TARGET_CHANGED()
	self:HideShow()
	self:GetWeaponSpeed()
	self:DoStats()
end

-- UpdateDisplay
hud.UpdateDisplay = hud.PLAYER_ENTERING_WORLD

-- HideShow
function hud:HideShow()
	if (hconf.enabled) then
if (ZekTest) then
	self:Show()
	self:SetAlpha(1)
	return
end
		if (InCombatLockdown()) then
			self:SetAlpha(0.7)
			self:Show()			-- Always show in-combat
			return
		end
		self:SetAlpha(0.2)

		if (not IsResting() and not UnitOnTaxi("player") and not IsMounted()) then
			self:Show()
			return
		end
	end

	self:Hide()
end

-- XPerl_Player_Energy_TickWatch
function XPerl_Player_Energy_OnUpdate(self, elapsed)
	local sparkPosition
	if (self.fsrEnabled) then
		local val = self:GetValue() + elapsed
		if (val > 5) then
			self:Hide()
			self.fsrEnabled = false
		else
			self:SetValue(val)
			sparkPosition = self:GetWidth() * (val / 5)
		end
	else
		local remainder = (GetTime() - self.EnergyTime) % 2
		self:SetValue(remainder)
		sparkPosition = self:GetWidth() * (remainder / 2)
	end

	self.spark:SetPoint("CENTER", self, "LEFT", sparkPosition, 0)
end

-- hud:MakeTicker()
function hud:MakeTicker()
	local c = CreateFrame("StatusBar", nil, self)
	self.energyTicker = c
	c:SetWidth(cw)
	c:SetHeight(2)
	c:SetStatusBarTexture("Interface\\Addons\\XPerl\\Images\\XPerl_FrameBack")
	c:SetScript("OnUpdate", TickerOnUpdate)

	c.spark = c:CreateTexture(nil, "OVERLAY")
	c.spark:SetTexture("Interface\\CastingBar\\UI-CastingBar-Spark")
	c.spark:SetBlendMode("ADD")
	c.spark:SetWidth(12)
	c.spark:SetHeight(12)
	c.spark:SetPoint("CENTER", 0, 0)

	c.EnergyTime = 0
	c:SetScript("OnUpdate", XPerl_Player_Energy_OnUpdate)

	self.MakeTicker = nil

	return c
end

-- hud:InitTicker()
function hud:InitTicker()
	local t = self.energyTicker
	if (not t) then
		t = self:MakeTicker()
	end
	t:SetPoint("TOP", self.manaBar, "BOTTOM", 0, 0)

	local p = UnitPowerType("player")
	if (p == 0) then
		t:Hide()
		t:SetStatusBarColor(0.4, 0.7, 1)
		t.spark:SetVertexColor(0.4, 0.7, 1)
		t:SetMinMaxValues(0, 5)
	elseif (p == 3) then
		t:SetStatusBarColor(1, 1, 0.5)
		t.spark:SetVertexColor(1, 1, 0.5)
		t:SetMinMaxValues(0, 2)
	else
		t:Hide()
	end
end

-- XPerl_Player_TickerShowHide
function hud:TickerShowHide()
	local f = self.energyTicker
	if (f) then
       		local t = UnitPowerType("player")
		local show

		if (not UnitIsDeadOrGhost("player")) then
			if (t == 3) then
	       			local e, m = UnitMana("player"), UnitManaMax("player")
				show = (e < m or UnitAffectingCombat("player") or IsStealthed()) and not UnitIsDeadOrGhost("player")
			end
		end

       		if (show) then
			if (t == 0) then
				f:SetStatusBarColor(0.4, 0.7, 1)
			else
				f:SetStatusBarColor(1, 1, 0.5)
			end
       			f:Show()
       		else
       			f:Hide()
       		end
	end
end

-- Setup
function hud:Setup()

	if (XPerlDB) then
		if (not XPerlDB.hud ) then
			XPerlDB.hud = {enabled = 1}
		end
		hconf = XPerlDB.hud
		conf = XPerlDB
		if (not hconf.counterHeight) then
			hconf.counterHeight = 10
		end
	end

	local events = {"UNIT_NAME_UPDATE", "UNIT_HEALTH", "UNIT_MAXHEALTH", "UNIT_MANA", "UNIT_MAXMANA", "UNIT_FACTION",
			"UNIT_ENERGY", "UNIT_MAXENERGY", "UNIT_RAGE", "UNIT_MAXRAGE", "UNIT_FOCUS", "UNIT_MAXFOCUS", "UNIT_LEVEL",
			"PLAYER_TARGET_CHANGED", "PLAYER_COMBO_POINTS", "PLAYER_REGEN_ENABLED", "PLAYER_REGEN_DISABLED", "UNIT_AURA",
			"UNIT_MODEL_CHANGED", "PLAYER_UPDATE_RESTING", "PLAYER_CONTROL_LOST", "PLAYER_CONTROL_GAINED", "PLAYER_ENTERING_WORLD",
			"RAID_TARGET_UPDATE", "UNIT_ATTACK_SPEED", "UNIT_STATS", "UPDATE_STEALTH", "UNIT_SPELLCAST_SUCCEEDED", "PLAYER_AURAS_CHANGED"}

	if (hconf.enabled) then
		for k,v in pairs(events) do
			self:RegisterEvent(v)
		end
		self:HideShow()
	else
		self:Hide()
		for k,v in pairs(events) do
			self:UnregisterEvent(v)
		end
	end
end

-- OnStartup
function hud:OnStartup(event, addon)

	if (event == "VARIABLES_LOADED") then
		self:UnregisterEvent(event)

		local _
		_, playerClass = UnitClass("player")

		if (playerClass ~= "ROGUE" and playerClass ~= "DRUID") then
			self.PLAYER_COMBO_POINTS = function() end

			if (playerClass ~= "WARRIOR" and playerClass ~= "MAGE" and playerClass ~= "PRIEST" and playerClass ~= "WARLOCK") then
				GetComboColour = nil
			end
		end

		if (playerClass == "HUNTER") then
			self.doRanged = true
		end

		if (playerClass == "ROGUE" or playerClas == "HUNTER" or playerClass == "WARRIOR" or playerClass == "SHAMAN") then
			self.doHasteBar = true
		end

		self:Create()

		self:SetScript("OnEvent", hud.OnEvent)

		self:Setup()
		self:HideShow()

		self.OnStartup = nil
	end
end

-- Test
function hud:Test()
	ZekTest = 1
	self:ShowComboBar(5)
	self:HideShow()
end

XPerl_RegisterOptionChanger(hud.Setup, hud)

hud:RegisterEvent("VARIABLES_LOADED")
hud:SetScript("OnEvent", hud.OnStartup)
