--[[

  XBar for WoW (World of Warcraft)

  By Dr. Doom (Shalune on Kargath)  
]]--

--Global variable so that other mod creators can test to see if XBAR is loaded.
XBAR_ENABLED = true;

-- Spacing constants
XBAR_SPBIG = 10;  -- Spacing for "spacers"
XBAR_SPSMALL = 1; -- Spacing between buttons

--Global vars
XBarModData = { };
XBarData = { };
XBarPlayerName = nil;
XBarOptionSet = "optionset-default";

-- global db for frame behavior tracking (replaced all the excess GetAttribute() and SetAttribute() function calls
XBarFrameDB = { };

local XBarInitialized=false;
local ver="1.24"; -- Mod version
local dbver="3";  -- Data format version

--==XBAR OPTIONS GUI==--
-- This is just glue until I can get it fully integrated

function XBarOptionsGUI_OnLoad()
	local XBarNest=getfenv()["XBarOptionsGUI"];
	XBarNest.name="XBar";
	XBarNest.parent=nil; -- empty for now, i might nest it later.
	XBarNest.okay=nil;
	XBarNest.cancel=nil;
	XBarNest.defaults=nil;
	InterfaceOptions_AddCategory(XBarNest);
end

--==XBAR FUNCTIONS==--
SLASH_XBAR1 = "/xbar";

function XBar_OnLoad()
	this:RegisterEvent("PLAYER_ENTERING_WORLD");

	SlashCmdList["XBAR"] = function(msg)
		XBar_SlashHandler(string.lower(msg));
	end
end

function XBar_GetVersion()
	return tonumber(ver),tonumber(dbver);
end

function XBar_SlashHandler(msg)
	if (strfind(msg, "config")) then
		if(XBarConfig:IsVisible()) then
			XBarConfig:Hide();
		else
			XBarConfig:Show();
		end
	elseif (strfind(msg, "show")) then
		msg=strsub(msg,6);
		msg=XBar_ValidateModName(msg);
		if (msg ~= "") then
			DEFAULT_CHAT_FRAME:AddMessage(msg..XBAR_MSG1)
			XBar_ShowHide(msg,2);
		end
	elseif (strfind(msg, "hide")) then
		msg=strsub(msg,6);
		msg=XBar_ValidateModName(msg);
		if (msg ~= "") then
			DEFAULT_CHAT_FRAME:AddMessage(msg..XBAR_MSG1)
			XBar_ShowHide(msg,1);
		end
	elseif (strfind(msg, "resetpos")) then
		msg=strsub(msg,10);
		msg=XBar_ValidateModName(msg);
		if (msg ~= "") then
			DEFAULT_CHAT_FRAME:AddMessage(msg..XBAR_MSG6);
			XBar_ResetPos(msg);
		end
	elseif (strfind(msg, "stopmoving")) then
		msg=strsub(msg,12);
		msg=XBar_ValidateModName(msg);
		if (msg ~= "") then
			DEFAULT_CHAT_FRAME:AddMessage(msg..XBAR_MSG11)
			getfenv()[msg]:StopMovingOrSizing();
		end
	elseif (strfind(msg, "toggle")) then
		msg=strsub(msg,8);
		msg=XBar_ValidateModName(msg);
		if (msg ~= "") then
			if (XBarData[XBarOptionSet]["mods"][msg]["toggle"]) then
				XBarData[XBarOptionSet]["mods"][msg]["toggle"]=false;
			else
				XBarData[XBarOptionSet]["mods"][msg]["toggle"]=true;
			end
			XBar_UpdateToggle(msg);
		end
	elseif (strfind(msg, "options")) then
		msg=strsub(msg,9);
		if (msg=="") then
			DEFAULT_CHAT_FRAME:AddMessage("XBar v"..ver);
			DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT8);
			DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT9);
			DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT10);
			DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT11);
			DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT12);
		elseif (strfind(msg,"view")) then
			local i;
			local n=tonumber(XBarData["noptionsets"]);
			
			DEFAULT_CHAT_FRAME:AddMessage("XBar Option sets:");
			for i=1,n do
				msg=tostring(i)..") ";
				if (XBarData[XBarPlayerName]==tostring(i)) then
					msg=msg.."*";
				end
				msg=msg..XBarData["optionset"..tostring(i)];
				DEFAULT_CHAT_FRAME:AddMessage(msg);
			end
		elseif (strfind(msg,"set")) then
			local i;
			local e=false;
			local n=XBarData["noptionsets"];
			
			msg=strsub(msg,5);
			i=tonumber(msg);
			if (i==nil) then
				e=true;
			elseif ((i<1) or (i>n)) then
				e=true;
			else
				-- Valid number, perform action
				XBarData[XBarPlayerName]=tostring(i);
				ReloadUI();
			end
			if (e) then
				DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG7.."'"..msg.."'");
			end
		elseif (strfind(msg,"delete")) then
			local i,t;
			local e=false;
			local n=tonumber(XBarData["noptionsets"]);
			
			msg=strsub(msg,8);
			i=tonumber(msg);
			msg=tostring(i);
			if (i==nil) then
				e=true;
			elseif ((i<2) or (i>n)) then
				e=true;
			elseif (XBarData[XBarPlayerName]==msg) then
				-- Can't delete our current set
				e=true;
			else
				t=XBarData["optionset"..i];
				XBarData["optionset-"..t]=nil;
				XBarData["optionset"..msg]=XBarData["optionset"..tostring(n)];
				XBarData["optionset"..tostring(n)]=nil;
				-- Valid number, perform action
				-- Whether last or not, players who are using it must use default now
				for x,v in pairs(XBarData) do
					if ((x~="dbver") and (strsub(x,1,9)~="optionset") and (type(v)=="string")) then
						-- By process of elimination, this has to be a player's option set
						if (XBarData[x]==msg) then
							XBarData[x]="1"; -- Found one that was set to this option set
						end
					end
				end

				-- If not the last one, overwrite
				if (i<n) then
					-- Deleted and repointed, now clean up.
					for x,v in pairs(XBarData) do
						if ((x~="dbver") and (strsub(x,1,9)~="optionset") and (type(v)=="string")) then
							-- By process of elimination, this has to be a player's option set
							if (XBarData[x]==tostring(n)) then
								XBarData[x]=msg; -- Found one that was set to this option set
							end
						end
					end
				end

				XBarData["noptionsets"]=n-1;
			end
			if (e) then
				DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG7.."'"..msg.."'");
			else
				DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG10.."'"..t.."'");
			end
		elseif (strfind(msg,"new")) then
			msg=strtrim(strsub(msg,5));
			local i;
			local n=tonumber(XBarData["noptionsets"]);
			local f=false;

			-- make sure its not in there first
			for i=1,n do
				if (XBarData["optionset"..tostring(i)]==msg) then
					DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG7..msg);
					f=true;
				end
			end
			if (msg=="") then
				f=true;
			end
			if (not f) then
				n=n+1;
				XBarData["noptionsets"]=n;
				XBarData["optionset"..tostring(n)]=msg;
				XBarData["optionset-"..msg]={
					["nmods"]=0,
					["mods"]={ },
				};
				DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG8..msg..XBAR_MSG9..tostring(n));
			end
		end
	else
		DEFAULT_CHAT_FRAME:AddMessage("XBar v"..ver);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT1);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT2);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT3);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT4);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT5);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT13);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT14);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT6);
		DEFAULT_CHAT_FRAME:AddMessage(XBAR_HELPTEXT7);
	end
end

function XBar_OnEvent()
	if (event == "PLAYER_ENTERING_WORLD") then
		-- Find the player name and save it
		XBar_BuildPlayerName();
		XBar_Start();
	end
end

function XBar_BuildPlayerName()
	-- We must setup the player's name and server to save data.
	if (UnitName("player") ~= UNKNOWNBEING and UnitName("player") ~= UNKNOWNOBJECT and UnitName("player") and not XBarPlayerName) then
		XBarPlayerName = UnitName("player").." of "..GetCVar("realmName");
	end
end

function XBar_GetOptionSet()
	return strsub(XBarOptionSet,11);
end

function XBar_Start()
	local resetdb=false;
	local i,n,m,on,os;

	if (XBarInitialized) then
		return; -- Only do this if we haven't done it yet
	end

	if (not XBarData) then
		resetdb=true;
	elseif (not XBarData[XBarOptionSet]) then
		resetdb=true;
	elseif (XBarData["dbver"] ~= dbver) then
		resetdb=true;
	end

	if (resetdb) then
		--Data structure not initialized for this player, so set one up
		XBarData={
			["dbver"]=dbver;
			["optionset-default"] = {};
			["noptionsets"] = 1;
			["optionset1"] = "default";
			[XBarPlayerName] = "1";
		};
		XBarData[XBarOptionSet]={
			["nmods"]=0,     ---# of mods registered, names will be values of mod1, mod2, etc...
			["mods"]={},     ---Collection of mod values
		};
	end

	-- Find the option set the player wants to use.
	on=XBarData[XBarPlayerName];
	if (on==nil) then
		XBarData[XBarPlayerName]="1";
		os="optionset-default";
	else
		-- Lets validate that option set
		n=XBarData["noptionsets"];
		if ((tonumber(on)>n) or (tonumber(on)<1)) then
			DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG7..tostring(on));
			on=1; -- reset to defaults
		end
		os="optionset-"..XBarData["optionset"..on];
	end

	XBarOptionSet=os;
	DEFAULT_CHAT_FRAME:AddMessage(XBAR_MSG4..XBar_GetOptionSet());

	-- Initialize all saved mods to "not loaded"
	n=XBarData[XBarOptionSet]["nmods"];
	if (n>0) then
		for i=1,n do
			m=XBarData[XBarOptionSet]["mod"..i];
			if (XBarModData[m] ~= nil) then
				XBarModData[m]["loaded"]=false;
			end
		end
	end
	XBarInitialized=true;
end

function XBar_RegisterAddon(mod)
	-- This function is used by plugins to let us know they are present and loaded.
	local n,v1,v2,i;

	-- Validate that the mod has loaded default settings
	if (not XBarModData[mod]) then
		-- Error message
		DEFAULT_CHAT_FRAME:AddMessage(mod..XBAR_MSG5);
	else
		if (XBarModData[mod]["enabled"]) then
			n=XBarData[XBarOptionSet]["nmods"];
			-- Scan to see if it's in there already
			v1=XBar_ValidateModName(mod);
			if (v1=="") then
				--not found, add it to the list
				i=n+1;
				XBarData[XBarOptionSet]["nmods"]=i;
				XBarData[XBarOptionSet]["mod"..i]=mod;
				XBarData[XBarOptionSet]["mods"][mod]={};
			end
			i=false;
			v1=XBarModData[mod]["dbver"];
			if (XBarData[XBarOptionSet]["mods"][mod]==nil) then
				i=true; -- No database at all
				DEFAULT_CHAT_FRAME:AddMessage(mod..XBAR_MSG2);
			elseif (XBarData[XBarOptionSet]["mods"][mod]["dbver"]~=v1) then
				i=true; -- Database version incorrect
				DEFAULT_CHAT_FRAME:AddMessage(mod..XBAR_MSG3);
			end
			-- Create user-space variables to store settings
			if (i) then
				XBarData[XBarOptionSet]["mods"][mod]={
					["dbver"]=v1;
					["x"]=150,
					["y"]=-150,
					["rp"]="TOPLEFT",
					["pt"]="TOPLEFT",
					["toggle"]=true,
					["horizontal"]=XBarModData[mod]["dhorizontal"],
					["hidebar"]=XBarModData[mod]["dhidebar"],
					["order"]=XBarModData[mod]["dorder"],
					["scale"]=XBarModData[mod]["dscale"],
					["tooltips"]=XBarModData[mod]["dtooltips"]
				};
				-- Add in space to save the user checkboxes
				n=XBarModData[mod]["nchecks"];
				if (n==nil) then
					n=0;
					XBarModData[mod]["nchecks"]=0;
				else
					for i=1,n do
						v1=XBarModData[mod]["check"..i];
						v2=XBarModData[mod]["dcheck"..i];
						if (v1) then
							XBarData[XBarOptionSet]["mods"][mod][v1]=v2;
						end
					end
				end
				-- Add in space to save the user sliders
				n=XBarModData[mod]["nsliders"];
				if (n==nil) then
					n=0;
					XBarModData[mod]["nsliders"]=0;
				else
					for i=1,n do
						v1=XBarModData[mod]["slider"..i];
						v2=XBarModData[mod]["dslider"..i];
						XBarData[XBarOptionSet]["mods"][mod][v1]=v2;
					end
				end
			end
			XBar_SetPos(mod); -- Set the mod to the last known coordinates just in case
			XBar_GenerateButtons(mod);
			XBarModData[mod]["loaded"]=true; -- mark the mod as loaded
		else
			XBarModData[mod]["loaded"]=false; -- mark the mod as unloaded
		end
	end
end

function XBar_GenerateButtons(mod)
	local i,n,b,f,g;
	local meta_idhack;

	n=XBarModData[mod]["nbuttons"];
	if (n==0) then
		return;
	end
	
	if XBarModData[mod]["loaded"] then
		-- Don't need to make buttons every time we zone
		return;
	end

	b=getfenv()[mod];
	f=CreateFrame("Button",mod.."ButtonToggle",UIParent,"XBarButtonBaseTemplate,SecureAnchorButtonTemplate");
	f:SetPoint("TOPLEFT",b,"TOPLEFT",10,0);
	f:SetWidth("16");
	f:SetHeight("16");
	if (XBarFrameDB[mod.."ButtonToggle"] == nil) then
		XBarFrameDB[mod.."ButtonToggle"] = {};
	end
	XBarFrameDB[mod.."ButtonToggle"]["XBarToggle"]=mod;
	f:SetScript("PostClick",XBar_Toggle);

	-- Creating a new state header to handle the state changes for the bar.
	g=CreateFrame("Button",mod.."StateHeader",UIParent,"SecureStateHeaderTemplate");
	-- Set the toggle button to send info to the new state header
	f:SetAttribute("*childstate*","^click");
	f:SetAttribute("anchorchild",g);
	-- Let the state header know which frame to update by adding it as a child, and give it the statemap for 'click'
	g:SetAttribute("addchild",getfenv()[mod]);
	g:SetAttribute("statemap-anchor-click","0:1;1:0");

	-- Set the XBar frame to only show when state is 1
	getfenv()[mod]:SetAttribute("showstates","1");

	for i=1,n do
		f=CreateFrame("Button",mod.."Button"..tostring(i),b,"XBarButtonTemplate");
		if (XBarFrameDB[mod.."Button"..tostring(i)] == nil) then
			XBarFrameDB[mod.."Button"..tostring(i)]={};
		end
		XBarFrameDB[mod.."Button"..tostring(i)]["XBarToggle"]=nil;
		if (i==1) then
			f:SetPoint("LEFT",mod.."ButtonToggle","RIGHT",1,0);
		else
			f:SetPoint("LEFT",mod.."Button"..(i-1),"RIGHT",1,0);
		end
		f:RegisterForClicks("LeftButtonUp","RightButtonUp","MiddleButtonUp","Button4Up","Button5Up");
	end
end

function XBar_Update(mod)
	local i,button,z,id,r1,r2,key,value,val2,rank,spelllist,meta_idhack,texture;
	local bar=getfenv()[mod];
	local barheader=getfenv()[mod.."StateHeader"];

	spelllist=getfenv()[mod.."Spells"];
	-- Exit this function if the buttons haven't been initialized yet.
	if (getfenv()[mod.."Button1"]==nil) then
		return;
	end
	if (XBarData[XBarOptionSet]) then
		-- This function cannot be performed in combat.
		if (InCombatLockdown() == nil) then
			i=XBarModData[mod]["nbuttons"];

			if (i~=nil) then
				for key=1,i do
					button=getfenv()[mod.."Button"..key];
					button:Hide();
				end
			end
			getfenv()[mod.."ButtonToggle"]:Show();
			XBar_UpdateToggle(mod);
			if (XBarData[XBarOptionSet]["mods"][mod]["hidebar"]) then
				barheader:SetAttribute("state","0");
			else
				barheader:SetAttribute("state","1");
			end
		end

		i=0;
		-- Look for callback functions
		local f1=getfenv()[XBarModData[mod]["ftexint"]];
		local f2=getfenv()[XBarModData[mod]["fbuttoncb"]];
		local f3=getfenv()[XBarModData[mod]["fbuttonid"]];

		for key,value in pairs(spelllist) do
			meta_idhack=false;
			if (strsub(value,1,1)=="&") then
				-- Parse meta commands
				id=strfind(value,"&",2);
				if (id) then
					rank=strsub(value,2,id-1);
					if (rank=="idhack") then
						meta_idhack=true;
					end
					value=strsub(value,id+1);
				else
					value=strsub(value,2);
				end
			end

			if ((value==" ") and (i>0)) then
				-- Spacer
				if (XBarFrameDB[mod.."Button"..i+1]==nil) then
					XBarFrameDB[mod.."Button"..i+1]={};
				end
				XBarFrameDB[mod.."Button"..i+1]["XBarSpacer"]=true;
			elseif (strsub(value,1,1)=="#") then
				-- Menu header, just ignore
			else
				id,rank = XBar_GetSpellID(value,XBarModData[mod]["booktype"]);

				if (f3~=nil) then
					id=f3(id,value);
				end

				if (id) then
					i = i + 1;
					if (XBarFrameDB[mod.."Button"..i]==nil) then
						XBarFrameDB[mod.."Button"..i]={};
					end
					if meta_idhack then
						XBarFrameDB[mod.."Button"..i]["XBarIDHack"]=0;
					end
					button = getfenv()[mod.."Button"..i];
					if (XBarFrameDB[mod.."Button"..i]["XBarSpacer"]==nil) then
						XBarFrameDB[mod.."Button"..i]["XBarSpacer"]=false;
					end

					-- This function cannot be performed in combat.
					if (XBarFrameDB[mod.."Button"..i]["XBarIDHack"] and not InCombatLockdown()) then
						button:SetAttribute("button*",nil);
						button:SetAttribute("type",nil);
						button:SetAttribute("spell*",nil);
						button:SetAttribute("ctrl-spell*",nil);
						if XBarFrameDB[mod.."Button"..i]["XBarIDHack"]==0 then
							-- Only need to override the OnClick event once
							button:SetScript("OnClick",XBar_IDHack);
							XBarFrameDB[mod.."Button"..i]["XBarIDHack"]=1;
						end
					elseif (not InCombatLockdown()) then
						button:SetAttribute("button*","type");
						button:SetAttribute("type", "spell");
						if ((rank ~= nil) and (rank ~= "")) then
							val2=value.."("..rank..")";
						else
							val2=value;
						end
						button:SetAttribute("spell*", val2);
						button:SetAttribute("ctrl-spell*", ATTRIBUTE_NOOP);
					end

	  				texture = GetSpellTexture(id, XBarModData[mod]["booktype"]);
					-- Call texture interceptor if it is there
					if (f1~=nil) then
						texture=f1(texture,value);
					end
					getfenv()[mod.."Button"..i.."Icon"]:SetTexture(texture);

					XBarFrameDB[mod.."Button"..i]["XBarID"]=id;
					XBarFrameDB[mod.."Button"..i]["XBarCD"]="SI"..tostring(id);
					XBarFrameDB[mod.."Button"..i]["XBarVI"]=key; -- Save the spell list index for addons to see where it corellates to
					XBarFrameDB[mod.."Button"..i]["XBarBT"]=XBarModData[mod]["booktype"];
					-- This function cannot be performed in combat.
					if not InCombatLockdown() then
						button:Show();
					end
					if (f2~=nil) then
						f2(button,value,i);
					end
					if not InCombatLockdown() then
						button:SetScale(XBarData[XBarOptionSet]["mods"][mod]["scale"]);
					end
				end
			end
		end
		XBar_Orient(mod);
		XBar_UpdateTT(mod);
		XBar_UpdateCooldowns(mod);

		-- This function cannot be performed in combat.
		if ((i == 0) and (InCombatLockdown() == nil)) then
			bar:Hide();
			getfenv()[mod.."ButtonToggle"]:Hide();
		end
		if not (InCombatLockdown() or XBarData[XBarOptionSet]["mods"][mod]["toggle"]) then
			getfenv()[mod.."ButtonToggle"]:Hide();
		end
	end
end

function XBar_UpdateTT(mod)
	local n=XBarModData[mod]["nbuttons"];
	local i;

	if (n==nil) then
		return;
	end

	for i = 1, n do
		XBarFrameDB[mod.."Button"..i]["XBarTT"]=XBarData[XBarOptionSet]["mods"][mod]["tooltips"];
	end
end

function XBar_UpdateToggle(mod)
	local i,b,t;

	if (XBarModData[mod]) then
		if XBarModData[mod]["loaded"] then
			i=getfenv()[mod.."ButtonToggleIcon"];
			b=getfenv()[mod];
			if b:IsVisible() then
				--XBarData[XBarOptionSet]["mods"][mod]["hidebar"] = false;
				t="Interface\\AddOns\\XBar\\xbarminus";
			else
				--XBarData[XBarOptionSet]["mods"][mod]["hidebar"] = true;
				t="Interface\\AddOns\\XBar\\xbarplus";
			end
			if (XBarData[XBarOptionSet]["mods"][mod]["toggle"]) then
				if not InCombatLockdown() then
					getfenv()[mod.."ButtonToggle"]:Show();
				end
			else
				if not InCombatLockdown() then
					getfenv()[mod.."ButtonToggle"]:Hide();
				end
				t="";
			end
			i:SetTexture(t);
		end
	end
end

function XBar_Orient(mod)
	local button;

	if (not XBarModData[mod]["enabled"]) then
		return;
	end

	local n=XBarModData[mod]["nbuttons"];
	local d1, d2, h, o, s, r, sp, a;
	local pt,an,ap,xd,yd,mag;

	if (n==nil) then
		return;
	end

	-- This function cannot be performed in combat.
    if ((InCombatLockdown() == nil) and (XBarData[XBarOptionSet])) then
		h=XBarData[XBarOptionSet]["mods"][mod]["horizontal"];
		o=XBarData[XBarOptionSet]["mods"][mod]["order"];
		r= (o == "za");
		if ((not h) and (not r)) then
			d1="TOP";
			d2="BOTTOM";
		elseif ((not h) and r) then
			d1="BOTTOM";
			d2="TOP";
		elseif (h and (not r)) then
			d1="LEFT";
			d2="RIGHT";
		elseif (h and r) then
			d1="RIGHT";
			d2="LEFT";
		else -- Default is horizontal without reversal
			d1="LEFT";
			d2="RIGHT";
			h=true;
			r=false;
		end

		-- Insert spacing
		if (h) then
			h=1;
		else
			h=0;
		end
		if ((h and not r) or (not h and r)) then
			s=1;
		else
			s=-1;
		end
		h=h * s;
		s=(abs(h)-1) * s;

		local fo=getfenv()[XBarModData[mod]["forientcb"]];

		for i = 1,n do
			button = getfenv()[mod.."Button"..i];
			if (button~=nil) then
				button:ClearAllPoints();
				if (XBarFrameDB[mod.."Button"..i]["XBarSpacer"]) then
					sp=XBAR_SPBIG;
				else
					sp=XBAR_SPSMALL;
				end
				if (i==1) then
					a=mod.."ButtonToggle";
				else
					a=mod.."Button"..(i-1);
				end
				if (fo) then
					pt,an,ap,xd,yd,mag=fo(i,d1,a,d2,h,s,sp);
				else
					pt=d1;
					an=a;
					ap=d2;
					xd=h;
					yd=s;
					mag=sp;
				end
				button:SetPoint(pt, an, ap, xd*mag, yd*mag);
			end
		end
	end
end

function XBar_RangeTimer(mod,elapsed)
	if XBarFrameDB[mod] == nil then
		XBarFrameDB[mod]={ };
	end
	if XBarFrameDB[mod]["RT"] == nil then
		XBarFrameDB[mod]["RT"]=elapsed;
	else
		XBarFrameDB[mod]["RT"]=XBarFrameDB[mod]["RT"]+elapsed
		if XBarFrameDB[mod]["RT"] >= 0.2 then
			XBar_UpdateCooldowns(mod);
			XBarFrameDB[mod]["RT"]=0;
		end
	end
end

function XBar_UpdateCooldowns(mod)
	local i,bi,bt,v,id,cd,f1,cooldown,start,duration,enable,texture;

	if (not XBarModData[mod]["loaded"]) then
		return;
	end

	local n=XBarModData[mod]["nbuttons"];
	if (n==nil) then
		return;
	end

	for i = 1, n do
		bi=getfenv()[mod.."Button"..i.."Icon"];
		bt=XBarFrameDB[mod.."Button"..i]["XBarBT"];
		if (bt) then
			id=XBarFrameDB[mod.."Button"..i]["XBarID"];
			v=GetSpellName(id,bt);
			if (v) then
				-- Notice we only call the texture interceptor if a booktype was specified (ie. its a spell)
				texture = GetSpellTexture(id, bt);

				f1=getfenv()[XBarModData[mod]["ftexint"]];
				-- Call texture interceptor if it is there
				if (f1~=nil) then
					texture=f1(texture,v);
				end

				-- Range check
				cd="playertarget";
				if (v=="Arcane Intellect" or v=="Pyroblast") then
					--DEFAULT_CHAT_FRAME:AddMessage("E: "..tostring(UnitExists(cd))..", F: "..tostring(UnitIsFriend(cd,"player"))..", A: "..tostring(IsAttackSpell(v))..", V:"..tostring(UnitIsVisible(cd)));
				end
				--[[
				This secion would be great if IsAttackSpell returned useful info.  Right now it only returns 1 for "Attack",
				but I need something that specifies whether or not a spell can be cast on an attackable unit.
				if (UnitExists(cd)) then
					if (UnitIsFriend(cd,"player") and IsAttackSpell(v)) then
						cd="player";
					elseif not (UnitIsFriend(cd,"player") or IsAttackSpell(v)) then
						cd="player";
					elseif not UnitIsVisible(cd) then
						cd="player";
					end
				]]--
				if not (UnitExists(cd) and UnitIsVisible(cd)) then
					cd="player";
				end

				bi:SetTexture(texture);
				start=IsUsableSpell(v)
				if start and SpellHasRange(v) then
					start = start and IsSpellInRange(v,cd);
				end
				if (start~=1) then
					bi:SetAlpha(0.5);
				else
					bi:SetAlpha(1.0);
				end
			end
			cd="S";
			id="I"..tostring(id);
		else
			-- If no booktype specified, look in the cooldown specifier attribute
			-- This should always be an item or a macro
			cd=XBarFrameDB[mod.."Button"..i]["XBarCD"];
			if (cd==nil) then
				cd="SInil"; -- Disables the cooldown
			end
			id=strsub(cd,2); -- Extract the spell/item name
			cd=strsub(cd,1,1);
			if cd=="S" then
				bt=BOOKTYPE_SPELL; -- Assume booktype-spell for spells
			end
		end
		start=0;
		duration=0;
		enable=1;
		if cd=="I" then
			start, duration, enable = GetItemCooldown(id);
		else
			cd=strsub(id,1,1);
			id=strsub(id,2);
			-- See whether this is a spell ID or name
			-- NOTE: Cooldown spell ID can be different from XBarID
			if cd=="I" then
				id=tonumber(id);
			end
			if type(id)=="string" then
				start, duration, enable = GetSpellCooldown(id);
			elseif id~=nil then
				start, duration, enable = GetSpellCooldown(id, bt);
			end
		end
		cooldown = getfenv()[mod.."Button"..i.."Cooldown"];
		if (start ~= nil) and (duration ~= nil) and (enable ~= nil) then
			CooldownFrame_SetTimer(cooldown,start, duration, enable);
		end
	end
end

function XBar_GetSpellID(spell,book,rank)
	local i = 0;
	local curRank=nil;
	local highestRank=nil;
	local spellName, spellRank;

	repeat
		i = i + 1;
		spellName, spellRank = GetSpellName(i, book);
		if (spellName == spell) then
			curRank = spellRank;
			if (rank ~= nil) then
				if (curRank==rank) then
					i=i+1;
					break;
				end
			end
		elseif (curRank) then
			break; -- break if we passed all the spells
		end
	until (spellName==nil);

	if (curRank==nil) then
		i=nil; -- didn't find it
	else
		i=i-1; -- found it, correct i.
	end

	return i,curRank;
end

function XBar_SetTooltip()
	local bn=this:GetName();
	if XBarFrameDB[bn] == nil then
		XBarFrameDB[bn] = {};
	end
	local id=XBarFrameDB[bn]["XBarID"];
	local tt=XBarFrameDB[bn]["XBarTT"];
	local toggle=XBarFrameDB[bn]["XBarToggle"];

	if (id==nil) then
		id=-1;
		tt=false;
	end

	if ((id >= 0) and tt) then
		GameTooltip_SetDefaultAnchor(GameTooltip, this);
		GameTooltip:SetSpell(id, XBarFrameDB[bn]["XBarBT"]);
		GameTooltip:Show();
	elseif (toggle ~= nil) then
		if (toggle=="-") then
			toggle=XBarFrameDB[bn]["XBarInfoText"];
		end
		GameTooltip_SetDefaultAnchor(GameTooltip, this);
		GameTooltip:SetText(toggle);
		GameTooltip:Show();
	end
end

function XBar_Scale(mod)
	local i,n,button;
	
	if (not XBarModData[mod]["loaded"]) then
		return;
	end
	
	if InCombatLockdown() then
		return;
	end

	n=XBarModData[mod]["nbuttons"];
	if (n==nil) then
		return;
	end

	for i = 1,n do
		button = getfenv()[mod.."Button"..i];
		button:SetScale(XBarData[XBarOptionSet]["mods"][mod]["scale"]);
	end
end

function XBar_ShowHide(mod,state)
	local frame,frameheader;
	-- state=0: Toggle
	-- state=1: Hide
	-- state=2: Show
	if (not XBarModData[mod]) then
		DEFAULT_CHAT_FRAME:AddMessage(mod);
		return;
	end
	if (not XBarModData[mod]["loaded"]) then
		return;
	end

	-- This function cannot be performed in combat.
	if (InCombatLockdown() == nil) then
		frame = getfenv()[mod];
		frameheader = getfenv()[mod.."StateHeader"];
		if (frame) then
			if(frame:IsVisible()) then
				if (state ~= 2) then
					XBarData[XBarOptionSet]["mods"][mod]["hidebar"]=true;
					frameheader:SetAttribute("state","0");
				end
			else
				if (state ~= 1) then
					XBarData[XBarOptionSet]["mods"][mod]["hidebar"]=false;
					frameheader:SetAttribute("state","1");
				end
			end
		end
	end
end

function XBar_ValidateModName(mod)
	local n=XBarData[XBarOptionSet]["nmods"];
	local v=false;
	local s1=strlower(strtrim(mod));
	local s2;

	for i=1,n do
		s2=XBarData[XBarOptionSet]["mod"..i];
		if (strlower(s2)==s1) then
			return s2;
		end
	end
	return "";
end

function XBar_SaveLoc(bar)
	local mod;

	mod=bar:GetName();
	if (mod) then
		if (not XBarModData[mod]["loaded"]) then
			return;
		end

		local p,rt,rp,x,y=bar:GetPoint();
		XBarData[XBarOptionSet]["mods"][mod]["x"] = x;
		XBarData[XBarOptionSet]["mods"][mod]["y"] = y;
		XBarData[XBarOptionSet]["mods"][mod]["rp"] = rp;
		XBarData[XBarOptionSet]["mods"][mod]["pt"] = p;
	end
end

function XBar_ResetPos(mod)
	if (not XBarModData[mod]) then
		return;
	end

	local bar=getfenv()[mod];
	bar:ClearAllPoints();
	bar:SetPoint("BOTTOM",nil,"CENTER");
	XBar_SaveLoc(bar);
end

function XBar_SetPos(mod)
	if (not XBarModData[mod]) then
		return;
	end

	local bar=getfenv()[mod];
	local x = XBarData[XBarOptionSet]["mods"][mod]["x"];
	local y = XBarData[XBarOptionSet]["mods"][mod]["y"];
	local rp = XBarData[XBarOptionSet]["mods"][mod]["rp"];
	local p = XBarData[XBarOptionSet]["mods"][mod]["pt"];

	bar:ClearAllPoints();
	bar:SetPoint(p,nil,rp,x,y);
end

function XBar_Toggle()
	if not IsControlKeyDown() then
		local mod=this:GetName();
		local fi=getfenv()[mod.."Icon"];
		local f;

		if (fi:GetTexture()~=nil) then
			mod=strsub(mod,-1*strlen(mod),-13); -- cut off the appendix to find the corresponding bar
			f=getfenv()[mod];
			if f:IsVisible() then
				XBarData[XBarOptionSet]["mods"][mod]["hidebar"]=false;
			else
				XBarData[XBarOptionSet]["mods"][mod]["hidebar"]=true;
			end
		end
		XBar_UpdateToggle(mod);
	end
end

function XBar_IDHack()
	local bn=this:GetName();
	if XBarFrameDB[bn]==nil then
		XBarFrameDB[bn]={};
	end
	local m=XBarFrameDB[bn]["XBarID"];
	local bt=XBarFrameDB[bn]["XBarBT"];
	if (m and bt) then
		CastSpell(m,bt);
	end
end

function XBar_Spellname(spellname)
	local r=spellname;
	local p;
	
	-- extract meta commands
	repeat
		p=strfind(r,"&");
		if (p) then
			p=strfind(r,"&",p+1);
			if (p) then
				r=strsub(r,p+1);
			else
				p=nil;
			end
		end
	until not p;
	
	return r;
end

function XBar_ShowCount(mod,index,count)
	local b=getfenv()[mod.."Button"..tostring(index).."Count"];
	local v=count;
	
	if (v>999) then
		v="*"
	elseif (v==1) then
		v="";
	else
		v=tostring(v);
	end
	b:SetText(v);
end

function XBar_CountSpells(mod)
	local rbuttons=0;
	local rchecks=0;
	local i,v,c;
	local spelllist=getfenv()[mod.."Spells"]

	-- This function performs the common task of enumerating the number of
	-- buttons and checkboxes in a mod (assuming 1 checkbox per spell).
	for i,v in pairs(spelllist) do
		c=strsub(v,1,1);
		
		if c==" " then
			-- This is a spacer
			rbuttons=rbuttons+1;
		elseif c=="#" then
			-- This is probably a menu header
			rchecks=rchecks+1;
		else
			-- Probably a spell
			rchecks=rchecks+1;
			rbuttons=rbuttons+1;
		end
	end

	return rbuttons,rchecks;
end

function XBar_OnMouseDown()
	local pfn=this:GetParent():GetName();
	if ((arg1 == "RightButton") and (IsControlKeyDown())) then
		local bn=this:GetName();
		local m=XBarFrameDB[bn]["XBarToggle"];
		if not XBarFrameDB[bn] then
			XBarFrameDB[bn] = {};
		end
		if XBarFrameDB[pfn]==nil then
			XBarFrameDB[pfn]={};
		end
		XBarFrameDB[pfn]["IsMoving"]=true;
		if ((m==nil) or (m=="-")) then
			this:GetParent():StartMoving();
			this:SetAttribute("type",nil);
		else
			getfenv()[m]:StartMoving();
		end
	end
end

function XBar_OnMouseUp()
	local pfn=this:GetParent():GetName();
	if (arg1 == "RightButton" and XBarFrameDB[pfn] and XBarFrameDB[pfn]["IsMoving"]) then
		local bn=this:GetName();
		local m=XBarFrameDB[bn]["XBarToggle"];
		if ((m==nil) or (m=="-")) then
			this:GetParent():StopMovingOrSizing();
			XBar_SaveLoc(this:GetParent());
			XBar_Update(this:GetParent():GetName());
		else
			getfenv()[m]:StopMovingOrSizing();
			XBar_SaveLoc(getfenv()[m]);
		end
		if XBarFrameDB[pfn]==nil then
			XBarFrameDB[pfn]={};
		end
		XBarFrameDB[pfn]["IsMoving"]=nil;
	end
end

--==XBAR CONFIG FUNCTIONS==--

function XBarConfig_OnShow()
	local i,n,mod,b;
	local cfg=this:GetName();

	if (not XBarData) then
		this:Hide();
		return;
	end
	if (not XBarData[XBarOptionSet]) then
		this:Hide();
		return;
	end

	n=XBarData[XBarOptionSet]["nmods"];
	if (n==nil) then
		this:Hide();
		return;
	end

	-- Initialize select fields to nothing
	if XBarFrameDB["XBarConfig"] == nil then
		XBarFrameDB["XBarConfig"] = {};
	end
	XBarFrameDB["XBarConfig"]["mod"]="";
	XBarFrameDB["XBarConfig"]["checkoption"]="";
	XBarFrameDB["XBarConfig"]["slideroption"]="";

	-- Start with all fields hidden
	XBarConfigCheckButtonH:Hide();
	XBarConfigCheckButtonO:Hide();
	XBarConfigCheckButtonV:Hide();
	XBarConfigCheckButtonTT:Hide();
	XBarConfigCheckButtonTG:Hide();
	XBarConfigSliderScale:Hide();
	XBarConfigCurCheck:Hide();
	XBarConfigCheckMenu:Hide();
	XBarConfigCurSlider:Hide();
	XBarConfigSliderMenu:Hide();
	XBarConfigSliderMulti:Hide();

	-- Search for loaded mods
	b=false;
	for i=1,n do
		mod=XBarData[XBarOptionSet]["mod"..i];
		if (XBarModData[mod]) then
		if (XBarModData[mod]["loaded"]) then
			b=true;
		end
		end
	end

	--Tell user whether we have anything installed or not.
	if (not b) then
		-- No data, display an error message
		XBarConfigCurMod:SetText(XBAR_NOMOD);
		XBarConfigModMenu:Hide();
	else
		XBarConfigCurMod:SetText(XBAR_SELMOD);
		XBarConfigModMenu:Show();
	end
end

function XBarConfigModMenu_Select()
	local mod=this.value;
	XBarConfigCurMod:SetText(mod);

	if (InCombatLockdown()) then
		return; -- can't do this in combat
	end
	
	XBarConfigCheckButtonH:Show();
	XBarConfigCheckButtonH:SetChecked(XBarData[XBarOptionSet]["mods"][mod]["horizontal"]);

	XBarConfigCheckButtonO:Show();
	XBarConfigCheckButtonO:SetChecked(XBarData[XBarOptionSet]["mods"][mod]["order"]=="za");

	XBarConfigCheckButtonV:Show();
	XBarConfigCheckButtonV:SetChecked(getfenv()[mod]:IsVisible());

	XBarConfigCheckButtonTG:Show();
	XBarConfigCheckButtonTG:SetChecked(getfenv()[mod.."ButtonToggleIcon"]:GetTexture()~=nil);

	-- Only show if the author did not disable trackings in the addon
	if (XBarModData[mod]["dtooltips"]) then
		XBarConfigCheckButtonTT:Show();
		XBarConfigCheckButtonTT:SetChecked(XBarData[XBarOptionSet]["mods"][mod]["tooltips"]);
	end

	XBarConfigSliderScale:Show();
	XBarConfigSliderScale:SetValue(XBarData[XBarOptionSet]["mods"][mod]["scale"]);

	-- Show custom checkbox and slider options
	if (XBarModData[mod]["nchecks"]>0) then
		XBarConfigCurCheck:Show();
		XBarConfigCheckMenu:Show();
		XBarConfigCurCheck:SetText(XBAR_SELOPTION);
	else
		XBarConfigCurCheck:Hide();
		XBarConfigCheckMenu:Hide();
		XBarConfigSliderMulti:Hide()
	end

	if (XBarModData[mod]["nsliders"]>0) then
		XBarConfigCurSlider:Show();
		XBarConfigSliderMenu:Show();
		XBarConfigCurSlider:SetText(XBAR_SELOPTION);
	else
		XBarConfigCurSlider:Hide();
		XBarConfigSliderMenu:Hide();
		XBarConfigSliderMulti:Hide()
	end
end

function XBarConfigModMenuFrame_OnLoad()
	-- Loading off the click event from the button itself so we can make sure the data is there.
	-- Reinitializes every time we click the button.
	local info = UIDropDownMenu_CreateInfo();
	local n,i,mod;

	n=XBarData[XBarOptionSet]["nmods"];
	for i=1,n do
		mod=XBarData[XBarOptionSet]["mod"..i]
		if (XBarModData[mod]) then
		if (XBarModData[mod]["loaded"]) then
			info.text = mod;
			info.notCheckable=true;
			info.value = mod;
			info.func = XBarConfigModMenu_Select;
			info.owner = this:GetParent();
			UIDropDownMenu_AddButton(info);
		end
		end
	end
end

function XBarConfigModMenu_OnClick()
	UIDropDownMenu_Initialize(XBarConfigModMenuFrame, XBarConfigModMenuFrame_OnLoad);
	ToggleDropDownMenu(1, nil, XBarConfigModMenuFrame, XBarConfigModMenu, 0, 0);
end

function XBarConfigCheckButtonV_OnClick()
	local mod=XBarConfigCurMod:GetText();
	local bar=getfenv()[mod];

	if not InCombatLockdown() then
		if (this:GetChecked()) then
			XBarData[XBarOptionSet]["mods"][mod]["hidebar"]=false;
			XBar_ShowHide(mod,2);
		else
			XBarData[XBarOptionSet]["mods"][mod]["hidebar"]=true;
			XBar_ShowHide(mod,1);
		end
	end
end

function XBarConfigCheckButtonH_OnClick()
	local mod=XBarConfigCurMod:GetText();

	if (this:GetChecked()) then
		XBarData[XBarOptionSet]["mods"][mod]["horizontal"] = true;
	else
		XBarData[XBarOptionSet]["mods"][mod]["horizontal"] = false;
	end
	XBar_Orient(mod);
end

function XBarConfigCheckButtonO_OnClick()
	local mod=XBarConfigCurMod:GetText();

	if (this:GetChecked()) then
		XBarData[XBarOptionSet]["mods"][mod]["order"] = "za";
	else
		XBarData[XBarOptionSet]["mods"][mod]["order"] = "az";
	end
	XBar_Orient(mod);
end

function XBarConfigCheckButtonTT_OnClick(mod)
	local mod=XBarConfigCurMod:GetText();

	if (this:GetChecked()) then
		XBarData[XBarOptionSet]["mods"][mod]["tooltips"] = true;
	else
		XBarData[XBarOptionSet]["mods"][mod]["tooltips"] = false;
	end
	XBar_UpdateTT(mod);
end

function XBarConfigCheckButtonTG_OnClick(mod)
	local mod=XBarConfigCurMod:GetText();

	if (this:GetChecked()) then
		XBarData[XBarOptionSet]["mods"][mod]["toggle"] = true;
	else
		XBarData[XBarOptionSet]["mods"][mod]["toggle"] = false;
	end
	XBar_UpdateToggle(mod);
end

function XBarConfigSliderScale_OnChange()
	local mod=XBarConfigCurMod:GetText();

	local v=this:GetValue();
	XBarData[XBarOptionSet]["mods"][mod]["scale"] = v;
	XBarConfigSliderScaleVal:SetText(format("%.2f", v));
	XBar_Scale(mod);
end

function XBarConfigCheckMenu_Select()
	local mod=XBarConfigCurMod:GetText();
	local v;
	local c=XBarModData[mod]["check"..this.value];

	if (this.checked) then
		v=false;
	else
		v=true;
	end
	
	XBarData[XBarOptionSet]["mods"][mod][c]=v;
	local f=getfenv()[XBarModData[mod]["foptioncb"]];

	if (f~=nil) then
		f(c,v);
	end
end

function XBarConfigCheckMenuFrame_OnLoad(level)
	-- Loading off the click event from the button itself so we can make sure the data is there.
	-- Reinitializes every time we click the button.
	local info = UIDropDownMenu_CreateInfo();
	local n,i,mod;
	local menukey=nil;

	mod=XBarConfigCurMod:GetText();

	n=XBarModData[mod]["nchecks"];
	for i=1,n do
		info.text = XBarModData[mod]["check"..i];
		info.func = XBarConfigCheckMenu_Select;
		info.owner = this:GetParent();
		info.hasArrow=false;
		info.notCheckable=false;
		if (info.text == "") then
			-- Its probably a menu
			info.func=nil;
			info.text = XBarModData[mod]["mcheck"..i];
			info.value = info.text;
			if (info.text ~= nil) then
				menukey=info.text;
				info.hasArrow=true;
				info.notCheckable=true;
				info.checked=false;
				if (level==1) then
					-- Always add the menu if it's the first one.
					UIDropDownMenu_AddButton(info,1);
				end
			end
		elseif (info.text ~= nil) then
			-- Only put something here if we need.
			info.value = i;
			info.checked=XBarData[XBarOptionSet]["mods"][mod][info.text];
			if ((level==2) and (menukey==UIDROPDOWNMENU_MENU_VALUE)) then
				UIDropDownMenu_AddButton(info,2);
			elseif ((level==1) and (menukey==nil)) then
				--haven't encountered any dropdown menus yet
				UIDropDownMenu_AddButton(info,1);
			end
		end
	end
end

function XBarConfigCheckMenu_OnClick()
	UIDropDownMenu_Initialize(XBarConfigCheckMenuFrame, XBarConfigCheckMenuFrame_OnLoad);
	ToggleDropDownMenu(1, nil, XBarConfigCheckMenuFrame, XBarConfigCheckMenu, 0, 0);
end

function XBarConfigSliderMenu_Select()
	local mod=XBarConfigCurMod:GetText();
	local v=this.value;
	local c=XBarModData[mod]["slider"..v];
	local min=XBarModData[mod]["slider"..v.."min"];
	local max=XBarModData[mod]["slider"..v.."max"];
	local fmt=XBarModData[mod]["slider"..v.."format"];
	local v2=XBarData[XBarOptionSet]["mods"][mod][c];

	XBarFrameDB["XBarConfig"]["sliderindex"]=v;
	XBarConfigSliderMultiText:SetText(c);
	XBarConfigSliderMulti:SetMinMaxValues(min,max);
	XBarConfigSliderMulti:SetValueStep(XBarModData[mod]["slider"..v.."step"]);
	XBarConfigSliderMulti:SetValue(v2);
	XBarConfigSliderMultiHigh:SetText(tostring(max));
	XBarConfigSliderMultiLow:SetText(tostring(min));
	XBarConfigSliderMultiVal:SetText(format(fmt, v2));
	XBarConfigSliderMulti:Show();
end

function XBarConfigSliderMenuFrame_OnLoad(level)
	-- Loading off the click event from the button itself so we can make sure the data is there.
	-- Reinitializes every time we click the button.
	local info = UIDropDownMenu_CreateInfo();
	local n,i,mod;
	local menukey=nil;

	mod=XBarConfigCurMod:GetText();

	n=XBarModData[mod]["nsliders"];
	for i=1,n do
		info.text = XBarModData[mod]["slider"..i];
		info.func = XBarConfigSliderMenu_Select;
		info.owner = this:GetParent();
		info.hasArrow=false;
		info.notCheckable=true;
		if (info.text == "") then
			-- Its probably a menu
			info.func=nil;
			info.text = XBarModData[mod]["mslider"..i];
			info.value = info.text;
			if (info.text ~= nil) then
				menukey=info.text;
				info.hasArrow=true;
				if (level==1) then
					-- Always add the menu if it's the first one.
					UIDropDownMenu_AddButton(info,1);
				end
			end
		elseif (info.text ~= nil) then
			-- Only put something here if we need.
			info.value = i;
			if ((level==2) and (menukey==UIDROPDOWNMENU_MENU_VALUE)) then
				UIDropDownMenu_AddButton(info,2);
			elseif ((level==1) and (menukey==nil)) then
				--haven't encountered any dropdown menus yet
				UIDropDownMenu_AddButton(info,1);
			end
		end
	end
end

function XBarConfigSliderMenu_OnClick()
	UIDropDownMenu_Initialize(XBarConfigSliderMenuFrame, XBarConfigSliderMenuFrame_OnLoad);
	ToggleDropDownMenu(1, nil, XBarConfigSliderMenuFrame, XBarConfigSliderMenu, 0, 0);
end

function XBarConfigSliderMulti_OnChanged()
	-- Don't update the value unless it's visible
	if (not this:IsVisible()) then
		return;
	end

	local mod=XBarConfigCurMod:GetText();
	local c=XBarConfigSliderMultiText:GetText();
	local v=this:GetValue();
	local i=XBarFrameDB["XBarConfig"]["sliderindex"];
	local fmt=XBarModData[mod]["slider"..i.."format"];

	XBarConfigSliderMultiVal:SetText(format(fmt, v));
	XBarData[XBarOptionSet]["mods"][mod][c]=v;
	local f=getfenv()[XBarModData[mod]["foptioncb"]];

	if (f~=nil) then
		f(c,v);
	end
end