--[[---------------------------------------------------------------------------------
  Copyright (c) 2007, James Whitehead II  
  All rights reserved.
  
  WowLua is an interactive interpreter for World of Warcraft
----------------------------------------------------------------------------------]]

WowLua = {
	VERSION = "WowLua 1.0 Interactive Interpreter",
}

local FONT_FILE = "Interface\\AddOns\\WowLua\\fonts\\VeraMono.ttf"

function WowLua:CreateFrame()
    -- Don't create this frame twice
	if self.frame then 
		self.frame:Show()
		return
	end

	self.frame = CreateFrame("Frame", "WowLuaFrame", UIParent)
	local frame = self.frame

	frame:SetHeight(250)
	frame:SetWidth(400)
	frame:SetPoint("CENTER", 0, 0)
	frame:SetFrameStrata("DIALOG")
	frame:SetToplevel(true)

	-- Change the visual style of the frame
	frame:SetBackdrop({
		bgFile = "Interface\\AddOns\\WowLua\\images\\backdrop.tga", 
		edgeFile = "Interface\\AddOns\\WowLua\\images\\borders.tga", tile = true,
		tileSize = 32, edgeSize = 16, 
		insets = {left = 16, right = 16, top = 16, bottom = 16}
	});

	frame:EnableMouse()
	frame:SetClampedToScreen(true)

	frame.titleBar = CreateFrame("Button", nil, frame)
	frame.titleBar:SetHeight(32)
	frame.titleBar:SetPoint("TOPLEFT", frame, "TOPLEFT", 2, -2)
	frame.titleBar:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -2, -2)
	frame:SetMovable(true)

	frame.titleBar:RegisterForDrag("LeftButton")
	frame.titleBar:SetScript("OnDragStart", function() frame:StartMoving() end)
	frame.titleBar:SetScript("OnDragStop", function() frame:StopMovingOrSizing() end)

	frame.headerLeft = frame.titleBar:CreateTexture(nil, "ARTWORK");
	frame.headerLeft:SetTexture("Interface\\AddOns\\WowLua\\images\\headCorner.tga");
	frame.headerLeft:SetWidth(32); frame.headerLeft:SetHeight(32);
	frame.headerLeft:SetPoint("TOPLEFT", 0, 0);

	frame.headerRight = frame.titleBar:CreateTexture(nil, "ARTWORK");
	frame.headerRight:SetTexture("Interface\\AddOns\\WowLua\\images\\headCorner.tga");
	frame.headerRight:SetTexCoord(1,0,0,1);
	frame.headerRight:SetWidth(32); frame.headerRight:SetHeight(32);
	frame.headerRight:SetPoint("TOPRIGHT", 0, 0);

	frame.header = frame.titleBar:CreateTexture(nil, "ARTWORK");
	frame.header:SetTexture("Interface\\AddOns\\WowLua\\images\\header.tga");
	frame.header:SetPoint("TOPLEFT", frame.headerLeft, "TOPRIGHT");
	frame.header:SetPoint("BOTTOMRIGHT", frame.headerRight, "BOTTOMLEFT");
		
	frame.title = frame.titleBar:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall");
	frame.title:SetWidth(200); frame.title:SetHeight(16);
	frame.title:SetPoint("TOP", 0, -2);
		
	frame.footerLeft = frame:CreateTexture(nil, "ARTWORK");
	frame.footerLeft:SetTexture("Interface\\AddOns\\WowLua\\images\\footCorner.tga");
	frame.footerLeft:SetWidth(32); frame.footerLeft:SetHeight(32);
	frame.footerLeft:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", 2, 2);

	frame.footerRight = frame:CreateTexture(nil, "ARTWORK");
	frame.footerRight:SetTexture("Interface\\AddOns\\WowLua\\images\\footCorner.tga");
	frame.footerRight:SetTexCoord(1,0,0,1);
	frame.footerRight:SetWidth(32); frame.footerRight:SetHeight(32);
	frame.footerRight:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -2, 2);

	frame.footer = frame:CreateTexture(nil, "ARTWORK");
	frame.footer:SetTexture("Interface\\AddOns\\WowLua\\images\\footer.tga");
	frame.footer:SetPoint("TOPLEFT", frame.footerLeft, "TOPRIGHT");
	frame.footer:SetPoint("BOTTOMRIGHT", frame.footerRight, "BOTTOMLEFT");

	frame.title:SetText(WowLua.VERSION);

	-- Create the scrolling message frame
	frame.output = CreateFrame("ScrollingMessageFrame", nil, frame)
	frame.output:SetPoint("TOPLEFT", 5, -30)
	frame.output:SetPoint("TOPRIGHT", -5, -30)
	frame.output:SetPoint("BOTTOM", 0, 48)

--	frame.output.bg = frame.output:CreateTexture("BACKGROUND")
--	frame.output.bg:SetAllPoints()
--	frame.output.bg:SetTexture(0, 0, 0, 0.2)

	frame.output:SetFont(FONT_FILE, 14)
	frame.output:SetJustifyH("LEFT")
	frame.output:SetScript("OnMouseWheel", function (self, delta)
											   if delta > 0 then
												   if IsShiftKeyDown() then 
													   self:ScrollToTop()
												   else
													   self:ScrollUp()
												   end
											   else
												   if IsShiftKeyDown() then
													   self:ScrollToBottom()
												   else
													   self:ScrollDown()
												   end
											   end
										   end)

	frame.output:EnableMouseWheel(1)
	frame.output:SetFading(false)
	frame.output:SetMaxLines(1e5)

	-- Create the prompt for the editbox
	frame.prompt = frame:CreateFontString("ARTWORK")
	frame.prompt:SetHeight(16)
	frame.prompt:SetFont(FONT_FILE, 14)
	frame.prompt:SetPoint("TOPLEFT", frame.output, "BOTTOMLEFT", 0, 0)
	frame.prompt:SetText("> ")

	-- Create the editbox
	frame.input = CreateFrame("EditBox", nil, frame)
	frame.input:SetAutoFocus(false)
	frame.input:SetHeight(16)
	frame.input:SetPoint("TOPLEFT", frame.prompt, "TOPRIGHT", 5, 0)
	frame.input:SetPoint("TOPRIGHT", frame.output, "BOTTOMRIGHT", 0, 0)
--	frame.input.bg = frame.input:CreateTexture("BACKGROUND")
--	frame.input.bg:SetAllPoints()
--	frame.input.bg:SetTexture(0, 0, 0, 0.2)

	-- Editbox behavior
	frame.input:SetScript("OnEscapePressed", function(self) self:ClearFocus() end)
	frame.input:SetFont(FONT_FILE, 14)
	frame.input:SetHistoryLines(100)

	-- Register for FAIAP
	self.indent.enable(frame.input)

	-- Process when pressing enter
	frame.input:SetScript("OnEnterPressed", function(self) WowLua:ProcessLine(self:GetText()) end)

	--
	-- Add the resize handle to the frame
	frame.resize = CreateFrame("Button", nil, frame)
	frame.resize:SetNormalTexture("Interface\\AddOns\\WowLua\\images\\resize.tga")
	frame.resize:SetHighlightTexture("Interface\\AddOns\\WowLua\\images\\resize.tga")
	frame.resize:SetWidth(20)
	frame.resize:SetHeight(20)
	frame.resize:SetPoint("BOTTOMRIGHT", 0, 0)

	frame.resize:SetScript("OnMouseDown", function(self) frame:StartSizing() end)
	frame.resize:SetScript("OnMouseUp", function(self) frame:StopMovingOrSizing() end)
	frame.resize:SetScript("OnHide", function(self) frame:StopMovingOrSizing() end)
	frame:SetResizable(true)
	frame:SetMinResize(250, 150)

	--
	-- Close Button

	frame.close = CreateFrame("Button", nil, frame.titleBar, "UIPanelCloseButton")
	frame.close:SetHeight(25)
	frame.close:SetWidth(25)
	frame.close:SetPoint("TOPRIGHT", -5, 3)
	frame.close:SetScript("OnClick", function(self) frame:Hide() end)

	-- In addition, make this frame closeable when pressing Escape
	table.insert(UISpecialFrames, "WowLuaFrame")

	-- The first time we create this window, add a version marker
	frame.output:AddMessage(WowLua.VERSION)
end

local function wowpad_print(...)
	local out = ""
	for i=1,select("#", ...) do
		-- Comma seperate values
		if i > 1 then
			out = out .. ", "
		end

		out = out .. tostring(select(i, ...))
	end
	WowLua.frame.output:AddMessage("|cff999999" .. out .. "|r")
end

local function scrubError(err, orig)
	local pre,excerpt,post = err:match("(%[string \")(.+)(\"%]:%d+:.+)")
	local code = orig:sub(1, #excerpt)
	if #excerpt < #orig then
		code = code .. "..."
	end
	code = code:gsub("\n", " ")
	
	err = pre .. code .. post
	return err
end

local function insertTimeSandbox(cmd)
	-- Use FAIAP's tokenizer to insert the timing code
	local time_code = "do local max_time = time() + 5; if time() >= max_time then error(\"maximum execution time (5s) exceeded\") end end "

	local start = 1
	local tokenType,pos = WowLua.indent.nextToken(cmd, 1)
	local postReturn

	while tokenType do
		local token = cmd:sub(start, pos -1)

		if tokenType == 4 then
			local token = cmd:sub(start, pos - 1)

			-- Insert the timing code
			if token == "return" then
				postReturn = true
				
				local pre = cmd:sub(1, start - 1)
				local post = cmd:sub(start, -1)
				cmd = pre .. time_code .. post
				pos = pos + #time_code
				
			elseif token == "end" then
				if postReturn then
					postReturn = nil
				else
					local pre = cmd:sub(1, start - 1)
					local post = cmd:sub(start, -1)
					cmd = pre .. time_code .. post
					pos = pos + #time_code
				end
			end
		end

		start = pos
		tokenType,pos = WowLua.indent.nextToken(cmd, pos)
	end

	return cmd
end

function WowLua:ProcessLine(text)
	-- escape any color codes:
	local output = text:gsub("\124", "\124\124")
	self.frame.output:AddMessage(self.frame.prompt:GetText() .. output)
	self.frame.input:SetText("")

	self.frame.input:AddHistoryLine(output)

	-- If they're using "= value" syntax, just print it
	text = text:gsub("^%s*=%s*(.+)", "print(%1)")

	-- Store this command into self.cmd in case we have multiple lines
	if self.cmd then
		self.cmd = self.cmd .. "\n" .. text
		self.orig = self.orig .. "\n" .. text
	else
		self.cmd = text
		self.orig = text
	end

	-- Add timing sandbox code
	self.cmd = insertTimeSandbox(self.cmd)

	-- Trim the command before we run it
	self.cmd = string.trim(self.cmd)

	-- Process the current command
	local func,err = loadstring(self.cmd)

	if not func then
		-- Check to see if this is just an unfinished block
		if err:sub(-7, -1) == "'<eof>'" then
			-- Change the prompt
			self.frame.prompt:SetText(">> ")
			return
		end

		err = scrubError(err, self.orig)
		self.frame.output:AddMessage("|cffff0000" .. err .. "|r")
		self.cmd = nil
		self.frame.prompt:SetText("> ")
	else
		-- Make print a global function
		local old_print = print
		print = wowpad_print

		-- Call the function
		local succ,err = pcall(func)

		-- Restore the value of print
		print = old_print

		if not succ then
			err = scrubError(err, self.orig)
			self.frame.output:AddMessage("|cffff0000" .. err .. "|r")
		end

		self.cmd = nil
		self.frame.prompt:SetText("> ")
	end
end

local function slashHandler(txt)
	WowLua:CreateFrame()
	if txt:match("%S") then
		WowLua:ProcessLine(txt)
	end

	WowLua.frame.input:SetFocus()
end

SLASH_WOWLUA1 = "/wowlua"
SLASH_WOWLUA2 = "/lua"
SlashCmdList["WOWLUA"] = slashHandler
