-- define a few globals
WhoHas = {}

WHOHAS_CONFIG_VERSION = 5;

WhoHasConfig = {
   enabled          = 1,
   totals           = 1,
   stacks           = 1,
   inbox            = 1,
   keyring          = 1,
   bags             = 1,
   equipment        = 1,
   allfactions      = nil,
-- added in v2
   version          = WHOHAS_CONFIG_VERSION,
   vault            = 1,
-- added in v3
   disableWorldTips = nil,
-- added in v4
   mines            = 1,
   ores             = 1,
-- added in v5
   ignore           = nil,
   allguilds        = nil,
   vaulttabs        = nil,
}

WhoHasData = {
}

-- set up a local environment
-- every symbol NAME after these three lines can be access from other addons
-- as "WhoHas.NAME".
local global = _G;
setmetatable(WhoHas, { __index = _G });
setfenv(1, WhoHas);

savedName        = "";
player           = "";
realm            = "";
faction          = "";
guild            = nil;
tooltipText      = {};
altCache         = {};
playerCache      = {};
vault            = {};
inventoryChanged = 1;
altsChanged      = 1;
vaultChanged     = 1;
fullVaultRefresh = 1;

config = global.WhoHasConfig;
data   = global.WhoHasData;

-- these are internal strings, not for display
WHOHAS_CATEGORY_INVENTORY = "inventory";
WHOHAS_CATEGORY_BANK      = "bank";
WHOHAS_CATEGORY_INBOX     = "inbox";
WHOHAS_CATEGORY_KEYRING   = "keyring";
WHOHAS_CATEGORY_EQUIPMENT = "equipment";
WHOHAS_CATEGORY_INVBAGS   = "invbags";
WHOHAS_CATEGORY_BANKBAGS  = "bankbags";
WHOHAS_CATEGORY_VAULT     = "vault";
WHOHAS_CATEGORY_TOTAL     = "total";
WHOHAS_CATEGORY_STACK     = "stack";

categories = {
   WHOHAS_CATEGORY_INVENTORY,
   WHOHAS_CATEGORY_BANK,
   WHOHAS_CATEGORY_INBOX,
   WHOHAS_CATEGORY_KEYRING,
   WHOHAS_CATEGORY_EQUIPMENT,
   WHOHAS_CATEGORY_INVBAGS,
   WHOHAS_CATEGORY_BANKBAGS,
}

-- import a few globals 
-- we can only do this with static tables and functions
-- plain variables and variables that bind to different tables
-- (such as the global "this") must be looked up explicitly each time
-- this is slightly faster than the metatable, but a pain to maintain
---- Lua stuff
--local pairs            = global.pairs;
--local ipairs           = global.ipairs;
--local table            = global.table;
--local string           = global.string;
--local getglobal        = global.getglobal;
---- WoW stuff
--local ItemRefTooltip   = global.ItemRefTooltip;
--local GameTooltip      = global.GameTooltip;
--local GetItemInfo      = global.GetItemInfo;
--local UnitName         = global.UnitName;
--local GetRealmName     = global.GetRealmName;
--local UnitFactionGroup = global.UnitFactionGroup;
--local GetGuildInfo     = global.GetGuildInfo;

-------------------------------------------------------------------------------
-- OnLoad
-------------------------------------------------------------------------------

function OnLoad()
   SlashCmdList["WHOHAS"] = SlashCommandHandler;
   table.insert(UISpecialFrames, "WhoHasConfigFrame");

   if (PossessionsData) then
      if (POSS_USEDBANKBAGS_CONTAINER) then
         backend = Scanner.Possessions.Lauchsuppe;
      else
         backend = Scanner.Possessions.Siz;
      end
   elseif (myProfile) then
      backend = Scanner.CharacterProfiler;
   else
      backend = Scanner.Default;
   end

   HookFunction("ReturnInboxItem", ReturnInboxItem);
   HookFunction("SendMail", SendMail);

   HookMethod(ItemRefTooltip, "SetHyperlink", ShowTooltip);

   HookScript(GameTooltip, "OnTooltipSetItem", OnTooltipSetItem);
   HookScript(GameTooltip, "OnTooltipCleared", OnTooltipCleared);

   if (Possessions_ItemButton_OnEnter) then
      Orig_Possessions_ItemButton_OnEnter = Possessions_ItemButton_OnEnter
      Possessions_ItemButton_OnEnter      = Possessions_ItemButton_OnEnter
   end

   if (IsAddOnLoaded("Blizzard_GuildBankUI")) then
      HookScript(GuildBankFrame, "OnShow", LoadGuildBank);
      HookFunction("GuildBankFrame_Update", ScanGuildBank);
   end

   for event in pairs(events) do
      this:RegisterEvent(event);
   end
end

-------------------------------------------------------------------------------
-- Slash commands
-------------------------------------------------------------------------------

function SlashCommandHandler(msg)
   if (not msg or msg == "") then
      ShowConfigFrame();
   else
      local cmd, arg = string.match(msg, "(.-) (.*)");
      if (cmd and arg and cmd == text.ignore) then
         arg = strtrim(arg, " \t\r\n'\"");
         local item = GetItemInfo(arg);
         if (item) then
            config.ignore = config.ignore or {};
            config.ignore[item] = 1;
            DEFAULT_CHAT_FRAME:AddMessage(string.format(formats.ignore, item));
         else
            DEFAULT_CHAT_FRAME:AddMessage(string.format(formats.noitem, arg));
         end
      else
         DEFAULT_CHAT_FRAME:AddMessage(text.usage);
      end
   end
end

-------------------------------------------------------------------------------
-- Utility functions
-------------------------------------------------------------------------------

function ShowConfigFrame()
   WhoHasConfigFrame:Show();
end

function InventoryChanged()
   inventoryChanged = 1;
end

-- pass non-nil to force a complete refresh
function VaultChanged(refresh)
   vaultChanged = 1;
   fullVaultRefresh = refresh;
end

function OptionsChanged()
   inventoryChanged = 1;
   altsChanged = 1;
   vaultChanged = 1;
   fullVaultRefresh = 1;
end

function UpdateGuild()
   if (guild ~= GetGuildInfo("player")) then
      guild = GetGuildInfo("player");
      VaultChanged();
   end
end

function DoNothing()
end

function debug(...)
   DEFAULT_CHAT_FRAME:AddMessage(...);
end

-------------------------------------------------------------------------------
-- Events
-------------------------------------------------------------------------------

events = {}

function OnEvent()
   local func = events[event];
   if (func) then
      func(arg1);
   end
end

function events.ADDON_LOADED(name)
   if (name == "WhoHas") then
      DEFAULT_CHAT_FRAME:AddMessage(backend:GetAnnounce());
      player  = UnitName("player");
      realm   = GetRealmName();
      faction = UnitFactionGroup("player");
      UpdateGuild();
   elseif (name == "Blizzard_GuildBankUI") then
      HookScript(GuildBankFrame, "OnShow", LoadGuildBank);
      HookFunction("GuildBankFrame_Update", ScanGuildBank);
   end
end

function events.VARIABLES_LOADED()
   -- make sure we have aliases to the loaded data
   config = WhoHasConfig;
   data   = WhoHasData;
   -- convert old config format to new
   if (not config.version or config.version < 2) then
      config.vault = 1;
      config.version = 2;
   end
   if (config.version < 4) then
      config.mines = 1;
      config.ores = 1;
   end
   if (config.version < WHOHAS_CONFIG_VERSION) then
      config.version = WHOHAS_CONFIG_VERSION;
   end
   if (LinkWrangler) then
      LinkWrangler.RegisterCallback("WhoHas", ShowTooltip, "refresh");
   end
end

events.PLAYER_GUILD_UPDATE    = UpdateGuild;
events.PLAYER_ENTERING_WORLD  = UpdateGuild;
events.UNIT_INVENTORY_CHANGED = InventoryChanged;
events.BAG_UPDATE             = InventoryChanged;

function OnShow()
   -- OnShow is the backup entry method.  This is only used for
   -- hover tooltips in the world and minimap.
   if (not GameTooltip.WhoHasShowing and not GameTooltip.WhoHasRecipe and not config.disableWorldTips) then
      GameTooltip.WhoHasShowing = 1;
      ShowTooltip(GameTooltip);
   end
end

function OnHide()
   GameTooltip.WhoHasShowing = nil;
   GameTooltip.WhoHasRecipe = nil;
end

function OnTooltipSetItem()
   -- OnTooltipSetItem is the preferred entry method.  This catches
   -- most tooltips.
   if (not this.WhoHasShowing) then
      -- recipes call OnTooltipSetItem twice for some reason, with
      -- an OnShow in between.  We only want to mod the tooltip
      -- after the second SetItem.
      if (not this.WhoHasRecipe) then
         local name, link = this:GetItem();
         if (link) then
            local _, _, _, _, _, type, subtype = GetItemInfo(link);
            if (type == "Recipe" and subtype ~= "Enchanting") then
               this.WhoHasRecipe = 1;
               return;
            end
         end
      end
      this.WhoHasShowing = 1;
      ShowTooltip(this);
   end
end

function OnTooltipCleared()
   this.WhoHasShowing = nil;
   this.WhoHasRecipe = nil;
end

function WhoHasConfigFrame_OnShow()
end

-------------------------------------------------------------------------------
-- Hooks
-------------------------------------------------------------------------------

function Possessions_ItemButton_OnEnter(args)
   -- don't doctor tooltips inside of Possessions
   skip = true;
   Orig_Possessions_ItemButton_OnEnter(args);
   skip = nil;
end

-------------------------------------------------------------------------------
-- Mail handling
-------------------------------------------------------------------------------

function SendMail(target, subject, body)
   if (config.inbox) then
      -- proper-case the name
      target = string.upper(string.sub(target, 1, 1)) .. string.lower(string.sub(target, 2));
      if (altCache[target]) then
         for i = 1, 12 do
            local item, _, qty, _ = GetSendMailItem(i);
            if (item) then
               altCache[target][item] = altCache[target][item] or {};
               altCache[target][item].inbox = (altCache[target][item].inbox or 0) + qty;
            end
         end
      end
      InventoryChanged()
   end
end

function ReturnInboxItem(mailID)
   if (config.inbox) then
      local _, _, sender, _ = GetInboxHeaderInfo(mailID);

      if (altCache[sender]) then
         for i = 1, 12 do
            local item, _, qty, _ = GetInboxItem(mailID, i);
            if (item) then
               altCache[sender][item] = altCache[sender][item] or {};
               altCache[sender][item].inbox = (altCache[sender][item].inbox or 0) + qty;
            end
         end
      end

      InventoryChanged()
   end
end

-------------------------------------------------------------------------------
-- Hooking functions
-------------------------------------------------------------------------------

function HookMethod(object, name, after, before)
   local func = object[name];
   if (func) then
      object[name] = function(self, ...)
                        if (before) then
                           before(self, ...);
                        end
                        r1, r2, r3, r4, r5, r6 = func(self, ...);
                        if (after) then
                           after(self, ...);
                        end
                        return r1, r2, r3, r4, r5, r6;
                     end
   end
end

function HookFunction(name, after, before)
   local func = global[name];
   if (func) then
      global[name] = function(...)
                    if (before) then
                       before(...);
                    end
                    r1, r2, r3, r4, r5, r6 = func(...);
                    if (after) then
                       after(...);
                    end
                    return r1, r2, r3, r4, r5, r6;
                 end
   end
end

function HookScript(object, name, after, before)
   local func = object:GetScript(name);
   if (func) then
      local newfunc = function(...)
                         if (before) then
                            before(...);
                         end
                         r1, r2, r3, r4, r5, r6 = func(...);
                         if (after) then
                            after(...);
                         end
                         return r1, r2, r3, r4, r5, r6;
                      end
      object:SetScript(name, newfunc);
   end
end

-------------------------------------------------------------------------------
-- Tooltip display
-------------------------------------------------------------------------------

function ShowTooltip(tooltip)
   if (tooltip and config.enabled and not skip) then
      local name = getglobal(tooltip:GetName().."TextLeft1"):GetText();
      if (not name or name == "") then
         return;
      end

      if (config.ignore and config.ignore[name]) then
         return;
      end

      UpdateCaches();
      UpdateTooltipText(name);
      AddTextToTooltip(tooltip);

      tooltip:Show();
   end
end

function UpdateCaches()
   if (altsChanged) then
      backend:ScanAlts();
      altsChanged = nil;
      savedName = "";
   end
   
   if (inventoryChanged) then
      backend:ScanPlayer();
      inventoryChanged = nil;
      savedName = "";
   end
   
   if (vaultChanged) then
      if (fullVaultRefresh) then
         vault = {};
         fullVaultRefresh = nil;
      end
      backend:ScanVault();
      vaultChanged = nil;
      savedName = "";
   end
end

function UpdateTooltipText(name)
   if (name ~= savedName) then
      tooltipText = {};
      savedName = name;
      GetText(name, tooltipText);
      
      if (config.mines) then
         while mines[name] do
            name = mines[name];
            table.insert(tooltipText, " ");
            table.insert(tooltipText, "(" .. name .. ")");
            if (GetText(name, tooltipText) == 0) then
               table.remove(tooltipText);
               table.remove(tooltipText);
            end
         end
      end
      
      if (config.ores) then
         while xlat[name] do
            name = xlat[name];
            table.insert(tooltipText, " ");
            table.insert(tooltipText, "(" .. name .. ")");
            if (GetText(name, tooltipText) == 0) then
               table.remove(tooltipText);
               table.remove(tooltipText);
            end
         end
      end
      
      if (config.ores) then
         if (enchant[savedName]) then
            name = enchant[savedName];
            table.insert(tooltipText, " ");
            table.insert(tooltipText, "(" .. name .. ")");
            if (GetText(name, tooltipText) == 0) then
               table.remove(tooltipText);
               table.remove(tooltipText);
            end
         end
      end
   end
end

function AddTextToTooltip(tooltip)
   for i, line in ipairs(tooltipText) do
      tooltip:AddLine(line);
   end
end

function GetText(name, text)
   local total = ListOwners(name, text);
   if (config.totals and total > 0) then
      table.insert(text, string.format(formats.total, total));
   end
   if (config.stacks) then
      local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount, itemEquipLoc, invTexture = GetItemInfo(name);
      if (itemStackCount and itemStackCount > 1) then
         table.insert(text, string.format(formats.stack, itemStackCount));
      end
   end
   return total;
end

function ListOwners(name, text)
   local total = 0;
   if (playerCache[name]) then
      total = total + ListChar(player, playerCache[name], text);
   end
   for charName, charData in pairs(altCache) do
      if (charData[name]) then
         total = total + ListChar(charName, charData[name], text);
      end
   end
   if (config.vault and vault) then
      total = total + ListVaults(name, text);
   end
   return total;
end

function ListChar(charName, charData, text)
   local total = 0;
   for i, category in ipairs(categories) do
      local count = charData[category];
      if count and count > 0 then
         table.insert(text, string.format(formats[category], count, charName));
         total = total + count;
      end
   end
   return total;
end

function ListVaults(name, text)
   local total = 0;
   for guild, data in pairs(vault) do
      if (guild and data) then
         if (data.hasTabs) then
            for tab = 1,6 do
               if (data[tab] and data[tab][name]) then
                  local count = data[tab][name];
                  if (config.allguilds) then
                     table.insert(text, string.format(formats.multivaulttab, count, guild, tab));
                  else
                     table.insert(text, string.format(formats.vaulttab, count, tab));
                  end
                  total = total + count;
               end
            end
         else
            if (data[name]) then
               local count = data[name];
               if (config.allguilds) then
                  table.insert(text, string.format(formats.multivault, count, guild));
               else
                  table.insert(text, string.format(formats.vault, count));
               end
               total = total + count;
            end
         end
      end
   end
   return total;
end

-------------------------------------------------------------------------------
-- Inventory scanning support
-------------------------------------------------------------------------------

Scanner = {}
Scanner.Default = { key = "none" }

function Scanner.Default:new(o)
   o = o or {};
   setmetatable(o, self);
   self.__index = self;
   return o;
end

function Scanner.Default:GetAnnounce()
   return support[self.key] or "";
end

function Scanner.Default:HasVaultData()
   return nil;
end

function Scanner.Default:ScanPlayer()
end

function Scanner.Default:ScanAlts()
end

function Scanner.Default:ScanVault()
   if (config.vault and data and data[realm]) then
      guild = GetGuildInfo("player");
      for who, items in pairs(data[realm]) do
         -- only show our vault if allguilds is not set,
         -- otherwise show only our faction, or vaults without a faction
         if (who == guild or (config.allguilds and items.faction == faction)) then
            if (config.vaulttabs) then
               vault[who] = items;
               vault[who].hasTabs = 1;
            else
               -- only scan the current guild and guilds we haven't scanned yet
               -- alt guild data can't change
               if (who == guild) then
                  vault[who] = nil;
               end
               if (not vault[who]) then
                  vault[who] = {};
                  for tab = 1, 6 do
                     if (items[tab]) then
                        for name, count in pairs(items[tab]) do
                           vault[who][name] = count + (vault[who][name] or 0);
                        end
                     end
                  end
               end
            end
         end
      end
   end
end

-------------------------------------------------------------------------------
-- Possessions support
-------------------------------------------------------------------------------

Scanner.Possessions = Scanner.Default:new();
Scanner.Possessions.Siz = Scanner.Possessions:new{ key = "SizPoss" };
Scanner.Possessions.Lauchsuppe = Scanner.Possessions:new{ key = "LSPoss" };

Scanner.Possessions.Lauchsuppe.slots = {
   Inventory = { 0, 1, 2, 3, 4 },
   Bank      = { -1, 5, 6, 7, 8, 9, 10, 11 },
   Keyring   = { -2 },
   Equipment = { -3 },
   Inbox     = { -4 },
   InvBags   = { -5 },
   BankBags  = { -6 }
}

Scanner.Possessions.Siz.slots = {
   Inventory = { 0, 1, 2, 3, 4 },
   Bank      = { -1, 5, 6, 7, 8, 9, 10, 11 },
   Equipment = { -2 },
   Inbox     = { -3 },
   Keyring   = { -4 },
   InvBags   = { -5 },
   BankBags  = { -6 },
   Vault     = { -7 }
}

function Scanner.Possessions.Siz:HasVaultData()
   -- Possessions doesn't track tabs
   return not config.vaulttabs;
end

function Scanner.Possessions:ScanPlayer()
   local charData;
   if (PossessionsData and PossessionsData[realm]) then
      charData = PossessionsData[realm][string.lower(player)];
   end
   if (charData) then
      playerCache = {};
      self:ScanChar(player, charData, playerCache);
   end
end

function Scanner.Possessions:ScanAlts()
   if (PossessionsData and PossessionsData[realm]) then
      for charName, charData in pairs(PossessionsData[realm]) do
         if (charName and charData and (config.allfactions or charData.faction == faction)) then
            -- Possessions lower-cases character names, annoyingly
            charName = string.upper(string.sub(charName, 1, 1)) .. string.sub(charName, 2);
            if (charName ~= player) then
               altCache[charName] = {};
               self:ScanChar(charName, charData, altCache[charName]);
            end
         end
      end
   end
end

function ClearVault()
   -- this is a placeholder to test zeroing out the vault instead of
   -- replacing the whole table
   vault = {};
end

function CleanupVault()
   -- this is meant to delete entries with zero counts, if we need to
end

function Scanner.Possessions.Siz:ScanVault()
   if (config.vaulttabs) then
      -- Possessions doesn't track tabs
      Scanner.Default.ScanVault(self);
   else
      vault = vault or {};
      if (config.vault and self.slots.Vault) then
         if (PossessionsData and PossessionsData[realm]) then
            guild = GetGuildInfo("player");
            for who, data in pairs(PossessionsData[realm]) do
               -- see if it's a vault
               -- note: this isn't portable.  fix the -7 index
               if (data and data.items and data.items[-7]) then
                  -- only show our vault if allguilds is not set,
                  -- otherwise show only our faction, or vaults without a faction
                  if (who == guild or (config.allguilds and (not data.faction or data.faction == faction))) then
                     -- only scan the current guild and guilds we haven't scanned yet
                     -- alt guild data can't change
                     if (who == guild) then
                        vault[who] = nil;
                     end
                     if (not vault[who]) then
                        vault[who] = {};
                        local bags = data.items;
                        for _, index in pairs(self.slots.Vault) do
                           if (bags[index]) then
                              for i, item in pairs(bags[index]) do
                                 if (item and item[1]) then
                                    local name = item[1];
                                    local count = item[3] or 1;
                                    vault[who][name] = count + (vault[who][name] or 0);
                                 end
                              end
                           end
                        end
                     end
                  end
               end
            end
         end
      end
   end
end

function Scanner.Possessions:ScanChar(charName, charData, cache)
   if (charData and charData.items) then
      self:ScanBags(charName, WHOHAS_CATEGORY_INVENTORY, charData.items, self.slots.Inventory, cache);
      self:ScanBags(charName, WHOHAS_CATEGORY_BANK, charData.items, self.slots.Bank, cache);
      if (config.inbox) then
         self:ScanBags(charName, WHOHAS_CATEGORY_INBOX, charData.items, self.slots.Inbox, cache);
      end
      if (config.keyring) then
         self:ScanBags(charName, WHOHAS_CATEGORY_KEYRING, charData.items, self.slots.Keyring, cache);
      end
      if (config.equipment) then
         self:ScanBags(charName, WHOHAS_CATEGORY_EQUIPMENT, charData.items, self.slots.Equipment, cache);
      end
      if (config.bags and self.slots.InvBags) then
         self:ScanBags(charName, WHOHAS_CATEGORY_INVBAGS, charData.items, self.slots.InvBags, cache);
      end
      if (config.bags and self.slots.BankBags) then
         self:ScanBags(charName, WHOHAS_CATEGORY_BANKBAGS, charData.items, self.slots.BankBags, cache);
      end
   end
end

function Scanner.Possessions:ScanBags(char, slot, bags, bagIndex, cache)
   for _, index in pairs(bagIndex) do
      if (bags[index]) then
         for i, item in pairs(bags[index]) do
            if (item and item[1]) then
               local name = item[1];
               local count = item[3] or 1;
               if (not cache[name]) then
                  cache[name] = {};
               end
               cache[name][slot] = count + (cache[name][slot] or 0);
            end
         end
      end
   end
end

-------------------------------------------------------------------------------
-- CharacterProfiler support
-------------------------------------------------------------------------------

Scanner.CharacterProfiler = Scanner.Default:new{ key = "CP" };

function Scanner.CharacterProfiler:ScanPlayer()
   local charData;
   if (myProfile and myProfile[realm] and myProfile[realm].Character) then
      charData = myProfile[realm].Character[player];
   end
   if (charData) then
      playerCache = {};
      self:doBags(charName, charData.Inventory, WHOHAS_CATEGORY_INVENTORY, formats.Inventory, playerCache);
      self:doBags(charName, charData.Bank, WHOHAS_CATEGORY_BANK, formats.Bank, playerCache);
      if (config.inbox) then
         self:doInbox(charName, charData.MailBox, WHOHAS_CATEGORY_INBOX, formats.Inbox, playerCache);
      end
   end
end

function Scanner.CharacterProfiler:ScanAlts()
   if (myProfile and myProfile[realm] and myProfile[realm].Character) then
      for charName, charData in pairs(myProfile[realm].Character) do
         if (charName ~= player and (config.allfactions or charData and charData.FactionEn == faction)) then
            altCache[charName] = {};
            self:doBags(charName, charData.Inventory, WHOHAS_CATEGORY_INVENTORY, formats.Inventory, altCache[charName]);
            self:doBags(charName, charData.Bank, WHOHAS_CATEGORY_BANK, formats.Bank, altCache[charName]);
            if (config.inbox) then
               self:doInbox(charName, charData.MailBox, WHOHAS_CATEGORY_INBOX, formats.Inbox, altCache[charName]);
            end
         end
      end
   end
end

function Scanner.CharacterProfiler:doBags(char, bags, slot, format, cache)
   if (bags) then
      for bag, bagData in pairs(bags) do
         if (bagData.Slots) then
            for i = 1, bagData.Slots do
               local item = bagData.Contents[i]
               if (item and item.Name) then
                  local count = item.Quantity or 1;
                  if (not cache[item.Name]) then
                     cache[item.Name] = {};
                  end
                  cache[item.Name][slot] = count + (cache[item.Name][slot] or 0);
               end
            end
         end
      end
   end
end

function Scanner.CharacterProfiler:doInbox(char, inbox, slot, format, cache)
   if (inbox) then
      for i, msg in ipairs(inbox) do
         if (msg) then
            if (msg.Item and msg.Item.Name) then
               -- pre-2.3 CP
               local item = msg.Item;
               local count = item.Quantity or 1;
               if (not cache[item.Name]) then
                  cache[item.Name] = {};
               end
               cache[item.Name][slot] = count + (cache[item.Name][slot] or 0);
            elseif (msg.Contents) then
               -- post-2.3 CP
               for i, item in pairs(msg.Contents) do
                  if (item and item.Name) then
                     local count = item.Quantity or 1;
                     if (not cache[item.Name]) then
                        cache[item.Name] = {};
                     end
                     cache[item.Name][slot] = count + (cache[item.Name][slot] or 0);
                  end
               end
            end
         end
      end
   end
end

-------------------------------------------------------------------------------
-- Guild Bank support
-------------------------------------------------------------------------------

-- these scanning functions are always hooked into the GuildBank, but
-- they don't actually gather any data if we can use data from another
-- backend instead.

-- so far, Siz'z Possessions is the only backend that qualifies, and then
-- only if the user doesn't want to see individual guild bank tabs.

function LoadGuildBank()
   if (config.vault and not backend:HasVaultData() and not vaultLoaded) then
      guild = GetGuildInfo("player");
      if (guild) then
         -- suppress updates temporarily
         config.vault = nil;
         for tab = 1, GetNumGuildBankTabs() do
            QueryGuildBankTab(tab);
         end
         -- restore updates and force update now
         vaultLoaded = 1;
         config.vault = 1;
         ScanGuildBank();
      end
   end
end

function ScanGuildBank()
   if (config.vault and not backend:HasVaultData()) then
      guild = GetGuildInfo("player");
      if (guild) then
         local cache = {};
         cache.faction = faction;
         for tab = 1, GetNumGuildBankTabs() do
            cache[tab] = {};
            for slot = 1, MAX_GUILDBANK_SLOTS_PER_TAB do
               local texture, count, _ = GetGuildBankItemInfo(tab, slot);
               if (texture) then
                  local link = GetGuildBankItemLink(tab, slot);
                  if (link) then
                     local name = GetItemInfo(link);
                     cache[tab][name] = count + (cache[tab][name] or 0);
                  end
               end
            end
         end
         data[realm] = data[realm] or {};
         data[realm][guild] = cache;
      end
   end
   VaultChanged();
end
