-----------------------------------
-----------------------------------
-- Whisp by Anea
--
-- Makes chatting fun again
-- Type /whisp in-game for options
-----------------------------------
-- core.lua
-- Main routines and functionionality
-----------------------------------

-- Create Ace3 instance
Whisp = LibStub("AceAddon-3.0"):NewAddon("Whisp", "AceConsole-3.0", "AceHook-3.0", "AceEvent-3.0", "AceTimer-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("Whisp")
Whisp.media = AceLibrary("LibSharedMedia-3.0")
Whisp:SetDefaultModuleState(false)


-- Default settings and variables
local defaults = {
	profile = {
		modules = {}, 										 -- Module settings
		version = 0,                       -- The version we are running
		timeout = 720,                     -- How long should conversations be saved in hours (0 indicates session)
		notifyDebug = false,               -- Shows debug messages
	},
	char = {
		linesShown = 10,                   -- How many lines to show in the panes
		paneSortDown = true,               -- Show messages in the pane newest down

		-- Customisation
		paneColor = {0.09, 0.09, 0.09, 0.8},  -- Color for the background of the panes
		paneBackground = "Blizzard Tooltip",
		paneBorder = "Blizzard Tooltip",
		paneBorderColor = {0.5, 0.5, 0.5, 0.8},
		paneFont = "Arial Narrow",
		fontSize = 14,
		timeFormat = "%H:%M:%S",
		timeColor = {1,1,1,1},
		timeColorByChannel = false,
		colorIncoming = {1,0.5,1},         -- Color for incoming messages
		colorOutgoing = {0.73,0.73,1},     -- Color for outgoing messages
	},
	realm = {
		chatHistory = {}                   -- All conversations
	},
}

-- local variables
Whisp.currentBuild = 20080704        -- Latest build
Whisp.tellTarget = nil               -- Current tell-target
Whisp.lastSender = nil               -- Person who last send you a tell
Whisp.editBoxFrame = {}              -- Pane to attach to the editbox
Whisp.tooltipFrame = {}              -- Pane to attach to fubar
Whisp.clickableTooltip = true        -- Ensures that the FuBar tooltip is clickable
Whisp.sessionStart = 0               -- Start time of this session

-----------------------------------
-----------------------------------
-- Initialisation functions

function Whisp:OnInitialize()
	self:TransferLegacyChatHistory()
	self.db = LibStub("AceDB-3.0"):New("WhispDB", defaults, "Default")
  self.sessionStart = time() - 30
  self:SetupOptions()
end

function Whisp:OnEnable()
  -- Cache incoming and outgoing whispers
  self:RegisterEvent("CHAT_MSG_WHISPER", Whisp.ChatEventIncoming)
  self:RegisterEvent("CHAT_MSG_WHISPER_INFORM", Whisp.ChatEventOutgoing)

  -- Schedules
  self:ScheduleRepeatingTimer(Whisp.CollectGarbage, 300)
  
	-- Load Modules  
	for k, v in self:IterateModules() do
		if self.db.profile.modules[k] ~= false then
			v:Enable()
		end
	end

	-- Versioning
	self:CheckVersion()
end

function Whisp:OnDisable()
  self:UnregisterAllEvents()

	-- UnLoad Modules  
	for k, v in self:IterateModules() do
		if self.db.profile.modules[k] ~= false then
			v:Disable()
		end
	end
end

-----------------------------------
-----------------------------------
-- Events

function Whisp:ChatEventIncoming(msg,plr)
  Whisp:UpdateChatHistory(msg,plr, plr, true)
  Whisp:SendMessage("WHISP_MESSAGE")
end

function Whisp:ChatEventOutgoing(msg,plr)
  Whisp:UpdateChatHistory(msg,plr, UnitName("player"), false)
  Whisp:SendMessage("WHISP_MESSAGE")
end

-----------------------------------
-----------------------------------
-- Generic Frame functions

function Whisp:CreateMyFrame(name, width)
  local frame = CreateFrame("Frame", "WHISP_"..name, UIParent)
  frame:ClearAllPoints()
  -- Size is required to be set or SetPoint() won't work later on
  frame:SetWidth(width)  
  frame:SetHeight(20)
  frame:SetMinResize(200,20)
  frame:SetFrameLevel(GameTooltip:GetFrameLevel() + 10)
  -- Setup textfields
  frame.text = frame:CreateFontString("WHISP_EDITBOXTEXT","OVERLAY","ChatFontNormal")
  frame.text:SetJustifyH("LEFT")
  frame.text:SetJustifyV("TOP")
  frame.text:SetTextColor(1.0,1.0,1.0,1.0)
  -- Set background and border
  frame:SetFrameStrata("TOOLTIP")
  Whisp:SkinMyFrame(frame)
  frame:Hide()
  return frame
end

function Whisp:UpdateMyFrame(frame, plr)
  frame.text:SetText(Whisp:GetChatHistory(plr, Whisp.db.char.linesShown))
  frame.text:SetPoint("TOPLEFT",frame,"TOPLEFT",5,-5)
  -- Adjust height of the frame to the height of the text
  frame.text:SetWidth(frame:GetRight() - frame:GetLeft() - 10)
  frame:SetHeight(frame.text:GetHeight()+15)
end

function Whisp:SkinMyFrame(frame)
  -- Set border and background texture
  local bg = Whisp.media:Fetch('background', Whisp.db.char.paneBackground, "Blizzard Tooltip")
  local ed = Whisp.media:Fetch('border', Whisp.db.char.paneBorder, "Blizzard Tooltip")
  frame:SetBackdrop({
      bgFile = bg, tile = true, tileSize = 16,
      edgeFile = ed, edgeSize = 16,
      insets = {left = 3, right = 3, top = 3, bottom = 3},
  })
  -- Set font
  local font = Whisp.media:Fetch('font', Whisp.db.char.paneFont)
  local _, _, style = frame.text:GetFont()
  frame.text:SetFont(font, Whisp.db.char.fontSize, style)

  -- Set background-color
  local c,d = Whisp.db.char.paneColor, Whisp.db.char.paneBorderColor
  frame:SetBackdropColor(c[1], c[2], c[3], c[4])
  frame:SetBackdropBorderColor(d[1], d[2], d[3], d[4])
end

-----------------------------------
-----------------------------------
-- Log window functions

function Whisp:ShowLogFrame(plr)
  if plr and Whisp.db.realm.chatHistory[plr] then
    local msg = Whisp:GetChatHistory(plr, #Whisp.db.realm.chatHistory[plr].time)
    if msg then
      Whisp_EditBox:SetText(msg)
      WhispUIParent:Show()
    end
  end
end

-----------------------------------
-----------------------------------
-- ChatHistory

-- Chat message event
function Whisp:UpdateChatHistory(msg, plr, snd, incoming)
  if not plr then return end
  if Whisp:HideAddonMessage(msg, plr) then return end
  plr = Whisp:ToName(plr)
  snd = Whisp:ToName(snd)
  Whisp.lastSender = plr
  -- Insert new message
  if not Whisp.db.realm.chatHistory[plr] then
    Whisp.db.realm.chatHistory[plr] = {sender = {}, message = {}, time = {}, incoming = {}}
  end
  tinsert(Whisp.db.realm.chatHistory[plr].sender, snd)
  tinsert(Whisp.db.realm.chatHistory[plr].message, msg)
  tinsert(Whisp.db.realm.chatHistory[plr].incoming, incoming)
  tinsert(Whisp.db.realm.chatHistory[plr].time, time())
end

-- Returns the combined recent messages from this player
function Whisp:GetChatHistory(plr, max)
  local msg = ""
  if plr then
    plr = Whisp:ToName(plr)
    if Whisp.db.realm.chatHistory[plr] then
      local n = #Whisp.db.realm.chatHistory[plr].message
      local output = {}
      -- Generate conversation
      for i= n>max and n-max or 1, n do
        local color = Whisp.db.realm.chatHistory[plr].incoming[i] and Whisp.db.char.colorIncoming or Whisp.db.char.colorOutgoing
        local t = Whisp:FormatTimeStamp(Whisp.db.realm.chatHistory[plr].time[i], color)
        local m = Whisp:Colorise(Whisp.db.realm.chatHistory[plr].message[i], Whisp:HexColor(color))
        local s = Whisp:FormatPlayerName(Whisp.db.realm.chatHistory[plr].sender[i])
        tinsert(output, t.." "..s..": "..m)
      end
      -- Sort output
      local pd = Whisp.db.char.paneSortDown
      for i = pd and 1 or #output, pd and #output or 1, pd and 1 or -1 do
        msg = msg .. output[i]
        if (pd and (i<#output)) or ((not pd) and (i>1)) then 
          msg = msg.."\n"
        end
      end
    end
  end
  if msg == "" then return nil end
  return msg
end

-- Garbage collection of message cache
function Whisp:CollectGarbage()
  local t = Whisp.sessionStart
  if Whisp.db.profile.timeout > 0 then 
    t = time() - (3600 * Whisp.db.profile.timeout)
  end

  -- Remove full entry of the player if the conversation has timed out
  for i,v in pairs(Whisp.db.realm.chatHistory) do
    local ct = Whisp.db.realm.chatHistory[i].time[#Whisp.db.realm.chatHistory[i].time]
    if ct and ct < t then
      Whisp.db.realm.chatHistory[i] = nil
    end
  end

  -- Remove individual entries of the player if they have timed out
  for i,v in pairs(Whisp.db.realm.chatHistory) do
    for j=1, #Whisp.db.realm.chatHistory[i].time do
      if Whisp.db.realm.chatHistory[i].time[j] and Whisp.db.realm.chatHistory[i].time[j] < t then
        tremove(Whisp.db.realm.chatHistory[i].sender,j)
        tremove(Whisp.db.realm.chatHistory[i].message,j)
        tremove(Whisp.db.realm.chatHistory[i].time,j)
        tremove(Whisp.db.realm.chatHistory[i].incoming,j)
      end
    end
  end

  Whisp:SendMessage("WHISP_MESSAGE")
end

-----------------------------------
-----------------------------------
-- Hide addon messages 

function Whisp:HideAddonMessage(msg, plr)
  if DBM and DBM.HiddenWhisperMessages[string.gsub(msg, "%%", "")] then
    return true
  elseif (strsub(msg, 1,4) == "LVBM") then
    return true
  end
  return false
end

-----------------------------------
-----------------------------------
-- Utility and other functions

-- Hooks to the utf8-library
local strutf8upper = _G.string.utf8upper
local strutf8lower = _G.string.utf8lower
local strutf8sub = _G.string.utf8sub

-- Output message to the chatframe
-- Level can be either:
-- 0: System message
-- 3: Debug
function Whisp:Msg(level, text)
  if not text then return end
  if level==0
  or level==3 and self.db.profile.notifyDebug then
    DEFAULT_CHAT_FRAME:AddMessage(text, 0.6, 0.6, 1)
  end
end

-- Create a link to a playername
function Whisp:FormatPlayerName(name)
  return "|cffffff00"..name.."|r"
end

function Whisp:FormatTimeStamp(t, c)
  if t < time() - 86400 then
    t = date("%d/%m - "..Whisp.db.char.timeFormat, t)
  else
    t = date(Whisp.db.char.timeFormat, t)
  end
  return Whisp:Colorise(t, Whisp:HexColor(Whisp.db.char.timeColorByChannel and c or Whisp.db.char.timeColor))
end

-- Returns the hexvalue of the rgb
function Whisp:HexColor(r, g, b)
  if type(r)=="table" then
    r, g, b = r[1], r[2], r[3]
  end
  return string.format("%02x%02x%02x", 255*r, 255*g, 255*b)
end

-- Colorise the text
function Whisp:Colorise(text, hexcolor)
  text = gsub(text, "(|c.-|H.-|r)", "|r%1|cff"..hexcolor)
  return "|cff"..hexcolor..text.."|r"
end

local nameCache = {}
setmetatable(nameCache, {__mode = "kv"})
-- Normalise a name to capital-lower format, pick name from cache if available
function Whisp:ToName(plr)
  if not nameCache[plr] then
    nameCache[plr] = strutf8upper(strutf8sub(plr, 1,1))..strutf8lower(strutf8sub(plr,2))
  end
  return nameCache[plr]
end

-- Functionality for versioning
function Whisp:CheckVersion()
  if not self.db.profile.version == self.currentBuild then
    self:Msg(0, format(L["Whisp v%s by Anea. Type |cffffffff/whisp|r or right-click the icon for options."], "|cffffffff"..self.db.profile.version.."|r"))
  end
  self.db.profile.version = self.currentBuild
end

-- Show a dialog box with given text. Executes the given functions on accept/cancel.
function Whisp:ShowNotification(text, acceptText, acceptFunc, cancelText, cancelFunc)
  StaticPopupDialogs["WHISP_NOTIFICATION"] = {
   text = text,
   button1 = acceptText,
   button2 = (type(cancelFunc)=="function") and cancelText or nil,
   OnAccept = acceptFunc,
   OnCancel = cancelFunc,
   timeout = 0,
   whileDead = 1,
   hideOnEscape = 1
  }
  StaticPopup_Show ("WHISP_NOTIFICATION");
end

-- Transfers the chathistory from old Ace2 database to the new one.
function Whisp:TransferLegacyChatHistory()
  if WhispDB and WhispDB.realms and WhispDB.profiles.Default.version > 0 and WhispDB.profiles.Default.version < 20080625 then
  	local old = WhispDB.realms
  	local new = {}
  	for k,v in pairs(old) do
  		local key = gsub(gsub(k, " %- Alliance", ""), " %- Horde", "")
  		new[key] = v  		
  	end
  	WhispDB = {}
		WhispDB.realm = new
  	Whisp:ShowNotification(format("|cff8888ffWhisp|r|cffff9977 v%s|r|cffffffff\n\nWelcome to the Ace3 version of Whisp.\nYour existing chathistory has just been transferred. Due to major changes in the structure of the addon other custom settings will need to be reapplied. My apologies for the inconvenience.\n\nAnea|r", Whisp.currentBuild), TEXT(OKAY), function() end)
  end
end