---------------------------------------
--- Local Variables -------------------
---------------------------------------

local PI = 3.14159265358979
local utime = 0
local silent = false
local onep = true

local bombx = {} 						--bomb x coordinate
local bomby = {} 						--bomb y coordinate
local bombt = {} 						--bomb timeout

local bicon = {} 						--bomb icons, created dynamically and reused

local vcnt  = 0							--version check
local vtout = 0
local vuser = {}
local verr = false

---------------------------------------
--- Helper Functions ------------------
---------------------------------------

local function Trunc(nr)
	return ceil(nr*100)/100					--only two decimal digits for text output
end

---------------------------------------

local function AddMsg(msg)
	DEFAULT_CHAT_FRAME:AddMessage(msg)
end

---------------------------------------

local function InRaid()
	return GetNumRaidMembers()>0				--are you in a raid?
end

---------------------------------------

local function InParty()
	return not InRaid() and GetNumPartyMembers()>0		--are you in a party (but not in a raid)?
end

---------------------------------------

local function SaveAllNames()
	local num,i,name
	vuser = {}

	num = GetNumRaidMembers()
	if (num>0) then						--read the names of raid members
		for i = 1, num do
			name = UnitName("raid" .. i)
			tinsert(vuser,name)
		end
		return
	end
	
	num = GetNumPartyMembers()
	if (num>0) then						--read the names of party members
		name = UnitName("player")
		tinsert(vuser,name)
		for i = 1, num do
			name = UnitName("party" .. i)
			tinsert(vuser,name)
		end
		return
	end
end

---------------------------------------

local function RemoveName(name)
	local i
	for i = #vuser, 1, -1 do
		if (vuser[i]==name) then
			tremove(vuser,i)
		end
	end
end

---------------------------------------
--- Coordinate Functions --------------
---------------------------------------

local function GetCoord()
	local x,y = Minimap:GetPingPosition()
	local t = {6,4.8,3.6,2.4,1.6,1}				--mapscale table
	local s = t[Minimap:GetZoom()+1]			--get mapzoom
	local f = -50						--multiplier to meter/yards and sign correction
	x = x * s * f
	y = y * s * f
	return x,y						--return true x/y coordinates in meter/yards
end

---------------------------------------

local function GetDir()
	--return select(9,Minimap:GetChildren()):GetFacing()	--get character direction in radian (north is zero)
	UpdateWorldMapArrowFrames()
	return PlayerArrowFrame:GetFacing() --get character direction in radian (north is zero)
end

---------------------------------------

local function GetDist(x,y)
	local p,q = GetCoord()
	p = x - p						--get vector
	q = y - q
	local d = sqrt(p*p+q*q) - 3.2				--calculate vector length, -3.2m deadzone between chars
	if (d<0) then
		d = 0						--deadzone correction, no negative distances
	end
	return d
end

---------------------------------------
--- Bomb Functions --------------------
---------------------------------------

local function AddBomb(x,y,tout)
	tinsert(bombx, x)					--add new bomb to tables
	tinsert(bomby, y)
	tinsert(bombt, GetTime()+tout)
end

---------------------------------------

local function RemoveOldBombs()
	local t = GetTime()
	for i = #bombt, 1, -1 do
		if (bombt[i]<t) then				--check timeout
			tremove(bombx,i)			--remove old bombs
			tremove(bomby,i)			--(bomb icons will be refreshed later)
			tremove(bombt,i)
		end
	end
end

---------------------------------------

local function GetNearestBombDist()
	local dist = 1000
	for i = 1, #bombt do
		local d = GetDist(bombx[i],bomby[i])
		if (d<dist) then
			dist = d				--set new minimum distance
		end
	end
	return dist
end

---------------------------------------

local function GetBombDir(i)					--complicated math stuff, don't worry about it as long as it works
	local x,y = GetCoord()
	x = bombx[i]-x						--get bomb vector
	y = bomby[i]-y
	local v = atan(y/x)/180*PI				--get angle in radian
	if (x<0) then						--arctan between 90 and 270 degrees needs correction
		v = PI + v
	end
	return mod(v+3/2*PI,2*PI)				--zero angle is north, therefore apply correction (add and mod)
end

---------------------------------------
--- Icon Functions --------------------
---------------------------------------

local function CreateBombIcon()
	local f = CreateFrame("Frame",nil,UIParent)		--create frame
	f:SetFrameStrata("BACKGROUND")
	f:SetWidth(32)
	f:SetHeight(32)

	local t = f:CreateTexture(nil,"BACKGROUND")		--create texture
	t:SetTexture("Interface\\Icons\\INV_Misc_Bomb_04.blp")
	t:SetAllPoints(f)
	f.texture = t

	f:SetPoint("CENTER",0,0)
	tinsert(bicon,f)					--insert into icon table
end

---------------------------------------

local function ShowBombIcon(dir,i)
	if (i>#bicon) then					--do we have enough icons?
		CreateBombIcon()				--otherwise dynamically create one
	end

	local deg = (GetDir()-dir)*57.3				--calculate angle delta in degree
	local c = 120						--radius
		
	local b = bicon[i]
	b:ClearAllPoints()
	b:SetPoint("CENTER",c*sin(deg),c*cos(deg)-30)		--set position
	b:Show()
end

---------------------------------------

local function HideBombIconsAbove(nr)
	for i = nr+1, #bicon do					--hide bombs above index nr
		bicon[i]:Hide()
	end
end

---------------------------------------

local function ShowIconsOfBombsInRange(range)			--scans all tables and displays bombs in range
	local j = 0
	for i = 1, #bombt do
		local d = GetDist(bombx[i],bomby[i])
		if (d<range) then				--display bomb if in range
			j = j + 1				--reuse old icons
			ShowBombIcon(GetBombDir(i),j)
		end
	end
	HideBombIconsAbove(j)					--hide all other bomb icons
end

---------------------------------------
--- Communication Functions -----------
---------------------------------------

local function PlaceBomb()
	local x,y = GetCoord()
	if not (InParty() or InRaid()) then
		AddBomb(x,y,10)					--if not in group then place bomb directly
	else
		SendAddonMessage("VoidReaverAlarm",		--if in group, send warning to everyone (including yourself)
			"CB" .. x .. "/" .. y,"RAID")		--bomb will be placed by EventMsg()
	end
end

---------------------------------------

local function PlaceTestBomb()					--same as PlaceBomb() but with 60sec despawn time
	local x,y = GetCoord()
	if not (InParty() or InRaid()) then
		AddBomb(x,y,60)
		AddMsg(VRALARM_TBOMBMSG1)
	else
		SendAddonMessage("VoidReaverAlarm",
			"CT" .. x .. "/" .. y,"RAID")
		if (onep) then
			AddMsg(VRALARM_PINGNOTE)
			onep = false
		end
	end
end

---------------------------------------

local function EventMsg(msg,sender)
	local prefix  = strsub(msg,1,2)				--parse msg
	local message = strsub(msg,3)
	
	if (prefix=="CB") then					--10sec normal mode
		local pos = strfind(message,"/")
		local x = tonumber(strsub(message,1,pos-1))
		local y = tonumber(strsub(message,pos+1))
		AddBomb(x,y,10)
	end

	if (prefix=="CT") then					--60sec test mode
		local pos = strfind(message,"/")
		local x = tonumber(strsub(message,1,pos-1))
		local y = tonumber(strsub(message,pos+1))
		AddBomb(x,y,60)
		AddMsg(sender .. VRALARM_TBOMBMSG2)
	end

	if (prefix=="VC") then					--version check request
		verr = true
		SendAddonMessage("VoidReaverAlarm",
			"VA" .. VRALARM_VER,"WHISPER",sender)
		verr = false
	end

	if (prefix=="VA") then					--version check answer
		vcnt = vcnt + 1
		AddMsg(vcnt .. ": " .. sender .. " v" .. message)
		RemoveName(sender)
	end
end

---------------------------------------
--- Main Functions --------------------
---------------------------------------

function VoidReaverAlarm_Load()					--initialization
	AddMsg(VRALARM_GREETING)

	SLASH_VOIDREAVERALARM1 = "/voidreaveralarm"
	SLASH_VOIDREAVERALARM2 = "/vra"
	SlashCmdList["VOIDREAVERALARM"] = VoidReaverAlarm_SlashHandler

	this:RegisterEvent("CHAT_MSG_ADDON")
	this:RegisterEvent("MINIMAP_PING")
	this:RegisterEvent("CHAT_MSG_MONSTER_YELL")
	this:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
end

---------------------------------------

function VoidReaverAlarm_SlashHandler(msg)			--slash handler
	if (msg=="") then
		AddMsg(VRALARM_HELP1)
		AddMsg(VRALARM_HELP2)
		AddMsg(VRALARM_HELP3)
		AddMsg(VRALARM_HELP4)
		AddMsg(VRALARM_HELP5);
	end

	if (msg=="test bomb") then
		local x,y = GetPlayerMapPosition("player")
		if (IsInInstance()) then
			PlaceTestBomb()
		else
			AddMsg("You must be inside an instance to test this feature.")
		end
	end
	
	if (msg=="version check") then
		vcnt = 0
		vtout = 3
		SaveAllNames()
		AddMsg(VRALARM_VERCHECK)
		SendAddonMessage("VoidReaverAlarm","VC","RAID")
	end

	if (msg=="silent on") then
		silent = true
		AddMsg(VRALARM_SILENTON)
	end

	if (msg=="silent off") then
		silent = false
		AddMsg(VRALARM_SILENTOFF)
	end
	
	if (msg=="blue off") then
		VoidReaverAlarmMainBackground:SetTexture(0.0, 0.0, 1.0, 0.0);
		AddMsg(VRALARM_BLUEOFF);
	end
	
	if (msg=="blue on") then
		VoidReaverAlarmMainBackground:SetTexture(0.0, 0.0, 1.0, 0.2);
		AddMsg(VRALARM_BLUEON);
	end
end

---------------------------------------

function VoidReaverAlarm_Update(sec)
	utime = utime + sec
	if (utime<0.025) then   --refresh throttle
		return
	end
	utime = 0  
			
	RemoveOldBombs()

	local d = GetNearestBombDist()				--display "blue screen" if necessary
	if (d<21 and not silent) then
		VoidReaverAlarmMainText:SetText(Trunc(d))	--distance
		VoidReaverAlarmMain:Show()
		ShowIconsOfBombsInRange(30)			--30m so that we don't run straight into another bomb
	else
		VoidReaverAlarmMain:Hide()
		HideBombIconsAbove(0)
	end

	if (vtout>0) then					--version check running?
		vtout = vtout - sec
		if (vtout<=0) and (#vuser>0) then		--check expired?
			AddMsg(VRALARM_NOVRA)
			local i
			for i = 1,#vuser do
				AddMsg(i .. ": " .. vuser[i])	--list missing users!
			end
		end
	end
	
	if (verr) then						--warn when version check answer fails!
		verr = false
		AddMsg(VRALARM_VERR)
	end
end

---------------------------------------

function VoidReaverAlarm_Event()
	if (event=="CHAT_MSG_ADDON") then
		if (arg1=="VoidReaverAlarm") then
			EventMsg(arg2,arg4)			--message from addon channel
		end
		return
	end
	if (GetRealZoneText() ~= VRALARM_TK) then 
		return;
	end

	if (event=="COMBAT_LOG_EVENT_UNFILTERED" and arg4==VRALARM_BOSSNAME and arg2=="SPELL_CAST_SUCCESS" and arg7==UnitName("player") and arg9==34190) then
		PlaceBomb()										--fixed for 2.4.2? *prays*
	end
	
	if (event=="MINIMAP_PING") then
		AddMsg(VRALARM_PING .. UnitName(arg1) .. ".")	--shows who is pinging
	end
	
	if (event=="CHAT_MSG_MONSTER_YELL") then
		if (IsRaidLeader()) then			--makes the raidleader ping when the fight starts (sync!)
			if (arg1==VRALARM_STARTYELL) then
				Minimap:PingLocation(0,0)
			end
		end
	end
end

---------------------------------------
---------------------------------------
---------------------------------------
