
local VisualHeal = LibStub("AceAddon-3.0"):NewAddon("VisualHeal", "AceConsole-3.0");
VisualHeal.Data = 
{
    Name = "VisualHeal",
    Version = 7, 
};

VisualHeal.EventFrame = CreateFrame("Frame");
VisualHeal.EventFrame:SetScript("OnEvent", function (this, event, ...) VisualHeal[event](VisualHeal, ...) end);
VisualHeal.HealComm = LibStub:GetLibrary("LibHealComm-3.0");
local L = LibStub("AceLocale-3.0"):GetLocale("VisualHeal");


--[ Frequently Accessed Globals ]--

local GetNumRaidMembers = GetNumRaidMembers;
local GetNumPartyMembers = GetNumPartyMembers;
local UnitIsUnit = UnitIsUnit;
local UnitName = UnitName;
local UnitPlayerOrPetInParty = UnitPlayerOrPetInParty;
local UnitPlayerOrPetInRaid = UnitPlayerOrPetInRaid;
local GetTime = GetTime;
local UnitHealth = UnitHealth;
local UnitHealthMax = UnitHealthMax;
local UnitClass = UnitClass;
local UnitCastingInfo = UnitCastingInfo;
local UnitIsPlayer = UnitIsPlayer;
local RAID_CLASS_COLORS = RAID_CLASS_COLORS;
local string = string;
local select = select;


--[ Settings ]--

local function setOptionHealBarScale(info, value)
    local a, b, c, d, e = VisualHealBar:GetPoint();
    local xscale = VisualHealBar:GetScale();
    VisualHealBar:SetScale(value);
    if (not b) then
        VisualHealBar:ClearAllPoints();
        VisualHealBar:SetPoint(a, b, c, d * xscale / value, e * xscale / value);
    end
    VisualHeal.db.profile.HealBarScale = value;
end

local function setOptionPlayerBarScale(info, value)
    local a, b, c, d, e = VisualHealPlayerBar:GetPoint();
    local xscale = VisualHealPlayerBar:GetScale();
    VisualHealPlayerBar:SetScale(value);
    if (not b) then
        VisualHealPlayerBar:ClearAllPoints();
        VisualHealPlayerBar:SetPoint(a, b, c, d * xscale / value, e * xscale / value);
    end
    VisualHeal.db.profile.PlayerBarScale = value;
end

local optionstable = 
{ 
    type = "group",
    icon = "",
    name = "VisualHeal",
    childGroups = "tree",
    args = 
    {
		guildversions = 
        {
			name = L["Dump Guild Versions"],
			type = "execute",
			order = 10,
			desc = L["Print the versions of LibHealComm-3.0 used by guild members"],
			func = function() 
                local versions = LibStub("LibHealComm-3.0"):GetGuildVersions();
                for player, version in pairs(versions) do                    
                    DEFAULT_CHAT_FRAME:AddMessage(player .. ": " .. tostring(version));
                end
            end,
		},
		groupversions = 
        {
			name = L["Dump Group Versions"],
			type = "execute",
			order = 11,
			desc = L["Print the versions of LibHealComm-3.0 used by group members"],
			func = function() 
                local versions = LibStub("LibHealComm-3.0"):GetRaidOrPartyVersions();
                for player, version in pairs(versions) do                    
                    DEFAULT_CHAT_FRAME:AddMessage(player .. ": " .. tostring(version));
                end
            end,
		},
		config = 
        {
			name = L["Config"],
			type = "execute",
			order = 1,
			desc = L["Configuration"],
			func = function() LibStub("AceConfigDialog-3.0"):Open("VisualHeal") end,
			guiHidden = true,
			dialogHidden = true,
			dropdownHidden = true,
		},
        healbar = 
        {
            name = L["Show HealBar"],
            type = "toggle",
            order = 2,
            desc = L["Toggles display of the HealBar when you are healing"],
            get = function() return VisualHeal.db.profile.ShowHealBar end,
            set = function() VisualHeal.db.profile.ShowHealBar = not VisualHeal.db.profile.ShowHealBar end,
        },
        playerbar = 
        {
            name = L["Show PlayerBar"],
            type = "toggle",
            order = 3,
            desc = L["Toggles display of the PlayerBar when heals are incoming to you"],
            get = function() return VisualHeal.db.profile.ShowPlayerBar end,
            set = function() VisualHeal.db.profile.ShowPlayerBar = not VisualHeal.db.profile.ShowPlayerBar end,
        },
        stickyhealbar = 
        {
            name = L["Sticky HealBar"],
            type = "toggle",
            order = 4,
            desc = L["If enabled the HealBar will stay on screen when your heal completes"],
            get = function() return VisualHeal.db.profile.StickyHealBar end,
            set = function() VisualHeal.db.profile.StickyHealBar = not VisualHeal.db.profile.StickyHealBar end,
        },
        healbarscale = 
        {
            name = L["Scale of the HealBar"],
            type = "range",
            order = 5,
            desc = L["Set the scale of the HealBar"],
            min = 0.5,
            max = 2.0,
            isPercent = true,
            get = function() return VisualHeal.db.profile.HealBarScale end,
            set = setOptionHealBarScale,
        },
        playerbarscale = 
        {
            name = L["Scale of the PlayerBar"],
            type = "range",
            order = 6,
            desc = L["Set the scale of the PlayerBar"],
            min = 0.5,
            max = 2.0,
            isPercent = true,
            get = function() return VisualHeal.db.profile.PlayerBarScale end,
            set = setOptionPlayerBarScale,
        },
        resetbars = 
        {
            name = L["Reset Bars"],
            type = "execute",
            order = 7,
            desc = L["Reset the HealBar and PlayerBar to default positions and scales"],
            func = function()
                VisualHealBar:ClearAllPoints();
                VisualHealBar:SetScale(1);
                VisualHeal.db.profile.HealBarScale = 1;
                VisualHealBar:SetPoint("CENTER", CastingBarFrame, "CENTER", 0, VisualHealBar.Offset);
                VisualHealPlayerBar:ClearAllPoints();
                VisualHealPlayerBar:SetScale(1);
                VisualHeal.db.profile.PlayerBarScale = 1;
                VisualHealPlayerBar:SetPoint("CENTER", CastingBarFrame, "CENTER", 0, VisualHealPlayerBar.Offset);
            end
        },
        show = 
        {
            name = L["Show Bars"],
            type = "execute",
            order = 8,
            desc = L["Show the HealBar and PlayerBar to allow moving them around"],
            func = function ()
                VisualHeal:HealBarHide();
                VisualHeal:PlayerBarHide();
                VisualHealPlayerBarTextLeft:SetText("");
                VisualHealPlayerBarTextRight:SetText("");
                VisualHealPlayerBarLeft:SetStatusBarColor(1, 1, 1);
                VisualHealPlayerBarLeft:SetValue(1);
                VisualHealPlayerBarMiddle:SetValue(1);
                VisualHealPlayerBarRight:SetValue(1);
                VisualHealPlayerBarSparkOne:SetPoint("CENTER", "VisualHealPlayerBarLeft", "LEFT", 0, 0)
                VisualHealPlayerBarSparkTwo:SetPoint("CENTER", "VisualHealPlayerBarLeft", "LEFT", 372/2, 0)
                VisualHealPlayerBar:Show();
                if (VisualHeal.IsHealer) then
                    VisualHealBarText:SetText("");
                    VisualHealBarLeft:SetStatusBarColor(1, 1, 1);
                    VisualHealBarLeft:SetValue(1);
                    VisualHealBarMiddle:SetValue(1);
                    VisualHealBarRight:SetValue(1);
                    VisualHealBarSparkOne:SetPoint("CENTER", "VisualHealBarLeft", "LEFT", 0, 0)
                    VisualHealBarSparkTwo:SetPoint("CENTER", "VisualHealBarLeft", "LEFT", 372/2, 0)
                    VisualHealBar:Show();
                end
            end
        },
    },
};


--[ Init/Enable/Disable ]--

function VisualHeal:OnInitialize()
    self.db = LibStub("AceDB-3.0"):New("VisualHealDB", {
        profile = {
            ShowHealBar = true,
            ShowPlayerBar = true,
            PlayerBarScale = 1.0,
            HealBarScale = 1.0,
            StickyHealBar = false,
        }
    });

    VisualHealPlayerBar:SetScript("OnUpdate", self.PlayerBarOnUpdate);

    LibStub("AceConfig-3.0"):RegisterOptionsTable("VisualHeal", optionstable, {"visualheal", "vh"})

    local playerClass = string.lower(select(2, UnitClass('player')));

    if (playerClass == "druid" or playerClass == "paladin" or playerClass == "priest" or playerClass == "shaman") then
        self.IsHealer = true;
    end

    self.PlayerName = self:UnitFullName('player');

    VisualHealBar:SetScale(self.db.profile.HealBarScale);
    VisualHealPlayerBar:SetScale(self.db.profile.PlayerBarScale);
end

function VisualHeal:OnEnable()
    self.EventFrame:RegisterEvent("UNIT_HEALTH");
    self.EventFrame:RegisterEvent("PLAYER_TARGET_CHANGED");
    self.EventFrame:RegisterEvent("PLAYER_FOCUS_CHANGED");
    self.HealComm.RegisterCallback(self, "HealComm_DirectHealStart");
    self.HealComm.RegisterCallback(self, "HealComm_DirectHealDelayed");
    self.HealComm.RegisterCallback(self, "HealComm_DirectHealStop");
    self.HealComm.RegisterCallback(self, "HealComm_HealModifierUpdate");
end

function VisualHeal:OnDisable()
    self.EventFrame:UnregisterEvent("UNIT_HEALTH");
    self.EventFrame:UnregisterEvent("PLAYER_TARGET_CHANGED");
    self.EventFrame:UnregisterEvent("PLAYER_FOCUS_CHANGED");
    self.HealComm.UnregisterCallback(self, "HealComm_DirectHealStart");
    self.HealComm.UnregisterCallback(self, "HealComm_DirectHealDelayed");
    self.HealComm.UnregisterCallback(self, "HealComm_DirectHealStop");
    self.HealComm.UnregisterCallback(self, "HealComm_HealModifierUpdate");

    self:HealBarHide();
    self:PlayerBarHide();
end


--[ Utilities ]--

function VisualHeal:UnitFullName(unit)
    local name, realm = UnitName(unit);
    if (realm and realm ~= "") then
        return name .. "-" .. realm;
    else
        return name;
    end
end

-- Returns true if the player is in a raid group
function VisualHeal:InRaid()
    return (GetNumRaidMembers() > 0);
end

-- Returns true if the player is in a party or a raid
function VisualHeal:InParty()
    return (GetNumPartyMembers() > 0);
end

-- Determine if player is in a battleground
function VisualHeal:InBattleground()
    return (select(2, IsInInstance()) == "pvp") and true or false
end

local maxHealthTab = 
{
    warrior=4100,
    paladin=4000,
    shaman=3500,
    rogue=3100,
    hunter=3100,
    druid=3100,
    warlock=2300,
    mage=2200,
    priest=2100
};
function VisualHeal:EstimateUnitHealth(unit)
    local _, class = UnitClass(unit);
    class = class or "Unknown";
    return ((UnitHealth(unit) or 0) * (maxHealthTab[string.lower(class)] or 4000) * (UnitLevel(unit) or 60) / 6000);
end

function VisualHeal:GetUnitID(targetName)
    return UnitExists(targetName) and targetName or (targetName == self:UnitFullName('target') and 'target') or (targetName == self:UnitFullName('focus') and 'focus') or (targetName == self:UnitFullName('targettarget') and 'targettarget') or (targetName == self:UnitFullName('focustarget') and 'focustarget') or nil;
end


--[ PlayerBar Functions ]--

local PlayerBarLastUpdate = GetTime();
function VisualHeal:PlayerBarOnUpdate()
    local time = GetTime();
    if (time > PlayerBarLastUpdate + 0.05) then
        PlayerBarLastUpdate = time;
        VisualHeal:PlayerBarUpdate();
    end		
end

function VisualHeal:PlayerBarShow()
    VisualHealPlayerBar.IsLive = true;
    VisualHealPlayerBar:Show();
    self:PlayerBarUpdate();
end

function VisualHeal:PlayerBarHide()
    VisualHealPlayerBar.IsLive = false;
    VisualHealPlayerBar:Hide();
end

function VisualHeal:PlayerBarUpdate()

    if (not VisualHealPlayerBar.IsLive) then
        return;
    end

    local now = GetTime();
    local hp, hppre, hppost;
    local _, incomingHeal, nextTime, nextSize, nextName = self.HealComm:UnitIncomingHealGet('player', now);

    if (not incomingHeal) then
        self:PlayerBarHide();
        return;
    end       
    
    -- Calculate time left to next heal
    local nextTimeLeft = nextTime - now;

    -- Determine health percentages
    hp = UnitHealth('player') / UnitHealthMax('player');
    hppre = hp + (nextSize * self.PlayerModifier) / UnitHealthMax('player');
    hppost = hp + (incomingHeal * self.PlayerModifier) / UnitHealthMax('player');

    -- Sanity check on values
    if (hp > 1.0) then 
        hp = 1.0; 
    end
    if (hppre > 2.0) then 
        hppre = 2.0;
    end
    if (hppost > 3.0) then 
        hppost = 3.0;
    end
    if (hp > hppre) then 
        hppre = hp;
    end
    if (hppre > hppost) then 
        hppost = hppre;
    end   

    -- Update bars
    VisualHealPlayerBarLeft:SetValue(hp);
    VisualHealPlayerBarMiddle:SetValue(hppre);
    VisualHealPlayerBarRight:SetValue(hppost);
    VisualHealPlayerBarSparkOne:SetPoint("CENTER", "VisualHealPlayerBarLeft", "LEFT", 372/2 * hp, 0);
    VisualHealPlayerBarSparkTwo:SetPoint("CENTER", "VisualHealPlayerBarLeft", "LEFT", 372/2 * ((incomingHeal == nextSize) and hp or hppre), 0);

    -- Set colour for health
    local r = hp < 0.5 and 1.0 or 2.0 * (1.0 - hp);
    local g = hp > 0.5 and 0.8 or 1.6 * hp;
    local b = 0.0;
    VisualHealPlayerBarLeft:SetStatusBarColor(r, g, b);

    -- Set text
    VisualHealPlayerBarTextLeft:SetText(nextName);
    VisualHealPlayerBarTextRight:SetFormattedText("%.1fs", nextTimeLeft);

    -- Set colour for remaining incoming heal
    VisualHealPlayerBarRight:SetStatusBarColor(0.4, 0.6, 0.4, 0.5);

    -- Set colour for next heal
    VisualHealPlayerBarMiddle:SetStatusBarColor(0.0, 0.8, 0.0, (1.5 - nextTimeLeft) / 1.5);
end

--[ HealBar Functions ]--

function VisualHeal:HealBarShow()
    VisualHealBar.IsLive = true;
    self:HealBarUpdate();
    VisualHealBar:Show();
end

function VisualHeal:HealBarHide()
    VisualHealBar.IsLive = false;
    VisualHealBar:Hide();
end

function VisualHeal:HealBarUpdate()

    if (not VisualHealBar.IsLive) then
        return;
    end

    -- Check validity of UnitID
    if (self.HealingTargetUnitID) then
        if (self:UnitFullName(self.HealingTargetUnitID) ~= self.HealingTargetName) then
            self.HealingTargetUnitID = nil;
        end
    end

    -- If no UnitID, try again to determine the UnitID
    if (not self.HealingTargetUnitID) then            
        self.HealingTargetUnitID = self:GetUnitID(self.HealingTargetName);
    end

    if (not self.HealingTargetUnitID) then
        -- Lost the unit
        VisualHealBarText:SetTextColor(255, 0, 0);
        VisualHealBarLeft:SetValue(0);
        VisualHealBarMiddle:SetValue(0);
        VisualHealBarRight:SetValue(0);
        VisualHealBarSparkOne:Hide();
        VisualHealBarSparkTwo:Hide();
        return;
    else
        VisualHealBar:Show();
    end

    VisualHealBarText:SetText(self.HealingTargetName);
    if (UnitIsPlayer(self.HealingTargetUnitID)) then
        local tab = RAID_CLASS_COLORS[select(2, UnitClass(self.HealingTargetUnitID)) or ""];
        VisualHealBarText:SetTextColor(tab.r, tab.g, tab.b);
    else
        VisualHealBarText:SetTextColor(230, 230, 0);
    end

    local hp, hppre, hppost, waste, incomingHeal;

    if (self.IsCasting) then
        incomingHeal = self.HealComm:UnitIncomingHealGet(self.HealingTargetUnitID, self.EndTime);
    else
        incomingHeal = self.HealComm:UnitIncomingHealGet(self.HealingTargetUnitID, GetTime() + 1000);
    end   

    -- Determine health percentages of healing target
    local unitHealthMax = UnitHealthMax(self.HealingTargetUnitID);
    if (unitHealthMax ~= 100) then
        -- Full info available
        local unitHealth = UnitHealth(self.HealingTargetUnitID);
        hp = unitHealth / unitHealthMax;
        if (incomingHeal) then
            hppre = (unitHealth + incomingHeal * self.HealingTargetModifier) / unitHealthMax;
        else
            hppre = hp;
        end
        if (self.IsCasting) then
            hppost = (unitHealth + ((incomingHeal or 0) + self.HealingSize) * self.HealingTargetModifier) / unitHealthMax; 
        else
            hppost = hppre;
        end
    else
        -- Estimate
        hp = UnitHealth(self.HealingTargetUnitID) / 100;        
        local factor = hp / self:EstimateUnitHealth(self.HealingTargetUnitID);
        if (incomingHeal) then
            hppre = hp + incomingHeal * self.HealingTargetModifier * factor;
        else
            hppre = hp;
        end
        if (self.IsCasting) then
            hppost = hp + ((incomingHeal or 0) + self.HealingSize) * self.HealingTargetModifier * factor; 
        else
            hppost = hppre;
        end
    end

    if (hp > 1.0) then 
        hp = 1.0; 
    end
    if (hppre > 2.0) then 
        hppre = 2.0;
    end
    if (hppost > 3.0) then 
        hppost = 3.0;
    end
    if (hp > hppre) then 
        hppre = hp;
    end
    if (hppre > hppost) then 
        hppost = hppre;
    end 

    -- Determine waste (in percent)
    if (hppost > 1.0 and hppost > hppre) then
        waste = (hppost - 1.0) / (hppost - hppre);
    else
        waste = 0.0;
    end
    if (waste > 1.0) then 
        waste = 1.0;
    end

    -- Calculate colour for overheal severity
    local red = waste > 0.1 and 1 or waste * 10;
    local green = waste < 0.1 and 1 or -2.5 * waste + 1.25;
    if (waste < 0.0) then 
        green = 1.0;
        red = 0.0;
    end

    -- Update bars
    VisualHealBarLeft:SetValue(hp);
    VisualHealBarMiddle:SetValue(hppre);
    VisualHealBarRight:SetValue(hppost);
    VisualHealBarSparkOne:SetPoint("CENTER", "VisualHealBarLeft", "LEFT", 372/2 * hp, 0)
    VisualHealBarSparkTwo:SetPoint("CENTER", "VisualHealBarLeft", "LEFT", 372/2 * hppre, 0)

    if (incomingHeal or self.IsCasting) then
        VisualHealBarSparkOne:Show();
        if (incomingHeal and self.IsCasting) then
            VisualHealBarSparkTwo:Show();
        else
            VisualHealBarSparkTwo:Hide();
        end
    else
        VisualHealBarSparkOne:Hide();        
    end

    -- Set colour for health
    VisualHealBarLeft:SetStatusBarColor(hp < 0.5 and 1.0 or 2.0 * (1.0 - hp), hp > 0.5 and 0.8 or 1.6 * hp, 0.0);

    -- Set colour for heal
    VisualHealBarRight:SetStatusBarColor(red, green, 0.0);
end


--[ Event Handlers ]--

function VisualHeal:UNIT_HEALTH(unit)
    if (self.HealingTargetUnitID and UnitIsUnit(self.HealingTargetUnitID, unit)) then 
        -- Update the heal bar when unit health changes on the healing target
        self:HealBarUpdate();
    end
    if (unit == 'player') then
        -- Update the player bar when the health of the player changes
        self:PlayerBarUpdate();
    end
end

function VisualHeal:PLAYER_FOCUS_CHANGED()
    self:HealBarUpdate();
end

function VisualHeal:PLAYER_TARGET_CHANGED()
    self:HealBarUpdate();
end

function VisualHeal:HealComm_DirectHealStart(event, healerName, healSize, endTime, ...)
    if (healerName == self.PlayerName) then
        self.IsCasting = true;
        self.HealingTargetName = ...;
        self.HealingTargetUnitID = self:GetUnitID(...);
        self.HealingSize = healSize;
        self.EndTime = endTime;
        self.HealingTargetModifier = self.HealComm:UnitHealModifierGet(...); 
        if (self.db.profile.ShowHealBar) then
            self:HealBarShow();
        end
    else        
        self:HealBarUpdate(); 
        for i = 1, select('#', ...) do
            local targetName = select(i, ...);
            if (self.db.profile.ShowPlayerBar and (targetName == self.PlayerName)) then
                self.PlayerModifier = self.HealComm:UnitHealModifierGet(targetName);
                self:PlayerBarShow();
            end
        end
    end
end

function VisualHeal:HealComm_DirectHealDelayed(event, healerName, healSize, endTime, ...)
    if (healerName == self.PlayerName) then
        self.EndTime = endTime;
        self:HealBarUpdate(); 
    else
        self:HealBarUpdate();
        self:PlayerBarUpdate();
    end
end

function VisualHeal:HealComm_DirectHealStop(event, healerName, healSize, succeeded, ...)
    if (healerName == self.PlayerName) then
        self.IsCasting = false;
        if (not self.db.profile.StickyHealBar) then
            self:HealBarHide();
        else
            self:HealBarUpdate();
        end
    else
        self:HealBarUpdate();
        self:PlayerBarUpdate();        
    end
end

function VisualHeal:HealComm_HealModifierUpdate(event, unit, targetName, healModifier)
    if (self.HealingTargetUnitID and UnitIsUnit(self.HealingTargetUnitID, unit)) then
        self.HealingTargetModifier = healModifier;
        self:HealBarUpdate();
    end
    if (UnitIsUnit(unit, 'player')) then
        self.PlayerModifier = healModifier;
        self:HealBarUpdate();
        self:PlayerBarUpdate();
    end
end
