--[[
	Hooks tooltips to be able to add custom information. Despite the corresponding
	library TipHooker, we are able to tell the count of items.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.TooltipHook = vendor.Vendor:NewModule("TooltipHook", "AceHook-2.1");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local self = vendor.TooltipHook;

local TOOLTIP_NAMES = {"GameTooltip", "ItemRefTooltip", "ShoppingTooltip", "AtlasLootTooltip"};
local METHOD_NAMES = {"SetHyperlink", "SetBagItem", "SetInventoryItem", "SetAuctionItem",
	"SetAuctionSellItem", "SetLootItem", "SetLootRollItem", "SetCraftSpell", "SetCraftItem",
	"SetTradeSkillItem", "SetInboxItem", "SetSendMailItem", "SetQuestItem",
	"SetTradePlayerItem", "SetTradeTargetItem", "SetMerchantItem",
	"SetBuybackItem", "SetSocketGem", "SetExistingSocketGem", "SetHyperlinkCompareItem",
	"SetGuildBankItem"};

	-- "SetQuestLogItem"
--[[
	Callback function for sorting appernders.
--]]
local function _SortAppenderCallback(appender1, appender2) 
	return appender1.prio < appender2.prio;
end
	
--[[
	Handles the tooltip call and notifies the appenders.
--]]
local function _HandleCall(self, tooltip, count)
	if (not self.db.profile.isActive) then 
		return;
	end
	if (not tooltip.GetItem) then 
		return;
	end
	local name, link = tooltip:GetItem();
	if (not name or string.len(name) == 0 or not link) then 
		return; 
	end
	if (not count) then
		count = 1;
	end
	-- print the into
	tooltip:AddLine(" ");
	local msg;
	if (vendor.AuctionHouse:IsNeutral()) then
		msg = "Neutral";
	else
		local _, faction = UnitFactionGroup("player");
		msg = GetRealmName().."-"..faction;
	end
	tooltip:AddDoubleLine(L["Vendor statistics"], "("..msg..")", 1, 1, 1, 1, 1, 1);
	for key, val in ipairs(self.appenders) do
		val.appender:AppendToGameTooltip(tooltip, link, name, count);
	end
	tooltip:Show();
end

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.TooltipHook = {
		type = "group",
		name = L["Tooltip"],
		desc = L["Tooltip"],
		args = {
			isActive = {
				type = "toggle",
				name = L["Activated"],
				desc = L["Selects whether any informations from vendor should be shown in the GameTooltip."],
				get = _GetValue,
				set = _SetValue,
				passValue = "isActive",
				order = 1,
			},
		}
	}
end

--[[
	Initializes the module.
--]]
function vendor.TooltipHook:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("TooltipHook");
	vendor.Vendor:RegisterDefaults("TooltipHook", "profile", {
		isActive = true,
	});
	_CreateOptions();	
end

--[[
	Enables the module and hooks all available tooltip functions.
--]]
function vendor.TooltipHook:OnEnable()
	self.appenders = {};
	-- hook all known tooltip implementations
	local frame = EnumerateFrames();
	while (frame) do
		if (frame:GetObjectType() == "GameTooltip") then
			local name = frame:GetName();
			if (name) then
				for _, v in ipairs(TOOLTIP_NAMES) do
					if strfind(name, v) then
						for _, methodName in ipairs(METHOD_NAMES) do
							if (type(frame[methodName]) == "function") then
								self:SecureHook(frame, methodName);
							end
						end
						break
					end
				end
			end
		end
		frame = EnumerateFrames(frame);
	end		
end
	
--[[
	Registers a new listener for GameTooltip integration.
	The given prio sorts the different modules, when integrating it
	into the tooltip. Low values will be printed first.
	The appenders have to implement the following method:
	AppendToGameTooltip(tooltip, itemLink, itemName, count)
--]]
function vendor.TooltipHook:AddAppender(appender, prio)
	local info = {appender = appender, prio = prio};
	table.insert(self.appenders, info);
	table.sort(self.appenders, _SortAppenderCallback);
end
	
--[[
	List of all available hooks.
--]]
function vendor.TooltipHook:SetInventoryItem(tooltip, unit, slot)
	local count = GetInventoryItemCount(unit, slot);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetAuctionItem(tooltip, type, index)
	local _, _, count = GetAuctionItemInfo(type, index);
	_HandleCall(self, tooltip, count);
end
	
function vendor.TooltipHook:SetAuctionSellItem(tooltip)
	local _, _, count = GetAuctionSellItemInfo();
	_HandleCall(self, tooltip, count);
end
	
function vendor.TooltipHook:SetLootItem(tooltip, slot)
	local _, _, count = GetLootSlotInfo(slot);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetLootRollItem(tooltip, slot)
	local _, _, count = GetLootRollItemInfo(slot);
	_HandleCall(self, tooltip, count);
end
	
function vendor.TooltipHook:SetBagItem(tooltip, bag, slot)
	local  _, count = GetContainerItemInfo(bag, slot);
	_HandleCall(self, tooltip, count);
end
	
function vendor.TooltipHook:SetCraftItem(tooltip, craftItemIndex, reagentIndex)
	local _, _, count = GetCraftReagentInfo(craftItemIndex, reagentIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetCraftSpell(tooltip, craftItemIndex)
	_HandleCall(self, tooltip, 1);
end
	
function vendor.TooltipHook:SetTradeSkillItem(tooltip, tradeSkillIndex, reagentIndex)
	if (reagentIndex) then
		local _, _, reagentCount = GetTradeSkillReagentInfo(tradeSkillIndex, reagentIndex);
		_HandleCall(self, tooltip, reagentCount);
	else
		_HandleCall(self, tooltip, 1);
	end
end
	
function vendor.TooltipHook:SetHyperlink(tooltip, link)
	if (tooltip.itemCount) then
		_HandleCall(self, tooltip, tooltip.itemCount);
	else
		_HandleCall(self, tooltip, 1);
	end
end

function vendor.TooltipHook:SetInboxItem(tooltip, index, itemIndex)
	local _, _, count = GetInboxItem(index, itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetSendMailItem(tooltip, itemIndex)
	local _, _, count = GetSendMailItem(itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetQuestItem(tooltip, type, itemIndex)
	local _, _, count = GetQuestItemInfo(type, itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetTradePlayerItem(tooltip, itemIndex)
	local _, _, count = GetTradePlayerItemInfo(itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetTradeTargetItem(tooltip, itemIndex)
	local _, _, count = GetTradeTargetItemInfo(itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetMerchantItem(tooltip, itemIndex)
	local _, _, _, count = GetMerchantItemInfo(itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetBuybackItem(tooltip, itemIndex)
	local _, _, _, count = GetBuybackItemInfo(itemIndex);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetSocketGem(tooltip, unit, slot)
	local count = GetInventoryItemCount(unit, slot);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetExistingSocketGem(tooltip, unit, slot)
	local count = GetInventoryItemCount(unit, slot);
	_HandleCall(self, tooltip, count);
end

function vendor.TooltipHook:SetHyperlinkCompareItem(tooltip)
	_HandleCall(self, tooltip, 1);
end

function vendor.TooltipHook:SetGuildBankItem(tooltip, tab, slot)
	local _, _, count = GetGuildBankItemInfo(tab, slot);
	_HandleCall(self, tooltip, count);
end
