--[[
	Gathers statistics for the auctions found.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.Statistic = vendor.Vendor:NewModule("Statistic");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local self = vendor.Statistic;

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Update the statistic for given item.
--]]
local function _UpdateStatistic(self, itemLink, minBids, buyouts, isNeutral)
	-- calculate the minbid average
	table.sort(minBids);
	table.sort(buyouts);
	local avgMinBid = vendor.Math.GetMedian(minBids);
	local avgBuyout = vendor.Math.GetMedian(buyouts);
	local itemInfo = vendor.Items:GetItemInfo(itemLink, isNeutral);
	if (not itemInfo) then
		itemInfo = vendor.Items:CreateItemInfo();
	end
	if (avgMinBid > 0) then
		-- update moving average of the medians
		if (itemInfo.avgMinBid > 0) then
			itemInfo.avgMinBid = vendor.Math.UpdateAverage(itemInfo.avgMinBid, self.db.profile.movingAverage, avgMinBid, 1);
		else
			itemInfo.avgMinBid = avgMinBid;
		end
	end
	if (avgBuyout > 0) then
		-- update moving average of the medians
		if (itemInfo.avgBuyout > 0) then
			itemInfo.avgBuyout = vendor.Math.UpdateAverage(itemInfo.avgBuyout, self.db.profile.movingAverage, avgBuyout, 1);
		else
			itemInfo.avgBuyout = avgBuyout;
		end
	end
	-- write back item
	vendor.Items:PutItemInfo(itemLink, isNeutral, itemInfo);
end

--[[
	Returns avgMinBid and avgBuyout for the given scan result.
--]]
local function _GetAvgs(result)
	local minBids = {};
	local buyouts = {};
	for i = 1, result:Size() do
		local itemLinkKey, time, timeLeft, count, minBid, minIncrement, buyout = result:Get(i);
		if (itemLinkKey) then
			if (minBid and minBid > 0) then
				table.insert(minBids, math.floor(minBid / count + 0.5));
			end
			if (buyout and buyout > 0) then
				table.insert(buyouts, math.floor(buyout / count + 0.5));
			end
		end
	end
	table.sort(minBids);
	table.sort(buyouts);
	local avgMinBid = vendor.Math.GetMedian(minBids);
	local avgBuyout = vendor.Math.GetMedian(buyouts);
	return avgMinBid, avgBuyout;
end
	
--[[
	Adds auction information to the GameTooltip of the currently available snapshot.
--]]
local function _AddCurrentAuctionInfo(self, tooltip, itemInfo, itemLink, count, isNeutral)
	if (self.db.profile.showCurrent) then
		local result = vendor.Scanner:GetScanSet(isNeutral, itemLink);
		if (result and result:Size() > 0) then
			tooltip:AddLine(L["Current auctions (%d)"]:format(result:Size()));
			local avgMinBid, avgBuyout = _GetAvgs(result);
			local msg1, msg2;
			-- avg minbid
			if (avgMinBid > 0) then
				if (count > 1) then
					msg1 = L["Bid (%d)"]:format(count);
					msg2 = vendor.Format.FormatMoney(avgMinBid, true).." ("..vendor.Format.FormatMoney(avgMinBid * count, true)..")";
				else
					msg1 = L["Bid"];
					msg2 = vendor.Format.FormatMoney(avgMinBid, true);
				end
				tooltip:AddDoubleLine(msg1, msg2);
			end
			if (avgBuyout > 0) then
				-- avg buyout
				if (count > 1) then
					msg1 = L["Buyout (%d)"]:format(count);
					msg2 = vendor.Format.FormatMoney(avgBuyout, true).." ("..vendor.Format.FormatMoney(avgBuyout * count, true)..")";
				else
					msg1 = L["Buyout"];
					msg2 = vendor.Format.FormatMoney(avgBuyout, true);
				end
				tooltip:AddDoubleLine(msg1, msg2);
			end
		end
	end
end

--[[
	Adds auction information to the GameTooltip.
--]]
local function _AddAuctionInfo(self, tooltip, itemInfo, itemLink, count, isNeutral)
	if (self.db.profile.showHistory) then
		local msg1, msg2;
		-- avg minbid
		if (itemInfo.avgMinBid > 0) then
			if (count > 1) then
				msg1 = L["Bid (%d)"]:format(count);
				msg2 = vendor.Format.FormatMoney(itemInfo.avgMinBid).." ("..vendor.Format.FormatMoney(itemInfo.avgMinBid * count)..")";
			else
				msg1 = L["Bid"];
				msg2 = vendor.Format.FormatMoney(itemInfo.avgMinBid);
			end
			tooltip:AddDoubleLine(msg1, msg2);
		end
		if (itemInfo.avgBuyout > 0) then
			-- avg buyout
			if (count > 1) then
				msg1 = L["Buyout (%d)"]:format(count);
				msg2 = vendor.Format.FormatMoney(itemInfo.avgBuyout).." ("..vendor.Format.FormatMoney(itemInfo.avgBuyout * count)..")";
			else
				msg1 = L["Buyout"];
				msg2 = vendor.Format.FormatMoney(itemInfo.avgBuyout);
			end
			tooltip:AddDoubleLine(msg1, msg2);
		end
	end
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.Statistic = {
		type = "group",
		name = L["Statistics"],
		desc = L["Statistics"],
		args = {
			showHistory = {
				type = "toggle",
				name = L["Show historically averages"],
				desc = L["Selects whether historically average values from auction scans should be shown in the GameTooltip."],
				get = _GetValue,
				set = _SetValue,
				passValue = "showHistory",
				order = 1,
			},
			showCurrent = {
				type = "toggle",
				name = L["Show current averages"],
				desc = L["Selects whether current average values from auction scans should be shown in the GameTooltip."],
				get = _GetValue,
				set = _SetValue,
				passValue = "showCurrent",
				order = 2,
			},
			movAvg = {
				type = "range",
				name = L["Moving average"],
				desc = L["Selects the number of (approximated) values, that should be taken for the moving average of the historically auction scan statistics."],
				min = 2,
				max = 256,
				step = 1,
				get = _GetValue,
				set = _SetValue,
				passValue = "movingAverage",
				order = 3,
			},
		}
	}
end

--[[
	Initializes the module.
--]]
function vendor.Statistic:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("Statistic");
	vendor.Vendor:RegisterDefaults("Statistic", "profile", {
		showHistory = true,
		showCurrent = true,
		movingAverage = 12,
	});
	_CreateOptions();	
end

--[[
	Initializes the module.
--]]
function vendor.Statistic:OnEnable()
	vendor.TooltipHook:AddAppender(self, 20);
	vendor.Scanner:AddScanSnapshotListener(self, 1);
end

--[[
	ScanResultListener interface method for new or updated
	Scansnapshots.
	The complete new scan will be considered, not only the difference, because we are
	working with the median.
--]]
function vendor.Statistic:ScanSnapshotUpdated(snapshot, isNeutral)
	vendor.Vendor:Debug("scan update enter");
	local currentItem = nil; -- current entry for gathering stats
	local minBids = {}; -- minbids collected for current item
	local buyouts = {}; -- buyouts collected for current item
   	for i = 1, snapshot:Size() do
      	local itemLinkKey, time, timeLeft, count, minBid, minIncrement, buyoutPrice = snapshot:Get(i);
      	if (itemLinkKey) then
      		if (currentItem and currentItem ~= itemLinkKey) then
      			-- a new item, handle the old one
				_UpdateStatistic(self, currentItem, minBids, buyouts, isNeutral);
				minBids = {}; -- need table:clear(), or sort(table, n)
				buyouts = {};
			end
			currentItem = itemLinkKey;
			if (minBid > 0) then
				table.insert(minBids, math.floor(minBid / count + 0.5));
      		end
			if (buyoutPrice > 0) then
				table.insert(buyouts, math.floor(buyoutPrice / count + 0.5));
   			end
      	end
   	end
	if (currentItem) then
		_UpdateStatistic(self, currentItem, minBids, buyouts, isNeutral);
	end
	vendor.Vendor:Debug("Statistic:ScanSnapshotUpdated leave");
end

--[[
	Callback for GameTooltip integration.
--]]
function vendor.Statistic:AppendToGameTooltip(tooltip, itemLink, itemName, count)
	local itemLinkKey = vendor.Items:GetItemLinkKey(itemLink);
	local isNeutral = vendor.AuctionHouse:IsNeutral();
	local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, isNeutral);
	if (itemInfo) then
		_AddAuctionInfo(self, tooltip, itemInfo, itemLink, count, isNeutral);
   	end
	_AddCurrentAuctionInfo(self, tooltip, itemInfo, itemLink, count, isNeutral);
end

--[[
	Returns minBid and buyout for the current auctions for the specified item or nil.
--]]
function vendor.Statistic:GetCurrentAuctionInfo(itemLink, isNeutral)
	local result = vendor.Scanner:GetScanSet(isNeutral, itemLink);
	if (result and result:Size() > 0) then
		return _GetAvgs(result);
	end
	return nil;
end
