--[[
	Listens to auction house scans and snipes for good oppurtunities.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.Sniper = vendor.Vendor:NewModule("Sniper", "AceEvent-2.0", "AceHook-2.1");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local self = vendor.Sniper;
local SNIPER_VERSION = 1;

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Resets the database for the current realm/server.
--]]
local function _ResetDb()
	vendor.Vendor:ResetDB("Sniper", "realm");
	vendor.Vendor:Print(L["Database of snipes for current realm where reset."]);
end

--[[
	Checks whether to snipe for a bookmarked item. Returns true and the reason, if the snipe should happen.
--]]
local function _SnipeBookmarked(itemLink, itemName, count, bid, buyout, highBidder)
	local data = self.db.realm.wanted[itemName];
	if (self.db.profile.snipeBookmarked and data) then
		local maxBidStr, maxBuyoutStr = strsplit(":", data);
		local maxBid = tonumber(maxBidStr);
		local maxBuyout = tonumber(maxBuyoutStr);
		maxBid = maxBid * count;
		maxBuyout = maxBuyout * count;
		if (buyout > 0 and maxBuyout > 0 and maxBuyout >= buyout) then
			local reason = L["Buyout is less or equal %s."]:format(vendor.Format.FormatMoney(maxBuyout));
			return true, reason;
		elseif (maxBid > 0 and maxBid >= bid and not highBidder) then
			local reason = L["Bid is less or equal %s."]:format(vendor.Format.FormatMoney(maxBid));
			return true, reason;
		end		
	end
	return false;
end

--[[
	Checks whether to snipe because of the sell price. Returns true and the reason, if the snipe should happen.
--]]
local function _SnipeSellPrices(itemLink, itemName, count, bid, buyout, highBidder)
	if (self.db.profile.snipeSellPrices) then
		local sellPrice = vendor.SellPrizes:GetSellPrice(itemLink);
		if (sellPrice and sellPrice > 0) then
			sellPrice = sellPrice * count;
			local money = GetMoney();
			local minProf = self.db.profile.minimumProfit * 100;
			if (buyout and buyout > 0 and buyout <= money and sellPrice > (buyout + minProf)) then
				local profit = vendor.Format.FormatMoney(sellPrice - buyout, true);
				local percent = math.floor(100.0 * (sellPrice - buyout) / buyout + 0.5);
				local reason = L["Buyout is %s less than sell price (%d%%)."]:format(profit, percent);
				vendor.Vendor:Debug("buyout: "..buyout.." sellPrice: "..sellPrice);
				return true, reason;
			elseif (bid and bid <= money and sellPrice > (bid + minProf) and not highBidder) then
				vendor.Vendor:Debug("bid: "..bid.." sellPrice: "..sellPrice);
				local profit = vendor.Format.FormatMoney(sellPrice - bid, true);
				local percent = math.floor(100.0 * (sellPrice - bid) / bid + 0.5);
				local reason = L["Bid is %s less than sell price (%d%%)."]:format(profit, percent);
				return true, reason;
			end		
		end
	end
	return false;
end

--[[
	Checks whether to snipe because of the average price. Returns true and the reason, if the snipe should happen.
--]]
local function _SnipeAverage(itemLink, itemName, count, bid, buyout, highBidder)
	if (self.db.profile.snipeAverage > 0) then
		local itemLinkKey = vendor.Items:GetItemLinkKey(itemLink);
		local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, isNeutral);
		if (itemInfo) then
			local minProf = self.db.profile.minimumProfit * 100;
			if (buyout and buyout > 0 and itemInfo.avgBuyout and itemInfo.avgBuyout > 0) then
				local avg = itemInfo.avgBuyout * count;
				local profit = avg - buyout;
				if (profit >= minProf) then
					local percent = math.floor(100.0 * profit / buyout + 0.5);
					if (percent >= self.db.profile.snipeAverage) then
						vendor.Vendor:Debug("buyout: "..buyout.." avg: "..avg.." profit: "..profit);
						local reason = L["Buyout for %s possible profit (%d%%)."]:format(vendor.Format.FormatMoney(profit, true), percent);
						return true, reason;
					end
				end
			end		
			if (bid and bid > 0 and itemInfo.avgMinBid and itemInfo.avgMinBid > 0) then
				local avg = itemInfo.avgMinBid * count;
				local profit = avg - bid;
				if (profit >= minProf) then
					local percent = math.floor(100.0 * profit / bid + 0.5);
					if (percent >= self.db.profile.snipeAverage) then
						vendor.Vendor:Debug("bid: "..bid.." avg: "..avg.." profit: "..profit);
						local reason = L["Bid for %s possible profit (%d%%)."]:format(vendor.Format.FormatMoney(profit, true), percent);
						return true, reason;
					end
				end
			end
		end
	end
	return false;
end

--[[
	Initializes the safety dialog for the database reset.
--]]
local function _CreateResetDialog()
	StaticPopupDialogs["SNIPER_DB_RESET"] = {
		text = L["Do you really want to reset the database?"];
	  	button1 = L["Yes"],
	  	button2 = L["No"],
	  	OnAccept = function()
	  		_ResetDb()
	  	end,
	  	timeout = 0,
	  	whileDead = 1,
	  	hideOnEscape = 1
	};	
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.Sniper = {
		type = "group",
		name = L["Sniper"],
		desc = L["Sniper"],
		args = {
			snipeBookmarked = {
				type = "toggle",
					name = L["Snipe bookmarked"],
					desc = L["Snipe for bookmarked items."],
					get = _GetValue,
					set = _SetValue,
					passValue = "snipeBookmarked",
					order = 1,
			},
			snipeSellPrices = {
				type = "toggle",
					name = L["Snipe sell prices"],
					desc = L["Snipe for items with higher sell prices."],
					get = _GetValue,
					set = _SetValue,
					passValue = "snipeSellPrices",
					order = 2,
			},
			snipeAverage = {
				type = "range",
					name = L["Snipe average"],
					desc = L["Snipe if the estimated profit according to the average values is higher as the given percent number. Set to zero percent to turn it off."],
					get = _GetValue,
					set = _SetValue,
					min = 0,
					max = 500,
					isPercent = true,
					step = 1,
					passValue = "snipeAverage",
					order = 3,
			},
			minimumProfit = {
				type = "range",
					name = L["Minimum profit"],
					desc = L["Minimum profit in silver that is needed before sniping for items. Will be ignored for bookmarked items."],
					get = _GetValue,
					set = _SetValue,
					min = 1,
					max = 1000,
					step = 1,
					passValue = "minimumProfit",
					order = 4,
			},
			show = {
				type = "toggle",
					name = L["Show snipes"],
					desc = L["Selects whether any existing snipes should be shown in the GameTooltip."],
					get = _GetValue,
					set = _SetValue,
					passValue = "showSnipes",
					order = 5,
			},
			reset = {
				type = "execute",
				name = L["Reset database"],
				desc = L["Resets the database of snipes for the current realm."],
				func = function() StaticPopup_Show("SNIPER_DB_RESET") end,
				order = 10,
			},
		}
	}
end

--[[
	Returns true, if the item may be bought depending from earlier decisions
	resp. the amount of money the player currently has.
--]]
local function _MayBuy(name, count, bid, buyout)
	local rtn = true;
	local money = GetMoney();
	if (bid and bid > 0 and bid > money) then
		rtn = false;
	elseif (buyout and buyout > 0 and buyout > money) then
		rtn = false;
	else
		if (count > 1) then
			if (bid) then
				bid = bid / count;
			end
			if (buyout) then
				buyout = buyout / count;
			end
		end
		local old = self.maxValues[name];
		if (old) then
			if (old.bid and old.bid > 0) then
				if (bid and bid > 0 and bid >= old.bid) then
					if (not buyout or buyout == 0 or buyout >= old.buyout) then
						rtn = false;
					end
				end
			end
		end
	end
	return rtn;
end

--[[
	Initializes the addon.
--]]
function vendor.Sniper:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("Sniper");
	vendor.Vendor:RegisterDefaults("Sniper", "realm", {
		wanted = {},
		version = SNIPER_VERSION
	});
	vendor.Vendor:RegisterDefaults("Sniper", "profile", {
		showSnipes = true,
		snipeBookmarked = true,
		snipeSellPrices = true,
		minimumProfit = 30,
		snipeAverage = 0.0,
	});
	_CreateResetDialog();
	_CreateOptions();
end

--[[
	Initializes the addon.
--]]
function vendor.Sniper:OnEnable()
	self.maxValues = {};
	vendor.Scanner:AddScanItemListener(self);
	self.snipeCreateDialog = vendor.SnipeCreateDialog:new();
	self:RegisterEvent("CHAT_MSG_SYSTEM");
	self:RegisterEvent("UI_ERROR_MESSAGE");
	vendor.TooltipHook:AddAppender(self, 20);
	vendor.Scanner:AddScanSnapshotListener(self);
end

--[[
	Informs the listener about the scan of the given item.
	Interface method of ScanItemListener.
 	@param index the current auction house index
	@param itemLink the link of the scanned item.
	@return askForBuy (bool), reason (string)
--]]
function vendor.Sniper:ItemScanned(index, itemLink, itemName)
	if (not self.paused) then
		local auctionName, _, count, _, _, _, minBid, minIncrement, buyout, bidAmount, highBidder, owner  = GetAuctionItemInfo("list", index);
		local playerName = UnitName("player");
		if (auctionName == itemName and playerName ~= owner and not highBidder) then
			local bid = minBid;
			if (bidAmount > 0) then
				bid = bidAmount + minIncrement;
			end
			local rtn = false;
			local reason = nil;
			if (_MayBuy(itemName, count, bid, buyout)) then
				rtn, reason = _SnipeBookmarked(itemLink, itemName, count, bid, buyout, highBidder);
				if (not rtn) then
					rtn, reason = _SnipeSellPrices(itemLink, itemName, count, bid, buyout, highBidder);
				end
				if (not rtn) then
					rtn, reason = _SnipeAverage(itemLink, itemName, count, bid, buyout, highBidder);
				end			
			end
			return rtn, reason;
		end
	end
	return nil;
end

--[[
	Updates the snipe information for the given item. If both prizes are zero, any
	existing snipe information will be removed for this item.
	@param name the name of the item to be sniped.
	@param maxBid the maximal bid to be accepted. Zero won't accept any bid.
	@param maxBuyout the maximal buyout to be accepted. Zero won't accept any buyout.
--]]
function vendor.Sniper:SetSnipeInfo(name, maxBid, maxBuyout)
	vendor.Vendor:Debug("SetSnipeInfo name: "..name.." maxBid: "..maxBid.." maxBuyout: "..maxBuyout);
	if (maxBid == 0 and maxBuyout == 0) then
		self.db.realm.wanted[name] = nil;
	else
		self.db.realm.wanted[name] = strjoin(":", maxBid, maxBuyout);
	end
end

--[[
	Returns maxBid and maxBuyout for the given item. 0 will be returned,
	if nothing is wanted for this item.
--]]
function vendor.Sniper:GetWanted(name)
	local data = self.db.realm.wanted[name];
	if (data) then
		local maxBidStr, maxBuyoutStr = strsplit(":", data);
		return tonumber(maxBidStr), tonumber(maxBuyoutStr);
	end
	return 0, 0;
end

--[[
	Places a snipe for the given item.
	Will only run inside of coroutines.
	@param auctionType should be "list".
	@param index the index in the auction list.
	@param bid will place a bid, if greater zero.
	@param buyout will buy the item, if greater zero.
	@param name for the final check, whether this is the correct item.
--]]
function vendor.Sniper:SnipeForItem(auctionType, index, bid, buyout, name)
	vendor.Vendor:Debug("SnipeForItem");
	local aname = GetAuctionItemInfo(auctionType, index);
	if (not aname or aname ~= name) then
		vendor.Vendor:Print(L["Item not found"]);
		self.buyMessage = L["Item not found"];
		self.buyPending = false;		
	elseif (bid and bid > 0) then
		vendor.Vendor:Debug("do bid");
		AuctionFrame.buyoutPrice = nil;
		PlaceAuctionBid(auctionType, index, bid);
	elseif (buyout and buyout > 0) then
		vendor.Vendor:Debug("do buy index: "..index.." buyout: "..buyout);
		AuctionFrame.buyoutPrice = buyout;
		PlaceAuctionBid(auctionType, index, AuctionFrame.buyoutPrice);
	end
end

--[[
	Waits for a pending item to be bought from auction house.
--]]
function vendor.Sniper:WaitForPendingBuy()
	while (self.buyPending) do
		coroutine.yield();
	end	
end

--[[
	Signals the sniper, that we soon will try to buy an item.
--]]
function vendor.Sniper:StartPendingBuy(itemName, count, bid, buyout)
	if (count > 1) then
		if (bid) then
			bid = bid / count;
		end
		if (buyout) then
			buyout = buyout / count;
		end
	end
	self.pendingBuy = {name = itemName, bid = bid, buyout = buyout};
	self.buyPending = true;
end

--[[
	Signals the sniper, that we won't buy an item.
--]]
function vendor.Sniper:StopPendingBuy()
	self.buyPending = nil;
	if (self.pendingBuy) then
		-- remember the name and values, so we don't ask again the same items in the current scan
		local old = self.maxValues[self.pendingBuy.name];
		local bid = self.pendingBuy.bid;
		local buyout = self.pendingBuy.buyout;
		if (old) then
			if (bid and (not old.bid or bid < old.bid)) then
				old.bid = bid;
			end
			if (buyout and (not old.buyout or buyout < old.buyout)) then
				old.buyout = buyout;
			end
		else
			self.maxValues[self.pendingBuy.name] = {bid = bid, buyout = buyout};
		end
		self.pendingBuy = nil;
	end
end

--[[
	Handles chat events to recognize the creation of auctions.
--]]
function vendor.Sniper:CHAT_MSG_SYSTEM(message)
	if (self.buyPending) then
		if (message == ERR_AUCTION_BID_PLACED) then
			self.buyMessage = L["Ok"];
			self.buyPending = false;
		end
	end	
end
	
--[[
	Handles error messages to recognize failures during creation of auctions.
--]] 
function vendor.Sniper:UI_ERROR_MESSAGE(message)
	if (self.buyPending) then
		if (message == ERR_ITEM_NOT_FOUND) then
			self.buyMessage = L["Item not found"];
			self.buyPending = false;			
		elseif (message == ERR_NOT_ENOUGH_MONEY) then
			self.buyMessage = L["Not enough money"];
			self.buyPending = false;
		elseif (message == ERR_AUCTION_BID_OWN) then
			self.buyMessage = L["Own auction"];
			self.buyPending = false;
		elseif (message == ERR_AUCTION_HIGHER_BID) then
			self.buyMessage = L["Higher bid"];
			self.buyPending = false;
		elseif (message == ERR_ITEM_MAX_COUNT) then
			self.buyMessage = L["No more items of this type possible"];
			self.buyPending = false;
		end
	end	
end

--[[
	Opens the snipe dialog. The name is optional.
--]]
function vendor.Sniper:OpenSnipeDialog(name)
	self.snipeCreateDialog:Show(name);
end

--[[
	Callback for Tooltip integration
--]]
function vendor.Sniper:AppendToGameTooltip(tooltip, itemLink, itemName, count)
	if (self.db.profile.showSnipes) then
		local maxBid, maxBuyout = self:GetWanted(itemName);
		if (maxBid > 0) then
			local msg1;
			local msg2;
			if (count > 1) then
				msg1 = L["Snipe bid (%d)"]:format(count);
				msg2 = vendor.Format.FormatMoney(maxBid, true).."("..vendor.Format.FormatMoney(maxBid * count, true)..")";
			else
				msg1 = L["Snipe bid"];
				msg2 = vendor.Format.FormatMoney(maxBid, true);			
			end
			tooltip:AddDoubleLine(msg1, msg2);
		end
		if (maxBuyout > 0) then
			local msg1;
			local msg2;
			if (count > 1) then
				msg1 = L["Snipe buyout (%d)"]:format(count);
				msg2 = vendor.Format.FormatMoney(maxBuyout, true).."("..vendor.Format.FormatMoney(maxBuyout * count, true)..")";
			else
				msg1 = L["Snipe buyout"];
				msg2 = vendor.Format.FormatMoney(maxBuyout, true);			
			end
			tooltip:AddDoubleLine(msg1, msg2);
		end
	end
end

--[[
	Pauses any snipes for the current scan.
--]]
function vendor.Sniper:PauseThisScan()
	self.paused = true;
end

--[[
	ScanResultListener interface method for new or updated
	Scansnapshots. The last scan is now finished.
--]]
function vendor.Sniper:ScanSnapshotUpdated(snapshot, isNeutral)
	self.paused = nil;
end
