--[[
	Adds a new tab to the auction house for more convinient selling of items.
	Enables the selling of many items at once, prefills the desired prizes and
	remembers them.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.Seller = vendor.Vendor:NewModule("Seller", "AceEvent-2.0", "AceHook-2.1");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local self = vendor.Seller;

local INVENTORY_LINE_HEIGHT = 27;
local INVENTORY_LINES = 11;
local BID_TYPE_PER_ITEM = 1;
local BID_TYPE_PER_STACK = 2;
local BID_TYPE_ALL = 3;
local PRIZE_MODEL_FIX = 1;
local PRIZE_MODEL_AVERAGE = 2;
local PRIZE_MODEL_MARKET = 3;
local DURATIONS = {720, 1440, 2880};
local STACK_DROPDOWN_ID = 1;
local COUNT_DROPDOWN_ID = 2;
local PRIZING_DROPDOWN_ID = 3;
local BIDTYPE_DROPDOWN_ID = 4;
local MARKET_PERCENT = 2;

--[[
	The refresh button has been clicked. A scan will be triggered.
--]]
local function _RefreshClicked(self) 
	if (self.item) then
		vendor.Scanner:Scan(self.item.link);
	elseif (self.lastItem) then
		vendor.Scanner:Scan(self.lastItem.link);
	end
end

--[[
	Tries to retrieve a link for an item with the given name in the inventory.
--]]
local function _FindInventoryItemLink(name)
   	for bag = 0, 4 do
      	for slot = 1, GetContainerNumSlots(bag) do
	 		local itemLink = GetContainerItemLink(bag, slot);
	 		if (itemLink) then
	    		local itemName = GetItemInfo(itemLink);
	    		if (name and name == itemName) then
	    			return itemLink; -- don't like this, but we need to get out of two loops
	    		end
	 		end
      	end
   	end
   	return nil;
end

--[[
	Returns "minBid, buyOut" for the given item, if known.
--]]
local function _GetItemAuctionSellValues(itemLink)
	local itemLinkKey = vendor.Items:GetItemLinkKey(itemLink);
	local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, vendor.AuctionHouse:IsNeutral());
	if (itemInfo) then
		return itemInfo.minBid, itemInfo.buyout;
	end
	return nil;
end

--[[
	Returns "avgMinBid, avgBuyout" for the given item, if known.
--]] 
local function _GetItemAuctionAvgValues(itemLink)
	local itemLinkKey = vendor.Items:GetItemLinkKey(itemLink);
	local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, vendor.AuctionHouse:IsNeutral());
	if (itemInfo) then
		return itemInfo.avgMinBid, itemInfo.avgBuyout;
	end
	return nil;
end

--[[
	Returns "avgMinBid, avgBuyout" for the given item for current auctions, if known.
--]] 
local function _GetItemAuctionCurrentValues(itemLink)
	local percent = MARKET_PERCENT / 100.0;
	local minBid, buyout = vendor.Statistic:GetCurrentAuctionInfo(itemLink, vendor.AuctionHouse:IsNeutral());
	if (minBid and minBid > 0) then
		minBid = math.floor(minBid - (percent * minBid) + 0.5);
		if (buyout and buyout > 0) then
			buyout = math.floor(buyout - (percent * buyout) + 0.5);
		end
	end
	if (minBid and buyout and buyout > 0 and minBid > buyout) then
		buyout = minBid; 
	end
	return minBid, buyout;
end

--[[
	Updates the deposit chown.
--]]
local function _UpdateDeposit(self)
	if (self.item) then
		local deposit = self.deposits[self.duration];
		local mult = 1;
		local stackCount = self.countDropDown:GetSelectedValue();
		if (type(stackCount) == "string") then
			-- sell all
			mult = self.item.count;
		else
			mult = self.stackDropDown:GetSelectedValue() * stackCount;
		end
		local copper = math.floor(mult * deposit + 0.5);
		MoneyFrame_Update("SalesFrameDeposit", copper);
	else
		MoneyFrame_Update("SalesFrameDeposit", 0);
	end
end

--[[
	Sets the given prizes for the item to be sold. 
--]]
local function _SetPrizes(self, minBid, buyout)
	MoneyInputFrame_SetCopper(self.startPriceBut, minBid);
	MoneyInputFrame_SetCopper(self.buyoutPriceBut, buyout);
	_UpdateDeposit(self);
end

--[[
	Sets the prizes for selling the item. 
--]]
local function _InitPrizes(self)
	vendor.Vendor:Debug("_InitPrizes");
	local minBid = 0;
	local buyout = 0;
	local mult = 1;
	if (self.item) then
		-- determine the multiplier for the displayed prize
		local bidType = self.bidTypeDropDown:GetSelectedValue();
		if (bidType == BID_TYPE_PER_STACK) then
			mult = self.stackDropDown:GetSelectedValue();
		elseif (bidType == BID_TYPE_ALL) then
			local count = self.countDropDown:GetSelectedValue();
			if (type(count) == "string") then
				-- sell all
				mult = self.item.count;
			else
				mult = self.stackDropDown:GetSelectedValue() * count;
			end
		end
		-- determine prizing model
		local prizeModel = self.prizingDropDown:GetSelectedValue();
		if (prizeModel == PRIZE_MODEL_FIX) then
			minBid, buyout = _GetItemAuctionSellValues(self.item.link);
		elseif (prizeModel == PRIZE_MODEL_AVERAGE) then
			minBid, buyout = _GetItemAuctionAvgValues(self.item.link);
		elseif (prizeModel == PRIZE_MODEL_MARKET) then
			minBid, buyout = _GetItemAuctionCurrentValues(self.item.link);
		end
		if (not minBid or minBid <= 0) then
			minBid = self.startPrize;
		end
	end
	_SetPrizes(self, minBid * mult, (buyout or 0) * mult);
end

--[[
	Updates the count dropdown button
--]]
local function _UpdateCountDropDown(self)
	if (not self.item) then
		return;
	end
	local counts = {};
	local stackCount = self.stackDropDown:GetSelectedValue();
	local n = self.item.count / stackCount;
	if (n < 1) then
		n = 1;
	end
	n = math.floor(n);
	local hasAdd = self.item.count > n * stackCount;
	local select = n;
	if (hasAdd) then
		select = ""..n.."+";
	end
	local i = 1;
	local prev = nil;
	-- add all integral stack sizes
	while i <= n do
		table.insert(counts, i);
		prev = i;
		i = i * 2;
	end
	-- add the last if there are more items
	if (hasAdd or prev and prev < n) then
		if (hasAdd) then
			table.insert(counts, select);
		else
			table.insert(counts, n);
		end
	end
	self.countDropDown:SetItems(counts, select);
end

--[[
	Updates the dropdown buttons accoring to the selected item
--]]
local function _UpdateDropDowns(self)
	-- stacksize
	local stackSize = math.min(self.item.stackCount, self.item.count);
	local stackSizes = self.stackSizes[stackSize];
	if (not stackSizes) then
		stackSizes = {};
		self.stackSizes[stackSize] = stackSizes;
		local i = 1;
		local prev = nil;
		while i <= stackSize do
			table.insert(stackSizes, i);
			prev = i;
			if (i < 5) then
				i = i + 1;
			else
				i = i * 2;
			end
		end
		if (prev and prev < stackSize) then
			table.insert(stackSizes, stackSize);
		end
	end
	if (self.item.preferedStackSize > 0) then
		self.stackDropDown:SetItems(stackSizes, self.item.preferedStackSize);
	else
		self.stackDropDown:SetItems(stackSizes, stackSize);
	end
	-- stack count
	_UpdateCountDropDown(self);
end

--[[
	Shows information if mouse is over the selected item
--]]
local function _OnEnterItem(self)
	GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
	if (self.item) then
		GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
		GameTooltip:SetHyperlink(self.item.link);
	else
		GameTooltip:SetText(L["Drag an item here to auction it"]);
	end
end

--[[
	Update the per item sell prizes and no. of stacks currently selected.
--]]
local function _UpdateSellInfo(self)
	local stackSize = self.stackDropDown:GetSelectedValue();
	self.minBidPerItem = MoneyInputFrame_GetCopper(self.startPriceBut);
	self.buyoutPerItem = MoneyInputFrame_GetCopper(self.buyoutPriceBut);
	local numStacks, items, _;
	local stackCount = self.countDropDown:GetSelectedValue();
	self.backlog = 0;
	if (type(stackCount) == "string") then
		-- we have a backdrop
		_, _, numStacks = string.find(stackCount, "^(%d+).*");
		self.backlog = self.item.count - (numStacks * stackSize);
		items = self.item.count;
	else
		numStacks = stackCount;
		items = numStacks * stackSize;
	end
	self.numStacks = numStacks;
	self.numItems = items;	
	local bidType = self.bidTypeDropDown:GetSelectedValue();
	if (bidType == BID_TYPE_PER_ITEM) then
		-- nothing to be done
	elseif bidType == BID_TYPE_PER_STACK then
		self.minBidPerItem = self.minBidPerItem / stackSize;
		self.buyoutPerItem = self.buyoutPerItem / stackSize;
	elseif bidType == BID_TYPE_ALL then
		self.minBidPerItem = self.minBidPerItem / items;
		self.buyoutPerItem = self.buyoutPerItem / items;			
	end
end

--[[
	Checks whether the current auction is valid and shows an error messages
	or enables the sell button accordingly.
--]]
local function _ValidateAuction(self)
	self.createBut:Show();
	self.createBut:Disable();
	self.buyoutErrorText:Hide();
	if (not self.item) then
		-- not item selected
		return;
	end
	local startPrize = MoneyInputFrame_GetCopper(self.startPriceBut);
	local buyoutPrize = MoneyInputFrame_GetCopper(self.buyoutPriceBut);
	-- Buyout price is less than the start price
	if (buyoutPrize > 0 and startPrize > buyoutPrize) then
		self.createBut:Hide();
		self.buyoutErrorText:Show();
		return;
	end
	-- Start price is 0 or greater than the max allowed
	if (startPrize < 1 or startPrize > MAXIMUM_BID_PRICE) then
		return;
	end
	_UpdateSellInfo(self);
	self.createBut:Enable();
end

--[[
	Selects the given itemLink for selling
--]]
local function _SelectItem(self, itemLink)
	if (not itemLink) then
		SalesFrameItemName:SetText("");
		SalesFrameItemCount:Hide();
		SalesFrameItem:SetNormalTexture(nil);
		self.lastItem = self.item;
		self.item = nil;
	else
		local name, link, stackCount, texture, count = self.inventory:GetItemInfo(itemLink);
		if (name) then
			SalesFrameItem:SetNormalTexture(texture);
			SalesFrameItemName:SetText(name);
			if (count > 1) then
				SalesFrameItemCount:SetText(count);
				SalesFrameItemCount:Show();
			else
				SalesFrameItemCount:Hide();
			end
			local itemLinkKey = vendor.Items:GetItemLinkKey(itemLink);
			local preferedStackSize = 0;
			local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, vendor.AuctionHouse:IsNeutral());
			if (itemInfo and itemInfo.stackSize <= count) then
				preferedStackSize = itemInfo.stackSize;			
			end
			self.item = {name = name, link = link, stackCount = stackCount, texture = texture, count = count, preferedStackSize = preferedStackSize, itemLinkKey = itemLinkKey};
			_UpdateDropDowns(self);
		else
			-- item disappaered after it had been selected
			_SelectItem(self, nil);
		end
	end
	_InitPrizes(self);
	_ValidateAuction(self);
end

--[[
	Picks the currently dragged item for selling
--]]
local function _PickItem(self)
	local type, itemID, itemLink = GetCursorInfo();
	if (type and type == "item") then
		-- use blizzard's auction slot to calculate deposit and start prize 
		ClickAuctionSellItemButton();
		local _, _, count, _, _, price = GetAuctionSellItemInfo();
		self.deposits = {};
		for _, duration in ipairs(DURATIONS) do
			self.deposits[duration] = CalculateAuctionDeposit(duration) / count;
		end;
		self.startPrize = price / count;
		_SelectItem(self, itemLink);
		-- now free blizzard's auction slot
		ClickAuctionSellItemButton();
		ClearCursor();
		-- display all known auctions about the item
		local scanSet = vendor.Scanner:GetScanSet(vendor.AuctionHouse:IsNeutral(), itemLink);
		self.scanResultFrame:Display(scanSet);
		vendor.Vendor:Debug("Check ScanSet freshness and size");
		if (scanSet:Size() == 0 or (scanSet:GetUpToDateness() >= self.db.profile.autoRefreshTime)) then
			vendor.Vendor:Debug("Refresh ScanSet");
			_RefreshClicked(self);
		end
	end
end

--[[
	Updates the list of inventory items.
--]]
local function _UpdateInventory(self)
	if (self.frame:IsVisible()) then
		self.inventory:Update();
		if (self.item) then
			_SelectItem(self, self.item.link);
		end
	 end
end

--[[
	Auction duration radio button has been clicked
--]]
local function _DurationSelected(self, id)
	PlaySound("igMainMenuOptionCheckBoxOn");
	self.shortBut:SetChecked(nil);
	self.mediumBut:SetChecked(nil);
	self.longBut:SetChecked(nil);
	if (id == 1) then
		self.shortBut:SetChecked(1);
		self.duration = DURATIONS[1];
	elseif (id == 2) then
		self.mediumBut:SetChecked(1);
		self.duration = DURATIONS[2];
	else
		self.longBut:SetChecked(1);
		self.duration = DURATIONS[3];
	end
	_UpdateDeposit(self);
end

--[[
	Creates an auction with the currently selected item.
--]]
local function _CreateAuction(self)
	if (self.item) then
		local itemLink = self.item.link;
		local runTime = self.duration;
		local stackSize = self.stackDropDown:GetSelectedValue();
		if (self.selectedStackSize) then
			-- a stackSize was selected manually, remember it
      		local itemInfo = vendor.Items:GetItemInfo(self.item.itemLinkKey, vendor.AuctionHouse:IsNeutral());
			if (not itemInfo) then
				itemInfo = vendor.Items:CreateItemInfo();
			end
			itemInfo.stackSize = self.selectedStackSize;
	 		vendor.Items:PutItemInfo(self.item.itemLinkKey, vendor.AuctionHouse:IsNeutral(), itemInfo);
			self.selectedStackSize = nil;
		end
		_UpdateSellInfo(self);
		local task = vendor.SellTask:new(self.inventory, itemLink, self.minBidPerItem, self.buyoutPerItem, runTime, stackSize, self.numStacks, self.backlog);
		vendor.TaskQueue:AddTask(task);
	end
end

--[[
	Creates the auction buttons.
--]]
local function _CreateAuctionButtons(self)
	-- item to be sold
	local f = self.frame:CreateFontString("SalesFrameItemText", "ARTWORK", "GameFontHighlightSmall");
	f:SetText(L["Auction Item"]);
	f:SetPoint("TOPLEFT", 28, -83);
	self.itemBut = CreateFrame("Button", "SalesFrameItem", self.frame);
	self.itemBut.ctrl = self; 
	self.itemBut:SetWidth(37);
	self.itemBut:SetHeight(37);
	self.itemBut:SetPoint("TOPLEFT", 28, -98);
	local f = self.itemBut:CreateFontString("SalesFrameItemName", "BACKGROUND", "GameFontNormal");
	f:SetWidth(124);
	f:SetHeight(30);
	f:SetPoint("TOPLEFT", self.itemBut, "TOPRIGHT", 5, 0);
	local f = self.itemBut:CreateFontString("SalesFrameItemCount", "OVERLAY", "NumberFontNormal");
	f:SetJustifyH("RIGHT");
	f:SetPoint("BOTTOMRIGHT", -5, 2);
	self.itemBut:RegisterForDrag("LeftButton");
	self.itemBut:SetScript("OnClick", function() _PickItem(this.ctrl); _ValidateAuction(this.ctrl); end);
	self.itemBut:SetScript("OnDragStart", function() _PickItem(this.ctrl); _ValidateAuction(this.ctrl); end);
	self.itemBut:SetScript("OnReceiveDrag", function() _PickItem(this.ctrl); end);
	self.itemBut:SetScript("OnEnter", function() _OnEnterItem(this.ctrl); end);
	self.itemBut:SetScript("OnLeave", function() GameTooltip:Hide(); end);
	self.itemBut:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square");
	
	-- duration of auction
	f = self.frame:CreateFontString("SalesFrameDurationText", "ARTWORK", "GameFontHighlightSmall");
	f:SetText(L["Auction Duration"]);
	f:SetPoint("TOPLEFT", 28, -167);

	self.shortBut = CreateFrame("CheckButton", "SalesFrameShort", self.frame, "VendorRadioButtonTemplate");
	self.shortBut:SetID(1);
	self.shortBut:SetPoint("TOPLEFT", SalesFrameDurationText, "BOTTOMLEFT", 3, -2);
	self.shortBut.ctrl = self;
	self.shortBut:SetScript("OnClick", function() _DurationSelected(this.ctrl, 1) end);
	getglobal("SalesFrameShortText"):SetText("12 "..GetText("HOURS", nil, 12));
	
	self.mediumBut = CreateFrame("CheckButton", "SalesFrameMedium", self.frame, "VendorRadioButtonTemplate");
	self.mediumBut:SetID(2);
	self.mediumBut:SetPoint("TOPLEFT", SalesFrameShort, "BOTTOMLEFT", 0, 0);
	self.mediumBut.ctrl = self;
	self.mediumBut:SetScript("OnClick", function() _DurationSelected(this.ctrl, 2) end);
	getglobal("SalesFrameMediumText"):SetText("24 "..GetText("HOURS", nil, 24));
	
	self.longBut = CreateFrame("CheckButton", "SalesFrameLong", self.frame, "VendorRadioButtonTemplate");
	self.longBut:SetID(2);
	self.longBut:SetPoint("TOPLEFT", SalesFrameMedium, "BOTTOMLEFT", 0, 0);
	self.longBut.ctrl = self;
	self.longBut:SetScript("OnClick", function() _DurationSelected(this.ctrl, 3) end);
	getglobal("SalesFrameLongText"):SetText("48 "..GetText("HOURS", nil, 48));
	
	self.duration = DURATIONS[1];
	self.shortBut:SetChecked(1);
	
	-- stacksize
	self.stackDropDown = vendor.DropDownButton:new("SalesFrameStackSize", self.frame, 70, L["Stack size"], L["Selects the size of the stacks to be sold."]);
	self.stackDropDown:SetPoint("TOPLEFT", 100, -150);
	self.stackDropDown:SetId(STACK_DROPDOWN_ID);
	self.stackDropDown:SetListener(self);
	
	-- count of "stacks"
	self.countDropDown = vendor.DropDownButton:new("SalesFrameCount", self.frame, 70, L["Amount"], L["Selects the number of stacks to be sold.\nThe number with the +-suffix sells\nalso any remaining items."]);
	self.countDropDown:SetPoint("TOPLEFT", SalesFrameStackSize, "BOTTOMLEFT", 0, -13);
 	self.countDropDown:SetId(COUNT_DROPDOWN_ID);
	self.countDropDown:SetListener(self);
	
	-- prizing model
	self.prizingDropDown = vendor.DropDownButton:new("SalesFramePrizing", self.frame, 70, L["Price calculation"], L["Selects the mode for calculating the sell prices.\nThe default mode Fixed price just select the last sell price."]);
	self.prizingDropDown:SetPoint("TOPLEFT", SalesFrameCount, "BOTTOMLEFT", 0, -13);
 	self.prizingDropDown:SetId(PRIZING_DROPDOWN_ID);
	self.prizingDropDown:SetListener(self);
    local prizeModels = {};
	table.insert(prizeModels, {value = PRIZE_MODEL_FIX, text = L["Fixed price"]});
	table.insert(prizeModels, {value = PRIZE_MODEL_MARKET, text = L["Market price - %d%%"]:format(MARKET_PERCENT)});
	table.insert(prizeModels, {value = PRIZE_MODEL_AVERAGE, text = L["Average price"]});
	self.prizingDropDown:SetItems(prizeModels, PRIZE_MODEL_FIX);

	-- bid type (per item, per stack, ...)
	self.bidTypeDropDown = vendor.DropDownButton:new("SalesFrameBidType", self.frame, 70, L["Bid type"], L["Selects which prices should be shown in the bid and buyout input fields."]);
	self.bidTypeDropDown:SetPoint("RIGHT", SalesFramePrizing, "LEFT", 30, 0);
	self.bidTypeDropDown:SetId(BIDTYPE_DROPDOWN_ID);
	self.bidTypeDropDown:SetListener(self);
    local bidTypes = {};
	table.insert(bidTypes, {value = BID_TYPE_PER_ITEM, text = L["Per item"]});
	table.insert(bidTypes, {value = BID_TYPE_PER_STACK, text = L["Stack"]});
	table.insert(bidTypes, {value = BID_TYPE_ALL, text = L["Overall"]});
	self.bidTypeDropDown:SetItems(bidTypes, BID_TYPE_PER_ITEM);
		
	-- starting prize
	f = self.frame:CreateFontString("SalesFramePriceText", "ARTWORK", "GameFontHighlightSmall");
	f:SetText(L["Starting Price"]);
	f:SetPoint("TOPLEFT", 28, -283);
	self.startPriceBut = CreateFrame("Frame", "SalesFrameStartPrice", self.frame, "MoneyInputFrameTemplate");
	self.startPriceBut:SetPoint("TOPLEFT", SalesFramePriceText, "BOTTOMLEFT", 3, -2);
	self.startPriceBut.controller = self;
	MoneyInputFrame_SetOnvalueChangedFunc(self.startPriceBut, function() _ValidateAuction(self) end);
	SalesFrameStartPriceGold:SetMaxLetters(6);
		
	-- buyout prize
	f = self.frame:CreateFontString("SalesFrameBuyoutText", "ARTWORK", "GameFontHighlightSmall");
	f:SetText(L["Buyout Price"].." |cff808080("..L["optional"]..")|r");
	f:SetPoint("TOPLEFT", SalesFramePriceText, 0, -30);
	self.buyoutPriceBut = CreateFrame("Frame", "SalesFrameBuyoutPrice", self.frame, "MoneyInputFrameTemplate");
	--- ?????? why not ????? self.buyoutPriceBut:SetPoint("TOPLEFT", SalesFrameBuyoutText, "BOTTOMLEFT", 3, -2);
	self.buyoutPriceBut:SetPoint("TOPLEFT", 31, -324);
	self.buyoutPriceBut.controller = self;
	MoneyInputFrame_SetOnvalueChangedFunc(self.buyoutPriceBut, function() _ValidateAuction(self) end);
	SalesFrameBuyoutPriceGold:SetMaxLetters(6);

	-- Set focus rules
	MoneyInputFrame_SetPreviousFocus(SalesFrameStartPrice, SalesFrameBuyoutPriceCopper);
	MoneyInputFrame_SetNextFocus(SalesFrameStartPrice, SalesFrameBuyoutPriceGold);
	MoneyInputFrame_SetPreviousFocus(SalesFrameBuyoutPrice, SalesFrameStartPriceCopper);
	MoneyInputFrame_SetNextFocus(SalesFrameBuyoutPrice, SalesFrameStartPriceGold);
	
	-- total deposit
	f = self.frame:CreateFontString("SalesFrameDepositText", "ARTWORK", "GameFontNormal");
	f:SetText(L["Deposit:"]);
	f:SetPoint("TOPLEFT", 28, -364);
	
	self.depositBut = CreateFrame("Frame", "SalesFrameDeposit", self.frame, "DepositFrameTemplate");
	self.depositBut:SetPoint("LEFT", SalesFrameDepositText, "RIGHT", 5, 0);
	self.depositBut.controller = self;
	
	-- up-to-dateness of the item list
	f = self.frame:CreateFontString("SalesFrameUpToDatenessText", "ARTWORK", "GameFontNormal");
	f:SetText("");
	f:SetPoint("BOTTOMLEFT", 300, 20);
	self.upToDatenessText = f;
	
	-- auction creation
	self.createBut = CreateFrame("Button", "SalesFrameCreate", self.frame, "UIPanelButtonTemplate");
	self.createBut:SetText(L["Create Auction"]);
	self.createBut:SetWidth(191);
	self.createBut:SetHeight(20);
	self.createBut:SetPoint("BOTTOMLEFT", 18, 39);
	self.createBut.ctrl = self;
	self.createBut:SetScript("OnClick", function() _CreateAuction(this.ctrl) end);
	
	-- auction creation not possible error text
	self.buyoutErrorText = self.frame:CreateFontString("SalesFrameBuyoutErrorText", "ARTWORK", "GameFontRedSmall");
	self.buyoutErrorText:SetText(L["Buyout < bid"]);
	self.buyoutErrorText:SetPoint("TOPLEFT", self.createBut, "TOPLEFT", 15, -5);
end

--[[
	Creates the central close etc. buttons.
--]]
local function _CreateFrameButtons(self)
	self.closeBut = vendor.AuctionHouse:CreateCloseButton(self.frame, "SalesFrameClose");
	
	self.refreshBut = vendor.GuiTools.CreateButton(self.frame, "UIPanelButtonTemplate", 86, 22, L["Refresh"], L["Refreshes the list of current auctions for the selected item to be sold."]); 
	CreateFrame("Button", "SalesFrameRefresh", self.frame, "UIPanelButtonTemplate");
	self.refreshBut:SetText(L["Refresh"]);
	self.refreshBut:SetWidth(86);
	self.refreshBut:SetHeight(22);
	self.refreshBut:SetPoint("BOTTOMLEFT", 212, 14);
	self.refreshBut:SetScript("OnClick", function() _RefreshClicked(self) end);
end

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.Seller = {
		type = "group",
		name = L["Seller"],
		desc = L["Seller"],
		args = {
			autoRefreshTime = {
				type = "range",
				name = L["Autorefresh time"],
				desc = L["Selects the minimum time in seconds that has to be passed, before the auction house is automatically scanned for the item to be sold."],
				min = 10,
				max = 360,
				step = 1,
				get = _GetValue,
				set = _SetValue,
				passValue = "autoRefreshTime",
				order = 1,
			},
		}
	}
end

--[[
	Initializes the module.
--]]
function vendor.Seller:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("Seller");
	vendor.Vendor:RegisterDefaults("Seller", "profile", {
		autoRefreshTime = 60,
	});
	_CreateOptions();	
end

--[[
	Initializes the module.
--]]
function vendor.Seller:OnEnable()
	self.frame = vendor.AuctionHouse:CreateTabFrame("AuctionFrameVendor", 4, "Vendor", self);
	self.vendorTabButton = vendor.AuctionHouse:CreateTabButton(L["Seller"], 4);
	self.isScanning = false;
	self.stackSizes = {};
	self.inventory = vendor.InventoryHandle:new();
	_CreateFrameButtons(self);
	_CreateAuctionButtons(self);
	self.scanResultFrame = vendor.CompactScanResultFrame:new("SalesScanResultFrame", self.frame, 15, self.upToDatenessText);
	self.scanResultFrame:SetPoint("TOPLEFT", 25, 33);
	self.lastItem = nil;
	self:RegisterEvent("BAG_UPDATE");
	self:RegisterEvent("AUCTION_HOUSE_CLOSED");
	self:RegisterEvent("NEW_AUCTION_UPDATE");
   	self:Hook("StartAuction", true);
	-- register for new ScanResults
	vendor.Scanner:AddScanSnapshotListener(self);	
end

--[[
	Updates the gui for displaying the frame (Interface method).
--]]
function vendor.Seller:UpdateTabFrame()
	AuctionFrameTopLeft:SetTexture("Interface\\Addons\\Vendor\\Media\\UI-AuctionFrame-Auction-TopLeft");
	AuctionFrameTop:SetTexture("Interface\\AuctionFrame\\UI-AuctionFrame-Auction-Top");
	AuctionFrameTopRight:SetTexture("Interface\\AuctionFrame\\UI-AuctionFrame-Auction-TopRight");
	AuctionFrameBotLeft:SetTexture("Interface\\AuctionFrame\\UI-AuctionFrame-Auction-BotLeft");
	AuctionFrameBot:SetTexture("Interface\\AuctionFrame\\UI-AuctionFrame-Auction-Bot");
	AuctionFrameBotRight:SetTexture("Interface\\Addons\\Vendor\\Media\\UI-AuctionFrame-Auction-BotRight");
end

--[[
	Returns the type of this auction house tab.
--]]
function vendor.Seller:GetTabType()
	return "seller";
end
	
--[[
	Shows the tabbed frame (Interface method).
--]]
function vendor.Seller:ShowTabFrame()
	self.frame:Show();
	_UpdateInventory(self);
	self.scanResultFrame:Update();
end

--[[
	Hides the tabbed frame (Interface method).
--]]
function vendor.Seller:HideTabFrame()
	self.frame:Hide();
end

--[[
	Interface method for DropDownButton.
--]]
function vendor.Seller:DropDownButtonSelected(button, value)
	if (self.item) then
		local id = button:GetId();
		local bidType = self.bidTypeDropDown:GetSelectedValue();
		local stackSize = self.stackDropDown:GetSelectedValue();
		if (id == STACK_DROPDOWN_ID) then
			-- new stacksize has been selected
			_UpdateCountDropDown(self);
			self.selectedStackSize = stackSize;
			-- only change the prizes, if they are shown per stack
			if (bidType == BID_TYPE_PER_STACK) then
				_SetPrizes(self, self.minBidPerItem * stackSize, self.buyoutPerItem * stackSize);
			end
		elseif (id == PRIZING_DROPDOWN_ID) then
			-- have to recalculate the prizes
			_InitPrizes(self);
		elseif (id == BIDTYPE_DROPDOWN_ID) then
			-- change prizes according to stacksize/numItems
			if (bidType == BID_TYPE_PER_ITEM) then
				_SetPrizes(self, self.minBidPerItem, self.buyoutPerItem);
			elseif (bidType == BID_TYPE_PER_STACK) then
				_SetPrizes(self, self.minBidPerItem * stackSize, self.buyoutPerItem * stackSize);
			elseif (bidType == BID_TYPE_ALL) then
				_SetPrizes(self, self.minBidPerItem * self.numItems, self.buyoutPerItem * self.numItems);
			end
		elseif (id == COUNT_DROPDOWN_ID) then
			-- change only needed if bid type is set to all
			if (bidType == BID_TYPE_ALL) then
				_SetPrizes(self, self.minBidPerItem * self.numItems, self.buyoutPerItem * self.numItems);
			end
		end
		_UpdateDeposit(self);
	end
end

--[[
	BAG_UPDATe event has been triggered.
--]]
function vendor.Seller:BAG_UPDATE()
	_UpdateInventory(self);
end

--[[
	Auction house has been closed
--]]
function vendor.Seller:AUCTION_HOUSE_CLOSED()
	_SelectItem(self, nil);
end

--[[
	Hooks the StartAuction function to remember the prizes.
--]]
function vendor.Seller:StartAuction(minBid, buyoutPrize, runTime)
   	-- we want to remember the "minBid" and "buyoutPrice" for the next time
   	local name, texture, count, quality, canUse, price = GetAuctionSellItemInfo();
   	if (name) then
      	-- unfortunately we need the itemLink, we have to scan the inventory
      	local itemLink = _FindInventoryItemLink(name);
      	if (itemLink) then
      		local itemLinkKey = vendor.Items:GetItemLinkKey(itemLink);
      		local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, vendor.AuctionHouse:IsNeutral());
			if (not itemInfo) then
				itemInfo = vendor.Items:CreateItemInfo();
			end
	 		itemInfo.minBid = math.floor(minBid / count);
	 		itemInfo.buyout = math.floor(buyoutPrize / count);
	 		vendor.Items:PutItemInfo(itemLinkKey, vendor.AuctionHouse:IsNeutral(), itemInfo);
      	end
   	end
   	self.hooks.StartAuction(minBid, buyoutPrize, runTime);
end

--[[
	Event callback for filling in last auction prizes for the given item.
--]]
function vendor.Seller:NEW_AUCTION_UPDATE()
	MoneyInputFrame_SetCopper(StartPrice, 0);
	MoneyInputFrame_SetCopper(BuyoutPrice, 0);
	-- we want to fill in old "minBid" and "buyoutPrice"
	local name, texture, count, quality, canUse, price = GetAuctionSellItemInfo();
	if (name) then
		-- unfortunately we need the itemLink, we have to scan the inventory
		local itemLink = _FindInventoryItemLink(name);
		if (itemLink) then
			local itemInfo = vendor.Items:GetItemInfo(itemLinkKey, vendor.AuctionHouse:IsNeutral());
			if (itemInfo) then
				if (itemInfo.minBid > 0) then
					MoneyInputFrame_SetCopper(StartPrice, itemInfo.minBid * count);
	 			end
	 			if (itemInfo.buyout > 0) then
	    			MoneyInputFrame_SetCopper(BuyoutPrice, itemInfo.buyout * count);
	 			end
      		end
      	end
   	end
end

--[[
	Returns information about the currently selected item, if any.
	@return itemName, itemLink
--]]
function vendor.Seller:GetSelectedItemInfo()
	if (self.item) then
		return self.item.name, self.item.link;
	end
	return nil;
end	

--[[
	ScanResultListener interface method for new or updated
	Scanresults.
--]]
function vendor.Seller:ScanSnapshotUpdated(scanSnapshot, isNeutral)
	vendor.Vendor:Debug("vendor.Seller:ScanSnapshotUpdated");
	local prizeModel = self.prizingDropDown:GetSelectedValue();
	if (prizeModel == PRIZE_MODEL_AVERAGE or prizeModel == PRIZE_MODEL_MARKET) then
		_InitPrizes(self);
	end
end

