--[[
	Shows sell prizes in the tooltip and updates them while visiting merchants.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.SellPrizes = vendor.Vendor:NewModule("SellPrizes", "AceEvent-2.0", "AceHook-2.1");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local itemPrice = AceLibrary("ItemPrice-1.1")
local self = vendor.SellPrizes;

local SELL_PRIZES_VERSION = 1;

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Extracts the item code from the given link.
--]]
function _ExtractItemId(link)
   	local _, _, rtn = strfind(link, "(%d+):")
	return tonumber(rtn);
end

--[[
	Resets the database for the current account.
--]]
local function _ResetDb()
	vendor.Vendor:ResetDB("SellPrizes", "account");
	vendor.Vendor:Print(L["Database of sell prizes for current account where reset."]);
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.SellPrizes = {
		type = "group",
		name = L["SellPrizes"],
		desc = L["SellPrizes"],
		args = {
			show = {
				type = "toggle",
				name = L["Show sell prizes"],
				desc = L["Selects whether the sell prizes should be shown in the GameTooltip."],
				get = _GetValue,
				set = _SetValue,
				passValue = "showPrizes",
				order = 1,
			},
			reset = {
				type = "execute",
				name = L["Reset database"],
				desc = L["Resets the database of sell prizes for the current account. Will not delete the predelivered default database."],
				func = function() StaticPopup_Show("SELL_PRIZES_DB_RESET") end,
				order = 2,
			},
		}
	}
end

--[[
	Initializes the safety dialog for the database reset.
--]]
local function _CreateResetDialog()
	StaticPopupDialogs["SELL_PRIZES_DB_RESET"] = {
		text = L["Do you really want to reset the database?"];
	  	button1 = L["Yes"],
	  	button2 = L["No"],
	  	OnAccept = function()
	  		_ResetDb()
	  	end,
	  	timeout = 0,
	  	whileDead = 1,
	  	hideOnEscape = 1
	};	
end

--[[
	Initializes the addon.
--]]
function vendor.SellPrizes:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("SellPrizes");
	vendor.Vendor:RegisterDefaults("SellPrizes", "account", {
		prizes = {},
		version = SELL_PRIZES_VERSION		
	});
	vendor.Vendor:RegisterDefaults("SellPrizes", "profile", {
		showPrizes = true,
	});
	_CreateResetDialog();
	_CreateOptions();
end

--[[
	Initializes the module.
--]] 
function vendor.SellPrizes:OnEnable()
	self.merchantVisible = false;   
   	self:RegisterEvent("MERCHANT_SHOW");
   	self:RegisterEvent("MERCHANT_CLOSED");
	self:SecureHook("SetTooltipMoney");
	vendor.TooltipHook:AddAppender(self, 10);
end

--[[
	Merchant frame has been opened.
--]]
function vendor.SellPrizes:MERCHANT_SHOW() 
   	self.merchantVisible = true;
	-- update sell prizes, if needed
	for bag = 0, 4 do
		for slot = 1, GetContainerNumSlots(bag) do
			itemLink = GetContainerItemLink(bag, slot);
			if (itemLink) then
				local itemId = _ExtractItemId(itemLink);
				local sellPrize = itemPrice:GetPriceById(itemId);
				if (not sellPrize or sellPrize <= 0) then
					-- show slot to read sell value from tooltip
					self.copper = nil;
					GameTooltip:SetBagItem(bag, slot);
					if (self.copper and self.copper > 0) then
						local _, quantity = GetContainerItemInfo(bag, slot);
						self.db.account.prizes[itemId] = self.copper / quantity;
						self.copper = nil
					end
				end
			end
		end
	end	
end

--[[
	Merchant frame has been closed.
--]]
function vendor.SellPrizes:MERCHANT_CLOSED() 
   	self.merchantVisible = false;
end

--[[
	Hooks corresponding function to remember sell value at merchant.
--]]  
function vendor.SellPrizes:SetTooltipMoney(tooltip, copper)
   	self.copper = copper;
end

--[[
	Returns the sell prize for the given item, or nil.
--]]
function vendor.SellPrizes:GetSellPrice(itemLink)
	local itemId = _ExtractItemId(itemLink);
	-- look for sell prize in ItemPrice
	local sellPrize = itemPrice:GetPriceById(itemId);
	if (not sellPrize) then
		-- check own db
		sellPrize = self.db.account.prizes[itemId];
	end
	return sellPrize;
end

--[[
	Callback for Tooltip integration
--]]
function vendor.SellPrizes:AppendToGameTooltip(tooltip, itemLink, itemName, count)
	if (not self.merchantVisible and self.db.profile.showPrizes) then
		local sellPrize = self:GetSellPrice(itemLink);
		if (sellPrize and sellPrize > 0) then
			local msg1, msg2;
			if (count > 1) then
				msg1 = L["Sales price (%d)"]:format(count);
				msg2 = vendor.Format.FormatMoney(sellPrize, true).." ("..vendor.Format.FormatMoney(count * sellPrize, true)..")";
			else
				msg1 = L["Sales price"];
				msg2 = vendor.Format.FormatMoney(sellPrize, true);
			end
			tooltip:AddDoubleLine(msg1, msg2);
		end
   	end
end
