--[[
	Scans the auction house with configurable requests.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]
   
vendor.Scanner = vendor.Vendor:NewModule("Scanner", "AceEvent-2.0", "AceHook-2.1");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local self = vendor.Scanner;
local SCANNER_VERSION = 6;
local QUALITY_INDEX = {
	[0] = vendor.Format.ColorizeQuality(L["Poor"], 0), 
	[1] = vendor.Format.ColorizeQuality(L["Common"], 1),
	[2] = vendor.Format.ColorizeQuality(L["Uncommon"], 2),
	[3] = vendor.Format.ColorizeQuality(L["Rare"], 3),
	[4] = vendor.Format.ColorizeQuality(L["Epic"], 4),
	[5] = vendor.Format.ColorizeQuality(L["Legendary"], 5),
	[6] = vendor.Format.ColorizeQuality(L["Artifact"], 6),
} 
local MIN_QUALITIES = {
	[QUALITY_INDEX[0]] = 0,
	[QUALITY_INDEX[1]] = 1,
	[QUALITY_INDEX[2]] = 2,
	[QUALITY_INDEX[3]] = 3,
	[QUALITY_INDEX[4]] = 4,
	[QUALITY_INDEX[5]] = 5,
	[QUALITY_INDEX[6]] = 6,
}

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
	self:ApplySettings();
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Sorts the scnasnaphot listeners according their order.
--]]
local function _SortScanSnapshotListeners(a, b)
	return a.order < b.order;
end

--[[
	Initializes the snapshots for the faction and neutral auction houses.
--]]
local function _InitSnapshots()
	self.snapshots = {
		[true] = vendor.ScanSnapshot:new(self.db.server.snapshot),
		[false] = vendor.ScanSnapshot:new(self.db.realm.snapshot)
	};
end

--[[
	Resets the database for the current realm/server.
--]]
local function _ResetDb()
	vendor.Vendor:ResetDB("Scanner", "realm");
	vendor.Vendor:ResetDB("Scanner", "server");
	vendor.Vendor:Print(L["Database of scan snapshots for current realm and server where reset."]);
	_InitSnapshots();
end

--[[
	Initializes the safety dialog for the database reset.
--]]
local function _CreateResetDialog()
	StaticPopupDialogs["SCANNER_DB_RESET"] = {
		text = L["Do you really want to reset the database?"];
	  	button1 = L["Yes"],
	  	button2 = L["No"],
	  	OnAccept = function()
	  		_ResetDb()
	  	end,
	  	timeout = 0,
	  	whileDead = 1,
	  	hideOnEscape = 1
	};	
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.Scanner = {
		type = "group",
		name = L["Scanner"],
		desc = L["Scanner"],
		args = {
			minQuality = {
				type = "text",
				name = L["Minimum quality"],
				desc = L["Selects the minimum quality for items to be scanned in the auction house."],
				get = _GetValue,
				set = _SetValue,
				validate = {QUALITY_INDEX[0], QUALITY_INDEX[1], QUALITY_INDEX[2], QUALITY_INDEX[3], QUALITY_INDEX[4], QUALITY_INDEX[5], QUALITY_INDEX[6]},
				passValue = "minQuality",
				order = 1,
			},
			reset = {
				type = "execute",
				name = L["Reset database"],
				desc = L["Resets the database of scan snapshots for the current realm and server."],
				func = function() StaticPopup_Show("SCANNER_DB_RESET") end,
				order = 2,
			},			
		}
	}
end

--[[
	Informs any listener about the update of the given snapshot.
--]]
local function _NotifySnapshotUpdate(self, snapshot, isNeutral)
	if (self.scanResultListeners) then
		for _, listener in ipairs(self.scanResultListeners) do
			assert(listener.listener.ScanSnapshotUpdated);
			vendor.Vendor:Debug("inform listener");
			listener.listener:ScanSnapshotUpdated(snapshot, isNeutral);
		end
	end
	vendor.Vendor:Debug("leave");
end

--[[
	Cleanup code at the end of a scan.
--]]
local function _StopScan(self)
	self.scanTask = nil;
	--self.scanDialog:ShowCloseButton();
	self.scanDialog:Hide();
end

--[[
	Initializes the module.
--]]
function vendor.Scanner:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("Scanner");
	vendor.Vendor:RegisterDefaults("Scanner", "realm", {
		version = SCANNER_VERSION,
		snapshot = {},
	});
	vendor.Vendor:RegisterDefaults("Scanner", "server", {
		version = SCANNER_VERSION,
		snapshot = {}		
	});
	vendor.Vendor:RegisterDefaults("Scanner", "profile", {
		minQuality = L["Common"],
	});
	_CreateResetDialog();
	_CreateOptions();	
	self.lru = vendor.LruCache:new(25);
end

--[[
	Initializes the scanner at startup and registers for the needed events.
--]]
function vendor.Scanner:OnEnable()
	self.scanDialog = vendor.ScanDialog:new();
	_InitSnapshots();
	self:RegisterEvent("AUCTION_ITEM_LIST_UPDATE");
	self:RegisterEvent("AUCTION_HOUSE_CLOSED");
	self:Hook("AuctionFrameBrowse_OnEvent", true);
	self:Hook("CanSendAuctionQuery", true);
end

--[[
	Performs a scan for the given item. nil will trigger a complete scan.
	@param name the name pattern or "" for a complete scan.
--]]
function vendor.Scanner:Scan(itemLink)
	if (self.scanTask) then
		error(L["There is already a running scan."]);
	else
		self.scanTask = vendor.ScanTask:new(itemLink, self.minQuality);
		vendor.TaskQueue:AddTask(self.scanTask);
	end	
end

--[[
	Hides the auction house browse buttons, used during scans.
--]]
function vendor.Scanner:HideAuctionHouseBrowseButtons()
   	for i=1, NUM_BROWSE_TO_DISPLAY do
      	local button = getglobal("BrowseButton"..i);
      	button:Hide();
   	end
   	BrowsePrevPageButton:Hide();
   	BrowseNextPageButton:Hide();
   	BrowseSearchCountText:Hide();
end

--[[
	Shows some status information for the scan on the list tab of the auction house.
--]]
function vendor.Scanner:ShowScanStatus(msg)
	self.scanDialog:ShowProgress(msg);
end

--[[
	Events triggered if the auction item list was updated. Will
	inform any current ScanTask about it.
--]]
function vendor.Scanner:AUCTION_ITEM_LIST_UPDATE()
	if (self.scanTask) then
		self.scanTask:AuctionListUpdate();
	end
end

--[[
	Events for auction house closes. Will cancel any running scan task.
--]]
function vendor.Scanner:AUCTION_HOUSE_CLOSED()
	if (self.scanTask) then
		self.scanTask:Cancel();
	end
	self.scanDialog:Hide();
end

--[[
	Hook to prevent blizzard seeing the auctions while we are scanning.
--]]
function vendor.Scanner:AuctionFrameBrowse_OnEvent()
	if (not self.scanTask) then
		self.hooks.AuctionFrameBrowse_OnEvent();
   	end
end

--[[
	Hook to prevent blizzard to flicker the "Search" button.
--]]
function vendor.Scanner:CanSendAuctionQuery()
	if (not self.scanTask) then
		return self.hooks.CanSendAuctionQuery();
   	end
	return nil;
end

--[[
	Returns the result of "CandSendQuery" bypassing the own hook.
--]]
function vendor.Scanner:MaySendAuctionQuery()
	return self.hooks.CanSendAuctionQuery();
end

--[[
	Returns the most current ScanSet for the specified item.
	@param isNeutral has to be set to true, for neutral auction houses.
	@param itemLink the item to be selected.
--]]
function vendor.Scanner:GetScanSet(isNeutral, itemLink)
	local key = vendor.Items:GetItemLinkKey(itemLink).."-"..vendor.Format.BoolToStr(isNeutral);
	local snap = self.lru:Get(key);
	if (not snap) then
		local snapshot = self.snapshots[isNeutral]; 
		snap = snapshot:GetScanSet(isNeutral, itemLink); 
		self.lru:Put(key, snap);
	end
	return snap;
end

--[[
	Registers a listener for updates for the scan snapshots. The order sorts the listeners.
	Small values come first. nil is interpreted as 1000.
	
	The listener has to implement the method "ScanSnapshotUpdated", with the
	following synposis:
	@param scanSnapshot the updated ScanSnapshot.
	@param isNeutral determines whether the snapshot was for a neutral auction house.
--]] 
function vendor.Scanner:AddScanSnapshotListener(listener, order)
	if (not self.scanResultListeners) then
		self.scanResultListeners = {};
	end
	table.insert(self.scanResultListeners, {listener = listener, order = order or 1000});
	table.sort(self.scanResultListeners, _SortScanSnapshotListeners);
end

--[[
	Adds a listener for items found during auction house scans.
	The listener has to implement the method "ItemScanned" with the following synopsis:
	@param index the current auction house index
	@param itemLink the link of the scanned item.
	@param itemName the name of the item.
--]]
function vendor.Scanner:AddScanItemListener(listener)
	if (not self.scanItemListeners) then
		self.scanItemListeners = {};
	end
	table.insert(self.scanItemListeners, listener);
end

--[[
	Informs the listeners about the given scanned item.
--]]
function vendor.Scanner:NotifyItemScanned(index, itemLink, itemName, count, bid, buyout)
	if (self.scanItemListeners) then
		for _, listener in ipairs(self.scanItemListeners) do
			local askForBuy, reason = listener:ItemScanned(index, itemLink, itemName);
			if (askForBuy) then
				vendor.Sniper:StartPendingBuy(itemName, count, bid, buyout);
				self.scanDialog:AskToBuy(itemLink, count, bid, buyout, reason, index);
				vendor.Sniper:WaitForPendingBuy();
			end
		end		
	end
end

--[[
	Stops the current scan, if any.
--]]
function vendor.Scanner:StopScan()
	if (self.scanTask) then
		self.scanTask:Cancel();
	end	
end

--[[
	Returns true, if the scanner is currently performing a scan.
--]]
function vendor.Scanner:IsScanning()
	return self.scanTask ~= nil;
end

--[[
	Sets the given complete snapshot produced by a scan task.
	Will assume that the current scan task is finished now.
--]]
function vendor.Scanner:SetSnapshot(snapshot, isNeutral)
	self.lru:Clear();
	self.snapshots[isNeutral] = snapshot;
	if (isNeutral) then
		self.db.server.snapshot = snapshot:GetData();
	else
		self.db.realm.snapshot = snapshot:GetData();
	end
	_NotifySnapshotUpdate(self, snapshot, isNeutral);
	_StopScan(self);

end

--[[
	Sets the given complete item snapshot produced by a scan task.
	Will assume that the current scan task is finished now.
--]]
function vendor.Scanner:SetItemSnapshot(itemLink, snapshot, isNeutral)
	local key = vendor.Items:GetItemLinkKey(itemLink).."-"..vendor.Format.BoolToStr(isNeutral);
	self.lru:Remove(key);
	local snap = self.snapshots[isNeutral];
	snap:DeleteResults(itemLink);
	snap:InsertResults(snapshot);
	_NotifySnapshotUpdate(self, snapshot, isNeutral);
	_StopScan(self);
end

--[[
	Lets the scan task inform the scanner, that it has abandoned the scan.
--]]
function vendor.Scanner:AbandonScan()
	_StopScan(self);
end

--[[
	Applys the possibly changed settings.
--]]
function vendor.Scanner:ApplySettings()
	self.minQuality = MIN_QUALITIES[self.db.profile.minQuality] or 1;
end
