--[[
	Immutable list of auctions which may be sorted in several ways.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

local AceOO = AceLibrary("AceOO-2.0");
vendor.ScanSet = AceOO.Class();

-- the several sort functions
local SORT_FUNCS = {
	["count"] = {
		[true] = function(a, b) 
			local _, _, _, acount = vendor.ScanResults.Unpack(a);
			local _, _, _, bcount = vendor.ScanResults.Unpack(b);
			return acount < bcount;
		end,
		[false] = function(a, b)
			local _, _, _, acount = vendor.ScanResults.Unpack(a);
			local _, _, _, bcount = vendor.ScanResults.Unpack(b);
			return acount > bcount;
		end
	},
	["owner"] = {
		[true] = function(a, b)
				local _, _, _, _, _, _, _, _, aowner = vendor.ScanResults.Unpack(a);
				local _, _, _, _, _, _, _, _, bowner = vendor.ScanResults.Unpack(b);
				return aowner < bowner;
			end,
		[false] = function(a, b)
			local _, _, _, _, _, _, _, _, aowner = vendor.ScanResults.Unpack(a);
			local _, _, _, _, _, _, _, _, bowner = vendor.ScanResults.Unpack(b);
			return aowner > bowner;
		end
	},
	["bidPerItem"] = {
		[true] = function(a, b)
				local _, _, _, acount, aminBid, _, _, abidAmount = vendor.ScanResults.Unpack(a);
				local _, _, _, bcount, bminBid, _, _, bbidAmount = vendor.ScanResults.Unpack(b);
				return (math.max(aminBid, abidAmount) / acount) < (math.max(bminBid, bbidAmount) / bcount);
			end,
		[false] = function(a, b)
				local _, _, _, acount, aminBid, _, _, abidAmount = vendor.ScanResults.Unpack(a);
				local _, _, _, bcount, bminBid, _, _, bbidAmount = vendor.ScanResults.Unpack(b);
				return (math.max(aminBid, abidAmount) / acount) > (math.max(bminBid, bbidAmount) / bcount);
			end
	},
	["buyoutPerItem"] = {
		[true] = function(a, b)
				local _, _, _, acount, _, _, abuyoutPrice = vendor.ScanResults.Unpack(a);
				local _, _, _, bcount, _, _, bbuyoutPrice = vendor.ScanResults.Unpack(b);
				return (abuyoutPrice / acount) < (bbuyoutPrice / bcount);
			end,
		[false] = function(a, b)
					local _, _, _, acount, _, _, abuyoutPrice = vendor.ScanResults.Unpack(a);
					local _, _, _, bcount, _, _, bbuyoutPrice = vendor.ScanResults.Unpack(b);
					return (abuyoutPrice / acount) > (bbuyoutPrice / bcount);
				end
	}
};

--[[
	Checks whether the list has to be sorted and does it in case.
--]]
local function _EnsureSorting(self)
	if (not self.sorted) then
		table.sort(self.index, self.sortFunc);
	end
	self.sorted = true;
end

--[[ 
	Creates a ScanSet instance with the given list of auctions.
--]]
function vendor.ScanSet.prototype:init(index, isNeutral, itemLink)
	vendor.ScanSet.super.prototype.init(self)
	self.index = index;
	self.sortFunc = SORT_FUNCS["owner"][true];
	self.isNeutral = isNeutral;
	self.itemLink = itemLink;
	-- search for eldest entry
	self.eldestEntry = nil;
	for i,v in pairs(self.index) do
		local key, time = vendor.ScanResults.Unpack(v);
		if (not self.eldestEntry or time < self.eldestEntry) then
			self.eldestEntry = time;
		end
	end
end

--[[
	Returns the total number of entries.
--]]
function vendor.ScanSet.prototype:Size()
	return table.getn(self.index);
end

--[[ 
	Returns the n'th entry starting from 1.
	@return itemLinkKey, time, timeLeft, count, minBid, minIncrement, buyoutPrice, bidAmount, owner
--]]
function vendor.ScanSet.prototype:Get(n)
	_EnsureSorting(self);
	local data = self.index[n];
	return vendor.ScanResults.Unpack(data);
end

--[[
	Selects the attribute for sorting, possible values are:
	"count", "owner", "bidPerItem" and "buyoutPerItem".
	Should never be called for the original snapshot, only 
	for copies of it! 
--]]
function vendor.ScanSet.prototype:Sort(sortType, ascending)
	self.sortFunc = SORT_FUNCS[sortType][ascending];
	self.sorted = false;
end

--[[
	Returns whether the auction set is for a neutral auction house.
--]] 
function vendor.ScanSet.prototype:IsNeutral()
	return self.isNeutral;
end

--[[
	Returns the selected item for the auctions.
--]]
function vendor.ScanSet.prototype:GetItemLink()
	return self.itemLink;
end

--[[
	Returns the up-to-dateness of the scan set. This is the age of the eldest
	entry in seconds or nil if no entries.
--]]
function vendor.ScanSet.prototype:GetUpToDateness()
	if (self.eldestEntry) then
		return time() - self.eldestEntry;
	end
	return nil;
end

