--[[
	Plugable frame for displaying ScanResults.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

local AceOO = AceLibrary("AceOO-2.0");
vendor.ScanResultFrame = AceOO.Class();

--[[
	Callback for updating the browse scrollbar.
--]]
function _BrowseUpdate(selfIn)
	local self = selfIn or this.ctrl;
	if (not self or not self.scanResult) then
		FauxScrollFrame_Update(self.browseScroll, 0, 1, AUCTIONS_BUTTON_HEIGHT);
		return;
	end
	-- display wheter we have results
	local n = self.scanResult:Size();
	if (n == 0) then
		self.noResultsText:Show();
	else
		self.noResultsText:Hide();
	end

	-- traverse the items to be displayed
	local offset = FauxScrollFrame_GetOffset(self.browseScroll);	
	for i=1, self.numItems do
		local index = offset + i;
		local button = self.browseButs[i];
		-- Show or hide auction buttons
		if (index > n) then
			button:Hide();
		else
			button:Show();
			local buttonName = button:GetName();
			local itemLinkKey, time, timeLeft, count, minBid, minIncrement, buyoutPrice, bidAmount, owner = self.scanResult:Get(index);
			assert(bidAmount);
			assert(buyoutPrice);
			local itemLink = vendor.Items:GetItemLink(itemLinkKey);
			local name, _, rarity, level, minLevel, itemType, subType, stackCount, equipLoc, texture = GetItemInfo(itemLink);
			assert(name); -- TODO handle nil with subfunction
			local canUse = IsUsableItem(itemLink);
			-- Resize according to scroller presence resp. absence
			local buttonHighlight = getglobal(buttonName.."Highlight");
			local buyoutMoneyFrame = getglobal(buttonName.."BuyoutMoneyFrame");
			if (n <= self.numItems) then
				getglobal(buttonName.."BuyoutText"):SetPoint("LEFT", buyoutMoneyFrame, "RIGHT", BUYOUT_TEXT_OFFSET_NO_SCROLL, 0)
				button:SetWidth(602);
				buttonHighlight:SetWidth(567);
			else
				getglobal(button:GetName().."BuyoutText"):SetPoint("LEFT", buyoutMoneyFrame, "RIGHT", BUYOUT_TEXT_OFFSET_WITH_SCROLL, 0)
				button:SetWidth(577);
				buttonHighlight:SetWidth(542);
			end
			-- remmeber item link
			local itemBut = getglobal(buttonName.."Item");
			itemBut.itemLink = itemLink;
			itemBut.itemCount = count;
			-- Set name and quality color
			local color = ITEM_QUALITY_COLORS[rarity];
			local itemName = getglobal(buttonName.."Name");
			itemName:SetText(name);
			itemName:SetVertexColor(color.r, color.g, color.b);
			-- Set level
			if (level > UnitLevel("player")) then
				getglobal(buttonName.."Level"):SetText(RED_FONT_COLOR_CODE..level..FONT_COLOR_CODE_CLOSE);
			else
				getglobal(buttonName.."Level"):SetText(level);
			end
			-- Set closing time
			getglobal(buttonName.."ClosingTimeText"):SetText(AuctionFrame_GetTimeLeftText(timeLeft));
			getglobal(buttonName.."ClosingTime").tooltip = AuctionFrame_GetTimeLeftTooltipText(timeLeft);
			-- Set item texture, count, and usability
			iconTexture = getglobal(buttonName.."ItemIconTexture");
			iconTexture:SetTexture(texture);
			if (not canUse) then
				iconTexture:SetVertexColor(1.0, 0.1, 0.1);
			else
				iconTexture:SetVertexColor(1.0, 1.0, 1.0);
			end
			itemCount = getglobal(buttonName.."ItemCount");
			if (count > 1) then
				itemCount:SetText(count);
				itemCount:Show();
			else
				itemCount:Hide();
			end
			-- Set high bid
			local moneyFrame = getglobal(buttonName.."MoneyFrame");
			local yourBidText = getglobal(buttonName.."YourBidText");
			local buyoutText = getglobal(buttonName.."BuyoutText");
			-- If not bidAmount set the bid amount to the min bid
			local displayedPrice;
			if (bidAmount == 0) then
				displayedPrice = minBid;
				requiredBid = minBid;
			else
				displayedPrice = bidAmount;
				requiredBid = bidAmount + minIncrement ;
			end
			MoneyFrame_Update(moneyFrame:GetName(), displayedPrice);
			
			local highBidder = nil; -- TODO add it to scan results
			if (highBidder) then
				yourBidText:Show();
			else
				yourBidText:Hide();
			end
				
			if (requiredBid >= MAXIMUM_BID_PRICE) then
				-- Lie about our buyout price
				buyoutPrice = requiredBid;
			end
			if (buyoutPrice > 0) then
				moneyFrame:SetPoint("RIGHT", buttonName, "RIGHT", 10, 10);
				MoneyFrame_Update(buyoutMoneyFrame:GetName(), buyoutPrice);
				buyoutMoneyFrame:Show();
				buyoutText:Show();
			else
				moneyFrame:SetPoint("RIGHT", buttonName, "RIGHT", 10, 3);
				buyoutMoneyFrame:Hide();
				buyoutText:Hide();
			end
			getglobal(buttonName.."HighBidder"):SetText(owner);
		end
	end
	-- Update scrollFrame
	FauxScrollFrame_Update(self.browseScroll, n, self.numItems, AUCTIONS_BUTTON_HEIGHT);	
end

--[[
	Show tooltip for item under mouse. The itemLink is attached to the button (this).
--]]
local function _BrowseItemOnEnter(self)
	if (this.itemLink) then
		GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
		GameTooltip:SetHyperlink(this.itemLink);
		GameTooltip.itemCount = this.itemCount; -- TooltipHook checks for it
	end
end

--[[ 
	Creates a new instance of the frame.
	@param name the unique name of the frame.
	@param parent the parent frame.
	@param numItems number of visible items.
--]]
function vendor.ScanResultFrame.prototype:init(name, parent, numItems)
	vendor.ScanResultFrame.super.prototype.init(self)
	self.numItems = numItems;
	self.scanResult = nil;
	self.browseButs = {};
	self.frame = CreateFrame("Frame", name, parent);
	self.frame:SetWidth(658);
	self.frame:SetHeight(347);	
	for i=1, numItems do
		local but = CreateFrame("Button", name.."BrowseButton"..i, self.frame, "BrowseButtonTemplate");
		but:SetID(i);
		if (i == 1) then
			but:SetPoint("TOPLEFT", 195, -110);
		else
			but:SetPoint("TOPLEFT", name.."BrowseButton"..(i-1), "BOTTOMLEFT", 0, 0);
		end
		-- shift the level column a bit to the left
		local shiftFrame = getglobal(but:GetName().."Level")
		shiftFrame:SetPoint("TOPLEFT", 190, 0);		
		-- shift the duration column a bit to the left
		shiftFrame = getglobal(but:GetName().."ClosingTime")
		shiftFrame:SetPoint("TOPLEFT", 235, 0);
		-- shift the owner column a bit to the left
		shiftFrame = getglobal(but:GetName().."HighBidder")
		shiftFrame:SetPoint("TOPLEFT", 305, 0);		
		-- set scripts for the item
		local item = getglobal(but:GetName().."Item");
		item:SetScript("OnEnter", function()
			this:GetParent():LockHighlight();
			_BrowseItemOnEnter(self); 
		end);
		item:SetScript("OnLeave", function()
			this:GetParent():UnlockHighlight();
			GameTooltip:Hide();
			ResetCursor();
		end);
		item:SetScript("OnClick", function()
--			if (IsModifiedClick()) then
--				HandleModifiedItemClick(GetAuctionItemLink("list", this:GetParent():GetID() + FauxScrollFrame_GetOffset(BrowseScrollFrame)));
--				else
--					BrowseButton_OnClick(this:GetParent());
--				end
		end);
		item:SetScript("OnUpdate", function()
			if (GameTooltip:IsOwned(this)) then
				_BrowseItemOnEnter(self);
			end
		end);
		but:Hide();
		self.browseButs[i] = but;
	end
	
	-- no results text
	self.noResultsText = self.frame:CreateFontString(name.."NoResultsText", "ARTWORK", "GameFontNormal");
	self.noResultsText:SetText(L["Found no auctions, press \"Refresh\" to update the list."]);
	self.noResultsText:SetPoint("TOP", 115, -120);
	self.noResultsText:Hide();
	
	-- scrollframe
	self.browseScroll = CreateFrame("ScrollFrame", name.."BrowseScrollFrame", parent, "FauxScrollFrameTemplate");
	self.browseScroll.ctrl = self;
	self.browseScroll:SetWidth(465);
	self.browseScroll:SetHeight(333);
	self.browseScroll:SetPoint("TOPRIGHT", 39, -78);
	self.browseScroll:SetScript("OnVerticalScroll", function() FauxScrollFrame_OnVerticalScroll(AUCTIONS_BUTTON_HEIGHT, _BrowseUpdate); end);
	self.browseScroll:SetScript("OnShow", function() _BrowseUpdate(self); end);
	
	local t = self.browseScroll:CreateTexture(nil, "ARTWORK");
	t:SetTexture("Interface\\PaperDollInfoFrame\\UI-Character-ScrollBar");
	t:SetWidth(31);
	t:SetHeight(256);
	t:SetPoint("TOPLEFT", self.browseScroll, "TOPRIGHT", -2, 5);
	t:SetTexCoord(0, 0.484375, 0, 1.0);

	t = self.browseScroll:CreateTexture(nil, "ARTWORK");
	t:SetTexture("Interface\\PaperDollInfoFrame\\UI-Character-ScrollBar");
	t:SetWidth(31);
	t:SetHeight(106);
	t:SetPoint("BOTTOMLEFT", self.browseScroll, "BOTTOMRIGHT", -2, -2);
	t:SetTexCoord(0.515625, 1.0, 0, 0.4140625);
	
	-- register for new ScanResults
	vendor.Scanner:AddScanResultListener(self);
end

--[[
	Selects a new ScanResult to be displayed by this frame.
	Nothing will be displayed if nil is set.
--]]
function vendor.ScanResultFrame.prototype:Display(scanResult)
	self.scanResult = scanResult;
	if (self.scanResult) then
		self.scanResult:SortPrice();
	end
	_BrowseUpdate(self);
end

--[[
	Update the currently displayed result list.
--]]
function vendor.ScanResultFrame.prototype:Update()
	_BrowseUpdate(self);
end

--[[
	Sets the position of the frame.
--]]
function vendor.ScanResultFrame.prototype:SetPoint(...)
	self.frame:SetPoint(...);
end

--[[
	ScanResultListener interface method for new or updated
	Scanresults.
--]]
function vendor.ScanResultFrame.prototype:ScanResultUpdated(scanResult)
	if (self.scanResult and scanResult:IsFinished()) then
		if (scanResult:IsComplete()) then
			if (self.scanResult:IsComplete()) then
				self.scanResult = scanResult;
			else
				self.scanResult = vendor.ScanResult:new(scanResult:GetData(), self.scanResult:IsNeutral(), self.scanResult:GetItemLink());
			end
		elseif (scanResult:GetName() == self.scanResult:GetName()) then
			self.scanResult = scanResult;
		end
	end
	if (self.scanResult) then
		self.scanResult:SortPrice();
	end
	_BrowseUpdate(self);
end