--[[
	Informs the user about the scan process or asks to buy items.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

local AceOO = AceLibrary("AceOO-2.0");
vendor.ScanDialog = AceOO.Class();
local L = AceLibrary("AceLocale-2.2"):new("Vendor");

--[[
	Wait until the user made a decison.
--]]
local function _WaitForDecision(self)
	while (not self.decisionMade) do
		coroutine.yield(); -- continue waiting
	end	
end

--[[
	Create the subframe for buying items.
--]]
local function _InitBuyFrame(self)
	local frame = CreateFrame("Frame", nil, self.frame);
	frame:SetWidth(320);
	frame:SetHeight(180);
	frame:SetPoint("TOPLEFT");
	self.buyFrame = frame;
	
	-- create the item icon
	local frameName = self.frame:GetName();
	local itemIcon = CreateFrame("Button", frameName.."ItemIcon", frame);
	itemIcon:SetWidth(50);
	itemIcon:SetHeight(50);
	itemIcon:SetPoint("TOPLEFT", 10, -20);
	itemIcon.ctrl = self;
	itemIcon:SetScript("OnEnter", function()
		local self = this.ctrl;
		if (self.itemLink) then 
			GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
			GameTooltip.itemCount = self.count;
			GameTooltip:SetHyperlink(self.itemLink);
		end
	end);
	itemIcon:SetScript("OnLeave", function() GameTooltip:Hide(); end);
	itemIcon:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square");
	self.itemIcon = itemIcon;
		
	-- and the item name
	local itemName = itemIcon:CreateFontString(frameName.."ItemName", "BACKGROUND", "GameFontNormalLarge");
	itemName:SetWidth(240);
	itemName:SetHeight(30);
	itemName:SetPoint("LEFT", itemIcon, "RIGHT", 5, 0);
	itemName:SetJustifyH("LEFT");
	self.itemName = itemName;
	
	-- and the item count
	local itemCount = itemIcon:CreateFontString(frameName.."ItemCount", "OVERLAY", "NumberFontNormal");
	itemCount:SetJustifyH("RIGHT");
	itemCount:SetPoint("BOTTOMRIGHT", -5, 2);
	self.itemCount = itemCount;
	
	local bidLabel = frame:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall");
	bidLabel:SetText(L["Bid"]);
	bidLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", 10, -76);
	
	local bt = frame:CreateFontString(nil, "ARTWORK", "GameFontNormal");
	bt:SetPoint("LEFT", bidLabel, "RIGHT", 10, 0);
	self.bidBt = bt;
	
	local buyoutLabel = frame:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall");
	buyoutLabel:SetText(L["Buyout"]);
	buyoutLabel:SetPoint("LEFT", bt, "RIGHT", 10, 0);
	
	local bt = frame:CreateFontString(nil, "ARTWORK", "GameFontNormal");
	bt:SetPoint("LEFT", buyoutLabel, "RIGHT", 10, 0);
	self.buyoutBt = bt;
	
	local reason = frame:CreateFontString(nil, "ARTWORK", "GameFontGreen");
	reason:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", 10, 69);
	self.reasonTxt = reason;

	bt = vendor.GuiTools.CreateButton(frame, "UIPanelButtonTemplate", 86, 22, L["Pause"], L["Pauses any snipes for the current scan."])
	bt:SetPoint("BOTTOMLEFT", 10, 40);
	bt.ctrl = self;
	bt:SetScript("OnClick", function()
		local self = this.ctrl; 
		vendor.Sniper:PauseThisScan();
		vendor.Sniper:StopPendingBuy();
		self.decisionMade = true;		
	end);

	bt = vendor.GuiTools.CreateButton(frame, "UIPanelButtonTemplate", 86, 22, L["Snipe"], L["Opens the snipe dialog for this item."])
	bt:SetPoint("BOTTOM", 0, 40);
	bt.ctrl = self;
	bt:SetScript("OnClick", function()
		local self = this.ctrl; 
		vendor.Sniper:OpenSnipeDialog(self.name); 
	end);
			
	bt = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	bt:SetText(L["Bid"]);
	bt:SetWidth(86);
	bt:SetHeight(22);
	bt:SetPoint("BOTTOMLEFT", 10, 10);
	bt.ctrl = self;
	bt:SetScript("OnClick", function() 
		local self = this.ctrl;
		vendor.Sniper:SnipeForItem("list", self.index, self.bid, 0, self.name);
		self.decisionMade = true; 
	end);
	
	bt = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	bt.ctrl = self;
	bt:SetText(L["Buyout"]);
	bt:SetWidth(86);
	bt:SetHeight(22);
	bt:SetPoint("BOTTOM", 0, 10);
	bt.ctrl = self;
	bt:SetScript("OnClick", function() 
		local self = this.ctrl;
		vendor.Sniper:SnipeForItem("list", self.index, 0, self.buyout, self.name);
		self.decisionMade = true; 
	end);
	self.doBuyoutBt = bt;
	
	bt = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	bt:SetText(L["Continue"]);
	bt:SetWidth(86);
	bt:SetHeight(22);
	bt:SetPoint("BOTTOMRIGHT", -10, 10);
	bt.ctrl = self;
	bt:SetScript("OnClick", function() 
		local self = this.ctrl;
		vendor.Sniper:StopPendingBuy();
		self.decisionMade = true; 
	end);
end

--[[
	Creates the subframe for the scan statistics.
--]]
local function _InitScanFrame(self)
	local frame = CreateFrame("Frame", nil, self.frame);
	frame:SetWidth(320);
	frame:SetHeight(180);
	frame:SetPoint("TOPLEFT");
	self.scanFrame = frame;
	
	local txt = frame:CreateFontString(nil, "ARTWORK", "GameFontNormal");
	txt:SetPoint("TOP", 0, -35);
	self.scanTxt = txt;
	
	bt = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	bt:SetText(L["Stop"]);
	bt:SetWidth(86);
	bt:SetHeight(22);
	bt:SetPoint("BOTTOM", 0, 10);
	bt:SetScript("OnClick", function() vendor.Scanner:StopScan(); end);
	self.scanBt = bt;
	
	bt = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate");
	bt:SetText(L["Close"]);
	bt:SetWidth(86);
	bt:SetHeight(22);
	bt:SetPoint("BOTTOM", self.scanBt, "TOPLEFT", 0, 0);
	bt.ctrl = self;
	bt:SetScript("OnClick", function() self.frame:Hide(); end);
	self.closeBt = bt;	
end

--[[
	Initilaizes the frame.
--]]
local function _InitFrame(self)
	-- create the frame
	local frameName = "VendorScanDialog";
	local frame = CreateFrame("Frame", frameName, UIParent, "VendorDialogTemplate");
	frame.ctrl = self;
	self.frame = frame;
	frame:SetWidth(320);
	frame:SetHeight(180);
	frame:SetPoint("CENTER");
	frame:SetFrameStrata("DIALOG");
	frame:SetMovable(true);
	frame:EnableMouse(true);
	frame:SetToplevel(true);
	frame:SetClampedToScreen(true);
	frame:SetScript("OnMouseDown", function() this:StartMoving() end);
	frame:SetScript("OnMouseUp", function() this:StopMovingOrSizing() end);
				
	-- create the title string
	local text = frame:CreateFontString(nil, "OVERLAY")
	text:SetPoint("TOP", frame, "TOP", 0, -10)
	text:SetFontObject("GameFontHighlightLarge")
	text:SetText(L["Auction Scan"])
	
	_InitBuyFrame(self);
	_InitScanFrame(self);
	self.scanFrame:Show();
	self.buyFrame:Hide();	
end

--[[ 
	Creates a new dialog instance.
--]]
function vendor.ScanDialog.prototype:init()
	vendor.ScanDialog.super.prototype.init(self)
	_InitFrame(self);
	self.frame:Hide();
end

--[[
	Asks the user to buy or bid the given item and performs it.
--]]
function vendor.ScanDialog.prototype:AskToBuy(itemLink, count, bid, buyout, reason, index)
	local isShown = self.frame:IsShown();
	self.scanFrame:Hide();
	self.buyFrame:Show();
	self.doBuyoutBt:Disable();
	self.bid = bid;
	self.buyout = buyout;
	self.index = index;
	self.count = count;
	if (itemLink) then
		local itemName, _, _, _, _, _, _, _, _, itemTexture = GetItemInfo(itemLink);
		if (itemName) then
			self.itemIcon:SetNormalTexture(itemTexture);
			self.itemName:SetText(itemName);
			if (count > 1) then
				self.itemCount:SetText(count);
				self.itemCount:Show();
			else
				self.itemCount:Hide();
			end
			self.itemLink = itemLink;
			self.bidBt:SetText(vendor.Format.FormatMoney(bid));
			self.buyoutBt:SetText(vendor.Format.FormatMoney(buyout));
			self.reasonTxt:SetText(reason);
			self.decisionMade = nil;
			self.name = itemName;
			if (buyout and buyout > 0) then
				self.doBuyoutBt:Enable();
			end
			self.frame:Show();
			_WaitForDecision(self);
			if (not isShown) then
				self.frame:Hide();
			end
		end
	end
	self.scanFrame:Show();
	self.buyFrame:Hide();
end

--[[
	Shows the given scan progress message.
--]]
function vendor.ScanDialog.prototype:ShowProgress(msg)
	if (not self.frame:IsShown()) then
		self.frame:Show();
	end
	if (not self.scanBt:IsShown()) then
		self.scanBt:Show();
	end
	if (self.closeBt:IsShown()) then
		self.closeBt:Hide();
	end
	self.scanTxt:SetText(msg);
end

--[[
	Hides the dialog frame.
--]]
function vendor.ScanDialog.prototype:Hide()
	self.frame:Hide();
end

--[[
	Converts the "Stop" to a "Close" button.
--]]
function vendor.ScanDialog.prototype:ShowCloseButton()
	self.closeBt:Show();
	self.scanBt:Hide();
end
