--[[
	Comprehensive database for item information used in vendor.
	The database is stored per "realm", where different factions are
	also counted as different realms. The neutral faction is saved 
	in the "server" space. 
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.Items = vendor.Vendor:NewModule("Items");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local ITEMS_VERSION = 2;
local self = vendor.Items;

--[[
	Resets the database for the current realm/server.
--]]
local function _ResetDb()
	vendor.Vendor:ResetDB("Items");
	vendor.Vendor:Print(L["Database of items for current realm/server where reset."]);
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.Items = {
		type = "group",
		name = L["Items"],
		desc = L["Items"],
		args = {
			reset = {
				type = "execute",
				name = L["Reset database"],
				desc = L["Resets the database of items for the current realm and server. This will delete all alltime statistics and sell prizes, so be careful."],
				func = function() StaticPopup_Show("VENDOR_ITEMS_DB_RESET") end,
			},
		}
	}
end

--[[
	Initializes the safety dialog for the database reset.
--]]
local function _CreateResetDialog()
	StaticPopupDialogs["VENDOR_ITEMS_DB_RESET"] = {
		text = L["Do you really want to reset the database?"];
	  	button1 = L["Yes"],
	  	button2 = L["No"],
	  	OnAccept = function()
	  		_ResetDb()
	  	end,
	  	timeout = 0,
	  	whileDead = 1,
	  	hideOnEscape = 1
	};	
end

--[[
	Initializes the module.
--]]
function vendor.Items:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("Items");
	vendor.Vendor:RegisterDefaults("Items", "realm", {
		index = {},
		version = ITEMS_VERSION
	})
	vendor.Vendor:RegisterDefaults("Items", "server", {
		index = {},
		version = ITEMS_VERSION
	})
	_CreateResetDialog();
	_CreateOptions();
end
	
--[[
	Returns the no. of items.
	@param isNeutral decides whether to access the neutral database.
--]]
function vendor.Items:Size(isNeutral)
	if (isNeutral) then
		return table.getn(self.db.server.index);
	end
	return table.getn(self.db.realm.index);
end

--[[
	Takes blizzard's item link and returns a shorter, normalized one.
--]]
function vendor.Items:GetItemLinkKey(itemLink)
	assert(itemLink);
	local _, _, itemString = string.find(itemLink, "^|%x+|H(.+)|h%[.+%]");
	local _, _, itemId, enchantId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId, uniqueId = string.find(itemString, "^item:(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%-?%d+):(%-?%d+)");
	return strjoin(":", itemId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId);			
end

--[[
	Takes the short, normailized item key and returns a blizzard compatible item link.
--]]
function vendor.Items:GetItemLink(itemLinkKey)
	local itemId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId = strsplit(":", itemLinkKey);
  	return "item:"..strjoin(":", itemId, 0, jewelId1, jewelId2, jewelId3, jewelId4, suffixId, 0);
end

--[[
	Returns all information known about the given item. The normalized item link
	is retrieved via Items:GetItemLinkKey() and is not compatible with
	Blizzard's item link. nil will be returned, if nothing is known about the item.
	@param itemLinkKey the normailzed item link.
	@param isNeutral determines whether to retrieve neutral item information (neutral auction houses)
	@return struct with avgMinBid, avgBuyout, minBid, buyout, stackSize 
--]]
function vendor.Items:GetItemInfo(itemLinkKey, isNeutral)
	local data;
	if (isNeutral) then
		data = self.db.server.index[itemLinkKey];
	else
		data = self.db.realm.index[itemLinkKey];
	end
	if (data) then
		local avgMinBid, avgBuyout, minBid, buyout, stackSize = strsplit(":", data);
		if (not stackSize) then
			stackSize = 0;
		end
		return {
			avgMinBid = tonumber(avgMinBid), avgBuyout = tonumber(avgBuyout), 
			minBid = tonumber(minBid), buyout = tonumber(buyout), stackSize = tonumber(stackSize)
		} 
	end
	return nil;
end

--[[
	Updates all information known about the given item. The normalized item link
	is retrieved via Items:GetItemLinkKey() and is not compatible with
	Blizzard's item link. 
	@param itemLinkKey the normailzed item link.
	@param isNeutral determines whether to retrieve neutral item information (neutral auction houses)
	@param itemInfo struct with avgMinBid, avgBuyout, minBid, buyout, stackSize
--]]
function vendor.Items:PutItemInfo(itemLinkKey, isNeutral, itemInfo)
	local data = strjoin(":", itemInfo.avgMinBid, itemInfo.avgBuyout, 
		itemInfo.minBid, itemInfo.buyout, itemInfo.stackSize);
	if (isNeutral) then
		self.db.server.index[itemLinkKey] = data;
	else
		self.db.realm.index[itemLinkKey] = data;
	end
end

--[[
	Returns an empty item info struct.
--]]
function vendor.Items:CreateItemInfo()
	return {avgMinBid = 0, avgBuyout = 0, minBid = 0, buyout = 0, stackSize = 0};
end
