--[[
	Offers functionality for the auction house, needed by Vendor.
	This includes the addition of new tabs and specifying the 
	tab to be selected when opening the auction house frame.

	Copyright (C) Udorn (Blackhand)

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
--]]

vendor.AuctionHouse = vendor.Vendor:NewModule("AuctionHouse", "AceEvent-2.0", "AceHook-2.1");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local waterFall = AceLibrary("Waterfall-1.0");
local self = vendor.AuctionHouse;

local START_TABS = {
	[L["Browse"]] = 1,
	[L["Bids"]] = 2,
	[L["Auctions"]] = 3, 
	[L["Vendor"]] = 4,
}

--[[
	Sets the given value in the profile.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
	self:ApplySettings();
end

--[[
	Returns the given value from the profile.
--]]
local function _GetValue(field)
	return self.db.profile[field]
end

--[[
	Options for this module.
--]]
local function _CreateOptions()
	vendor.Vendor.options.args.AuctionHouse = {
		type = "group",
		name = L["AuctionHouse"],
		desc = L["AuctionHouse"],
		args = {
			toggle = {
				type = "text",
				name = L["Start tab"],
				desc = L["Tab to be selected when opening the auction house."],
				get = _GetValue,
				set = _SetValue,
				validate = {L["Browse"], L["Bids"], L["Auctions"], L["Vendor"]},
				passValue = "startTab",
			},
		}
	}
end

--[[
	Called for the tab button.
--]]
local function _TabOnClick(self, id)
	-- ensure the correct no. of tabs
	PanelTemplates_SetNumTabs(AuctionFrame, self.numFrames);
	-- hide all custom tabs
	for _, frame in pairs(self.frames) do
		frame:HideTabFrame();
	end
	if (id and id > 3) then
		-- hide blizzard frames, set tab and play sound
		PanelTemplates_SetTab(AuctionFrame, id);
		AuctionFrame.type = self.frames[id]:GetTabType();
		
		AuctionFrameAuctions:Hide();
		AuctionFrameBrowse:Hide();
		AuctionFrameBid:Hide();
		PlaySound("igCharacterInfoTab");
		self.frames[id]:UpdateTabFrame();
		self.frames[id]:ShowTabFrame();
	end
	self.type = AuctionFrame.type;
end

--[[
	Creates the buttons for some calls like scan or snipe.
--]]
local function _CreateButtons(self)
	local snipeBt = vendor.GuiTools.CreateButton(AuctionFrame, "UIPanelButtonTemplate", 86, 20, L["Snipe"], L["Opens a window to select maximum prices for an item\nto be sniped during the next auction house scans."]);
	snipeBt:SetPoint("TOPRIGHT", -21, -14);
	snipeBt:SetScript("OnClick", function() vendor.Sniper:OpenSnipeDialog(); end);	
		
	local scanBt = vendor.GuiTools.CreateButton(AuctionFrame, "UIPanelButtonTemplate", 86, 20, L["Scan"], L["Starts to scan the auction house to update\nstatistics and snipe for items."]);
	scanBt:SetPoint("RIGHT", snipeBt, "LEFT", 3, 0);
	scanBt:SetScript("OnClick", function() vendor.Scanner:Scan(); end);
	scanBt:SetScript("OnUpdate", function()
		if (vendor.Scanner:IsScanning()) then
			this:Disable();
		else
			this:Enable();
		end
	end);
	
	local confBt = vendor.GuiTools.CreateButton(AuctionFrame, "UIPanelButtonTemplate", 106, 20, L["Configuration"], L["Opens a configuration dialog for Vendor."]);
	confBt:SetPoint("RIGHT", scanBt, "LEFT", 3, 0);
	confBt:SetScript("OnClick", function()
		if (waterFall:IsOpen("Vendor")) then 
			waterFall:Close("Vendor"); 
		else
			waterFall:Open("Vendor");
		end
	end);
end

--[[
	Initializes the module.
--]]
function vendor.AuctionHouse:OnInitialize()
	self.db = vendor.Vendor:AcquireDBNamespace("AuctionHouse");
	vendor.Vendor:RegisterDefaults("AuctionHouse", "profile", {
		startTab = L["Browse"],
	})
	_CreateOptions();
end

--[[
	Initializes the module.
--]]
function vendor.AuctionHouse:OnEnable()
	self.frames = {};
	self.numFrames = 3;	
	self:RegisterEvent("AUCTION_HOUSE_SHOW");
	self:SecureHook("AuctionFrameTab_OnClick");
	_CreateButtons(self);
end

--[[
	Hooks opening of the auction house to select start tab.
--]]
function vendor.AuctionHouse:AUCTION_HOUSE_SHOW()
	if (self.startTab) then
		AuctionFrameTab_OnClick(self.startTab);
	end
end

--[[
	Hook for auction frame tab clicks
--]]
function vendor.AuctionHouse:AuctionFrameTab_OnClick(id)
	_TabOnClick(self, id);
end

--[[
	Returns true, if the currently opened auction house is neutral.
	If no auction house is open, false will be returned.
--]]
function vendor.AuctionHouse:IsNeutral()
	if (not AuctionFrame:IsVisible()) then
		return false;
	end
	return GetAuctionHouseDepositRate() ~= 5;
end

--[[
	Constructor for tabbed frames.
	@param name the name of the created frame
	@id the tab sequence starting with 4
	@tabFrame the controler for the created frame. It has to implement the following methods:
	
	UpdateTabFrame: Updates the frame for displaying it. This is the right place to 
	switch the background.

	ShowTabFrame: Displays the frame object.
		
	HideTabFrame: Hides the frame object.
	
	GetTabType: Returns an unqiue string identifier like "vendor" to be used
	for GetCurrentAuctionHouseTab.
--]]
function vendor.AuctionHouse:CreateTabFrame(name, id, title, tabFrame)
	local frame = CreateFrame("Frame", name, AuctionFrame);
	frame:SetWidth(758);
	frame:SetHeight(447);
	frame:SetPoint("TOPLEFT");
	local f = frame:CreateFontString("BrowseTitle", "BACKGROUND", "GameFontNormal");
	f:SetText(title);
	f:SetPoint("TOP", 0, -18);
	frame:Hide();
	if (not self.frames[id]) then
		self.numFrames = self.numFrames + 1;
	end
	self.frames[id] = tabFrame;
	return frame;
end

--[[
	Constructor for tab buttons
	@param text the text to be displayed on the button
	@param id the sequence of the button, startig with 4
--]]
function vendor.AuctionHouse:CreateTabButton(text, id)
	local but = CreateFrame("Button", "AuctionFrameTab"..id, AuctionFrame, "AuctionTabTemplate");
	but:SetID(id);
	but:SetText(text);
	but:SetPoint("TOPLEFT", "AuctionFrameTab"..(id - 1), "TOPRIGHT", -8, 0);
	but:SetScript("OnClick", function() _TabOnClick(vendor.AuctionHouse, id) end);
	return but;
end

--[[
	Constructor for close button of the tabs.
--]]
function vendor.AuctionHouse:CreateCloseButton(parent, name)
	local but = CreateFrame("Button", name, parent, "UIPanelButtonTemplate");
	but:SetText(L["Close"]);
	but:SetWidth("80");
	but:SetHeight("22");
	but:SetPoint("BOTTOMRIGHT", parent, "BOTTOMRIGHT", 66, 14);
	but:SetScript("OnClick", function() CloseAuctionHouse(); end);
	return but;
end

--[[
	Returns the type of tab currently shown in the auction house. Possible 
	values are the ones from blizzard "list", "bidder", "owner" and
	the vendor defined ones. The Tabs of vendor have to implement the
	method GetTabType to retrieve it from them.
--]]
function vendor.AuctionHouse:GetCurrentAuctionHouseTab()
	return self.type;
end

--[[
	Applys the possibly changed settings.
--]]
function vendor.AuctionHouse:ApplySettings()
	self.startTab = START_TABS[self.db.profile.startTab] or 1;
end

