--[[
	The entry point of the addon.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

vendor.Vendor = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceHook-2.1", "AceEvent-2.0", "AceModuleCore-2.0", "AceDebug-2.0", "AceDB-2.0");
local L = AceLibrary("AceLocale-2.2"):new("Vendor");
local waterFall = AceLibrary("Waterfall-1.0");

local self = vendor.Vendor;

--[[
	Sets the given new value for the specified field.
--]]
local function _SetValue(field, value)
	self.db.profile[field] = value;
	self:ApplySettings();
end

--[[
	Returns the current value for the specified field.
--]]
local function _GetValue(field)
	return self.db.profile[field];	
end

--[[
	Applies settings locally, without calling the modules.
--]]
local function _ApplySettings()
	-- apply settings locally	
	self:SetDebugging(self.db.profile.debug);
	if (not self.db.profile.useTooltips) then	
		vendor.Vendor.options.args.gui.args.tooltipFadeTime.disabled = true;
	else
		vendor.Vendor.options.args.gui.args.tooltipFadeTime.disabled = false;
	end
end

--[[
	All options of the addon.
--]]
local function _CreateOptions(self)
	self.options = { 
    type = "group",
	name = L["Vendor"],
	desc = L["Vendor"],
    args = {
		conf = {
			type = "execute",
			name = L["Configuration"],
			desc = L["Opens a configuration window."],
			func = function()
				if (waterFall:IsOpen("Vendor")) then 
					waterFall:Close("Vendor"); 
				else
					waterFall:Open("Vendor");
				end				 
			end
		},
		gui = {
			type = "group",
			name = L["Gui"],
			desc = L["Gui"],
			args = {
				tooltips = {
					type = "toggle",
					name = L["Tooltips"],
					desc = L["Selects whether tooltips for the gui elements of vendor should be activated."],
					get = _GetValue,
					set = _SetValue,
					order = 1,
					passValue = "useTooltips"
				},
				tooltipFadeTime = {
						type = "range",
						name = L["Fade time"],
						desc = L["Selects the time in secs to wait until the tooltip fades in."],
						min = 0.0,
						max = 10.0,
						step = 0.1,
						get = _GetValue,
						set = _SetValue,
						order = 2,
						passValue = "tooltipFadeTime"
					},					
			}
		}		
    },
	}
end

--[[
	Handles the disabling of the debugger.
--]]
local function _OnDebugDisable()
	self.db.profile.debug = false;
end

--[[
	Handles the enabling of the debugger.
--]]
local function _OnDebugEnable()
	self.db.profile.debug = true;
end

--[[
	Initializes the addon.
--]]
function vendor.Vendor:OnInitialize()
	self:RegisterDB("VendorDb");
	self:RegisterDefaults("profile", {
		useTooltips = true,
		tooltipFadeTime = 0.7,
	});
	self.OnDebugEnable = _OnDebugEnable; -- to be able to remember the debug state
	self.OnDebugDisable = _OnDebugDisable;
	_CreateOptions(self);
	waterFall:Register("Vendor", "aceOptions", self.options,
						"title", "Vendor",
						"treeLevels", 2,
						"colorR", 1, "colorG", 0.3, "colorB", 0.3);
	self:RegisterChatCommand({"/vendor"}, self.options);
end

--[[
	Enables the addon.
--]]
function vendor.Vendor:OnEnable()
	self:ApplySettings();	
end

--[[
	Enables a profile.
--]]
function vendor.Vendor:OnProfileEnable()
	self:ApplySettings();
end

--[[
	Displays the given error message
--]]
function vendor.Vendor:Error(msg)
	message(L["Error: %s"]:format(msg));
end

--[[
	Calls ApplySettings for all modules.
--]]
function vendor.Vendor:ApplySettings()
	_ApplySettings();
	-- apply settings for all modules
	for name, module in self:IterateModules() do
		if module.ApplySettings then
			module:ApplySettings();
		end
	end
end
