--[[
	Wraps a dropdown button.
	
	Copyright (C) Udorn (Blackhand)
	
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.	
--]]

local AceOO = AceLibrary("AceOO-2.0");

vendor.DropDownButton = AceOO.Class();

--[[
	Callback for selecting items.
--]]
local function _DropDownOnClick(arg1, arg2)
	UIDropDownMenu_SetSelectedValue(arg1.button, this.value);
	if (arg1.listener) then
		arg1.listener:DropDownButtonSelected(arg1, this.value);
	end
end

--[[
	Callback for initializing a dropdown button.
--]]
local function _DropDownInitialize()
	local button = getglobal(UIDROPDOWNMENU_INIT_MENU);
	local info = UIDropDownMenu_CreateInfo();
	info.owner = button:GetParent();
	info.func = _DropDownOnClick;
	info.arg1 = button.ctrl;
	for key, value in ipairs(button.ctrl.items) do
		if (type(value) == "table") then
			info.text = value.text;
			info.value = value.value;
		else
			info.text = value;
			info.value = value;			
		end
		info.checked = nil;
		UIDropDownMenu_AddButton(info);
	end
end

--[[
	Constructor.
--]]
function vendor.DropDownButton.prototype:init(name, parent, width, title, tooltipText)
	vendor.DropDownButton.super.prototype.init(self)
	self.button = CreateFrame("Frame", name, parent, "UIDropDownMenuTemplate");
	UIDropDownMenu_SetWidth(width, self.button);
	local f = self.button:CreateFontString(name.."Name", "OVERLAY", "GameFontHighlightSmall");
	f:SetText(title);
	f:SetPoint("BOTTOMLEFT", name, "TOPLEFT", 20, 0);
	if (tooltipText) then
		local bt = getglobal(name.."Button");
		vendor.GuiTools.AddFadingTooltip(bt, tooltipText);
	end
	self.button.ctrl = self;
	self:Clear();
end

--[[
	Sets a unique identifier for the button.
--]]
function vendor.DropDownButton.prototype:SetId(id)
	self.id = id;
end

--[[
	Returns the identifier of the button, if any.
--]]
function vendor.DropDownButton.prototype:GetId()
	return self.id;
end

--[[
	Sets a listener for selection changes. The listener has to implement the method
	"DropDownButtonSelected" with the following synopsis:
	@param button The corresponding DropDownButton
	@param value The selected value
--]]
function vendor.DropDownButton.prototype:SetListener(listener)
	self.listener = listener;
end

--[[
	Sets the position of the button.
--]]
function vendor.DropDownButton.prototype:SetPoint(...)
	self.button:SetPoint(...);
end

--[[
	Clears the list of selectable items
--]]
function vendor.DropDownButton.prototype:Clear()
	self.items = {};
	UIDropDownMenu_ClearAll(self.button);
end

--[[
	Sets the list of selectable items from the given table.
--]]
function vendor.DropDownButton.prototype:SetItems(itemTable, selectedValue)
	self.items = itemTable;
	UIDropDownMenu_Initialize(self.button, _DropDownInitialize);
	if (selectedValue) then
		self:SelectValue(selectedValue);
	end
end

--[[
	Selects the given value, that should be in the value list.
--]]
function vendor.DropDownButton.prototype:SelectValue(item)
	UIDropDownMenu_SetSelectedValue(self.button, item);
end

--[[
	Returns the currently selected value.
--]]
function vendor.DropDownButton.prototype:GetSelectedValue()
	return UIDropDownMenu_GetSelectedValue(self.button);
end
