--[[--------------------------------------------------------
-- Factionizer, a reputation management tool --
------------------------------------------------------------
CODE INDEX (search on index for fast access):
_01_ Variables
_02_ Default values
_03_ OnLoad
_04_ OnEvent
_05_ Slash Command
_06_ OnUpdate
_07_ Spell functions
_08_ dragging and resizing
_09_ Tools
_10_ Information
_11_ urbin addon listing
_12_ colour handling
_13_ profile handling
]]----------------------------------------------------------

-- todo: make colours stored

--------------------------
-- _01_ Variables
--------------------------
URH_NAME = "UrbinRange"
URH_Version = GetAddOnMetadata(URH_NAME, "Version")
URH_PlayerKey = ""
URH_InitComplete = false

local URH_TargetFrameState = 0
local URH_OldTargetFrameStat = 0
local URH_mouse_x
local URH_mouse_y
local URH_forceUpdate = false

--------------------------
-- _02_ Default values
--------------------------
URH_HELP_COLOUR = "|cFFFFFF7F"

URH_FontScale = 1

URH_RangeText = {}
URH_RangeText.Melee = URH_TXT.melee
URH_RangeText.Ranged = URH_TXT.range
URH_RangeText.DeadZone = URH_TXT.deadzone
URH_RangeText.OutOfRange = URH_TXT.outofrange

--------------------------
-- _03_ OnLoad
--------------------------

------------------------------------------------------------
function URH_OnLoad()
	SlashCmdList["URH"] = URH_SlashCommand
	SLASH_URH1 = "/urbinrange"
	SLASH_URH2 = "/rh"
	SLASH_URH3 = "/ur"

	URH_PlayerKey = UnitName("player") .. " of " .. GetCVar("realmName")

	--this:RegisterEvent("VARIABLES_LOADED") 	-- spellbook not yet available
	this:RegisterEvent("SPELLS_CHANGED")			-- spellbook available, always fired after vars are loaded

	URH_TargetFrameState = 0
	URH_Old_TargetFrameState = 0

	URH_RegisterUrbinAddon(URH_NAME, URH_About)
end

--------------------------
-- _04_ OnEvent
--------------------------

------------------------------------------------------------
function URH_OnEvent()
	if (event == "SPELLS_CHANGED") then
		if (not URH_InitComplete) then
			if (not URH_Data) then
				URH_Data = {}
				URH_Data.Version = URH_Version
			end
			if (not URH_Data.Version or (URH_Data.Version < "20400.2") or (not URH_Data[URH_PlayerKey])) then
				URH_DefaultSettings()
			end

			if (not URH_Data[URH_PlayerKey].Colours or
			    not URH_Data[URH_PlayerKey].Colours.Bg or
			    not URH_Data[URH_PlayerKey].Colours.Border or
			    not URH_Data[URH_PlayerKey].Colours.Font) then
				URH_SetDefaultColours()
			end

			if (not URH_Data.Profiles) then
				URH_Data.Profiles = {}
			end
			if (not URH_Data.Profiles[URH_PlayerKey]) then
				URH_Data.Profiles[URH_PlayerKey] = true
			end
			URH_InitComplete = true
		end
	end
end

--------------------------
-- _05_ Slash Command
--------------------------

------------------------------------------------------------
function URH_SlashCommand(msg)
	if not msg then
		URH_ShowStatus()
	else
		local msgLower = string.lower(msg)
		local words = URH_GetWords(msg)
		local wordsLower = URH_GetWords(msgLower)
		local size = URH_TableSize(wordsLower)

		if (size>0) then
			if (wordsLower[0]=="def" or wordsLower[0]=="default") then
				URH_DefaultSettings()

			elseif (wordsLower[0]=="melee") then
				local name = ""
				for i = 1, size-1 do
					if (name ~= "") then
						name = name.." "
					end
					name = name..wordsLower[i]
				end
				URH_SetMeleeName(name)
			elseif (wordsLower[0]=="range") then
				local name = ""
				for i = 1, size-1 do
					if (name ~= "") then
						name = name.." "
					end
					name = name..wordsLower[i]
				end
				URH_SetRangedName(name)
			elseif (wordsLower[0]=="check") then
				URH_CheckSpells(true, true)

			elseif (wordsLower[0]=="dump") then
				URH_Dump()

			elseif (wordsLower[0]=="enable")  then
				URH_Data[URH_PlayerKey].Enabled = 1
				URH_ForceUpdate()
				URH_ShowStatus()
			elseif (wordsLower[0]=="disable")  then
				URH_Data[URH_PlayerKey].Enabled = 0
				URH_ForceUpdate()
				URH_ShowStatus()
			elseif (wordsLower[0]=="toggle")  then
				URH_Data[URH_PlayerKey].Enabled = not URH_Data[URH_PlayerKey].Enabled
				URH_ForceUpdate()
				URH_ShowStatus()

			elseif (wordsLower[0]=="resize")  then
				URH_Data[URH_PlayerKey].Move = false
				URH_Data[URH_PlayerKey].Resize = true
				URH_ShowStatus()
			elseif (wordsLower[0]=="move")  then
				URH_Data[URH_PlayerKey].Move = true
				URH_Data[URH_PlayerKey].Resize = false
				URH_ShowStatus()
			elseif (wordsLower[0]=="both")  then
				URH_Data[URH_PlayerKey].Move = true
				URH_Data[URH_PlayerKey].Resize = true
				URH_ShowStatus()
			elseif (wordsLower[0]=="lock")  then
				URH_Data[URH_PlayerKey].Move = false
				URH_Data[URH_PlayerKey].Resize = false
				URH_ShowStatus()
			elseif (wordsLower[0]=="reset")  then
				URH_ResetSizeAndPos()
				URH_ShowStatus()

			elseif (wordsLower[0]=="colour" or wordsLower[0]=="color" or wordsLower[0]=="col") then
				if (size>3) then
					URH_Print("Colour is ["..wordsLower[3].."]")

					local col = ""
					if (wordsLower[2]=="melee") then
						col = "Melee"
					elseif (wordsLower[2]=="range" or wordsLower[2]=="ranged") then
						col = "Ranged"
					elseif (wordsLower[2]=="dead") then
						col = "DeadZone"
					elseif (wordsLower[2]=="oor" or wordsLower[2]=="out" or wordsLower[2]=="outofrange") then
						col = "OutOfRange"
					else
						URH_Print("Unknown range ["..wordsLower[2].."]")

						URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.command.." ["..URH_HELP_COLOUR..msgLower.."|r]")
						URH_Help()
					end

					if (col ~= "") then
						if (wordsLower[1]=="bg") then
							URH_SetBgColour(col, wordsLower[3])
							URH_ForceUpdate()
						elseif (wordsLower[1]=="border") then
							URH_SetBorderColour(col, wordsLower[3])
							URH_ForceUpdate()
						elseif (wordsLower[1]=="font") then
							URH_SetFontColour(col, wordsLower[3])
							URH_ForceUpdate()
						else
							URH_Print("Unknown colour ["..wordsLower[1].."]")

							URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.command.." ["..URH_HELP_COLOUR..msgLower.."|r]")
							URH_Help()
						end
					end
				elseif (size>1) then
					if (wordsLower[1]=="default" or wordsLower[1]=="def") then
						URH_SetDefaultColours()
						URH_ShowColours()
						URH_ForceUpdate()
					else
						URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.command.." ["..URH_HELP_COLOUR..msgLower.."|r]")
						URH_Help()
					end
				else
					URH_ShowColours()
				end

			elseif (wordsLower[0]=="profile") then
				if (size>1) then
					if (wordsLower[1]=="list") then
						URH_ListProfiles()
					else
						local name = ""
						for i = 1, size-1 do
							if (name ~= "") then
								name = name.." "
							end
							name = name..words[i]
						end
						URH_ImportProfile(name)
					end
				else
					URH_ListProfiles()
				end

			elseif (wordsLower[0]=="status") then
				URH_ShowStatus()

			elseif (wordsLower[0]=="test") then
				URH_Test()

			elseif (wordsLower[0]=="help") then
				URH_Help()
			elseif (wordsLower[0]=="about") then
				URH_About()
			elseif (wordsLower[0]=="urbin") then
				URH_ListUrbinAddonDetails()
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.command.." ["..URH_HELP_COLOUR..msgLower.."|r]")
				URH_Help()
			end
		else
			URH_ShowStatus()
		end
	end
end


--------------------------
-- _06_ OnUpdate
--------------------------

------------------------------------------------------------
function URH_ForceUpdate()
	URH_forceUpdate = true
	URH_OnUpdate()
end

------------------------------------------------------------
function URH_OnUpdate()
	if (not URH_Data[URH_PlayerKey].Enabled) then
		if (URH_TargetRangeInfo:IsVisible()) then
			URH_TargetRangeInfo:Hide()
		end
		URH_forceUpdate = false
		return
	end

	if (UnitExists("target") and (not UnitIsDead("target")) and UnitCanAttack("player", "target")) then
		-- enemy target exists -> show range frame
		index = ""

		if (not URH_TargetRangeInfo:IsVisible()) then
			URH_TargetRangeInfo:Show()
			URH_RangeInfoFontFrame:SetScale(URH_FontScale)
		end

		if (URH_Data[URH_PlayerKey].MeleeName == "") then
			text = URH_TXT.meleeNotset
			URH_TargetRangeInfo:SetBackdropColor(1.0,1.0,1.0)
			URH_TargetRangeInfo:SetBackdropBorderColor(1.0,1.0,1.0)
			URH_TargetRange:SetTextColor(0.5,0.5,0.5)
			URH_TargetRange:SetText(text)

		elseif (URH_Data[URH_PlayerKey].RangeName == "") then
			text = URH_TXT.rangedNotSet
			URH_TargetRangeInfo:SetBackdropColor(1.0,1.0,1.0)
			URH_TargetRangeInfo:SetBackdropBorderColor(1.0,1.0,1.0)
			URH_TargetRange:SetTextColor(0.5,0.5,0.5)
			URH_TargetRange:SetText(text)

		elseif (IsSpellInRange(URH_Data[URH_PlayerKey].MeleeName) == 1) then
			index = "Melee"
			URH_TargetFrameState = 1

		elseif (IsSpellInRange(URH_Data[URH_PlayerKey].RangeName) == 1) then
			index = "Ranged"
			URH_TargetFrameState = 2

		elseif (CheckInteractDistance("target", 4)) then
			index = "DeadZone"
			URH_TargetFrameState = 3

		else
			index = "OutOfRange"
			URH_TargetFrameState = 4
		end

		if (((URH_TargetFrameState ~= URH_Old_TargetFrameState) or URH_forceUpdate) and (index ~= "")) then
			URH_TargetRangeInfo:SetBackdropColor(URH_Data[URH_PlayerKey].Colours.Bg[index].r, URH_Data[URH_PlayerKey].Colours.Bg[index].g, URH_Data[URH_PlayerKey].Colours.Bg[index].b, URH_Data[URH_PlayerKey].Colours.Bg[index].a)
			URH_TargetRangeInfo:SetBackdropBorderColor(URH_Data[URH_PlayerKey].Colours.Border[index].r, URH_Data[URH_PlayerKey].Colours.Border[index].g, URH_Data[URH_PlayerKey].Colours.Border[index].b, URH_Data[URH_PlayerKey].Colours.Border[index].a)
			URH_TargetRange:SetTextColor(URH_Data[URH_PlayerKey].Colours.Font[index].r, URH_Data[URH_PlayerKey].Colours.Font[index].g, URH_Data[URH_PlayerKey].Colours.Font[index].b)
			URH_TargetRange:SetText(URH_RangeText[index])
			URH_Old_TargetFrameState = URH_TargetFrameState
			URH_forceUpdate = false
		end

	else
		-- no target -> hide range frame
		if (URH_TargetRangeInfo:IsVisible()) then
			URH_TargetRangeInfo:Hide()
			URH_TargetFrameState = 0
			URH_Old_TargetFrameState = 0
			URH_forceUpdate = false
		end
	end
end

--------------------------
-- _07_ Spell functions
--------------------------

------------------------------------------------------------
function URH_DefaultSettings()
	URH_Data.Version = URH_Version
	URH_Data[URH_PlayerKey] = {}
	URH_Data[URH_PlayerKey].Enabled = 1
	URH_Data[URH_PlayerKey].Resize = false
	URH_Data[URH_PlayerKey].Move = true
	URH_Data[URH_PlayerKey].HealName = nil	-- remove any old entries, will not support healing range, done better by other addons

	local playerClass, englishClass = UnitClass("player")
	URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.settingSpell.." ["..playerClass.."]")
	if (englishClass == "HUNTER") then
		-- hunters can use guns
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.autoShot
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.wingClip

	elseif (englishClass == "ROGUE") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.throw
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.sinisterStrike

	elseif (englishClass == "WARRIOR") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.throw
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	elseif (englishClass == "PRIEST") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.shoot
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	elseif (englishClass == "WARLOCK") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.shoot
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	elseif (englishClass == "MAGE") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.shoot
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	elseif (englishClass == "PALADIN") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.righteousDefense
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	elseif (englishClass == "SHAMAN") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.lightningBolt
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	elseif (englishClass == "DRUID") then
		URH_Data[URH_PlayerKey].RangeName = URH_TXT.wrath
		URH_Data[URH_PlayerKey].MeleeName = URH_TXT.attack

	end

	URH_CheckSpells(true, true)

	URH_SetDefaultColours()
end

------------------------------------------------------------
function URH_SetMeleeName(name)
	if not name then return end
	URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.setMelee.." ["..name.."]");

	oldName = URH_Data[URH_PlayerKey].MeleeName
	URH_Data[URH_PlayerKey].MeleeName = name

	local _, melee, _, _ = URH_CheckSpells(true, false)
	if (melee == 0) then
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.keepMelee.." ["..oldName.."]")
		URH_Data[URH_PlayerKey].MeleeName = oldName
	end
end

------------------------------------------------------------
function URH_SetRangedName(name)
	if not name then return end
	URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.setRanged.." ["..name.."]");

	oldName = URH_Data[URH_PlayerKey].RangeName
	URH_Data[URH_PlayerKey].RangeName = name

	local _, _, ranged, _ = URH_CheckSpells(false, true)
	if (ranged == 0) then
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.keepRanged.." ["..oldName.."]")
		URH_Data[URH_PlayerKey].RangeName = oldName
	end
end

------------------------------------------------------------
function URH_CheckSpells(printMelee, printRanged)
	local success = 0
	local melee = 0
	local range = 0

	mname = string.lower(URH_Data[URH_PlayerKey].MeleeName)
	rname = string.lower(URH_Data[URH_PlayerKey].RangeName)

	-- todo: can we get a linkable spell name?

	local i = 1
	while true do
		local spellName, spellRank = GetSpellName(i, BOOKTYPE_SPELL)
		if not spellName then
			do break end
		end
		local link = GetSpellLink(spellName)
		if not link then
			link = "["..spellName.."]"
		end
		nname = string.lower(spellName)
		if (melee == 0) and (nname == mname) then
			URH_Data[URH_PlayerKey].MeleeName = spellName
			melee = 1
			if (printMelee) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.verifyMelee.." "..link)
			end
		elseif (range == 0) and (nname == rname) then
			URH_Data[URH_PlayerKey].RangeName = spellName
			range = 1
			if (printRanged) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.verifyRanged.." "..link)
			end
		end

		-- stop when done
		if (melee == 1 and range == 1) then
			success = 1
			break
		end

		i = i + 1
	end

	if (melee == 0) then
		if (printMelee) then
			URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.missMelee.." ["..URH_Data[URH_PlayerKey].MeleeName.."]")
		end
		URH_Data[URH_PlayerKey].MeleeName = ""
	end
	if (range == 0) then
		if (printRanged) then
			URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.missRanged.." ["..URH_Data[URH_PlayerKey].RangeName.."]")
		end
		URH_Data[URH_PlayerKey].RangeName = ""
	end

	return success, melee, range
end

--------------------------
-- _08_ dragging and resizing
--------------------------

------------------------------------------------------------
function URH_OnDragStart()
	if (URH_Data[URH_PlayerKey].Resize == true) then
		local hotleft = URH_TargetRangeInfo:GetLeft()*UIParent:GetScale() + 7
		local hottop = URH_TargetRangeInfo:GetTop()*UIParent:GetScale() - 7
		local hotbottom = URH_TargetRangeInfo:GetBottom()*UIParent:GetScale() + 7
		local hotright = URH_TargetRangeInfo:GetRight()*UIParent:GetScale() - 7
		if (mouse_x < hotleft and mouse_y > hottop) then
			URH_TargetRangeInfo:StartSizing("TOPLEFT")
		elseif (mouse_x > hotright and mouse_y > hottop) then
			URH_TargetRangeInfo:StartSizing("TOPRIGHT")
		elseif (mouse_x > hotright and mouse_y < hotbottom) then
			URH_TargetRangeInfo:StartSizing("BOTTOMRIGHT")
		elseif (mouse_x < hotleft and mouse_y < hotbottom) then
			URH_TargetRangeInfo:StartSizing("BOTTOMLEFT")
		elseif (mouse_x > hotright) then
			URH_TargetRangeInfo:StartSizing("RIGHT")
		elseif (mouse_x < hotleft) then
			URH_TargetRangeInfo:StartSizing("LEFT")
		elseif (mouse_y > hottop) then
			URH_TargetRangeInfo:StartSizing("TOP")
		elseif (mouse_y < hotbottom) then
			URH_TargetRangeInfo:StartSizing("BOTTOM")

		elseif (URH_Data[URH_PlayerKey].Move == true) then
			URH_TargetRangeInfo:StartMoving()
		end

	elseif (URH_Data[URH_PlayerKey].Move == true) then
		URH_TargetRangeInfo:StartMoving()
	end
end

------------------------------------------------------------
function URH_OnDragStop()
	URH_TargetRangeInfo:StopMovingOrSizing()
end

------------------------------------------------------------
function URH_OnMouseDown()
	mouse_x, mouse_y = GetCursorPosition()
end

------------------------------------------------------------
function URH_ResetSizeAndPos()
	URH_TargetRangeInfo:StopMovingOrSizing()
	URH_TargetRangeInfo:SetWidth(135)	-- synch with XML if changed there
	URH_TargetRangeInfo:SetHeight(35)	-- synch with XML if changed there

	-- todo: set position
end

--------------------------
-- _09_ Tools
--------------------------

------------------------------------------------------------
function URH_Print(msg)
	DEFAULT_CHAT_FRAME:AddMessage(msg)
end

------------------------------------------------------------
function URH_TableSize(info)
	local result = 0
	if info then
		for item in pairs(info) do result = result + 1 end
	end
	return result
end

------------------------------------------------------------
function URH_GetWords(str)
	local ret = {};
	local pos=0;
	local index=0
	while(true) do
		local word;
		_,pos,word=string.find(str, "^ *([^%s]+) *", pos+1);
		if(not word) then
			return ret;
		end
		ret[index]=word
		index = index+1
	end
end

------------------------------------------------------------
function URH_BoolToEnabled(b)
	local result = URH_TXT.disabled
	if b then result = URH_TXT.enabled end
	return result
end

------------------------------------------------------------
function URH_StringConditional(s)
	local result = s

	if (s == nil) or (s == "") then
		result = URH_TXT.disabled
	end

	return result
end

--------------------------
-- _10_ Information
--------------------------

------------------------------------------------------------
function URH_Help()
	URH_Print(" ", true)
	URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.help)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh melee <spell> "..URH_HELP_COLOUR..URH_TXT.helpMelee)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh range <spell> "..URH_HELP_COLOUR..URH_TXT.helpRanged)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh default "..URH_HELP_COLOUR..URH_TXT.helpDefault)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh resize | move | both | lock | reset "..URH_HELP_COLOUR..URH_TXT.helpDrag)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh colour [ { default | {bg | border | font} { melee | range | dead | out } xAARRGGBB } ] "..URH_HELP_COLOUR..URH_TXT.helpColour)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh profile [ list | <name> ] "..URH_HELP_COLOUR..URH_TXT.helpProfile)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh enable | disable | toggle")
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh check "..URH_HELP_COLOUR..URH_TXT.helpCheck)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh status "..URH_HELP_COLOUR..URH_TXT.helpStatus)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh help "..URH_HELP_COLOUR..URH_TXT.helphelp)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh about "..URH_HELP_COLOUR..URH_TXT.helpabout)
	URH_Print(URH_HELP_COLOUR..URH_TXT.usage..":|r /rh urbin "..URH_HELP_COLOUR..URH_TXT.help_urbin);
end

------------------------------------------------------------
function URH_About(urbin)
	local ver = GetAddOnMetadata(URH_NAME, "Version");
	local date = GetAddOnMetadata(URH_NAME, "X-Date");
	local author = GetAddOnMetadata(URH_NAME, "Author");
	local web = GetAddOnMetadata(URH_NAME, "X-Website");

	if (author ~= nil) then
		URH_Print(URH_NAME.." "..URH_TXT.by.." "..URH_HELP_COLOUR..author.."|r", true);
	end
	if (ver ~= nil) then
		URH_Print("  "..URH_TXT.version..": "..URH_HELP_COLOUR..ver.."|r", true);
	end
	if (date ~= nil) then
		URH_Print("  "..URH_TXT.date..": "..URH_HELP_COLOUR..date.."|r", true);
	end
	if (web ~= nil) then
		URH_Print("  "..URH_TXT.web..": "..URH_HELP_COLOUR..web.."|r", true);
	end

	if (urbin) then
		URH_Print("  "..URH_TXT.slash..": "..URH_HELP_COLOUR..SLASH_URH2.."|r", true);
	else
		URH_ListUrbinAddons(URH_NAME)
	end
end

------------------------------------------------------------
function URH_ShowStatus()
	URH_Print(" ")
	URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.status)
	URH_Print("   "..URH_TXT.statEnabled..": "..URH_HELP_COLOUR..URH_BoolToEnabled(URH_Data[URH_PlayerKey].Enabled).."|r")
	URH_Print("   "..URH_TXT.statResize..": "..URH_HELP_COLOUR..URH_BoolToEnabled(URH_Data[URH_PlayerKey].Resize).."|r")
	URH_Print("   "..URH_TXT.statMove..": "..URH_HELP_COLOUR..URH_BoolToEnabled(URH_Data[URH_PlayerKey].Move).."|r")
	URH_Print("   "..URH_TXT.statMelee..": "..URH_HELP_COLOUR..URH_StringConditional(URH_Data[URH_PlayerKey].MeleeName).."|r")
	URH_Print("   "..URH_TXT.statRanged..": "..URH_HELP_COLOUR..URH_StringConditional(URH_Data[URH_PlayerKey].RangeName).."|r")
end

------------------------------------------------------------
function URH_Dump()
			if (IsSpellInRange(URH_Data[URH_PlayerKey].MeleeName) == 1) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.melee.." ["..URH_Data[URH_PlayerKey].MeleeName.."] "..URH_TXT.isinrange)
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.melee.." ["..URH_Data[URH_PlayerKey].MeleeName.."] "..URH_TXT.outofrange)
			end
			if (IsSpellInRange(URH_Data[URH_PlayerKey].RangeName) == 1) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.ranged.." ["..URH_Data[URH_PlayerKey].RangeName.."] "..URH_TXT.isinrange)
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.ranged.." ["..URH_Data[URH_PlayerKey].RangeName.."] "..URH_TXT.outofrange)
			end
			if (CheckInteractDistance("target", 4)) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.interact.." "..URH_TXT.isinrange)
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.interact.." "..URH_TXT.outofrange)
			end
end

------------------------------------------------------------
function URH_Test()
	URH_ExtractColourPerc("0.5,1,0.3,0")
end

--------------------------
-- _11_ urbin addon listing
--------------------------

------------------------------------------------------------
function URH_RegisterUrbinAddon(name, about)
	if (not name) then
		return
	end
	if (not URBIN_AddonList) then
		URBIN_AddonList = {}
	end
	URBIN_AddonList[name] = {}
	URBIN_AddonList[name].name = name
	URBIN_AddonList[name].about = about
end

------------------------------------------------------------
function URH_ListUrbinAddons(name)
	if (not URBIN_AddonList) then
		return
	end

	local addons = ""
	for p in pairs(URBIN_AddonList) do
		if (URBIN_AddonList[p].name ~= name) then
			if (addons ~= "") then
				addons = addons..", "
			end
			addons = addons..URBIN_AddonList[p].name
		end
	end

	if (addons ~= "") then
		URH_Print(" ", true);
		URH_Print("  "..URH_TXT.urbin..": "..URH_HELP_COLOUR..addons.."|r", true);
	end
end

------------------------------------------------------------
function URH_ListUrbinAddonDetails()
	for p in pairs(URBIN_AddonList) do
		if (URBIN_AddonList[p].about) then
			URBIN_AddonList[p].about(true)
		end
	end
end

--------------------------
-- _12_ colour handling
--------------------------

------------------------------------------------------------
function URH_SetDefaultColours()
	URH_Data[URH_PlayerKey].Colours = {}
	URH_Data[URH_PlayerKey].Colours.Bg = {}
	URH_Data[URH_PlayerKey].Colours.Bg.Melee = {}
	URH_Data[URH_PlayerKey].Colours.Bg.Melee.r = 0
	URH_Data[URH_PlayerKey].Colours.Bg.Melee.g = 1
	URH_Data[URH_PlayerKey].Colours.Bg.Melee.b = 0
	URH_Data[URH_PlayerKey].Colours.Bg.Melee.a = 0.7
	URH_Data[URH_PlayerKey].Colours.Bg.Ranged = {}
	URH_Data[URH_PlayerKey].Colours.Bg.Ranged.r = 0
	URH_Data[URH_PlayerKey].Colours.Bg.Ranged.g = 0
	URH_Data[URH_PlayerKey].Colours.Bg.Ranged.b = 1
	URH_Data[URH_PlayerKey].Colours.Bg.Ranged.a = 0.7
	URH_Data[URH_PlayerKey].Colours.Bg.DeadZone = {}
	URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.r = 1
	URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.g = 0.5
	URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.b = 0
	URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.a = 0.7
	URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange = {}
	URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.r = 1
	URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.g = 0
	URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.b = 0
	URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.a = 0.7

	URH_Data[URH_PlayerKey].Colours.Border = {}
	URH_Data[URH_PlayerKey].Colours.Border.Melee = {}
	URH_Data[URH_PlayerKey].Colours.Border.Melee.r = 0
	URH_Data[URH_PlayerKey].Colours.Border.Melee.g = 1
	URH_Data[URH_PlayerKey].Colours.Border.Melee.b = 0
	URH_Data[URH_PlayerKey].Colours.Border.Melee.a = 0.7
	URH_Data[URH_PlayerKey].Colours.Border.Ranged = {}
	URH_Data[URH_PlayerKey].Colours.Border.Ranged.r = 0
	URH_Data[URH_PlayerKey].Colours.Border.Ranged.g = 0
	URH_Data[URH_PlayerKey].Colours.Border.Ranged.b = 1
	URH_Data[URH_PlayerKey].Colours.Border.Ranged.a = 0.7
	URH_Data[URH_PlayerKey].Colours.Border.DeadZone = {}
	URH_Data[URH_PlayerKey].Colours.Border.DeadZone.r = 1
	URH_Data[URH_PlayerKey].Colours.Border.DeadZone.g = 0.5
	URH_Data[URH_PlayerKey].Colours.Border.DeadZone.b = 0
	URH_Data[URH_PlayerKey].Colours.Border.DeadZone.a = 0.7
	URH_Data[URH_PlayerKey].Colours.Border.OutOfRange = {}
	URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.r = 1
	URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.g = 0
	URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.b = 0
	URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.a = 0.7

	URH_Data[URH_PlayerKey].Colours.Font = {}
	URH_Data[URH_PlayerKey].Colours.Font.Melee = {}
	URH_Data[URH_PlayerKey].Colours.Font.Melee.r = 1
	URH_Data[URH_PlayerKey].Colours.Font.Melee.g = 1
	URH_Data[URH_PlayerKey].Colours.Font.Melee.b = 1
	URH_Data[URH_PlayerKey].Colours.Font.Ranged = {}
	URH_Data[URH_PlayerKey].Colours.Font.Ranged.r = 1
	URH_Data[URH_PlayerKey].Colours.Font.Ranged.g = 1
	URH_Data[URH_PlayerKey].Colours.Font.Ranged.b = 1
	URH_Data[URH_PlayerKey].Colours.Font.DeadZone = {}
	URH_Data[URH_PlayerKey].Colours.Font.DeadZone.r = 1
	URH_Data[URH_PlayerKey].Colours.Font.DeadZone.g = 1
	URH_Data[URH_PlayerKey].Colours.Font.DeadZone.b = 1
	URH_Data[URH_PlayerKey].Colours.Font.OutOfRange = {}
	URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.r = 1
	URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.g = 1
	URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.b = 1
end

------------------------------------------------------------
function URH_SetBgColour(range, colour)
	if (URH_Data[URH_PlayerKey].Colours.Bg[range]) then
		if (colour) then
			local r,g,b,a = URH_ExtractColour(colour)
			if (r and g and b and a) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.settingBgColour.." ["..range.."] "..URH_TXT.to.." ["..colour.."]")
				URH_Data[URH_PlayerKey].Colours.Bg[range].r = r
				URH_Data[URH_PlayerKey].Colours.Bg[range].g = g
				URH_Data[URH_PlayerKey].Colours.Bg[range].b = b
				URH_Data[URH_PlayerKey].Colours.Bg[range].a = a

				URH_ShowColours()
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.couldNotExtractBgColour.." ["..colour.."]")
			end
		else
			URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.bgColourNil.." ["..range.."]")
		end
	else
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.unknownRange.." ["..range.."]")
	end
end

------------------------------------------------------------
function URH_SetBorderColour(range, colour)
	if (URH_Data[URH_PlayerKey].Colours.Border[range]) then
		if (colour) then
			local r,g,b,a = URH_ExtractColour(colour)
			if (r and g and b and a) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.settingBorderColour.." ["..range.."] "..URH_TXT.to.." ["..colour.."]")
				URH_Data[URH_PlayerKey].Colours.Border[range].r = r
				URH_Data[URH_PlayerKey].Colours.Border[range].g = g
				URH_Data[URH_PlayerKey].Colours.Border[range].b = b
				URH_Data[URH_PlayerKey].Colours.Border[range].a = a

				URH_ShowColours()
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.couldNotExtractBorderColour.." ["..colour.."]")
			end
		else
			URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.borderColourNil.." ["..range.."]")
		end
	else
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.unknownRange.." ["..range.."]")
	end
end

------------------------------------------------------------
function URH_SetFontColour(range, colour)
	if (URH_Data[URH_PlayerKey].Colours.Font[range]) then
		if (colour) then
			local r,g,b,a = URH_ExtractColour(colour)
			if (r and g and b) then
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.settingFontColour.." ["..range.."] "..URH_TXT.to.." ["..colour.."]")
				URH_Data[URH_PlayerKey].Colours.Font[range].r = r
				URH_Data[URH_PlayerKey].Colours.Font[range].g = g
				URH_Data[URH_PlayerKey].Colours.Font[range].b = b

				URH_ShowColours()
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.couldNotExtractFontColour.." ["..colour.."]")
			end
		else
			URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.fontColourNil.." ["..range.."]")
		end
	else
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.unknownRange.." ["..range.."]")
	end
end

------------------------------------------------------------
function URH_ShowColours()
	local colour = ""

	URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.colours..URH_HELP_COLOUR.." (|r"..URH_TXT.bg..URH_HELP_COLOUR..", |r"..URH_TXT.border..URH_HELP_COLOUR..", |r"..URH_TXT.font..URH_HELP_COLOUR..")|r")
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Bg.Melee.a, URH_Data[URH_PlayerKey].Colours.Bg.Melee.r, URH_Data[URH_PlayerKey].Colours.Bg.Melee.g, URH_Data[URH_PlayerKey].Colours.Bg.Melee.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Bg.Melee.a, URH_Data[URH_PlayerKey].Colours.Bg.Melee.r, URH_Data[URH_PlayerKey].Colours.Bg.Melee.g, URH_Data[URH_PlayerKey].Colours.Bg.Melee.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Border.Melee.a, URH_Data[URH_PlayerKey].Colours.Border.Melee.r, URH_Data[URH_PlayerKey].Colours.Border.Melee.g, URH_Data[URH_PlayerKey].Colours.Border.Melee.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Border.Melee.a, URH_Data[URH_PlayerKey].Colours.Border.Melee.r, URH_Data[URH_PlayerKey].Colours.Border.Melee.g, URH_Data[URH_PlayerKey].Colours.Border.Melee.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(1, URH_Data[URH_PlayerKey].Colours.Font.Melee.r, URH_Data[URH_PlayerKey].Colours.Font.Melee.g, URH_Data[URH_PlayerKey].Colours.Font.Melee.b)
	colour = colour..URH_PercToColourString(1, URH_Data[URH_PlayerKey].Colours.Font.Melee.r, URH_Data[URH_PlayerKey].Colours.Font.Melee.g, URH_Data[URH_PlayerKey].Colours.Font.Melee.b)
	colour = colour.."|r"
	URH_Print("    "..URH_HELP_COLOUR..URH_TXT.melee..":|r "..colour)
	colour = ""
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Bg.Ranged.a, URH_Data[URH_PlayerKey].Colours.Bg.Ranged.r, URH_Data[URH_PlayerKey].Colours.Bg.Ranged.g, URH_Data[URH_PlayerKey].Colours.Bg.Ranged.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Bg.Ranged.a, URH_Data[URH_PlayerKey].Colours.Bg.Ranged.r, URH_Data[URH_PlayerKey].Colours.Bg.Ranged.g, URH_Data[URH_PlayerKey].Colours.Bg.Ranged.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Border.Ranged.a, URH_Data[URH_PlayerKey].Colours.Border.Ranged.r, URH_Data[URH_PlayerKey].Colours.Border.Ranged.g, URH_Data[URH_PlayerKey].Colours.Border.Ranged.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Border.Ranged.a, URH_Data[URH_PlayerKey].Colours.Border.Ranged.r, URH_Data[URH_PlayerKey].Colours.Border.Ranged.g, URH_Data[URH_PlayerKey].Colours.Border.Ranged.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(1, URH_Data[URH_PlayerKey].Colours.Font.Ranged.r, URH_Data[URH_PlayerKey].Colours.Font.Ranged.g, URH_Data[URH_PlayerKey].Colours.Font.Ranged.b)
	colour = colour..URH_PercToColourString(1, URH_Data[URH_PlayerKey].Colours.Font.Ranged.r, URH_Data[URH_PlayerKey].Colours.Font.Ranged.g, URH_Data[URH_PlayerKey].Colours.Font.Ranged.b)
	colour = colour.."|r"
	URH_Print("    "..URH_HELP_COLOUR..URH_TXT.range..":|r "..colour)
	colour = ""
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.a, URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.r, URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.g, URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.a, URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.r, URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.g, URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Border.DeadZone.a, URH_Data[URH_PlayerKey].Colours.Border.DeadZone.r, URH_Data[URH_PlayerKey].Colours.Border.DeadZone.g, URH_Data[URH_PlayerKey].Colours.Border.DeadZone.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Border.DeadZone.a, URH_Data[URH_PlayerKey].Colours.Border.DeadZone.r, URH_Data[URH_PlayerKey].Colours.Border.DeadZone.g, URH_Data[URH_PlayerKey].Colours.Border.DeadZone.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(1, URH_Data[URH_PlayerKey].Colours.Font.DeadZone.r, URH_Data[URH_PlayerKey].Colours.Font.DeadZone.g, URH_Data[URH_PlayerKey].Colours.Font.DeadZone.b)
	colour = colour..URH_PercToColourString(1, URH_Data[URH_PlayerKey].Colours.Font.DeadZone.r, URH_Data[URH_PlayerKey].Colours.Font.DeadZone.g, URH_Data[URH_PlayerKey].Colours.Font.DeadZone.b)
	colour = colour.."|r"
	URH_Print("    "..URH_HELP_COLOUR..URH_TXT.deadzone..":|r "..colour)
	colour = ""
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.a, URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.r, URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.g, URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.a, URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.r, URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.g, URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.a, URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.r, URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.g, URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.b)
	colour = colour..URH_PercToColourString(URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.a, URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.r, URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.g, URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.b)
	colour = colour.."  "
	colour = colour..URH_PercToColour(1, URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.r, URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.g, URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.b)
	colour = colour..URH_PercToColourString(1, URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.r, URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.g, URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.b)
	colour = colour.."|r"
	URH_Print("    "..URH_HELP_COLOUR..URH_TXT.outofrange..":|r "..colour)
end

------------------------------------------------------------
function URH_ExtractColour(colour)
	local r=nil
	local g=nil
	local b=nil
	local a=nil

	if (colour) then
		if (string.sub(colour, 1, 1) == "x") then
			r,g,b,a = URH_ExtractColourHex(colour)
		else
			r,g,b,a = URH_ExtractColourPerc(colour)
		end
	end

	return r,g,b,a
end

------------------------------------------------------------
function URH_ExtractColourHex(colour)
	local r=nil
	local g=nil
	local b=nil
	local a=nil

	if (colour) then
		-- xAARRGGBB
		local hex = string.sub(colour, 2)
		local alpha, rhex, ghex, bhex = string.sub(hex, 1,2), string.sub(hex, 3, 4), string.sub(hex, 5, 6), string.sub(hex, 7, 8)
		a = tonumber(alpha, 16)/255
		r = tonumber(rhex, 16)/255
		g = tonumber(ghex, 16)/255
		b = tonumber(bhex, 16)/255

		--URH_Print("Hex: "..tostring(hex).." - alpha: "..tostring(alpha).." - red: "..tostring(rhex).." - green: "..tostring(ghex).." - blue: "..tostring(bhex))
	end

	return r,g,b,a
end

------------------------------------------------------------
function URH_ExtractColourPerc(colour)
	local r=nil
	local g=nil
	local b=nil
	local a=nil

	if (colour) then
		-- todo: extract 4 values of the form v,v,v,v (where v=0 .. 0.5 .. 1)

		local start = 1
		local s,e,p = string.find(colour, "(%d+%.?%d-)", start)

		URH_Print("Result: ["..colour.."] - s: ["..tostring(s).."] - e: ["..tostring(e).."] - p: ["..tostring(p).."]")
	end

	return r,g,b,a
end

------------------------------------------------------------
function URH_PercToColour(a, r, g, b)
	return string.format("|c%02X%02X%02X%02X", a*255, r*255, g*255, b*255)
end

------------------------------------------------------------
function URH_PercToColourString(a, r, g, b)
	return string.format("x%02X%02X%02X%02X", a*255, r*255, g*255, b*255)
end

--------------------------
-- _13_ profile handling
--------------------------

------------------------------------------------------------
function URH_ListProfiles()
	if URH_Data.Profiles then
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.listProfiles)
		for p in pairs(URH_Data.Profiles) do
			URH_Print("   ["..URH_HELP_COLOUR..p.."|r]")
		end
	end
end

------------------------------------------------------------
function URH_ImportProfile(name)
	local copied = false
	if (name and URH_Data.Profiles) then
		if (URH_Data.Profiles[name]) then
			-- remember to update this when adding new members
			--
			-- it would be nicer to iterate over the data in the other profile but
			-- I am not sure whehter nested tables (such as colours etc) would be
			-- handled correctly
			if (URH_Data[name].Enabled and
			    URH_Data[name].Resize and
			    URH_Data[name].Move and
			    URH_Data[name].RangeName and
			    URH_Data[name].MeleeName and
			    URH_Data[name].Colours) then	-- using the fact, that if [Colours] is defined, all subfields are always also set -> not checking sub fields of colour
				local oldMelee = URH_Data[URH_PlayerKey].MeleeName
				local oldRange = URH_Data[URH_PlayerKey].RangeName

				URH_Data[URH_PlayerKey].Enabled = URH_Data[name].Enabled
				URH_Data[URH_PlayerKey].Resize = URH_Data[name].Resize
				URH_Data[URH_PlayerKey].Move = URH_Data[name].Move

				URH_Data[URH_PlayerKey].RangeName = URH_Data[name].RangeName
				URH_Data[URH_PlayerKey].MeleeName = URH_Data[name].MeleeName

				URH_Data[URH_PlayerKey].Colours = {}
				URH_Data[URH_PlayerKey].Colours.Bg = {}
				URH_Data[URH_PlayerKey].Colours.Bg.Melee = {}
				URH_Data[URH_PlayerKey].Colours.Bg.Melee.r = URH_Data[name].Colours.Bg.Melee.r
				URH_Data[URH_PlayerKey].Colours.Bg.Melee.g = URH_Data[name].Colours.Bg.Melee.g
				URH_Data[URH_PlayerKey].Colours.Bg.Melee.b = URH_Data[name].Colours.Bg.Melee.b
				URH_Data[URH_PlayerKey].Colours.Bg.Melee.a = URH_Data[name].Colours.Bg.Melee.a
				URH_Data[URH_PlayerKey].Colours.Bg.Ranged = {}
				URH_Data[URH_PlayerKey].Colours.Bg.Ranged.r = URH_Data[name].Colours.Bg.Ranged.r
				URH_Data[URH_PlayerKey].Colours.Bg.Ranged.g = URH_Data[name].Colours.Bg.Ranged.g
				URH_Data[URH_PlayerKey].Colours.Bg.Ranged.b = URH_Data[name].Colours.Bg.Ranged.b
				URH_Data[URH_PlayerKey].Colours.Bg.Ranged.a = URH_Data[name].Colours.Bg.Ranged.a
				URH_Data[URH_PlayerKey].Colours.Bg.DeadZone = {}
				URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.r = URH_Data[name].Colours.Bg.DeadZone.r
				URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.g = URH_Data[name].Colours.Bg.DeadZone.g
				URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.b = URH_Data[name].Colours.Bg.DeadZone.b
				URH_Data[URH_PlayerKey].Colours.Bg.DeadZone.a = URH_Data[name].Colours.Bg.DeadZone.a
				URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange = {}
				URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.r = URH_Data[name].Colours.Bg.OutOfRange.r
				URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.g = URH_Data[name].Colours.Bg.OutOfRange.g
				URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.b = URH_Data[name].Colours.Bg.OutOfRange.b
				URH_Data[URH_PlayerKey].Colours.Bg.OutOfRange.a = URH_Data[name].Colours.Bg.OutOfRange.a

				URH_Data[URH_PlayerKey].Colours.Border = {}
				URH_Data[URH_PlayerKey].Colours.Border.Melee = {}
				URH_Data[URH_PlayerKey].Colours.Border.Melee.r = URH_Data[name].Colours.Border.Melee.r
				URH_Data[URH_PlayerKey].Colours.Border.Melee.g = URH_Data[name].Colours.Border.Melee.g
				URH_Data[URH_PlayerKey].Colours.Border.Melee.b = URH_Data[name].Colours.Border.Melee.b
				URH_Data[URH_PlayerKey].Colours.Border.Melee.a = URH_Data[name].Colours.Border.Melee.a
				URH_Data[URH_PlayerKey].Colours.Border.Ranged = {}
				URH_Data[URH_PlayerKey].Colours.Border.Ranged.r = URH_Data[name].Colours.Border.Ranged.r
				URH_Data[URH_PlayerKey].Colours.Border.Ranged.g = URH_Data[name].Colours.Border.Ranged.g
				URH_Data[URH_PlayerKey].Colours.Border.Ranged.b = URH_Data[name].Colours.Border.Ranged.b
				URH_Data[URH_PlayerKey].Colours.Border.Ranged.a = URH_Data[name].Colours.Border.Ranged.a
				URH_Data[URH_PlayerKey].Colours.Border.DeadZone = {}
				URH_Data[URH_PlayerKey].Colours.Border.DeadZone.r = URH_Data[name].Colours.Border.DeadZone.r
				URH_Data[URH_PlayerKey].Colours.Border.DeadZone.g = URH_Data[name].Colours.Border.DeadZone.g
				URH_Data[URH_PlayerKey].Colours.Border.DeadZone.b = URH_Data[name].Colours.Border.DeadZone.b
				URH_Data[URH_PlayerKey].Colours.Border.DeadZone.a = URH_Data[name].Colours.Border.DeadZone.a
				URH_Data[URH_PlayerKey].Colours.Border.OutOfRange = {}
				URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.r = URH_Data[name].Colours.Border.OutOfRange.r
				URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.g = URH_Data[name].Colours.Border.OutOfRange.g
				URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.b = URH_Data[name].Colours.Border.OutOfRange.b
				URH_Data[URH_PlayerKey].Colours.Border.OutOfRange.a = URH_Data[name].Colours.Border.OutOfRange.a

				URH_Data[URH_PlayerKey].Colours.Font = {}
				URH_Data[URH_PlayerKey].Colours.Font.Melee = {}
				URH_Data[URH_PlayerKey].Colours.Font.Melee.r = URH_Data[name].Colours.Font.Melee.r
				URH_Data[URH_PlayerKey].Colours.Font.Melee.g = URH_Data[name].Colours.Font.Melee.g
				URH_Data[URH_PlayerKey].Colours.Font.Melee.b = URH_Data[name].Colours.Font.Melee.b
				URH_Data[URH_PlayerKey].Colours.Font.Ranged = {}
				URH_Data[URH_PlayerKey].Colours.Font.Ranged.r = URH_Data[name].Colours.Font.Ranged.r
				URH_Data[URH_PlayerKey].Colours.Font.Ranged.g = URH_Data[name].Colours.Font.Ranged.g
				URH_Data[URH_PlayerKey].Colours.Font.Ranged.b = URH_Data[name].Colours.Font.Ranged.b
				URH_Data[URH_PlayerKey].Colours.Font.DeadZone = {}
				URH_Data[URH_PlayerKey].Colours.Font.DeadZone.r = URH_Data[name].Colours.Font.DeadZone.r
				URH_Data[URH_PlayerKey].Colours.Font.DeadZone.g = URH_Data[name].Colours.Font.DeadZone.g
				URH_Data[URH_PlayerKey].Colours.Font.DeadZone.b = URH_Data[name].Colours.Font.DeadZone.b
				URH_Data[URH_PlayerKey].Colours.Font.OutOfRange = {}
				URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.r = URH_Data[name].Colours.Font.OutOfRange.r
				URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.g = URH_Data[name].Colours.Font.OutOfRange.g
				URH_Data[URH_PlayerKey].Colours.Font.OutOfRange.b = URH_Data[name].Colours.Font.OutOfRange.b

				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.copiedProfile1.." ["..name.."] "..URH_TXT.copiedProfile2)
			else
				URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.dataMissing1.." ["..name.."] "..URH_TXT.dataMissing2)
			end
		else
			URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.profileNotFound1.." ["..name.."] "..URH_TXT.profileNotFound2)
		end
	else
		URH_Print(URH_HELP_COLOUR..URH_NAME..":|r "..URH_TXT.profileError)
	end
end
