-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 63272 $"):match("%d+")), ("$Date: 2008-03-01 02:05:29 -0500 (Sat, 01 Mar 2008) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[
This class provides an AceModuleCore framework for defining UnitStatus
extensions.  A UnitState module can be defined based on this framework
and be used by UAB (and a UAB user) to recognize when specific unit status
events have occurred.
--]]
uabUnitStatus = uab:NewModule("UnitStatus", "AceModuleCore-2.0")
uabUnitStatus:SetModuleMixins("AceEvent-2.0", "AceDebug-2.0", "AceModuleCore-2.0")

uabUnitStatus.core = uab
uabUnitStatus.unitstatus = {}
uabUnitStatus.globalunitstatus = {}
uabUnitStatus.statusgained = {}

function uabUnitStatus:OnInitialize()
    self.super.OnInitialize(self)
end

function uabUnitStatus:GetAllUnitStatus()
	return self.unitstatus
end

function uabUnitStatus:RegisterUnitStatus(group, menuname, module, profile, gainnotified)
    if not menuname then
        menuname = group
        group = "main"
    end
    
	if not self.unitstatus[group] then
        self.unitstatus[group] = {}
    end
    self.unitstatus[group][menuname] = true
    self.globalunitstatus[menuname] = { module = module, profile = profile, gainnotified = gainnotified, usedby = {} }
end

function uabUnitStatus:UnregisterUnitStatus(group, menuname)
    if not menuname then
        menuname = group
        group = "main"
    end
    
    if self.unitstatus[group] then
        self.unitstatus[group][menuname] = nil
        if self.globalunitstatus[menuname].gainnotified and self.statusgained[menuname] then
            for unit, _ in pairs(self.statusgained[menuname]) do
                self.statusgained[menuname][unit] = nil
            end
        end
        self.globalunitstatus[menuname] = nil
    end
end

function uabUnitStatus:StatusPriority(name)
	local status = self:GetUnitStatusConfig(name)
	
	return status and status.priority
end

function uabUnitStatus:StatusColor(name)
	local status = self:GetUnitStatusConfig(name)
	
	return status and status.color
end

function uabUnitStatus:IsStatusGained(name, unit, action)
    local info = self.globalunitstatus[name]
    if info then

--        if info.gainnotified then
--            if self.statusgained[name] and self.statusgained[name][unit] then
--                local status = self.statusgained[name][unit]
--                return true, status.enabled, status.start, status.duration
--            else
--                return nil
--            end
--        end

        local module = info.module
        if module and not module.db.profile.disabled then
            local profile = info.profile
            if profile and not profile.disabled then
                return module:IsStatusGained(name, unit, action)
            end
        end
    end
end

function uabUnitStatus:UnitStatusGained(name, unit, enabled, start, duration)
    if not self.statusgained[name] then
        self.statusgained[name] = {}
    end
    
    if not self.statusgained[name][unit] then
        self.statusgained[name][unit] = { enabled = enabled, start = start, duration = duration }
    else
        local status = self.statusgained[name][unit]
        
        status.enabled = enabled
        status.start = start
        status.duration = duration
    end

    self:TriggerEvent("UnitActionBars_UnitStatus_Gained", name, unit)
end

function uabUnitStatus:UnitStatusLost(name, unit)
    if self.statusgained[name] then
        self.statusgained[name][unit] = nil
    end

    self:TriggerEvent("UnitActionBars_UnitStatus_Lost", name, unit)
end

function uabUnitStatus:GetUnitStatusConfig(name)
    local info = self.globalunitstatus[name]
    if info then
        local module = info.module
        if module and not module.db.profile.disabled then
            local profile = info.profile
            if profile and not profile.disabled then
                return profile
            end
        end
    end
end

--[[
Call this method to indicate that the specified wrapper makes use of the specified unit effect.
--]]
function uabUnitStatus:SetEffectUse(uniteffect, wrapper)
    self:Debug("Wrapper uses unit status: "..uniteffect)
    local info = self.globalunitstatus[uniteffect]
    if info then
        if not info.usedby[wrapper] then
            info.usedby[wrapper] = true
            
            local module = info.module
            if module and not module.db.profile.disabled then
                local profile = info.profile
                if profile and not profile.disabled then
                    if not info.inuse then
                        info.inuse = true
                        self:Debug("Enabling unit status: "..uniteffect)
                        module:SetEffectUse(uniteffect)
                    end
                end
            end
        end
    end
end

--[[
Call this method to indicate that the specified wrapper no longer makes use of the specified unit effect.
--]]
function uabUnitStatus:ClearEffectUse(uniteffect, wrapper)
    self:Debug("Clearing wrapper use of unit status: "..uniteffect)
    local info = self.globalunitstatus[uniteffect]
    if info then
        if info.usedby[wrapper] then
            info.usedby[wrapper] = nil
            
            info.inuse = nil
            for w, _ in pairs(info.usedby) do
                info.inuse = true
                break
            end
            
            if not info.inuse then
                local module = info.module
                if module and not module.db.profile.disabled then
                    local profile = info.profile
                    if profile and not profile.disabled then
                        self:Debug("Disabling unit status: "..uniteffect)
                        module:ClearEffectUse(uniteffect)
                    end
                end
            end
        end
    end
end

uabUnitStatus.modulePrototype.core = uabUnitStatus
uabUnitStatus.modulePrototype.statusgained = {}
uabUnitStatus.modulePrototype.unitStatusDefaultDB = {
    disabled = nil,
    priority = 50,
    color = { r = 1, g = 1, b = 1, a = 1, }
}

function uabUnitStatus.modulePrototype:OnInitialize()
	if not self.db then
		self.core.core:RegisterDefaults(self.name, "profile", self.defaultDB or self.unitStatusDefaultDB)
		self.db = self.core.core:AcquireDBNamespace(self.name)
	end

    self.core.core.options.args.unitstatus.args[self.name] = self:CreateUnitStatusOptions()
    
    self:SetDebugging(self.db.profile.debugging)
    self.core.core.options.args.debug.args[self.core.name..":"..self.name] = {
        type = 'toggle',
        name = self.core.name..":"..self.name,
        desc = "Enable debugging for the "..self.core.name..":"..self.name.." module",
        get = function()
            return self:IsDebugging()
        end,
        set = function(v)
            self.db.profile.debugging = v
            self:SetDebugging(self.db.profile.debugging)
        end,
    }
end

function uabUnitStatus.modulePrototype:GetUnitClass(unit)
    local unitclass = UnitClass(unit)
    local class
    
    if unitclass then
        class = string.lower(unitclass)
    end
    
    return class
end

function uabUnitStatus.modulePrototype:RegisterUnitStatus(group, menuname, profile, gainnotified)
	uabUnitStatus:RegisterUnitStatus(group, menuname, self, profile, gainnotified)
end

function uabUnitStatus.modulePrototype:UnregisterUnitStatus(group, menuname, profile)
	uabUnitStatus:UnregisterUnitStatus(group, menuname, self, profile)
end

function uabUnitStatus.modulePrototype:UnitStatusGained(name, unit, enabled, start, duration)
    uabUnitStatus:UnitStatusGained(name, unit, enabled, start, duration)
end

function uabUnitStatus.modulePrototype:UnitStatusLost(name, unit)
    uabUnitStatus:UnitStatusLost(name, unit)
end

function uabUnitStatus.modulePrototype:IsStatusGained(name, unit, action)
	if name == self.name then
	    if not unit then return end
        
        local class = self:GetUnitClass(unit)
        
        if not class or self.db.profile.classfilter[class] then
    		return self.statusgained[unit]
        end
	end
end

function uabUnitStatus.modulePrototype:SetStatusGained(name, unit)
	if name == self.name then
	    if not unit then return end
        
        local class = self:GetUnitClass(unit)
        
        if not class or self.db.profile.classfilter[class] then
		    self.statusgained[unit] = true
            uabUnitStatus:UnitStatusGained(name, unit)
        end
	end
end

function uabUnitStatus.modulePrototype:SetStatusLost(name, unit)
	if name == self.name then
	    if not unit then return end
        
        local class = self:GetUnitClass(unit)
        
        if not class or self.db.profile.classfilter[class] then
    		self.statusgained[unit] = nil
            uabUnitStatus:UnitStatusLost(name, unit)
        end
	end
end

function uabUnitStatus.modulePrototype:CreateUnitStatusOptions()
    return self:CreateUnitStatusOption(self.db.profile, self.name, self.name, true)
end

function uabUnitStatus.modulePrototype:EnableUnitStatus(name, displayname)
end

function uabUnitStatus.modulePrototype:DisableUnitStatus(name, displayname)
end

--[[
Optional override
This method is called when the specified unit effect is initially used by at least one wrapper.
--]]
function uabUnitStatus.modulePrototype:SetEffectUse(name)
end

--[[
Optional override
This method is called when the specified unit effect is no longer used by any wrappers.
--]]
function uabUnitStatus.modulePrototype:ClearEffectUse(name)
end

function uabUnitStatus.modulePrototype:CreateUnitStatusOption(profile, name, displayname, addclassfilter)
    local options
    local me = self

    if not name then
        name = self.name
    end

    if not displayname then
        displayname = name
    end

    options = {
        type = 'group',
        name = name,
        desc = "Unit status settings for the "..displayname.." unit status",
        args = {
            enabled = {
                type = 'toggle',
                name = "Enabled",
                desc = "When checked, enable the use of the "..displayname.." unit status",
                order = 1,
                get = function()
                    return not profile.disabled
                end,
                set = function(v)
                    profile.disabled = not v
                    if profile.disabled then
                        self:DisableUnitStatus(name, displayname)
                    else
                        self:EnableUnitStatus(name, displayname)
                    end
                end,
            },
            
            spacing3 = {
                name = " ",
                type = 'header',
                order = 5,
            },

            priority = {
                type = 'range',
                name = "Priority",
                desc = displayname .. " priority",
                disabled = function() return name == "default" end,
                order = 10,
                min = 1,
                max = 100,
                step = 1,
                isPercent = false,
                get = function()
                    return profile.priority
                end,
                set = function(value)
                    profile.priority = value
                end,
            },

            highlight = {
                type = 'color',
                name = "Border highlight color",
                desc = "Set the border hightlight color when "..displayname.." is in effect",
                order = 20,
                hasAlpha = true,
                get = function()
                    local c = profile.color
                    return c.r, c.g, c.b, c.a
                end,
                set = function(r, g, b, a)
                    local c = profile.color
                    c.r, c.g, c.b, c.a = r, g, b, a
                end,
            },
        },
    }
    
    if addclassfilter then
        if not profile.classfilter then
            profile.classfilter = {
                druid = true,
                hunter = true,
                mage = true,
                paladin = true,
                priest = true,
                rogue = true,
                shaman = true,
                warlock = true,
                warrior = true,
            }
        end
        
        options.args.classfilter = {
            type = 'group',
            name = "Class Filter",
            desc = "Set the class filter for this status",
            order = 30,
            args = {
                druid = {
                    type = 'toggle',
                    name = "Druid",
                    desc = "Check the status for a druid",
                    order = 10,
                    get = function() return profile.classfilter.druid end,
                    set = function(v) profile.classfilter.druid = v end,
                },
                hunter = {
                    type = 'toggle',
                    name = "Hunter",
                    desc = "Check the status for a hunter",
                    order = 20,
                    get = function() return profile.classfilter.hunter end,
                    set = function(v) profile.classfilter.hunter = v end,
                },
                mage = {
                    type = 'toggle',
                    name = "Mage",
                    desc = "Check the status for a mage",
                    order = 30,
                    get = function() return profile.classfilter.mage end,
                    set = function(v) profile.classfilter.mage = v end,
                },
                paladin = {
                    type = 'toggle',
                    name = "Paladin",
                    desc = "Check the status for a paladin",
                    order = 40,
                    get = function() return profile.classfilter.paladin end,
                    set = function(v) profile.classfilter.paladin = v end,
                },
                priest = {
                    type = 'toggle',
                    name = "Priest",
                    desc = "Check the status for a priest",
                    order = 50,
                    get = function() return profile.classfilter.priest end,
                    set = function(v) profile.classfilter.priest = v end,
                },
                rogue = {
                    type = 'toggle',
                    name = "Rogue",
                    desc = "Check the status for a rogue",
                    order = 60,
                    get = function() return profile.classfilter.rogue end,
                    set = function(v) profile.classfilter.rogue = v end,
                },
                shaman = {
                    type = 'toggle',
                    name = "Shaman",
                    desc = "Check the status for a shaman",
                    order = 70,
                    get = function() return profile.classfilter.shaman end,
                    set = function(v) profile.classfilter.shaman = v end,
                },
                warlock = {
                    type = 'toggle',
                    name = "Warlock",
                    desc = "Check the status for a warlock",
                    order = 80,
                    get = function() return profile.classfilter.warlock end,
                    set = function(v) profile.classfilter.warlock = v end,
                },
                warrior = {
                    type = 'toggle',
                    name = "Warrior",
                    desc = "Check the status for a warrior",
                    order = 90,
                    get = function() return profile.classfilter.warrior end,
                    set = function(v) profile.classfilter.warrior = v end,
                },
            },
        }
    end
    
    return options
end


