-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 43089 $"):match("%d+")), ("$Date: 2007-07-07 15:10:03 -0500 (Sat, 07 Jul 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

uab.state_data = {}
uab.secureheaderobjs = {}
uab.framegroupobjs = {}
uab.activatorobjs = {}
uab.actionsetobjs = {}
uab.framegroup_data = {}
uab.unitgroup_data = {}
uab.postponedNewActivatorFrameControl = {}

--[[--------------------------------------------------------------------
 Methods that set or retrieve current object state information.  These
 methods are typically "get" or "set" methods.  They normally just
 perform a very simple retrieval of data, or setting of data.
--]]--------------------------------------------------------------------

function uab.CalculateState(activatorcode, unitgroupcode, framegroupcode, stancecode)
    local state

    state =         (unitgroupcode*uab.constants.unitgroupstates)
    state = state + (activatorcode*uab.constants.activatorstates)
    state = state + (stancecode*uab.constants.stancestates)
    state = state + (framegroupcode*uab.constants.framegroupstates)

    return state
end

--[[------------------------------------------------
---------------- SecureHeader Array ----------------
--]]------------------------------------------------
function uab:GetSecureHeaderObj(code)
    if not self.secureheaderobjs[code] then
        local activatorcode = floor(code /  self.constants.activatorstates)
        local unitgroupcode = floor((code - (activatorcode * self.constants.activatorstates)) / self.constants.unitgroupstates)
    
        self:SetSecureHeaderObj(code, self.classes.SecureHeader:new(activatorcode, unitgroupcode))
    end

    return self.secureheaderobjs[code]
end

function uab:SetSecureHeaderObj(code, obj)
    self.secureheaderobjs[code] = obj
end

function uab:GetSecureHeaderCode(activatorcode, unitgroupcode)
    if not unitgroupcode then
        unitgroupcode = 0
    end
    
    if not activatorcode then
        activatorcode = 0
    end

    local code = unitgroupcode * self.constants.unitgroupstates
    code = code + (activatorcode * self.constants.activatorstates)

    return code
end

--[[------------------------------------------------
----------- ActivatorFrameControl Array ------------
--]]------------------------------------------------
function uab:ActivatorFrameControlObjs(state)
    return self.state_data[state].activatorframecontrolobjs
end

--[[------------------------------------------------
----------------- FrameGroup Array -----------------
--]]------------------------------------------------
function uab:GetFrameGroupObj(code)
    return self.framegroupobjs[code]
end

function uab:SetFrameGroupObj(code, obj)
    self.framegroupobjs[code] = obj
end

function uab:GetFrameGroupCode(state)
    if self.state_data[state] and self.state_data[state].framegroupcode then
        return self.state_data[state].framegroupcode
    end
    
    -- The result of "state / X" where X is framegroupstates gives you the framegroup
    return floor(state / uab.constants.framegroupstates)
end

function uab:GetFrameGroupModuleName(code)
    local obj = self:GetFrameGroupObj(code)
    return obj:GetModuleName()
end

function uab:GetFrameGroupName(code)
    local obj = self:GetFrameGroupObj(code)
    return obj:GetName()
end

--[[------------------------------------------------
----------------- UnitGroup Array ------------------
--]]------------------------------------------------
function uab:GetUnitGroupCode(state)
    if self.state_data[state] and self.state_data[state].unitgroupcode then
        return self.state_data[state].unitgroupcode
    end
    
    -- The result of "state mod X" where X is framegroupstates gives you the state within the framegroup
    local group = math.fmod(state, uab.constants.framegroupstates)
    -- When you have a state value within a framegroup, the result of "state mod X" where X is the activatorstates gives you the state within the activator
    group = math.fmod(group, uab.constants.activatorstates)
    -- When you have a state value within an activator, the result of "state mod X" where X is the stancestates gives you the unitgroup
    return math.fmod(group, uab.constants.stancestates)
end    

function uab:GetUnitGroupName(code)
    return self.constants.unitgroupstatename[code]
end

--[[------------------------------------------------
----------------- Activator Array ------------------
--]]------------------------------------------------
function uab:GetActivatorCode(state)
    if self.state_data[state] and self.state_data[state].activatorcode then
        return self.state_data[state].activatorcode
    end
    
    -- The result of "state mod X" where X is framegroupstates gives you the state within the framegroup
    local activator = math.fmod(state, uab.constants.framegroupstates)
    -- When you have a state value within a framegroup, the result of "state / X" where X is the activatorstates gives you the activator
    return floor(activator / uab.constants.activatorstates)
end    

function uab:GetActivatorObj(code)
    return self.activatorobjs[code]
end

function uab:SetActivatorObj(code, activatorobj)
    self.activatorobjs[code] = activatorobj
end

--[[------------------------------------------------
----------------- Action Set Array -----------------
--]]------------------------------------------------
function uab:GetActionSetObj(code)
    return self.actionsetobjs[code]
end

function uab:SetActionSetObj(code, actionsetobj)
    self.actionsetobjs[code] = actionsetobj
end

function uab:GetActionSetCodeForName(name)
    for code, actionsetobj in pairs(self.actionsetobjs) do
        if uab:GetActionSetObj(code):GetName() == name then
            return code
        end
    end
end

function uab:ActionSetName(code)
    return uab:GetActionSetObj(code):GetName()
end

--[[------------------------------------------------
------------------- Stance Array -------------------
--]]------------------------------------------------
function uab:GetStanceCode(state)
    if self.state_data[state] and self.state_data[state].stancecode then
        return self.state_data[state].stancecode
    end

    -- The result of "state mod X" where X is framegroupstates gives you the state within the framegroup
    local stancecode = math.fmod(state, uab.constants.framegroupstates)
    -- When you have a state value within a framegroup, the result of "state mod X" where X is the activatorstates gives you the state within the activator
    stancecode = math.fmod(stancecode, uab.constants.activatorstates)
    -- When you have a state value within an activator, the result of "state / X" where X is the stancestates gives you the stance
    return floor(stancecode / uab.constants.stancestates)
end    

--[[--------------------------------------------------------------------
 Methods that operate on the object.  These methods either apply changes
 to this object, or to other objects based on changes in the add-on
 configuration.  These methods are "action" methods.  That is, their
 name indicates that an action is taking place.  For example:
 "DoProcessing", "ApplyNewestChanges", "SynchronizeComplicatedState"
--]]--------------------------------------------------------------------

--[[------------------------------------------------
---------------- FrameGroup Enable -----------------
--]]------------------------------------------------
function uab:DefaultFrameGroupEnabled(enabled)
    if enabled then
        -- If we enable the use of a default frame group, some frames that were ignored
        -- because they didn't match any enabled frame group might now need to be registered
        -- again to match the default frame group.
    else
        -- If we disable the use of a default frame group, frames that were registered
        -- to the default frame group now need to be ignored.
    end
end

function uab:FrameGroupEnabled(framegroupcode, enabled)
    if enabled then
        -- We need to act like this framegroupcode was just created.  Basically,
        -- this means that all settings for this framegroup should be used where
        -- appropriate, and any frame for this framegroup that is registered with
        -- the default framegroup should be unregistered and re-registered to
        -- this framegroup.
    else
        -- We need to act like this framegroupcode does not exist.  Basically,
        -- this means that all settings for this framegroup should be ignored,
        -- and any frame registered with this framegroup should be unregistered
        -- and re-registered to the default frame group (or not re-registered).
    end
end

--[[------------------------------------------------
---------------- DirectCast Change -----------------
--]]------------------------------------------------
function uab:ApplyDirectCastToActivatorFrameControl(state, ctrlobj, hostile_count, friendly_count)
    if not hostile_count or not friendly_count then
        local activatorcode = self.state_data[state].activatorcode
        local stancecode = self.state_data[state].stancecode
        local stanceobj = self.state_data[state].activatorobj:GetStanceObj(stancecode)

        if not hostile_count then
            hostile_count = stanceobj:NumHostileActions()
        end
        
        if not friendly_count then
            local unitgroupcode = self.state_data[state].unitgroupcode
            
            friendly_count = stanceobj:NumActions(unitgroupcode)
        end
    end
    
    ctrlobj:ClearHostileCast()

    if self.db.profile.directcast and hostile_count and hostile_count == 1 then
        ctrlobj:SetHostileDirectCast()
    elseif hostile_count > 0 then
        ctrlobj:SetHostileActionButtonCast()
    end
    
    if self.db.profile.directcast and friendly_count and friendly_count == 1 then
        ctrlobj:SetFriendlyDirectCast()
    else
        ctrlobj:SetFriendlyActionButtonCast()
    end
end

local function ApplyNewActivatorFrameControl(afc)
    local state_data = uab.state_data[afc.state]
    
    if state_data.relpoint ~= "cursor" and state_data.relpoint ~= "screen" then
        afc.afc:ReApplySetPointAttributes()
    end

    if afc.new then    
        state_data.secureheaderobj:StatesChanged()
    
        afc.asug:SetSecureHeaderFromCode(state_data.secureheadercode)
        afc.asug:ReApplyLayoutAttributes()        
    end

    uab:ApplyDirectCastToActivatorFrameControl(afc.state, afc.afc)
    afc.afc:SetSecureHeaderFromCode(state_data.secureheadercode)
end

local function ReApplyDirectCast()
    for state, data in pairs(uab.state_data) do
        local stanceobj = data.activatorobj:GetStanceObj(data.stancecode)
        local hostile_count = stanceobj:NumHostileActions()
        local friendly_count = stanceobj:NumActions(data.unitgroupcode)

        for _, ctrlobj in pairs(uab:ActivatorFrameControlObjs(state)) do
            uab:ApplyDirectCastToActivatorFrameControl(state, ctrlobj, hostile_count, friendly_count)
        end
    end
end

function uab:SetDirectCast()
    if not self.db.profile.directcast then
        self.db.profile.directcast = true

        ReApplyDirectCast()    
    end
end

function uab:ClearDirectCast()
    if self.db.profile.directcast then
        self.db.profile.directcast = nil

        ReApplyDirectCast()    
    end
end

function uab:ApplyDirectCast()
    ReApplyDirectCast()    
end

function uab:RemoveActivatorFrameControlObj(activatorframecontrolobj)
    local state = activatorframecontrolobj:GetState()
    for idx, val in pairs(self.state_data[state].activatorframecontrolobjs) do
        if val == activatorframecontrolobj then
            self.state_data[state].activatorframecontrolobjs[idx] = nil
        end
    end
end

--[[------------------------------------------------
----------- NewActivatorFrameControlObj ------------
--]]------------------------------------------------
function uab:NewActivatorFrameControlObj(activatorframecontrolobj)
    -- It is possible (likely) that more than one frame control will have the
    -- same state value.  This occurs when an activator is hooked to multiple
    -- frames in the same frame group (e.g. party frames).
    local state = activatorframecontrolobj:GetState()
    local state_data
    local newafc
    
    local activatorcode = self:GetActivatorCode(state)
    local stancecode = self:GetStanceCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    
    if self.state_data[state] then
        table.insert(self.state_data[state].activatorframecontrolobjs, activatorframecontrolobj)
        
        state_data = self.state_data[state]
    else
        state_data = {}
        
        newafc = true
        self.state_data[state] = {}
        state_data = self.state_data[state]
    
        state_data.scale = uab.db.profile.scale.default.value
        state_data.framegroupcode = framegroupcode
        state_data.unitgroupcode = unitgroupcode
        state_data.activatorcode = activatorcode
        state_data.activatorobj = self:GetActivatorObj(activatorcode)
        state_data.stancecode = stancecode
        state_data.activatorframecontrolobjs = {}
        table.insert(state_data.activatorframecontrolobjs, activatorframecontrolobj)

        state_data.activatorstanceunitgroupobj = state_data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)

        if uab.db.profile.activators[activatorcode].independent then
            if uab.db.profile.activators[activatorcode].independentgroup and
                uab.db.profile.activators[activatorcode].independentgroup[self:GetUnitGroupName(unitgroupcode)] then
                state_data.secureheadercode = uab:GetSecureHeaderCode(activatorcode, unitgroupcode)
            else
                state_data.secureheadercode = uab:GetSecureHeaderCode(activatorcode, nil)
            end
        else
            if uab.db.profile.independent[self:GetUnitGroupName(unitgroupcode)] then
                state_data.secureheadercode = uab:GetSecureHeaderCode(nil, activatorcode)
            else
                state_data.secureheadercode = uab:GetSecureHeaderCode(nil, nil)
            end
        end
    	
        state_data.secureheaderobj = uab:GetSecureHeaderObj(state_data.secureheadercode)

        state_data.scale = self:GetStateScale(state)
        state_data.relpoint = uab:GetStateRelPoint(state)
        state_data.point = uab:GetStatePoint(state)
        state_data.x = uab:GetStateOffsetX(state)
        state_data.y = uab:GetStateOffsetY(state)
        state_data.layoutmgrname, state_data.layoutname = uab:GetStateLayoutMgrAndLayoutName(state)
    end

    local activatorstanceunitgroupobj = state_data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
    local afc = { state = state, new = newafc, afc = activatorframecontrolobj, asug = activatorstanceunitgroupobj }

    if self.postponeApplyNewActivatorFrameControl then
        table.insert(self.postponedNewActivatorFrameControl, afc)
    else
        ApplyNewActivatorFrameControl(afc)
    end
end

--[[------------------------------------------------
---------------- Changing the Scale ----------------
--]]------------------------------------------------
function uab:GetStateScale(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)

    local scale = self.framegroup_data[framegroupcode] and self.framegroup_data[framegroupcode].scale
    if not scale then
        scale = self.unitgroup_data[unitgroupcode] and self.unitgroup_data[unitgroupcode].scale
        if not scale then
            scale = uab.db.profile.scale.default.value
        end
    end

    return scale
end

local function ApplyScaleAttribute(headers)
    for headerobj, _ in pairs(headers) do
        headerobj:ReApplyScaleAttribute()
    end
end

function uab:SetFrameGroupScale(framegroupcode, scale)
    if not self.framegroup_data[framegroupcode] then
        self.framegroup_data[framegroupcode] = {}
    end
    
    self.framegroup_data[framegroupcode].scale = scale

    local changed_headers = {}
    for state, data in pairs(self.state_data) do
        local fg = data.framegroupcode
        if fg == framegroupcode then
            data.scale = scale or self:GetStateScale(state)
            changed_headers[data.secureheaderobj] = true
        end
    end

    ApplyScaleAttribute(changed_headers)
end

function uab:SetUnitGroupScale(unitgroupcode, scale)
    if not self.unitgroup_data[unitgroupcode] then
        self.unitgroup_data[unitgroupcode] = {}
    end
    
    self.unitgroup_data[unitgroupcode].scale = scale
    if not scale then
        scale = uab.db.profile.scale.default.value
    end

    local changed_headers = {}
    for state, data in pairs(self.state_data) do
        local ucode = data.unitgroupcode
        if unitgroupcode == ucode then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].scale then
                data.scale = scale
                changed_headers[data.secureheaderobj] = true
            end
        end
    end

    ApplyScaleAttribute(changed_headers)
end

function uab:SetGlobalScale(scale)
    uab.db.profile.scale.default.value = scale

    local changed_headers = {}
    for state, data in pairs(self.state_data) do
        local unitgroupcode = data.unitgroupcode
        if not self.unitgroup_data[unitgroupcode] or not self.unitgroup_data[unitgroupcode].scale then
            local framegroupcode = data.framegroupcode
            if self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].scale then
                data.scale = scale
                changed_headers[data.secureheaderobj] = true
            end
        end
    end

    ApplyScaleAttribute(changed_headers)
end

--[[------------------------------------------------
----------- Changing the Relative Point ------------
--]]------------------------------------------------

function uab:GetStateRelPoint(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)

    local relpoint = self.framegroup_data[framegroupcode] and self.framegroup_data[framegroupcode].relpoint
    if not relpoint then
        relpoint = self.unitgroup_data[unitgroupcode] and self.unitgroup_data[unitgroupcode].relpoint
        if not relpoint then
            relpoint = uab.db.profile.position.default.relpoint
        end
    end

    return relpoint
end

local function ApplyPointAttribute(headers, ctrls)
    for headerobj, _ in pairs(headers) do
        headerobj:ReApplyRelPointAttribute()
    end

    for state, _ in pairs(ctrls) do
        for _, ctrlobj in pairs(uab:ActivatorFrameControlObjs(state)) do
            ctrlobj:ReApplySetPointAttributes()
        end
    end
end

function uab:SetFrameGroupRelPoint(framegroupcode, relpoint)
    if not self.framegroup_data[framegroupcode] then
        self.framegroup_data[framegroupcode] = {}
    end
    
    self.framegroup_data[framegroupcode].relpoint = relpoint

    local changed_ctrls = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local fg = data.framegroupcode
        if fg == framegroupcode then
            data.relpoint = relpoint or self:GetStateRelPoint(state)
            if data.relpoint == "cursor" or data.relpoint == "screen" then
                changed_headers[data.secureheaderobj] = true
            else
                changed_ctrls[state] = true
            end
        end
    end

    ApplyPointAttribute(changed_headers, changed_ctrls)
end

function uab:SetUnitGroupRelPoint(unitgroupcode, relpoint)
    if not self.unitgroup_data[unitgroupcode] then
        self.unitgroup_data[unitgroupcode] = {}
    end
    
    self.unitgroup_data[unitgroupcode].relpoint = relpoint
    if not relpoint then
        relpoint = uab.db.profile.position.default.relpoint
    end

    local changed_ctrls = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local ucode = data.unitgroupcode
        if unitgroupcode == ucode then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].relpoint then
                data.relpoint = relpoint
                changed_headers[data.secureheaderobj] = true
                if relpoint == "cursor" or relpoint == "screen" then
                    changed_headers[data.secureheaderobj] = true
                else
                    changed_ctrls[state] = true
                end
            end
        end
    end

    ApplyPointAttribute(changed_headers, changed_ctrls)
end

function uab:SetGlobalRelPoint(relpoint)
    uab.db.profile.position.default.relpoint = relpoint

    local changed_ctrls = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local unitgroupcode = data.unitgroupcode
        if not self.unitgroup_data[unitgroupcode] or not self.unitgroup_data[unitgroupcode].relpoint then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].relpoint then
                data.relpoint = relpoint
                changed_headers[data.secureheaderobj] = true
                if relpoint == "cursor" or relpoint == "screen" then
                    changed_headers[data.secureheaderobj] = true
                else
                    changed_ctrls[state] = true
                end
            end
        end
    end

    ApplyPointAttribute(changed_headers, changed_ctrls)
end
    
--[[------------------------------------------------
---------------- Changing the Point ----------------
--]]------------------------------------------------

function uab:GetStatePoint(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)

    local point = self.framegroup_data[framegroupcode] and self.framegroup_data[framegroupcode].point
    if not point then
        point = self.unitgroup_data[unitgroupcode] and self.unitgroup_data[unitgroupcode].point
        if not point then
            point = uab.db.profile.position.default.point
        end
    end

    return point
end

function uab:SetFrameGroupPoint(framegroupcode, point)
    if not self.framegroup_data[framegroupcode] then
        self.framegroup_data[framegroupcode] = {}
    end
    
    self.framegroup_data[framegroupcode].point = point

    local changed_ctrls = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local fg = data.framegroupcode
        if fg == framegroupcode then
            data.point = point or self:GetStatePoint(state)

            if not data.relpoint or data.relpoint == "cursor" or data.relpoint == "screen" then
                changed_headers[data.secureheaderobj] = true
            end
            if not data.relpoint or (data.relpoint ~= "cursor" and data.relpoint ~= "screen") then
                changed_ctrls[state] = true
            end
        end
    end

    ApplyPointAttribute(changed_headers, changed_ctrls)
end

function uab:SetUnitGroupPoint(unitgroupcode, point)
    if not self.unitgroup_data[unitgroupcode] then
        self.unitgroup_data[unitgroupcode] = {}
    end
    
    self.unitgroup_data[unitgroupcode].point = point
    if not point then
        point = uab.db.profile.position.default.point
    end

    local changed_ctrls = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local ucode = data.unitgroupcode
        if unitgroupcode == ucode then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].point then
                data.point = point
                changed_headers[data.secureheaderobj] = true
                if not data.relpoint or data.relpoint == "cursor" or data.relpoint == "screen" then
                    changed_headers[data.secureheaderobj] = true
                end
                if not data.relpoint or (data.relpoint ~= "cursor" and data.relpoint ~= "screen") then
                    changed_ctrls[state] = true
                end
            end
        end
    end

    ApplyPointAttribute(changed_headers, changed_ctrls)
end

function uab:SetGlobalPoint(point)
    uab.db.profile.position.default.point = point

    local changed_ctrls = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local unitgroupcode = data.unitgroupcode
        if not self.unitgroup_data[unitgroupcode] or not self.unitgroup_data[unitgroupcode].point then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].point then
                data.point = point
                if data.relpoint == "cursor" or data.relpoint == "screen" then
                    changed_headers[data.secureheaderobj] = true
                else
                    changed_ctrls[state] = true
                end
            end
        end
    end

    ApplyPointAttribute(changed_headers, changed_ctrls)
end

--[[------------------------------------------------
-------------- Changing the Offset X ---------------
--]]------------------------------------------------

function uab:GetStateOffsetX(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)

    local x = self.framegroup_data[framegroupcode] and self.framegroup_data[framegroupcode].x
    if not x then
        x = self.unitgroup_data[unitgroupcode] and self.unitgroup_data[unitgroupcode].x
        if not x then
            x = uab.db.profile.position.default.x
        end
    end

    return x
end

function uab:SetFrameGroupOffsetX(framegroupcode, x)
    if not self.framegroup_data[framegroupcode] then
        self.framegroup_data[framegroupcode] = {}
    end
    
    self.framegroup_data[framegroupcode].x = x

    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local fg = data.framegroupcode
        if fg == framegroupcode then
            data.x = x or self:GetStateOffsetX(state)
            changed_headers[data.secureheaderobj] = true
        end
    end

    for headerobj, _ in pairs(changed_headers) do
        headerobj:ReApplyOffsetXAttribute()
    end
end

function uab:SetUnitGroupOffsetX(unitgroupcode, x)
    if not self.unitgroup_data[unitgroupcode] then
        self.unitgroup_data[unitgroupcode] = {}
    end
    
    self.unitgroup_data[unitgroupcode].x = x
    if not x then
        x = uab.db.profile.position.default.x
    end

    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local ucode = data.unitgroupcode
        if unitgroupcode == ucode then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].point then
                data.x = x
                changed_headers[data.secureheaderobj] = true
            end
        end
    end

    for headerobj, _ in pairs(changed_headers) do
        headerobj:ReApplyOffsetXAttribute()
    end
end

function uab:SetGlobalOffsetX(x)
    uab.db.profile.position.default.x = x

    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local unitgroupcode = data.unitgroupcode
        if not self.unitgroup_data[unitgroupcode] or not self.unitgroup_data[unitgroupcode].point then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].point then
                data.x = x
                changed_headers[data.secureheaderobj] = true
            end
        end
    end

    for headerobj, _ in pairs(changed_headers) do
        headerobj:ReApplyOffsetXAttribute()
    end
end
    
--[[------------------------------------------------
-------------- Changing the Offset Y ---------------
--]]------------------------------------------------

function uab:GetStateOffsetY(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)

    local y = self.framegroup_data[framegroupcode] and self.framegroup_data[framegroupcode].y
    if not y then
        y = self.unitgroup_data[unitgroupcode] and self.unitgroup_data[unitgroupcode].y
        if not y then
            y = uab.db.profile.position.default.y
        end
    end

    return y
end

function uab:SetFrameGroupOffsetY(framegroupcode, y)
    if not self.framegroup_data[framegroupcode] then
        self.framegroup_data[framegroupcode] = {}
    end
    
    self.framegroup_data[framegroupcode].y = y
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local fg = data.framegroupcode
        if fg == framegroupcode then
            data.y = y or self:GetStateOffsetY(state)
            changed_headers[data.secureheaderobj] = true
        end
    end

    for headerobj, _ in pairs(changed_headers) do
        headerobj:ReApplyOffsetYAttribute()
    end
end

function uab:SetUnitGroupOffsetY(unitgroupcode, y)
    if not self.unitgroup_data[unitgroupcode] then
        self.unitgroup_data[unitgroupcode] = {}
    end
    
    self.unitgroup_data[unitgroupcode].y = y
    if not y then
        y = uab.db.profile.position.default.y
    end

    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local ucode = data.unitgroupcode
        if unitgroupcode == ucode then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].y then
                data.y = y
                changed_headers[data.secureheaderobj] = true
            end
        end
    end

    for headerobj, _ in pairs(changed_headers) do
        headerobj:ReApplyOffsetYAttribute()
    end
end

function uab:SetGlobalOffsetY(y)
    uab.db.profile.position.default.y = y

    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local unitgroupcode = data.unitgroupcode
        if not self.unitgroup_data[unitgroupcode] or not self.unitgroup_data[unitgroupcode].y then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].y then
                data.y = y
                changed_headers[data.secureheaderobj] = true
            end
        end
    end

    for headerobj, _ in pairs(changed_headers) do
        headerobj:ReApplyOffsetYAttribute()
    end
end

--[[------------------------------------------------
--------------- Changing the Layout ----------------
--]]------------------------------------------------

function uab:GetStateLayoutMgrAndLayoutName(state)
    local framegroupcode = self:GetFrameGroupCode(state)
    local unitgroupcode = self:GetUnitGroupCode(state)

    local layoutmgrname = self.framegroup_data[framegroupcode] and self.framegroup_data[framegroupcode].layoutmgrname
    local layoutname = "fgroup:"..uab:GetFrameGroupModuleName(framegroupcode)..":"..uab:GetFrameGroupName(framegroupcode)
    if not layoutmgrname then
        layoutmgrname = self.unitgroup_data[unitgroupcode] and self.unitgroup_data[unitgroupcode].layoutmgrname
        layoutname = "group:"..uab:GetUnitGroupName(unitgroupcode)
        
        if not layoutmgrname then
            layoutmgrname = uab.db.profile.layout
            layoutname = "main"
        end
    end
    
    return layoutmgrname, layoutname
end

local function ApplyLayoutAttribute(headers, asugobjs)
    for headercode, _ in pairs(headers) do
        uab:GetSecureHeaderObj(headercode):ReApplyLayoutAttributes()
    end
    
    for asugobj, _ in pairs(asugobjs) do
        asugobj:ReApplyLayoutAttributes()
    end
end

function uab:SetFrameGroupLayoutManagerName(framegroupcode, layoutmgrname)
    if not self.framegroup_data[framegroupcode] then
        self.framegroup_data[framegroupcode] = {}
    end
    
    self.framegroup_data[framegroupcode].layoutmgrname = layoutmgrname
    local layoutname = "fgroup:"..uab:GetFrameGroupModuleName(framegroupcode)..":"..uab:GetFrameGroupName(framegroupcode)
    
    local changedunitgroupobjs = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local fg = data.framegroupcode
        if fg == framegroupcode then
            if not layoutmgrname then
                data.layoutmgrname, data.layoutname = self:GetStateLayoutMgrAndLayoutName(state)
            else
                data.layoutmgrname = layoutmgrname
                data.layoutname = layoutname
            end
            
            local activatorcode = data.activatorcode
            local stancecode = data.stancecode
            local unitgroupcode = data.unitgroupcode
            
            local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
            changedunitgroupobjs[activatorstanceunitgroupobj] = true
            changed_headers[activatorstanceunitgroupobj:GetSecureHeaderCode()] = true
        end
    end

    ApplyLayoutAttribute(changed_headers, changedunitgroupobjs)
end

function uab:SetUnitGroupLayoutManagerName(unitgroupcode, layoutmgrname)
    if not self.unitgroup_data[unitgroupcode] then
        self.unitgroup_data[unitgroupcode] = {}
    end
    
    self.unitgroup_data[unitgroupcode].layoutmgrname = layoutmgrname
    local layoutname = "group:"..uab:GetUnitGroupName(unitgroupcode)
    
    if not layoutmgrname then
        layoutmgrname = uab.db.profile.layout
        layoutname = "main"
    end
    
    local changedunitgroupobjs = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local ucode = data.unitgroupcode
        if unitgroupcode == ucode then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].layoutmgrname then
                data.layoutmgrname = layoutmgrname
                data.layoutname = layoutname
                
                local activatorcode = data.activatorcode
                local stancecode = data.stancecode
                
                local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
                changedunitgroupobjs[activatorstanceunitgroupobj] = true
                changed_headers[activatorstanceunitgroupobj:GetSecureHeaderCode()] = true
            end
        end
    end

    ApplyLayoutAttribute(changed_headers, changedunitgroupobjs)
end

function uab:SetActionSetLayoutManagerName(actionsetcode, layoutmgrname)
    local actionsetname = self:ActionSetName(actionsetcode)
    local actionsetobj = self:GetActionSetObj(actionsetcode)

    uab.db.profile.actionsets[actionsetname].layout = layoutmgrname
    actionsetobj:SetLayoutManager(layoutmgrname)

    for code, activatorobj in pairs(self.activatorobjs) do
        if activatorobj:HasActionSetForCode(actionsetcode) then
            for stancecode = 0, uab.MaxCurrentClassStances(), 1 do
                for unitgroupcode, _ in pairs(self.constants.unitgroupstatename) do
                    activatorobj:GetUnitGroupObj(stancecode, unitgroupcode):ReApplyLayoutAttributes()
                end
            end
        end
    end
end

function uab:SetGlobalLayoutManagerName(layoutmgrname)
    uab.db.profile.layoutmgrname = layoutmgrname
    
    local layoutname = "main"
    local changedunitgroupobjs = {}
    local changed_headers = {}
    
    for state, data in pairs(self.state_data) do
        local unitgroupcode = data.unitgroupcode
        if not self.unitgroup_data[unitgroupcode] or not self.unitgroup_data[unitgroupcode].layoutmgrname then
            local framegroupcode = data.framegroupcode
            if not self.framegroup_data[framegroupcode] or not self.framegroup_data[framegroupcode].layoutmgrname then
                data.layoutmgrname = layoutmgrname
                data.layoutname = layoutname

                local activatorcode = data.activatorcode
                local stancecode = data.stancecode
                local unitgroupcode = data.unitgroupcode
                
                local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
                changedunitgroupobjs[activatorstanceunitgroupobj] = true
                changed_headers[activatorstanceunitgroupobj:GetSecureHeaderCode()] = true
            end
        end
    end

    ApplyLayoutAttribute(changed_headers, changedunitgroupobjs)
end

function uab:LayoutSettingsChanged(layoutmgrname, layoutname)
    local changedunitgroupobjs = {}
    local changed_headers = {}
    local found

    for state, data in pairs(self.state_data) do
        if data.layoutmgrname == layoutmgrname and data.layoutname == layoutname then
            local activatorcode = data.activatorcode
            local stancecode = data.stancecode
            local unitgroupcode = data.unitgroupcode
            
            local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
        
            changedunitgroupobjs[activatorstanceunitgroupobj] = true
            changed_headers[activatorstanceunitgroupobj:GetSecureHeaderCode()] = true
            found = true
        end
    end

    if not found then
        -- If no state matches the layoutname, check to see if it matches
        -- a layout.  If so, then add the activatorstanceunitgroup instances
        -- for all activators associated with that action set.
        local _, _, actionsetname = string.find(layoutname, "^actions:(.+)$")
        if actionsetname then
            local code = self:GetActionSetCodeForName(actionsetname)
            if code then
                for activatorcode, activatorobj in pairs(self.activatorobjs) do
                    if activatorobj:HasActionSetForCode(code) then
                        for stancecode = 0, uab.MaxCurrentClassStances(), 1 do
                            for unitgroupcode, _ in pairs(self.constants.unitgroupstatename) do
                                local activatorstanceunitgroupobj = activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
                                changedunitgroupobjs[activatorstanceunitgroupobj] = true
                            end
                        end
                    end
                end
            end
        end
    end

    ApplyLayoutAttribute(changed_headers, changedunitgroupobjs)
end

--[[------------------------------------------------
-------------------- (private) ---------------------
--------------- SecureHeader Change ----------------
--]]------------------------------------------------

local function SetSecureHeaderForStates(activatorcode, unitgroupcode)
    local changedheaders = {}
    local changedunitgroupobjs = {}
    
    for state, data in pairs(uab.state_data) do
        local a = data.activatorcode
        local ug = data.unitgroupcode
        local newheadercode
        
        if a == activatorcode or activatorcode == nil then
            if ug == unitgroupcode or unitgroupcode == nil then
                local ugname = uab:GetUnitGroupName(ug)
                
                if uab.db.profile.activators[a].independent then
                    if uab.db.profile.activators[a].independentgroup and
                        uab.db.profile.activators[a].independentgroup[ugname] then
                        newheadercode = uab:GetSecureHeaderCode(a, ug)
                    else
                        newheadercode = uab:GetSecureHeaderCode(a, nil)
                    end
                else
                    if uab.db.profile.independent[ugname] then
                        newheadercode = uab:GetSecureHeaderCode(nil, ug)
                    else
                        newheadercode = uab:GetSecureHeaderCode(nil, nil)
                    end
                end

                if newheadercode ~= data.secureheadercode then
                    local newheaderobj = uab:GetSecureHeaderObj(newheadercode)
                    
                    if not changedheaders[newheaderobj] then
                        changedheaders[newheaderobj] = true
                    end
                    
                    if not changedheaders[data.secureheaderobj] then
                        changedheaders[data.secureheaderobj] = true
                    end
                    
                    data.secureheadercode = newheadercode
                    data.secureheaderobj = uab:GetSecureHeaderObj(newheadercode)
                    
                    local activatorcode = data.activatorcode
                    local unitgroupcode = data.unitgroupcode
                    local stancecode = data.stancecode
                    local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)
        
                    if not changedunitgroupobjs[activatorstanceunitgroupobj] then
                        changedunitgroupobjs[activatorstanceunitgroupobj] = state
                    end
        
                    for _, ctrlobj in pairs(uab:ActivatorFrameControlObjs(state)) do
                        ctrlobj:SetSecureHeaderFromCode(data.secureheadercode)
                    end
                end
            end
        end
    end
    
    for headerobj, _ in pairs(changedheaders) do
        headerobj:StatesChanged()
    end

    for activatorstanceunitgroupobj, state in pairs(changedunitgroupobjs) do
        activatorstanceunitgroupobj:SetSecureHeaderFromCode(uab.state_data[state].secureheadercode)
        activatorstanceunitgroupobj:ReApplyLayoutAttributes()
    end
end

--[[------------------------------------------------
---------------- Independent Change ----------------
--]]------------------------------------------------

function uab:ToggleIndependentUnitGroupSecureHeader(unitgroupcode)
    local ugname = self:GetUnitGroupName(unitgroupcode)
                
    if self.db.profile.independent[ugname] then
        self.db.profile.independent[ugname] = nil
    else
        self.db.profile.independent[ugname] = true
    end
    
    SetSecureHeaderForStates(nil, unitgroupcode)
end

function uab:ToggleIndependentActivatorSecureHeader(activatorcode)
    if self.db.profile.activators[activatorcode] and self.db.profile.activators[activatorcode].independent then
        self.db.profile.activators[activatorcode].independent = nil
    else
        if not self.db.profile.activators[activatorcode] then
            self.db.profile.activators[activatorcode] = {}
        end

        self.db.profile.activators[activatorcode].independent = true
    end

    SetSecureHeaderForStates(activatorcode, nil)
end

function uab:ClearIndependentUnitGroupSecureHeader(unitgroupcode)
    self.db.profile.independent[self:GetUnitGroupName(unitgroupcode)] = nil
    
    SetSecureHeaderForStates(nil, unitgroupcode)
end

function uab:ClearIndependentActivatorSecureHeader(activatorcode)
    if self.db.profile.activators[activatorcode] then
        self.db.profile.activators[activatorcode].independent = nil
    end
    
    SetSecureHeaderForStates(activatorcode, nil)
end

function uab:SetIndependentUnitGroupSecureHeader(unitgroupcode)
    self.db.profile.independent[self:GetUnitGroupName(unitgroupcode)] = true
    
    SetSecureHeaderForStates(nil, unitgroupcode)
end

function uab:SetIndependentActivatorSecureHeader(activatorcode)
    if not self.db.profile.activators[activatorcode] then
        self.db.profile.activators[activatorcode] = {}
    end
    self.db.profile.activators[activatorcode].independent = true

    SetSecureHeaderForStates(activatorcode, nil)
end

function uab:SetIndependentActivatorUnitGroupSecureHeader(activatorcode, unitgroupcode)
    if not uab.db.profile.activators[activatorcode] then
        uab.db.profile.activators[activatorcode] = {}
    end
    
    if not uab.db.profile.activators[activatorcode].independentgroup then
        uab.db.profile.activators[activatorcode].independentgroup = {}
    end

    uab.db.profile.activators[activatorcode].independentgroup[self:GetUnitGroupName(unitgroupcode)] = true
    SetSecureHeaderForStates(activatorcode, unitgroupcode)
end

function uab:ClearIndependentActivatorUnitGroupSecureHeader(activatorcode, unitgroupcode)
    if uab.db.profile.activators[activatorcode] and uab.db.profile.activators[activatorcode].independentgroup then
        uab.db.profile.activators[activatorcode].independentgroup[self:GetUnitGroupName(unitgroupcode)] = nil
    end

    SetSecureHeaderForStates(activatorcode, unitgroupcode)
end

function uab:ApplyIndependent()
    SetSecureHeaderForStates(nil, nil)
end

--[[------------------------------------------------
----------------- StayOpen Change ------------------
--]]------------------------------------------------

function uab:ApplyStayOpen()
    self:IterateActivators("ReApplyStayOpen")
end

function uab:SetGlobalStayOpen()
    if not self.db.profile.stayopen then
        self.db.profile.stayopen = true
    end

    self:ApplyStayOpen()
end

function uab:ClearGlobalStayOpen()
    if self.db.profile.stayopen then
        self.db.profile.stayopen = nil
    end

    self:ApplyStayOpen()
end

function uab:DefaultActivatorStayOpen(activatorcode)
    if self.db.profile.activators[activatorcode] and self.db.profile.activators[activatorcode].stayopen then
        self.db.profile.activators[activatorcode].stayopen.enabled = nil
        self.db.profile.activators[activatorcode].stayopen = nil
    end

    self:GetActivatorObj(activatorcode):ReApplyStayOpen()
end

function uab:SetActivatorStayOpen(activatorcode)
    if not self.db.profile.activators[activatorcode] then
        self.db.profile.activators[activatorcode] = {}
    end

    if not self.db.profile.activators[activatorcode].stayopen then
        self.db.profile.activators[activatorcode].stayopen = {}
    end

    self.db.profile.activators[activatorcode].stayopen.enabled = true

    self:GetActivatorObj(activatorcode):ReApplyStayOpen()
end

function uab:ClearActivatorStayOpen(activatorcode)
    if not self.db.profile.activators[activatorcode] then
        self.db.profile.activators[activatorcode] = {}
    end

    if not self.db.profile.activators[activatorcode].stayopen then
        self.db.profile.activators[activatorcode].stayopen = {}
    end

    self.db.profile.activators[activatorcode].stayopen.enabled = nil

    self:GetActivatorObj(activatorcode):ReApplyStayOpen()
end

--[[------------------------------------------------
----------------- ActionSet Change -----------------
--]]------------------------------------------------

function uab:ActionSetChanged(actionsetcode)
    local changedunitgroupobjs = {}

    for state, data in pairs(self.state_data) do
        local activatorcode = data.activatorcode
        local stancecode = data.stancecode
        local unitgroupcode = data.unitgroupcode

        local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)

        if activatorstanceunitgroupobj:IsActionSetForCode(actionsetcode) then
            changedunitgroupobjs[activatorstanceunitgroupobj] = true
        end
    end
    
    for activatorstanceunitgroupobj, _ in pairs(changedunitgroupobjs) do
        activatorstanceunitgroupobj:ReApplyLayoutAttributes()
    end
    
    self:ApplyDirectCast()
end

function uab:GetActionData(asname, stancecode, unitgroupcode, actionindex)
    local actionsetdata = self.db.profile.actionsets[asname]
    local stancedata = actionsetdata.stances[stancecode]
    
    if unitgroupcode ~= -1 then
        groupname = uab.constants.unitgroupstatename[unitgroupcode]
    else
        groupname = "hostile"
    end
    
    local unitgroupdata = stancedata[groupname]

    return unitgroupdata[actionindex]
end

function uab:GetWrapper(actionsetobj, stancecode, unitgroupcode, actionindex)
    local wrapper
     
    if unitgroupcode ~= -1 then
        wrapper = actionsetobj:GetActions(stancecode, unitgroupcode)[actionindex]
    else
        wrapper = actionsetobj:GetHostileActions(stancecode)[actionindex]
    end
    
    return wrapper
end

function uab:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)
    local actionsetobj = self:GetActionSetObj(actionsetcode)
    local wrapper = self:GetWrapper(actionsetobj, stancecode, unitgroupcode, actionindex)
    local actiondata = self:GetActionData(actionsetobj:GetName(), stancecode, unitgroupcode, actionindex)
    
    return wrapper, actiondata
end     

function uab:SetActionDefaultHideUnusable(actionsetcode, stancecode, unitgroupcode, actionindex)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)
    
    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    wrapper:SetDefaultHideUnusable()

    actiondata.sethideunusable = nil
end
    
function uab:ClearActionDefaultHideUnusable(actionsetcode, stancecode, unitgroupcode, actionindex)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)
     
    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    wrapper:ClearDefaultHideUnusable()

    actiondata.sethideunusable = true
end

function uab:SetActionHideUnusable(actionsetcode, stancecode, unitgroupcode, actionindex)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)
     
    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    wrapper:SetHideUnusable()

    actiondata.hideunusable = true
end
    
function uab:ClearActionHideUnusable(actionsetcode, stancecode, unitgroupcode, actionindex)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    wrapper:ClearHideUnusable()

    actiondata.hideunusable = nil
end

function uab:SetActionUsePlayerUnit(actionsetcode, stancecode, unitgroupcode, actionindex)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    wrapper:SetUsePlayerUnit()

    actiondata.useplayerunit = true
end
    
function uab:ClearActionUsePlayerUnit(actionsetcode, stancecode, unitgroupcode, actionindex)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    wrapper:ClearUsePlayerUnit()

    actiondata.useplayerunit = nil
end

function uab:SetActionUnitStatusVisualEffect(actionsetcode, stancecode, unitgroupcode, actionindex, unitstatus, visualeffect)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if not wrapper.effects[unitstatus] then
        wrapper.effects[unitstatus] = {}
    end

    local neweffects = {}
    for idx, veffect in pairs(wrapper.effects[unitstatus]) do
        neweffects[idx] = veffect
    end

    neweffects[visualeffect] = true

    wrapper:SetUnitEffectVisualEffects(unitstatus, neweffects)

    if not actiondata.effect[unitstatus] then
        actiondata.effect[unitstatus] = {}
    end

    actiondata.effect[unitstatus][visualeffect] = true
end
    
function uab:ClearActionUnitStatusVisualEffect(actionsetcode, stancecode, unitgroupcode, actionindex, unitstatus, visualeffect)
    local wrapper
    local actiondata

    wrapper, actiondata = self:GetActionInfo(actionsetcode, stancecode, unitgroupcode, actionindex)

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if wrapper.effects[unitstatus] then
        local neweffects = {}
        for idx, veffect in pairs(wrapper.effects[unitstatus]) do
            neweffects[idx] = veffect
        end
    
        neweffects[visualeffect] = nil
    
        wrapper:SetUnitEffectVisualEffects(unitstatus, neweffects)
    end

    if actiondata.effect[unitstatus] then
        actiondata.effect[unitstatus][visualeffect] = nil

        -- if there are no more visual effects for this unit status, then
        -- clear that unit status array

        local any
        for effect, _ in pairs(actiondata.effect[unitstatus]) do
            any = true
        end

        if not any then
            actiondata.effect[unitstatus] = nil
        end
    end
end
    
--[[------------------------------------------------
------------- Traverse all Action Sets -------------
--]]------------------------------------------------

function uab:IterateActionSets(fcn, params)
    for code, actionsetobj in pairs(self.actionsetobjs) do
        actionsetobj[fcn](actionsetobj, params)
    end
end
    
--[[------------------------------------------------
------------- Traverse all Activators --------------
--]]------------------------------------------------

function uab:IterateActivatorsCallGlobal(fcn, params)
    for code, activatorobj in pairs(self.activatorobjs) do
        fcn(activatorobj, params)
    end
end

function uab:IterateActivators(fcn, params)
    for code, activatorobj in pairs(self.activatorobjs) do
        activatorobj[fcn](activatorobj, params)
    end
end

function uab:StartPostponeApplyNewActivatorFrameControl()
    self.postponeApplyNewActivatorFrameControl = true
    
    for idx, afc in pairs(self.postponedNewActivatorFrameControl) do
        self.postponedNewActivatorFrameControl[idx].state = nil
        self.postponedNewActivatorFrameControl[idx].afc = nil
        self.postponedNewActivatorFrameControl[idx].asug = nil
        self.postponedNewActivatorFrameControl[idx] = nil
    end
end

function uab:EndPostponeApplyNewActivatorFrameControl()
    local afc_applied = {}
    local secureheader_applied = {}
    local asug_applied = {}
    
    self.postponeApplyNewActivatorFrameControl = nil
    
    for idx, afc_info in pairs(self.postponedNewActivatorFrameControl) do
        local state_data = self.state_data[afc_info.state]
        local secureheaderobj = self:GetSecureHeaderObj(state_data.secureheadercode)
        
        if not afc_applied[afc_info.afc] then
            afc_applied[afc_info.afc] = true
            if state_data.relpoint ~= "cursor" and state_data.relpoint ~= "screen" then
                afc_info.afc:ReApplySetPointAttributes()
            end

            uab:ApplyDirectCastToActivatorFrameControl(afc_info.state, afc_info.afc)
            afc_info.afc:SetSecureHeaderFromCode(state_data.secureheadercode)
        end
        
        if not secureheader_applied[secureheaderobj] then
            secureheader_applied[secureheaderobj] = true
            secureheaderobj:StatesChanged()
        end
        
        if not asug_applied[afc_info.asug] then
            asug_applied[afc_info.asug] = true
    
            afc_info.asug:SetSecureHeaderFromCode(state_data.secureheadercode)
            afc_info.asug:ReApplyLayoutAttributes()        
        end
    end

    for idx, _ in pairs(self.postponedNewActivatorFrameControl) do
        self.postponedNewActivatorFrameControl[idx].state = nil
        self.postponedNewActivatorFrameControl[idx].afc = nil
        self.postponedNewActivatorFrameControl[idx].asug = nil
        self.postponedNewActivatorFrameControl[idx] = nil
    end

    for p, _ in pairs(afc_applied) do
        afc_applied[p] = nil
    end
    
    for p, _ in pairs(secureheader_applied) do
        secureheader_applied[p] = nil
    end

    for p, _ in pairs(asug_applied) do
        asug_applied[p] = nil
    end
end

function uab:AssociateActivatorToActionSet(activatorcode, actionsetcode)
    local activatorobj = self:GetActivatorObj(activatorcode)
    local actionsetobj = self:GetActionSetObj(actionsetcode)
    local pending = self:SetPendingAttributes()
    
    self:StartPostponeApplyNewActivatorFrameControl()

    if not self.db.profile.activators[activatorcode] then
        self.db.profile.activators[activatorcode] = {}
    end

    self.db.profile.activators[activatorcode].inuse = true

    if not self.db.profile.activators[activatorcode].actionsets then
        self.db.profile.activators[activatorcode].actionsets = {}
    end

    table.insert(self.db.profile.activators[activatorcode].actionsets, actionsetobj:GetName())
    activatorobj:AddActionSetCode(actionsetcode)

    local changedunitgroupobjs = {}

    for state, data in pairs(self.state_data) do
        if data.activatorcode == activatorcode then
            local stancecode = data.stancecode
            local unitgroupcode = data.unitgroupcode
    
            local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)

            changedunitgroupobjs[activatorstanceunitgroupobj] = true
        end
    end
    
    for activatorstanceunitgroupobj, _ in pairs(changedunitgroupobjs) do
        activatorstanceunitgroupobj:ReApplyLayoutAttributes()
    end
    
    self:EndPostponeApplyNewActivatorFrameControl()

    if pending then    
        self:ClearPendingAttributes()
    end
end

function uab:DisassociateActivatorToActionSet(activatorcode, actionsetcode)
    local activatorobj = self:GetActivatorObj(activatorcode)
    local actionsetobj = self:GetActionSetObj(actionsetcode)

    if self.db.profile.activators[activatorcode] then
        if not self.db.profile.activators[activatorcode].actionsets then
            self.db.profile.activators[activatorcode].actionsets = {}
        end
    end

    local changedunitgroupobjs = {}

    for state, data in pairs(self.state_data) do
        if data.activatorcode == activatorcode then
            local stancecode = data.stancecode
            local unitgroupcode = data.unitgroupcode
    
            local activatorstanceunitgroupobj = data.activatorobj:GetUnitGroupObj(stancecode, unitgroupcode)

            changedunitgroupobjs[activatorstanceunitgroupobj] = true
        end
    end

    if self.db.profile.activators[activatorcode] then
        self.db.profile.activators[activatorcode].inuse = nil
        
        for idx, name in pairs(self.db.profile.activators[activatorcode].actionsets) do
            if name == actionsetobj:GetName() then
                self.db.profile.activators[activatorcode].actionsets[idx] = nil
            else
                self.db.profile.activators[activatorcode].inuse = true
            end
        end
    end

    activatorobj:RemoveActionSetCode(actionsetcode)
    
    for activatorstanceunitgroupobj, _ in pairs(changedunitgroupobjs) do
        activatorstanceunitgroupobj:ReApplyLayoutAttributes()
    end
end

function uab:DeleteActionSetByCode(actionsetcode)
    for code, activatorobj in pairs(self.activatorobjs) do
        if activatorobj:HasActionSetForCode(actionsetcode) then
            uab:DisassociateActivatorToActionSet(activatorobj.activatorcode, actionsetcode)
        end
    end

    local name = self:GetActionSetObj(actionsetcode):GetName()

    uab.ClearTable(self.db.profile.actionsets[name])
    self.db.profile.actionsets[name] = nil
    self.actionsetobjs[actionsetcode] = nil
end

