-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 76273 $"):match("%d+")), ("$Date: 2008-06-08 14:57:21 -0400 (Sun, 08 Jun 2008) $"):match("%d%d%d%d%-%d%d%-%d%d"))

uabSpellAction = uab:NewModule("SpellAction", "AceEvent-2.0")

uabSpellAction.core = uab
uabSpellAction.spell_indexes = {}

function uabSpellAction:OnInitialize()
	uabSpellAction.super.OnInitialize(self)
    self:RegisterEvent("SPELLS_CHANGED", "LoadSpells") -- some spells added/removed
end

function uabSpellAction:GetRankTemplate(rank)
    local template = "Rank (%d+)"

    if rank then
        local start, finish = string.find(rank, "Rank (%d+)")
        if start then
            if (string.len(rank) - (start-1)) > finish then
                local afterstd = string.sub(rank, finish+1)
                local len = string.len(afterstd)
        
                if len > 0 then
                    template = "Rank (%d+)" .. string.sub(afterstd, 1, len)
                end
            end
        else
            template = rank
        end
    end

    return template
end

function uabSpellAction:GetRankFormatTemplate(rank)
    local template = "Rank %d"

    if rank then
        local start, finish = string.find(rank, "Rank (%d+)")
        if start then
            if (string.len(rank) - (start-1)) > finish then
                local afterstd = string.sub(rank, finish+1)
                local len = string.len(afterstd)
        
                if len > 0 then
                    template = "Rank %d" .. string.sub(afterstd, 1, len)
                end
            end
        else
            template = rank
        end
    end
    
    return template
end

function uabSpellAction:LoadSpells()
    self.spellsloaded = true

    -- Either we have not cached the spell indexes yet, or the spells have changed.  Cache them now

    for name, data in pairs(self.spell_indexes) do
        data.maxindex = nil
        data.maxrank = nil
        if data.ranks then
            for ranktemplate, ranks in pairs(data.ranks) do
                for rank, _ in pairs(ranks) do
                    ranks[rank] = nil
                end
    
                data.ranks[ranktemplate] = nil
            end
            data.ranks = nil
        end
        self.spell_indexes[name] = nil
    end

    self.spell_indexes = {}
    
    local idx = 0

    while true do
        idx = idx + 1
        local name, rank = GetSpellName(idx, BOOKTYPE_SPELL)
        local texture

        if not name then break end

        if not self.spell_indexes[name] then
            self.spell_indexes[name] = {}
        end

        if rank then
            local start, _, thisrank = string.find(rank, "Rank (%d+)")
            if start then
                if not self.spell_indexes[name].ranks then
                    self.spell_indexes[name].ranks = {}
                end
    
                local ranknum = tonumber(thisrank)
                local ranktemplate = self:GetRankTemplate(rank)
                
                if not self.spell_indexes[name].ranks[ranktemplate] then
                    self.spell_indexes[name].ranks[ranktemplate] = {}
                    self.spell_indexes[name].ranks[ranktemplate].formattemplate = self:GetRankFormatTemplate(rank)
                end

                if not self.spell_indexes[name].ranks[ranktemplate].ranks then
                    self.spell_indexes[name].ranks[ranktemplate].ranks = {}
                end
    
                self.spell_indexes[name].ranks[ranktemplate].ranks[ranknum] = idx
                if not self.spell_indexes[name].ranks[ranktemplate].maxrank or ranknum > self.spell_indexes[name].ranks[ranktemplate].maxrank then
                    self.spell_indexes[name].ranks[ranktemplate].maxrank = ranknum
                    self.spell_indexes[name].ranks[ranktemplate].maxindex = idx
                    if not self.spell_indexes[name].maxrank or ranknum > self.spell_indexes[name].maxrank then
                        self.spell_indexes[name].maxrank = ranknum
                        self.spell_indexes[name].maxindex = idx
                    end
                end
            else
                self.spell_indexes[name].maxindex = idx
            end
        else
            self.spell_indexes[name].maxindex = idx
        end
    end
end

function uabSpellAction:GetSpellIdx(spellname, spellranktemplate, spellranktext)
    if not self.spellsloaded then
        self:LoadSpells()
    end

    if self.spell_indexes[spellname] then
        if self.spell_indexes[spellname].ranks then
            if self.spell_indexes[spellname].ranks[spellranktemplate] then
                if not spellranktext then
                    return self.spell_indexes[spellname].ranks[spellranktemplate].maxindex
                else
                    local _, _, rank = string.find(spellranktext, "Rank (%d+)")
                    local ranknum = tonumber(rank)
                    
                    if self.spell_indexes[spellname].ranks[spellranktemplate].ranks[ranknum] then
                        return self.spell_indexes[spellname].ranks[spellranktemplate].ranks[ranknum]
                    else
                        return self.spell_indexes[spellname].ranks[spellranktemplate].maxindex
                    end
                end
            else
                return self.spell_indexes[spellname].maxindex
            end
        else
            return self.spell_indexes[spellname].maxindex
        end
    end
end

function uabSpellAction:GetMaxRank(spellname, ranktemplate)
    if not self.spellsloaded then
        self:LoadSpells()
    end

    if self.spell_indexes[spellname] then
        local spell = self.spell_indexes[spellname]

        if not ranktemplate then
            ranktemplate = "Rank (%d+)"
        end
        
        if spell.ranks and spell.ranks[ranktemplate] then
            if spell.ranks[ranktemplate].maxrank then
                return string.format(spell.ranks[ranktemplate].formattemplate, spell.ranks[ranktemplate].maxrank)
            end
        end
    end
end

function uabSpellAction:SetTooltipSpell(tooltip, name, ranktext, usemaxrank)
    if not self.spellsloaded then
        self:LoadSpells()
    end

    local spellidx
    local start
    local ranktemplate = self:GetRankTemplate(ranktext)

    if ranktext then
        start, _, _ = string.find(ranktext, "Rank (%d+)")
    end
    
    if start and not usemaxrank then
        -- not using max rank spell, so find this exact spell in the list

        spellidx = self:GetSpellIdx(name, ranktemplate, ranktext)
    else
        -- using max rank spell, so find a matching spell, with the highest rank (using ranktemplate as a template to match the rank string)

        spellidx = self:GetSpellIdx(name, ranktemplate)
    end

    if spellidx then
        local sname, srank = GetSpellName(spellidx, BOOKTYPE_SPELL)

        tooltip:SetSpell(spellidx, SpellBookFrame.bookType)

        self:Debug("SetTooltipSpell: ("..sname..", "..(srank or "nil")..", "..spellidx..")")

        return sname, srank, spellidx
    else
        self:Debug("SetTooltipSpell: ("..name..", "..(ranktext or "nil")..", not found)")
    end
end

uab.classes.SpellAction = uab.mixins.aceOO.Class(uab.classes.BaseAction, "AceEvent-2.0")
uab.classes.SpellAction.gratuity = AceLibrary("Gratuity-2.0")

function uab.classes.SpellAction.IsActive(name, rank)
    local ranktemplate = uabSpellAction:GetRankTemplate(rank)
    local spellidx = uabSpellAction:GetSpellIdx(name, ranktemplate, rank)

    if spellidx then
        local active
        if IsCurrentCast then
            active = IsCurrentCast(spellidx, BOOKTYPE_SPELL)
        else
            active = IsCurrentSpell(spellidx, BOOKTYPE_SPELL)
        end
        return active
    end
end

function uab.classes.SpellAction.GetCount(name, ranktext, userank)
    -- set the tooltip so that we can get the "reagents" text from it, if there are reagents.
    uabSpellAction:SetTooltipSpell(uab.classes.SpellAction.gratuity, name, ranktext, usemaxrank)

    local count    
    local reagent = select(3, uab.classes.SpellAction.gratuity:Find(SPELL_REAGENTS.."(.+)", 2, 99, false, true))

    if reagent then
        if reagent:sub(0, 2) == "|c" then
            reagent = reagent:sub(11, -3)
        end

        count = 0

        -- now, try to locate the reagent in inventory/bags
        local itemCount = uab.classes.ItemAction.GetCount(reagent)

        if itemCount then
            count = tonumber(itemCount)
        end
    end
    
    return count
end

function uab.classes.SpellAction.GetCooldown(name, rank)
    local spellidx = uabSpellAction:GetSpellIdx(name, uabSpellAction:GetRankTemplate(rank), rank)

    if spellidx then    
        return GetSpellCooldown(spellidx, BOOKTYPE_SPELL)
    end
end

function uab.classes.SpellAction.prototype:init(data)
	uab.classes.SpellAction.super.prototype.init(self)
	
	self.data = data
	if self.data.max then
		self.text =	self.data.name
	elseif self.data.rank then
		self.text = self.data.name..self.data.rank
	else
		self.text = self.data.name
	end

    self.attribs = {}

    self.attribs["type"] = "spell"
    self.attribs["spell"] = self.text
    
	self:SetIcon()
end

function uab.classes.SpellAction.prototype:Dump(dumper)
    dumper:Dump("Spell Action")
    dumper:IncIndent()
    dumper:Dump("Name     : "..self.text)
    dumper:Dump("Icon     : "..self.data.icon)
    dumper:Dump("Spell    : "..self.data.name)
    dumper:Dump("Rank     : "..(self.data.rank or "nil"))
    dumper:Dump("Max      : "..((self.data.max and "true") or "false"))
    dumper:Dump("Attributes")
    dumper:IncIndent()
    
    for attr, val in pairs(self.attribs) do
        dumper:Dump(attr.." : "..val)
    end
    
    dumper:DecIndent()
    dumper:DecIndent()
end

-- Override
function uab.classes.SpellAction.prototype:ShowTooltip(frame)
    GameTooltip:SetOwner(frame, "ANCHOR_RIGHT")
    GameTooltip:SetBackdropColor(0.1, 0.1, 0.1, 0.8)
    
    local name, srank, spellidx = uabSpellAction:SetTooltipSpell(GameTooltip, self.data.name, self.data.rank, self.data.max)

    if srank then
        if not GameTooltipTextRight1:GetText() or not GameTooltipTextRight1:IsVisible() then
            GameTooltipTextRight1:SetText(srank)
            GameTooltipTextRight1:Show()
            GameTooltipTextRight1:ClearAllPoints()
            GameTooltipTextRight1:SetPoint("TOPRIGHT", GameTooltip, "TOPRIGHT", -10, -10)
            GameTooltipTextRight1:SetTextColor(0.5, 0.5, 0.5, 1)
        end
    else
        GameTooltipTextRight1:SetText()
        GameTooltipTextRight1:Hide()
    end
end

-- Override
function uab.classes.SpellAction.prototype:GetCount()
    return uab.classes.SpellAction.GetCount(self.data.name, self.data.rank, self.data.max)
end

-- Override
function uab.classes.SpellAction.prototype:GetAttributes()
    return self.attribs
end

-- Override
function uab.classes.SpellAction.prototype:IsUsable()
    return IsUsableSpell(self:GetSpellName())
end

-- Override
function uab.classes.SpellAction.prototype:GetClassName()
	return "SpellAction"
end

-- Override
function uab.classes.SpellAction.prototype:IsActive()
    return uab.classes.SpellAction.IsActive(self.data.name, self.data.rank)
end

-- Override
function uab.classes.SpellAction.prototype:GetCooldown()
    return uab.classes.SpellAction.GetCooldown(self.data.name, self.data.rank)
end

-- Override
function uab.classes.SpellAction.prototype:GetSpellName()
	return self.data.name
end

function uab.classes.SpellAction.prototype:GetHasRank()
    if uabSpellAction:GetMaxRank(self.data.name, uabSpellAction:GetRankTemplate(self.data.rank)) then
        return true
    end
end

function uab.classes.SpellAction.prototype:GetRank()
	if not self.data.max then
        if self.data.rank then
            return self.data.rank
        end
    end
end

-- Override optional
function uab.classes.SpellAction.prototype:GetRankTemplate()
    return uabSpellAction:GetRankTemplate(self.data.rank)
end

-- Override optional
function uab.classes.SpellAction.prototype:GetRankText(idx)
    local template = uabSpellAction:GetRankFormatTemplate(self.data.rank)

    return string.format(template, idx)
end

-- Override
function uab.classes.SpellAction.prototype:PickupAction()
    local spellidx = uabSpellAction:GetSpellIdx(self.data.name, uabSpellAction:GetRankTemplate(self.data.rank), self.data.rank)
    
    if spellidx then
        PickupSpell(spellidx, BOOKTYPE_SPELL)
        return true
    end
end

--[[---------------------------------------------------------------------------
Helper methods, to be used only by the class itself
-----------------------------------------------------------------------------]]
-- Override
function uab.classes.SpellAction.prototype:SetIcon()
    self.icon = self.data.icon

    if not self.icon then
        local spellidx = uabSpellAction:GetSpellIdx(self.data.name, uabSpellAction:GetRankTemplate(self.data.rank), self.data.rank)
    
        if spellidx then
            self.icon = GetSpellTexture(spellidx, BOOKTYPE_SPELL)
        end
    end
    
    self.data.icon = self.icon
end

