-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 57743 $"):match("%d+")), ("$Date: 2007-12-31 20:23:52 -0500 (Mon, 31 Dec 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[
A SecureHeader presents an interface to the the Blizzard/WoW secure header
frame for the functionality that UAB needs.
--]]
uab.classes.SecureHeader = uab.mixins.aceOO.Class()
uab.classes.SecureHeader.headers = {}

function uab.classes.SecureHeader:Dump(dumper)
    for header, _ in pairs(self.headers) do
        header:Dump(dumper)
    end
end

function uab.classes.SecureHeader:ApplyPendingAttributes()
    for header, _ in pairs(self.headers) do
        header:ApplyPendingAttributes()
    end
end

function uab.classes.SecureHeader:HideAllHeaders()
    for header, _ in pairs(self.headers) do
        header:Hide()
    end
end

function uab.classes.SecureHeader:SetStanceState()
    for header, _ in pairs(self.headers) do
        header:SetStanceState()
    end
end

function uab.classes.SecureHeader.prototype:init(activatorcode, unitgroupcode)
	uab.classes.SecureHeader.super.prototype.init(self)

    self.pendingattributes = {}
    self.activatorcode = activatorcode
    self.unitgroupcode = unitgroupcode

    self.secureheadercode = self.unitgroupcode*uab.constants.unitgroupstates
    self.secureheadercode = self.secureheadercode + (self.activatorcode*uab.constants.activatorstates)

	self.name = "UAB_SH_"..self.secureheadercode

	--[[
	Create the header frame.
	--]]

    self.framename = self.name
    
    self.frame = getglobal(self.framename)
    if not self.frame then
        self.frame = CreateFrame("Frame", self.framename, UIParent, "SecureStateHeaderTemplate")
    end

    local frame = self.frame

    frame.uabobj = self

    uab.SetFrameLevelAndStrata(frame, 1)
    
    self:SetAttribute(frame, "state", "0")

    self:SetAttribute(frame, "useparent-unit", true)
    self:SetAttribute(frame, "useparent-unitsuffix", true)
    self:SetAttribute(frame, "statemap-anchor", "$input")
    
    uab.classes.SecureHeader.headers[self] = true

--    self.cancelobj = uab.classes.ActionButtonCancel:new(self, self.secureheadercode)
    
    self:SetStatemapStance()
    self:SetStanceState()
end

function uab.classes.SecureHeader.prototype:Dump(dumper)
    dumper:Dump("SecureHeader : "..self.name)
    dumper:IncIndent()
    
    local parent = self.frame:GetParent()
    local parentname
    
    if parent then
        parentname = parent:GetName()
    end
    
    if not parentname then
        parentname = "UIParent"
    end
    
    dumper:Dump("Code  : "..self.secureheadercode)
    dumper:Dump("Parent: "..parentname)
    dumper:DumpFrameAttribute(self.frame, "state")
    dumper:DumpFrameAttribute(self.frame, "useparent-unit")
    dumper:DumpFrameAttribute(self.frame, "useparent-unitsuffix")
    dumper:DumpFrameAttribute(self.frame, "statemap-anchor")
    
    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            dumper:DumpFrameAttribute(self.frame, "statemap-anchor-"..state)
        end
    end

    for stance = 0, uab.MaxCurrentClassStances(), 1 do
        dumper:DumpFrameAttribute(self.frame, "statemap-stance-"..stance)
    end
    
    dumper:DumpFrameAttribute(self.frame, "headscale")
    dumper:DumpFrameAttribute(self.frame, "headofspoint")
    dumper:DumpFrameAttribute(self.frame, "headofsrelpoint")
    dumper:DumpFrameAttribute(self.frame, "headofsx")
    dumper:DumpFrameAttribute(self.frame, "headofsy")

    dumper:DecIndent()
end

function uab.classes.SecureHeader.prototype:Hide()
    self:SetAttribute(self.frame, "state", "0")
end

function uab.classes.SecureHeader.prototype:ApplyPendingAttributes()
	for name, val in pairs(self.pendingattributes) do
		for frame, data in pairs(val) do
    		uab.SetFrameAttribute(frame, name, data.val)
            val[frame] = nil
        end
        self.pendingattributes[name] = nil
	end
end

function uab.classes.SecureHeader.prototype:SetAttribute(frame, attribute, val)
    if uab:IsPendingAttributes() then
        if not self.pendingattributes[attribute] then
            self.pendingattributes[attribute] = {}
        end
        self.pendingattributes[attribute][frame] = { val = val }
    else
        self:ApplyPendingAttributes()
        uab.SetFrameAttribute(frame, attribute, val)
    end
end

function uab.classes.SecureHeader.prototype:StateChangedByBlizzard(newstate)
--    uab.print("SecureHeader ("..self.secureheadercode..") state changed to "..newstate)
end

function uab.classes.SecureHeader.prototype:SetStatemapStance()
    local old_states = {}

    local stance_map
    
    for stance = 1, uab.MaxCurrentClassStances(), 1 do
        local val = "[stance:"..stance.."] "..stance
        if not stance_map then
            stance_map = ""
        else
            stance_map = stance_map .. "; "
        end
        stance_map = stance_map .. val
    end

    if uab.MaxCurrentClassStances() > 0 then
        stance_map = stance_map .. "; 0"
    else
        stance_map = "0"
    end

    RegisterStateDriver(self.frame, "stance", stance_map)

    --[[
    Since the states have changed, we need to ensure that a stance change causes the correct
    state change to occur based on our current state, given the new states that we are associated with.
    
    For a stance change, the current state determines the new state based on the value for the
    statemap-stance-# attribute for the header (# is the stancecode).  The value for this attribute
    is a state-specific rule that is evaluated to determine the new state.  Stance changing does not change
    the activator, the framegroup, or the unitgroup.  The goal is to determine the correct set of buttons
    to display, given the new stance.  The buttons that are displayed are not affected by the framegroup
    but the framegroup does, or could, affect how the buttons are displayed (layout and positioning configuration).
    Therefore, we need to know all the states for a given combination of activator, framegroup, and unitgroup.  If
    we call this a "stategroup", then for each stategroup, we need to create a state-rule that indicates that if we
    are in any of the states in the stategroup, we should transition to the state in the stategroup associated with
    the new stance.  We need to do this for all of stategroups that we create and set the statemap-stance-#
    attribute for the header to a value that concatenates all of these rules, each separated from the next by a
    ';' character.
    
    Finally, we need to do this for each of the possible stances.
    --]]
    
    for stance = 0, uab.MaxCurrentClassStances(), 1 do
        local statemapstanceval
        
        --[[
            statemapstanceval is a state-mapped value.  A state-mapped
            value is a value that maps from a current state to a new
            state.  The value is a collection of "maps".  Each "map"
            converts a set of current states to a new state.  Each "map" 
            is formed as follows:
            
            oldstate0, oldstate1, oldstate2 : newstate
            
            oldstate0 - oldstate9 : these are all the states for
                a given activator, framegroup, unitgroup combination
            newstate : this is the state to transition to for that same
                activatr, framegroup, unitgroup combination for the
                current stance
        --]]

        for state, data in pairs(uab.state_data) do
            if data.secureheadercode == self.secureheadercode then
                local activatorcode = data.activatorcode
                local unitgroupcode = data.unitgroupcode
                local framegroupcode = data.framegroupcode

                --[[
                Add this state to the list of stance-independent states for
                this combination of activator, unitgroup, and framegroup
                --]]
                local stanceindependentstate = uab.CalculateState(activatorcode, unitgroupcode, framegroupcode, 0)
                if not old_states[stanceindependentstate] then
                    old_states[stanceindependentstate] = {}
                end
                
                old_states[stanceindependentstate][state] = true
            end            
        end

        --[[
        We now have all the states for each combination of activator, unitgroup,
        and framegroup.  For each combination, we need to build a value that looks
        like this:

            oldstate0, oldstate1, oldstate2 : newstate

        We then concatenate all of these values together (separated by ';') to build
        the entire attribute value.
        --]]

        for stanceindependentstate, states in pairs(old_states) do
            local activatorcode = uab:GetActivatorCode(stanceindependentstate)
            local unitgroupcode = uab:GetUnitGroupCode(stanceindependentstate)
            local framegroupcode = uab:GetFrameGroupCode(stanceindependentstate)
            local stancestate = uab.CalculateState(activatorcode, unitgroupcode, framegroupcode, stance)
            local stancestatemap

            for oldstate, _ in pairs(states) do
                if not stancestatemap then
                    stancestatemap = ""
                else
                    stancestatemap = stancestatemap .. ","
                end
                
                stancestatemap = stancestatemap .. oldstate
            end

            --[[
            We have all the old states separated by ','.  Add the stance-state
            to the end of the value, separated by ':'
            --]]
            stancestatemap = stancestatemap .. ":" .. stancestate
            
            --[[
            Append this "map" to the full value for this stance.
            --]]
            if not statemapstanceval then
                statemapstanceval = ""
            else
                statemapstanceval = statemapstanceval .. ";"
            end

            statemapstanceval = statemapstanceval .. stancestatemap
        end

        for state, states in pairs(old_states) do
            for newstate, _ in pairs(states) do
                states[newstate] = nil
            end
            old_states[state] = nil
        end

        if statemapstanceval then
            -- Set the attribute value for this state
            self:SetAttribute(self.frame, "statemap-stance-"..stance, statemapstanceval)
        end
    end
end

function uab.classes.SecureHeader.prototype:SetStanceState()
    self:SetAttribute(self.frame, "state", ""..GetShapeshiftForm(true))
end

local function AddStateScale(state, data, values)
    if not values.scale then
        values.scale = {}
    end

    if not values.scale[data.scale] then
        values.scale[data.scale] = ""
    else
        values.scale[data.scale] = values.scale[data.scale] .. ","
    end

    values.scale[data.scale] = values.scale[data.scale] .. state
end

local function ApplyStateScale(self, frame, values)
    local headscale

    if values.scale then
        for scale, states in pairs(values.scale) do
            if not headscale then
                headscale = ""
            else
                headscale = headscale .. ";"
            end
            
            headscale = headscale .. states .. ":" .. scale
        end
    end
    
    self:SetAttribute(frame, "headscale", headscale)
end

function uab.classes.SecureHeader.prototype:ReApplyScaleAttribute()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStateScale(state, data, values)
        end
    end

    ApplyStateScale(self, self.frame, values)
end

local function AddStateRelPoint(state, data, values)
    if not values.relpoint then
        values.relpoint = {}
    end

    relpoint = data.relpoint
    if relpoint ~= "cursor" and relpoint ~= "screen" then
        relpoint = "CENTER"
    end

    if not values.relpoint[relpoint] then
        values.relpoint[relpoint] = ""
    else
        values.relpoint[relpoint] = values.relpoint[relpoint] .. ","
    end

    values.relpoint[relpoint] = values.relpoint[relpoint] .. state
end

local function ApplyStateRelPoint(self, frame, values)
    local headofsrelpoint

    if values.relpoint then
        for relpoint, states in pairs(values.relpoint) do
            if not headofsrelpoint then
                headofsrelpoint = ""
            else
                headofsrelpoint = headofsrelpoint .. ";"
            end
            
            headofsrelpoint = headofsrelpoint .. states .. ":" .. relpoint
        end
    end
    
    self:SetAttribute(frame, "headofsrelpoint", headofsrelpoint)
end

function uab.classes.SecureHeader.prototype:ReApplyRelPointAttribute()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStateRelPoint(state, data, values)
        end
    end

    ApplyStateRelPoint(self, self.frame, values)
end

local function AddStatePoint(state, data, values)
    if not values.point then
        values.point = {}
    end

    local point = data.point

    if data.relpoint ~= "cursor" and data.relpoint ~= "screen" then
        point = "CENTER"
    end

    if not values.point[point] then
        values.point[point] = ""
    else
        values.point[point] = values.point[point] .. ","
    end

    values.point[point] = values.point[point] .. state
end

local function ApplyStatePoint(self, frame, values)
    local headofspoint

    if values.point then
        for point, states in pairs(values.point) do
            if not headofspoint then
                headofspoint = ""
            else
                headofspoint = headofspoint .. ";"
            end
            
            headofspoint = headofspoint .. states .. ":" .. point
        end
    end
    
    self:SetAttribute(frame, "headofspoint", headofspoint)
end

function uab.classes.SecureHeader.prototype:ReApplyPointAttribute()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStatePoint(state, data, values)
        end
    end

    ApplyStatePoint(self, self.frame, values)
end

local function AddStateOffsetX(state, data, values)
    if not values.x then
        values.x = {}
    end

    local x = data.x

    if data.relpoint ~= "cursor" and data.relpoint ~= "screen" then
        x = 0
    end

    if not values.x[x] then
        values.x[x] = ""
    else
        values.x[x] = values.x[x] .. ","
    end

    values.x[x] = values.x[x] .. state
end

local function ApplyStateOffsetX(self, frame, values)
    local headofsx

    if values.x then
        for x, states in pairs(values.x) do
            if not headofsx then
                headofsx = ""
            else
                headofsx = headofsx .. ";"
            end
            
            headofsx = headofsx .. states .. ":" .. x
        end
    end
    
    self:SetAttribute(frame, "headofsx", headofsx)
end

function uab.classes.SecureHeader.prototype:ReApplyOffsetXAttribute()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStateOffsetX(state, data, values)
        end
    end

    ApplyStateOffsetX(self, self.frame, values)
end

local function AddStateOffsetY(state, data, values)
    if not values.y then
        values.y = {}
    end

    local y = data.y

    if data.relpoint ~= "cursor" and data.relpoint ~= "screen" then
        y = 0
    end

    if not values.y[y] then
        values.y[y] = ""
    else
        values.y[y] = values.y[y] .. ","
    end

    values.y[y] = values.y[y] .. state
end

local function ApplyStateOffsetY(self, frame, values)
    local headofsy

    if values.y then
        for y, states in pairs(values.y) do
            if not headofsy then
                headofsy = ""
            else
                headofsy = headofsy .. ";"
            end
            
            headofsy = headofsy .. states .. ":" .. y
        end
    end
    
    self:SetAttribute(frame, "headofsy", headofsy)
end

function uab.classes.SecureHeader.prototype:ReApplyOffsetYAttribute()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStateOffsetY(state, data, values)
        end
    end

    ApplyStateOffsetY(self, self.frame, values)
end

function uab.classes.SecureHeader.prototype:ReApplyLayoutAttributes()
    -- the only attributes we are concerned with, are the attributes that
    -- apply to the cancel button.  There is one cancel button for each header
    -- and it shows/hides based on the state.  Pass this request to the cancel
    -- action button so that it can set the attributes appropriately.
--    self.cancelobj:ReApplyLayoutAttributes()
end

local function AddStateAnchor(state, data, values)
    if not values.anchor then
        values.anchor = {}
    end

    values.anchor[state] = true
end

local function ApplyStateAnchor(self, frame, values)
    --[[
    We need to indicate the behavior when an activatorframecontrol attempts to
    change the state of this header.  FastChangeUnit controls this.
    
    When it is enabled, a request to change to another state causes us to immediately
    change to that state unless we are in that state, in which case, we do nothing.
    
    When it is disabled, a request to change to another state causes us to hide
    if we are shown, but not in that state, do nothing if we are already in the
    state, and show if we are hidden.
    --]]

    if values.anchor then
        if not uab.db.profile.fastchangeunit then
            for state, _ in pairs(values.anchor) do
                self:SetAttribute(frame, "statemap-anchor-"..state, "0:$input;"..state..":"..state..";*:0")
            end
        else
            for state, _ in pairs(values.anchor) do
                self:SetAttribute(frame, "statemap-anchor-"..state, ""..state..":"..state..";*:$input")
            end
        end
    end
end

function uab.classes.SecureHeader.prototype:ReApplyFastChangeUnitAttributes()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStateAnchor(state, data, values)
        end
    end

    ApplyStateAnchor(self, self.frame, values)
end

function uab.classes.SecureHeader.prototype:StatesChanged()
    local values = {}

    for state, data in pairs(uab.state_data) do
        if data.secureheadercode == self.secureheadercode then
            AddStateScale(state, data, values)
            AddStateRelPoint(state, data, values)
            AddStatePoint(state, data, values)
            AddStateOffsetX(state, data, values)
            AddStateOffsetY(state, data, values)
            AddStateAnchor(state, data, values)
        end
    end

    ApplyStateScale(self, self.frame, values)
    ApplyStateRelPoint(self, self.frame, values)
    ApplyStatePoint(self, self.frame, values)
    ApplyStateOffsetX(self, self.frame, values)
    ApplyStateOffsetY(self, self.frame, values)
    ApplyStateAnchor(self, self.frame, values)
    
    self:ReApplyLayoutAttributes()
    self:SetStatemapStance()
end
    
function uab.classes.SecureHeader.prototype:GetFrame()
    return self.frame
end

