-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 42775 $"):match("%d+")), ("$Date: 2007-07-04 16:50:18 -0500 (Wed, 04 Jul 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[
This controls the configuration settings (and display of a sample set of
actions based on those settings) for each ActionSetLayout.  Each
BarActionSetLayout is associated with an BarActionSetLayoutConfig, and each
BarActionSetLayoutConfig refers to that same BarActionSetLayout instance.
--]]
uabOvalLayoutManager.LayoutConfig = uab.mixins.aceOO.Class(uabLayoutManager.interfaces.IActionSetLayoutConfig, uab.mixins.aceEvent)
uabOvalLayoutManager.LayoutConfig.count = 0

function uabOvalLayoutManager.LayoutConfig.prototype:init(layout, settings)
    uabOvalLayoutManager.LayoutConfig.super.prototype.init(self)

	self.settings = settings
    self.layout = layout

	uabOvalLayoutManager.LayoutConfig.count = uabOvalLayoutManager.LayoutConfig.count + 1
	self.number = uabOvalLayoutManager.LayoutConfig.count

	self:SettingsChanged()
end

function uabOvalLayoutManager.LayoutConfig.prototype:SettingsChanged(settings)
	if not self.settings then
		self.settings = {}
	end

    if not settings then
        settings = self.settings
    end
    
    self.settings.growClockwise = (settings and settings.growClockwise) or "true"
    self.settings.anchor = (settings and settings.anchor) or uabLayoutManager.constants.anchor.left
    self.settings.startLocation = (settings and settings.startLocation) or uabLayoutManager.constants.placement.topleft
    self.settings.hspacing = (settings and settings.hspacing) or 0
    self.settings.vspacing = (settings and settings.vspacing) or 0
    self.settings.horizontal = (settings and settings.horizontal) or 0
    self.settings.vertical = (settings and settings.vertical) or 0
    self.settings.hidecancel = (settings and settings.hidecancel)

    --[[
    Update the controls in the frame to show the new settings
    --]]
    
    self:SetGrowClockwise(self.settings.growClockwise)
    self:SetStartLocation(self.settings.startLocation)
    self:SetAnchor(self.settings.anchor)
    self:SetHSpacing(self.settings.hspacing)
    self:SetVSpacing(self.settings.vspacing)
    self:SetHorizontal(self.settings.horizontal)
    self:SetVertical(self.settings.vertical)
    self:SetHideCancel(self.settings.hidecancel)
end

function uabOvalLayoutManager.LayoutConfig.prototype:UseDefaults(bypassNotification)
	if not self.settings then
	    self.settings = {}
	end

    self.settings.growClockwise = "true"
    self.settings.anchor = uabLayoutManager.constants.anchor.left
    self.settings.startLocation = uabLayoutManager.constants.placement.topleft
    self.settings.hspacing = 0
    self.settings.vspacing = 0
    self.settings.horizontal = 0
    self.settings.vertical = 0
    self.settings.hidecancel = nil
    
	self:SettingsChanged()
	
    if not bypassNotification then
        self.layout:LayoutSettingsChanged()
    end
end

function uabOvalLayoutManager.LayoutConfig.prototype:MenuOptionArgs()
	--[[
	Build an AceOptions table of the options to be used to configure this
	config instance.  This table is inserted into a parent menu as a set of
	options.  If there are several options, just define them at the top level
	of the returned table.  Do not create a 'group' at the top level to enclose
	the options.
	--]]
	
	local options = {
        
        clockwise = {
            type = 'toggle',
            name = "Grow clockwise",
            desc = "When buttons are added, add them in a clockwise direction",
            order = 105,
            get = function()
                return self.settings.growClockwise == "true"
            end,
            set = function(v)
                if v then
					self:SetGrowClockwise("true")
				else
				    self:SetGrowClockwise("false")
                end
            end,
        },
        startLocation = {
            type = 'text',
            name = "First button location",
            desc = "Placement of the first button, relative to the cancel button",
            order = 110,
            get = function()
                return uabLayoutManager.PlacementDisplayValue(self.settings.startLocation)
            end,
            set = function(start)
                if start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.topleft) then
                    self:SetStartLocation(uabLayoutManager.constants.placement.topleft)
                elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.top) then
                    self:SetStartLocation(uabLayoutManager.constants.placement.top)
                elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.topright) then
                    self:SetStartLocation(uabLayoutManager.constants.placement.topright)
				elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.right) then
                	self:SetStartLocation(uabLayoutManager.constants.placement.right)
				elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottomright) then
                	self:SetStartLocation(uabLayoutManager.constants.placement.bottomright)
				elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottom) then
                	self:SetStartLocation(uabLayoutManager.constants.placement.bottom)
				elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottomleft) then
                	self:SetStartLocation(uabLayoutManager.constants.placement.bottomleft)
				elseif start == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.left) then
                	self:SetStartLocation(uabLayoutManager.constants.placement.left)
				end
            end,

            validate = {
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.topleft),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.top),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.topright),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.right),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottomright),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottom),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottomleft),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.left),
			},
        },
        anchor = {
            type = 'text',
            name = "Anchor location",
            desc = "Long side of the oval on the left/right or top/bottom",
            order = 120,
            get = function()
                return uabLayoutManager.AnchorDisplayValue(self.settings.anchor)
            end,
            set = function(anchor)
                if anchor == uabLayoutManager.AnchorDisplayValue(uabLayoutManager.constants.anchor.left) then
	                self:SetAnchor(uabLayoutManager.constants.anchor.left)
				elseif anchor == uabLayoutManager.AnchorDisplayValue(uabLayoutManager.constants.anchor.top) then
	                self:SetAnchor(uabLayoutManager.constants.anchor.top)
				end
            end,

            validate = {
				uabLayoutManager.AnchorDisplayValue(uabLayoutManager.constants.anchor.left),
				uabLayoutManager.AnchorDisplayValue(uabLayoutManager.constants.anchor.top),
			},
        },
        
        space1 = {
            type = 'header',
            name = " ",
            order = 140,
        },
        
        hspacing = {
            type = 'range',
            name = "Horizontal Spacing",
            desc = "Amount of horizontal space between buttons",
            order = 144,
            min = -20,
            max = 20,
            step = 1,
            isPercent = false,
            get = function()
                return self.settings.hspacing
            end,
            set = function(v)
                self:SetHSpacing(v)
            end,
        },
        vspacing = {
            type = 'range',
            name = "Vertical Spacing",
            desc = "Amount of vertical space between buttons",
            order = 148,
            min = -20,
            max = 20,
            step = 1,
            isPercent = false,
            get = function()
                return self.settings.vspacing
            end,
            set = function(v)
                self:SetVSpacing(v)
            end,
        },
	}
	
	return options
end

--[[---------------------------------------------------------------------------
Helper methods, to be used only by the class itself
-----------------------------------------------------------------------------]]

function uabOvalLayoutManager.LayoutConfig.prototype:SetStartLocation(value)
    local notify = (value ~= self.settings.startLocation)
        
    self.settings.startLocation = value
    if self.settings.anchor == uabLayoutManager.constants.anchor.top then
        if self.settings.startLocation == uabLayoutManager.constants.placement.left or self.settings.startLocation == uabLayoutManager.constants.placement.right then
            self:SetAnchor(uabLayoutManager.constants.anchor.left)
        end
    elseif value == uabLayoutManager.constants.anchor.left then
        if self.settings.startLocation == uabLayoutManager.constants.placement.top or self.settings.startLocation == uabLayoutManager.constants.placement.bottom then
            self:SetAnchor(uabLayoutManager.constants.anchor.top)
        end
    end
    
    if notify then
        self.layout:LayoutSettingsChanged()
    end        
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetAnchor(value)
    local notify = (value ~= self.settings.anchor)
    
    self.settings.anchor = value
    
    if self.settings.anchor == uabLayoutManager.constants.anchor.top then
        if self.settings.startLocation == uabLayoutManager.constants.placement.left or self.settings.startLocation == uabLayoutManager.constants.placement.right then
            self:SetStartLocation(uabLayoutManager.constants.placement.top..self.settings.startLocation)
        end
    elseif value == uabLayoutManager.constants.anchor.left then
        if self.settings.startLocation == uabLayoutManager.constants.placement.top or self.settings.startLocation == uabLayoutManager.constants.placement.bottom then
            self:SetStartLocation(self.settings.startLocation..uabLayoutManager.constants.placement.left)
        end
    end
    
    if notify then
        self.layout:LayoutSettingsChanged()
    end        
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetGrowClockwise(value)
    local newval
    
    if value and (tonumber(value) == 1 or value == "true") then
        newval = "true"
    else
        newval = "false"
    end
    
    local notify = (newval ~= self.settings.growClockwise)
        
    self.settings.growClockwise = newval
    
    if notify then
        self.layout:LayoutSettingsChanged()
    end        
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetHSpacing(value)
    local notify = (value ~= self.settings.hspacing)

    self.settings.hspacing = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetVSpacing(value)
    local notify = (value ~= self.settings.vspacing)

    self.settings.vspacing = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetHorizontal(value)
    local notify = (value ~= self.settings.horizontal)

    self.settings.horizontal = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetVertical(value)
    local notify = (value ~= self.settings.vertical)

    self.settings.vertical = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabOvalLayoutManager.LayoutConfig.prototype:SetHideCancel(value)
    local notify = (value ~= self.settings.hidecancel)

    self.settings.hidecancel = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabOvalLayoutManager.LayoutConfig.prototype:GetHorizontal()
    return self.settings.horizontal or 0
end

function uabOvalLayoutManager.LayoutConfig.prototype:GetVertical()
    return self.settings.vertical or 0
end

function uabOvalLayoutManager.LayoutConfig.prototype:GetHideCancel()
    return self.settings.hidecancel
end


