-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 63272 $"):match("%d+")), ("$Date: 2008-03-01 02:05:29 -0500 (Sat, 01 Mar 2008) $"):match("%d%d%d%d%-%d%d%-%d%d"))

uabLayoutManager = uab:NewModule("LayoutManager", "AceModuleCore-2.0")
uabLayoutManager:SetModuleMixins("AceDebug-2.0", "AceModuleCore-2.0")

uabLayoutManager.constants = {}

-- Layout anchor side constants
uabLayoutManager.constants.anchor = {}
uabLayoutManager.constants.anchor.top = "top"
uabLayoutManager.constants.anchor.left = "left"

function uabLayoutManager.AnchorDisplayValue(value)
    if value == uabLayoutManager.constants.anchor.top then
        return "Top & Bottom"
    else
        return "Left & Right"
    end
end

-- Layout placement position constants
uabLayoutManager.constants.placement = {}
uabLayoutManager.constants.placement.left = "l"
uabLayoutManager.constants.placement.topleft = "tl"
uabLayoutManager.constants.placement.top = "t"
uabLayoutManager.constants.placement.topright = "tr"
uabLayoutManager.constants.placement.right = "r"
uabLayoutManager.constants.placement.bottomright = "br"
uabLayoutManager.constants.placement.bottom = "b"
uabLayoutManager.constants.placement.bottomleft = "bl"

function uabLayoutManager.PlacementDisplayValue(value)
    if value == uabLayoutManager.constants.placement.top then
        return "Top"
    elseif value == uabLayoutManager.constants.placement.topright then
        return "Top Right"
    elseif value == uabLayoutManager.constants.placement.topleft then
        return "Top Left"
    elseif value == uabLayoutManager.constants.placement.bottom then
        return "Bottom"
    elseif value == uabLayoutManager.constants.placement.bottomright then
        return "Bottom Right"
    elseif value == uabLayoutManager.constants.placement.bottomleft then
        return "Bottom Left"
    elseif value == uabLayoutManager.constants.placement.left then
        return "Left"
    else
        return "Right"
    end
end

-- Layout grow direction constants
uabLayoutManager.constants.grow = {}
uabLayoutManager.constants.grow.up = "u"
uabLayoutManager.constants.grow.down = "d"
uabLayoutManager.constants.grow.left = "l"
uabLayoutManager.constants.grow.right = "r"

function uabLayoutManager.GrowDirectionDisplayValue(value)
    if value == uabLayoutManager.constants.grow.up then
        return "Up"
    elseif value == uabLayoutManager.constants.grow.down then
        return "Down"
    elseif value == uabLayoutManager.constants.grow.left then
        return "Left"
    else
        return "Right"
    end
end

uabLayoutManager.interfaces = {}

uabLayoutManager.interfaces.IActionSetLayout = uab.mixins.aceOO.Interface {
	ArrangeButtons = "function",
	GetConfig = "function",
	SetSettings = "function",
}

uabLayoutManager.interfaces.IActionSetLayoutConfig = uab.mixins.aceOO.Interface {
	SettingsChanged = "function", -- SettingsChanged(settings) or SettingsChanged()
	UseDefaults = "function", -- UseDefaults()
	MenuOptionArgs = "function", -- MenuOptionArgs()
    GetHorizontal = "function", -- GetHorizontal()
    GetVertical = "function", -- GetVertical()
    GetHideCancel = "function", -- GetHideCancel()
    SetHorizontal = "function", -- SetHorizontal()
    SetVertical = "function", -- SetVertical()
    SetHideCancel = "function", -- SetHideCancel()
}

uabLayoutManager.core = uab
uabLayoutManager.layouts = {}
uabLayoutManager.layoutswithdefault = {}

table.insert(uabLayoutManager.layoutswithdefault, "Use default")

function uabLayoutManager:OnInitialize()
	self.super.OnInitialize(self)

    --[[
    Add the menu options to configure the default layout settings.
    --]]
    self.core.options.args.lookandfeel.args.defaultanchorscale.args.layout = {
        type = 'group',
        name = "Layout",
        desc = "Select and configure button layout options",
		order = 10,
        args = {
            select = {
                type = 'text',
                name = "Select",
                usage = "<name>",
                desc = "Select the current layout to use",
                order = 10,
                get = function()
                	local mgr = self:Get(self.core.db.profile.layout)
                	return mgr:GetDisplayName()
				end,
                set = function(name)
                	self.core.db.profile.layout = self:GetByDisplayName(name).name

                    local layout = self:GetLayout(self.core.db.profile.layout, "main")
                    local config = layout:GetConfig()
                    local layoutoptions = self:MenuOptionArgs(config)
                    
                    self.core.options.args.lookandfeel.args.defaultanchorscale.args.layout.args.configure.args = layoutoptions
                    
                    uab:SetGlobalLayoutManagerName(self.core.db.profile.layout)
                end,
                validate = self.layouts
            },
            configure = {
                type = 'group',
                name = "Configure",
                desc = "Configure current layout options",
                order = 20,
                args = {}
            },
        },
    }
end

function uabLayoutManager:MenuOptionArgs(config)
    local options = config:MenuOptionArgs()
    local maxorder = 0

    for _, option in pairs(options) do
        option.order = option.order + 2
        if option.order > maxorder then
            maxorder = option.order
        end
    end

    options.usedefaults = {
        type = 'execute',
        name = "Defaults",
        desc = "Return all layout settings to their default values",
        order = 1,
        func = function()
            config:UseDefaults()
        end,
    }
        
    options.base_space0 = {
        type = 'header',
        name = " ",
        order = 2,
    }
        
    options.base_space2 = {
        type = 'header',
        name = " ",
        order = maxorder+1,
    }
        
    options.horizontal = {
        type = 'text',
        name = "Horizontal",
        desc = "Horizontal offset (positive is right, negative is left, 0 is over cursor)",
        usage = "<number>",
        order = maxorder+2,
        get = function()
            return config:GetHorizontal() or 0
        end,
        set = function(value)
            if value and tonumber(value) then
                config:SetHorizontal(math.floor(tonumber(value)))
            end
        end,
        validate = function(value)
            if value and tonumber(value) then
                local v = math.floor(tonumber(value))
                if tostring(v) == value then
                    return true
                end
            end
        end,
    }
        
    options.vertical = {
        type = 'text',
        name = "Vertical",
        desc = "Vertical offset (positive is up, negative is down, 0 is over cursor)",
        usage = "<number>",
        order = maxorder+3,
        get = function()
            return config:GetVertical() or 0
        end,
        set = function(value)
            if value and tonumber(value) then
                config:SetVertical(math.floor(tonumber(value)))
            end
        end,
        validate = function(value)
            if value and tonumber(value) then
                local v = math.floor(tonumber(value))
                if tostring(v) == value then
                    return true
                end
            end
        end,
    }
        
    options.hidecancel = {
        type = 'toggle',
        name = "Hide cancel button",
        desc = "When checked, the cancel button will not be shown, and all positioning will act as though there is no cancel button",
        order = maxorder+4,
        get = function()
            return config:GetHideCancel()
        end,
        set = function(v)
            config:SetHideCancel(v)
        end,
	}

    return options
end

function uabLayoutManager:AddLayout(displayname)
    table.insert(self.layouts, displayname)
    table.insert(self.layoutswithdefault, displayname)
end

function uabLayoutManager:OnEnable()
	self.super.OnEnable(self)

    --[[
    Add the options for all the unitgroups
    --]]
    self.core.GroupIterate(
        function(self, context, name, info)
            self.core.options.args.lookandfeel.args.groupanchorscale.args[info.name].args.layout = {
                type = 'group',
                name = "Layout",
                desc = "Configure the layout to use for action buttons in the group",
                order = 25,
                args = self:GroupLayoutOptions(info.code)
            }
        end,
        self,
        nil)

    --[[
    Add the options for all the action sets
    --]]
    self:AddActionSetOptions()

    local layout = self:GetLayout(self.core.db.profile.layout, "main")
    local config = layout:GetConfig()
    local layoutoptions = self:MenuOptionArgs(config)
    
    self.core.options.args.lookandfeel.args.defaultanchorscale.args.layout.args.configure.args = layoutoptions
end

function uabLayoutManager:AddActionSetOptions()
    local uab = self.core

    for name, options in pairs(uab.options.args.lookandfeel.args.actionsets.args) do
        uab.ClearTable(uab.options.args.lookandfeel.args.actionsets.args[name])
        uab.options.args.lookandfeel.args.actionsets.args[name] = nil
    end

    local count = 10
    for name, info in pairs(uab.db.profile.actionsets) do
        local code = uab:GetActionSetCodeForName(name)

        uab.options.args.lookandfeel.args.actionsets.args[name] = {
            type = 'group',
            name = name,
            desc = "Configure the layout for the action buttons in the '"..name.."' action set",
            order = count,
            args = self:ActionSetLayoutOptions(code, info)
        }

        count = count + 10
    end

end

function uabLayoutManager:GetByDisplayName(displayname)
    for name, module in pairs(self.modules) do
        if module:GetDisplayName() == displayname then
            if module:IsEnabled() then
                return module
            end
        end
    end
end

function uabLayoutManager:Get(mgrname)
    for name, module in pairs(self.modules) do
        if name == mgrname then
            if module:IsEnabled() then
                return module
            end
        end
    end
end

function uabLayoutManager:GetLayout(mgrname, layoutname)
    local layoutmgrobj = self:Get(mgrname)

    return layoutmgrobj and layoutmgrobj:GetLayout(layoutname)
end

function uabLayoutManager:GroupLayoutOptions(unitgroupcode)
    local unitgroupname = uab.constants.unitgroupstatename[unitgroupcode]
    local layoutname = "group:"..unitgroupname
    local options = {}

    options.configure = {
        type = 'group',
        name = "Configure",
        desc = "Configure current layout options for this group",
        disabled = function()
            if not self.core.db.profile.layouts[unitgroupname] then
                return true
            end
        end,
        order = 20,
        args = {}
    }

    options.select = {
        type = 'text',
        name = "Select",
        usage = "<name>",
        desc = "Select the layout to use for this group",
        order = 10,
        get = function()
            if not self.core.db.profile.layouts[unitgroupname] then
                return "Use default"
            else
                local mgr = self:Get(self.core.db.profile.layouts[unitgroupname])
                return mgr:GetDisplayName()
            end
        end,
        set = function(name)
            if name == "Use default" then
                self.core.db.profile.layouts[unitgroupname] = nil
                uab.ClearTable(options.configure.args)
                options.configure.args = {}
            else
                self.core.db.profile.layouts[unitgroupname] = self:GetByDisplayName(name).name
            
                local layout = self:GetLayout(self.core.db.profile.layouts[unitgroupname], layoutname)
                local config = layout:GetConfig()
                local layoutoptions = self:MenuOptionArgs(config)
                
                options.configure.args = layoutoptions
            end

            uab:SetUnitGroupLayoutManagerName(unitgroupcode, self.core.db.profile.layouts[unitgroupname])
        end,
        validate = self.layoutswithdefault
    }

    if self.core.db.profile.layouts[unitgroupname] then
        local layout = self:GetLayout(self.core.db.profile.layouts[unitgroupname], layoutname)
        local config = layout:GetConfig()
        local layoutoptions = self:MenuOptionArgs(config)
        
        options.configure.args = layoutoptions
    end

    uab:SetUnitGroupLayoutManagerName(unitgroupcode, self.core.db.profile.layouts[unitgroupname])
    
    return options
end

function uabLayoutManager:FrameGroupLayoutOptions(framegroupcode, profile)
    local layoutname = "fgroup:"..uab:GetFrameGroupModuleName(framegroupcode)..":"..uab:GetFrameGroupName(framegroupcode)
    local options = {}
    
    options.configure = {
        type = 'group',
        name = "Configure",
        desc = "Configure current layout options",
        disabled = function()
            if not profile.layout then
                return true
            end
        end,
        order = 20,
        args = {}
    }

    options.select = {
        type = 'text',
        name = "Select",
        usage = "<name>",
        desc = "Select the current layout to use",
        order = 10,
        get = function()
            if not profile.layout then
                return "Use default"
            else
                local mgr = self:Get(profile.layout)
                return mgr:GetDisplayName()
            end
        end,
        set = function(name)
            if name == "Use default" then
                profile.layout = nil
                
                uab.ClearTable(options.configure.args)
                options.configure.args = {}
            else
                profile.layout = self:GetByDisplayName(name).name
            
                local layout = self:GetLayout(profile.layout, layoutname)
                local config = layout:GetConfig()
                local layoutoptions = self:MenuOptionArgs(config)
                
                options.configure.args = layoutoptions
            end

            uab:SetFrameGroupLayoutManagerName(framegroupcode, profile.layout)
        end,
        validate = self.layoutswithdefault
    }
    
    if profile.layout then
        local layout = self:GetLayout(profile.layout, layoutname)
        local config = layout:GetConfig()
        local layoutoptions = self:MenuOptionArgs(config)
        
        options.configure.args = layoutoptions
    end

    uab:SetFrameGroupLayoutManagerName(framegroupcode, profile.layout)
    
    return options
end

function uabLayoutManager:ActionSetLayoutOptions(actionsetcode, profile)
    local actionsetname = uab:ActionSetName(actionsetcode)
    local layoutname = "actions:"..actionsetname
    local options = {}
    
    options.configure = {
        type = 'group',
        name = "Configure",
        desc = "Configure current layout options",
        disabled = function()
            if not profile.layout then
                return true
            end
        end,
        order = 20,
        args = {}
    }

    options.select = {
        type = 'text',
        name = "Select",
        usage = "<name>",
        desc = "Select the current layout to use",
        order = 10,
        get = function()
            if not profile.layout then
                return "Use default"
            else
                local mgr = self:Get(profile.layout)
                return mgr:GetDisplayName()
            end
        end,
        set = function(name)
            if name == "Use default" then
                profile.layout = nil
                
                uab.ClearTable(options.configure.args)
                options.configure.args = {}
            else
                profile.layout = self:GetByDisplayName(name).name
            
                local layout = self:GetLayout(profile.layout, layoutname)
                local config = layout:GetConfig()
                local layoutoptions = self:MenuOptionArgs(config)
                
                options.configure.args = layoutoptions
            end

            uab:SetActionSetLayoutManagerName(actionsetcode, profile.layout)
        end,

        validate = self.layoutswithdefault
    }
    
    if profile.layout then
        local layout = self:GetLayout(profile.layout, layoutname)
        local config = layout:GetConfig()
        local layoutoptions = self:MenuOptionArgs(config)
                
        options.configure.args = layoutoptions
    end

    uab:SetActionSetLayoutManagerName(actionsetcode, profile.layout)
    
    return options
end

--[[
This class is the base class for all "layout" classes to use.  It provides the basic functions that
all layouts need.  This includes methods to manipulate the positioning information for buttons that
are shown from this layout, methods to change the configuration, and methods to set the positioning
information for buttons using this layout.
--]]
uabLayoutManager.Layout = uab.mixins.aceOO.Class()

function uabLayoutManager.Layout.prototype:init()
    uabLayoutManager.Layout.super.prototype.init(self)
end

function uabLayoutManager.Layout.prototype:Initialize(config, name, context)
    self.name = name .. "_" .. context
    self.basename = name
    self.config = config
    self.context = context
    self.havesettings = nil
    self.positions = {}
end

function uabLayoutManager.Layout.prototype:ResetPositions()
    if self.positions then
        uab.ClearTable(self.positions)
    end
    self.positions = {}
end

function uabLayoutManager.Layout.prototype:GetButtonPoint(button)
    local h = (self.config and self.config.settings and self.config.settings.horizontal) or 0
    local v = (self.config and self.config.settings and self.config.settings.vertical) or 0
    local point = {}

    point.x = self.positions[button].x - h
    point.y = self.positions[button].y - v
    point.point = self.positions[button].point

    return point
end

function uabLayoutManager.Layout.prototype:SetButtonPoint(button, point, x, y)
    local settings = {}
    local h = (self.config and self.config.settings and self.config.settings.horizontal) or 0
    local v = (self.config and self.config.settings and self.config.settings.vertical) or 0
    
    settings.point = point
    settings.x = x+h
    settings.y = y+v
    
    self.positions[button] = settings
end

function uabLayoutManager.Layout.prototype:FormatCancelBtn(cancelbtn)
    -- Nothing to do here at this time.  Would be a good idea to resize
    -- the cancel button frame, hide the text frame for the button frame
end

function uabLayoutManager.Layout.prototype:FormatActionBtn(btn)
    -- Nothing to do here at this time.  Would be a good idea to resize
    -- the cancel button frame, hide the text frame for the button frame
end

function uabLayoutManager.Layout.prototype:SetPointForStates(btn, states, index)
    local posdata = self.positions[index]
    if not posdata then
        uab.print("Unable to find position information for index ("..index..") of '"..self.name.."' for frame '"..btn.frame:GetName().."'")
    else
        btn:SetPointForStates(states, posdata.point, posdata.x, posdata.y)
    end
end

function uabLayoutManager.Layout.prototype:LayoutSettingsChanged()
    self.havesettings = nil
    uab:LayoutSettingsChanged(self.basename, self.context)
end

function uabLayoutManager.Layout.prototype:GetConfig()
	return self.config
end

function uabLayoutManager.Layout.prototype:SetSettings(settings)
    self.config:SettingsChanged(settings)
end

function uabLayoutManager.Layout.prototype:ArrangeButtons(cancelbtn, actionbtns, states)
    uabLayoutManager:Debug(self.name .. " : ArrangeButtons")

    if not self.havesettings then
        self:ResetPositions()
        self:SetupPositions()
        self.havesettings = true
    end

    local found
    for idx = 1, 24, 1 do
        local btn = actionbtns[idx]
        if btn then
            self:FormatActionBtn(btn)
            found = found or btn.isenabled
            if btn.isenabled then
                btn:AddShowStates(states)
                self:SetPointForStates(btn, states, idx)
            end
        end
    end
	
    --[[
    Now, position the cancel button centered relative to the header (the
    parent of the cancel button)
    --]]
    if cancelbtn then
        self:FormatCancelBtn(cancelbtn)
        if self.positions.cancel and found then
            cancelbtn:AddShowStates(states)
            self:SetPointForStates(cancelbtn, states, "cancel")
        end
    end
end

--[[
LayoutManager module prototype

When implementing a LayoutManager module:
 - variables to set
 	- self.defaultDB to define your own default settings for all layouts, or
          to hold any settings you want to use.  Be sure to include the following
          settings as they are used to control the use of a LayoutManager:
 	    - disabled
 - OnInitialize functionality to implement
    - set self.defaultDB here if needed
    - call super.OnInitialize(self)
    - set any other variables you may need and perform any additional first-time
	initialization
 - Methods to implement
    - OnEnable (optional)
    - OnDisable (optional)
    - GetLayout(contextName) - required
        - Used to allocate a new layout in the specified context.  The settings
          to use are maintained by you related to the specified context.  If you
          have no settings for that context, you must initialize the returned layout
          to use default settings appropriate for that layout.
    - GetDisplayName() - required
        - Used to retrieve the name to appear in the FuBar drop-down menu for this
          layout manager.
--]]

uabLayoutManager.modulePrototype.core = uabLayoutManager
uabLayoutManager.modulePrototype.controls = {}
uabLayoutManager.modulePrototype.layoutManagerDefaultDB = {
    disabled = nil,
}

function uabLayoutManager.modulePrototype:OnInitialize()
    if not self.db then
        self.core.core:RegisterDefaults(self.name, "profile", self.defaultDB or self.layoutManagerDefaultDB)
        self.db = self.core.core:AcquireDBNamespace(self.name)
    end

    self.debugging = self.db.profile.debug
    if self:IsEnabled() then
	    self.core:AddLayout(self:GetDisplayName())
	end
    
    self:SetDebugging(self.db.profile.debugging)
    self.core.core.options.args.debug.args[self.core.name..":"..self.name] = {
        type = 'toggle',
        name = self.core.name..":"..self.name,
        desc = "Enable debugging for the "..self.core.name..":"..self.name.." module",
        get = function()
            return self:IsDebugging()
        end,
        set = function(v)
            self.db.profile.debugging = v
            self:SetDebugging(self.db.profile.debugging)
        end,
    }
end

function uabLayoutManager.modulePrototype:IsEnabled()
    return not self.db.profile.disabled
end

