-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 42775 $"):match("%d+")), ("$Date: 2007-07-04 16:50:18 -0500 (Wed, 04 Jul 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[
This controls the configuration settings (and display of a sample set of
actions based on those settings) for each ActionSetLayout.  Each
BarActionSetLayout is associated with an BarActionSetLayoutConfig, and each
BarActionSetLayoutConfig refers to that same BarActionSetLayout instance.
--]]
uabBarLayoutManager.LayoutConfig = uab.mixins.aceOO.Class(uabLayoutManager.interfaces.IActionSetLayoutConfig, uab.mixins.aceEvent)
uabBarLayoutManager.LayoutConfig.count = 0

function uabBarLayoutManager.LayoutConfig.prototype:init(layout, settings)
    uabBarLayoutManager.LayoutConfig.super.prototype.init(self)

    self.settings = settings
    self.layout = layout

    uabBarLayoutManager.LayoutConfig.count = uabBarLayoutManager.LayoutConfig.count + 1
    self.number = uabBarLayoutManager.LayoutConfig.count

    self:SettingsChanged()
end

function uabBarLayoutManager.LayoutConfig.prototype:SettingsChanged(settings)
    if not self.settings then
        self.settings = {}
    end

    if not settings then
        settings = self.settings
    end
    
    self.settings.overflowClockwise = (settings and settings.overflowClockwise) or "true"
    self.settings.placement = (settings and settings.placement) or uabLayoutManager.constants.placement.right
    self.settings.growDirection = (settings and settings.growDirection) or uabLayoutManager.constants.grow.right
    self.settings.growLimit = (settings and settings.growLimit) or 4
    self.settings.growSpacing = (settings and settings.growSpacing) or 0
    self.settings.overflowSpacing = (settings and settings.overflowSpacing) or 0
    self.settings.horizontal = (settings and settings.horizontal) or 0
    self.settings.vertical = (settings and settings.vertical) or 0
    self.settings.hidecancel = (settings and settings.hidecancel)

    -- Make sure all the settings properly coordinate with one another
    
    self:SetOverflowClockwise(self.settings.overflowClockwise)
    self:SetPlacement(self.settings.placement)
    self:SetGrowDirection(self.settings.growDirection)
    self:SetGrowLimit(self.settings.growLimit)
    self:SetGrowSpacing(self.settings.growSpacing)
    self:SetOverflowSpacing(self.settings.overflowSpacing)
    self:SetHorizontal(self.settings.horizontal)
    self:SetVertical(self.settings.vertical)
    self:SetHideCancel(self.settings.hidecancel)
end

function uabBarLayoutManager.LayoutConfig.prototype:UseDefaults(bypassNotification)
	if not self.settings then
	    self.settings = {}
	end

    self.settings.overflowClockwise = "true"
    self.settings.placement = uabLayoutManager.constants.placement.right
    self.settings.growDirection = uabLayoutManager.constants.grow.right
    self.settings.growLimit = 4
    self.settings.growSpacing = 0
    self.settings.overflowSpacing = 0
    self.settings.horizontal = 0
    self.settings.vertical = 0
    self.settings.hidecancel = nil

    self:SettingsChanged()
	
    if not bypassNotification then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:MenuOptionArgs()
	--[[
	Build an AceOptions table of the options to be used to configure this
	config instance.  This table is inserted into a parent menu as a set of
	options.  If there are several options, just define them at the top level
	of the returned table.  Do not create a 'group' at the top level to enclose
	the options.
	--]]
	
	local options = {
        
        growlimit = {
            type = 'range',
            name = "Grow Limit",
            desc = "Maximum number of buttons to place in each row before expanding to the next row",
            order = 105,
            min = 1,
            max = 24,
            step = 1,
            isPercent = false,
            get = function()
                return self.settings.growLimit
            end,
            set = function(limit)
                self:SetGrowLimit(limit)
            end,
        },
        placement = {
            type = 'text',
            name = "First button placement",
            desc = "Placement of the first button, relative to the cancel button",
            order = 110,
            get = function()
                return uabLayoutManager.PlacementDisplayValue(self.settings.placement)
            end,
            set = function(placement)
                if placement == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.top) then
                    self:SetPlacement(uabLayoutManager.constants.placement.top)
				elseif placement == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottom) then
                	self:SetPlacement(uabLayoutManager.constants.placement.bottom)
				elseif placement == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.left) then
                	self:SetPlacement(uabLayoutManager.constants.placement.left)
				elseif placement == uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.right) then
                	self:SetPlacement(uabLayoutManager.constants.placement.right)
				end
            end,

            validate = {
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.top),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.bottom),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.left),
				uabLayoutManager.PlacementDisplayValue(uabLayoutManager.constants.placement.right),
			},
        },
        direction = {
            type = 'text',
            name = "Grow direction",
            desc = "The direction in which new buttons are added in each row",
            order = 120,
            get = function()
                return uabLayoutManager.GrowDirectionDisplayValue(self.settings.growDirection)
            end,
            set = function(direction)
                if direction == uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.up) then
	                self:SetGrowDirection(uabLayoutManager.constants.grow.up)
				elseif direction == uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.down) then
	                self:SetGrowDirection(uabLayoutManager.constants.grow.down)
				elseif direction == uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.left) then
	                self:SetGrowDirection(uabLayoutManager.constants.grow.left)
				elseif direction == uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.right) then
	                self:SetGrowDirection(uabLayoutManager.constants.grow.right)
				end
            end,

            validate = {
				uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.up),
				uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.down),
				uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.left),
				uabLayoutManager.GrowDirectionDisplayValue(uabLayoutManager.constants.grow.right),
			},
        },
        clockwise = {
            type = 'toggle',
            name = "Overflow clockwise",
            desc = "When a bar grows past the grow limit, add another row relative to the last bar in a clockwise direction",
            disabled = function() return not self.cansetdirection end,
            order = 130,
            get = function()
                return self.settings.overflowClockwise == "true"
            end,
            set = function(v)
                if v then
					self:SetOverflowClockwise("true")
				else
				    self:SetOverflowClockwise("false")
                end
            end,
        },

        space1 = {
            type = 'header',
            name = " ",
            order = 140,
        },
        
        growspacing = {
            type = 'range',
            name = "Grow Spacing",
            desc = "Amount of space between buttons in a row",
            order = 143,
            min = -20,
            max = 20,
            step = 1,
            isPercent = false,
            get = function()
                return self.settings.growSpacing
            end,
            set = function(v)
                self:SetGrowSpacing(v)
            end,
        },
        overflowspacing = {
            type = 'range',
            name = "Overflow Spacing",
            desc = "Amount of space between each row of buttons",
            order = 146,
            min = -20,
            max = 20,
            step = 1,
            isPercent = false,
            get = function()
                return self.settings.overflowSpacing
            end,
            set = function(v)
                self:SetOverflowSpacing(v)
            end,
        },
	}
	
	return options
end

--[[---------------------------------------------------------------------------
Helper methods, to be used only by the class itself
-----------------------------------------------------------------------------]]

function uabBarLayoutManager.LayoutConfig.prototype:SetPlacement(value)
    local notify = (value ~= self.settings.placement)

    self.settings.placement = value

    self:SynchronizeGrowPattern()

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetGrowDirection(value)
    local notify = (value ~= self.settings.growDirection)

    self.settings.growDirection = value

    self:SynchronizeGrowPattern()

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetGrowLimit(value)
    local notify = (value ~= self.settings.growLimit)

    self.settings.growLimit = value
    
    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetOverflowClockwise(value)
    local newval

    if value and (tonumber(value) == 1 or value == "true") then
        newval = "true"
    else
        newval = "false"
    end

    local notify = (newval ~= self.settings.overflowClockwise)

    self.settings.overflowClockwise = newval
    
    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetGrowSpacing(value)
    local notify = (value ~= self.settings.growSpacing)

    self.settings.growSpacing = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetOverflowSpacing(value)
    local notify = (value ~= self.settings.overflowSpacing)

    self.settings.overflowSpacing = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetHorizontal(value)
    local notify = (value ~= self.settings.horizontal)

    self.settings.horizontal = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetVertical(value)
    local notify = (value ~= self.settings.vertical)

    self.settings.vertical = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:SetHideCancel(value)
    local notify = (value ~= self.settings.hidecancel)

    self.settings.hidecancel = value

    if notify then
        self.layout:LayoutSettingsChanged()
    end
end

function uabBarLayoutManager.LayoutConfig.prototype:GetHorizontal()
    return self.settings.horizontal or 0
end

function uabBarLayoutManager.LayoutConfig.prototype:GetVertical()
    return self.settings.vertical or 0
end

function uabBarLayoutManager.LayoutConfig.prototype:GetHideCancel()
    return self.settings.hidecancel
end

function uabBarLayoutManager.LayoutConfig.prototype:EnableOverflowDirection()
    self.cansetdirection = true
end
    	
function uabBarLayoutManager.LayoutConfig.prototype:DisableOverflowDirection()
    self.cansetdirection = nil
end
    	
function uabBarLayoutManager.LayoutConfig.prototype:SynchronizeGrowPattern()
    self:EnableOverflowDirection()
    	
    if self.settings.placement and self.settings.growDirection then
        if self.settings.placement == uabLayoutManager.constants.placement.top then
            if self.settings.growDirection == uabLayoutManager.constants.grow.down then
                self:SetGrowDirection(uabLayoutManager.constants.grow.up)
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.right then
                self:SetOverflowClockwise("false")
                self:DisableOverflowDirection()
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.left then
                self:SetOverflowClockwise("true")
                self:DisableOverflowDirection()
            end
        elseif self.settings.placement == uabLayoutManager.constants.placement.bottom then
            if self.settings.growDirection == uabLayoutManager.constants.grow.up then
                self:SetGrowDirection(uabLayoutManager.constants.grow.down)
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.right then
                self:SetOverflowClockwise("true")
                self:DisableOverflowDirection()
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.left then
                self:SetOverflowClockwise("false")
                self:DisableOverflowDirection()
            end
        elseif self.settings.placement == uabLayoutManager.constants.placement.left then
            if self.settings.growDirection == uabLayoutManager.constants.grow.right then
                self:SetGrowDirection(uabLayoutManager.constants.grow.left)
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.up then
                self:SetOverflowClockwise("false")
                self:DisableOverflowDirection()
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.down then
                self:SetOverflowClockwise("true")
                self:DisableOverflowDirection()
            end
        elseif self.settings.placement == uabLayoutManager.constants.placement.right then
            if self.settings.growDirection == uabLayoutManager.constants.grow.left then
                self:SetGrowDirection(uabLayoutManager.constants.grow.right)
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.up then
                self:SetOverflowClockwise("true")
                self:DisableOverflowDirection()
            elseif self.settings.growDirection == uabLayoutManager.constants.grow.down then
                self:SetOverflowClockwise("false")
                self:DisableOverflowDirection()
            end
        end
    end
end

