-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 63272 $"):match("%d+")), ("$Date: 2008-03-01 02:05:29 -0500 (Sat, 01 Mar 2008) $"):match("%d%d%d%d%-%d%d%-%d%d"))

uabFrameGroup = uab:NewModule("FrameGroup", "AceModuleCore-2.0")
uabFrameGroup:SetModuleMixins("AceDebug-2.0", "AceModuleCore-2.0")

uabFrameGroup.core = uab
uabFrameGroup.framegroups = {}
uabFrameGroup.count = 1

local defaultFrameGroupName = "Default Frame Group"

function uabFrameGroup:OnInitialize()
	self.super.OnInitialize(self)
end

function uabFrameGroup:OnEnable()
	self.super.OnEnable(self)
end

function uabFrameGroup:GetMainOptionsMenu()
    return self.core.options.args.lookandfeel.args.framegroups
end

function uabFrameGroup:GetFrameGroup(frame, returnifdisabled)
	for name, module in pairs(self.modules) do
        if name ~= defaultFrameGroupName and module:IsFrameGroup(frame) then
            return (module:IsEnabled() or returnifdisabled) and module
		end
	end

    local module = self.modules[defaultFrameGroupName]
    
    return (module:IsEnabled() or returnifdisabled) and module
end

function uabFrameGroup:GetFrameGroupCode(frame)
	for name, module in pairs(self.modules) do
        if name ~= defaultFrameGroupName and module:IsFrameGroup(frame) then
            return module:IsEnabled() and module.framegroupcode
		end
	end

    local module = self.modules[defaultFrameGroupName]

    return module and module:IsEnabled() and module.framegroupcode
end

function uabFrameGroup:GetUnitGroupCode(frame)
	for name, module in pairs(self.modules) do
        if name ~= defaultFrameGroupName and module:IsFrameGroup(frame) then
            local unit = uab.GetFrameUnit(frame)
            local code = unit and uab.GetUnitGroupCodeFromUnitFrame(frame)
        
            return module:IsEnabled() and (code or module:GetUnitGroupCode())
		end
	end

    return uab.GetUnitGroupCodeFromUnitFrame(frame)
end

function uabFrameGroup:AddActivatorFrameControlForFrameGroup(framegroupcode, activatorframecontrolobj)
	for name, module in pairs(self.modules) do
        if module.framegroupcode == framegroupcode then
            module:AddActivatorFrameControl(activatorframecontrolobj)
            return
		end
	end
end

function uabFrameGroup:RemoveActivatorFrameControlForFrameGroup(framegroupcode, activatorframecontrolobj)
	for name, module in pairs(self.modules) do
        if module.framegroupcode == framegroupcode then
            module:RemoveActivatorFrameControl(activatorframecontrolobj)
            return
		end
	end
end

local scaleAndAnchor = uab.mixins.aceOO.Class()

function scaleAndAnchor.prototype:init(code, profile)
    scaleAndAnchor.super.prototype.init(self)
    self.framegroupcode = code
    self.profile = profile
end

function scaleAndAnchor.prototype:IsDefaultScale()
    return self.profile.usedefaultscale
end

function scaleAndAnchor.prototype:IsDefaultPosition()
    return self.profile.usedefaultanchor
end

function scaleAndAnchor.prototype:UseDefaultScale()
    return {
        type = 'toggle',
        name = "Default scale",
        desc = "Use default scale value for this frame group",
        get = function()
            return self.profile.usedefaultscale
        end,
        set = function(v)
            self.profile.usedefaultscale = v

            if not v then
                if not self.profile.scale then
                    self.profile.scale = uab.db.profile.scale.default.value
                end
                
                uab:SetFrameGroupScale(self.framegroupcode, self.profile.scale)
            else
                uab:SetFrameGroupScale(self.framegroupcode, nil)
            end
        end,
    }
end

function scaleAndAnchor.prototype:UseDefaultAnchor()
    return {
        type = 'toggle',
        name = "Default anchor",
        desc = "Use default action button anchor information for this frame group",
        get = function()
            return self.profile.usedefaultanchor
        end,
        set = function(v)
            self.profile.usedefaultanchor = v
            
            if not v then
                if not self.profile.point then
                    self.profile.point = uab.db.profile.position.default.point
                end

                uab:SetFrameGroupPoint(self.framegroupcode, self.profile.point)
                
                if not self.profile.relpoint then
                    self.profile.relpoint = uab.db.profile.position.default.relpoint
                end

                uab:SetFrameGroupRelPoint(self.framegroupcode, self.profile.relpoint)
                
                if not self.profile.x then
                    self.profile.x = uab.db.profile.position.default.x
                end

                uab:SetFrameGroupOffsetX(self.framegroupcode, self.profile.x)
                
                if not self.profile.y then
                    self.profile.y = uab.db.profile.position.default.y
                end
                
                uab:SetFrameGroupOffsetY(self.framegroupcode, self.profile.y)
            else
                uab:SetFrameGroupPoint(self.framegroupcode, nil)
                uab:SetFrameGroupRelPoint(self.framegroupcode, nil)
                uab:SetFrameGroupOffsetX(self.framegroupcode, nil)
                uab:SetFrameGroupOffsetY(self.framegroupcode, nil)
            end
        end,
    }
end

function scaleAndAnchor.prototype:GetScale()
    return self.profile.scale or uab.db.profile.scale.default.value
end

function scaleAndAnchor.prototype:SetScale(v)
    self.profile.scale = v
    uab:SetFrameGroupScale(self.framegroupcode, v)
end

function scaleAndAnchor.prototype:GetPoint()
    return self.profile.point or uab.db.profile.position.default.point or "CENTER"
end

function scaleAndAnchor.prototype:SetPoint(v)
    self.profile.point = v
    uab:SetFrameGroupPoint(self.framegroupcode, v)
end

function scaleAndAnchor.prototype:GetRelPoint()
    return self.profile.relpoint or uab.db.profile.position.default.relpoint or "cursor"
end

function scaleAndAnchor.prototype:SetRelPoint(v)
    self.profile.relpoint = v
    uab:SetFrameGroupRelPoint(self.framegroupcode, v)
end

function scaleAndAnchor.prototype:GetHorizontal()
    return self.profile.x or uab.db.profile.position.default.x or 0
end

function scaleAndAnchor.prototype:SetHorizontal(v)
    self.profile.x = v
    uab:SetFrameGroupOffsetX(self.framegroupcode, v)
end

function scaleAndAnchor.prototype:GetVertical()
    return self.profile.y or uab.db.profile.position.default.y or 0
end

function scaleAndAnchor.prototype:SetVertical(v)
    self.profile.y = v
    uab:SetFrameGroupOffsetY(self.framegroupcode, v)
end

--[[
FrameGroup module prototype

When implementing a FrameGroup module:
 - variables to set
 	- self.defaultDB to define your own default settings
 	    be sure to include the following settings as they are used to control
 	    the matching of frames, and the scaling and positioning of uab buttons
 	    in relation to these frames:
 	    - disabled (used to determine whether the frame group is to be used or
                    not.  If the frame group matches multiple types of frames
                    this is checked at the frame group module level, and also
                    at the level of the options for a specific frame)
 	    - expression (used to match frames to this frame group)
 	    - expressions (used to match frames to this frame group - table of expression)
 	    - usedefaultscale
		- scale, if not using the uab defaults that the user defines.
        - usedefaultanchor
        - point, relpoint, x, y for positioning, if not using the uab defaults
          that the user defines.
 	- self.unitgroup to define the name of the unitgroup with which this frame
        group is associated
 - OnInitialize functionality to implement
    - set self.defaultDB here if needed
    - call super.OnInitialize(self)
    - set any other variables you may need and perform any additional first-time
		initialization
 - Methods to implement
	- OnEnable (optional)
	- OnDisable (optional)
    - OnInitialize (optional)
    - AddFrameGroupOptions
    - SetLayoutConfigureOptions
--]]

uabFrameGroup.modulePrototype.core = uabFrameGroup
uabFrameGroup.modulePrototype.controls = {}
uabFrameGroup.modulePrototype.frameGroupDefaultDB = {
    disabled = nil,
    expression = nil,
    usedefaultscale = true,
    usedefaultanchor = true,
}

-- Override optional
function uabFrameGroup.modulePrototype:OnInitialize()
	if not self.db then
		self.core.core:RegisterDefaults(self.name, "profile", self.defaultDB or self.frameGroupDefaultDB)
		self.db = self.core.core:AcquireDBNamespace(self.name)
	end

    self.debugging = self.db.profile.debug
    if self.name ~= defaultFrameGroupName then
        self.id = self.core.count
        self.core.count = self.core.count + 1
    else
        self.id = 0
    end

    self.framegroupcode = uab.constants.framegroupstates*self.id
    self.core.core:SetFrameGroupObj(self.framegroupcode, self)
end

function uabFrameGroup.modulePrototype:OnEnable()
    local optionsmenu = self:GetOptionsMenu()
    
    self.basicoptions = self:CreateOptions()

    optionsmenu.args[self.name] = self.basicoptions
    optionsmenu.args[self.name].order = (self.framegroupcode+1)
    
    self:SetDebugging(self.db.profile.debugging)
    
    self.core.core.options.args.debug.args[self.core.name..":"..self.name] = {
        type = 'toggle',
        name = self.core.name..":"..self.name,
        desc = "Enable debugging for the "..self.core.name..":"..self.name.." module",
        get = function()
            return self:IsDebugging()
        end,
        set = function(v)
            self.db.profile.debugging = v
            self:SetDebugging(self.db.profile.debugging)
        end,
    }
    
    if self.db.profile.usedefaultscale then
        uab:SetFrameGroupScale(self.framegroupcode, nil)
    else
        uab:SetFrameGroupScale(self.framegroupcode, self.db.profile.scale)
    end

    if self.db.profile.usedefaultanchor then
        uab:SetFrameGroupPoint(self.framegroupcode, nil)
        uab:SetFrameGroupRelPoint(self.framegroupcode, nil)
        uab:SetFrameGroupOffsetX(self.framegroupcode, nil)
        uab:SetFrameGroupOffsetY(self.framegroupcode, nil)
    else
        uab:SetFrameGroupPoint(self.framegroupcode, self.db.profile.point)
        uab:SetFrameGroupRelPoint(self.framegroupcode, self.db.profile.relpoint)
        uab:SetFrameGroupOffsetX(self.framegroupcode, self.db.profile.x)
        uab:SetFrameGroupOffsetY(self.framegroupcode, self.db.profile.y)
    end

    uab:SetFrameGroupLayoutManagerName(self.framegroupcode, self.db.profile.layout)
end

-- Override optional
function uabFrameGroup.modulePrototype:GetUnitGroupCode()
    if self.db.profile.unitgroup then
        return uab.constants.groupInfo[self.db.profile.unitgroup].code
    end
end

-- Override expected
function uabFrameGroup.modulePrototype:GetModuleName()
end

-- Override not recommended
function uabFrameGroup.modulePrototype:GetName()
    return self.name
end

--[[
The default implementation returns the main "framegroup" dropdown menu.

-- Override recommended

--]]
function uabFrameGroup.modulePrototype:GetOptionsMenu()
    return uabFrameGroup:GetMainOptionsMenu()
end

-- Override optional
function uabFrameGroup.modulePrototype:AddActivatorFrameControl(activatorframecontrolobj)
end

-- Override optional
function uabFrameGroup.modulePrototype:RemoveActivatorFrameControl(activatorframecontrolobj)
end

-- Override not recommended
function uabFrameGroup.modulePrototype:SetLayoutConfigureOptions(args)
	self.basicoptions.args[self.name].args.layout.args.configure.args = args
end

-- Override not recommended
function uabFrameGroup.modulePrototype:IsDefaultFrameGroup()
end

-- Override not recommended
function uabFrameGroup.modulePrototype:IsEnabled()
    return not self.db.profile.disabled
end

function uabFrameGroup.modulePrototype:SetEnabled(v)
    self.db.profile.disabled = not v
    uab:FrameGroupEnabled(self.framegroupcode, v)
end

-- Override not recommended
function uabFrameGroup.modulePrototype:IsFrameGroup(frame)
    if self:MatchFrame(frame, self.db.profile) then
        return true
    end
end

-- Override not recommended
function uabFrameGroup.modulePrototype:MatchFrame(frame, defn)
    if not frame or not frame:GetName() then return end
    
    local framename = frame:GetName()

    if defn.expressions then    
        for _, exp in pairs(defn.expressions) do
            if string.find(framename, exp) then
                return true
            end
        end
    end
    
    if defn.expression then
        return string.find(framename, defn.expression)
    end
end

-- Override not recommended
function uabFrameGroup.modulePrototype:CreateOptions()

    local options
    local me = self

    local scaleanchor = scaleAndAnchor:new(self.framegroupcode, self.db.profile)
    
    options = {
        type = 'group',
        name = me.name,
        desc = "Use, anchor, scale, and match expression for frames in this frame group",
        args = uab.ScaleAndAnchorOptions(scaleanchor)
    }

    options.args.enabled = {
        type = 'toggle',
        name = "Enabled",
        desc = "When checked, enable the use of UAB action sets for frames in this frame group",
        order = 10,
        get = function()
            return me:IsEnabled()
        end,
        set = function(v)
            me:SetEnabled(v)
        end,
    }
    
    options.args.spacing_enabled = {
        name = " ",
        type = 'header',
        order = 15,
    }
            
    options.args.layout = {
        type = 'group',
        name = "Layout",
        desc = "Configure the layout for the action buttons when associated with frames in this frame group",
        order = 20,
        args = uabLayoutManager:FrameGroupLayoutOptions(me.framegroupcode, me.db.profile)
    }
    
    options.args.spacing_layout = {
        name = " ",
        type = 'header',
        order = 25,
    }

    return options
end

--[[---------------------------------------------------------
THe following frame group is used for all unit frames that
don't match any *real* frame group.
-----------------------------------------------------------]]

local uabDefaultFrameGroup = uabFrameGroup:NewModule(defaultFrameGroupName)

uabDefaultFrameGroup.defaultDB = {
    usedefaultscale = true,
    usedefaultanchor = true,
}

function uabDefaultFrameGroup:OnInitialize()
    self.super.OnInitialize(self)
end

function uabDefaultFrameGroup:IsEnabled()
    return uab.db.profile.usedefaultframegroup
end

function uabDefaultFrameGroup:SetEnabled(v)
    uab.db.profile.usedefaultframegroup = v
    uab:FrameGroupEnabled(self.framegroupcode, v)
end

function uabDefaultFrameGroup:IsFrameGroup(frame)
    return true
end

function uabDefaultFrameGroup:IsDefaultFrameGroup()
    return true
end

-- Override expected
function uabDefaultFrameGroup:GetModuleName()
    return "Default"
end

function uabDefaultFrameGroup:CreateOptions()
    local options = self.super.CreateOptions(self)

    options.args.warndefault = {
        type = 'toggle',
        name = "Warn on default",
        desc = "When checked, print a warning when using default setting for unknown frames.  When not checked, defaulted frames cause no warning",
        order = options.args.enabled.order + 1,
        get = function()
            return uab.db.profile.warndefaultframegroup
        end,
        set = function(v)
            uab.db.profile.warndefaultframegroup = v
        end,
    }
    
    return options
end
