-- Update the UAB version if the version of this file is later than any other already loaded
uab.CheckVersion(tonumber(("$Revision: 54416 $"):match("%d+")), ("$Date: 2007-11-12 16:16:09 -0500 (Mon, 12 Nov 2007) $"):match("%d%d%d%d%-%d%d%-%d%d"))

--[[
Class to manage the config dialog for the add-on.  Handles all config dialog functionality.
--]]

local ActionSetConfigFrame = UAB_ConfigFrameTab2Frame

function ActionSetConfigFrame:UabTabFrameShow()
end
            
function ActionSetConfigFrame:UabTabFrameHide()
end

function ActionSetConfigFrame:UabInit()
    local framename = self:GetName()
    
    self.uabframes = {}
    self.uabframes.actionbtns = getglobal(framename .. "ActionButtons")

    self:CreateActionButtons()

--    self.uabframes.actionsetcopy = getglobal(self:GetName() .. "ActionSetCopyTo")
    self.uabframes.actionsetnew = getglobal(framename .. "ActionSetNew")
    self.uabframes.actionsetdelete = getglobal(framename .. "ActionSetDelete")

--    self.uabframes.actionsetcopy:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabActionSetCopy() end)
    self.uabframes.actionsetnew:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabActionSetNew() end)
    self.uabframes.actionsetdelete:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabActionSetDelete() end)

    framename = self.uabframes.actionbtns:GetName()
    
    self.uabframes.stancecopy = getglobal(framename.."StanceCopyActions")
    self.uabframes.stanceclear = getglobal(framename.."StanceClearActions")
    self.uabframes.unitgroupcopy = getglobal(framename.."UnitGroupCopyActions")
    self.uabframes.unitgroupclear = getglobal(framename.."UnitGroupClearActions")

    self.uabframes.stancecopy:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabStanceCopy() end)
    self.uabframes.stanceclear:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabStanceClear() end)
    self.uabframes.unitgroupcopy:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabUnitGroupCopy() end)
    self.uabframes.unitgroupclear:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabUnitGroupClear() end)

    self:AddActionSets()

    self:InitActionSetControls()

    self:CurrentActionSetChanged()
end

function ActionSetConfigFrame:CreateActionButtons()
    self.uabactionbtns = {}

    for idx = 1, 24, 1 do
        self:CreateActionButton(idx)
    end    
end

function ActionSetConfigFrame:CreateActionButton(idx)
    local relinfo = {}

--    local parent = self.uabframes.actionbtns
    local parent = getglobal(self.uabframes.actionbtns:GetName().."RealButtons")

    relinfo.point = "TOPLEFT"

    if idx == 1 then
        relinfo.relpoint = "TOPLEFT"
        relinfo.relto = parent
        relinfo.x = 0 -- 30
        relinfo.y = 0 -- -73
    elseif idx == 5 or idx == 9 or idx == 13 or idx == 17 or idx == 21 then
        relinfo.relpoint = "BOTTOMLEFT"
        relinfo.relto = self.uabactionbtns[idx-4].frame
        relinfo.x = 0
        relinfo.y = -5
    else
        relinfo.relpoint = "TOPRIGHT"
        relinfo.relto = self.uabactionbtns[idx-1].frame
        relinfo.x = 5
        relinfo.y = 0
    end

    local frame = CreateFrame("CheckButton", self:GetName() .. "ActionButtons_Action"..idx, parent, "UAB_ActionButtonTemplate")

    frame:SetScale(0.8)
    frame:SetID(idx)
    frame:SetScript("OnClick", function(self, button, down) ActionSetConfigFrame:UabSelectAction(button, self) end)
    frame:ClearAllPoints()
    frame:SetPoint(relinfo.point, relinfo.relto, relinfo.relpoint, relinfo.x, relinfo.y)
    frame:Show()
    getglobal(frame:GetName().."ActionNum"):SetText(idx..".")
    fontName, fontHeight, fontFlags = getglobal(frame:GetName().."ActionNum"):GetFont()
    
    getglobal(frame:GetName().."ActionNum"):SetFont(fontName, 10, fontFlags)
    
    getglobal(frame:GetName().."Icon"):SetTexture("Interface\\AddOns\\UnitActionBars2\\images\\EmptySlot")

    frame:RegisterForDrag("LeftButton")
    frame:SetBackdropColor(0, 0, 0)

    frame:SetScript("OnDragStart", function(self, motion) ActionSetConfigFrame:OnDragStart(self) end)
    frame:SetScript("OnEnter", function(self, motion) ActionSetConfigFrame:ShowActionTooltip(self) end)
    frame:SetScript("OnReceiveDrag", function(self) ActionSetConfigFrame:OnActionReceiveDrag(self) end)
    frame:SetScript("OnLeave", function(self, motion) GameTooltip:Hide() end)
    
    self.uabactionbtns[idx] = {}
    self.uabactionbtns[idx].frame = frame
end

function ActionSetConfigFrame:RefreshCurrentActionSet()
    local associated
    
    if self.uabselectedactionsetcode then
        self:RefreshSelectedActionSetName()
    end

    self:CurrentActionContextChanged()
end

function ActionSetConfigFrame:RefreshSelectedActionSetName()
    local frame = getglobal(self:GetName().."ActionSet")
    
    self:InitActionSetDropDown()
    UIDropDownMenu_SetSelectedName(frame, uab:GetActionSetObj(self.uabselectedactionsetcode):GetName())
end

function ActionSetConfigFrame:CurrentActionContextChanged()
    local frame = getglobal(self:GetName().."ActionButtons")
    
    -- Show the actions for the currently selected action context.  That is, for the
    -- currently selected action set, stance, and unit group.
    if self.uabselectedactionsetcode then
        frame:Show()
        self.uabselectedactionindex = nil
        self:RefreshActions()
        self:CurrentActionChanged()
    else
        frame:Hide()
    end
end

function ActionSetConfigFrame:GetSelectedActions()
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local actions
    if self.uabselectedunitgroupcode ~= -1 then
        actions = actionsetobj:GetActions(self.uabselectedstancecode, self.uabselectedunitgroupcode)
    else
        actions = actionsetobj:GetHostileActions(self.uabselectedstancecode)
    end
    return actions
end


function ActionSetConfigFrame:RefreshActions()
    self:RefreshActionControlVisibility()
    self:ClearAllActions()
    if self.uabselectedactionsetcode then
        local frame = getglobal(self:GetName().."ActionButtons")
    
        -- set the values for the various text/drop down controls
        -- to the proper values based on the currently selected
        -- action for the current stance and unit group of the
        -- current action set.

        local actions = self:GetSelectedActions()
        for idx = 1, 24, 1 do
            local actionframe = getglobal(self:GetName().."ActionButtons_Action"..idx)
            local iconframe = getglobal(actionframe:GetName().."Icon")

            if actions[idx] and actions[idx].actionobj then
                -- actions[idx] is an ActionButton.  We should show
                -- the icon for this action in the appropriate control

                local wrapper = actions[idx].actionobj
                local icon = wrapper:GetIcon()

                iconframe:SetTexture(icon)
            else
                iconframe:SetTexture("Interface\\AddOns\\UnitActionBars2\\images\\EmptySlot")
            end
        end
    end
end

function ActionSetConfigFrame:CurrentActionChanged()
    for idx = 1, 24, 1 do
        if idx ~= self.uabselectedactionindex then
            local actionframe = getglobal(self:GetName().."ActionButtons_Action"..idx)
            actionframe:SetChecked(nil)
        end
    end

    self:RefreshActionControlVisibility()
    self:RefreshActionControlContents()
end

function ActionSetConfigFrame:UabSelectAction(button, action)
    if button == "LeftButton" then
        local actiondata = self:ReceiveActionDrop()
        if actiondata then
            local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
            self:ProcessReceiveDrag(actionsetobj, action, actiondata)
            return
        end
    end

    local checked = action:GetChecked()
    local id = action:GetID()

    self.uabselectedactionindex = nil

    if checked then
        self.uabselectedactionindex = action:GetID()
        local wrapper = self:GetSelectedActions()[self.uabselectedactionindex]
        if not wrapper then
            self.uabselectedactionindex = nil
        end
    end

    self:CurrentActionChanged()        
end

function ActionSetConfigFrame:RefreshActionControlVisibility()
    local frame = getglobal(self:GetName().."ActionButtons")

    if self.uabselectedactionindex then
        getglobal(frame:GetName().."ActionClear"):Show()
        getglobal(frame:GetName().."PostClickMessage"):Show()
        getglobal(frame:GetName().."CurrentAction"):Show()
        getglobal(frame:GetName().."RankLabel"):Show()
        getglobal(frame:GetName().."UseRank"):Show()
        getglobal(frame:GetName().."UnitStatusGroupLabel"):Show()
        getglobal(frame:GetName().."UnitStatusGroup"):Show()
        getglobal(frame:GetName().."UnitStatusLabel"):Show()
        getglobal(frame:GetName().."UnitStatus"):Show()
        getglobal(frame:GetName().."DefaultHideUnusable"):Show()
        getglobal(frame:GetName().."HideUnusable"):Show()
        getglobal(frame:GetName().."UsePlayerUnit"):Show()
        getglobal(frame:GetName().."ActionEffects"):Show()
    else
        getglobal(frame:GetName().."ActionClear"):Hide()
        getglobal(frame:GetName().."PostClickMessage"):Hide()
        getglobal(frame:GetName().."CurrentAction"):Hide()
        getglobal(frame:GetName().."RankLabel"):Hide()
        getglobal(frame:GetName().."UseRank"):Hide()
        getglobal(frame:GetName().."UnitStatusGroupLabel"):Hide()
        getglobal(frame:GetName().."UnitStatusGroup"):Hide()
        getglobal(frame:GetName().."UnitStatusLabel"):Hide()
        getglobal(frame:GetName().."UnitStatus"):Hide()
        getglobal(frame:GetName().."DefaultHideUnusable"):Hide()
        getglobal(frame:GetName().."HideUnusable"):Hide()
        getglobal(frame:GetName().."UsePlayerUnit"):Hide()
        getglobal(frame:GetName().."ActionEffects"):Hide()
    end
end

function ActionSetConfigFrame:RefreshActionControlContents()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actionframe = getglobal(frame:GetName().."CurrentAction")
    local iconframe = getglobal(actionframe:GetName().."Icon")
    local wrapper

    iconframe:SetTexture("Interface\\AddOns\\UnitActionBars2\\images\\EmptySlot")
    if self.uabselectedactionindex then
        local ranklabel = getglobal(frame:GetName().."RankLabel")
        local userank = getglobal(frame:GetName().."UseRank")
        local defaulthideunusable = getglobal(frame:GetName().."DefaultHideUnusable")
        local hideunusable = getglobal(frame:GetName().."HideUnusable")
        local useplayerunit = getglobal(frame:GetName().."UsePlayerUnit")
        
        wrapper = self:GetSelectedActions()[self.uabselectedactionindex]
        if wrapper then
            local actionobj = wrapper:GetActionObj()
            local icon = wrapper:GetIcon()
        
            iconframe:SetTexture(icon)
        
            if actionobj:GetHasRank() then
                ranklabel:Show()
                userank:Show()
                self:InitRankDropDown()
            else
                ranklabel:Hide()
                userank:Hide()
            end
            
            defaulthideunusable:SetChecked(wrapper:GetDefaultHideUnusable())
            if wrapper:GetDefaultHideUnusable() then
                hideunusable:SetChecked(nil)
                hideunusable:Hide()
            else
                hideunusable:SetChecked(wrapper:GetHideUnusable())
                hideunusable:Show()
            end

            useplayerunit:SetChecked(wrapper:GetUsePlayerUnit())
            
            self:RefreshActionEffects()
        end
    end
end

function ActionSetConfigFrame:RefreshActionEffects()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")

    local wrapper = self:GetSelectedActions()[self.uabselectedactionindex]
    local actionobj = wrapper:GetActionObj()

    local borderhighlight = getglobal(actioneffects:GetName().."BorderHighlight")
    local alpha50 = getglobal(actioneffects:GetName().."Alpha50")
    local blink = getglobal(actioneffects:GetName().."Blink")
    local hide = getglobal(actioneffects:GetName().."Hide")
    local notusable = getglobal(actioneffects:GetName().."NotUsable")
    local showcooldown = getglobal(actioneffects:GetName().."ShowCooldown")

    local actioneffects = wrapper.effects[self.uabselectedunitstatus]

    borderhighlight:SetChecked(actioneffects and actioneffects.highlight)
    alpha50:SetChecked(actioneffects and actioneffects.lowalpha)
    blink:SetChecked(actioneffects and actioneffects.blink)
    hide:SetChecked(actioneffects and actioneffects.noalpha)
    notusable:SetChecked(actioneffects and actioneffects.notusable)
    showcooldown:SetChecked(actioneffects and actioneffects.showcooldown)
end

function ActionSetConfigFrame:ClearAllActions()
    for idx = 1, 24, 1 do
        local actionframe = getglobal(self:GetName().."ActionButtons_Action"..idx)
        local iconframe = getglobal(actionframe:GetName().."Icon")

        iconframe:SetTexture("Interface\\AddOns\\UnitActionBars2\\images\\EmptySlot")
    end
end

function ActionSetConfigFrame:AddActionSets()
    local frame = getglobal(self:GetName().."ActionSet")
    
    frame.uabactionsetdata = {}

    self.uabselectedactionsetcode = nil
    self:RefreshActionSets()
end

function ActionSetConfigFrame:RefreshActionSets()
    self:SetupActionSets()
    local frame = getglobal(self:GetName().."ActionSet")
    if not self.uabselectedactionsetcode then
        if frame.uabactionsetdata[1] then
            self.uabselectedactionsetcode = frame.uabactionsetdata[1].index
        else
            self.uabselectedactionsetcode = nil
        end
    end
    self:InitActionSetDropDown()
end
        
function ActionSetConfigFrame:SetupActionSets()
    local frame = getglobal(self:GetName().."ActionSet")
	local names = {}
	local sortednames = {}

    for code, actionsetobj in pairs(uab.actionsetobjs) do
        names[actionsetobj:GetName()] = code
        table.insert(sortednames, actionsetobj:GetName())
    end

	table.sort(sortednames)

    for idx, _ in pairs(frame.uabactionsetdata) do
        frame.uabactionsetdata[idx] = nil
    end

	for idx = 1, #sortednames, 1 do
        frame.uabactionsetdata[idx] = { name = sortednames[idx], index = names[sortednames[idx]] }
    end
end

function ActionSetConfigFrame:InitActionSetDropDown()
    local frame = getglobal(self:GetName().."ActionSet")
    UIDropDownMenu_Initialize(frame,
        function()
            ActionSetConfigFrame:FillActionSetDropDown()
        end)
	UIDropDownMenu_SetWidth(110, frame)
	UIDropDownMenu_SetButtonWidth(24, frame)
	UIDropDownMenu_JustifyText("LEFT", frame)
end

function ActionSetConfigFrame:FillActionSetDropDown()
    local frame = getglobal(self:GetName().."ActionSet")
    local selected = nil
    
    for idx, data in pairs(frame.uabactionsetdata) do
        local dropdowndata = {
            text = data.name,
            func = uab.DropDownSelect,
            arg1 = self,
            arg2 = { func = "SelectActionSetDropDown", arg1 = data }
        }
        UIDropDownMenu_AddButton(dropdowndata)
        if self.uabselectedactionsetcode == data.index then
            selected = data.name
        end
    end

    UIDropDownMenu_SetSelectedName(frame, selected)
end

function ActionSetConfigFrame:SelectActionSetDropDown(selector, selection)
    local frame = getglobal(self:GetName().."ActionSet")
    UIDropDownMenu_SetSelectedName(frame, selection.name)
    self.uabselectedactionsetcode = selection.index
    self:CurrentActionSetChanged()
end

function ActionSetConfigFrame:CurrentActionSetChanged()
    self:RefreshCurrentActionSet()
end

function ActionSetConfigFrame:InitActionSetControls()
    -- Set up the stance and unit group drop downs with the static set of values, and
    -- configure the controls to call the proper methods when a new selection is made
    -- from the control.
    self.uabselectedstancecode = 0
    self.uabselectedunitgroupcode = 1
    self.uabselectedunitstatusgroup = "main"
    self.uabselectedunitstatus = "Default"

    local frame = getglobal(self:GetName().."ActionButtons")
    local framename = frame:GetName()
    local actioneffects = getglobal(framename.."ActionEffects")
    local actioneffectsname = actioneffects:GetName()

    getglobal(framename.."ActionClear"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabActionClearOnClick() end)
    getglobal(framename.."PostClickMessage"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabPostClickMessageOnClick() end)

    getglobal(framename.."DefaultHideUnusable"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleDefaultHideUnusableSelected() end)
    getglobal(framename.."HideUnusable"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleHideUnusableSelected() end)
    getglobal(framename.."UsePlayerUnit"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleUsePlayerUnitSelected() end)
    getglobal(actioneffectsname.."BorderHighlight"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleBorderHighlightSelected() end)
    getglobal(actioneffectsname.."Alpha50"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleLowAlphaSelected() end)
    getglobal(actioneffectsname.."Blink"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleBlinkSelected() end)
    getglobal(actioneffectsname.."Hide"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleHideSelected() end)
    getglobal(actioneffectsname.."NotUsable"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleNotUsableSelected() end)
    getglobal(actioneffectsname.."ShowCooldown"):SetScript("OnClick", function(self, button) ActionSetConfigFrame:UabToggleShowCooldownSelected() end)

    self:InitStanceDropDown()
    self:InitUnitGroupDropDown()
    self:InitUnitStatusGroupDropDown()
    self:RefreshUnitStatusDropDown()
    
    -- Clear all the "action" controls to have no action associated with them, and
    -- hide all the controls that only show relevant data when an action is selected.
    self:CurrentActionContextChanged()
end

function ActionSetConfigFrame:UabActionClearOnClick()
    -- Clear the selected action control to have no action associated with it.
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local actions = self:GetSelectedActions()
    local actionframe = getglobal(self:GetName().."ActionButtons_Action"..self.uabselectedactionindex)
    local iconframe = getglobal(actionframe:GetName().."Icon")

    iconframe:SetTexture("Interface\\AddOns\\UnitActionBars2\\images\\EmptySlot")
    actionframe:SetChecked(nil)

    local dbactionset = uab.db.profile.actionsets[actionsetobj:GetName()]
    local dbstance = dbactionset.stances[self.uabselectedstancecode]
    local dbgroup
     
    if self.uabselectedunitgroupcode ~= -1 then
        dbgroup = dbstance[uab.constants.unitgroupstatename[self.uabselectedunitgroupcode]]
    else
        dbgroup = dbstance.hostile
    end

    uab.ClearTable(dbgroup[self.uabselectedactionindex])
    dbgroup[self.uabselectedactionindex] = nil

    uab:SetPendingAttributes()

    -- Clear the selected action in the current unit group of the current stance of the current action set
    if self.uabselectedunitgroupcode ~= -1 then
        actionsetobj:ClearAction(self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
        uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributesForUnitGroup, { unitgroupcode = self.uabselectedunitgroupcode, stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })
    else
        actionsetobj:ClearHostileAction(self.uabselectedstancecode, self.uabselectedactionindex)
        uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributes, { stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })
    end

    uab:ClearPendingAttributes()

    self.uabselectedactionindex = nil
    self:CurrentActionChanged()
end

function ActionSetConfigFrame:UabPostClickMessageOnClick()
    --[[
    Open a dialog that contains a single text window, an OK button,
    and a Cancel button.
    --]]
    
    UAB_ActionMessage:UabInit(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    UAB_ActionMessage:Show()
end

function ActionSetConfigFrame:UabToggleDefaultHideUnusableSelected()
    local frame = getglobal(self:GetName().."ActionButtons")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if getglobal(frame:GetName().."DefaultHideUnusable"):GetChecked() then
        getglobal(frame:GetName().."HideUnusable"):Hide()
        uab:SetActionDefaultHideUnusable(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    else
        getglobal(frame:GetName().."HideUnusable"):Show()
        uab:ClearActionDefaultHideUnusable(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    end
end
    
function ActionSetConfigFrame:UabToggleHideUnusableSelected()
    local frame = getglobal(self:GetName().."ActionButtons")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if getglobal(frame:GetName().."HideUnusable"):GetChecked() then
        uab:SetActionHideUnusable(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    else
        uab:ClearActionHideUnusable(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    end
end
    
function ActionSetConfigFrame:UabToggleUsePlayerUnitSelected()
    local frame = getglobal(self:GetName().."ActionButtons")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if getglobal(frame:GetName().."UsePlayerUnit"):GetChecked() then
        uab:SetActionUsePlayerUnit(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    else
        uab:ClearActionUsePlayerUnit(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    end
end
    
function ActionSetConfigFrame:UabToggleBorderHighlightSelected()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")
    local borderhighlight = getglobal(actioneffects:GetName().."BorderHighlight")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if borderhighlight:GetChecked() then
        uab:SetActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "highlight")
    else
        uab:ClearActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "highlight")
    end
end

function ActionSetConfigFrame:UabToggleLowAlphaSelected()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")
    local alpha50 = getglobal(actioneffects:GetName().."Alpha50")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if alpha50:GetChecked() then
        uab:SetActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "lowalpha")
    else
        uab:ClearActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "lowalpha")
    end
end

function ActionSetConfigFrame:UabToggleBlinkSelected()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")
    local blink = getglobal(actioneffects:GetName().."Blink")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if blink:GetChecked() then
        uab:SetActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "blink")
    else
        uab:ClearActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "blink")
    end
end

function ActionSetConfigFrame:UabToggleHideSelected()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")
    local hide = getglobal(actioneffects:GetName().."Hide")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if hide:GetChecked() then
        uab:SetActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "noalpha")
    else
        uab:ClearActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "noalpha")
    end
end

function ActionSetConfigFrame:UabToggleNotUsableSelected()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")
    local notusable = getglobal(actioneffects:GetName().."NotUsable")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if notusable:GetChecked() then
        uab:SetActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "notusable")
    else
        uab:ClearActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "notusable")
    end
end

function ActionSetConfigFrame:UabToggleShowCooldownSelected()
    local frame = getglobal(self:GetName().."ActionButtons")
    local actioneffects = getglobal(frame:GetName().."ActionEffects")
    local showcooldown = getglobal(actioneffects:GetName().."ShowCooldown")

    -- Toggling the value for this checkbox causes a change in a setting that needs
    -- to be persisted after a re-log (or UI reload).  Telling the action wrapper
    -- that the setting changed does not cause the setting to be persisted.  We also
    -- have to make the change in the persistent storage data.

    if showcooldown:GetChecked() then
        uab:SetActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "showcooldown")
    else
        uab:ClearActionUnitStatusVisualEffect(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, self.uabselectedunitstatus, "showcooldown")
    end
end

function ActionSetConfigFrame:InitStanceDropDown()
    local frame = getglobal(self:GetName().."ActionButtonsStance")
    UIDropDownMenu_Initialize(frame,
        function()
            ActionSetConfigFrame:FillStanceDropDown()
        end)
	UIDropDownMenu_SetWidth(126, frame)
	UIDropDownMenu_SetButtonWidth(24, frame)
	UIDropDownMenu_JustifyText("LEFT", frame)
end

function ActionSetConfigFrame:FillStanceDropDown()
    local frame = getglobal(self:GetName().."ActionButtonsStance")
    local selected = nil

    for stancecode = 0, uab.MaxCurrentClassStances(), 1 do
        local text = "Stance " .. stancecode

        if stancecode == 0 then
            text = text .. " (Normal)"
        else
            name = uab.GetStanceName(stancecode)

            if name then
                text = text .. " ("..name..")"
            else
                text = text .. " (N/A)"
            end                
        end

        UIDropDownMenu_AddButton({
            text = text,
            func = uab.DropDownSelect,
            arg1 = self,
            arg2 = { func = "SelectStanceDropDown", arg1 = { code = stancecode, name = text } }
        })
        if self.uabselectedstancecode == stancecode then
            selected = text
        end
    end    

    UIDropDownMenu_SetSelectedName(frame, selected)
end

function ActionSetConfigFrame:SelectStanceDropDown(selector, selection)
    local frame = getglobal(self:GetName().."ActionButtonsStance")
    UIDropDownMenu_SetSelectedName(frame, selection.name)
    self.uabselectedstancecode = selection.code
    self:CurrentStanceChanged()
end

function ActionSetConfigFrame:CurrentStanceChanged()
    self:CurrentActionContextChanged()
end

function ActionSetConfigFrame:InitUnitGroupDropDown()
    local frame = getglobal(self:GetName().."ActionButtonsUnitGroup")
    UIDropDownMenu_Initialize(frame,
        function()
            ActionSetConfigFrame:FillUnitGroupDropDown()
        end)
	UIDropDownMenu_SetWidth(126, frame)
	UIDropDownMenu_SetButtonWidth(24, frame)
	UIDropDownMenu_JustifyText("LEFT", frame)
end

function ActionSetConfigFrame:FillUnitGroupDropDown()
    local frame = getglobal(self:GetName().."ActionButtonsUnitGroup")
    local selected = nil

    for name, info in pairs(uab.constants.groupInfo) do
        UIDropDownMenu_AddButton({
            text = info.displayName,
            func = uab.DropDownSelect,
            arg1 = self,
            arg2 = { func = "SelectUnitGroupDropDown", arg1 = info }
        })
        if self.uabselectedunitgroupcode == info.code then
            selected = info.displayName
        end
    end    

    UIDropDownMenu_AddButton({
        text = "Hostile",
        func = uab.DropDownSelect,
        arg1 = self,
        arg2 = { func = "SelectUnitGroupDropDown", arg1 = -1 }
    })
    if self.uabselectedunitgroupcode == -1 then
        selected = "Hostile"
    end

    UIDropDownMenu_SetSelectedName(frame, selected)
end

function ActionSetConfigFrame:SelectUnitGroupDropDown(selector, selection)
    local frame = getglobal(self:GetName().."ActionButtonsUnitGroup")
    if selection == -1 then
        UIDropDownMenu_SetSelectedName(frame, "Hostile")
        self.uabselectedunitgroupcode = -1
    else
        UIDropDownMenu_SetSelectedName(frame, selection.displayName)
        self.uabselectedunitgroupcode = selection.code
    end
    self:CurrentUnitGroupChanged()
end

function ActionSetConfigFrame:CurrentUnitGroupChanged()
    self:CurrentActionContextChanged()
end

function ActionSetConfigFrame:InitUnitStatusGroupDropDown()
    local frame = getglobal(self:GetName().."ActionButtons")
    local unitstatusgroup = getglobal(frame:GetName().."UnitStatusGroup")
    
    UIDropDownMenu_Initialize(unitstatusgroup,
        function()
            ActionSetConfigFrame:FillUnitStatusGroupDropDown()
        end)
	UIDropDownMenu_SetWidth(110, unitstatusgroup)
	UIDropDownMenu_SetButtonWidth(24, unitstatusgroup)
	UIDropDownMenu_JustifyText("LEFT", unitstatusgroup)
end

local function GetUnitEffectGroups()
    local index = 1
    local selected = nil

    local options = {}

	local uniteffects = uabUnitStatus:GetAllUnitStatus()
	local sorted = {}

	for name, _ in pairs(uniteffects) do
		table.insert(sorted, name)
	end
	
	table.sort(sorted)

	for idx = 1, #sorted, 1 do
		table.insert(options, { text = sorted[idx],
								func = uab.DropDownSelect,
								arg2 = { func = "SelectUnitStatusGroupDropDown", arg1 = sorted[idx] } })
	end		

    return options
end
    
function ActionSetConfigFrame:FillUnitStatusGroupDropDown()
    local frame = getglobal(self:GetName().."ActionButtons")
    local unitstatusgroup = getglobal(frame:GetName().."UnitStatusGroup")
    local index = 1
    local selected = nil

    local options = GetUnitEffectGroups()

    for index, data in pairs(options) do
        data.arg1 = self
		if (index > UIDROPDOWNMENU_MAXBUTTONS) then
			break
		end

        UIDropDownMenu_AddButton(data)
        if self.uabselectedunitstatusgroup == data.arg2.arg1 then
            selected = self.uabselectedunitstatusgroup
        end
    end

    UIDropDownMenu_SetSelectedName(unitstatusgroup, selected)
end

function ActionSetConfigFrame:SelectUnitStatusGroupDropDown(selector, selection)
    local frame = getglobal(self:GetName().."ActionButtons")
    local unitstatusgroup = getglobal(frame:GetName().."UnitStatusGroup")
    
    UIDropDownMenu_SetSelectedName(unitstatusgroup, selection)
    if self.uabselectedunitstatusgroup ~= selection then
        self.uabselectedunitstatusgroup = selection
        self.uabselectedunitstatus = nil
        
        self:CurrentUnitStatusGroupChanged()
    end
end

function ActionSetConfigFrame:CurrentUnitStatusGroupChanged()
    self:RefreshUnitStatusDropDown()
    self:CurrentActionUnitStatusContextChanged()
end

function ActionSetConfigFrame:RefreshUnitStatusDropDown()
    local frame = getglobal(self:GetName().."ActionButtons")
    local unitstatus = getglobal(frame:GetName().."UnitStatus")
    
    UIDropDownMenu_Initialize(unitstatus,
        function()
            ActionSetConfigFrame:FillUnitStatusDropDown()
        end)
	UIDropDownMenu_SetWidth(110, unitstatus)
	UIDropDownMenu_SetButtonWidth(24, unitstatus)
	UIDropDownMenu_JustifyText("LEFT", unitstatus)
end
    
function ActionSetConfigFrame:GetUnitEffects()
    local index = 1
    local selected = nil

    local options = {}

	local uniteffects = uabUnitStatus:GetAllUnitStatus()
	local sorted = {}

    if uniteffects[self.uabselectedunitstatusgroup] then
        for name, _ in pairs(uniteffects[self.uabselectedunitstatusgroup]) do
            table.insert(sorted, name)
        end
	end
	
	table.sort(sorted)

	for idx = 1, #sorted, 1 do
		table.insert(options, { text = sorted[idx],
								func = uab.DropDownSelect,
                                arg1 = self,
								arg2 = { func = "SelectUnitStatusDropDown", arg1 = sorted[idx] } })
	end		

    return options
end

function ActionSetConfigFrame:FillUnitStatusDropDown()
    local frame = getglobal(self:GetName().."ActionButtons")
    local unitstatus = getglobal(frame:GetName().."UnitStatus")
    local index = 1
    local selected = nil

    local options = self:GetUnitEffects()

    for index, data in pairs(options) do
		if (index > UIDROPDOWNMENU_MAXBUTTONS) then
			break
		end

        UIDropDownMenu_AddButton(data)
        if not self.uabselectedunitstatus then
            self.uabselectedunitstatus = data.arg2.arg1
        end

        if self.uabselectedunitstatus == data.arg2.arg1 then
            selected = self.uabselectedunitstatus
        end
    end

    UIDropDownMenu_SetSelectedName(unitstatus, selected)
end

function ActionSetConfigFrame:SelectUnitStatusDropDown(selector, selection)
    local frame = getglobal(self:GetName().."ActionButtons")
    local unitstatus = getglobal(frame:GetName().."UnitStatus")
    
    UIDropDownMenu_SetSelectedName(unitstatus, selection)
    if self.uabselectedunitstatus ~= selection then
        self.uabselectedunitstatus = selection
        
        self:CurrentUnitStatusChanged()
    end
end

function ActionSetConfigFrame:CurrentUnitStatusChanged()
    self:CurrentActionUnitStatusContextChanged()
end

function ActionSetConfigFrame:CurrentActionUnitStatusContextChanged()
    self:RefreshActionEffects()
end

function ActionSetConfigFrame:UabActionSetNew()
    UAB_NewActionSet:UabInit(self.uabselectedactionsetcode)
    UAB_NewActionSet:Show()
end

function ActionSetConfigFrame:UabActionSetDelete()
    uab:SetPendingAttributes()

    uab:DeleteActionSetByCode(self.uabselectedactionsetcode)

    self.uabselectedactionsetcode = nil
    self:RefreshActionSets()
    uabLayoutManager:AddActionSetOptions()
    uab:BuildActivatorOptions(true)
    self:CurrentActionSetChanged()

    uab:ClearPendingAttributes()
end

function ActionSetConfigFrame:UabStanceCopy()
    UAB_CopyStanceActions:UabInit(self.uabselectedactionsetcode, self.uabselectedstancecode)
    
    UAB_CopyStanceActions:Show()
end

function ActionSetConfigFrame:ClearActionSetStanceActions(actionsetcode, stancecode)
    local actionsetobj = uab:GetActionSetObj(actionsetcode)
    local actionsetdata = uab.db.profile.actionsets[actionsetobj:GetName()]

    -- First, clear the hostile and normal actions for this stance of the action set
    -- object. We must do this before removing the actions from the profile, since the
    -- action set object references the profile information. Clearing hostile actions
    -- will cause an update that will reference the normal action data. If the profile
    -- data is cleared already, this will cause an error.
    for idx = 1, 24, 1 do
        actionsetobj:ClearHostileAction(stancecode, idx)
    end

    for name, info in pairs(uab.constants.groupInfo) do
        for idx = 1, 24, 1 do
            actionsetobj:ClearAction(stancecode, info.code, idx)
        end
    end

    -- Now, clear the profile action data for this stance of this action set, since the
    -- profile action data is no longer referenced by any action set object or activator.
    if actionsetdata.stances[stancecode] then
        uab.ClearTable(actionsetdata.stances[stancecode])
        actionsetdata.stances[stancecode] = nil
    end
end

function ActionSetConfigFrame:UabStanceClear()
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local actionsetdata = uab.db.profile.actionsets[actionsetobj:GetName()]

    uab:SetPendingAttributes()

    self:ClearActionSetStanceActions(self.uabselectedactionsetcode, self.uabselectedstancecode)
    
    uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributes, { stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })

    uab:ClearPendingAttributes()

    self:CurrentActionContextChanged()
end

function ActionSetConfigFrame:UabUnitGroupCopy()
    UAB_CopyGroupActions:UabInit(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode)
    
    UAB_CopyGroupActions:Show()
end

function ActionSetConfigFrame:ClearActionSetStanceUnitGroupActions(actionsetcode, stancecode, unitgroupcode)
    local actionsetobj = uab:GetActionSetObj(actionsetcode)
    local actionsetdata = uab.db.profile.actionsets[actionsetobj:GetName()]
    local stancedata = actionsetdata.stances[stancecode]
    local currentunitgroupname
     
    -- First, clear the actions for the unit group of this action set object. We must do
    -- this before removing the actions from the profile, since the action set object
    -- references the profile information. Clearing actions will cause an update that will
    -- reference the profile action data. If the profile data is cleared already, this
    -- will cause an error.
    if unitgroupcode ~= -1 then
        currentunitgroupname = uab.constants.unitgroupstatename[unitgroupcode]
    else
        currentunitgroupname = "hostile"
    end

    for idx = 1, 24, 1 do
        if unitgroupcode ~= -1 then
            actionsetobj:ClearAction(stancecode, unitgroupcode, idx)
        else
            actionsetobj:ClearHostileAction(stancecode, idx)
        end
    end

    -- Now, clear the profile action data for this unit group of this stance of this
    -- action set, since the profile action data is no longer referenced by any action
    -- set object or activator.
    if stancedata and stancedata[currentunitgroupname] then
        uab.ClearTable(stancedata[currentunitgroupname])
        stancedata[currentunitgroupname] = nil
    end

end

function ActionSetConfigFrame:UabUnitGroupClear()
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local actionsetdata = uab.db.profile.actionsets[actionsetobj:GetName()]
    local stancedata = actionsetdata.stances[self.uabselectedstancecode]
    local currentunitgroupname
     
    uab:SetPendingAttributes()

    self:ClearActionSetStanceUnitGroupActions(self.uabselectedactionsetcode, self.uabselectedstancecode, self.uabselectedunitgroupcode)
    
    if self.uabselectedunitgroupcode ~= -1 then
        uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributesForUnitGroup, { unitgroupcode = self.uabselectedunitgroupcode, stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })
    else
        uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributes, { stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })
    end

    uab:ClearPendingAttributes()

    self:CurrentActionContextChanged()
end

function ActionSetConfigFrame:ShowActionTooltip(action)
    local id = action:GetID()
    local wrapper = self:GetSelectedActions()[id]
    
    if wrapper then
        wrapper:ShowActionTooltip(action)
    end
end

function ActionSetConfigFrame:OnDragStart(action)
    local id = action:GetID()
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local wrapper = self:GetSelectedActions()[id]

    if wrapper then
        if wrapper:GetActionObj():PickupAction() then
            -- First, clear the selected (dragging) action, and refresh the controls to show that this action
            -- is no longer there.

            self.uabselectedactionindex = id
            self:UabActionClearOnClick()
        end
    end
end

function ActionSetConfigFrame:OnActionReceiveDrag(action)
    local id = action:GetID()
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local actiondata = self:ReceiveActionDrop()

    if actiondata then
        self:ProcessReceiveDrag(actionsetobj, action, actiondata)
    end
end

function ActionSetConfigFrame:ProcessReceiveDrag(actionsetobj, action, actiondata)
    local id = action:GetID()

    uab:SetPendingAttributes()

    actiondata.effect = {}
    self.uabselectedactionindex = id

    self:SetupNewActionSetAction(actionsetobj, actiondata)
    
    action:SetChecked(true)
end

function ActionSetConfigFrame:ReceiveActionDrop()
    local actiondata
    
    if CursorHasMacro() then
        local _, index, _ = GetCursorInfo()
        
        actiondata = self:GetCursorMacroAction(index)
    elseif CursorHasSpell() then
        local _, slot, spelltype = GetCursorInfo()

        if spelltype == "spell" then
            actiondata = self:GetCursorSpellAction(slot)
        end
    elseif CursorHasItem() then
        local _, id, link = GetCursorInfo()
        
        actiondata = self:GetCursorItemAction(id, link)
    end
    
    ClearCursor()
    
    return actiondata
end

function ActionSetConfigFrame:GetCursorSpellAction(slot)
    local name, rank = GetSpellName(slot, BOOKTYPE_SPELL)
    local usemax

    -- for right now, if the spell has a rank, and the rank name is not a simple name (e.g., Rank 1 vs. Rank 1: Turtle)
    -- then we will use the exact rank for the spell instead of using the max.  This only temporarily allows non-standard
    -- ranked spells to be used, since the rank drop down will list the ranks as numbers to select, so a special ranked
    -- spell might get de-selected in preference to a normal ranked spell because the default "Rank #" rank will be the
    -- only one that is selectable.
    if rank then
        local start, _, thisrank = string.find(rank, "Rank (%d+)")
        if start then
            local simplerank = "Rank "..thisrank
            if rank ~= simplerank then
                usemax = true
            end
        else
            usemax = true
        end
    else
        usemax = true
    end

    return {
        classname = "SpellAction",
        data = {
            max = usemax,
            name = name,
            rank = "("..rank..")",
        }
    }
end

function ActionSetConfigFrame:GetCursorItemAction(id, link)
    local sName, sLink, iRarity, iLevel, iMinLevel, sType, sSubType, iStackCount = GetItemInfo(id)

    return {
        classname = "ItemAction",
        data = {
            name = sName,
        }
    }
end

function ActionSetConfigFrame:GetCursorMacroAction(index)
    local name, texture, macrobody, localVar = GetMacroInfo(index)

    return {
        classname = "MacroAction",
        data = {
            name = name,
        }
    }
end

function ActionSetConfigFrame:GetRanks()
    local ranks = {}
    local data = {}
    
    data.text = "Max"
    data.func = uab.DropDownSelect
    data.arg1 = self
    data.arg2 = { func = "SelectRank", arg1 = "Max" }

    table.insert(ranks, data)
    
    local wrapper = self:GetSelectedActions()[self.uabselectedactionindex]
    if wrapper and wrapper:GetActionObj() and wrapper:GetActionObj():GetHasRank() then
        local actionobj = wrapper:GetActionObj()
        local _, _, maxrank = string.find(uabSpellAction:GetMaxRank(actionobj.data.name, actionobj:GetRankTemplate()), "Rank (%d+)")
    
        if maxrank then
            for idx = 1, maxrank, 1 do
                data = {}
                
                data.text = "Rank "..idx
                data.func = uab.DropDownSelect
                data.arg1 = self
                data.arg2 = { func = "SelectRank", arg1 = actionobj:GetRankText(idx) }
        
                table.insert(ranks, data)
            end
        end
    end

    return ranks
end

function ActionSetConfigFrame:FillUseRank()
    local frame = getglobal(self:GetName().."ActionButtons")
    local userank = getglobal(frame:GetName().."UseRank")
    local ranks = self:GetRanks()
    local wrapper = self:GetSelectedActions()[self.uabselectedactionindex]
    if wrapper and wrapper:GetActionObj() and wrapper:GetActionObj():GetHasRank() then
        local actionobj = wrapper:GetActionObj()
        local selected
    
        UIDropDownMenu_AddButton(ranks[1])
        if actionobj.data.max or not actionobj:GetHasRank() then
            selected = 1
        end        
    
        for idx = 2, # ranks, 1 do
            UIDropDownMenu_AddButton(ranks[idx])
            if not actionobj.data.max and actionobj.data.rank == "("..ranks[idx].arg2.arg1..")" then
                selected = idx
            end        
        end
    
        UIDropDownMenu_SetSelectedID(userank, selected)
    end
end

function ActionSetConfigFrame:InitRankDropDown()
    local frame = getglobal(self:GetName().."ActionButtons")
    local userank = getglobal(frame:GetName().."UseRank")
    UIDropDownMenu_Initialize(userank,
        function()
            ActionSetConfigFrame:FillUseRank()
        end)
	UIDropDownMenu_SetWidth(110, userank)
	UIDropDownMenu_SetButtonWidth(24, userank)
	UIDropDownMenu_JustifyText("LEFT", userank)
end

function ActionSetConfigFrame:SetupDBAction(actionsetobj, actiondata)
    local dbactionset = uab.db.profile.actionsets[actionsetobj:GetName()]
    
    local dbstance = dbactionset.stances[self.uabselectedstancecode]
    local dbgroup
     
    if self.uabselectedunitgroupcode ~= -1 then
        if not dbstance[uab.constants.unitgroupstatename[self.uabselectedunitgroupcode]] then
            dbstance[uab.constants.unitgroupstatename[self.uabselectedunitgroupcode]] = {}
        end

        dbgroup = dbstance[uab.constants.unitgroupstatename[self.uabselectedunitgroupcode]]
    else
        if not dbstance.hostile then
            dbstance.hostile = {}
        end

        dbgroup = dbstance.hostile
    end

    if dbgroup[self.uabselectedactionindex] then
        uab.ClearTable(dbgroup[self.uabselectedactionindex])
        dbgroup[self.uabselectedactionindex] = nil
    end

    dbgroup[self.uabselectedactionindex] = actiondata
end

function ActionSetConfigFrame:SetupNewActionSetAction(actionsetobj, actiondata)
    if self.uabselectedunitgroupcode ~= -1 then
        actionsetobj:ClearAction(self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex)
    else
        actionsetobj:ClearHostileAction(self.uabselectedstancecode, self.uabselectedactionindex)
    end

    self:SetupDBAction(actionsetobj, actiondata)
    
    uab:SetPendingAttributes()

    if self.uabselectedunitgroupcode ~= -1 then
        wrapper = uabActionWrapper:CreateActionWrapper(actiondata)
        actionsetobj:AddAction(self.uabselectedstancecode, self.uabselectedunitgroupcode, self.uabselectedactionindex, wrapper)
    else
        wrapper = uabActionWrapper:CreateActionWrapper(actiondata, true)
        actionsetobj:AddHostileAction(self.uabselectedstancecode, self.uabselectedactionindex, wrapper)
    end

    self:RefreshActions()
    self:CurrentActionChanged()

    if self.uabselectedunitgroupcode ~= -1 then
        uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributesForUnitGroup, { unitgroupcode = self.uabselectedunitgroupcode, stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })
    else
        uab:IterateActivatorsCallGlobal(uabConfig.ReApplyASUGLayoutAttributes, { stancecode = self.uabselectedstancecode, actionsetobj = actionsetobj })
    end

    uab:ClearPendingAttributes()
end

function ActionSetConfigFrame:GetActionSetDBActionInfo(actionsetobj)
    local dbactionset = uab.db.profile.actionsets[actionsetobj:GetName()]
    local dbstance = dbactionset.stances[self.uabselectedstancecode]
    local dbgroup
     
    if self.uabselectedunitgroupcode ~= -1 then
        dbgroup = dbstance[uab.constants.unitgroupstatename[self.uabselectedunitgroupcode]]
    else
        dbgroup = dbstance.hostile
    end

    return dbgroup[self.uabselectedactionindex]
end

function ActionSetConfigFrame:SelectRank(selector, selection)
    local frame = getglobal(self:GetName().."ActionButtons")
    local userank = getglobal(frame:GetName().."UseRank")
    
    UIDropDownMenu_SetSelectedID(userank, selector:GetID())
    
    local actionsetobj = uab:GetActionSetObj(self.uabselectedactionsetcode)
    local actioninfo = self:GetActionSetDBActionInfo(actionsetobj)
    
    local actiondata = uab.CloneTable(actioninfo)
    
    if selection == "Max" then
        actiondata.data.max = true
        actiondata.data.rank = nil
    else
        actiondata.data.max = nil
        actiondata.data.rank = "("..selection..")"
    end

    self:SetupNewActionSetAction(actionsetobj, actiondata)
end

uabConfig:RegisterConfigTabFrame(2, ActionSetConfigFrame)
