uab = AceLibrary("AceAddon-2.0"):new(
    "AceConsole-2.0",
    "AceDebug-2.0",
    "AceEvent-2.0",
    "AceDB-2.0",
    "AceModuleCore-2.0"
)
uab:SetModuleMixins("AceDebug-2.0", "AceEvent-2.0", "AceModuleCore-2.0")

function uab.CheckVersion(maybeversion, maybedate)
    if not uab.version or uab.revision < maybeversion then
        uab.version = "r"..maybeversion
        uab.revision = maybeversion
        uab.date = maybedate
    end
end

uab.CheckVersion(tonumber(("$Revision: 75047 $"):match("%d+")), ("$Date: 2008-05-24 16:45:01 -0400 (Sat, 24 May 2008) $"):match("%d%d%d%d%-%d%d%-%d%d"))

function uab.modulePrototype:OnInitialize()
    if not self.core then
        self.core = uab
    end

	if not self.db then
		self.core:RegisterDefaults(self.name, "profile", self.defaultDB or {})
		self.db = self.core:AcquireDBNamespace(self.name)
	end

    self:SetDebugging(self.db.profile.debugging)
    self.core.options.args.debug.args[self.name] = {
        type = 'toggle',
        name = self.name,
        desc = "Enable debugging for the "..self.name.." module",
        get = function()
            return self:IsDebugging()
        end,
        set = function(v)
            self.db.profile.debugging = v
            self:SetDebugging(self.db.profile.debugging)
        end,
    }
end

function uab.modulePrototype:OnEnable()
end

function uab.modulePrototype:SetFrameAttribute(frame, attrib, val, state)
    uab.SetFrameAttributeDebug(self, frame, attrib, val, state)
end

function uab.modulePrototype:SetUnknownFrameAttribute(frame, attrib, val, state)
    uab.SetUnknownFrameAttributeDebug(self, frame, attrib, val, state)
end

function uab.modulePrototype:SetBoolFrameAttribute(frame, attrib, val, state)
    uab.SetBoolFrameAttributeDebug(self, frame, attrib, val, state)
end

function uab.modulePrototype:SetFrameFrameAttribute(frame, attrib, val, state)
    uab.SetFrameFrameAttributeDebug(self, frame, attrib, val, state)
end

uab.hasIcon = "Interface\\CURSOR\\\Cast"
uab.defaultPosition = "RIGHT"
uab.hideWithoutStandby = true
uab.independentProfile = true
uab.hasNoColor = true
uab.cannotDetachTooltip = true
uab.ispendingattributes = nil
uab.pendingattributes = {}

UAB2Debug = {}

function uab.DumpToDebugTable(msg)
    if msg then table.insert(UAB2Debug, msg) end
end

function uab.print(msg)
    if msg then DEFAULT_CHAT_FRAME:AddMessage(msg) end
end

function uab.SetFrameAttribute(frame, attrib, val, state)
    uab.SetFrameAttributeDebug(uab, frame, attrib, val, state)
end

function uab.SetUnknownFrameAttribute(frame, attrib, val, state)
    uab.SetUnknownFrameAttributeDebug(uab, frame, attrib, val, state)
end

function uab.SetBoolFrameAttribute(frame, attrib, val, state)
    uab.SetBoolFrameAttributeDebug(uab, frame, attrib, val, state)
end

function uab.SetFrameFrameAttribute(frame, attrib, val, state)
    uab.SetFrameFrameAttributeDebug(uab, frame, attrib, val, state)
end

function uab.SetFrameLevelAndStrata(frame, level)
    frame:SetFrameLevel(level)
    frame:SetFrameStrata("HIGH")  --FULLSCREEN_DIALOG
end

function uab:IsPendingAttributes()
    return self.ispendingattributes
end

function uab:SetPendingAttributes()
    if not self:IsPendingAttributes() then
        self.ispendingattributes = true
        return true
    end
end

function uab:ClearPendingAttributes()
    self.ispendingattributes = nil
    self:ApplyPendingAttributes()
    if self.stancectrlsecureheader then
        self.stancectrlsecureheader:ApplyPendingAttributes()
    end
    uab.classes.SecureHeader:ApplyPendingAttributes()
    self:IterateActivators("ApplyAllPendingAttributes")
end

function uab:ApplyPendingAttributes()
	for name, val in pairs(self.pendingattributes) do
		for frame, data in pairs(val) do
            self.SetFrameAttribute(frame, name, data.val)
            val[frame] = nil
        end
        self.pendingattributes[name] = nil
	end
end

function uab:SetAttribute(frame, attribute, val)
    if self:IsPendingAttributes() then
        if not self.pendingattributes[attribute] then
            self.pendingattributes[attribute] = {}
        end
        self.pendingattributes[attribute][frame] = { val = val }
    else
        self:ApplyPendingAttributes()
        self.SetFrameAttribute(frame, attribute, val)
    end
end

function uab:IsPostponeSetAttr()
    if self.postponesetattr then
        return true
    end
end

function uab:StartPostponeAttributes()
    self.postponesetattr = {}
end

function uab:AddPostponeSetAttr(frame, attrib, val, state)
    table.insert(self.postponesetattr, { frame = frame, attrib = attrib, val = val, state = state })
end

function uab:EndPostponeSetAttr()
    uab.print("EndPostponeSetAttr - Count: "..#uab.postponesetattr)
    for idx = 1, #self.postponesetattr, 1 do
        local info = self.postponesetattr[idx]
        if info and info.frame and info.attrib then
            info.frame:SetAttribute(info.attrib, info.val)
            info.frame = nil
            info.attrib = nil
            info.val = nil
            info.state = nil
        end

        self.postponesetattr[idx] = nil
    end
    
    self.postponesetattr = nil
end

function uab.SetFrameAttributeDebug(dbgobj, frame, attrib, val, state)
    if not uab.postponesetattr then
        frame:SetAttribute(attrib, val)
    else
        uab:AddPostponeSetAttr(frame, attrib, val, state)
    end
end

function uab.SetUnknownFrameAttributeDebug(dbgobj, frame, attrib, val, state)
    if not uab.postponesetattr then
        frame:SetAttribute(attrib, val)
    else
        uab:AddPostponeSetAttr(frame, attrib, val, state)
    end
end

function uab.SetBoolFrameAttributeDebug(dbgobj, frame, attrib, val, state)
    if not uab.postponesetattr then
        frame:SetAttribute(attrib, val)
    else
        uab:AddPostponeSetAttr(frame, attrib, val, state)
    end
end

function uab.SetFrameFrameAttributeDebug(dbgobj, frame, attrib, val, state)
    if not uab.postponesetattr then
        frame:SetAttribute(attrib, val)
    else
        uab:AddPostponeSetAttr(frame, attrib, val, state)
    end
end

uab.enabledframes = {}
uab.classes = {}
uab.constants = {}

uab.constants.buttonWidth = 38
uab.constants.buttonHeight = 38

uab.constants.unitgroupstates = 1

-- this is 10 instead of 5 to account for the "hostile" group, though it is not really included
-- and to round it up to an easy to read number.
uab.constants.unitgroupcount = 10
uab.constants.unitgroupstatename = {
    [1] = "player",
    [2] = "pet",
    [3] = "partyraid",
    [4] = "focus",
    [5] = "target",
}

uab.constants.groupInfo = {
	player =    { name = "player",    displayName = "Player",     code = 1 },
	pet =       { name = "pet",       displayName = "Pet",        code = 2 },
	partyraid = { name = "partyraid", displayName = "Party/Raid", code = 3 },
	focus =     { name = "focus",     displayName = "Focus",      code = 4 },
	target =    { name = "target",    displayName = "Any Target", code = 5 },
}

-- Total number of states within a stance
uab.constants.stancestates = uab.constants.unitgroupcount

-- At most 10 stances for each activator
uab.constants.stancecount = 10
uab.constants.stances = {
    ["PRIEST"] = {
        [1] = "Shadowform",
        [2] = "Spirit of Redemption",
    },
    ["SHAMAN"] = {
        [1] = "Ghost Wolf",
    },
}
uab.constants.classmaxstance = {
    ["DRUID"] = 8,      -- aquatic, bear (dire bear), cat, flight, moonkin (tree of life), travel
    ["HUNTER"] = 0,     -- no shapeshifting
    ["MAGE"] = 0,       -- no shapeshifting (no, not even invisibility)
    ["PALADIN"] = 8,    -- corresponds to the 8 auras (devotion, retribution, contentration, sanctity, shadow resist, frost resist, fire resist, crusader)
    ["PRIEST"] = 2,     -- shadow, redemption
    ["ROGUE"] = 1,      -- stealth
    ["SHAMAN"] = 1,     -- ghost wolf
    ["WARLOCK"] = 0,    -- no shapeshifting
    ["WARRIOR"] = 3,    -- battle, defensive, berserker
}

function uab.MaxCurrentClassStances()
    local _, class = UnitClass("player")

    if class == "PRIEST" then
        local haveshadow
        local havespirit

        local numTabs = GetNumTalentTabs()
        for tab=1, numTabs, 1 do
            local numTalents = GetNumTalents(tab)
            for idx = 1, numTalents, 1 do
                local name, icon, tier, col, cur, max, isEx, meets = GetTalentInfo(tab, idx)
                if name == "Spirit of Redemption" then
                    if cur > 0 then
                        havespirit = true
                    end
                elseif name == "Shadowform" then
                    if cur > 0 then
                        haveshadow = true
                    end
                end
            end
        end
        
        if haveshadow and havespirit then
            return 2
        elseif haveshadow or havespirit then
            return 1
        else
            return 1
        end
    end
    
    return uab.constants.classmaxstance[class]
end

-- Total number of states within each activator
uab.constants.activatorstates = uab.constants.unitgroupcount*uab.constants.stancecount

-- We allow at most 100 activators.  The first 40 are tied to specific key-click combinations.  The last
-- 60 are user-defined, and can only be activated/deactivated by direct slash command, or function-call.
uab.constants.activatorcount = 100
uab.constants.keyclickactivatorcount = 40
uab.constants.activatorInfo = {
    [0]  = { keyclick = true, button = "RightButton",  prefixshift = true },
    [1]  = { keyclick = true, button = "RightButton",                      prefixctrl = true },
    [2]  = { keyclick = true, button = "RightButton",                                         prefixalt = true },
    [3]  = { keyclick = true, button = "RightButton",  prefixshift = true, prefixctrl = true },
    [4]  = { keyclick = true, button = "RightButton",  prefixshift = true,                    prefixalt = true },
    [5]  = { keyclick = true, button = "RightButton",                      prefixctrl = true, prefixalt = true },
    [6]  = { keyclick = true, button = "RightButton",  prefixshift = true, prefixctrl = true, prefixalt = true },
    [7]  = { keyclick = true, button = "RightButton" },
    
    [8]  = { keyclick = true, button = "LeftButton",   prefixshift = true },
    [9]  = { keyclick = true, button = "LeftButton",                       prefixctrl = true },
    [10] = { keyclick = true, button = "LeftButton",                                          prefixalt = true },
    [11] = { keyclick = true, button = "LeftButton",   prefixshift = true, prefixctrl = true },
    [12] = { keyclick = true, button = "LeftButton",   prefixshift = true,                    prefixalt = true },
    [13] = { keyclick = true, button = "LeftButton",                       prefixctrl = true, prefixalt = true },
    [14] = { keyclick = true, button = "LeftButton",   prefixshift = true, prefixctrl = true, prefixalt = true },
    [15] = { keyclick = true, button = "LeftButton" },
    
    [16] = { keyclick = true, button = "MiddleButton", prefixshift = true },
    [17] = { keyclick = true, button = "MiddleButton",                     prefixctrl = true },
    [18] = { keyclick = true, button = "MiddleButton",                                        prefixalt = true },
    [19] = { keyclick = true, button = "MiddleButton", prefixshift = true, prefixctrl = true },
    [20] = { keyclick = true, button = "MiddleButton", prefixshift = true,                    prefixalt = true },
    [21] = { keyclick = true, button = "MiddleButton",                     prefixctrl = true, prefixalt = true },
    [22] = { keyclick = true, button = "MiddleButton", prefixshift = true, prefixctrl = true, prefixalt = true },
    [23] = { keyclick = true, button = "MiddleButton" },
    
    [24] = { keyclick = true, button = "Button4",      prefixshift = true },
    [25] = { keyclick = true, button = "Button4",                          prefixctrl = true },
    [26] = { keyclick = true, button = "Button4",                                             prefixalt = true },
    [27] = { keyclick = true, button = "Button4",      prefixshift = true, prefixctrl = true },
    [28] = { keyclick = true, button = "Button4",      prefixshift = true,                    prefixalt = true },
    [29] = { keyclick = true, button = "Button4",                          prefixctrl = true, prefixalt = true },
    [30] = { keyclick = true, button = "Button4",      prefixshift = true, prefixctrl = true, prefixalt = true },
    [31] = { keyclick = true, button = "Button4" },
    
    [32] = { keyclick = true, button = "Button5",      prefixshift = true },
    [33] = { keyclick = true, button = "Button5",                          prefixctrl = true },
    [34] = { keyclick = true, button = "Button5",                                             prefixalt = true },
    [35] = { keyclick = true, button = "Button5",      prefixshift = true, prefixctrl = true },
    [36] = { keyclick = true, button = "Button5",      prefixshift = true,                    prefixalt = true },
    [37] = { keyclick = true, button = "Button5",                          prefixctrl = true, prefixalt = true },
    [38] = { keyclick = true, button = "Button5",      prefixshift = true, prefixctrl = true, prefixalt = true },
    [39] = { keyclick = true, button = "Button5" },
}

-- This is the max number of states for each frame group that is created.
uab.constants.framegroupstates = uab.constants.unitgroupcount*uab.constants.stancecount*uab.constants.activatorcount

-- The result of "state / X" where X is framegroupstates gives you the framegroup
-- The result of "state mod X" where X is framegroupstates gives you the state within the framegroup
-- When you have a state value within a framegroup, the result of "state / X" where X is the activatorstates gives you the activator
-- When you have a state value within a framegroup, the result of "state mod X" where X is the activatorstates gives you the state within the activator
-- When you have a state value within an activator, the result of "state / X" where X is the stancestates gives you the stance
-- When you have a state value within an activator, the result of "state mod X" where X is the stancestates gives you the unitgroup

uab.mixins = {}
uab.mixins.aceConsole = AceLibrary("AceConsole-2.0")
uab.mixins.aceDB = AceLibrary("AceDB-2.0")
uab.mixins.aceEvent = AceLibrary("AceEvent-2.0")
uab.mixins.aceOO = AceLibrary("AceOO-2.0")
uab.mixins.aura = AceLibrary("SpecialEvents-Aura-2.0")
uab.mixins.banzai = AceLibrary("LibBanzai-2.0")
uab.mixins.tablet = AceLibrary("Tablet-2.0")
uab.mixins.waterfall = AceLibrary("Waterfall-1.0")
uab.mixins.dewdrop = AceLibrary("Dewdrop-2.0")

--[[
Default profile configuration settings
--]]
uab.profileDefaults = {
    tooltips = true,
    directcast = true,
    fastchangeunit = true,
    checkusable = true,
    blinkcycle = 1.0,
    borderwidth = 2,
    alpha = 0.5,
    usedefaultframegroup = nil,
    warndefaultframegroup = true,
    hideunusable = nil,
    activators = {
    },
    actionsets = {
    },
    groupstayopen = {
    },
    independent = {
    },
    framegroups = {
    },
    scale = {
        player = {
			usedefault = true,
		},
        pet = {
			usedefault = true,
		},
        partyraid = {
			usedefault = true,
		},
        focus = {
			usedefault = true,
		},
        target = {
			usedefault = true,
		},
        default = {
			value = 1.0,
		},
    },
    position = {
        player = {
			usedefault = true,
		},
        pet = {
			usedefault = true,
		},
        partyraid = {
			usedefault = true,
		},
        focus = {
			usedefault = true,
		},
        target = {
			usedefault = true,
		},
        default = {
			point = "CENTER",
            relpoint = "cursor",
            x = 0.0,
            y = 0.0,
		},
    },
    layout = "uab.Bar",
    layouts = {
    },
}

function uab.ScaleAndAnchorOptions(obj)
    local args = {
        scale = {
            type = 'range',
            name = "Scale",
            desc = "Action button scale",
            disabled = function() return obj:IsDefaultScale() end,
            order = 120,
            min = 0.05,
            max = 2.00,
            step = 0.05,
            isPercent = true,
            get = function()
                return obj:GetScale()
            end,
            set = function(value)
                obj:SetScale(value)
            end,
        },
        point = {
            type = 'text',
            name = "Point",
            usage = "<name>",
            desc = "Select the point on the cancel button that positioning in based from",
            disabled = function() return obj:IsDefaultPosition() end,
            order = 130,
            get = function()
                return obj:GetPoint()
            end,
            set = function(pt)
                obj:SetPoint(pt)
            end,
            validate = {
                "BOTTOM",
                "BOTTOMLEFT",
                "BOTTOMRIGHT",
                "CENTER",
                "LEFT",
                "RIGHT",
                "TOP",
                "TOPLEFT",
                "TOPRIGHT",
            },
        },
        relativepoint = {
            type = 'text',
            name = "Relative Point",
            usage = "<name>",
            desc = "Select the point the buttons are relative to",
            disabled = function() return obj:IsDefaultPosition() end,
            order = 140,
            get = function()
                return obj:GetRelPoint()
            end,
            set = function(pt)
                obj:SetRelPoint(pt)
            end,
            validate = {
                "cursor",
                "screen",
                "BOTTOM",
                "BOTTOMLEFT",
                "BOTTOMRIGHT",
                "CENTER",
                "LEFT",
                "RIGHT",
                "TOP",
                "TOPLEFT",
                "TOPRIGHT",
            },
        },
        horizontal = {
            type = 'text',
            name = "Horizontal",
            desc = "Horizontal offset (positive is right, negative is left, 0 is over cursor)",
            disabled = function() return obj:IsDefaultPosition() end,
            usage = "<number>",
            order = 150,
            get = function()
                return obj:GetHorizontal() or 0
            end,
            set = function(value)
                if value and tonumber(value) then
                    obj:SetHorizontal(math.floor(tonumber(value)))
                end
            end,
            validate = function(value)
                if value and tonumber(value) then
                    local v = math.floor(tonumber(value))
                    if tostring(v) == value then
                        return true
                    end
                end
            end,
        },
        vertical = {
            type = 'text',
            name = "Vertical",
            desc = "Vertical offset (positive is up, negative is down, 0 is over cursor)",
            disabled = function() return obj:IsDefaultPosition() end,
            usage = "<number>",
            order = 160,
            get = function()
                return obj:GetVertical() or 0
            end,
            set = function(value)
                if value and tonumber(value) then
                    obj:SetVertical(math.floor(tonumber(value)))
                end
            end,
            validate = function(value)
                if value and tonumber(value) then
                    local v = math.floor(tonumber(value))
                    if tostring(v) == value then
                        return true
                    end
                end
            end,
        },
    }

    local usedefaultscale = obj:UseDefaultScale()
    if usedefaultscale then
        args.usedefaultscale = usedefaultscale
        args.usedefaultscale.order = 100
    end
    
    local usedefaultanchor = obj:UseDefaultAnchor()
    if usedefaultanchor then
        args.usedefaultanchor = usedefaultanchor
        args.usedefaultanchor.order = 110
    end

    if usedefaultscale or usedefaultanchor then
        args.spacing3 = {
            type = 'header',
            name = " ",
            order = 115,
        }
    end

    return args
end

local defaultScaleAndAnchor = {}

function defaultScaleAndAnchor:IsDefaultScale()
end

function defaultScaleAndAnchor:IsDefaultPosition()
end

function defaultScaleAndAnchor:UseDefaultScale()
end

function defaultScaleAndAnchor:UseDefaultAnchor()
end

function defaultScaleAndAnchor:GetScale()
    return uab.db.profile.scale.default.value
end

function defaultScaleAndAnchor:SetScale(v)
    uab.db.profile.scale.default.value = v
    uab:SetGlobalScale(v)
end

function defaultScaleAndAnchor:GetPoint()
    return uab.db.profile.scale.default.point
end

function defaultScaleAndAnchor:SetPoint(v)
    uab.db.profile.scale.default.point = v
    uab:SetGlobalPoint(v)
end

function defaultScaleAndAnchor:GetRelPoint()
    return uab.db.profile.scale.default.relpoint
end

function defaultScaleAndAnchor:SetRelPoint(v)
    uab.db.profile.scale.default.relpoint = v
    uab:SetGlobalRelPoint(v)
end

function defaultScaleAndAnchor:GetHorizontal()
    return uab.db.profile.scale.default.x or 0
end

function defaultScaleAndAnchor:SetHorizontal(v)
    uab.db.profile.scale.default.x = v
    uab:SetGlobalOffsetX(v)
end

function defaultScaleAndAnchor:GetVertical()
    return uab.db.profile.scale.default.y or 0
end

function defaultScaleAndAnchor:SetVertical(v)
    uab.db.profile.scale.default.y = v
    uab:SetGlobalOffsetY(v)
end

--[[
Constant menu/slash commands.
--]]
uab.options = {
    type = 'group',
    args = {
        activators = {
            type = 'group',
            name = "Activators",
            desc = "Create, modify, and remove activators",
            disabled = function() return InCombatLockdown() end,
    		order = 10,
            args = {},
        },
        
        lookandfeel = {
            type = 'group',
            name = "Look and Feel",
            desc = "Options that affect how buttons appear.  Includes layout, positioning, scaling, and button behavior",
            disabled = function() return InCombatLockdown() end,
    		order = 20,
            args = {
                spacing2 = {
            		name = "Button Appearance",
            		type = 'header',
            		order = 22,
            	},

                defaultanchorscale = {
                    type = 'group',
                    name = "Default",
                    desc = "Default button positioning and layout",
            		order = 30,
                    args = uab.ScaleAndAnchorOptions(defaultScaleAndAnchor)
                },
                                        
                groupanchorscale = {
                    type = 'group',
                    name = "Groups",
                    desc = "Anchor, scale, and behavior information for each unit group",
                    order = 40,
                    args  = {},
                },
                
                actionsets = {
                    type = 'group',
                    name = "Action Set Layouts",
                    desc = "Layout settings for each action set",
                    order = 45,
                    args  = {},
                },
                
                framegroups = {
                    type = 'group',
                    name = "Frame groups",
                    desc = "Frame groupings.  Enable/disable frames for UAB support, and define the anchor, scale, and behavior information for each frame group",
                    disabled = function() return InCombatLockdown() end,
            		order = 55,
                    args = {},
                },
                
                spacing3 = {
            		name = " ",
            		type = 'header',
            		order = 56,
            	},
            
                behaviorhdr = {
            		name = "Global Behavior",
            		type = 'header',
            		order = 57,
            	},

                directcast = {
                    type = 'toggle',
                    name = "Direct cast",
                    desc = "Use direct click-casting when action set contains a single action",
                    order = 60,
                    get = function()
                        return uab.db.profile.directcast
                    end,
                    set = function(v)
                        if v then
                            uab:SetDirectCast()
                        else
                            uab:ClearDirectCast()
                        end
                    end,
                },
                tooltip = {
                    type = 'toggle',
                    name = "Tooltips",
                    desc = "Show action tooltips",
            		order = 70,
                    get = function()
                        return uab.db.profile.tooltips
                    end,
                    set = function(v)
                        uab.db.profile.tooltips = v
                    end,
                },
--[[
                fastchangeunit = {
                    type = 'toggle',
                    name = "Fast change unit",
                    desc = "When selected, clicking on a different unit frame when action buttons are already shown will immediately reposition the buttons to the new unit frame and associate any action with that unit.  When not selected, clicking a new unit frame closes the action buttons",
            		order = 90,
                    get = function()
                        return uab.db.profile.fastchangeunit
                    end,
                    set = function(v)
                        uab.db.profile.fastchangeunit = v
                        uab.main:FastChangeUnitChanged()
                    end,
                },
--]]                
            }
    	},
    
        unitstatus = {
            type = 'group',
            name = "Unit Status",
            desc = "Unit status settings",
            disabled = function() return InCombatLockdown() end,
    		order = 40,
            args = {},
    	},

        effectsettings = {
            type = 'group',
            name = "Effect Settings",
            desc = "Button indicator settings",
            disabled = function() return InCombatLockdown() end,
            order = 50,
            args = {
                highlight = {
                    type = 'range',
                    name = "Highlight Width",
                    desc = "Width of the highlight border, when active.",
                    order = 10,
                    min = 1,
                    max = 5,
                    step = 1,
                    isPercent = false,
                    get = function()
                        return uab.db.profile.borderwidth
                    end,
                    set = function(value)
                        uab.db.profile.borderwidth = value
                        uab.main:HighlightBorderChanged()
                    end,
                },
                blinkrate = {
                    type = 'range',
                    name = "Blink Rate",
                    desc = "The rate at which the action button blinks.  This is the complete cycle time from visible back to visible.",
                    order = 20,
                    min = 0.1,
                    max = 2.0,
                    step = 0.1,
                    isPercent = false,
                    get = function()
                        return uab.db.profile.blinkcycle
                    end,
                    set = function(value)
                        uab.db.profile.blinkcycle = value
                    end,
                },
                lowalpha = {
                    type = 'range',
                    name = "Low alpha",
                    desc = "The alpha value to use when displaying a button with low alpha.",
                    order = 30,
                    min = 0.05,
                    max = 1.0,
                    step = 0.05,
                    isPercent = false,
                    get = function()
                        return uab.db.profile.alpha
                    end,
                    set = function(value)
                        uab.db.profile.alpha = value
                    end,
                },
                checkusable = {
                    type = 'toggle',
                    name = "Check usable",
                    desc = "When checked, examine whether an action is usable, and if not usable, show as unusable in addition to other action button indicators.",
                    disabled = function() return uab.db.profile.hideunusable end,
                    order = 40,
                    get = function()
                        return uab.db.profile.checkusable or uab.db.profile.hideunusable
                    end,
                    set = function(value)
                        uab.db.profile.checkusable = value
                    end,
                },
                hideunusable = {
                    type = 'toggle',
                    name = "Hide unusable",
                    desc = "When checked, hide actions if they are not usable.  This setting does not affect actions whose action effects make the action display as not usable.",
                    order = 50,
                    get = function()
                        return uab.db.profile.hideunusable
                    end,
                    set = function(v)
                        uab.db.profile.hideunusable = v
                        if v then
                            uab.db.profile.checkusable = true
                        end
                        uabActionButton:UnitStatusChanged()
                    end,
                },
            },
        },
        
        debug = {
            type = 'group',
            name = "Debug",
            desc = "Enable debug messages for UAB modules",
            disabled = function() return InCombatLockdown() end,
            order = 100,
            args = {},
        },
        
        dump = {
            type = 'execute',
            name = "Dump",
            desc = "Dump the current state of all the uab objects",
            order = 110,
            func = function()
                uab:Dump()
            end,
        },
    },
}

uabDumper = {}
uabDumper.dumplines = {}
uabDumper.indent = 0

function uabDumper:Clear()
    for idx, text in pairs(self.dumplines) do
        self.dumplines[idx] = nil
    end
end

function uabDumper:IncIndent()
    self.indent = self.indent + 1
end

function uabDumper:DecIndent()
    self.indent = self.indent - 1
end

function uabDumper:DumpFrameAttribute(frame, attr)
    local fname = frame:GetName()
    if not fname then
        fname = "Unknown frame (no name)"
    end
    local hdr = "Attribute ("..fname..") - "..attr
    local val = frame:GetAttribute(attr)
    if not val then
        self:Dump(hdr.." : nil")
    elseif type(val) == "string" then
        self:Dump(hdr.." : "..val)
    elseif type(val) == "boolean" then
        self:Dump(hdr.." : true")
    elseif type(val) == "table" and val.GetName then
        self:Dump(hdr.." : "..val:GetName())
    else
        self:Dump(hdr.." : raw value")
    end
end

function uabDumper:Dump(text)
    local dumptext = ""
    for idx = 1, self.indent, 1 do
        dumptext = dumptext .. "    "
    end
    dumptext = dumptext .. text
    
    table.insert(self.dumplines, dumptext)
end

function uabDumper:Write()
    for idx, text in pairs(self.dumplines) do
        table.insert(UAB2Debug, self.dumplines[idx])
    end
end

function uab:DumpRegisteredFrames(dumper)
    dumper:Dump("Registered frames")
    dumper:IncIndent()
    for frame, _ in pairs(self.enabledframes) do
        dumper:DumpFrameAttribute(frame, "stateheader")
    end
    dumper:DecIndent()
end

function uab:Dump()
    uabDumper:Clear()
    
    self:DumpRegisteredFrames(uabDumper)
    self.stancectrlsecureheader:Dump(uabDumper)
    uab.classes.SecureHeader:Dump(uabDumper)
    self:IterateActionSets("Dump", uabDumper)
    self:IterateActivators("Dump", uabDumper)
    
    uabDumper:Write()
end

local groupScaleAndAnchor = uab.mixins.aceOO.Class()

function groupScaleAndAnchor.prototype:init(unitgroup)
    groupScaleAndAnchor.super.prototype.init(self)
    self.unitgroup = unitgroup
    self.profilescale = uab.db.profile.scale[self.unitgroup.name]
    self.profileposition = uab.db.profile.position[self.unitgroup.name]
end

function groupScaleAndAnchor.prototype:SetGroup(unitgroup)
    self.unitgroup = unitgroup
end

function groupScaleAndAnchor.prototype:IsDefaultScale()
    return self.profilescale.usedefault
end

function groupScaleAndAnchor.prototype:IsDefaultPosition()
    return self.profileposition.usedefault
end

function groupScaleAndAnchor.prototype:UseDefaultScale()
    return {
        type = 'toggle',
        name = "Default scale",
        desc = "Use default scale value for this group",
        get = function()
            return self.profilescale.usedefault
        end,
        set = function(v)
            self.profilescale.usedefault = v

            if not v then
                if not self.profilescale.value then
                    self.profilescale.value = uab.db.profile.scale.default.value
                end
                
                uab:SetUnitGroupScale(self.unitgroup.code, self.profilescale.value)
            else
                uab:SetUnitGroupScale(self.unitgroup.code, nil)
            end
        end,
    }
end

function groupScaleAndAnchor.prototype:UseDefaultAnchor()
    return {
        type = 'toggle',
        name = "Default anchor",
        desc = "Use default action button anchor information for this group",
        get = function()
            return self.profileposition.usedefault
        end,
        set = function(v)
            self.profileposition.usedefault = v
            
            if not v then
                if not self.profileposition.point then
                    self.profileposition.point = uab.db.profile.position.default.point
                end

                uab:SetUnitGroupPoint(self.unitgroup.code, self.profileposition.point)
                
                if not self.profileposition.relpoint then
                    self.profileposition.relpoint = uab.db.profile.position.default.relpoint
                end

                uab:SetUnitGroupRelPoint(self.unitgroup.code, self.profileposition.relpoint)
                
                if not self.profileposition.x then
                    self.profileposition.x = uab.db.profile.position.default.x
                end

                uab:SetUnitGroupOffsetX(self.unitgroup.code, self.profileposition.x)
                
                if not self.profileposition.y then
                    self.profileposition.y = uab.db.profile.position.default.y
                end
                
                uab:SetUnitGroupOffsetY(self.unitgroup.code, self.profileposition.y)
            else
                uab:SetUnitGroupPoint(self.unitgroup.code, nil)
                uab:SetUnitGroupRelPoint(self.unitgroup.code, nil)
                uab:SetUnitGroupOffsetX(self.unitgroup.code, nil)
                uab:SetUnitGroupOffsetY(self.unitgroup.code, nil)
            end
        end,
    }
end

function groupScaleAndAnchor.prototype:GetScale()
    return self.profilescale.value or uab.db.profile.scale.default.value
end

function groupScaleAndAnchor.prototype:SetScale(v)
    self.profilescale.value = v
    uab:SetUnitGroupScale(self.unitgroup.code, v)
end

function groupScaleAndAnchor.prototype:GetPoint()
    return self.profileposition.point or uab.db.profile.position.default.point or "CENTER"
end

function groupScaleAndAnchor.prototype:SetPoint(v)
    self.profileposition.point = v
    uab:SetUnitGroupPoint(self.unitgroup.code, v)
end

function groupScaleAndAnchor.prototype:GetRelPoint()
    return self.profileposition.relpoint or uab.db.profile.position.default.relpoint or "cursor"
end

function groupScaleAndAnchor.prototype:SetRelPoint(v)
    self.profileposition.relpoint = v
    uab:SetUnitGroupRelPoint(self.unitgroup.code, v)
end

function groupScaleAndAnchor.prototype:GetHorizontal()
    return self.profileposition.x or uab.db.profile.position.default.x or 0
end

function groupScaleAndAnchor.prototype:SetHorizontal(v)
    self.profileposition.x = v
    uab:SetUnitGroupOffsetX(self.unitgroup.code, v)
end

function groupScaleAndAnchor.prototype:GetVertical()
    return self.profileposition.y or uab.db.profile.position.default.y or 0
end

function groupScaleAndAnchor.prototype:SetVertical(v)
    self.profileposition.y = v
    uab:SetUnitGroupOffsetY(self.unitgroup.code, v)
end

function uab:SetupUnitGroupAnchorScaleOptions()
    uab.GroupIterate(
        function(self, context, name, info)
            local scaleanchor = groupScaleAndAnchor:new(info)
            
            scaleanchor:SetGroup(info)
            context.count = context.count + 1

            local options = {
                type = 'group',
                name = info.displayName,
                desc = "Set the position, scale, and behavior, for the action buttons when related to frames in this group",
                order = context.count,
                args = uab.ScaleAndAnchorOptions(scaleanchor)
            }

            options.args.spacing1 = {
                name = " ",
                type = 'header',
                order = 30,
            }

            uab.options.args.lookandfeel.args.groupanchorscale.args[info.name] = options
            
            if uab.db.profile.scale[info.name].usedefault then
                uab:SetUnitGroupScale(info.code, nil)
            else
                uab:SetUnitGroupScale(info.code, uab.db.profile.scale[info.name].value)
            end

            if uab.db.profile.position[info.name].usedefault then
                uab:SetUnitGroupPoint(info.code, nil)
                uab:SetUnitGroupRelPoint(info.code, nil)
                uab:SetUnitGroupOffsetX(info.code, nil)
                uab:SetUnitGroupOffsetY(info.code, nil)
            else
                uab:SetUnitGroupPoint(info.code, uab.db.profile.position[info.name].point)
                uab:SetUnitGroupRelPoint(info.code, uab.db.profile.position[info.name].relpoint)
                uab:SetUnitGroupOffsetX(info.code, uab.db.profile.position[info.name].x)
                uab:SetUnitGroupOffsetY(info.code, uab.db.profile.position[info.name].y)
            end
        end,
        nil,
        { count = 100 })
end

function uab:BuildActivatorOptions(skipapplysettings)
    for name, _ in pairs(self.options.args.activators.args) do
        self.ClearTable(self.options.args.activators.args[name])
        self.options.args.activators.args[name] = nil
    end

    for code, data in pairs(self.db.profile.activators) do
        local activatorobj = self:GetActivatorObj(code)
        local activatorcode = activatorobj.activatorcode
        local include

        if data.actionsets then
            -- Go through all the action sets for this activator, adding that action
            -- set code to the action set codes for the activator object
            for _, name in pairs(data.actionsets) do
                for actionsetcode, obj in pairs(self.actionsetobjs) do
                    if obj:GetName() == name then
                        include = true
                        break
                    end
                end
                if include then
                    break
                end
            end
        end

        if include then
            local name
            local options = self.options.args.activators
            
            name = data.name
            local info = self.constants.activatorInfo[activatorcode]
            local mousename

            if info then    
                if info.prefixshift then
                    if info.prefixctrl then
                        if info.prefixalt then
                            mousename = "Shift+Ctrl+Alt "..info.button
                        else
                            mousename = "Shift+Ctrl "..info.button
                        end
                    else
                        if info.prefixalt then
                            mousename = "Shift+Alt "..info.button
                        else
                            mousename = "Shift "..info.button
                        end
                    end
                elseif info.prefixctrl then
                    if info.prefixalt then
                        mousename = "Ctrl+Alt "..info.button
                    else
                        mousename = "Ctrl "..info.button
                    end
                elseif info.prefixalt then
                    mousename = "Alt "..info.button
                else
                    mousename = info.button
                end
            end

            if not data.name then
                data.name = mousename
            end

            local argname            
            if data.name ~= mousename then
                argname = data.name .. "(" .. mousename ..")"
            else
                argname = data.name
            end

            options.args[data.name] = {
                type = 'group',
                name = argname,
                desc = "Set the options for this activator",
                order = math.fmod(activatorcode, 8) + 20,
                args = {}
            }

            local args = options.args[data.name]

            args.args.stayopen = {
                type = 'text',
                name = "Stay open",
                desc = "Specify the show/hide behavior of the action buttons when clicked.",
                order = 4,
                get = function()
                    if not self.db.profile.activators[code] or not self.db.profile.activators[code].stayopen then
                        return "Use default"
                    end
                    
                    if self.db.profile.activators[code].stayopen.enabled then
                        return "Yes"
                    else
                        return "No"
                    end
                end,
                set = function(v)
                    if v == "Use default" then
                        self:DefaultActivatorStayOpen(activatorcode)
                    elseif v == "Yes" then
                        self:SetActivatorStayOpen(activatorcode)
                    else
                        self:ClearActivatorStayOpen(activatorcode)
                    end
                end,
                validate = {
                    "Yes",
                    "No",
                    "Use default",
                },
            }
    
            args.args.name = {
                type = 'text',
                name = "Name",
                desc = "Set the name for this activator.",
                usage = "<name>",
                order = 5,
                get = function()
                    return data.name
                end,
                set = function(v)
                    data.name = v
                    self:BuildActivatorOptions(true)
                    self.mixins.dewdrop:Close()
                end,
                validate = function(v)
                    local found
                    for _, adata in pairs(self.db.profile.activators) do
                        if adata.name and adata.name == v then
                            found = true
                            break
                        end
                    end
                    return not found
                end,
            }
    
            args.args.independent = {
                type = 'toggle',
                name = "Independent show/hide",
                desc = "When set, action buttons for this activator show and hide indepedently from other activators.",
                order = 6,
                disabled = function()
                    return not self.db.profile.activators[code] or not self.db.profile.activators[code].inuse
                end,
                get = function()
                    return self.db.profile.activators[code] and self.db.profile.activators[code].independent
                end,
                set = function(v)
                    if v then
                        self:SetIndependentActivatorSecureHeader(activatorcode)
                    else
                        self:ClearIndependentActivatorSecureHeader(activatorcode)
                    end
                end,
            }
            
            args.args.groupindependent = {
                type = 'group',
                name = "Unit Group show/hide",
                desc = "Indicate the unit groups that show/hide action buttons independent from other unit groups for this activator.",
                order = 7,
                disabled = function()
                    return not self.db.profile.activators[code] or not self.db.profile.activators[code].inuse or not self.db.profile.activators[code].independent
                end,
                args = {}
            }

            if not skipapplysettings then    
                if self.db.profile.activators[code] and self.db.profile.activators[code].stayopen then
                    if self.db.profile.activators[code].stayopen.enabled then
                        self:SetActivatorStayOpen(activatorcode)
                    else
                        self:ClearActivatorStayOpen(activatorcode)
                    end
                else
                    self:DefaultActivatorStayOpen(activatorcode)
                end
    
                if self.db.profile.activators[code] and self.db.profile.activators[code].independent then
                    self:SetIndependentActivatorSecureHeader(activatorcode)
                else
                    self:ClearIndependentActivatorSecureHeader(activatorcode)
                end
            end
    
            self.GroupIterate(
                function(self, context, name, info)
                    context.args[name] = {
                        type = 'toggle',
                        name = info.displayName,
                        desc = "When set, action buttons for this unit group show and hide indepedently from other unit groups for this activator.",
                        order = context.count,
                        get = function()
                            return uab.db.profile.activators[context.activatorname] and
                                uab.db.profile.activators[context.activatorname].independentgroup and
                                uab.db.profile.activators[context.activatorname].independentgroup[name]
                        end,
                        set = function(v)
                            if v then
                                uab:SetIndependentActivatorUnitGroupSecureHeader(context.activatorcode, info.code)
                            else
                                uab:ClearIndependentActivatorUnitGroupSecureHeader(context.activatorcode, info.code)
                            end
                        end,
                    }

                    if not context.skipapplysettings then
                        if uab.db.profile.activators[context.activatorname] and
                            uab.db.profile.activators[context.activatorname].independentgroup and
                            uab.db.profile.activators[context.activatorname].independentgroup[name] then
                            uab:SetIndependentActivatorUnitGroupSecureHeader(context.activatorcode, info.code)
                        else
                            uab:ClearIndependentActivatorUnitGroupSecureHeader(context.activatorcode, info.code)
                        end
                    end
    
                    context.count = context.count + 10
                end,
                self,
                {   skipapplysettings = skipapplysettings,
                    args = args.args.groupindependent.args,
                    count = 10,
                    activatorname = code,
                    activatorcode = activatorcode
                })
        end
    end

    self.options.args.activators.args.stayopen = {
        type = 'toggle',
        name = "Stay open",
        desc = "Keep actions displayed after use.  Clicking cancel closes actions",
        order = 5,
        get = function()
            return self.db.profile.stayopen
        end,
        set = function(v)
            if v then
                self:SetGlobalStayOpen()
            else
                self:ClearGlobalStayOpen()
            end
        end,
    }

    self.options.args.activators.args.groupindependent = {
        type = 'group',
        name = "Unit Group show/hide",
        desc = "Indicate the unit groups that show/hide action buttons independent from other unit groups for non-independent activators.",
        order = 6,
        args = {}
    }

    self.GroupIterate(
        function(self, context, name, info)
            context.args[name] = {
                type = 'toggle',
                name = info.displayName,
                desc = "When set, action buttons for this unit group show and hide indepedently from other unit groups for this activator.",
                order = context.count,
                get = function()
                    return uab.db.profile.independent[name]
                end,
                set = function(v)
                    if v then
                        uab:SetIndependentUnitGroupSecureHeader(info.code)
                    else
                        uab:ClearIndependentUnitGroupSecureHeader(info.code)
                    end
                end,
            }

            if not context.skipapplysettings then
                if uab.db.profile.independent[name] then
                    uab:SetIndependentUnitGroupSecureHeader(info.code)
                else
                    uab:ClearIndependentUnitGroupSecureHeader(info.code)
                end
            end
            
            context.count = context.count + 10
        end,
        self,
        {   skipapplysettings = skipapplysettings,
            args = self.options.args.activators.args.groupindependent.args,
            count = 10
        })

    self.options.args.activators.args.spacing1 = {
        name = " ",
        type = 'header',
        order = 8,
    }

    if not skipapplysettings then
        if self.db.profile.stayopen then
            self:SetGlobalStayOpen()
        else
            self:ClearGlobalStayOpen()
        end
    end

    self:TriggerEvent("UnitActionBars_RefreshActivators")
end

function uab:CreateActivators()
    for num, info in pairs(self.constants.activatorInfo) do
        self:SetActivatorObj(num, self.classes.Activator:new(num))
    end
end

local nextactionsetcode = 1

function uab:CreateActionSet(name)
    local code = nextactionsetcode
    local actionsetobj = self.classes.ActionSet:new(code, name)

    nextactionsetcode = nextactionsetcode + 1

    self:SetActionSetObj(code, actionsetobj)
    return code, actionsetobj
end

function uab:CreateNewActionSet(name)
    local code, actionsetobj = self:CreateActionSet(name)

    self.db.profile.actionsets[name] = {}

    local actionset = self.db.profile.actionsets[name]

    actionset.stances = {}
    for stancecode = 0, uab.MaxCurrentClassStances(), 1 do
        actionset.stances[stancecode] = {}
        actionset.stances[stancecode].hostile = {}
        for gname, info in pairs(self.constants.groupInfo) do
            actionset.stances[stancecode][gname] = {}
        end
    end

    return code, actionsetobj
end

function uab:CreateActionSets()
    for name, data in pairs(self.db.profile.actionsets) do
        local code, actionsetobj = self:CreateActionSet(name)
        
        -- Now, add the action buttons for each of the unit groups of each
        -- stance to this action set.

        for stancecode = 0, uab.MaxCurrentClassStances(), 1 do
            local unitgroupactions = data.stances[stancecode]
            if unitgroupactions then
                for unitgroupname, actions in pairs(unitgroupactions) do
                    if unitgroupname ~= "hostile" then
                        local unitgroupcode = self.constants.groupInfo[unitgroupname].code
        
                        for idx, actiondata in pairs(actions) do
                            local wrapper = uabActionWrapper:CreateActionWrapper(actiondata)
                            actionsetobj:AddAction(stancecode, unitgroupcode, idx, wrapper)
                        end
                    else
                        for idx, actiondata in pairs(actions) do
                            local wrapper = uabActionWrapper:CreateActionWrapper(actiondata, true)
                            actionsetobj:AddHostileAction(stancecode, idx, wrapper)
                        end
                    end
                end
            end
        end
        
        code = code + 1
    end
end

function uab:AssociateActionSetsToActivators()
    local actionsetcodes = {}
    
    for code, data in pairs(self.db.profile.activators) do
        local activatorobj = self:GetActivatorObj(code)

        if data.actionsets then
            -- Go through all the action sets for this activator, adding that action
            -- set code to the action set codes for the activator object
            for _, name in pairs(data.actionsets) do
                for actionsetcode, obj in pairs(self.actionsetobjs) do
                    if obj:GetName() == name then
                        data.inuse = true
                        activatorobj:AddActionSetCode(actionsetcode)
                    end
                end
            end
        end
    end
end

function uab:GetActivatorActionSetName(name)
    if not self.db.profile.actionsets[name] then
        return name
    elseif not self.db.profile.actionsets[key .. "_migrated"] then
        return key .. "_migrated"
    else                
        local found = true
        local idx = 0
        
        while found do
            idx = idx + 1
            found = self.db.profile.actionsets[key .. "_migrated_" .. idx]
        end
        
        return key .. "_migrated_" .. idx
    end
end

function uab:MigrateSavedVariableData()
                
    -- If a version 1 savedvariables file, then derive the activator code
    -- from the data and ignore the name (code).  If a version 2 savedvariables
    -- file, then the code identifies the activator.

    if not self.db.profile.version or self.db.profile.version < 2 then
        self.db.profile.version = 2
        
        uab.print("UnitActionBars: Migrating to version 2...")
        
        local activatorcodes = {}

    	for key, val in pairs(self.db.profile.activators) do
    	    if key ~= "" then
                if not val.actionset then
                    local actiondata = {}
        
                    actiondata.groups = {}
                    for _, info in pairs(self.constants.groupInfo) do
                        actiondata.groups[info.name] = uab.CloneTable(val.groups[info.name])
                    end
    
                    for n, v in pairs(val.groups) do
                        val.groups[n] = nil
                    end
                    
                    val.groups = nil
    
                    val.actionset = self:GetActivatorActionSetName(key)
                    self.db.profile.actionsets[val.actionset] = actiondata
                end
            end
            activatorcodes[key] = true
    	end

        for code, _ in pairs(activatorcodes) do
            local activatorname = code

            if code ~= "" then
                local old_data = self.db.profile.activators[code]
                local data = uab.CloneTable(self.db.profile.activators[code])
                local activatorcode = code
                
                for num, info in pairs(uab.constants.activatorInfo) do
                    if data.button == info.button then
                        if (data.prefixshift and info.prefixshift) or
                            (not data.prefixshift and not info.prefixshift) then
                            if (data.prefixctrl and info.prefixctrl) or
                                (not data.prefixctrl and not info.prefixctrl) then
                                if (data.prefixalt and info.prefixalt) or
                                    (not data.prefixalt and not info.prefixalt) then
                                    activatorcode = num
                                    break
                                end
                            end
                        end
                    end
                end
    
                -- Allow any number of action sets to be associated with this
                -- activator.  This means converting from actionset to actionsets.

                data.name = activatorname    
                data.prefixshift = nil
                data.prefixctrl = nil
                data.prefixalt = nil
                data.button = nil
                data.actionsets = {}
                table.insert(data.actionsets, old_data.actionset)
                if old_data.actionset then
                    data.inuse = true
                end
                data.actionset = nil
                
                self.db.profile.activators[activatorcode] = data
            end
            
            self.db.profile.activators[code] = nil
        end

        local actionsetgroups = {}

        for name, data in pairs(self.db.profile.actionsets) do
            -- Now, for each of the unit groups, copy the action information
            -- for that unit group to all stances

            data.stances = {}
            
            for group, unitgroupinfo in pairs(data.groups) do
                actionsetgroups[group] = true
            end

            for group, _ in pairs(actionsetgroups) do
                local unitgroupinfo = data.groups[group]

                -- Skip unit groups that we don't support anymore
                if uab.constants.groupInfo[group] then
                    if not data.stances[0] then
                        data.stances[0] = {}
                    end

                    -- unitgroupinfo information is structured as follows:
                    --  enabled (boolean) - we ignore this, as it was never used
                    --  actions (indexed table) - these are the actions to copy
                    data.stances[0][group] = uab.CloneTable(unitgroupinfo.actions)
                end
                    
                data.groups[group] = nil
            end
            
            data.groups = nil
            
            for group, _ in pairs(actionsetgroups) do
                actionsetgroups[group] = nil
            end
        end
        
        uab.print("UnitActionBars: Migration to version 2 completed")
    end
end

function uab:ConnectToUnitFrames()
	self:RegisterAllFrames()
end

function uab:RegisterAllFrames()
    local pending = self:SetPendingAttributes()
    
    -- Register all frames that snuck in before we did =)
    self:StartPostponeApplyNewActivatorFrameControl()
    
    for frame in pairs(self.ccframes) do
		self:RegisterFrame(frame)
    end
    
    self:EndPostponeApplyNewActivatorFrameControl()
    
    if pending then
        self:ClearPendingAttributes()
    end
end

function uab:UnregisterAllFrames()
    -- Unregister all the frames we currently have registered
    for frame in pairs(self.ccframes) do
		self:UnregisterFrame(frame)
    end
end

function uab:CCFNewIndex(t, k, v)
    local fname = k:GetName()
    if not fname then
        fname = "Unknown frame (no name)"
    end

	if self.ccframesEnabled and not self.ccframesNewIndex then
	    if v == nil then
	        self:UnregisterFrame(k)
	    else
	        self:RegisterFrame(k)
	    end
	end
	
    --[[
	If there was someone loaded before us, and they catch
    ClickCastFrames changes, then let them have the event
    and let them do the rawset.  First, they need to process
    the event, and second, if they process it, they will
    either do the rawset or pass it to the next one in line
    as we are doing here.
    --]]
    
    if self.ccframesNewIndex then
        self.ccframesNewIndex(t,k,v)
    else
        if v == nil then
            rawset(self.ccframes, k, nil)
        else
            rawset(self.ccframes, k, v)
        end
    end
end

function uab:HookClickCastFrames()
	if not self.ccframesHooked then
	    self.ccframesHooked = true
	    
		ClickCastFrames = ClickCastFrames or {}
		self.ccframes = ClickCastFrames

	    local mt = getmetatable(ClickCastFrames)
	    if not mt then
	        mt = {}
	    end

	    self.ccframesNewIndex = mt.__newindex

        if self.ccframesNewIndex then
            --[[
            Another click-cast add-on is already loaded.  We
            should note this, and not attach ourselves.  Any
            further actions will be retained in case at some
            reload, the other click-cast add-on is not loaded
            but the actions will have no effect in the current
            session.
            --]]
            
            uab.print("Another click-casting add-on is already loaded.")
            uab.print("If you want to use UnitActionBars, please disable the other add-on.")
            uab.print("You may set up action sets, actions, and general settings, but those")
            uab.print("settings will not have any effect, as UnitActionBars will not attach")
            uab.print("to any frame to enable click-casting.")
        end
        
	    local newindex = function(t,k,v)
	        uab:CCFNewIndex(t, k, v)
	    end

	    mt.__newindex = newindex

	    ClickCastFrames = setmetatable(ClickCastFrames, mt)
	end
	
    local tbl = {
		PlayerFrame,
		PetFrame,
		PartyMemberFrame1,
		PartyMemberFrame2,
		PartyMemberFrame3,
		PartyMemberFrame4,
		PartyMemberFrame1PetFrame,
		PartyMemberFrame2PetFrame,
		PartyMemberFrame3PetFrame,
		PartyMemberFrame4PetFrame,
		TargetFrame,
		TargetofTargetFrame,
    }

    for i, frame in pairs(tbl) do
		if not ClickCastFrames[frame] then
            ClickCastFrames[frame] = true
        end
    end

	self:RegisterAllFrames()
end

function uab:UnhookClickCastFrames()
	--[[
	We don't actually disconnect from the metatable, as someone may
	have connected after us.  Instead, we just indicate that we should
	not process any registrations now.  They will still be in the table
	when we are re-enabled, and will be registered at that time.
	--]]

	self.ccframesEnabled = nil
end

function uab:RegisterFrame(frame)
    if not self.enabledframes[frame] then
        if InCombatLockdown() then
            self:DelayRegisterFrame(frame)
            return
        end

        local fname = frame:GetName()
        if not fname then
            fname = "Unknown frame (no name)"
        end
    
        local fgroup = uabFrameGroup:GetFrameGroup(frame, true)
        local enabled = fgroup and fgroup:IsEnabled()
        
        if enabled then
            if fgroup:IsDefaultFrameGroup() and uab.db.profile.warndefaultframegroup then
                uab.print("Unable to find frame group for '"..fname.."'.  Using default settings.")
            end
    
            self:EnableClickCastFrame(frame)
        end
    end
end

function uab:EnableClickCastFrame(frame)
    local pending = self:SetPendingAttributes()
    
    local fname = frame:GetName() or "Unknown frame (no name)"
    
    self.enabledframes[frame] = true

    self:SetAttribute(frame, "stateheader", self.stancectrlsecureheader.frame)

    local postpone = not self.postponeApplyNewActivatorFrameControl
    
    if postpone then
        self:StartPostponeApplyNewActivatorFrameControl()
    end

    self:IterateActivators("RegisterFrame", frame)

    if postpone then
        self:EndPostponeApplyNewActivatorFrameControl()
    end

    if pending then
        self:ClearPendingAttributes()
    end

    self:TriggerEvent("UnitActionBars_FrameRegistered", frame)
end

function uab:UnregisterFrame(frame)
    if InCombatLockdown() then
        self:DelayUnregisterFrame(frame)
        return
    end

    self:DisableClickCastFrame(frame)
end

function uab:DisableClickCastFrame(frame)
    local pending = self:SetPendingAttributes()
    
    self.enabledframes[frame] = nil
	self:IterateActivators("UnregisterFrame", frame)

    if pending then
        self:ClearPendingAttributes()
    end

    self:TriggerEvent("UnitActionBars_FrameUnregistered", frame)
end

function uab:DelayRegisterFrame(frame)
	self.combatRegistrations[frame] = true
end

function uab:DelayUnregisterFrame(frame)
	self.combatUnregistrations[frame] = true
end

function uab:DelayedFrameRegister()
    self:RegisterDelayedFrames()
    self:UnregisterDelayedFrames()
end

function uab:RegisterDelayedFrames()
    local pending = self:SetPendingAttributes()
    
	for frame, _ in pairs(self.combatRegistrations) do
        local fname = frame:GetName() or "Unknown frame (no name)"
        uab.print("UAB: Registering queued frame "..fname)
		self:RegisterFrame(frame)
        self.combatRegistrations[frame] = nil
	end

    if pending then
        self:ClearPendingAttributes()
    end
end

function uab:UnregisterDelayedFrames()
    local pending = self:SetPendingAttributes()
    
	for frame, _ in pairs(self.combatUnregistrations) do
        local fname = frame:GetName() or "Unknown frame (no name)"
        uab.print("UAB: Unregistering queued frame "..fname)
		self:UnregisterFrame(frame)
        self.combatUnregistrations[frame] = nil
	end

    if pending then
        self:ClearPendingAttributes()
    end
end

function uab:AttachHooks()
    self.combatRegistrations = {}
    self.combatUnregistrations = {}
    self.ccframesEnabled = true
    
    self:HookClickCastFrames()
    self:RegisterEvent("PLAYER_REGEN_ENABLED")
end

function uab:PLAYER_REGEN_ENABLED()
    --[[
    When we leave combat, we register and unregister any frames that requested to register/unregister
	during combat.
    --]]
    
	self:DelayedFrameRegister()
end

function uab:OnInitialize()
    self:RegisterDB("UAB2DB")
    self:RegisterDefaults('profile', self.profileDefaults)

    UAB2Debug = {}

    self:SetDebugging(self.db.profile.debugging)
    
    self.options.args.debug.args.UnitActionBars = {
        type = 'toggle',
        name = "UnitActionBars",
        desc = "Enable debugging for the UnitActionBars module",
        get = function()
            return self:IsDebugging()
        end,
        set = function(v)
            self.db.profile.debugging = v
            self:SetDebugging(self.db.profile.debugging)
        end,
    }
    
    self:RegisterChatCommand({ "/unitactionbars", "/uab" }, self.options)

	self.mixins.waterfall:Register('UnitActionBars',
        'aceOptions', self.options,
        'title', "UnitActionBars",
        'treeLevels', 7,
        'colorR', 0.8, 'colorG', 0.8, 'colorB', 0.8)
    
    self:RegisterChatCommand({"/uabw"}, function()
        self.mixins.waterfall:Open('UnitActionBars')
    end)
end

function uab:Setup()
    self:SetPendingAttributes()

    if uab.db.profile.directcast then
        self:SetDirectCast()
    else
        self:ClearDirectCast()
    end

    if self.db.profile.stayopen then
        self:SetGlobalStayOpen()
    else
        self:ClearGlobalStayOpen()
    end

    self:SetupUnitGroupAnchorScaleOptions()

    self:MigrateSavedVariableData()

    self:CreateActivators()
    self:CreateActionSets()
    self:AssociateActionSetsToActivators()
    self:ApplyStayOpen()

    self:BuildActivatorOptions()
    
    self.stancectrlsecureheader = uab.classes.StanceCtrlSecureHeader:new()

    self:ClearPendingAttributes()
end

function uab:OnEnable(first)
    self:Setup()
    self:AttachHooks()
    uab.classes.SecureHeader:SetStanceState()
    self.stancectrlsecureheader:SetStanceState()
    self:TriggerEvent("UnitActionBars_InitializationCompleted")
end

function uab:FindActivatorFrameControlForFrame(activatorname, unitframename, stancecode)
    if not stancecode then
        stancecode = GetShapeshiftForm(true)
    end

    for code, data in pairs(self.db.profile.activators) do
        if data.name and tostring(data.name) == activatorname then
            local frame = getglobal(unitframename)
            return self:GetActivatorObj(code):GetActivatorFrameControlForFrame(frame, stancecode)
        end
    end
end

function uab.DropDownSelect(frame, functionData)
    frame[functionData.func](frame, this, functionData.arg1, functionData.arg2, functionData.arg3)
end

function uab.GroupIterate(fcn, self, context)
    local key, val
    for key, val in pairs(uab.constants.groupInfo) do
        local result = fcn and fcn(self, context, key, val)
        if result then
            return result
        end
    end
end

function uab.GetFrameUnit(frame)
    local unit = SecureButton_GetModifiedUnit(frame)
    if not unit and frame.GetUnit then
        unit = frame:GetUnit()
    end

    return unit
end

function uab.GetUnitGroupCodeFromUnitFrame(frame)
    local unit = uab.GetFrameUnit(frame)
    local info
    
    if unit then
        if unit == "player" or unit == "pet" or "focus" == unit then
            info = uab.constants.groupInfo[unit]
        elseif unit:find("target$") then
            info = uab.constants.groupInfo.target
        else
            info = uab.constants.groupInfo.partyraid
        end
    end
    
    if not info then
        info = uab.constants.groupInfo.partyraid
    end
    
    return info.code
end

function uab.CloneTable(tbl)
	if type(tbl) == "table" then
        local newtbl = {}

        for val, data in pairs(tbl) do
            newtbl[val] = uab.CloneTable(data)
        end
        
        return newtbl
    else
        return tbl
    end
end

function uab.ClearTable(tbl)
	if type(tbl) == "table" then
        for val, data in pairs(tbl) do
            uab.ClearTable(data)
            tbl[val] = nil
        end
    end
end

function uab.GetStanceName(code)
    local _, class = UnitClass("player")
    if class == "PRIEST" then
        local haveshadow
        local havespirit

        local numTabs = GetNumTalentTabs()
        for tab=1, numTabs, 1 do
            local numTalents = GetNumTalents(tab)
            for idx = 1, numTalents, 1 do
                local name, icon, tier, col, cur, max, isEx, meets = GetTalentInfo(tab, idx)
                if name == "Spirit of Redemption" then
                    if cur > 0 then
                        havespirit = true
                    end
                elseif name == "Shadowform" then
                    if cur > 0 then
                        haveshadow = true
                    end
                end
            end
        end

        local stances = uab.constants.stances[class]

        if code == 0 then
            return stances and stances[code]
        elseif code == 1 then
            if haveshadow then
                return stances and stances[code]
            elseif havespirit then
                return stances and stances[code+1]
            else
                return
            end
        elseif haveshadow and havespirit then
            return stances and stances[code]
        end
    else
        local icon, name, active, castable = GetShapeshiftFormInfo(code)
        if name then
            return name
        end
    
        local stances = uab.constants.stances[class]

        return stances and stances[code]
    end
end

function uab.DoStartMoving(frame, btn)
	if (btn == "LeftButton") then
		frame:SetUserPlaced(true)
		frame:StartMoving()
		frame.ismoving = true
	end
end
    			
function uab.DoStopMoving(frame)
	if (frame.ismoving) then
		frame:StopMovingOrSizing()
		frame:SetUserPlaced(true)
		frame.ismoving = nil
	end
end

